# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>
#          Olivier Tilloy <olivier@fluendo.com>


from elisa.core.utils.i18n import install_translation
from elisa.core.action import ContextualAction


_ = install_translation('poblesec')


class Action(object):
    """Contextul action for a Controller object.

    The intended use is for controllers to subclass this and override the run()
    method, that can do everything you want, assuming to deal with a specific
    controller class.

    @ivar controller: the controller on which to run the action
    @type controller: L{elisa.plugins.pigment.pigment_controller.PigmentController}
    """

    icon = ''
    title = ''
    subtitle = ''

    def __init__(self, controller):
        self.controller = controller

    def run(self):
        raise NotImplementedError()


class ActionPathNotSetError(Exception):
    pass


class LinkAction(Action):
    crumb_title = None
    path = None
    args = {}

    def run(self):
        if self.path is None:
            raise ActionPathNotSetError()

        crumb_title = self.crumb_title
        if crumb_title is None:
            crumb_title = self.title

        browser = self.controller.frontend.retrieve_controllers('/poblesec/browser')[0]
        dfr = browser.history.append_controller(self.path, crumb_title,
                                                **self.args)

        return dfr


###############################################################################
#
# NEW GENERATION CONTEXTUAL ACTIONS
#
###############################################################################


class OpenControllerAction(ContextualAction):
    """
    Specialized implementation of L{ContextualAction} that given a path to a
    controller attempts to create it and navigate to it when the action is
    executed.
    Any keyword arguments passed to the action when it is created will be
    also passed as keyword arguments to the controller when it is created.

    @ivar controller: the controller on which to run the action
    @type controller: L{elisa.plugins.pigment.pigment_controller.PigmentController}
    @ivar path: the path of the controller to be created and navigated to
    @type path: L{basestring}
    """

    label = _('Open')

    def __init__(self, controller, path, **kwargs):
        super(OpenControllerAction, self).__init__(controller)
        self.path = path
        self._kwargs = kwargs

    def open_controller(self, path, title, **kwargs):
        """
        Attempt to create the controller with the specified L{path} and L{title}
        and then navigate to it.
        Any keyword arguments will be passed as keyword arguments to the
        controller's constructor, merged with the keyword arguments passed to
        this action's constructor.
        
        @ivar path: the path of the controller to be created and navigated to
        @type path: L{basestring}
        @ivar title: the title to be assigned to the new controller
        @type title: L{basestring}
        """
        browser = self.controller.frontend.retrieve_controllers('/poblesec/browser')[0]
        kwargs.update(self._kwargs)
        return browser.history.append_controller(path, title, **kwargs)

    def execute(self, item):
        """
        Default implementation of L{execute} that simply call L{open_controller}
        with an empty title
        
        @ivar item: the current item that is the target of the action
        @type item: L{object}
        """
        return self.open_controller(self.path, '')

class OpenControllerActionWithItem(OpenControllerAction):
    """
    Specialized implementation of L{OpenControllerAction} that pass the current
    item as the first argument to the controller's initialization function.
    """
    def execute(self, item):
        """
        Trigger the creation and opening of the controller specified by L{path}
        and pass the current item along to it.
        
        @ivar item: the current item on which to run the action
        @type item: L{object}
        """
        self._kwargs['item'] = item
        return self.open_controller(self.path, '', **self._kwargs)
    
