/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include "MRI.h"

static void CalculateColorMap (unsigned short *, double, double, double);

int
MRI_BalanceColors (MRI *mri, MRI_balance balance)
{
	unsigned short **R = mri->R;
	unsigned short **B = mri->B;
	unsigned short **G = mri->G;
	int x, y;

	for (y = 0; y < mri->height; y++)
		for (x = 0; x < mri->width; x++) {
			R[y][x] = ((int)R[y][x] * balance.rgain) >> 8;
			G[y][x] = ((int)G[y][x] * balance.ggain) >> 8;
			B[y][x] = ((int)B[y][x] * balance.bgain) >> 8;
		}
	return 1;
}

int
MRI_EnhanceColors (MRI *mri, const char *contrast, const char *gammaVal)
{
	unsigned short **R = mri->R;
	unsigned short **G = mri->G;
	unsigned short **B = mri->B;
        int height = mri->height;
	int width = mri->width;
	unsigned short map[65536];
	int count[65536];
	int	i, j, x, y;
	int maxpixel;
	int maxR, maxG, maxB, maxval;
	int minR, minG, minB, minval;
	double excl = atof (contrast) * width * height * 3.0 / 2000.0;
	double scale;
	double base;
	const char *gammaR, *gammaG, *gammaB;

	gammaR = gammaG = gammaB = gammaVal;
	maxpixel = 0;
	for (y = 0; y < height; y++)
	    for (x = 0; x < width; x++) {
		if (R[y][x] > maxpixel) maxpixel = R[y][x];
		if (G[y][x] > maxpixel) maxpixel = G[y][x];
		if (B[y][x] > maxpixel) maxpixel = B[y][x];
	    }
	maxR = maxG = maxB = 0;
	minR = minG = minB = 65535;
	for (i = 0; i < 65536; i++) count[i] = 0;
	CalculateColorMap (map, atof (gammaVal), 0.0, 1.0);
	for (y = 0; y < height; y++)
	    for (x = 0; x < width; x++) {
		unsigned short t = map[R[y][x]];
		count[t]++;
		if (t > maxR) maxR = t;
		if (t < minR) minR = t;
	    }
	CalculateColorMap (map, atof(gammaVal), 0.0, 1.0);
	for (y = 0; y < height; y++)
	    for (x = 0; x < width; x++) {
		unsigned short t = map[G[y][x]];
		count[t]++;
		if (t > maxG) maxG = t;
		if (t < minG) minG = t;
	    }
	CalculateColorMap (map, atof(gammaVal), 0.0, 1.0);
	for (y = 0; y < height; y++)
	    for (x = 0; x < width; x++) {
		unsigned short t = map[B[y][x]];
		count[t]++;
		if (t > maxB) maxB = t;
		if (t < minB) minB = t;
	    }
#ifdef DEBUG
	fprintf (stderr, "maxR = %d, maxG=%d, maxB=%d\n", maxR, maxG, maxB);
	fprintf (stderr, "minR = %d, minG=%d, minB=%d\n", minR, minG, minB);
#endif
	if (maxG > maxR)
		maxval = maxB > maxG ? maxB : maxG;
	else
		maxval = maxB > maxR ? maxB : maxR;
	if (minG < minR)
		minval = minB < minG ? minB : minG;
	else
		minval = minB < minR ? minB : minR;

	if (excl > 0.0) {
		for (i = 1, j = 0; j < excl; j += count[i++])
			;
		base = i - 1;
		for (i = 65535, j = 0; j < excl; j += count[i--])
			;
		scale = 65535.0 / (i - base + 1);
	}
	else {
		base = 0.0;
		scale = excl < 0.0 ? 1.0 : (65535.0 / maxval);
	}

	CalculateColorMap (map, atof(gammaVal), base, scale);
	for (y = 0; y < height; y++)
		for (x = 0; x < width; x++)
			R[y][x] = map[R[y][x]];
	CalculateColorMap (map, atof(gammaVal), base, scale);
	for (y = 0; y < height; y++)
		for (x = 0; x < width; x++)
			G[y][x] = map[G[y][x]];
	CalculateColorMap (map, atof(gammaVal), base, scale);
	for (y = 0; y < height; y++)
		for (x = 0; x < width; x++)
			B[y][x] = map[B[y][x]];
	return 1;
}

static void
CalculateColorMap (unsigned short *map, double gamma, double base, double scale)
{
	double g = 1.0/gamma;
	int	i;

	for (i = 0; i <= 65535; i++) {
		double v = (pow (i/65535.0, g) - base/65535.0) * scale;
		v = v * 65535.0 + 0.5;
		map[i] = (int)v;
		if (v > 65535.0) map[i] = 65535;
		if (v < 0.0) map[i] = 0;
	}
}

struct MRIEnhanceData {
	struct link *next;
	int	width;
	int	y, height;
	struct MRI_ScanLine **rows;
	unsigned short map[65536];
	int count[65536];
	int maxval;
	int minval;
	double contrast, gamma;
};

void
MRIEnhanceStart (void *private, int width, int height, int freedata)
{
	struct MRIEnhanceData *wd = private;
	int i;
	wd->width = width;
	wd->height = height;
	(*wd->next->start) (wd->next->private, width, height, freedata);
	wd->rows = malloc (sizeof (struct MRI_ScanLine *) * height);
	if (wd->rows == (struct MRI_ScanLine **)0) {
		fprintf (stderr, "MRIEnhanceStart: malloc failed\n");
		exit (1);
	}
	wd->y = 0;
	wd->minval = 65535;
	wd->maxval = 0;
	for (i = 0; i < 65536; i++) wd->count[i] = 0;
	CalculateColorMap (wd->map, wd->gamma, 0.0, 1.0);
}

void
MRIEnhanceRow (void *private, void *data)
{
	struct MRIEnhanceData *wd = private;
	struct MRI_ScanLine *sl = data;
	unsigned short *map = wd->map;
	unsigned short *R = sl->R;
	unsigned short *G = sl->G;
	unsigned short *B = sl->B;
	int x;

	wd->rows[wd->y++] = sl;
	for (x = 0; x < wd->width; x++) {
		unsigned short t;
		t = map[R[x]];
		wd->count[t]++;
		if (t > wd->maxval) wd->maxval = t;
		if (t < wd->minval) wd->minval = t;
		t = map[G[x]];
		wd->count[t]++;
		if (t > wd->maxval) wd->maxval = t;
		if (t < wd->minval) wd->minval = t;
		t = map[B[x]];
		wd->count[t]++;
		if (t > wd->maxval) wd->maxval = t;
		if (t < wd->minval) wd->minval = t;
	}
}

void
MRIEnhanceClose (void *private)
{
	struct MRIEnhanceData *wd = private;
	int i, j, x, y;
	double excl = wd->contrast * wd->width * wd->height * 3.0 / 2000.0;
	double base, scale;
	unsigned short *map = wd->map;

	if (excl > 0.0) {
		for (i = 1, j = 0; j < excl; j += wd->count[i++])
			;
		base = i - 1;
		for (i = 65535, j = 0; j < excl; j += wd->count[i--])
			;
		scale = 65535.0 / (i - base + 1);
	}
	else {
		base = 0.0;
		scale = excl < 0.0 ? 1.0 : (65535.0 / wd->maxval);
	}
	fprintf (stderr, "base=%g, scale=%g, maxval=%d\n", base, scale, wd->maxval);

	CalculateColorMap (map, wd->gamma, base, scale);
	for (y = 0; y < wd->height; y++) {
		struct MRI_ScanLine *sl = wd->rows[y];
		unsigned short *R = sl->R;
		unsigned short *G = sl->G;
		unsigned short *B = sl->B;
		for (x = 0; x < wd->width; x++) {
			R[x] = map[R[x]];
			G[x] = map[G[x]];
			B[x] = map[B[x]];
		}
		(*wd->next->row) (wd->next->private, sl);
	}
	(*wd->next->close) (wd->next->private);
	free (wd->rows);
	free (wd->next);
	free (wd);
}

struct link *
GenMRIEnhance (double contrast, const char *gamma, struct link *next)
{
	struct MRIEnhanceData *wd = malloc (sizeof (struct MRIEnhanceData));
	struct link *ep = malloc (sizeof (*ep));

	if (wd == (struct MRIEnhanceData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = MRIEnhanceStart;
	ep->row = MRIEnhanceRow;
	ep->close = MRIEnhanceClose;
	ep->private = wd;
	wd->next = next;
	wd->contrast = contrast;
	wd->gamma = atof (gamma);
	return ep;
}
