/*
 *  Copyright (c) by Shuu Yamaguchi <shuu@dotaster.com>
 *
 *  $Id: loader.c,v 3.7 2002/01/25 05:33:19 shuu Exp shuu $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 */
#include	<unistd.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<limits.h>
#include	<string.h>
#include	<stdio.h>
#include	<sys/mman.h>
#include	<errno.h>
#include	<sys/utsname.h>
#include	<sys/wait.h>

#include	"murasaki.h"
#include	"env.h"

#define	MODPROBLE_FILE "/proc/sys/kernel/modprobe"
#define	MODULE_BASEDIR	"/lib/modules/"
#define	CHAR_COMMENT	'#'

#define	FORK_OFF	0
#define	FORK_ON		1

#define	SCRIPT_MAX_ARG	3
#define CLASS_NAME_MAX	16

static char *
get_loader(void)
{
	int fd;
	char ld[PATH_MAX+1];
	ssize_t len;
	char *module_loader;

	if ((fd = open(MODPROBLE_FILE,O_RDONLY)) == -1)
		return NULL;
	if ((len = read(fd,ld,PATH_MAX)) <= 0)
		return NULL;
	close(fd);
	if (ld[len-1] == '\n')
		ld[len-1] = '\0';
	else
		ld[len] = '\0';
	if (access(ld,X_OK) != 0)
		return NULL;
	if((module_loader = strdup(ld)) == NULL)
		return NULL;
	DPRINTF(LOG_LEVEL,"loader = \"%s\"",module_loader);

	return module_loader;
}

static void
executing(char *path,char *argv[],int process)
{
	pid_t pid;

	if (process == FORK_ON) {
		pid = fork();
		switch(pid) {
		case -1:
			syslog(LOG_LEVEL,"fork error err=%d",errno);
			break;
		case 0:	/* child */
			execvp(path,argv);
			beep(INVALID);
			syslog(LOG_LEVEL,"exec error \"%s\" err=%d",path,errno);
			break;
		default:	/* parent */
			wait(NULL);
			break;
		}
	} else {
		execvp(path,argv);
		beep(INVALID);
		syslog(LOG_LEVEL,"exec error \"%s\" err=%d",path,errno);
	}
}

static void
exec_script(char *argv[],int action)
{
	int i;
	char *cmd[SCRIPT_MAX_ARG];

	if (action == ACTION_ADD)
		cmd[1] = "start";
	else 
		cmd[1] = "stop";
	cmd[2] = NULL;
	for(i=0;argv[i] != NULL;i++) {
		cmd[0] = argv[i];
		DPRINTF(LOG_LEVEL,"\"%s\" \"%s\"",cmd[0],cmd[1]);
		if (argv[i+1] == NULL) {
			executing(cmd[0],cmd,FORK_OFF);
		} else {
			executing(cmd[0],cmd,FORK_ON);
		}
	}
}

static int
get_module(char **modules,char *fname,void *config)
{
	struct stat st;
	int fd;
	char *mp;
	int count;

	if ((fd = open(fname,O_RDONLY)) == -1)
		return 0;
	if (fstat(fd,&st) == -1)
		return -1;
	if ((mp = mmap(0,st.st_size,PROT_READ,MAP_SHARED,fd,0)) == MAP_FAILED)
		return -1;
	close(fd);

	if ((count = match_config(modules,mp,mp+st.st_size,config)) == 0) {
		if (debug)
			syslog(LOG_LEVEL,"Not found in %s", fname);
	}
	munmap(mp,st.st_size);

	return count;
}


char *
make_defaultpath(char *fname)
{
	struct utsname uts;
	static char fullpath[PATH_MAX];

	if (uname(&uts) == -1)
		return NULL;
	if (sizeof(MODULE_BASEDIR)-1 + 1 + strlen(fname) + strlen(uts.release) > PATH_MAX)
		return NULL;
	strcpy(fullpath,MODULE_BASEDIR);
	strcpy(fullpath+sizeof(MODULE_BASEDIR)-1,uts.release);
	strcat(fullpath+sizeof(MODULE_BASEDIR),"/");
	strcat(fullpath+sizeof(MODULE_BASEDIR),fname);

	return fullpath;
}


int
load_module(struct mu_op *opp,int action)
{
	int i;
	char *loader;
	char *default_path;
	int count,sum;

	inform_config(opp->config);	/* print device configuration */

	if ((loader = get_loader()) == NULL) {
		syslog(LOG_LEVEL,"fork error");
		return INVALID;
	}

	/* create command */
	module_list[0] = loader;
	if (action == ACTION_ADD)
		module_list[1] = "-as";
	else 	/* ACTION_REMOVE */
		module_list[1] = "-asr";
	/*
	 * 1st: search murasaki.XXXmap (XXX:usb,pci)
	 * 2nd: search modules.XXXmap
	 */
	if ((count = get_module(module_list,opp->mappath,opp->config)) == -1)
		return INVALID;
	sum = count;
	if ((default_path = make_defaultpath(opp->default_mapname)) == NULL)
		return INVALID;
	if ((count = get_module(module_list,default_path,opp->config)) == -1)
		return INVALID;
	sum += count;
	DPRINTF(LOG_LEVEL,"module sum = %d",sum);
	if (sum == 0) {	/* not found */
		beep(INVALID);
		syslog(LOG_LEVEL,"The device match nothing in mapfile");
		syslog(LOG_LEVEL,"Please change MODULE in following line to appropriate module name, add it to %s",opp->mappath);
		print_config(opp->config);
		return INVALID;
	}
	beep(GOOD);
	/*
	 * adding depended modules & expanding alias modules 
	 */
	if (get_depend(module_list) == INVALID) {
		DPRINTF(LOG_LEVEL,"get_depend return INVALID");
		return INVALID;
	}
	/*
	 * If module name begins with "alias-", it should be expand name and
	 * be not register module_list
	 */
	for (i = START_ARG;module_list[i] != NULL;i++) {
		if (strncmp(module_list[i],ALIAS_STR,sizeof(ALIAS_STR)-1) == 0)
			remove_from_list(module_list,i);
	}
	/* get script names */
	if (get_script(script_list,module_list) == INVALID) {
		DPRINTF(LOG_LEVEL,"get_script return INVALID");
		return INVALID;
	}

	DPRINT_LIST(module_list);
	/*
	 * if module is "none", do nothing.
	 */
	sum = count_of_list(module_list);
	sum -= START_ARG;
	if (sum == 1 && strcmp(module_list[START_ARG],"none") == 0) {
		DPRINTF(LOG_LEVEL,"\"none\" -> do nothing");
		return GOOD;
	}
	/* 
	 * module_list[0]:"modprobe"
	 * module_list[1]:"-as" or "-asr"
	 * module_list[2-]: module names
	 * module_list[END]: NULL names
	 */
	if (count_of_list(script_list) != 0) {
		executing(module_list[0],module_list,FORK_ON);
		exec_script(script_list,action);	/* not come back */
	}
	executing(module_list[0],module_list,FORK_OFF);
	/* not reached */
#ifdef	KERNEL_JOB	
	free(module_name);
	for(i = START_ARG+1;module_list[i] != NULL;i++)
		free(module_list[i]);
	for(i = 0;script_list[i] != NULL;i++)
		free(script_list[i]);
#endif	/* KERNEL_JOB */

	return GOOD;
}
	
#if 0
int
exec_class(char *class,int action)
{
	char class_name[CLASS_NAME_MAX+3];	/* 3: '['+']'+'\0' */
	char *list[2];

	if (strlen(class) > CLASS_NAME_MAX)
		return INVALID;
	class_name[0] = '[';
	strcpy(class_name+1,class);
	strcat(class_name+1,"]");
	list[0] = class_name;
	list[1] = NULL;
	if (get_script(CALL_PATH,call_list,list,0,LIST_MAX) == INVALID)
		return INVALID;
	exec_script(class_list,action);	/* not come back */
}
#endif
