//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: synth.h,v 1.6 2002/02/11 10:10:57 muse Exp $
//
//  This file is derived from IIWU Synth and modified
//    for MusE.
//  Parts of IIWU are derived from Smurf Sound Font Editor.
//  Parts of Smurf Sound Font Editor are derived from
//    awesfx utilities
//  Smurf:  Copyright (C) 1999-2000 Josh Green
//  IIWU:   Copyright (C) 2001 Peter Hanappe
//  MusE:   Copyright (C) 2001 Werner Schweer
//  awesfx: Copyright (C) 1996-1999 Takashi Iwai
//=========================================================

#ifndef _IIWU_SYNTH_H
#define _IIWU_SYNTH_H

class SFont;

#define Gen_MaxValid 	Gen_Dummy - 1	/* maximum valid generator */
#define Gen_Count	Gen_Dummy	/* count of generators */
#define GenArrSize sizeof(SFGenAmount)*Gen_Count	/* gen array size */

// int sfont_preset_compare_func (const void* a, const void* b);

/* sfont file chunk sizes */
#define SFPHDRSIZE      38
#define SFBAGSIZE       4
#define SFMODSIZE       10
#define SFGENSIZE       4
#define SFIHDRSIZE	22
#define SFSHDRSIZE	46

// SFData *sfload_file(FILE*);

#define FAIL	0
#define OK	      1

enum {
      ErrWarn, ErrFatal, ErrStatus, ErrCorr, ErrEof, ErrMem, Errno,
      ErrRead, ErrWrite
      };

#include <list>

#define IIWU_NUM_CHANNELS      16
#define IIWU_NUM_PROGRAMS      129
#define IIWU_NUM_MOD           64
#define IIWU_CENTS_HZ_SIZE     1200
#define IIWU_VEL_CB_SIZE       128
#define IIWU_CB_AMP_SIZE       961
#define IIWU_PAN_SIZE          1002

#ifndef PI
#define PI                     3.141592654
#endif

/* Choose the type of interpolation */
// #define USE_TRUNCATION
// #define USE_LINEAR_INTERPOLATION
#define USE_CUBIC_INTERPOLATION


enum iiwu_loop {
      IIWU_UNLOOPED,
      IIWU_LOOP_DURING_RELEASE,
      IIWU_NOTUSED,
      IIWU_LOOP
      };

enum iiwu_mod_flags {
      IIWU_MOD_POSITIVE = 0,
      IIWU_MOD_NEGATIVE = 1,
      IIWU_MOD_UNIPOLAR = 0,
      IIWU_MOD_BIPOLAR = 2,
      IIWU_MOD_LINEAR = 0,
      IIWU_MOD_CONCAVE = 4,
      IIWU_MOD_CONVEX = 8,
      IIWU_MOD_SWITCH = 12,
      IIWU_MOD_GC = 0,
      IIWU_MOD_CC = 16
      };

enum iiwu_mod_src {
      IIWU_MOD_NONE = 0,
      IIWU_MOD_VELOCITY = 2,
      IIWU_MOD_KEY = 3,
      IIWU_MOD_KEYPRESSURE = 10,
      IIWU_MOD_CHANNELPRESSURE = 13,
      IIWU_MOD_PITCHWHEEL = 14,
      IIWU_MOD_PITCHWHEELSENS = 16
      };

//---------------------------------------------------------
//   EnvStatus
//---------------------------------------------------------

enum EnvStatus {
      ENV_SKIP,
      ENV_DELAY,
      ENV_ATTACK,
      ENV_HOLD,
      ENV_DECAY,
      ENV_SUSTAIN,
      ENV_RELEASE,
      ENV_OFF
      };

class SynthProc;
class Channel;

/***************************************************************
 *
 *         TYPE DEFINITIONS & FUNCTION DECLARATIONS
 */

#if defined(USE_TRUNCATION)

typedef char iiwu_interp_coeff_t;

#elif defined(USE_LINEAR_INTERPOLATION)
typedef struct {
      double a0, a1;
      } iiwu_interp_coeff_t;

#elif defined(USE_CUBIC_INTERPOLATION)
typedef struct {
      double a0, a1, a2, a3;
      } iiwu_interp_coeff_t;

#else
#error No interpolation defined
#endif


/*
 *  phase
 */

#define IIWU_INTERP_BITS        8
#define IIWU_INTERP_BITS_MASK   0xff000000
#define IIWU_INTERP_BITS_SHIFT  24
#define IIWU_INTERP_MAX         256

#define IIWU_FRACT_MAX ((double)4294967296.0)

//---------------------------------------------------------
//   Phase
//---------------------------------------------------------

class Phase {
      int      _index;
      unsigned _fract;

   public:
      Phase() {}
      Phase(int a)       { _index = a; _fract = 0; }
      Phase(double a)    { set(a); }

      void setInt(int a) { _index = a; _fract = 0; }
      int  index() const { return _index; }
      void dump() const  { printf("Phase %d-%ud\n", _index, _fract); }
      void set(double a) {
            _index = int(a);
            _fract = (u_int32_t) (((double)(a) - (double)(_index)) * IIWU_FRACT_MAX);
            }
      friend bool operator< (const Phase& a, const Phase& b) {
            return ((a._index < b._index) || ((a._index == b._index) && (a._fract < b._fract)));
            }
      friend bool operator== (const Phase& a, const Phase& b) {
            return ((a._index == b._index) && (a._fract == b._fract));
            }
      friend bool operator> (const Phase& a, const Phase& b) {
            return ((a._index > b._index) || ((a._index == b._index) && (a._fract > b._fract)));
            }
      void operator+=(const Phase& b) {
            _fract += b._fract;
            _index += b._index + (_fract < b._fract);
            }
      void operator-=(const Phase& b) {
            _index -= b._index - (_fract < b._fract);
            _fract -= b._fract;
            }
      int interp_index() const {
            return (int)((_fract & IIWU_INTERP_BITS_MASK) >> IIWU_INTERP_BITS_SHIFT);
            }
      };

//---------------------------------------------------------
//   Mod
//---------------------------------------------------------

class Mod {
   public:
      unsigned char dest;
      unsigned char src1;
      unsigned char flags1;
      unsigned char src2;
      unsigned char flags2;
      double amount;

   public:
      void clone(Mod* src);
      void set_source1(int src, int flags);
      void set_source2(int src, int flags);
      void set_dest(int dst) { dest = dst; }
      void set_amount(double a) { amount = a; }
      double get_value(Channel* chan, SynthProc* sp);
      };

#define iiwu_mod_has_source(mod,cc,ctrl)  \
( ((((mod)->src1 == ctrl) && (((mod)->flags1 & IIWU_MOD_CC) != 0) && (cc != 0)) \
   || ((((mod)->src1 == ctrl) && (((mod)->flags1 & IIWU_MOD_CC) == 0) && (cc == 0)))) \
|| ((((mod)->src2 == ctrl) && (((mod)->flags2 & IIWU_MOD_CC) != 0) && (cc != 0)) \
    || ((((mod)->src2 == ctrl) && (((mod)->flags2 & IIWU_MOD_CC) == 0) && (cc == 0)))))


class SFont;
class ISynth;
class Preset;
class Sample;

//---------------------------------------------------------
//   Channel
//---------------------------------------------------------

class Channel {
      char _chan;
      char _hbank;
      char _lbank;
      char _prog;
      Preset* _preset;
      int _pitch_bend;
      char _cc[128];

   public:
      int key_pressure;
      int channel_pressure;
      int pitch_wheel_sensitivity;

   public:
      Channel();
      ~Channel() {}
      void setChannum(int num);

      const Preset* preset() const        { return _preset; }
      void setPreset(Preset* val)         { _preset = val;   }

      char chan() const                   { return _chan; }
      void setChan(char c)                { _chan = c; }

      char hbank() const                  { return _hbank; }
      char lbank() const                  { return _lbank; }
      char prog() const                   { return _prog; }
      void setHbank(char val)             { _hbank = val;  }
      void setLbank(char val)             { _lbank = val;  }
      void setProg(char val)              { _prog = val;  }

      void noteon(ISynth* synth, int key, int vel);
      int pitchBend() const               { return _pitch_bend; }
      void setPitchBend(int val)          { _pitch_bend = val;  }
      int cc(int num) const               { return _cc[num];    }
      void setCc(int num, int value)      { _cc[num] = value;   }
      };

//---------------------------------------------------------
//   SynthProc
//    synthesis process
//---------------------------------------------------------

class SynthProc {
      double _pitch;                // the pitch in midicents
      ISynth* synth;
      Sample* _sample;
      bool _sustained;

   public:
      SynthProc* next;
      unsigned char key;            // the key, quick acces for noteoff
      unsigned char vel;            // the velocity
      Channel* channel;

      Gen gen[GEN_LAST];
      Mod mod[IIWU_NUM_MOD];

      int mod_count;
      unsigned int ticks;
      Phase phase;                  // the phase of the sample wave

      /* basic parameters */
      double attenuation;           // the attenuation in centibels
      double root_pitch;

      /* sample and loop start and end points */
      int start_offset;
      int end_offset;
      int loop_start_offset;
      int loop_end_offset;

      /* vol env */
      EnvStatus volenv_state;       // the state of the volume envelope
      double volenv_val;            // the value of the volume envelope
      unsigned int volenv_delay;    // duration delay phase in ticks
      double volenv_attack;
      double volenv_hold;           // ticks
      double volenv_decay;
      double volenv_sustain;
      double volenv_release;

      /* mod env */
      EnvStatus modenv_state;       // the state of the modulation envelope */
      double modenv_val;             /* the value of the modulation envelope */
      unsigned int modenv_delay;
      double modenv_attack;
      double modenv_hold;
      double modenv_decay;
      double modenv_sustain;
      double modenv_release;
      double modenv_to_fc;
      double modenv_to_pitch;
      /* mod lfo */
      double modlfo_val;             /* the value of the modulation LFO */
      unsigned int modlfo_delay;        /* the delay of the lfo in samples */
      double modlfo_incr;            /* the lfo frequency is converted to a per-buffer increment */
      double modlfo_to_fc;
      double modlfo_to_pitch;
      double modlfo_to_vol;
      /* vib lfo */
      double viblfo_val;             /* the value of the vibrato LFO */
      unsigned int viblfo_delay;           /* the delay of the lfo in samples */
      double viblfo_incr;            /* the lfo frequency is converted to a per-buffer increment */
      double viblfo_to_pitch;
      /* for the resonant filter */
      double fres;                   /* the resonance frequency, in cents (not absolute cents) */
      double q;                      /* the q-factor */
      double w1, w2;                 /* delayed samples */
      /* pan */
      double pan;
      /* for debugging purposeses */
      void update_param(int gen);

   public:
      SynthProc(ISynth*);
      ~SynthProc() {}
      void init(Channel* channel, Sample* sample);
      void write(int len, float* mono, float* left, float* right);
      void modulate(int cc, int ctrl, int val);
      void noteoff();
      void off();
      void optimize();
      void add_mod(Mod* mod);
      double pitch() const          { return _pitch;     }
      int chan() const              { return channel->chan(); }
      bool sustained() const        { return _sustained; }
      void setSustained(bool val)   { _sustained = val;  }
      bool released() const         { return volenv_state == ENV_RELEASE; }
      };

#define _SAMPLEMODE()   ((int)(gen[GEN_SAMPLEMODE].val))
#define _GEN(_n)        (gen[_n].val + gen[_n].mod)

//---------------------------------------------------------
//   ISynth
//---------------------------------------------------------

class ISynth : public Mess {
      static bool initialized;

      enum SfOp { SF_REPLACE, SF_ADD, SF_REMOVE };

      bool _gmMode;
      int _masterVol;         // 0 - 0x3fff
      double startFrame;      // synthi start time
      unsigned curSample;

      int maxBufferSize;
      mutable SFont* nextSFont;       // for getNextParam

      void noteon(int chan, int key, int vel);
      void noteoff(int chan, int key);
      void sysex(const unsigned char* data, int len);
      void gmOn(bool);
      void sysexSoundFont(SfOp op, const char* data);
      void setMasterVol(int val)   { _masterVol = val;  }

      void program_change(int chan, char program);
      void hbank_select(int chan, char bank);
      void lbank_select(int chan, char bank);
      void setCc(int chan, int ctrl, int val);
      void pitch_bend(int chan, int val);
      void deleteSFonts();
      void allNotesOff();
      void resetAllController(int);

      const MidiPatch* getFirstPatch(int ch) const;
      static void initSynth();
      virtual void write(int n, float** ports, int offset);

   protected:
      virtual void processEvent(MEvent*);

   public:
      SynthProc* alloc_sp(Channel* chan, int key, int vel);
      SFont* sfont;               // the loaded soundfont
      Channel channel[IIWU_NUM_CHANNELS]; // the channels

      SynthProc* spFree;   // the free synthesis processes
      SynthProc* spBusy;   // the busy synthesis processes

      float* mono_buf;

      //----threads:
      pthread_t midiThread;
      void midiRun();

   public:
      ISynth(const char* classname);
      ~ISynth();

      void stop(SynthProc*);
      void start(SynthProc*);

      int masterVol() const        { return _masterVol; }
      bool init();

      /** Load a SoundFont. The newly loaded SoundFont will be put on top
       * of the SoundFont stack. Presets are searched starting from the
       * SoundFont on the top of the stack, working the way down the stack
       * until a preset is found.  */
      int sfload(const char* filename);

      /** Get the name of the next preset of a SoundFont */
      char* next_preset(int font);
      void program_reset();
      SFont* get_font(int num);

      void dump();      // debugging

      //
      //   experimental MESSS interface:
      //
      virtual const char* getPatchName(int, int, int, int, MType);
//      virtual const MidiPatch* getFirstPatch(int ch) const;
      virtual const MidiPatch* getNextPatch(int ch, const MidiPatch*) const;
      virtual bool getFirstParameter(const char*& name, const char*& value) const;
      virtual bool getNextParameter(const char*& name, const char*& value) const;
      virtual void setParameter(const char* name, const char* value);
      };

#endif  /* _IIWU_SYNTH_H */
