#include <stdio.h>
#include "bigtime.h"
#include "globals.h"
#include <qpalette.h>
#include <qlabel.h>
#include <qcolor.h>
#include <qlayout.h>

#include "song.h"
#include "app.h"

extern int mtcType;

//
// the bigtime widget
// display is split into several parts to avoid flickering.
//

//---------------------------------------------------------
//   BigTime
//---------------------------------------------------------

BigTime::BigTime(QWidget* parent, MusE* se) :
	QWidget(parent, 0, WType_TopLevel | WStyle_StaysOnTop)
      {
      seq = se;

      seq->setBigTimeChecked(true); // dont know why i have to do this...

      barLabel      = new QLabel(this);
      beatLabel     = new QLabel(this);
      tickLabel     = new QLabel(this);
      hourLabel     = new QLabel(this);
      minLabel      = new QLabel(this);
      secLabel      = new QLabel(this);
      frameLabel    = new QLabel(this);

      sep1 = new QLabel(this);
      sep2 = new QLabel(this);
      sep3 = new QLabel(this);
      sep4 = new QLabel(this);
      sep5 = new QLabel(this);

      sep1->setText(".");
      sep2->setText(".");
      sep3->setText(":");
      sep4->setText(":");
      sep5->setText(":");

      oldbar = oldbeat = oldtick = oldhour = oldmin = oldsec = oldframe = -1;

      setString(-1);

      setBgColor(black);
      setFgColor(red);
      resize(320, 100);
      QFont f("Courier");
      setFont(f);
      setCaption("muse: Bigtime");

      }

//---------------------------------------------------------
//   setString
//---------------------------------------------------------

bool BigTime::setString(int v)
      {
      if (v < 0) {
            barLabel->setText("----");
            beatLabel->setText("--");
            tickLabel->setText("---");
	    hourLabel->setText("--");
            minLabel->setText("--");
            secLabel->setText("--");
            frameLabel->setText("--");
	    oldbar = oldbeat = oldtick = oldhour = oldmin = oldsec = oldframe = -1;
            return true;
            }

      int bar, beat, tick;
      sigmap.tickValues(v, &bar, &beat, &tick);

      double time = tempomap.tick2time(v);
      int hour = int(time) / 3600;
      int min  = (int(time) / 60) % 60;
      int sec  = int(time) % 60;
      double rest = time - (min * 60 + sec);
      switch(mtcType) {
          case 0:     // 24 frames sec
                rest *= 24;
                break;
          case 1:     // 25
                rest *= 25;
                break;
          case 2:     // 30 drop frame
                rest *= 30;
                break;
          case 3:     // 30 non drop frame
                rest *= 30;
                break;
      }
      int frame = int(rest);
//      int subframe = int((rest-frame)*100);

      QString s;

      if(oldbar != bar) {
	      s.sprintf("%04d", bar+1);
	      barLabel->setText(s);
	      oldbar = bar;
      }

      if(oldbeat != beat) {
	      s.sprintf("%02d", beat+1);
	      beatLabel->setText(s);
	      oldbeat = beat;
      }

      if(oldtick != tick) {
	      s.sprintf("%03d", tick);
	      tickLabel->setText(s);
	      oldtick = tick;
      }

      if(oldhour != hour) {
	      s.sprintf("%02d", hour);
	      hourLabel->setText(s);
	      oldhour = hour;
      }

      if(oldmin != min) {
	      s.sprintf("%02d", min);
	      minLabel->setText(s);
	      oldmin = min;
      }

      if(oldsec != sec) {
	      s.sprintf("%02d", sec);
	      secLabel->setText(s);
	      oldsec = sec;
      }

      if(oldframe != frame) {
	      s.sprintf("%02d", frame);
	      frameLabel->setText(s);
	      oldframe = frame;
      }


      return false;
      }

//---------------------------------------------------------
//   setPos
//---------------------------------------------------------

void BigTime::setPos(int idx, int v, bool)
      {
      if(idx == 0)
      		setString(v);
      	
      }

//---------------------------------------------------------
//   resizeEvent
//---------------------------------------------------------

void BigTime::resizeEvent(QResizeEvent *ev)
      {
	
	QFont f = font();
	int fs = f.pixelSize();
	int hspace = 20;

	while(fontMetrics().width("00:00:00:00") < (ev->size().width() - hspace*2))
	      {
	        fs+=2;
		f.setPixelSize(fs);
		setFont(f);
	      }

	while(fontMetrics().width("00:00:00:00") > (ev->size().width() - hspace*2))
              {
                fs-=2;
		f.setPixelSize(fs);
		setFont(f);
	      }


	// check widget height
/*
	if( ev->size().height() < fs*3.5 )
		{
		  resize( ev->size().width(), fs*3.5 );
		  return;
		}
*/

	int digitWidth = fontMetrics().width("0");
	
	int vspace = (ev->size().height() - (fs*2)) / 3;
	int tickY = vspace;
	int timeY = vspace*2 + fs;

	barLabel->resize(digitWidth*4, fs);
	beatLabel->resize(digitWidth*2, fs);
	tickLabel->resize(digitWidth*3, fs);
	hourLabel->resize(digitWidth*2, fs);
	minLabel->resize(digitWidth*2, fs);
	secLabel->resize(digitWidth*2, fs);
	frameLabel->resize(digitWidth*2, fs);

	sep1->resize(digitWidth, fs);
	sep2->resize(digitWidth, fs);
	sep3->resize(digitWidth, fs);
	sep4->resize(digitWidth, fs);
	sep5->resize(digitWidth, fs);
	
	barLabel->move(		hspace + (digitWidth*0), tickY);
	sep1->move(		hspace + (digitWidth*4), tickY);
	beatLabel->move(	hspace + (digitWidth*5), tickY);
	sep2->move(		hspace + (digitWidth*7), tickY);
	tickLabel->move(	hspace + (digitWidth*8), tickY);

	hourLabel->move(	hspace + (digitWidth*0), timeY);
	sep3->move(		hspace + (digitWidth*2), timeY);
	minLabel->move(		hspace + (digitWidth*3), timeY);
	sep4->move(		hspace + (digitWidth*5), timeY);
	secLabel->move(		hspace + (digitWidth*6), timeY);
	sep5->move(		hspace + (digitWidth*8), timeY);
	frameLabel->move(	hspace + (digitWidth*9), timeY);
      }

//---------------------------------------------------------
//   closeEvent
//---------------------------------------------------------

void BigTime::closeEvent(QCloseEvent *ev)
      {
        seq->setBigTimeChecked(false);
        QWidget::closeEvent(ev);
      }

//---------------------------------------------------------
//   setForegroundColor
//---------------------------------------------------------

void BigTime::setFgColor(QColor c)
      {
        QColorGroup cg = palette().active();
        cg.setColor(QColorGroup::Foreground, c);
        QPalette newpalette(cg, cg, cg);
        setPalette(newpalette);

        barLabel->setPalette(newpalette);
        beatLabel->setPalette(newpalette);
        tickLabel->setPalette(newpalette);
        hourLabel->setPalette(newpalette);
        minLabel->setPalette(newpalette);
        secLabel->setPalette(newpalette);
        frameLabel->setPalette(newpalette);

        sep1->setPalette(newpalette);
        sep2->setPalette(newpalette);
        sep3->setPalette(newpalette);
        sep4->setPalette(newpalette);
        sep5->setPalette(newpalette);
      }

//---------------------------------------------------------
//   setBackgroundColor
//---------------------------------------------------------

void BigTime::setBgColor(QColor c)
      {
        QColorGroup cg = palette().active();
        cg.setColor(QColorGroup::Background, c);
        QPalette newpalette(cg, cg, cg);
        setPalette(newpalette);

        barLabel->setPalette(newpalette);
        beatLabel->setPalette(newpalette);
        tickLabel->setPalette(newpalette);
        hourLabel->setPalette(newpalette);
        minLabel->setPalette(newpalette);
        secLabel->setPalette(newpalette);
        frameLabel->setPalette(newpalette);

        sep1->setPalette(newpalette);
        sep2->setPalette(newpalette);
        sep3->setPalette(newpalette);
        sep4->setPalette(newpalette);
        sep5->setPalette(newpalette);

	setBackgroundColor(c);
      }

//---------------------------------------------------------
//   getForegroundColor
//---------------------------------------------------------

QColor BigTime::getFgColor()
      {
      	return sep1->palette().active().foreground();
      }

//---------------------------------------------------------
//   getBackgroundColor
//---------------------------------------------------------

QColor BigTime::getBgColor()
      {
      	return sep1->palette().active().background();
      }
