/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_utilites.cpp
 * @brief Some common utility functions.
 * 
 */

#include "myx_gc.h"

#ifdef _WINDOWS
  #include <direct.h>
#else
  #include <stdlib.h>
  #define MAX_PATH PATH_MAX
#endif // ifdef _WINDOWS

#include <errno.h>
#include <glib.h>

#include "myx_gc_utilities.h"

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Converts the given string, which is supposed to be an UTF-8 encoded text into an UTF-16 string.
 * @param Source Contains the source string encoded in UTF-8.
 * @return The converted string in UTF-16 encoding.
 */
wstring Utf8ToUtf16(const string& Source)
{
  wstring Result;

#ifdef _WINDOWS
  const char* Raw = Source.c_str();
  int Size = MultiByteToWideChar(CP_UTF8, 0, Raw, -1, NULL, 0);
  wchar_t* Buffer = new wchar_t[Size];
  MultiByteToWideChar(CP_UTF8, 0, Raw, -1, Buffer, Size);
  Result += Buffer;
  delete[] Buffer;
#else
  gunichar2 *tmp;
  tmp= g_utf8_to_utf16(Source.c_str(), Source.length(), NULL, NULL, NULL);
  Result= (wchar_t*)tmp;
  g_free(tmp);
#endif

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Converts the given string into an ANSI string.
 * @param Source Contains the source string encoded in UTF-16.
 * @return The converted string in ANSI encoding.
 * @note The current user locale is used to convert the Unicode string to ANSI.
 */
string Utf16ToANSI(const wstring& Source)
{
  string Result;

#ifdef _WINDOWS
  const wchar_t* Raw = Source.c_str();
  int Size = WideCharToMultiByte(CP_ACP, 0, Raw, -1, NULL, 0, NULL, NULL);
  char* Buffer = new char[Size];
  WideCharToMultiByte(CP_ACP, 0, Raw, -1, Buffer, Size, NULL, NULL);
  Result += Buffer;
  delete[] Buffer;
#else
  gchar *tmp;
  tmp= g_utf16_to_utf8((gunichar2*)Source.c_str(), Source.length(), NULL, NULL, NULL);
  Result= tmp;
  g_free(tmp);
#endif

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Converts the given string, which is supposed to be an UTF-8 encoded text into an ANSI string.
 * @param Source Contains the source string encoded in UTF-8.
 * @return The converted string in ANSI encoding.
 * @note The current user locale is used to convert the Unicode string to ANSI.
 */
string Utf8ToANSI(const string& Source)
{
#ifdef _WINDOWS
  return Utf16ToANSI(Utf8ToUtf16(Source));
#else
  return Source;
#endif
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief User defined file handling callback for handling file i/o in png lib.
 *        We have to make it so because of the FILE allocation and png being in a DLL on Windows.
 */
void PNGRead(png_structp png_ptr, png_bytep data, png_size_t length)
{
  fread(data, length, 1, (FILE*) png_ptr->io_ptr);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Loads a png file.
 * @param Filename An ANSI encoded file name (can contain path info) to the png file.
 * @return Returns a pointer to a TImage structure containing the image data.
 * @note The return memory must be freed using FreeImage().
 */
TImage* LoadPNG(const string& Filename)
{
  TImage *Image = NULL;
  FILE* File;
  png_structp PNG;
  png_infop Info;
  png_uint_32 Width, Height;
  int depth, junk, color_type;
  int Offset;

  File = fopen(Filename.c_str(), "rb");
  try
  {
    if (File) 
    {
      png_byte Magic[8];
      if (fread(Magic, sizeof(Magic), 1, File) != 1)
      {
        fprintf(stderr, "can't read '%s': %s\n", Filename.c_str(), strerror(errno));
        fclose(File);
        return NULL;
      };
      if (!png_check_sig(Magic, sizeof(Magic)))
      {
        fclose(File);
        return NULL;
      };

      PNG = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
      if (!PNG)
      {
        fclose(File);
        return NULL;
      };
      
      Info = png_create_info_struct(PNG);
      if (!Info) 
      {
        fclose(File);
        png_destroy_read_struct(&PNG, NULL, NULL);
        return NULL;
      };

      png_set_read_fn(PNG, File, &PNGRead);

      png_init_io(PNG, File);
      png_set_sig_bytes(PNG, sizeof(Magic));

      png_read_info(PNG, Info);

      png_get_IHDR(PNG, Info, &Width, &Height, &depth, &color_type,
                  &junk, &junk, &junk);

      /* sanity check */
      if (Width < 1 || Height < 1) 
      {
        fclose(File);
        png_destroy_read_struct(&PNG, &Info, NULL);
        return NULL;
      }

      Image = (TImage*) malloc(sizeof(TImage));
      if (!Image)
      {
        fclose(File);
        png_destroy_read_struct(&PNG, &Info, NULL);
        return NULL;
      };

      Image->Data = (unsigned char*) malloc(Width * Height * Info->channels);
      if (!Image->Data) 
      {
        fclose(File);
        png_destroy_read_struct(&PNG, &Info, NULL);
        return NULL;
      };

      Image->Width = Width;
      Image->Height = Height;
      Image->ColorType = (TColorType) Info->color_type;
      Image->Channels = Info->channels;

      // normalize to 8bpp with alpha channel
      if (color_type == PNG_COLOR_TYPE_PALETTE && depth <= 8)
        png_set_expand(PNG);

      if (color_type == PNG_COLOR_TYPE_GRAY && depth <= 8)
        png_set_expand(PNG);

      if (png_get_valid(PNG, Info, PNG_INFO_tRNS))
        png_set_expand(PNG);

      if (depth == 16)
        png_set_strip_16(PNG);

      if (color_type == PNG_COLOR_TYPE_GRAY ||
          color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
        png_set_gray_to_rgb(PNG);

      /* do the transforms */
      png_read_update_info(PNG, Info);

      Offset= 0;
      /* read Data */
      for (unsigned int I = 0; I < Height; I++)
      {
        png_read_row(PNG, &Image->Data[Offset], NULL);
        Offset += Info->channels * Width;
      }
      png_read_end(PNG, Info);
      png_destroy_read_struct(&PNG, &Info, NULL);

      fclose(File);
    };
  }
  catch (...)
  {
    if (Image)
      FreeImage(Image);
    Image = NULL;
  };

  return Image;
}

//----------------------------------------------------------------------------------------------------------------------

void FreeImage(TImage* Image)
{
  free(Image->Data);
  free(Image);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Returns the current working folder (ANSI encoded).
 * @return The current working folder.
 */
string GetCurrentDir(void)
{
  char Buffer[MAX_PATH];

  getcwd(Buffer, sizeof(Buffer));

  return string(Buffer);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Sets the current working folder (folder name must be ANSI encoded).
 * @param Folder The new folder to be set.
 */
void SetCurrentDir(const string& Folder)
{
  chdir(Folder.c_str());
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief ExtractFilePath extracts the drive and directory parts of the given filename. The resulting string is the 
          leftmost characters of FileName, up to and including the colon or backslash that separates the path information 
          from the name and extension. The resulting string is empty if FileName contains no drive and directory parts.
 * @param Filename The file name (ANSI encoded) of which the path is to be extracted.
 * @return The extracted path part (ANSI encoded).
 */
string ExtractFilePath(const string& Filename)
{
  gchar* Raw = g_path_get_dirname(Filename.c_str());

  string Result(Raw);
  g_free(Raw);

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Platform neutral file open routine.
 * @param Filename The name of file encode in UTF-8.
 * @param OpenMode The mode how to open the file (the same as used for fopen calls).
 * @return A pointer to a FILE structure if the file could be opened, NULL otherwise.
 */
FILE* OpenFile(string Filename, const char* OpenMode)
{
#ifdef _WINDOWS
  wstring Name = Utf8ToUtf16(Filename);
  wstring Mode = Utf8ToUtf16(string(OpenMode));
  return _wfopen(Name.c_str(), Mode.c_str());
#else
  FILE *file;
  char * local_filename;

  if (!(local_filename = g_filename_from_utf8(Filename.c_str(), -1, NULL, NULL, NULL)))
    return NULL;

  file = fopen(local_filename, OpenMode);

  g_free(local_filename);

  return file;
#endif // #ifdef _WINDOWS
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method to sort left/right and bottom/top coordinates so that for left/bottom are always smaller than right/top.
 */
TBounds SortBounds(const TBounds& Bounds)
{
  TBounds Result;

  Result.Left = (Bounds.Left <= Bounds.Right) ? Bounds.Left : Bounds.Right;
  Result.Right = (Bounds.Left <= Bounds.Right) ? Bounds.Right : Bounds.Left;
  Result.Top = (Bounds.Top <= Bounds.Bottom) ? Bounds.Top : Bounds.Bottom;
  Result.Bottom = (Bounds.Top <= Bounds.Bottom) ? Bounds.Bottom : Bounds.Top;

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Determines whether both bounds overlap. 
 *
 * @param Bounds1 One of the bounds to compare.
 * @param Bounds2 The other bounds to compare.
 * @return True if both bounds overlap each other, otherwise false.
 */
bool BoundsIntersect(const TBounds& Bounds1, const TBounds& Bounds2)
{
  return (Bounds1.Right > Bounds2.Left) && (Bounds1.Left < Bounds2.Right) && 
    (Bounds1.Bottom > Bounds2.Top) && (Bounds1.Top < Bounds2.Bottom);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Examines the given bounds and returns whether it is empty or not.
 *
 * @param Bounds The bounds to examine.
 * @return True if the bounds are empty, false otherwise.
 */
bool BoundsAreEmpty(const TBounds& Bounds)
{
  return (Bounds.Left == Bounds.Right) && (Bounds.Top == Bounds.Bottom);
}

//----------------- CBoundingBoxComputer -------------------------------------------------------------------------------

CBoundingBoxComputer::CBoundingBoxComputer(void)
{
  // The bound box initializes itself to all zero.
}

//----------------------------------------------------------------------------------------------------------------------

CBoundingBoxComputer::CBoundingBoxComputer(const TBoundingBox& InitialBox)
{
  FCurrentBox = InitialBox;
}

//----------------------------------------------------------------------------------------------------------------------

void CBoundingBoxComputer::IncludeVertex(const TVertex& Vertex)
{
  if (Vertex.X < FCurrentBox.UpperLeftX)
    FCurrentBox.UpperLeftX = Vertex.X;
  if (Vertex.X > FCurrentBox.LowerRightX)
    FCurrentBox.LowerRightX = Vertex.X;
  if (Vertex.Y < FCurrentBox.UpperLeftY)
    FCurrentBox.UpperLeftY = Vertex.Y;
  if (Vertex.Y > FCurrentBox.LowerRightY)
    FCurrentBox.LowerRightY = Vertex.Y;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Transforms the given coordinate using the provided matrix.
 *
 * @param Matrix The matrix with which to do the transformation. It is a usual 4x4 matrix, although only the upper-left 
 *               2x2 entries are used.
 * @param X The X coordinate.
 * @param Y The Y coordinate.
 * @return The transformed vertex.
 */
TVertex CBoundingBoxComputer::Transform(GLfloat* Matrix, const float& X, const float& Y)
{
  TVertex Result;

  Result.X = X * Matrix[0] + Y * Matrix[4];
  Result.Y = X * Matrix[1] + Y * Matrix[5];

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current bounding box.
 *
 * @return The current bounding box.
 */
TBoundingBox CBoundingBoxComputer::BoundingBox()
{
  return FCurrentBox;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the new box and merges it with the current bounding box. The new data is not required to be ordered.
 *
 * @param NewBox A new bounding box to merge in.
 */
void CBoundingBoxComputer::Include(GLfloat* Matrix, TBoundingBox* NewBox)
{
  float LLX;
  float URX;
  if (NewBox->UpperLeftX < NewBox->LowerRightX)
  {
    LLX = NewBox->UpperLeftX;
    URX = NewBox->LowerRightX;
  }
  else
  {
    LLX = NewBox->LowerRightX;
    URX = NewBox->UpperLeftX;
  };

  float LLY;
  float URY;
  if (NewBox->UpperLeftY < NewBox->LowerRightY)
  {
    LLY = NewBox->UpperLeftY;
    URY = NewBox->LowerRightY;
  }
  else
  {
    LLY = NewBox->LowerRightY;
    URY = NewBox->UpperLeftY;
  };

  float LLZ;
  float URZ;
  if (NewBox->UpperLeftZ < NewBox->LowerRightZ)
  {
    LLZ = NewBox->UpperLeftZ;
    URZ = NewBox->LowerRightZ;
  }
  else
  {
    LLZ = NewBox->LowerRightZ;
    URZ = NewBox->UpperLeftZ;
  };

  TVertex V;
  if (Matrix != NULL)
    V = Transform(Matrix, LLX, LLY);
  else
  {
    V.X = LLX;
    V.Y = LLY;
  };
  IncludeVertex(V);
  if (Matrix != NULL)
    V = Transform(Matrix, URX, URY);
  else
  {
    V.X = URX;
    V.Y = URY;
  };
  IncludeVertex(V);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the vertex and merges it with the current bounding box.
 *
 * @param Vertex The vertex to be included
 */
void CBoundingBoxComputer::Include(GLfloat* Matrix, const TVertex& Vertex)
{
  TVertex V;
  if (Matrix != NULL)
    V = Transform(Matrix, Vertex.X, Vertex.Y);
  else
    V = Vertex;
  IncludeVertex(V);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the X and Y values and merges them with the current bounding box.
 *
 * @param X The X coordinate of the point to include.
 * @param Y The Y coordinate of the point to include.
 */
void CBoundingBoxComputer::Include(GLfloat* Matrix, const float& X, const float& Y)
{
  TVertex V;
  if (Matrix != NULL)
    V = Transform(Matrix, X, Y);
  else
  {
    V.X = X;
    V.Y = Y;
  };
  IncludeVertex(V);
}

//----------------------------------------------------------------------------------------------------------------------

