/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <netdude/nd_debug.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_protocol.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_raw_protocol.h>
#include <netdude/nd.h>
#include <protocols/snap/nd_proto_snap.h>

#include <nd_fddi.h>
#include <nd_fddi_callbacks.h>

#include <net/bpf.h>

static ND_ProtoField fddi_fields[] = {
  { ND_VAL_FIELD, N_("Frame\nCtrl.\n(0x%.2x)"), N_("Frame Control"),    8,  nd_fddi_fc_cb    },
  { ND_VAL_FIELD, N_("Dst. addr.\n(%s)\n"),     N_("Destination host"), 48, nd_fddi_dhost_cb },
  { ND_VAL_FIELD, N_("Src. addr.\n(%s)\n"),     N_("Source host"),      48, nd_fddi_shost_cb },
  { 0, NULL, NULL, 0, NULL }
};

static ND_Protocol *fddi;

static gboolean
fddi_header_complete(const ND_Packet *packet, guchar *data)
{
  return (data + 13 <= nd_packet_get_end(packet));
}


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return _("FDDI Plugin");
}


const char *
description(void)
{
  return _("A plugin providing support for the FDDI protocol.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return ("0.1.0");
}


ND_Protocol *
init(void)
{
  fddi = nd_proto_new("FDDI", ND_PROTO_LAYER_LINK, DLT_FDDI);
			     		
  fddi->create_gui      = nd_fddi_create_gui;
  fddi->set_gui         = nd_fddi_set_gui;
  fddi->init_packet     = nd_fddi_init_packet;
  fddi->header_complete = nd_fddi_header_complete;

  /* This one isn't really necessary, just for illustration */
  fddi->is_stateful = FALSE;

  /* We're using a button table to display the protocol content,
     so we need to hook them in here: */
  fddi->fields       = fddi_fields;
  fddi->header_width = 104;
	     
  return fddi;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_fddi_create_gui(ND_Trace *trace, ND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);

  return table;
}


void       
nd_fddi_set_gui(const ND_Packet *packet, ND_ProtoInfo *pinf)
{
  struct fddi_header *fh;
  
  fh = (struct fddi_header*) nd_packet_get_data(packet, fddi, pinf->inst.nesting);

  nd_fddi_set_gui_fc(pinf, fh);
  nd_fddi_set_gui_dhost(pinf, fh);
  nd_fddi_set_gui_shost(pinf, fh);
}


void       
nd_fddi_init_packet(ND_Packet *packet, guchar *data, guchar *data_end)
{
  ND_Protocol *payload_proto;
  struct fddi_header *fh;

  if (!fddi_header_complete(packet, data))
    {
      nd_raw_proto_get()->init_packet(packet, data, data_end);
      return;
    }

  nd_packet_add_proto_data(packet, fddi, data, data_end);  
  
  /* Cast the data pointer into your protocol's header */
  fh = (struct fddi_header *) data;

  /* Check the appriopriate header field value to demultiplex
     packet initialization up to the next correct protocol: */

  if ((fh->fddi_fc >= 0x50 && fh->fddi_fc <= 0x5F) ||
      (fh->fddi_fc >= 0xD0 && fh->fddi_fc <= 0xD7))
    {
      /* This is an LLC frame. Pass on to the LLX/SNAP implementation: */
      payload_proto = nd_proto_registry_find(ND_PROTO_LAYER_LINK, ND_PROTO_SNAP);
      payload_proto->init_packet(packet, data + 13, data_end);

      return;
    }

  /* Otherwise, we don't handle this yet. Sorry. */
}


gboolean
nd_fddi_header_complete(const ND_Packet *packet, guint nesting)
{
  guchar      *data;

  if (!packet)
    return FALSE;

  data = nd_packet_get_data(packet, fddi, nesting);

  return fddi_header_complete(packet, data);
  TOUCH(nesting);
}


/* Misc helper stuff below --------------------------------------------- */

void
nd_fddi_set_gui_fc(ND_ProtoInfo *pinf, struct fddi_header *fh)
{
  nd_proto_field_set(pinf, &fddi_fields[0], DATA_TO_PTR(fh->fddi_fc));
}


void
nd_fddi_set_gui_dhost(ND_ProtoInfo *pinf, struct fddi_header *fh)
{
  char s[MAXPATHLEN];
  
  g_snprintf(s, MAXPATHLEN, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x",
	     fh->fddi_dhost[0], fh->fddi_dhost[1],
	     fh->fddi_dhost[2], fh->fddi_dhost[3],
	     fh->fddi_dhost[4], fh->fddi_dhost[5]);

  nd_proto_field_set(pinf, &fddi_fields[1], s);
}


void
nd_fddi_set_gui_shost(ND_ProtoInfo *pinf, struct fddi_header *fh)
{
  char s[MAXPATHLEN];

  g_snprintf(s, MAXPATHLEN, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x",
	     fh->fddi_shost[0], fh->fddi_shost[1],
	     fh->fddi_shost[2], fh->fddi_shost[3],
	     fh->fddi_shost[4], fh->fddi_shost[5]);

  nd_proto_field_set(pinf, &fddi_fields[2], s);
}


ND_Protocol *
nd_fddi_get(void)
{
  return fddi;
}

