/*  VER 079  TAB P   $Id: logmsg.c,v 1.10 1999/03/31 03:38:28 src Exp $
 *
 *  handle error messages and such...
 *
 *  copyright 1996, 1997 Egil Kvaleberg, egil@kvaleberg.no
 *  the GNU General Public License applies
 *
 *  $Log: logmsg.c,v $
 *  Revision 1.10  1999/03/31 03:38:28  src
 *  Moved errno.h to common.h
 *
 *  Revision 1.9  1998/10/25 04:08:56  src
 *  A little more debugging on lockfiles.
 *
 *  Revision 1.8  1998/09/09 07:32:12  src
 *  Version 1.1
 *
 *  Revision 1.7  1998/09/03 02:49:30  src
 *  Fixed stuff detected by -Wall
 *
 *  Revision 1.6  1998/07/12 09:39:28  src
 *  newsx version 1.0
 */

#include "common.h"
#include "proto.h"
#include "options.h"

#if WITHOUT_ERRLOG
#  undef HAVE_SYSLOG_H
#endif
#if HAVE_SYSLOG_H
#  include <syslog.h>
#endif

static int line_not_clean;

/*
 *  NOTE: we conveniently avoid including "proto.h"
 *        here, so that it doesn't crash with the log_msg()
 *        definition if we must use varargs.h
 */
#if HAVE_VPRINTF
#  if HAVE_STDARG_H
#    include <stdarg.h>
#  else
#    if HAVE_VARARGS_H
#      include <varargs.h>
#      define USE_VA_ALIST 1
#    else
#      undef HAVE_VPRINTF /* we don't know how to use it anyway */
#    endif
#  endif
#endif

#ifndef HAVE_VPRINTF
    /* 
     *  try to make a surrogate 
     *  we assume that on those architectures where this trick
     *  doesn't work there we will surely have stdarg.h or varargs.h
     */
#define vsprintf(buf,  fmt, ap) sprintf(buf,  fmt, arg1, arg2, arg3, arg4)
#define vfprintf(file, fmt, ap) fprintf(file, fmt, arg1, arg2, arg3, arg4)
#endif

/* 
 *  open log facility
 */     
void 
log_open(void)
{
    /* open syslog if required */
#if HAVE_SYSLOG_H
#   ifndef WITH_ERRLOG
#      ifdef LOG_NEWS
#        define WITH_ERRLOG LOG_NEWS
#      else
#        ifdef LOG_USER
#          define WITH_ERRLOG LOG_USER
#        endif
#      endif
#   endif
#endif

#ifdef WITH_ERRLOG
    openlog(pname, LOG_PID, WITH_ERRLOG);
#endif
    /* openlog is not really required - syslog will do it for us */
}

/*
 *  show something is happening
 */
void
show_activity(void)
{
    fputc('.',stderr);
    fflush(stderr);
    ++line_not_clean;
}

/*
 *  ensure a clean line
 */
static void
clean_line(void)
{
    if (line_not_clean) {
	fputc('\n',stderr);
	line_not_clean = 0;
    }
}

/*
 *  make error message from error number
 */
char *
str_error(int err)
{
#if HAVE_STRERROR
    return strerror(err);
#else
    static char buf[30];
    sprintf("errno %d",err);
#endif
}

/*
 *  log a message   
 *
 *  the type is:
 *    L_ERR     syslog LOG_ERR
 *    L_ERRno   syslog LOG_ERR, consult errno
 *    L_INFO    syslog LOG_INFO
 *    L_DEBUG    debug
 *    L_GET      debug, w/crlf
 *    L_PUT      debug, w/crlf
 */     
#if HAVE_VPRINTF
#  if USE_VA_ALIST
void 
log_msg(type, va_alist)
    int type;
    va_dcl
#  else
void 
log_msg(int type, char *fmt, ...)
#  endif
#else
void 
log_msg(int type, char *fmt, char *arg1,char *arg2,char *arg3,char *arg4)
#endif
{
    int e;
    char buf[BUFSIZ]; /* BUG: do we risk overwriting it? */

#if HAVE_VPRINTF
    va_list ap;
#  if USE_VA_ALIST
    /* traditional method */
    char *fmt;
    va_start(ap);
    fmt = va_arg(ap,char *);
#  else
    /* ANSI method, HAVE_STDARG_H */
    va_start(ap, fmt);
#  endif
#endif

    switch (type) {
    default:
    case L_ERRno:
    case L_ERR:
	e = errno;
	vsprintf(buf, fmt, ap);
	if (type == L_ERRno) {
	    sprintf(buf + strlen (buf), ": %s", str_error(e));
	}
	strcat(buf, "\n");
#if HAVE_SYSLOG_H
	if (!debug_opt) {
	    syslog(LOG_ERR, buf);
	} else 
#endif
	{
	    clean_line();
	    fprintf(stderr, "%s: %s", pname, buf);
	    fflush(stderr);
	}
	break;

    case L_INFO:
	vsprintf(buf, fmt, ap);
	strcat(buf, "\n");
#if HAVE_SYSLOG_H
	if (!debug_opt) {
	    syslog(LOG_INFO, buf);
	} else 
#endif
	{
	    clean_line();
	    fprintf(stderr, "%s", buf);
	    fflush(stderr);
	}
	break;

    case L_GET:
	if (debug_opt >= 3) {
	    clean_line();
	    fprintf(stderr, "<- ");
	    vfprintf(stderr, fmt, ap);
	    fflush(stderr);
	}
	break;

    case L_PUT:
	if (debug_opt >= 3) {
	    clean_line();
	    fprintf(stderr, "->");
	    vfprintf(stderr, fmt, ap);
	    fflush(stderr);
	}
	break;

    case L_DEBUGMORE:
	if (debug_opt < 5) break;
    case L_DEBUG: 
	if (debug_opt >= 1) {
	    clean_line();
	    vfprintf(stderr, fmt, ap);
	    fprintf(stderr, "\n");
	    fflush(stderr);
	}
	break;
    }
#if HAVE_VPRINTF
    va_end(ap);
#endif
}


/*
 *  debug message
 *  used by dbz module
 */     
#if HAVE_VPRINTF
#  if USE_VA_ALIST
void 
debug_dbz(va_alist)
    va_dcl
#  else
void 
debug_dbz(char *fmt, ...)
#  endif
#else
void 
debug_dbz(char *fmt, char *arg1,char *arg2,char *arg3,char *arg4)
#endif
{
#if HAVE_VPRINTF
    va_list ap;
#  if USE_VA_ALIST
    /* traditional method */
    char *fmt;
    va_start(ap);
    fmt = va_arg(ap,char *);
#  else
    /* ANSI method, HAVE_STDARG_H */
    va_start(ap, fmt);
#  endif
#endif

    if (debug_opt >= 6) {
	clean_line();
	fprintf(stderr, "dbz: ");
	vfprintf(stderr, fmt, ap);
	fflush(stderr);
    }
#if HAVE_VPRINTF
    va_end(ap);
#endif
}
