/* Written by and copyright Maurizio Sartori <masar@MasarLabs.com>. */

/* seek: extract rows at selected offsets from a large, indexed table.

   Copyright (C) 1998-2001 Maurizio Sartori <masar@MasarLabs.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


   Usage:  seek [options] table < offset_list

   Options:
       -R|--raw-input
            The input offset list is not in table format.

       -x|--hex
            Input offset are expressed in hexadecimal values.

       -N|--no-header
            Do not print the table header on STDOUT.

       -i|--info
            Instead of the actual table content, a new table is
            printed, containing information on the requested offset(s).
            The information printed is the start, end and size (in bytes)
            of the relevant rows, and a flag that tells whether the
            latter are empty, i.e. contain just blanks and/or tabs.

       -b|--blank
            Same as '--info', but only blank lines, if any, are taken into
            account.


   This operator takes a list of byte-offsets on STDIN and prints on STDOUT
   the corresponding table records from the table specified as a command
   line argument. The list of offsets can be built with the 'index'
   operator, and 'seek' expects it to be a one-column table, like this:

           lpos
           ----
           228
           1117
           518
           1225

   The column name, 'lpos' in the example, can take any name, provided
   that the list has a valid NoSQL table format. If option '-R' is
   given, then the offsets are expected to be a blank- , tab- or
   newline-separated raw list of numbers on STDIN, i.e. they are not
   expected to to have a valid NoSQL table header.

   If option '-x' is given, then the offsets are expected to be base 16
   numbers instead of the usual base 10 numbers.

   Examples :

     seek table < offset_table

     or

     echo "228 1117 518 1225" | seek -R table
     echo "a28 1b1F 5c8 1425" | seek -R -x table


 ****************************************************************************/

/*
static char Psz_RCS [] =
    "$Id: seek.c,v 1.1.1.1 2003/03/17 09:49:20 carlo Exp $";
*/

/****************************************************************************/

#include <stdio.h>
#include <ctype.h>


/****************************************************************************/
/*                          Function prototypes                             */
/****************************************************************************/
void LineOutput( FILE * file, const char * pszStr );
void LineSkip( FILE * file, const char * pszStr );
void LineInfo( FILE * file, const int fBlank, const char * pszStr );

/****************************************************************************/
/*                              LineOutput ()                               */
/****************************************************************************/

void LineOutput( FILE * file, const char * pszStr )
{
   int chChar;

   chChar = fgetc( file );
   while ( chChar != '\n' )
   {
      if ( chChar == EOF )
      {
         fprintf( stderr, pszStr );
         exit(1);
      }

      fputc( chChar, stdout );
      chChar = fgetc( file );
   }
   fputc( chChar, stdout );
}

/****************************************************************************/
/*                              LineInfo ()                                 */
/****************************************************************************/

void LineInfo( FILE * file, const int fBlank, const char * pszStr )
{
   int chChar;
   long iStart = 0;
   long iEnd = 0;
   long iSize = 0;
   char cBlank = 'y';

   iStart = ftell( file );

   chChar = fgetc( file );
   while ( chChar != '\n' )
   {
      if ( chChar == EOF )
      {
         fprintf( stderr, pszStr );
         exit(1);
      }

      /* Check whether it is a blank row */

      if ( chChar != ' ' && chChar != '\t' ) cBlank = 'n';

      chChar = fgetc( file );
   }
   iEnd = ftell( file );

   /* Exclude the trailing newline/EOF from line size */
   iEnd -=2;
   iSize = iEnd - iStart;

   if ( fBlank )
   {
     if ( cBlank == 'y' )
     {
       printf( "%ld\t%ld\t%ld\t%c\n", iStart, iEnd, iSize, cBlank );
     }
   }
   else printf( "%ld\t%ld\t%ld\t%c\n", iStart, iEnd, iSize, cBlank );
}

/****************************************************************************/
/*                               LineSkip ()                                */
/****************************************************************************/
void LineSkip( FILE * file, const char * pszStr )
{
   int chChar;

   chChar = fgetc( file );
   while ( chChar != '\n' )
   {
      if ( chChar == EOF )
      {
         fprintf( stderr, pszStr );
         exit(1);
      }

      chChar = fgetc( file );
   }
}

/****************************************************************************/
/*                                  main ()                                 */
/****************************************************************************/
int main( int argc, char **argv )
{
   int    fRaw = 0;
   int    fHdr = 1;
   int    fInfo = 0;
   int    fBlank = 0;
   FILE * fileDb = NULL;
   int    iInx;
   int    chChar;
   long   lOffset;
   int    iValue[256];
   int    iBase;

   for ( iInx = 0; iInx < 256; iInx++ )
   {
      iValue[iInx] = 0;
   }

   iValue['0'] = 1;
   iValue['1'] = 2;
   iValue['2'] = 3;
   iValue['3'] = 4;
   iValue['4'] = 5;
   iValue['5'] = 6;
   iValue['6'] = 7;
   iValue['7'] = 8;
   iValue['8'] = 9;
   iValue['9'] = 10;
   iBase = 10;

   for ( iInx = 1; iInx < argc; iInx++ )
   {
      if ( !strcmp( argv[iInx], "-R" )
        || !strcmp( argv[iInx], "--raw-input" ) )
      {
         fRaw = 1;
      }
      else if ( !strcmp ( argv[iInx], "-x" )
            || !strcmp ( argv[iInx], "--hex" ) )
      {
         iValue['a'] = iValue['A'] = 11;
         iValue['b'] = iValue['B'] = 11;
         iValue['c'] = iValue['C'] = 11;
         iValue['d'] = iValue['D'] = 11;
         iValue['e'] = iValue['E'] = 11;
         iValue['f'] = iValue['F'] = 11;
         iBase = 16;
      }
      else if ( !strcmp( argv[iInx], "-N" )
        || !strcmp( argv[iInx], "--no-header" ) )
      {
         fHdr = 0;
      }
      else if ( !strcmp( argv[iInx], "-i" )
        || !strcmp( argv[iInx], "--info" ) )
      {
         fInfo = 1;
      }
      else if ( !strcmp( argv[iInx], "-b" )
        || !strcmp( argv[iInx], "--blank" ) )
      {
         fBlank = 1;
         fInfo = 1;
      }
      else if ( fileDb )
      {
         fprintf( stderr, "seek: invalid number of parameters\n" );
         return 1;
      }
      else
      {
         fileDb = fopen( argv[iInx], "rt" );
         if ( !fileDb )
         {
            fprintf( stderr, "seek: cannot open \"%s\"\n", argv[iInx] );
            return 1;
         }
      }
   }

   if ( !fileDb )
   {
      fprintf( stderr, "seek: no database specified\n" );
      return 1;
   }

   /*************************************************************************/
   /* Put headers (first and second line of the database)                   */
   /*************************************************************************/
   if ( fHdr && !fInfo )
   {
     LineOutput( fileDb, "Invalid Database Header 1\n" );
     LineOutput( fileDb, "Invalid Database Header 2\n" );
   }

   /*************************************************************************/
   /* Skip headers if not Raw Mode                                          */
   /*************************************************************************/
   if ( !fRaw )
   {
      LineSkip( stdin, "seek: invalid input header 1\n" );
      LineSkip( stdin, "seek: invalid input header 2\n" );
   }

   /* Print the output table header if '--info' was requested. */

   if ( fHdr && fInfo )
     printf("Start\tEnd\tSize\tBlank\n-----\t---\t----\t-----\n");

   /*************************************************************************/
   /* Process offsets                                                       */
   /*************************************************************************/
   lOffset = 0;
   chChar = fgetc( stdin );
   while ( chChar != EOF )
   {
      if (iValue [chChar])
      {
         lOffset *= iBase;
         lOffset += iValue[chChar] - 1;
      }
      else if ( isspace( chChar ) )
      {
         if ( lOffset )
         {
            if ( fseek( fileDb, lOffset, SEEK_SET ) )
            {
               fprintf( stderr, "seek: cannot seek\n" );
               exit(1);
            }

            if ( !fInfo )
              LineOutput( fileDb, "seek: invalid file line\n" );
            else
              LineInfo( fileDb, fBlank, "seek: invalid file line\n" );
         }

         lOffset = 0;
      }
      else
      {
         fprintf( stderr, "seek: invalid character \"%c\"\n", chChar );
         exit(1);
      }

      chChar = fgetc( stdin );
   }

   return 0;
}

/* End of program */
