/*
 *  Copyright (C) 1998-2001 Luca Deri <deri@ntop.org>
 *                          Portions by Stefano Suin <stefano@ntop.org>
 *                      
 *		 	    http://www.ntop.org/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "ntop.h"
#include "globals-report.h"

static int sqlSocket = -1;

/* Global */
static struct sockaddr_in dest;

/* **************************************** */

static void openSQLsocket(char* dstHost, int dstPort) {
  struct hostent *hostAddr = gethostbyname(dstHost);
  
  if(hostAddr == NULL) {
    traceEvent(TRACE_INFO, "Unable to resolve address '%s'\n", dstHost);
    exit(-1);
  }
	
  memcpy(&dest.sin_addr.s_addr, hostAddr->h_addr_list[0],
	 hostAddr->h_length);
  dest.sin_family      = AF_INET;
  dest.sin_port        = (int)htons((unsigned short int)dstPort);

  sqlSocket = socket (AF_INET, SOCK_DGRAM, 0);
  
  if(sqlSocket <= 0) {
    traceEvent(TRACE_INFO, "Unable to open SQLsocket\n");
      exit(-1);
  } else {
    traceEvent(TRACE_INFO, "Open channel with ntop SQL client running @ %s:%d\n", dstHost, dstPort);
  }
}

/* **************************************** */

void handleDbSupport(char* addr /* host:port */, int* enableDBsupport) {
  char *hostName, *strtokState;
  int  portNumber;

  if((addr == NULL) || (addr[0] == '\0'))
    return;

  hostName = strtok_r(addr, ":", &strtokState);
  portNumber = atoi(strtok_r(NULL, ":", &strtokState));

  if((hostName == NULL) || (portNumber == 0)) {
    traceEvent(TRACE_WARNING, "WARNING: invalid value specified for '-b' parameter. \n"
          "         It should be host:port.\n");
    return;
  } else {
    (*enableDBsupport) = 1;
    openSQLsocket(hostName, portNumber); /* *** SQL Engine *** */
  }
}

/* **************************************** */

void closeSQLsocket(void) {
#ifndef WIN32
  close(sqlSocket);
#else
  closesocket(sqlSocket);
#endif     
}

/* **************************************** */

void updateHostNameInfo(unsigned long numeric, char* symbolic) {
  char *hostName;
  struct in_addr addr;
  char buf[32];
  char sqlBuf[BUF_SIZE];
  u_int idx;

  if(!capturePackets) return;

  addr.s_addr = numeric;
  hostName = _intoa(addr, buf, sizeof(buf));

  /* Search the instance and update its name */

#ifdef MULTITHREADED
  accessMutex(&addressResolutionMutex, "updateHostNameInfo");
#endif
    
  idx = findHostIdxByNumIP(addr);

  if(idx != NO_PEER) {
    if(device[actualDeviceId].hash_hostTraffic[idx] != NULL) {

      if(strlen(symbolic) >= (MAX_HOST_SYM_NAME_LEN-1)) 
	symbolic[MAX_HOST_SYM_NAME_LEN-2] = '\0';
      strcpy(device[actualDeviceId].hash_hostTraffic[idx]->hostSymIpAddress, symbolic);
    }
  }

#ifdef MULTITHREADED
    releaseMutex(&addressResolutionMutex);
#endif

  if(sqlSocket != -1) {
    if(strcmp(hostName, symbolic) != 0) {  
      /* [1] Delete */
      if(snprintf(sqlBuf, sizeof(sqlBuf), "DELETE FROM NameMapper WHERE IPaddress = '%s'", hostName) < 0) 
	traceEvent(TRACE_ERROR, "Buffer overflow!");
      sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, 
	     (struct sockaddr *)&dest, sizeof(dest));
      
      /* [2] Insert */
      if(snprintf(sqlBuf, sizeof(sqlBuf), "INSERT INTO NameMapper (IPaddress, Name)"
		  " VALUES ('%s', '%s')", hostName, symbolic) < 0)
	traceEvent(TRACE_ERROR, "Buffer overflow!");
      sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, 
	     (struct sockaddr *)&dest, sizeof(dest));
    }
  }

#ifdef HAVE_MYSQL
  mySQLupdateHostNameInfo(numeric, symbolic);
#endif
}

/* **************************************** */

void updateHostTraffic(HostTraffic *el) {
  char theDate[32], theDate2[32];
  char sqlBuf[2*BUF_SIZE];
  struct tm t;

  if((sqlSocket == -1)
     || (broadcastHost(el))
     || (el->hostNumIpAddress[0] == '\0'))
    return;

  /* Fixes below courtesy of Andreas Pfaller <a.pfaller@pop.gun.de> */
  strftime(theDate2, 32, "%Y-%m-%d %H:%M:%S", localtime_r(&el->firstSeen, &t));

  /* Added by David Moore <davem@mitre.org> */
  strftime(theDate, 32, "%Y-%m-%d %H:%M:%S", localtime_r(&el->lastSeen, &t));  

  /* ****************************** */

  if(snprintf(sqlBuf, sizeof(sqlBuf), "UPDATE Hosts SET "
	      "PktSent = %llu, "
	      "PktRcvd = %llu, "
	      "PktMulticastSent = %llu, "
	      "PktBroadcastSent = %llu, "
	      "DataSent = %llu, "
	      "DataRcvd = %llu, "
	      "DataMulticastSent = %llu, "
	      "DataBroadcastSent = %llu, "
	      "FirstSeen = '%s', "
	      "LastSeen = '%s'"
	      " WHERE IPaddress = '%s'",
	      (el->pktSent),
	      (el->pktReceived),
	      (el->pktMulticastSent),
	      (el->pktBroadcastSent),
	      (el->bytesSent),
	      (el->bytesReceived),
	      (el->bytesMulticastSent),
	      (el->bytesBroadcastSent),
	      theDate2, theDate,
	      el->hostNumIpAddress) < 0) traceEvent(TRACE_ERROR, "Buffer overflow!");
  
  sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));
    
  /* ****************************** */

  if(snprintf(sqlBuf, sizeof(sqlBuf), "UPDATE IPtraffic SET "
	  "TCPSentLocally = %llu, "
	  "TCPSentRemotely = %llu, "
	  "TCPrcvdLocally = %llu, "
	  "TCPrcvdFromRemote = %llu, "
	  "UDPSentLocally = %llu, "
	  "UDPSentRemotely = %llu, "
	  "UDPrcvdLocally = %llu, "
	  "UDPrcvdFromRemote = %llu, "
	  "ICMPsent = %llu, "
	  "ICMPrcvd = %llu, "
 	  "OSPFsent = %llu, "
	  "OSPFrcvd = %llu, "
	  "IGMPsent = %llu, "
	  "IGMPrcvd = %llu "
	  " WHERE IPaddress = '%s'",
	  (el->tcpSentLocally),
	  (el->tcpSentRemotely),
	  (el->tcpReceivedLocally),
	  (el->tcpReceivedFromRemote),
	  (el->udpSentLocally),
	  (el->udpSentRemotely),
	  (el->udpReceivedLocally),
	  (el->udpReceivedFromRemote),
	  (el->icmpSent),
	  (el->icmpReceived),
	  (el->ospfSent),
	  (el->ospfReceived),
	  (el->igmpSent),
	  (el->igmpReceived),
	  el->hostNumIpAddress) < 0) traceEvent(TRACE_ERROR, "Buffer overflow!");

  sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));

  /* ****************************** */

  if(snprintf(sqlBuf, sizeof(sqlBuf), "UPDATE NonIPTraffic SET "
	  "IPXsent = %llu, "
	  "IPXrcvd = %llu, "
 	  "OSIsent = %llu, "
	  "OSIrcvd = %llu, "
	  "DLCsent = %llu, "
	  "DLCrcvd = %llu, "
 	  "ARPsent = %llu, "
	  "ARPrcvd = %llu, "
	  "DECNETsent = %llu, "
	  "DECNETrcvd = %llu, "
 	  "ATALKsent = %llu, "
	  "ATALKrcvd = %llu, "
	  "NBIOSsent = %llu, "
	  "NBIOSrcvd = %llu, "
 	  "OtherSent = %llu, "
	  "OtherRcvd = %llu "
	  " WHERE IPaddress = '%s'",
	  (el->ipxSent),
	  (el->ipxReceived),
	  (el->osiSent),
	  (el->osiReceived),
	  (el->dlcSent),
	  (el->dlcReceived),
	  (el->arp_rarpSent),
	  (el->arp_rarpReceived),
	  (el->decnetSent),
	  (el->decnetReceived),
	  (el->appletalkSent),
	  (el->appletalkReceived),
	  (el->netbiosSent),
	  (el->netbiosReceived),
	  (el->otherSent),
	  (el->otherReceived),
	  el->hostNumIpAddress) < 0) traceEvent(TRACE_ERROR, "Buffer overflow!");

  sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));
}

/* **************************************** */

void notifyHostCreation(HostTraffic *el) {
  char sqlBuf[BUF_SIZE];

  if((sqlSocket == -1) || broadcastHost(el))
    return;

  /* [1] Delete */

  if(el->hostNumIpAddress[0] != '\0') {
    if(snprintf(sqlBuf, sizeof(sqlBuf), "DELETE FROM Hosts WHERE IPaddress = '%s'", 
		el->hostNumIpAddress) < 0) 
      traceEvent(TRACE_ERROR, "Buffer overflow!");
  } else {
    if(snprintf(sqlBuf, sizeof(sqlBuf), "DELETE FROM Hosts WHERE MACaddress = '%s'", 
		el->ethAddressString) < 0) 
      traceEvent(TRACE_ERROR, "Buffer overflow!");
  }
  
  sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));
  
  /* [2] Insert */
  if(snprintf(sqlBuf, sizeof(sqlBuf), "INSERT INTO Hosts (IPaddress, MACaddress, NICvendor)"
	  " VALUES ('%s', '%s', '%s')",
	  el->hostNumIpAddress,
	  el->ethAddressString,
	  getVendorInfo(el->ethAddress, 0)) < 0) traceEvent(TRACE_ERROR, "Buffer overflow!");

  sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));

  if(el->hostNumIpAddress[0] != '\0') {
    if(snprintf(sqlBuf, sizeof(sqlBuf), "DELETE FROM NonIPTraffic WHERE IPaddress = '%s'",
		el->hostNumIpAddress) < 0) 
      traceEvent(TRACE_ERROR, "Buffer overflow!");
    sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));

    if(snprintf(sqlBuf, sizeof(sqlBuf), "INSERT INTO NonIPTraffic (IPaddress) VALUES ('%s')", 
		el->hostNumIpAddress) < 0) 
     traceEvent(TRACE_ERROR, "Buffer overflow!");
    sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));

    if(snprintf(sqlBuf, sizeof(sqlBuf), "DELETE FROM IPtraffic WHERE IPaddress = '%s'", 
		el->hostNumIpAddress) < 0) 
      traceEvent(TRACE_ERROR, "Buffer overflow!");
    sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));

    if(snprintf(sqlBuf, sizeof(sqlBuf), "INSERT INTO IPtraffic (IPaddress) VALUES ('%s')", 
		el->hostNumIpAddress) < 0) 
      traceEvent(TRACE_ERROR, "Buffer overflow!");
    sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));
  }

  /* traceEvent(TRACE_INFO, "%s\n", buf); */

}

/* **************************************** */

void notifyTCPSession(IPSession *session) {
  HostTraffic *server, *client;
  char dt1[32], dt2[32];
  struct tm t;
  char sqlBuf[BUF_SIZE];

  if((sqlSocket == -1) 
     || (session->initiatorIdx == NO_PEER)
     || (session->remotePeerIdx == NO_PEER))
    return;

  client = device[actualDeviceId].hash_hostTraffic[checkSessionIdx(session->initiatorIdx)];
  server = device[actualDeviceId].hash_hostTraffic[checkSessionIdx(session->remotePeerIdx)];

  strftime(dt1, 32, "%Y-%m-%d %H:%M:%S", localtime_r(&session->firstSeen, &t));  
  strftime(dt2, 32, "%Y-%m-%d %H:%M:%S", localtime_r(&session->lastSeen, &t));  

  if(snprintf(sqlBuf, sizeof(sqlBuf), "INSERT INTO TCPsessions (Client, Server, ClientPort, "
	  "ServerPort, DataSent, DataRcvd, FirstSeen, LastSeen)"
	  " VALUES ('%s', '%s', '%d', '%d', '%llu', '%llu', '%s', '%s')",
	  client->hostNumIpAddress,
	  server->hostNumIpAddress,
	  session->sport,
	  session->dport,
	  (session->bytesSent), 
	  (session->bytesReceived),
	  dt1, dt2) < 0) 
    traceEvent(TRACE_ERROR, "Buffer overflow!");

#ifndef DEBUG
  traceEvent(TRACE_INFO, "%s\n", sqlBuf);
#endif

  sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));
}

/* **************************************** */

void updateDBOSname(HostTraffic *el) {
  char sqlBuf[BUF_SIZE];

  if((sqlSocket == -1) 
     || (el->osName == NULL)
     || (el->osName[0] == '\0'))
    return;
  
  /* traceEvent(TRACE_INFO, "%s@%s\n", el->osName, el->hostNumIpAddress); */

  if(snprintf(sqlBuf, sizeof(sqlBuf), "UPDATE Hosts SET "
	  "OsName = '%s' WHERE IPaddress = '%s'",
	  el->osName, el->hostNumIpAddress) < 0) 
    traceEvent(TRACE_ERROR, "Buffer overflow!");
  
  sendto(sqlSocket, sqlBuf, strlen(sqlBuf), 0, (struct sockaddr *)&dest, sizeof(dest));
}

