/* upsfetch - library for UPS communications from client programs

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include "upsfetch.h"
#include "timehead.h"

	int	upserror, syserrno;

const char *upsstrerror (int errnum)
{
	static char	buf[SMALLBUF];

	buf[0] = '\0';

	switch (errnum) {
		case UPSF_UNKNOWN:
			return "Unknown error";
			break;
		case UPSF_VARNOTSUPP:
			return "Variable not supported by UPS";
			break;
		case UPSF_NOSUCHHOST:
			return "No such host";
			break;
		case UPSF_SENDFAILURE:
			snprintf (buf, sizeof(buf), "Send failure: %s",
			         strerror(syserrno));
			break;
		case UPSF_RECVFAILURE:
			snprintf (buf, sizeof(buf), "Receive failure: %s",
			         strerror(syserrno));
			break;
		case UPSF_SOCKFAILURE:
			snprintf (buf, sizeof(buf), "Socket failure: %s",
			         strerror(syserrno));
			break;
		case UPSF_BINDFAILURE:
			snprintf (buf, sizeof(buf), "bind failure: %s",
			         strerror(syserrno));
			break;
		case UPSF_RECVTIMEOUT:
			return "Receive timeout";
			break;
		case UPSF_NOCOMM1:
			return "Data source is stale";
			break;
		case UPSF_UNKNOWNUPS:
			return "Unknown ups name";
			break;
		case UPSF_ACCESSDENIED:
			return "Access denied";
			break;
		case UPSF_CONNFAILURE:
			snprintf (buf, sizeof(buf), "Connection failure: %s",
			         strerror(syserrno));
			break;
		case UPSF_READERROR:
			snprintf (buf, sizeof(buf), "Read error: %s",
			         strerror(syserrno));
			break;
		case UPSF_WRITEERROR:
			snprintf (buf, sizeof(buf), "Write error: %s",
			         strerror(syserrno));
			break;
		case UPSF_INVALIDPW:
			return "Invalid password";
			break;
		case UPSF_BADPASSWD:
			return "Password incorrect";
			break;
		case UPSF_WRONGVAR:
			return "Received wrong variable";
			break;
		case UPSF_UNKNOWNLVL:
			return "Unknown level";
			break;
		case UPSF_UNKNOWNCMD:
			return "Unknown command";
			break;
		case UPSF_DATASTALE:
			return "Data stale";
			break;
		case UPSF_VARUNKNOWN:
			return "Variable not recognized by server";
			break;
		default:
			snprintf(buf, sizeof(buf), "Unknown error 0x%04x", errnum);
			break;
	}

	return (buf);
}

/* common error condition handler */
int errcheck (const char *buf)
{
	if (!strncmp(buf, "ERR", 3)) {
		upserror = UPSF_UNKNOWN;		/* default error */

		if (!strncmp(buf, "ERR ACCESS-DENIED", 17)) 
			upserror = UPSF_ACCESSDENIED;

		if (!strncmp(buf, "ERR UNKNOWN-UPS", 15))
			upserror = UPSF_UNKNOWNUPS;

		if (!strncmp(buf, "ERR INVALID-PASSWORD", 20))
			upserror = UPSF_INVALIDPW;

		if (!strncmp(buf, "ERR PASSWORD-INCORRECT", 22))
			upserror = UPSF_INVALIDPW;

		if (!strncmp(buf, "ERR UNKNOWN-COMMAND", 19))
			upserror = UPSF_UNKNOWNCMD;

		if (!strncmp(buf, "ERR DATA-STALE", 14))
			upserror = UPSF_DATASTALE;

		if (!strncmp(buf, "ERR VAR-NOT-SUPPORTED", 21))
			upserror = UPSF_VARNOTSUPP;		

		if (!strncmp(buf, "ERR VAR-UNKNOWN", 15))
			upserror = UPSF_VARUNKNOWN;		

		if (!strncmp(buf, "ERR CMD-NOT-SUPPORTED", 21))
			upserror = UPSF_CMDNOTSUPP;

		return (-1);
	}

	/* legacy failure msg - deprecated */
	if (!strncmp(buf, "Unknown command", 15)) {
		upserror = UPSF_UNKNOWNCMD;
		return (-1);
	}

	return 0;
}

/* send <req> to <host> and put the answer in <buf> */

int fetch (const char *hostin, const char *req, char *buf, int buflen)
{
	int	res, port = PORT, fd, fromlen, err;
	struct	sockaddr_in xmit, dest, from;
	struct	hostent *serv;
	char	sbuf[LARGEBUF], host[LARGEBUF], *cpos;
	fd_set	rfd;
	struct	timeval tv;

	upserror = UPSF_UNKNOWN;

	snprintf (host, sizeof(host), "%s", hostin);
	cpos = strstr (host, ":");

	if (cpos != NULL) {
		cpos[0] = 0;
		port = atoi(++cpos);
	}

	if ((serv = gethostbyname(host)) == (struct hostent *) NULL) {
		upserror = UPSF_NOSUCHHOST;
		return (-1);
	}

	fd = socket (AF_INET, SOCK_DGRAM, 0);
	if (fd < 0) {
		upserror = UPSF_SOCKFAILURE;
		syserrno = errno;
		return (-1);
	}

	memset (&xmit, '\0', sizeof(xmit));
	xmit.sin_family = AF_INET;

	res = bind (fd, (struct sockaddr *) &xmit, sizeof(xmit));
	if (res < 0) {
		upserror = UPSF_BINDFAILURE;
		syserrno = errno;
		close (fd);
		return (-1);
	}

	memset (&dest, '\0', sizeof(dest));
	memcpy (&dest.sin_addr, serv->h_addr, serv->h_length);
	dest.sin_family = AF_INET;
	dest.sin_port = htons(port);

	snprintf (sbuf, sizeof(sbuf), "%s", req);
	res = sendto (fd, sbuf, strlen(sbuf), 0, 
	             (struct sockaddr *) &dest, sizeof(dest));

	if (res < 0) {
		syserrno = errno;
		upserror = UPSF_SENDFAILURE;
		close (fd);
		return (-1);	/* failure */
	}

	FD_ZERO (&rfd);
	FD_SET (fd, &rfd);
	tv.tv_sec = 2;	/* 2.0 seconds allowed for reply */
	tv.tv_usec = 0;

	res = select (fd + 1, &rfd, NULL, NULL, &tv);

	if (res > 0) {
		memset (buf, '\0', buflen);
		fromlen = sizeof(from);
		res = recvfrom (fd, buf, buflen, 0, (struct sockaddr *) &from,
		                &fromlen);
	}
	else {
		syserrno = errno;
		upserror = UPSF_RECVTIMEOUT;
		close (fd);
		return (-1);	/* failure */
	}

	if (res < 0) {
		syserrno = errno;
		upserror = UPSF_RECVFAILURE;
		close (fd);
		return (-1);	/* failure */
	}

	close(fd);

	if ((err = errcheck(buf)) != 0)
		return err;

	upserror = 0;
	return 1;	/* success */
}

/* warn about old upsd quirks, but still handle them for now */
static void warn_oldupsd(char *tag)
{
	fprintf(stderr, "upsfetch: old upsd detected (ANS %s)\n", tag);
}

int getupsvar(const char *hostin, const char *varname, char *buf, int buflen)
{
	char	cmd[SMALLBUF], tmp[LARGEBUF], *ptr, *host, *upsname, hosttmp[LARGEBUF],
		vartmp[SMALLBUF];
	int	ret;

	/* host = ups-1@localhost */

	snprintf(hosttmp, sizeof(hosttmp), "%s", hostin);
	ptr = strstr(hosttmp, "@");

	if (ptr != NULL) {
		ptr[0] = 0;
		upsname = hosttmp;
		host = ptr + 1;
		snprintf(cmd, sizeof(cmd), "REQ %s@%s", varname, upsname);
	}
	else {
		upsname = NULL;
		host = hosttmp;
		snprintf(cmd, sizeof(cmd), "REQ %s", varname);
	}

	ret = fetch(host, cmd, tmp, sizeof(tmp));

	if (ret != 1)
		return ret;

	tmp[strlen(tmp) - 1] = 0;

	/* ANS <varname> <answer> */

	/* check variable name for sanity */
	snprintf(vartmp, sizeof(vartmp), "%s", &tmp[4]);
	vartmp [strlen(varname)] = 0;

	if (strcasecmp(vartmp, varname) != 0) {
		upserror = UPSF_WRONGVAR;
		return -1;		/* failure */
	}

	if (upsname == NULL)		/* skip "ANS <varname> " */
		ptr = tmp + strlen(varname) + 5;
	else				/* skip "ANS <varname>@<upsname> " */
		ptr = tmp + strlen(varname) + 5 + strlen(upsname) + 1;

	/* --- legacy stuff for older upsd versions --- */

	/* old upsd versions returned some errors as ANS <var> <error> */

	if (!strcmp(ptr, "DATA-STALE")) {
		warn_oldupsd("DATA-STALE");
		upserror = UPSF_DATASTALE;
		return -1;	/* failure */
	}

	if (!strcmp(ptr, "NOT-SUPPORTED")) {
		warn_oldupsd("NOT-SUPPORTED");
		upserror = UPSF_VARNOTSUPP;
		return -1;	/* failure */
	}

	if (!strcmp(ptr, "UNKNOWN")) {
		warn_oldupsd("UNKNOWN");
		upserror = UPSF_VARUNKNOWN;
		return -1;	/* failure */
	}

	if (!strcmp(ptr, "UNKNOWN-UPS")) {
		warn_oldupsd("UNKNOWN-UPS");
		upserror = UPSF_UNKNOWNUPS;
		return -1;	/* failure */
	}

	memset(buf, '\0', buflen);
	strlcpy(buf, ptr, buflen);

	return ret;
}

int getupsvarlist (const char *hostin, char *buf, int buflen)
{
	char	tmp[LARGEBUF], *ptr, *upsname, *host, cmd[128], hosttmp[LARGEBUF];
	int	ret;

	/* host = ups-1@localhost */

	snprintf (hosttmp, sizeof(hosttmp), "%s", hostin);
	ptr = strstr (hosttmp, "@");

	if (ptr != NULL) {
		ptr[0] = 0;
		host = ptr + 1;
		upsname = hosttmp;
		snprintf (cmd, sizeof(cmd), "LISTVARS %s", upsname);
	}
	else {
		upsname = NULL;
		host = hosttmp;
		snprintf (cmd, sizeof(cmd), "LISTVARS");
	}

	ret = fetch (host, cmd, tmp, sizeof(tmp));

	if (ret != 1)
		return (ret);

	/* VARS <varlist> */
	tmp[strlen(tmp) - 1] = 0;

	if (upsname == NULL)
		ptr = tmp + 5;	/* skip "VARS " */
	else
		ptr = tmp + 5 + strlen(upsname) + 2;  /* skip "VARS <name> " */

	memset (buf, '\0', buflen);
	strlcpy (buf, ptr, buflen);

	return (ret);
}

int upsconnect (const char *hostin)
{
	struct	sockaddr_in local, server;
	struct	hostent *serv;
	int	fd, port = PORT;
	char	*cpos, *host;

	host = xstrdup (hostin);
	cpos = strstr (host, ":");
	if (cpos != NULL) {
		cpos[0] = 0;
		port = atoi(++cpos);
	}

	if ((serv = gethostbyname(host)) == (struct hostent *) NULL) {
		upserror = UPSF_NOSUCHHOST;
		return (-1);
	}

	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		upserror = UPSF_SOCKFAILURE;
		syserrno = errno;
		return (-1);
	}		

	memset (&local, '\0', sizeof(struct sockaddr_in));
	local.sin_family = AF_INET;
	local.sin_port  = htons (INADDR_ANY);

	memset (&server, '\0', sizeof(struct sockaddr_in));
	server.sin_family = AF_INET;

	server.sin_port = htons (port);
	memcpy (&server.sin_addr, serv->h_addr, serv->h_length);

	if (bind (fd, (struct sockaddr *) &local, sizeof(struct sockaddr_in)) == -1) {
		upserror = UPSF_BINDFAILURE;
		syserrno = errno;
		close (fd);
		return (-1);
	}

	if (connect (fd, (struct sockaddr *) &server, sizeof(struct sockaddr_in)) == -1) {
		upserror = UPSF_CONNFAILURE;
		syserrno = errno;
		close (fd);
		return (-1);
	}

	return (fd);
}

void closeupsfd (int fd)
{
	shutdown (fd, 2);
	close (fd);
}

/* read from <fd> until a LF */
int ups_readline (int fd, char *tmp, int buflen)
{
	int	tmpptr = 0, ret;
	char	ch;

	for (;;) {
		ret = read (fd, &ch, 1);

		if (ret < 1) {
			upserror = UPSF_READERROR;
			syserrno = errno;
			return (ret);
		}

		if ((ch == 10) || (tmpptr >= buflen))
			return 0;

		tmp[tmpptr++] = ch;
	}

	/* NOTREACHED */
	return 0;
}

int getupsvarlistfd (int fd, const char *upsname, char *buf, int buflen)
{
	char	tmp[LARGEBUF], *ptr, cmd[128];
	int	ret, err;

	if (upsname == NULL)
		snprintf (cmd, sizeof(cmd), "LISTVARS\n");
	else
		snprintf (cmd, sizeof(cmd), "LISTVARS %s\n", upsname);

	ret = write (fd, cmd, strlen(cmd));

	if (ret < 1) {
		upserror = UPSF_WRITEERROR;
		syserrno = errno;
		return (-1);
	}

	memset (tmp, '\0', sizeof(tmp));

	if ((err = ups_readline (fd, tmp, buflen)) != 0)
		return (err);

	if ((err = errcheck (tmp)) != 0)
		return (err);

	/* VARS <varlist> */
	if (upsname == NULL)
		ptr = tmp + 5;	/* skip "VARS " */
	else
		ptr = tmp + 5 + strlen(upsname) + 2;  /* skip "VARS <name> " */

	memset (buf, '\0', buflen);
	strlcpy (buf, ptr, buflen);

	return (ret);
}

int getupsvarfd (int fd, const char *upsname, const char *varname, char *buf, int buflen)
{
	char	cmd[SMALLBUF], tmp[LARGEBUF], *ptr;
	int	ret, err;

	if (upsname == NULL)
		snprintf (cmd, sizeof(cmd), "REQ %s\n", varname);
	else
		snprintf (cmd, sizeof(cmd), "REQ %s@%s\n", varname, upsname);

	ret = write (fd, cmd, strlen(cmd));

	if (ret < 1) {
		upserror = UPSF_WRITEERROR;
		syserrno = errno;
		return (-1);
	}

	memset (tmp, '\0', sizeof(tmp));

	if ((err = ups_readline (fd, tmp, buflen)) != 0)
		return (err);

	if ((err = errcheck (tmp)) != 0)
		return (err);

	/* ANS <varname> <answer> */

	if (upsname == NULL)		/* skip "ANS <varname> " */
		ptr = tmp + strlen(varname) + 5;
	else				/* skip "ANS <varname>@<upsname> " */
		ptr = tmp + strlen(varname) + 5 + strlen(upsname) + 1;

	/* --- legacy stuff for older upsd versions --- */

	/* old upsd versions returned some errors as ANS <var> <error> */

	if (!strcmp(ptr, "DATA-STALE")) {
		warn_oldupsd("DATA-STALE");
		upserror = UPSF_DATASTALE;
		return -1;	/* failure */
	}

	if (!strcmp(ptr, "NOT-SUPPORTED")) {
		warn_oldupsd("NOT-SUPPORTED");
		fprintf(stderr, "upsfetch: old upsd detected! (ANS NOT-SUPPORTED)\n");
		upserror = UPSF_VARNOTSUPP;
		return -1;	/* failure */
	}

	if (!strcmp(ptr, "UNKNOWN-UPS")) {
		warn_oldupsd("UNKNOWN-UPS");
		upserror = UPSF_UNKNOWNUPS;
		return (-1);	/* failure */
	}

	memset(buf, '\0', buflen);
	strlcpy(buf, ptr, buflen);

	return ret;
}

/* send a raw buffer to upsd with basic error checking */
int upssendraw(int fd, const char *cmd)
{
	int	ret;

	upserror = UPSF_UNKNOWN;

	ret = write(fd, cmd, strlen(cmd));

	if (ret < 1) {
		upserror = UPSF_WRITEERROR;
		syserrno = errno;
		return -1;
	}

	upserror = 0;
	return 1;		/* success */
}

/* read a response from upsd with basic error checking */
int upsreadraw(int fd, char *buf, int buflen)
{
	char	tmp[SMALLBUF];
	int	err;

	memset(tmp, '\0', sizeof(tmp));

	if ((err = ups_readline(fd, tmp, sizeof(tmp))) != 0)
		return err;

	if ((err = errcheck(tmp)) != 0)
		return err;

	strlcpy(buf, tmp, buflen);

	upserror = 0;
	return 1;	/* success */
}

