/* 
   $Id: aeg.c,v 1.3 2001/06/08 09:47:18 r_sieger Exp $
   model specific routines for AEG Protect S. xxxx models
   
   Copyright (C) 2001  
   Ralf Sieger   <ralf.sieger@gmx.net>

   from:

   bestups.c - model specific routines for Best-UPS Fortress models
   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
   
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define INFOMAX  16
#define ENDCHAR  13	/* replies end with CR */
#define UPSDELAY  5
#define MAXTRIES 10

/* global variables */
int	shmok = 1;
char	*model = NULL;
float	lowvolt = 0, highvolt = 0, voltrange = 0;
int	linenorm = 230;
int	sddelay = 90;	/* wait 90 seconds for shutdown */
int	restore = 0;	/* turn ups on again if utility is available */
int	debug = 0;
int	quiet = 0;

void initinfo (void)
{
	create_info (INFOMAX, shmok);

	/* now set up room for all future variables that are supported */
	addinfo (INFO_MFR, "", 0, 0);
	addinfo (INFO_MODEL, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_UPSTEMP, "", 0, 0);
}

/* TODO: adapt to the standard upscommon open/send/recv calls */
void setup_serial(void)
{	
	struct	termios	tio;

	if (tcgetattr (upsfd, &tio) == -1)
		fatal("tcgetattr");

	tio.c_iflag = IXON | IXOFF;
	tio.c_oflag = 0;
	tio.c_cflag = (B2400 | CS8 | CREAD | HUPCL | CLOCAL);
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0; 

#ifdef HAVE_CFSETISPEED
	cfsetispeed (&tio, B2400);
	cfsetospeed (&tio, B2400);
#endif

	if (tcsetattr (upsfd, TCSANOW, &tio) == -1)
		fatal("tcsetattr");
}

static char readchar(void)
{
	char	c;
	int	ret;

	ret = read (upsfd, &c, 1);

	if (ret > 0)
		return (c & 0x7f);
	
	return 0;
}

/* TODO: also roll this into upscommon */
static int xrecv(int len, char *buf)
{
	char	ch, *p;
	int	count = 0;

	p = buf;
	*p = '\0';

	while ((ch = readchar ()) && (count < len)) {
		if (ch == ENDCHAR) {
			*p = '\0';
			return 0;
		}
	
		*p++ = ch;
		count++;
	}

	*p = '\0';
	return 0;
}

static void ups_sync()
{
	char	buf[256];
	int	tries = 0;

	if ( !quiet) {
		printf ("Syncing with UPS: ");
		fflush (stdout);
	}

	for (;;) {
		tries++;
		if (tries > MAXTRIES)
			fatalx("\nFailed - giving up...");
		upssend("%s", "\rQ1\r");
		if ( !quiet) {
			printf (".");
			fflush (stdout);
		}
		sleep (UPSDELAY);
		if ( !quiet) {
			printf (".");
			fflush (stdout);
		}
		xrecv (sizeof (buf), buf);
		if ( !quiet) {
			printf (".");
			fflush (stdout);
		}
		if (buf[0] == '(')
			break;			
	}

	if ( !quiet) 
		printf (" done\n");
}

/* power down the attached load immediately */
void forceshutdown(const char *port)
{
	char	temp[256], stat[32];
	int	tries = 0;

	upslogx(LOG_INFO, "Initiating UPS shutdown");
	
	if ( !quiet) 
		printf ("Initiating forced UPS shutdown!\n");

	upsfd = open(port, O_RDWR | O_NDELAY);
	if (upsfd == -1)
		fatal("open %s", port);

	setup_serial();

	/* basic idea: find out line status and send appropriate command */

	if ( !quiet) {
		printf ("Checking status: ");
		fflush (stdout);
	}
	for (;;) {
		tries++;
		if (tries > MAXTRIES) {
			fatalx("Failed - giving up...");
			exit (1);
		}
		upssend ("\rQ1\r");
		if ( !quiet) {
			printf (".");
			fflush (stdout);
		}
		sleep (UPSDELAY);
		if ( !quiet) {
			printf (".");
			fflush (stdout);
		}
		xrecv (sizeof (temp), temp);
		if ( !quiet) {
			printf (".");
			fflush (stdout);
		}
		if ((temp[0] == '(') && (strlen(temp) == 46))
			break;
	}

	if ( !quiet) 
		printf (" done\n");

	sscanf (temp, "%*s %*s %*s %*s %*s %*s %*s %s", stat);

	/* on battery: send S01<cr>, ups will return by itself on utility */
	/* on line: send S01R0001<cr>, ups will cycle and return soon */

	upssend("%s", "S01");

	if (restore) {
		if ( !quiet) 
			printf ("sending shutdown+restore command...\n");
		upssend("%s", "R0001");
	} else {
		if ( !quiet) 
			printf ("sending normal shutdown command...\n");
	}

	upssend("%s", "\r");	/* end sequence */

	if (sddelay > 0) {
		if ( !quiet) 
			printf ("Waiting for poweroff...\n");
		sleep (sddelay);
		if ( !quiet) 
			printf ("Hmm, did the shutdown fail?  Oh well...\n");
		exit (1);                               
	}
	exit (0);
}

static void setmodel(const char *abbr, const char *va)
{
	char	temp[256];

	if (!strcmp(abbr, "AEG")) {
		setinfo (INFO_MFR, "AEG SVS");
		snprintf (temp, sizeof(temp), "Protect S. %s", va);
		setinfo (INFO_MODEL, temp);
		return;
	}


	setinfo(INFO_MFR, "%s", "Unknown");
	setinfo(INFO_MODEL, "Unknown %s (%s)", abbr, va);

	printf ("Unknown model detected - please report this ID: '%s'\n", abbr);
}

static void ups_update()
{
	char	utility[16], loadpct[16], acfreq[16], battvolt[16],
		upstemp[16], stat[16], buf[256], temp[VALSIZE];
	float	bvoltp;

	upssend("%s", "\rQ1\r");
	sleep (UPSDELAY); 
	xrecv (sizeof (buf), buf);

	if ((strlen(buf) < 46) || (buf[0] != '('))
		return;
	
	if (debug)
		printf ("buf Q1: [%s]\n", buf);

	sscanf (buf, "%*c%s %*s %*s %s %s %s %s %s", utility, loadpct, 
	        acfreq, battvolt, upstemp, stat);
	
	setinfo(INFO_UTILITY, "%s", utility);

	bvoltp = ((atof (battvolt) - lowvolt) / voltrange) * 100.0;

	if (bvoltp > 100.0)
		bvoltp = 100.0;

	setinfo(INFO_BATTPCT, "%02.1f", bvoltp);

	strcpy (temp, "");

	if (stat[0] == '0')
		strcat (temp, "OL ");		/* on line */
	else
		strcat (temp, "OB ");		/* on battery */

	if (stat[1] == '1')
		strcat (temp, "LB ");		/* low battery */

	if (stat[2] == '1') {		/* boost or trim in effect */
		if (atoi(utility) < linenorm)
			strcat (temp, "BOOST");

		if (atoi(utility) > linenorm)
			strcat (temp, "TRIM");
	}

	/* lose trailing space if present */
	if (temp[strlen(temp)-1] == ' ')
		temp[strlen(temp)-1] = 0;

	setinfo(INFO_STATUS, "%s", temp);
	setinfo(INFO_ACFREQ, "%s", acfreq);
	setinfo(INFO_LOADPCT, "%s", loadpct);
	setinfo(INFO_UPSTEMP, "%s", upstemp);

	writeinfo();
}

void instcmd (int auxcmd, int dlen, char *data)
{
	/* TODO: reply to upsd? */

	switch (auxcmd) {
		case CMD_BTEST0:	/* stop battery test */
			upssendchar ('C');
			upssendchar ('T');
			upssendchar (ENDCHAR);
			break;
		case CMD_BTEST1:	/* start battery test */
			upssendchar ('T');
			upssendchar (ENDCHAR);
			break;
		default:
			upslogx(LOG_INFO, "instcmd: unknown type 0x%04x",
			        auxcmd);
	}
}

void setuphandlers(void)
{
	upsh.instcmd = instcmd;
}

static void run_debug()
{
	
	while (1)
		ups_update ();
}

static void usage(const char *prog)
{
	printf ("usage: %s [-hDqkr] [-d <num>] [-l <num>] -m <model> <device>\n", prog);
	printf ("-h		- display this help\n");
	printf ("-D		- run in debug mode\n");
	printf ("-q		- quiet: do not print to stdout\n");
	printf ("-k		- force shutdown\n");
	printf ("-r		- turn ups on again if utility returns\n");
	printf ("-d <num>	- wait <num> seconds after sending shutdown command\n");
	printf ("-l <num>	- line voltage, default 230\n");
	printf ("-m <model>	- model number\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
	printf ("\n");
	printf ("Example: %s -m 1000 /dev/ttyS0\n", prog);
	exit (1);
}

int main(int argc, char **argv)
{
	char	*prog, *portname;
	int	i;

	prog = argv[0];

	while ((i = getopt(argc, argv, "+d:qDrm:hk:l:")) != EOF) {
		switch (i) {
			case 'l':
				linenorm = atoi (optarg);
				break;
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'q':
				quiet = 1;
				break;
			case 'D':
				debug = 1;
				break;
			case 'r':
				restore = 1;
				break;
			case 'k':
				forceshutdown(optarg);
				break;
			case 'h':
				usage (prog);
				break;
			case 'm':
				model = strdup (optarg);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		usage (prog);
		exit (1);
	}
	if ( !model) {
		printf ("error: no model number given\n");
		exit (1);
	}
	
	if ( !quiet) {
		printf ("Network UPS Tools - ");
		printf ("AEG Protect driver v0.01 (%s)\n", UPS_VERSION);
	}
	openlog ("aeg", LOG_PID, LOG_FACILITY);
	
	if (debug)
		printf ("Running in debug mode\n");
	
	droproot();

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/aeg-%s", STATEPATH, portname);

	upsfd = open(argv[0], O_RDWR | O_NDELAY);
	if (upsfd == -1)
		fatal("open %s", argv[0]);

	setup_serial();
	ups_sync();
	
	/* ToDo: check if these ranges are
		 true for all models. */
	lowvolt = 1.66;
	highvolt = 2.30;
	voltrange = highvolt - lowvolt;

	initinfo();
	createmsgq();	/* try to create IPC message queue */
	setuphandlers();
	setmodel ("AEG", model);

        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST0);
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);
	if ( !quiet) 
		printf ("Detected %s %s on %s\n", getdata(INFO_MFR), getdata(INFO_MODEL), argv[0]);

	if (debug) {
		run_debug ();
	} else {
		background();
		while (1) {
			ups_update();

			if (getupsmsg(2))       /* TODO: remove debug scaffolding */
				upslogx(LOG_INFO, "Received a message from upsd");
		}
	}

	return 0;
}
