/* bestups.c - model specific routines for Best-UPS Fortress models

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "main.h"

#define ENDCHAR  13	/* replies end with CR */
#define UPSDELAY  5
#define MAXTRIES 10

static	char	*modelname, *va;
static	float	lowvolt = 0, highvolt = 0, voltrange = 0;
static	int	linenorm = 0, poll_failures = 0;
static	int	inverted_bypass_bit = 0;

static void setmodel(const char *abbr, const char *va)
{
	if (!strcmp(abbr, "FOR")) {
		setinfo(INFO_MFR, "%s", "Best Power");
		setinfo(INFO_MODEL, "Fortress %s", va);
		return;
	}

	if (!strcmp(abbr, "FTC")) {
		setinfo(INFO_MFR, "%s", "Best Power");
		setinfo(INFO_MODEL, "Fortress Telecom %s", va);
		return;
	}

	if (!strcmp(abbr, "PRO")) {
		setinfo(INFO_MFR, "%s", "Best Power");
		setinfo(INFO_MODEL, "Patriot Pro %s", va);
		inverted_bypass_bit = 1;
		return;
	}

	if (!strcmp(abbr, "PR2")) {
		setinfo(INFO_MFR, "%s", "Best Power");
		setinfo(INFO_MODEL, "Patriot Pro II %s", va);
		inverted_bypass_bit = 1;
		return;
	}

	if (!strcmp(abbr, "520")) {
		setinfo(INFO_MFR, "%s", "Sola Australia");
		setinfo(INFO_MODEL, "Sola 520 %s", va);
		return;
	}

	if (!strcmp(abbr, "620")) {
		setinfo(INFO_MFR, "%s", "Sola Australia");
		setinfo(INFO_MODEL, "Sola 620 %s", va);
		return;
	}

	setinfo(INFO_MFR, "%s", "Unknown");
	setinfo(INFO_MODEL, "Unknown %s (%s)", abbr, va);

	printf ("Unknown model detected - please report this ID: '%s'\n", abbr);

	return;
}

void instcmd (int auxcmd, int dlen, char *data)
{
	/* TODO: reply to upsd? */

	switch (auxcmd) {
		case CMD_BTEST0:	/* stop battery test */
			upssendchar ('C');
			upssendchar ('T');
			upssendchar (13);
			break;
		case CMD_BTEST1:	/* start battery test */
			upssendchar ('T');
			upssendchar (13);
			break;
		default:
			upslogx(LOG_INFO, "instcmd: unknown type 0x%04x", auxcmd);
	}
}

void upsdrv_initinfo(void)
{
	addinfo (INFO_MFR, "", 0, 0);
	addinfo (INFO_MODEL, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_OUTVOLT, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_BATTVOLT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_UPSTEMP, "", 0, 0);
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST0);
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);

	setmodel(modelname, va);
	printf("Detected %s %s on %s\n", getdata(INFO_MFR), getdata(INFO_MODEL), device_path);

	upsh.instcmd = instcmd;
}

static void setup_serial(void)
{	
	struct	termios	tio;

	if (tcgetattr(upsfd, &tio) == -1)
		fatal("tcgetattr");

	tio.c_iflag = IXON | IXOFF;
	tio.c_oflag = 0;
	tio.c_cflag = (CS8 | CREAD | HUPCL | CLOCAL);
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0; 

#ifdef HAVE_CFSETISPEED
	cfsetispeed(&tio, B2400);
	cfsetospeed(&tio, B2400);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	if (tcsetattr(upsfd, TCSANOW, &tio) == -1)
		fatal("tcsetattr");
}

void ups_sync(void)
{
	char	buf[256];
	int	tries = 0, ret;

	printf("Syncing with UPS: ");
	fflush(stdout);

	for (;;) {
		tries++;
		if (tries > MAXTRIES)
			fatalx("\nFailed - giving up...");

		printf(".");
		fflush(stdout);
		upssend("%s", "\rQ1\r");
		printf(".");
		fflush(stdout);
		sleep(UPSDELAY);
		printf(".");
		fflush(stdout);

		ret = upsrecv(buf, sizeof(buf), ENDCHAR, "");

		if ((ret > 0) && (buf[0] == '('))
			break;
	}

	printf(" done\n");
}

void upsdrv_shutdown(void)
{
	char	temp[256], stat[32];
	int	tries = 0, ret;

	/* basic idea: find out line status and send appropriate command */

	printf("Checking status: ");
	fflush(stdout);

	for (;;) {
		tries++;
		if (tries > MAXTRIES) {
			printf("\nFailed - giving up...\n");
			exit(1);
		}
		upssend("\rQ1\r");
		printf(".");
		fflush(stdout);
		sleep(UPSDELAY);
		printf(".");
		fflush(stdout);

		ret = upsrecv(temp, sizeof(temp), ENDCHAR, "");

		if ((ret > 0) && (temp[0] == '(') && (strlen(temp) == 46))
			break;
	}

	printf(" done\n");

	sscanf(temp, "%*s %*s %*s %*s %*s %*s %*s %s", stat);

	/* on battery: send S01<cr>, ups will return by itself on utility */
	/* on line: send S01R0001<cr>, ups will cycle and return soon */

	upssend("%s", "S01");

	if (stat[0] == '0') {			/* on line */
		printf ("On line, sending shutdown+return command...\n");
		upssend("%s", "R0001");
	}
	else
		printf ("On battery, sending normal shutdown command...\n");

	upssend("%s", "\r");	/* end sequence */
}

void ups_ident(void)
{
	char	buf[256], *ptr, *com;
	int	i, tries = 0, ret;

	printf ("Identifying UPS: ");
	fflush (stdout);

	for (;;) {
		tries++;
		if (tries > MAXTRIES) {
			printf ("\nFailed - giving up...\n");
			exit (1);
		}

		printf(".");
		fflush(stdout);
		upssend("%s", "\r\rID\r");
		printf(".");
		fflush(stdout);
		sleep(UPSDELAY);
		printf(".");
		fflush(stdout);

		ret = upsrecv(buf, sizeof(buf), ENDCHAR, "");

		if ((ret > 0) && (buf[0] != '(') && (strlen(buf) >= 25) &&
			(strlen(buf) <= 27))
			break;
	}

	printf(" done\n");

	modelname = va = NULL;

	/* FOR,750,120,120,20.0,27.6 */
	ptr = buf;

	for (i = 0; i < 6; i++) {
		com = strchr (ptr, ',');

		if (com)
			*com = '\0';

		switch (i) {
			case 0: modelname = xstrdup (ptr);
				break;
			case 1: va = xstrdup (ptr);
				break;
			case 2: linenorm = atoi(ptr);
				break;
			case 4: lowvolt = atof (ptr);
				break;
			case 5: highvolt = atof (ptr);
				voltrange = highvolt - lowvolt;
				break;
			default:
				break;
		}

		if (com)
			ptr = com + 1;
	}
}

void pollfail(char *why)
{
	poll_failures++;

	/* ignore the first few since these UPSes tend to drop characters */
	if (poll_failures == 3)
		upslogx(LOG_ERR, why);

	return;
}

void upsdrv_updateinfo(void)
{
	char	utility[16], outvolt[16], loadpct[16], acfreq[16], 
		battvolt[16], upstemp[16], stat[16], buf[256], 
		temp[VALSIZE];
	float	bvoltp;
	int	ret;

	upssend("%s", "\rQ1\r");
	sleep(UPSDELAY); 

	ret = upsrecv(buf, sizeof(buf), ENDCHAR, "");

	if (strlen(buf) < 46) {
		pollfail("Poll failed: short read from UPS");
		return;
	}

	if (strlen(buf) > 46) {
		pollfail("Poll failed: oversized read from UPS");
		return;
	}

	if (buf[0] != '(') {
		pollfail("Poll failed: invalid start character");
		return;
	}

	/* only say this if it got high enough to log a failure note */
	if (poll_failures >= 3)
		upslogx(LOG_NOTICE, "UPS poll succeeded");

	poll_failures = 0;

	sscanf(buf, "%*c%s %*s %s %s %s %s %s %s", utility, outvolt, 
		loadpct, acfreq, battvolt, upstemp, stat);
	
	bvoltp = ((atof (battvolt) - lowvolt) / voltrange) * 100.0;

	if (bvoltp > 100.0)
		bvoltp = 100.0;

	setinfo(INFO_BATTVOLT, "%s", battvolt);
	setinfo(INFO_UTILITY, "%s", utility);
	setinfo(INFO_OUTVOLT, "%s", outvolt);
	setinfo(INFO_LOADPCT, "%s", loadpct);
	setinfo(INFO_ACFREQ, "%s", acfreq);
	setinfo(INFO_UPSTEMP, "%s", upstemp);
	setinfo(INFO_BATTPCT, "%02.1f", bvoltp);

	strcpy(temp, "");

	if (stat[0] == '0')
		strcat(temp, "OL ");		/* on line */
	else
		strcat(temp, "OB ");		/* on battery */

	if (stat[1] == '1')
		strcat(temp, "LB ");		/* low battery */

	if (stat[2] == (inverted_bypass_bit ? '0' : '1')) {
		/* boost or trim in effect */
		if (atof(utility) < atof(outvolt))
			strcat(temp, "BOOST");

		if (atof(utility) > atof(outvolt))
			strcat(temp, "TRIM");
	}

	/* lose trailing space if present */
	if (temp[strlen(temp)-1] == ' ')
		temp[strlen(temp)-1] = 0;

	setinfo(INFO_STATUS, "%s", temp);

	writeinfo();
}

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Best UPS driver 0.45 (%s)\n", UPS_VERSION);
}

void upsdrv_initups(void)
{
	open_serial(device_path, B2400);
	setup_serial();

	/* don't let upscommon warn about it since it happens way too often */
	/* this driver does its own checking with better handling */
        flag_timeoutfailure = -1;

	ups_sync();
	ups_ident();
}

/* tell main how many entries we need */
int upsdrv_infomax(void)
{
	return 16;
}

