/* main.c - Network UPS Tools driver core

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "main.h"

	unsigned int sddelay = 90;	/* wait 90 seconds for shutdown */
	const char *progname = NULL, *upsname = NULL;
	const char *device_name = NULL, *device_path = NULL;
	const char *statepath = NULL;
	int	do_forceshutdown = 0;
	vartab_t	*vartab_h = NULL;
	int	experimental_driver = 0;

	/* stuff from upscommon */
	extern	int	do_lock_port;
	extern	char	*pidfn;
	extern	int	use_init_fn;	/* TODO: remove in the future */

	/* stuff from common */
	extern	int	upslog_flags;

/* power down the attached load immediately */
static void forceshutdown(void)
{
	upslogx(LOG_NOTICE, "Initiating UPS shutdown");
	printf("Initiating forced UPS shutdown!\n");

	do_lock_port = 0;
	upsdrv_shutdown();

	if (sddelay > 0) {
		printf("Waiting for poweroff...\n");
		sleep(sddelay);
		printf("Hmm, did the shutdown fail?  Oh well...\n");
		exit(1);
	}
	exit(0);
}

void usage(void)
{
	printf ("usage: %s [-h] [-a <id>] [-d <num>] [-k] [-x <var>=<val>] [<device>]\n", 
	        progname);
	printf("\n");
	printf("Examples:\n");
	printf("         %s -a myups       - new method: use ups.conf section myups\n",
		progname);
	printf("         %s /dev/ttyS0     - old method: start up on port /dev/ttyS0\n", 
		progname);
	exit(1);
}

void help(void)
{
	vartab_t	*tmp;

	printf ("usage: %s [-h] [-a <id>] [-d <num>] [-k] [-x <var>=<val>] [<device>]\n", 
	        progname);

	printf (
		"  -a <id>        - autoconfig using ups.conf section <id>\n"
		"                 - note: -x options after -a override ups.conf settings\n"
		"  -d <num>       - wait <num> seconds after sending shutdown command\n"
		"  -k             - force shutdown\n"
		"  -h             - display this help\n"
		"  -x <var>=<val> - set driver variable <var> to <val>\n"
		"                 - example: -x cable=940-0095B\n"
		"\n"
		"  <device>       - /dev entry corresponding to UPS port\n"
		"                 - Only optional when using -a!\n"
		"\n");

	if (vartab_h) {
		tmp = vartab_h;

		printf ("Acceptable values for -x in this driver:\n\n");

		while (tmp) {
			if (tmp->vartype == VAR_VALUE)
				printf("%40s : -x %s=<value>\n", 
				       tmp->desc, tmp->var);
			else
				printf("%40s : -x %s\n", tmp->desc, tmp->var);
			tmp = tmp->next;
		}
	}

	upsdrv_help();
	exit(1);
}

/* cram var [= <val>] data into storage */
static void storeval(const char *var, char *val)
{
	vartab_t	*tmp, *last;

	tmp = last = vartab_h;

	while (tmp) {
		last = tmp;

		/* sanity check */
		if (!tmp->var) {
			tmp = tmp->next;
			continue;
		}

		/* later definitions overwrite earlier ones */
		if (!strcasecmp(tmp->var, var)) {
			if (tmp->val)
				free(tmp->val);

			if (val)
				tmp->val = xstrdup(val);

			tmp->found = 1;
			return;
		}

		tmp = tmp->next;
	}

	printf("Error: -x %s is not valid for this driver.\n\n", var);
	help();
}

/* retrieve the value of variable <var> if possible */
char *getval(const char *var)
{
	vartab_t	*tmp = vartab_h;

	while (tmp) {
		if (!strcasecmp(tmp->var, var))
			return(tmp->val);
		tmp = tmp->next;
	}

	return NULL;
}

/* see if <var> has been defined, even if no value has been given to it */
int testvar(const char *var)
{
	vartab_t	*tmp = vartab_h;

	while (tmp) {
		if (!strcasecmp(tmp->var, var))
			return tmp->found;
		tmp = tmp->next;
	}

	return 0;	/* not found */
}

/* callback from driver - create the table for -x/conf entries */
void addvar(int vartype, const char *name, const char *desc)
{
	vartab_t	*tmp, *last;

	tmp = last = vartab_h;

	while (tmp) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(vartab_t));

	tmp->vartype = vartype;
	tmp->var = xstrdup(name);
	tmp->val = NULL;
	tmp->desc = xstrdup(desc);
	tmp->found = 0;
	tmp->next = NULL;

	if (last)
		last->next = tmp;
	else
		vartab_h = tmp;
}	

/* handle -x / ups.conf config details that are for this part of the code */
static int main_arg(char *var, char *val)
{
	/* flags for main: just 'nolock' for now */

	if (!strcmp(var, "nolock")) {
		do_lock_port = 0;
		return 1;	/* handled */
	}

	/* any other flags are for the driver code */
	if (!val)
		return 0;

	/* variables for main: port and sddelay */

	if (!strcmp(var, "port")) {
		device_path = xstrdup(val);
		device_name = xbasename(device_path);
		return 1;	/* handled */
	}

	if (!strcmp(var, "sddelay")) {
		sddelay = atoi(val);
		return 1;	/* handled */
	}

	/* only for upsdrvctl - ignored here */
	if (!strcmp(var, "sdorder"))
		return 1;	/* handled */

	return 0;	/* unhandled, pass it through to the driver */
}

void do_upsconf_args(char *confupsname, char *var, char *val)
{
	/* ignore global stuff in here */
	if (!confupsname)
		return;

	/* no match = not for us */
	if (strcmp(confupsname, upsname) != 0)
		return;

	if (main_arg(var, val))
		return;

	/* flags (no =) now get passed to the driver-level stuff */
	if (!val) {
		storeval(var, NULL);
		return;
	}

	/* don't let the user shoot themselves in the foot */
	if (!strcmp(var, "driver")) {
		if (strcmp(val, progname) != 0)
			fatalx("Error: UPS [%s] is for driver %s, but I'm %s!\n",
				confupsname, val, progname);
		return;
	}

	/* everything else must be for the driver */

	storeval(var, val);
}

/* split -x foo=bar into 'foo' and 'bar' */
static void splitxarg(char *inbuf)
{
	char	*eqptr, *val, *buf;

	/* make our own copy - avoid changing argv */
	buf = xstrdup(inbuf);

	eqptr = strchr(buf, '=');

	if (!eqptr)
		val = NULL;
	else {
		*eqptr++ = '\0';
		val = eqptr;
	}

	/* see if main handles this first */
	if (main_arg(buf, val))
		return;

	/* otherwise store it for later */
	storeval(buf, val);
}

/* two pipes for two different sync points during startup */

#define SERIALIZE_INIT 1
#define SERIALIZE_SETM1 2
#define SERIALIZE_SETM2 3
#define SERIALIZE_WAITM1 4
#define SERIALIZE_WAITM2 5

void serialize(int op)
{
	static	int	pipe1[2], pipe2[2];
	int	ret;
	char	ch;

	switch (op) {

		case SERIALIZE_INIT:
			ret = pipe(pipe1);

			if (ret != 0) {
				fatal("serialize: pipe");
				exit(1);
			}

			ret = pipe(pipe2);

			if (ret != 0) {
				fatal("serialize: pipe");
				exit(1);
			}

			break;

		case SERIALIZE_SETM1:
			close(pipe1[0]);
			close(pipe1[1]);
			break;

		case SERIALIZE_WAITM1:
			close(pipe1[1]);
			ret = read(pipe1[0], &ch, 1);
			close(pipe1[0]);
			break;

		case SERIALIZE_SETM2:
			close(pipe2[0]);
			close(pipe2[1]);
			break;

		case SERIALIZE_WAITM2:
			close(pipe2[1]);
			ret = read(pipe2[0], &ch, 1);
			close(pipe2[0]);
			break;
	}
}

static void runparent(void)
{
	close_statefd();

	/* allow the client to exit from its WAITM1 call */
	serialize(SERIALIZE_SETM1);

	/* wait for the client to call info_ready() and set M2 */
	serialize(SERIALIZE_WAITM2);

	_exit(0);
}


/* special version to deal with the nasty sync issues in here */
static void main_background(void)
{
	int	pid;

	if ((pid = fork()) < 0)
		fatal("Unable to enter background");

	xbit_set(&upslog_flags, UPSLOG_SYSLOG);
	xbit_clear(&upslog_flags, UPSLOG_STDERR);

	close(0);
	close(1);
	close(2);

	if (pid != 0)
		runparent();

	/* child */

	/* make fds 0-2 point somewhere defined */
	if (open("/dev/null", O_RDWR) != 0)
		fatal("open /dev/null");
	dup(0);
	dup(0);

#ifdef HAVE_SETSID
	setsid();		/* make a new session to dodge signals */
#endif

	/* wait until after parent closes the statefd */
	serialize(SERIALIZE_WAITM1);

	/* load up info with a full set of data */
	upsdrv_updateinfo();

	/* put the state file where it's supposed to be */
	info_ready();

	/* let the parent exit now that we're ready to roll */
	serialize(SERIALIZE_SETM2);

	upslogx(LOG_INFO, "Startup successful");
}

int main(int argc, char **argv)
{
	int	i;
#ifdef HAVE_MMAP
	int 	shmok = 0;
#else
	int 	shmok = 1;
#endif

	droproot();

	upsdrv_banner();

	if (experimental_driver) {
		printf("Warning: This is an experimental driver.\n");
		printf("Some features may not function correctly.\n\n");
	}

	progname = xbasename(argv[0]);
	openlog(progname, LOG_PID, LOG_FACILITY);

	if ((statepath = getenv("NUT_STATEPATH")) == NULL)
		statepath = STATEPATH;

	/* build the driver's extra (-x) variable table */
	upsdrv_makevartable();

	while ((i = getopt(argc, argv, "+a:d:kDhx:")) != EOF) {
		switch (i) {
			case 'a':
				upsname = optarg;
				read_upsconf(1);
				break;
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'k':
				do_forceshutdown = 1;
				break;
			case 'D':
				nut_debug_level++;
				break;
			case 'V':
				printf("Network UPS Tools (%s)\n", UPS_VERSION);
				exit(0);
			case 'x':
				splitxarg(optarg);
				break;
			case 'h':
				help();
				break;
			default:
				usage();
				break;
		}
	}

	argc -= optind;
	argv += optind;
	optind = 1;

	/* we need to get the port from somewhere */
	if (argc < 1) {
		if (!device_path) {
			fprintf(stderr, "Error: You must specify a port name in ups.conf or on the command line.\n");
			usage();
		}
	}

	/* allow argv to override the ups.conf entry if specified */
	else {
		device_path = argv[0];
		device_name = xbasename(device_path);
	}

	snprintf(statefn, sizeof(statefn), "%s/%s-%s", 
		statepath, progname, device_name);

	pidfn = xmalloc(SMALLBUF);
	snprintf(pidfn, SMALLBUF, "%s/%s-%s.pid", 
		ALTPIDPATH, progname, device_name);

	upsdebugx(1, "debug level is '%d'", nut_debug_level);

	upsdrv_initups();

	if (do_forceshutdown)
		forceshutdown();

	if (chdir(statepath))
		fatal("Can't chdir to %s", statepath);

	/* TODO: remove when all old drivers are dead */
	use_init_fn = 1;

	create_info(upsdrv_infomax(), shmok);
	createmsgq();

	upsdrv_initinfo();

	if (nut_debug_level == 0) {
		serialize(SERIALIZE_INIT);
		main_background();
		writepid(pidfn);
	} else {

		/* no sync to worry about, so rename it and get going */
		info_ready();

	}

	for (;;) {
		upsdrv_updateinfo();

		if (getupsmsg(2))       
			upslogx(LOG_DEBUG, "Received a message from upsd");
	}

	return 0;
}
