/*  mge-ellipse.c - monitor MGE UPS for NUT with SHUT protocol 
 *                  (like Ellipse models)
 * 
 *  Copyright (C) 2001-2002 Philippe Marzouk <philm@users.sourceforge.net>
 *  some parts are Copyright (C) Russel Kroll <rkroll@exploits.org>
 *  some parts are Copyright (C) Canon driver developpers for Gphoto
 *         see http://www.gphoto.org for details
 *  All rights reserved.
 *
 * $Id: mge-ellipse.c,v 1.2 2002/02/15 22:56:11 phil Exp phil $
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *            59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


#include <sys/ioctl.h>

#include "main.h"
#include "timehead.h"
#include "mge-ellipse.h"

const struct shutdata commands[] = {
  {0xA1, 0x01, 0x02},		/* PRESENT_STATUS */
  {0xA1, 0x01, 0x16},		/* BATT_CAPACITY */
  {0x21, 0x09, 0x0F},		/* SET_SHUTDOWN_DELAY */
  {0xA1, 0x01, 0x0F},		/* GET_SHUTDOWN_DELAY  */
  {0x21, 0x09, 0x0C},		/* SET_LOW_BATT_LEVEL */
  {0xA1, 0x01, 0x0C},		/* GET_LOW_BATT_LEVEL */
  {0xA1, 0x01, 0x0E},		/* PERCENT_LOAD */
  {0x21, 0x09, 0x11},		/* SET_STARTUP_DELAY */
  {0xA1, 0x01, 0x11},		/* GET_STARTUP_DELAY */
  {0xA1, 0x01, 0x10},		/* IPRODUCT */
  {0}
};

#define MAX_TRY	4

/* global variables */
int shmok = 1;
int debug = 0;
int commstatus = 0;
int lowbatt = 30;
int ondelay = 13;
int offdelay = 120;

void
upsdrv_initinfo (void)
{
  if (debug)
    upslogx (LOG_DEBUG, "entering initinfo()\n");

  create_info (INFOMAX, shmok);

  addinfo (INFO_MFR, "MGE UPS", FLAG_STRING, 0);
  /*FIXME To be obtained from the UPS */
  addinfo (INFO_MODEL, "Ellipse", FLAG_STRING, 0);
  addinfo (INFO_STATUS, "", FLAG_STRING, 0);
  addinfo (INFO_LOADPCT, "", 0, 0);
  addinfo (INFO_RUNTIME, "", 0, 0);
  addinfo (INFO_BATTPCT, "", 0, 0);
  addinfo (INFO_BADPACKS, "", 0, 0);
  addinfo (INFO_BATT_CHARGE, "", 0, 0);
  addinfo (INFO_ALRM_BADOUTPUT, "", 0, 0);
  addinfo (INFO_ALRM_UPSSHUT, "", 0, 0);
  addinfo (INFO_ALRM_OVERLOAD, "", 0, 0);
  addinfo (INFO_ALRM_GENERAL, "", 0, 0);
  addinfo (INFO_GRACEDELAY, "", 0, 0);
  addinfo (INFO_WAKEDELAY, "", 0, 0);
  addinfo (INFO_INSTCMD, "", 0, CMD_SHUTDOWN);

  upsh.instcmd = instcmd;
}

void
upsdrv_updateinfo (void)
{

  if (debug)
    upslogx (LOG_DEBUG, "entering upsdrv_updateinfo()");

  if (commstatus == 0) {
    if (e_ups_start () != 0) {
      upslogx(LOG_NOTICE, "No communication with UPS, retrying");
      sleep (10);
    } else {
      if (debug)
        upslogx (LOG_NOTICE, "Communication with UPS established");

      /* e_identify_ups(); */
			
      if (lowbatt != 30)
        e_set_lowbatt ();
							
      if (e_set_autorestart () != 0)
        ondelay = 13;		/* default value */
			
			/* informational as the setting of this value will start
			 * the shutdown process */
      setinfo (INFO_GRACEDELAY, "%i", offdelay);
			
      writeinfo ();
		}
  } else {
    e_request_ups ();

    writeinfo ();
  }
}


void
upsdrv_shutdown (void)
{
  unsigned char *pkt, data[3];
  int try, ret = 0;

  if (debug)
    upslogx (LOG_DEBUG, "entering upsdrv_shutdown()\n");
	
	pkt = malloc (11); /* Max size of a packet */
  if (!pkt) {
	  upslogx(LOG_NOTICE, "(RQ) error while trying to allocate 11 bytes!!");
		/* we have no other choice than exit at this time */
		exit(-1);
	}

	memset (pkt, 0x00, 11);
	
  /* if called right after itself, communication will not work immediatly 
   * so we add some delay */
  for (try = 0; try < MAX_TRY; try++) {
    if (e_ups_start () == 0)
      break;
    sleep (2);
  }

  if (commstatus == 0)
    printf ("No communication with UPS, trying to continue...\n");

  if (e_set_autorestart () != 0)
    printf
      ("Could not set autorestart, the UPS may not restart when power comes back\n");


  /* programming shutdown duration (120 seconds by default) */


  data[0] = offdelay & 0xff;
  data[1] = offdelay >> 8;
  data[2] = 0;			/* 65535 is a maximum */

  e_packet_send (SET_SHUTDOWN_DELAY, data, 3);

  /* verify the UPS got it right */
  e_packet_send (GET_SHUTDOWN_DELAY, NULL, 0);

	for (try=0; try < MAX_TRY; try++) {
    ret = e_packet_recv (GET_SHUTDOWN_DELAY, pkt);
		if ((ret == 0) || (ret == -1))
      break;
	}

	if (ret != 0)
    upslogx(LOG_NOTICE, "could not communicate with UPS correctly");

  if (pkt) {
    if ((offdelay - 1) == (pkt[3] | pkt[4] << 8)) {
      upslogx (LOG_NOTICE, "Shutdown delay set to %i", offdelay);
    }
    else {
      upslogx (LOG_NOTICE,
	       "Could not set shutdown delay, default is 120 seconds");
      printf
	("Could not set shutdown delay, the UPS will shut after the default 120 seconds");
    }
  }

	free (pkt);
}

void
upsdrv_help (void)
{
  if (debug)
    upslogx (LOG_DEBUG, "entering upsdrv_help");
}

/* list flags and values that you want to receive via -x */
void
upsdrv_makevartable (void)
{
  if (debug)
    upslogx (LOG_DEBUG, "entering upsdrv_makevartable()");

  addvar (VAR_VALUE, "lowbatt", "Set low battery level, in % (default=30).");
  addvar (VAR_VALUE, "offdelay", "Shutdown delay, in seconds (default=120).");
  addvar (VAR_VALUE, "ondelay", "Startup delay, in ten seconds units (default=13).");
  addvar (VAR_VALUE, "debug", "Turn debugging on");
}

void
upsdrv_banner (void)
{
  printf ("Network UPS Tools - MGE Ellipse UPS driver 0.4 (%s)\n\n",
	  UPS_VERSION);
}

void
upsdrv_initups (void)
{
  if (debug)
    upslogx (LOG_DEBUG, "entering upsdrv_initups()");


  /* initialize serial port */
  open_serial (device_path, B2400);

  setline (1);

  /* get debugging */
  if (testvar ("debug"))
    debug++;

  /* get battery lowlevel */
  if (getval ("lowbatt"))
    lowbatt = atoi (getval ("lowbatt"));

  /* on delay */
  if (getval ("ondelay"))
    ondelay = atoi (getval ("ondelay"));
  /* shutdown delay */
  if (getval ("offdelay"))
    offdelay = atoi (getval ("offdelay"));
}

/* tell main how many items you need */
int
upsdrv_infomax (void)
{
  /* every addinfo() consumes an entry.  plan ahead, but don't go
   * overboard, since this is an easy way to waste memory.
   */

  return INFOMAX;
}

void
instcmd (int auxcmd, int datalen, char *data)
{

  switch (auxcmd) {
  case CMD_SHUTDOWN:
    upsdrv_shutdown ();
    break;
  default:
    upslogx (LOG_NOTICE, "%s", "instcmd: unknown type 0x%04x");
    break;
  }

}


/*
 * set RTS to on and DTR to off
 * 
 * set : 1 to set comm
 * set : 0 to stop comm
 */
void
setline (int set)
{
  int dtr = TIOCM_DTR;
  int rts = TIOCM_RTS;

  if (debug)
    upslogx (LOG_DEBUG, "setline(%i)\n", set);

  if (set == 1) {
    ioctl (upsfd, TIOCMBIC, &dtr);
    ioctl (upsfd, TIOCMBIS, &rts);
  }
  else {
    ioctl (upsfd, TIOCMBIS, &dtr);
    ioctl (upsfd, TIOCMBIC, &rts);
  }
}

/*****************************************************************************
 * e_ups_start ()
 * 
 * initiate communication with the UPS
 *
 * return 0 on success, -1 on failure
 *
 *****************************************************************************/
int
e_ups_start ()
{
  unsigned char c;
  int try;

  if (debug)
    upslogx (LOG_DEBUG, "entering e_ups_start()\n");

  for (try = 0; try < MAX_TRY; try++) {
    if ((serial_send (PING, L_PING)) == -1) {
      upslogx (LOG_NOTICE, "Communication error while writing to port");
      return -1;
    }
    c = serial_read (1000);
    if (c == 0x16) {
      commstatus = 1;
      return 0;
    }
  }


  commstatus = 0;
  return -1;
}

/*****************************************************************************
 * e_request_ups ()
 * 
 * Handle requests to the UPS and call the right function if a notification
 * is received
 * 
 * return 0 on success, -1 on failure
 * 
 *****************************************************************************/
int
e_request_ups ()
{
  unsigned char *pkt; 
	int try, ret = 0, command = PERCENT_LOAD;

  if (commstatus == 0)
    return -1;

  if (debug)
    upslogx (LOG_DEBUG, "entering e_request_ups ()\n");
	
	pkt = malloc (11); /* Max size of a packet */
  if (!pkt) {
	  upslogx(LOG_NOTICE, "(RQ) error while trying to allocate 11 bytes!!");
		/* maybe we should exit at this time ? */
		return -1;
	}

	memset (pkt, 0x00, 11);

  while ( commstatus != 0 ) {
	  switch (command) {
      case PRESENT_STATUS:
        command = BATT_CAPACITY;
	      break;
	    case BATT_CAPACITY:
	      command = PERCENT_LOAD;
	      break;
	    default:
	      command = PRESENT_STATUS;
	      break;
	  }
	
		
	  ret = e_packet_send (command, NULL, 0);
    if (ret != 0) {
      if (ret == -3) { /* notification being received */
        usleep(500);
        serial_send ("\x15", 1); /* send a NACK to ask for a resend */
      } else {
	      if (debug)
	        upslogx (LOG_DEBUG, "(RQ) error while sending");
				
	      commstatus = 0;
				free (pkt);
	      return -1;
	    }
    }

		for (try=0; try < MAX_TRY; try++) {
	  	ret = e_packet_recv (command, pkt);
			if (ret == -1)           /* we might be receiving a notification */
        serial_send("\x15",1); /*	so let's send a NACK and try again */
			if (ret == 0)
        break;
		}
		
		if (ret != 0) {
        free (pkt);
        commstatus = 0;
        return -1;
		}
		
	  if (pkt) {
      dump_hex ("RECV", pkt, 11);

	    if (pkt[0] == 0x85) /* Notification received */
		    command--; /* We'll have to resend the command */
				
		  switch (pkt[2]) {
			  case 0x02:
			    e_ups_status (pkt);
			    break;
			  case 0x16:
				  e_autonomy (pkt);
				  break;
			  case 0x0e:
				  e_ups_load (pkt);
				  break;
			  default:
				  upslogx(LOG_NOTICE, "I do not know how to handle response type %x", pkt[2]);
				  break;
			}

			writeinfo();
			sleep(2);
	  }
	}

	  free(pkt);
    commstatus = 0;
    return -1;
}

/*****************************************************************************
 * e_identify_ups ()
 * 
 * get the product info from the UPS
 *
 * return 0 on success, -1 on failure
 *
 *****************************************************************************/
int
e_identify_ups ()
{
  unsigned char *pkt;
	int try, ret = 0;

  if (commstatus == 0)
    return -1;

  if (debug)
    upslogx (LOG_DEBUG, "entering e_identify_ups()\n");
	
	pkt = malloc (11); /* Max size of a packet */
  if (!pkt) {
	  upslogx(LOG_NOTICE, "(ID) error while trying to allocate 11 bytes!!");
		/* maybe we should exit at this time ? */
		return -1;
	}
	
	memset (pkt, 0x00, 11);

  if (e_packet_send (IPRODUCT, NULL, 0) != 0) {
    if (debug)
      upslogx (LOG_DEBUG, "(ID) error while sending");
		
		free (pkt);
    commstatus = 0;
    return -1;
  }

	for (try=0; try < MAX_TRY; try++) {
    ret = e_packet_recv (IPRODUCT, pkt);
		if ((ret == 0) || (ret == -1))
      break;
	}

	if (ret != 0) {
    free (pkt);
    commstatus = 0;
    return -1;
  }
	
  if (pkt) {
    return 0;
  }
	
	free (pkt);
  commstatus = 0;
  return -1;
}

/*****************************************************************************
 * e_ups_status (unsigned char *pkt)
 * 
 * Get general status of the UPS
 *
 * pkt       - response packet from the UPS
 * 
 * return nothing
 * 
 *****************************************************************************/
void
e_ups_status (unsigned char *pkt)
{
  char value[255] = { 0 };

  if (1 == (pkt[3] >> 0 & 01))
    strcat (value, "OL ");

  setinfo (INFO_BATT_CHARGE, "%i", pkt[3] >> 1 & 01);

  if (1 == (pkt[3] >> 2 & 01))
    strcat (value, "OB ");

  if (1 == (pkt[3] >> 3 & 01))
    strcat (value, "LB ");

  setinfo (INFO_BADPACKS, "%i", pkt[3] >> 4 & 01);

  if ((pkt[3] >> 5 & 01) == 1)
    setinfo (INFO_ALRM_BADOUTPUT, "%i", 0);
  else
    setinfo (INFO_ALRM_BADOUTPUT, "%i", 1);

  setinfo (INFO_ALRM_UPSSHUT, "%i", pkt[3] >> 6 & 01);

  setinfo (INFO_ALRM_OVERLOAD, "%i", pkt[3] >> 7 & 01);

  setinfo (INFO_ALRM_GENERAL, "%i", pkt[3] >> 8 & 01);

  if (value[strlen (value) - 1] == ' ')
    value[strlen (value) - 1] = 0;
  setinfo (INFO_STATUS, "%s", value);

}

/*****************************************************************************
 * e_ups_load (unsigned char *pkt)
 * 
 * Get load of the UPS
 *
 * pkt       - response packet from the UPS
 *
 * return nothing
 * 
 *****************************************************************************/
void
e_ups_load (unsigned char *pkt)
{
  setinfo (INFO_LOADPCT, "%i", pkt[4]);
}

/*****************************************************************************
 * e_autonomy (unsigned char *pkt)
 * 
 * Get UPS autonomy
 *
 * pkt       - response packet from the UPS
 * 
 * return nothing
 * 
 *****************************************************************************/
void
e_autonomy (unsigned char *pkt)
{
  setinfo (INFO_RUNTIME, "%i", ((pkt[4] & 0xFF) | (pkt[5] << 8)) / 60);
  setinfo (INFO_BATTPCT, "%u", pkt[3]);
}

/****************************************************************************
 * e_set_autorestart ()
 * 
 * select Autorestart mode
 *
 * return 0 on success, -1 on failure, -2 on unsupported
 * 
 *****************************************************************************/
int
e_set_autorestart ()
{
  unsigned char *pkt, data[3];
	int ret = 0;

  if (debug)
    upslogx (LOG_DEBUG, "entering e_set_autorestart()\n");

  if (commstatus == 0)
    return -1;
	
	pkt = malloc (11); /* Max size of a packet */
  if (!pkt) {
	  upslogx(LOG_NOTICE, "(AU) error while trying to allocate 11 bytes!!");
		/* maybe we should exit at this time ? */
		return -1;
	}

	memset (pkt, 0x00, 11);

  data[0] = ondelay & 0xff;
  data[1] = ondelay >> 8;
  data[2] = 0;			/* 65535 is a maximum */

  if (e_packet_send (SET_STARTUP_DELAY, data, 3) != 0) {
    free (pkt);
    commstatus = 0;
    return -1;
  }

  /* verify the UPS got it right */
  ret = e_get_autorestart();
	if ((ret == 0) || (ret == -2))
    return ret;

	free (pkt);
  commstatus = 0;
  return -1;
}

/****************************************************************************
 * e_get_autorestart ()
 * 
 * get Autorestart value
 *
 * return 0 on success, -1 on failure, -2 on unsupported
 * 
 *****************************************************************************/
int
e_get_autorestart ()
{
  unsigned char *pkt;
  int try, ret = 0;
	
	if (debug)
    upslogx (LOG_DEBUG, "entering e_get_autorestart()\n");

  if (commstatus == 0)
    return -1;
	
	pkt = malloc (11); /* Max size of a packet */
  if (!pkt) {
	  upslogx(LOG_NOTICE, "(AU) error while trying to allocate 11 bytes!!");
		/* maybe we should exit at this time ? */
		return -1;
	}

	memset (pkt, 0x00, 11);

	if (e_packet_send (GET_STARTUP_DELAY, NULL, 0) != 0) {
    free (pkt);
    commstatus = 0;
    return -1;
  }

	pkt = malloc (11); /* Max size of a packet */
  if (!pkt) {
	  upslogx(LOG_NOTICE, "(GAU) error while trying to allocate 11 bytes!!");
		/* maybe we should exit at this time ? */
		return -1;
	}

	memset (pkt, 0x00, 11);


  for (try=0; try < MAX_TRY; try++) {
  ret = e_packet_recv (GET_STARTUP_DELAY, pkt);
		if ((ret == 0) || (ret == -1))
      break;
	}

	if (ret != 0) {
    free (pkt);
		commstatus = 0;
		return -1;
	}
	
  if (pkt) {
    if ((ondelay - 1) == (pkt[3] | pkt[4] << 8)) {
      setinfo (INFO_WAKEDELAY, "%i", ondelay);
			free (pkt);
      return 0;
    }
    else {
      upslogx (LOG_NOTICE, "Could not set startup delay");
      setinfo (INFO_WAKEDELAY, "%i", (pkt[3] | pkt[4] << 8) + 1);
			free (pkt);
      return -1;
    }
  }

  free (pkt);
  commstatus = 0;
  return -1;
}

/****************************************************************************
 * e_set_lowbatt ()
 * 
 * set low battery threshold (default is 30%)
 *
 * return 0 on success, -1 on failure, -2 on unsupported
 * 
 *****************************************************************************/
int
e_set_lowbatt ()
{
  unsigned char *pkt, data[3];
	int ret = 0;

  if (debug)
    upslogx (LOG_DEBUG, "entering e_set_lowbatt()\n");

  if (commstatus == 0)
    return -1;

	pkt = malloc (11); /* Max size of a packet */
  if (!pkt) {
	  upslogx(LOG_NOTICE, "(LB) error while trying to allocate 11 bytes!!");
		/* maybe we should exit at this time ? */
		return -1;
	}

	memset (pkt, 0x00, 11);
	
  data[0] = lowbatt & 0xff;
  data[1] = lowbatt >> 8;
  data[2] = 0;			/* 65535 is a maximum */

  if (e_packet_send (SET_LOWBATT_LEVEL, data, 3) != 0) {
    free (pkt);
    commstatus = 0;
    return -1;
  }

  /* verify the UPS got it right */
	ret = e_get_lowbatt();
	if ((ret == 0) || (ret == -2))
    return ret;


  free (pkt);
  commstatus = 0;
  return -1;
}

/****************************************************************************
 * e_get_lowbatt ()
 * 
 * get low battery threshold (default is 30%)
 *
 * return 0 on success, -1 on failure, -2 on unsupported
 * 
 *****************************************************************************/
int
e_get_lowbatt ()
{
  unsigned char *pkt;
	int try, ret = 0;

  if (debug)
    upslogx (LOG_DEBUG, "entering e_get_lowbatt()\n");

  if (commstatus == 0)
    return -1;

	pkt = malloc (11); /* Max size of a packet */
  if (!pkt) {
	  upslogx(LOG_NOTICE, "(GLB) error while trying to allocate 11 bytes!!");
		/* maybe we should exit at this time ? */
		return -1;
	}

	memset (pkt, 0x00, 11);
	
	if (e_packet_send (GET_LOWBATT_LEVEL, NULL, 0) != 0) {
    free (pkt);
    commstatus = 0;
    return -1;
  }

	for (try=0; try < MAX_TRY; try++) {
    ret = e_packet_recv (GET_LOWBATT_LEVEL, pkt);
		if ((ret == 0) || (ret == -1))
      break;
	}

  if (ret != 0) {
    free (pkt);
		commstatus = 0;
		return -1;
	}
	
  if (pkt) {
    if ((lowbatt - 1) == (pkt[3] | pkt[4] << 8)) {
      upslogx (LOG_NOTICE, "Low battery level set to %i", lowbatt);
      return 0;
    }
    else {
      upslogx (LOG_NOTICE, "Lowbatt level should be set but verify failed");
      return 0; /* FIXME the value returned by this query do no match
                          with what other send */
    }
  }

  free (pkt);
  commstatus = 0;
  return -1;
}
	
/*****************************************************************************
 * e_wait_ack()
 *
 * wait for an ACK packet
 *
 * returns 0 on success, -1 on error, -2 on NACK, -3 on NOTIFICATION
 * 
 *****************************************************************************/
int
e_wait_ack (void)
{
  int try;
  unsigned char c;

  if (debug)
    upslogx (LOG_DEBUG, "entering e_wait_ack()\n");

  for (try = 0; try < MAX_TRY; try++) {
    c = serial_read (DEFAULT_TIMEOUT);
    if (c == 0x06) {
      if (debug)
	upslogx (LOG_DEBUG, "ACK received");
      return 0;
    }
    else if (c == 0x15)
      return -2;
    else if (c == 0x85)
      return -3;
  }

  return -1;
}

/****************************************************************************
 * char_read (char *bytes, int size, int read_timeout)
 *
 * reads size bytes from the serial port
 * 
 * bytes     - buffer to store the data
 * size      - size of the data to get
 * read_timeout - serial timeout (in milliseconds)
 * 
 * return -1 on error, -2 on timeout, 0 on success
 * 
 *****************************************************************************/
int
char_read (char *bytes, int size, int read_timeout)
{
  struct timeval serial_timeout;
  fd_set readfs;
  int readen = 0;
  int rc = 0;

  FD_ZERO (&readfs);
  FD_SET (upsfd, &readfs);

  serial_timeout.tv_usec = (read_timeout % 1000) * 1000;
  serial_timeout.tv_sec = (read_timeout / 1000);

  rc = select (upsfd + 1, &readfs, NULL, NULL, &serial_timeout);
  if (0 == rc)
    return -2;			/* timeout */
  if (FD_ISSET (upsfd, &readfs)) {
    int now = read (upsfd, bytes, size - readen);

    if (now < 0) {
      return -1;
    }
    else {
      bytes += now;
      readen += now;
    }
  }
  else {
    return -1;
  }
  return readen;
}

/***************************************************************************
 * serial_read (int read_timeout)
 *  
 * return data one byte at a time
 *
 * read_timeout - serial timeout (in milliseconds)
 * 
 * returns the byte on success, -1 on error, -2 on timeout
 * 
 ****************************************************************************/
int
serial_read (int read_timeout)
{
  static unsigned char cache[512];
  static unsigned char *cachep = cache;
  static unsigned char *cachee = cache;
  int recv;
  /* if still data in cache, get it */
  if (cachep < cachee) {
    return (int) *cachep++;
  }

  recv = char_read (cache, 1, read_timeout);

  if (recv == -1)
    return -1;

  if (recv == -2)
    return -2;

  cachep = cache;
  cachee = cache + recv;
  cachep = cache;
  cachee = cache + recv;

  if (recv) {
    /*   fprintf(stderr,"received: %x\n",(int) *cachep);  */
    return (int) *cachep++;
  }

  return -1;
}

/*****************************************************************************
 * serial_send (char *buf, int len)
 *
 * write the content of buf to the serial port
 *
 * buf       - data to send
 * len       - lenght of data to send
 * 
 * returns number of bytes written on success, -1 on error
 * 
 *****************************************************************************/
int
serial_send (char *buf, int len)
{
  tcflush (upsfd, TCIFLUSH);
  dump_hex ("sent", buf, len);
  return write (upsfd, buf, len);
}

/*****************************************************************************
 * e_packet_send(int command, char *data, int datalen)
 * 
 * creates the data packet to send to the ups
 *
 * command   - from shutdataType
 * data      - additionnal data to send if multi-part command (NULL else)
 * datalen   - length of additionnal data (<=3)
 *
 * returns 0 on success, -1 on failure
 *
 *****************************************************************************/
int
e_packet_send (int command, char *data, int datalen)
{
  char pkt[11] = { 0 };
  int i = 0, try = 0;

  if (debug)
    upslogx (LOG_DEBUG, "entering e_packet_send()\n");

  if (!data)
    pkt[0] = 0x81;
  else
    pkt[0] = 0x01;

  pkt[1] = 0x88;		/* first command packet has always 8 data bytes */

  pkt[2] = commands[command].cmd_type1;
  pkt[3] = commands[command].cmd_type2;

  pkt[4] = commands[command].reqid;

  pkt[5] = 0x03;		/* maybe the report type should not be hardcoded ? */

  /* interface LSB and MSB, always 0 (at least for Ellipse models */
  pkt[6] = 0x00;
  pkt[7] = 0x00;

  if (!data)
    pkt[8] = 0x20;		/* this is from the spec, snooping gave me 0x20 */
  else				/* SET_REPORT CASE */
    pkt[8] = 0x04;		/* length of data packet, I only know of 4 bytes command */
  /* so let's hardcode it for the moment */

  pkt[9] = 0x00;		/* length MSB, always 0 */

  /* checksum */
  for (i = 2; i < 10; i++)
    pkt[10] ^= pkt[i];

  for (try = 1; try < MAX_TRY; try++) {
    if (serial_send (pkt, 11) < 0) {
      return -1;
    }
    i = e_wait_ack ();
    if (i == 0) {
      if (debug)
	upslogx (LOG_DEBUG, "received ACK");
      break;
    }
    else if ((i == -1) || (i == -3)) {
      return i;
    }
  }

  /* if after MAX_TRY we keep receiving NACK then we got a comm problem */
  if (i == -2)
    return -1;

  /* process additional data */
  if (!data) {
    return 0;
  }
  else {
    pkt[0] = 0x81;
    pkt[1] = 0x44;
    pkt[2] = commands[command].reqid;
    /* data */
    for (i = 0; i < datalen; i++)
      pkt[3 + i] = data[i];
    /* checksum */
    for (i = 2; i <= 2 + datalen; i++)
      pkt[3 + datalen] ^= pkt[i];

    for (try = 1; try < MAX_TRY; try++) {
      if (serial_send (pkt, 3 + datalen + 1) < 0)
	return -1;
      i = e_wait_ack ();
      if (i == 0) {
	if (debug)
	  upslogx (LOG_DEBUG, "received ACK for data");
	return 0;
      }
      else if (i == -1)
	return -1;
    }
  }

  return -1;
}

/*****************************************************************************
 * e_packet_recv(int command, unsigned char *pkt)
 * 
 * get a packet back from the ups and acknowledge it
 *
 * command   - from shutdataType
 * pkt       - where to store the packet received
 *
 * return 0 on success, -1 on failure, -2 on NACK
 *
 *****************************************************************************/
int
e_packet_recv (int command, unsigned char *pkt)
{
  unsigned char c;
  int i, try, j, checksum = 0;

  if (debug)
    upslogx (LOG_DEBUG, "entering e_packet_recv()\n");

  i = 0;
  try = 0;
  while (try < MAX_TRY) {
    c = serial_read (DEFAULT_TIMEOUT);
    switch (c) {
    case -1:			/* error while reading serial port */
      upslogx (LOG_NOTICE, "error reading from port");
      return -1;
      break;
    case -2:			/* timeout */
      try++;
      break;
    case 0x15:			/* NACK received */
      if (i == 0)
        return -2;
      break;
    default:
      if ((i == 2) && (c != commands[command].reqid) && (pkt [0] != 0x85)) {
        if (debug)
	        upslogx (LOG_DEBUG, "received %x while expecting %x",
		             c, commands[command].reqid);
          return -1;
        }

      pkt[i] = c;
      i++;
      try = 0;

      if (i >= 2) {
	if (i == ((pkt[1] / 0x11) + 3)) {
	  for (j = 2; j < i - 1; j++) {
	    checksum ^= pkt[j];
	  }

	  if (pkt[i - 1] == checksum) {
	    if (debug)
	      upslogx (LOG_DEBUG, "sending ACK");
	    serial_send ("\x06", 1);
	    return 0;
	  }
	  else {
	    serial_send ("\x15", 1);
	    upslogx (LOG_NOTICE, "communication error, checksum invalid");
	    return -2;
	  }
	}
      }
    }
  }
  return -1;
}


/*****************************************************************************
 *
 * dump_hex
 *
 * Dumps a memory area as hex on the screen.
 *
 * msg  - Info message for the dump
 * buf  - the memory buffer to dump
 * len  - length of the buffer
 *
 ****************************************************************************/


#define NIBBLE(_i)    (((_i) < 10) ? '0' + (_i) : 'A' + (_i) - 10)

void
dump_hex (const char *msg, const unsigned char *buf, int len)
{
  int i;
  int nlocal;
  const unsigned char *pc;
  char *out;
  const unsigned char *start;
  char c;
  char line[100];

  if (debug == 0) {
    return;
  }
  start = buf;
/*  fprintf (stderr, "%s: (%d bytes)\n", msg, len); */
	upslogx (LOG_DEBUG, "%s: (%d bytes)", msg, len);
  while (len > 0) {
    snprintf (line, sizeof (line), "%08lx: ", (unsigned long) (buf - start));
    out = line + 10;

    for (i = 0, pc = buf, nlocal = len; i < 16; i++, pc++) {
      if (nlocal > 0) {
	c = *pc;

	*out++ = NIBBLE ((c >> 4) & 0xF);
	*out++ = NIBBLE (c & 0xF);

	nlocal--;
      }
      else {
	*out++ = ' ';
	*out++ = ' ';
      }				/* end else */
      *out++ = ' ';
    }				/* end for */

    *out++ = '-';
    *out++ = ' ';

    for (i = 0, pc = buf, nlocal = len;
	 (i < 16) && (nlocal > 0); i++, pc++, nlocal--) {
      c = *pc;

      if ((c < ' ') || (c >= 126)) {
	c = '.';
      }

      *out++ = c;
    }				/* end for */

    *out++ = 0;

/*    fprintf (stderr, "%s\n", line); */
		upslogx(LOG_DEBUG, "%s", line);

    buf += 16;
    len -= 16;
  }				/* end while */
}				/* end dump */
