/* optiups.c - model specific routines for Opti-UPS units

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define INFOMAX 16
#define UPS_DELAY 150000
#define ENDCHAR 10
#define IGNCHARS ""

#define DRIVER_VERSION "0.11.alpha.1"
#define SYSLOG_DEBUG 1

	int	shmok = 1, maxbattvolts = 0, cap_upstemp = 0;
	float	lowvolt = 0, voltrange;
	int	sddelay = 10;	/* wait 10 seconds for shutdown */
	int	shutdowncmdset = 0; /* The shutdown command set to use */


void printheader(void)
{
	printf ("Network UPS Tools %s - Opti-UPS driver %s\n", UPS_VERSION, DRIVER_VERSION);
	printf ("modified by Ted A. Kuban <teddy@ladograd.ru>");
	openlog ("optiups", LOG_PID, LOG_FACILITY);

	printf ("\n*** EXPERIMENTAL VERSION ***\n");
	printf ("Please send strangeness/success/bug reports so this can become\n");
	printf ("a stable version soon.\n\n");
}

void usage(const char *prog)
{
	printf ("usage: %s [-h] [-t <command>][-d <num>] [-s <num>] [-k] <device>\n", prog);
	printf ("Example: %s /dev/cuaa1\n", prog);
	exit (1);
}

void help (const char *prog)
{
	printheader();
	printf ("usage: %s [-h] [-t <command>][-d <num>] [-s <num>] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-d <num> - wait <num> seconds after sending shutdown command\n");
	printf ("-h       - display this help\n");
	printf ("-k       - force shutdown\n");
	printf ("-s <num> - use shutdown command set <num> for forced shutdown\n");
	printf ("           0 - combined (default), 1 - simple, 2 - delayed\n");
	printf ("-t       - send custom command to port and print result\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
	exit (1);
}


/* send command , wait for an answer 
   in the form <data><endchar> and store in buf */
void upsrequest(char *upscmd, char *buf, int buflen)
{
	int	i = -1;
#ifdef SYSLOG_DEBUG
	upslogx(LOG_DEBUG, "Sending %s", upscmd);
#endif
	while	(upscmd[++i] != 0) {
		upssendchar (upscmd[i]);
	}
	usleep (UPS_DELAY);
	upsrecv (buf, buflen, ENDCHAR, IGNCHARS);
#ifdef SYSLOG_DEBUG
	upslogx(LOG_DEBUG, "Get %s", buf);
#endif
}

void initinfo (void)
{
	create_info(INFOMAX, shmok);

	addinfo(INFO_MFR, "", 0, 0);
	addinfo(INFO_MODEL, "", 0, 0);
	addinfo(INFO_STATUS, "", 0, 0);
	addinfo(INFO_UTILITY, "", 0, 0);
	addinfo(INFO_LOADPCT, "", 0, 0);
	addinfo(INFO_UPSTEMP, "", 0, 0);
	addinfo(INFO_BATTPCT, "", 0, 0);
	addinfo(INFO_LOWXFER, "", 0, 0);
	addinfo(INFO_HIGHXFER, "", 0, 0);
	addinfo(INFO_ACFREQ, "", 0, 0);
}

void openser(void)
{
	struct	termios	tio;
	int	fd, flags;
 
	fd = open("/dev/ttyS0", O_RDWR | O_NONBLOCK);

	if (fd < 0)
		fatal("Port not available.");

	tio.c_cflag = 0 | CREAD | CS8 | HUPCL | CLOCAL;
	tio.c_iflag = 0 | IXANY | IMAXBEL | IXOFF;
	tio.c_oflag = 0 | ONLCR;
	tio.c_lflag = 0 | ECHOE | ECHOKE | ECHOCTL | PENDIN;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

#ifdef HAVE_CFSETISPEED
	cfsetispeed (&tio, B2400);
	cfsetospeed (&tio, B2400);
#endif

	tcsetattr (fd, TCSANOW, &tio);

	flags = 0;
	ioctl (fd, TIOCMSET, &flags);

	upsfd = fd;
}

void getbaseinfo (char *port)
{
	char	temp[256], mch[8], mtemp[8];
	int	flags, sl;

	/* kill DTR and RTS */

	flags = 0;
	ioctl (upsfd, TIOCMSET, &flags);

	/* manufacturer */

	upsrequest ("IM", temp, sizeof(temp));
	setinfo(INFO_MFR, "%s", temp);

	/* model parsing */

	upsrequest ("IO", temp, sizeof(temp));
	sl = strlen (temp);
	snprintf (mch, sizeof(mch), "%c%c", temp[sl-2], temp[sl-1]);
	strlcpy(mtemp, temp, sizeof(mtemp));
	mtemp [sl-2] = 0;

	switch (mch[0]) {
		case 'E': setinfo(INFO_MODEL, "PowerES %ses", mtemp);
			  break;

		case 'P': setinfo(INFO_MODEL, "PowerPS %sps", mtemp);
			  break;

		case 'V': setinfo(INFO_MODEL, "PowerVS %svs", mtemp);
			  break;

		default: setinfo(INFO_MODEL, "Unknown (%s)", temp);
			  break;
	}

	printf ("Detected %s %s on %s\n", getdata(INFO_MFR), getdata(INFO_MODEL), port);

	/* low transfer */
	upsrequest ("FL", temp, sizeof(temp));
	setinfo(INFO_LOWXFER, "%s", temp);

	/* high transfer */
	upsrequest ("FH", temp, sizeof(temp));
	setinfo(INFO_HIGHXFER, "%s", temp);
	return;
}

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
{
	char	temp[256], temp2[256];
	int	tval, battv;
	float	battf;

	upsrequest ("AG", temp, sizeof(temp));
	tval = strtol (temp, 0, 16);

	strcpy (temp2, "");

	if (tval & 8)
		strcat (temp2, "OVER ");	/* overload */
	if (tval & 16)
		strcat (temp2, "RB ");		/* replace batt */
	if (tval & 32)
		strcat (temp2, "OB ");		/* on battery */
	else
		strcat (temp2, "OL ");		/* on line */
	if (tval & 64)
		strcat (temp2, "LB ");		/* low battery */

	if (temp2[strlen(temp2)-1] == ' ')
		temp2[strlen(temp2)-1] = 0;
	setinfo(INFO_STATUS, "%s", temp2);

	/* input voltage */
	upsrequest ("NV", temp, sizeof(temp));
	setinfo(INFO_UTILITY, "%s", temp);

	/* ups load percent */
	upsrequest ("OL", temp, sizeof(temp));
	setinfo(INFO_LOADPCT, "%s", temp);

	/* ups temp */
	upsrequest ("BT", temp, sizeof(temp));
	setinfo(INFO_UPSTEMP, "%s", temp);

	/* battery charge */
	upsrequest ("BV", temp, sizeof(temp));

	/* battery voltage range: 10.4 - 13.0 VDC */
	battv = atoi(temp);
	battf = (((battv / 10.0) - 10.4) / 2.6) * 100.0;
	if (battf > 100.0)
		battf = 100.0;
	setinfo(INFO_BATTPCT, "%2.1f", battf);

	/* input freq */
	upsrequest ("FF", temp, sizeof(temp));
	setinfo(INFO_ACFREQ, "%2.1f", atof(temp) / 10.0);	

	writeinfo();
}

/* power down the attached load immediately */
void forceshutdown(char *port)
{
	char	temp[256];
	int	tval;

	upslogx(LOG_INFO, "Initiating UPS shutdown");
	printf ("Initiating forced UPS shutdown on port %s!\n", port);

	open_serial (port, B2400);

	upsrequest ("IM", temp, sizeof(temp));
	printf("UPS Model - %s", temp);
	if (strcmp(temp, "OPTI-UPS") != 0)
		printf ("Detection failed.  Trying a shutdown command anyway.\n");

	/* check the line status */

	upsrequest ("AG", temp, sizeof(temp));
	tval = strtol (temp, 0, 16);

	/* Send a combined set of shutdown commands which can work better */
	/* if the UPS gets power during shutdown process */
	/* Specifically it sends both the soft shutdown 'S' */
	/* and the powerdown after grace period - '@000' commands */
	printf ("UPS - currently %s - sending shutdown/powerdown\n",
	       (tval & 16) ? "on battery" : "on-line");
	upsrequest("BT", temp, sizeof(temp));
	if (sddelay > 0) {
		sleep (sddelay);
		upsrequest("BV", temp, sizeof(temp)); /* delayed shutdown */
		printf ("Hmm, did the shutdown fail?  Oh well...\n");
		upsrequest ("AG", temp, sizeof(temp));
		tval = strtol (temp, 0, 16);
		printf ("UPS - currently %s",
		       (tval & 16) ?	"on battery - waiting for empty battery\n"
					: "on-line - rebooting\n");
		exit (1);
	}
	exit (0);
}

int main (int argc, char **argv)
{
	char	*portname, *prog, temp[256];
	char	*testcommand = NULL;
	int	i;
	int	killpower = 0;

	/* TODO: getopt */
	prog = argv[0];

	printheader();

	while ((i = getopt(argc, argv, "+hkd:s:t:")) != EOF) {
		switch (i) {
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'k':
				printf("This function are in testing!\n");
				killpower = 1;
				/* forceshutdown(optarg); */
				break;
			case 'h':
				help(prog);
				break;
			case 's':
				shutdowncmdset = atoi(optarg);
				break;
			case 't':
				testcommand = optarg;
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1) {
		printf ("Error: no device specified!\n");
		usage (prog);
	}

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL)
		fatalx("Unable to abbreviate %s", argv[0]);

	if (killpower == 1) {
		forceshutdown(argv[0]);
		exit(0);
	}


	droproot();

	snprintf (statefn, sizeof(statefn), "%s/optiups-%s", STATEPATH,
	          portname);

	/* FIXME: why is open_serial so stupid in this module? */
	open_serial (argv[0], B2400);
/*	openser(); */

	if (testcommand != NULL) {
		upsrequest (testcommand, temp, sizeof(temp));
		printf("Command %s; UPS return:\n", testcommand);
		printf("%s\n", temp);
		exit(0);
	}

	initinfo();

	createmsgq();		/* try to create IPC message queue */

	getbaseinfo(argv[0]);

 	background();

	for (;;) {
		updateinfo();
		/* wait up to 2 seconds for a message from the upsd */
		if (getupsmsg(2))
			upslogx(LOG_INFO, "Received a message from upsd");
	}
}
