/* Copyright (C) 2002-2004  Mark Andrew Aikens <marka@desert.cx>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: IO.cxx,v 1.11 2004/08/22 16:04:55 marka Exp $
 */
using namespace std;

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdexcept>
#include "config.h"
#include "IO.h"
#ifdef ENABLE_LINUX_PPDEV
#  include "LinuxPPDevIO.h"
#endif
#include "DirectPPIO.h"
#include "Util.h"


IO *IO::acquire(char *name, int port) {
#ifdef ENABLE_LINUX_PPDEV
	if(strcasecmp(name, "LinuxPPDev") == 0) {
		return new LinuxPPDevIO(port);
	}
#endif
	if(strcasecmp(name, "DirectPP") == 0) {
		return new DirectPPIO(port);
	}
	throw runtime_error("Unknown IO driver selected");
}


IO::IO(int port) {
	long value;

	if(config->get_integer("io", "tdly_stretch", &value)) {
		this->tdly_stretch = value;
	} else {
		this->tdly_stretch = 0;
	}

	if(config->get_integer("io", "tset_stretch", &value)) {
		this->tset_stretch = value;
	} else {
		this->tset_stretch = 0;
	}

	if(config->get_integer("io", "thld_stretch", &value)) {
		this->thld_stretch = value;
	} else {
		this->thld_stretch = 0;
	}

	if(config->get_integer("io", "gen_stretch", &value)) {
		this->gen_stretch = value;
	} else {
		this->gen_stretch = 0;
	}
}


IO::~IO() {
}


void IO::usleep(unsigned long us) {
	struct timeval now, later;

	us += this->gen_stretch;
	if(us >= 10000) {	/* 10ms */
		later.tv_sec = us / 1000000;
		later.tv_usec = us % 1000000;
		select(0, NULL, NULL, NULL, &later);
		return;
	}

	gettimeofday(&later, NULL);
	later.tv_sec += us / 1000000;
	later.tv_usec += us % 1000000;
	while(later.tv_usec > 1000000) {
		later.tv_usec -= 1000000;
		later.tv_sec += 1;
	}
	while(1) {
		gettimeofday(&now, NULL);
		if((now.tv_sec > later.tv_sec) ||
		  ((now.tv_sec == later.tv_sec) && (now.tv_usec > later.tv_usec)))
			break;
	}
}


void IO::shift_bits_out(uint32_t bits, int numbits, int tset, int thold) {
	while(numbits > 0) {
		this->set_clk(true);
		this->set_data(bits & 0x01);
		this->usleep(tset + this->tset_stretch);/* Delay for data setup time */
		this->set_clk(false);                    /* Falling edge */
		this->usleep(thold + this->thld_stretch);/* Delay for data hold time */
		bits >>= 1;
		numbits--;
	}
}


uint32_t IO::shift_bits_in(int numbits, int tdly) {
	uint32_t data, mask;

	data = 0;
	mask = 0x00000001;
	this->set_data(true);
	while(numbits > 0) {
		this->set_clk(true);
		this->usleep(tdly + this->tdly_stretch);
		if(this->read_data())
			data |= mask;
		this->set_clk(false);
		this->usleep(tdly + this->tdly_stretch);
		mask <<= 1;
		numbits--;
	}
	this->set_data(false);
	return data;
}
