/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "StateManager.h"
#include <cstdlib>
#include <cassert>
#include <iostream>
#include <fstream>
#include <map>
#include "OmGtk.h"
#include "PortModel.h"
#include "PluginModel.h"
#include "PatchController.h"
#include "PatchModel.h"
#include "OmGtkApp.h"
#include "OmPatchBayArea.h"
#include "Controller.h"

using std::cerr; using std::cout; using std::endl;
using std::map; using std::string;
using LibOmClient::PatchModel;

namespace OmGtk {

using namespace LibOmClient;


StateManager::StateManager()
: m_patch_path("/usr/share/om/patches:/usr/local/share/om/patches"),
  m_audio_port_color(0x3F5885FF),
  m_control_port_color(0x3F8558FF)
{
}


StateManager::~StateManager()
{
}


/** Loads settings from the rc file.  Passing no parameter will load from
 * the default location.
 */
void
StateManager::load_settings(string filename)
{	
	if (filename == "")
		filename = string(getenv("HOME")).append("/.omgtkrc");
	
	std::ifstream is;
	is.open(filename.c_str(), std::ios::in);

	if ( ! is.good()) {
		cout << "[StateManager] Unable to open settings file " << filename << endl;
		return;
	} else {
		cout << "[StateManager] Loading settings from " << filename << endl;
	}

	string s;

	is >> s;
	if (s != "file_version") {
		cerr << "[StateManager] Corrupt settings file, load aborted." << endl;
		is.close();
		return;
	}
	
	is >> s;
	if (s != "1") {
		cerr << "[StateManager] Unknown settings file version number, load aborted." << endl;
		is.close();
		return;
	}
	
	is >> s;
	if (s != "patch_path") {
		cerr << "[StateManager] Corrupt settings file, load aborted." << endl;
		is.close();
		return;
	}

	is >> s;
	m_patch_path = s;
	
	apply_settings();

	is.close();
}


/** Saves settings to rc file.  Passing no parameter will save to the
 * default location.
 */
void
StateManager::save_settings(string filename)
{
	if (filename == "")
		filename = string(getenv("HOME")).append("/.omgtkrc");
	
	std::ofstream os;
	os.open(filename.c_str(), std::ios::out);
	
	if ( ! os.good()) {
		cout << "[StateManager] Unable to write to setting file " << filename << endl;
		return;
	} else {
		cout << "[StateManager] Saving settings to " << filename << endl;
	}
	
	os << "file_version 1" << endl;
	os << "patch_path " << m_patch_path << endl;
	
	os.close();
}


/** Applies the current loaded settings to whichever parts of the app
 * need updating (ie setting all the patch windows to be anti-alised).
 */
void
StateManager::apply_settings()
{
	controller->set_patch_path(m_patch_path);
}


int
StateManager::get_port_color(const PortModel* pi)
{
	assert(pi != NULL);
	
	if (pi->is_control()) {
		return m_control_port_color;
	} else if (pi->is_audio()) {
		return m_audio_port_color;
	}
	
	cerr << "[StateManager] Unknown port type!  Port will be bright red, this is an error." << endl;
	return 0xFF0000FF;
}

/*
Coord
StateManager::get_window_location(const string& id) 
{
	return m_window_locations[id];
}


void
StateManager::set_window_location(const string& id, Coord loc) 
{
	m_window_locations[id] = loc;
}


Coord
StateManager::get_window_size(const string& id) 
{
	return m_window_sizes[id];
}


void
StateManager::set_window_size(const string& id, Coord size) 
{
	m_window_sizes[id] = size;
}*/


} // namespace OmGtk
