/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef CONNECTION_H
#define CONNECTION_H

#include <cstdlib>
#include "Array.h"
#include "MaidObject.h"
#include "types.h"
#include "OutputPort.h"

namespace Om {

class Port;
class InputPort;


/** A class to represent a single inbound connection for an InputPort.
 *
 * This can be a group of ports (ie coming from a polyphonic node) or
 * a single port.  This class exists basically as an abstraction of mixing
 * down polyphonic inputs, so InputPort can just deal with mixing down
 * multiple input sources without worrying about polyphonic stuff.
 *
 * \ingroup engine
 */
class Connection : public MaidObject
{
public:
	Connection(OutputPort* const src_port, InputPort* const dst_port);
	virtual ~Connection();

	void prepare_buffers();

	OutputPort* const src_port() const { return m_src_port; }
	InputPort*  const dst_port() const { return m_dst_port; }

	/** Used by some (recursive) events to prevent double disconnections */
	bool pending_disconnection()       { return m_pending_disconnection; }
	void pending_disconnection(bool b) { m_pending_disconnection = b; }
	
	/** Get the buffer for a particular voice.
	 * A Connection is smart - it knows the destination port requesting the
	 * buffer, and will return accordingly (ie the same buffer for every voice
	 * in a mono->poly connection).
	 */
	inline const sample* const buffer(uint voice) const
	{
		if (m_is_poly_to_mono) {
			return m_local_buffer;
		} else {
			if (m_src_port->poly() == 1)
				return m_src_port->buffer(0);
			else
				return m_src_port->buffer(voice);
		}
	}
	
private:
	// Disallow copies
	Connection(const Connection& copy)
	: m_src_port(NULL), m_dst_port(NULL)
	{ exit(EXIT_FAILURE); }
	Connection& operator=(const Connection& copy) { exit(EXIT_FAILURE); }

	OutputPort* const m_src_port;
	InputPort* const  m_dst_port;
	sample*           m_local_buffer;  // Only used for poly->mono connections
	bool              m_is_poly_to_mono;
	size_t            m_buffer_size;
	bool              m_pending_disconnection;
};


} // namespace Om

#endif // CONNECTION_H
