/*
 * 
 * Oregano, a tool for schematic capture and simulation of electrical circuits.
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999, 2000, 2001  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "config.h"
#include <gnome.h>
#include <glade/glade.h>
#include <locale.h>
#include <signal.h>
#include "main.h"
#include "dialogs.h"
#include "schematic.h"
#include "schematic-view.h"
#include "cursors.h"
#include "load-library.h"
#include "load-schematic.h"
#include "load-common.h"
#include "oregano-config.h"
#include "stock.h"

OreganoApp oregano; 
static char **startup_files = NULL;
int oregano_debugging;

static void
quit (void)
{
	gtk_main_quit ();
}

static gboolean
quit_hook (GtkObject *object,
	   guint signal_id,
	   guint n_params,
	   GtkArg *params,
	   gpointer data)
{
	gtk_main_quit ();
	return TRUE;
}

static void
session_die (void)
{
	/* FIXME: The session is ending. Save unsaved work etc. */
	quit ();
}

int
main (int argc, char **argv)
{
	GnomeClient *client;
	Schematic *schematic;
	SchematicView *schematic_view;
	char *msg;
	int i;

	poptContext ctx;
	const struct poptOption oregano_popt_options [] = {
		{ "debug", '\0', POPT_ARG_INT, &oregano_debugging, 0,
		  N_("Enables some debugging functions"), N_("LEVEL") },
		{ NULL, '\0', 0, NULL, 0 }
	};

	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

	gnome_init_with_popt_table ("oregano", VERSION, argc, argv, oregano_popt_options, 0, &ctx);

	cursors_init ();
	glade_gnome_init ();
	stock_init ();

	/* Disable the stupid crash dialog for now. */
	signal (SIGSEGV, SIG_DFL);

	oregano_config_load ();

	gtk_widget_set_default_colormap (gdk_rgb_get_cmap ());
	gtk_widget_set_default_visual (gdk_rgb_get_visual ());

	if (!g_file_exists (OREGANO_GLADEDIR "/sim-settings.glade")) {
		msg = g_strdup_printf (_("You seem to be running Oregano without\n"
					 "having it installed properly on your system.\n\n"
					 "Please install Oregano and try again."));

		oregano_error (msg);
		g_free (msg);
		return 1;
	}

	setlocale (LC_NUMERIC, "C");

	/* Connect to session manager. */
	client = gnome_master_client ();
	gtk_signal_connect (GTK_OBJECT (client), "die", 
			    GTK_SIGNAL_FUNC (session_die), NULL);

	oregano_lookup_libraries();
	if (oregano.libraries == NULL) {
		oregano_error (_("Could not find a parts library.\n\n"
				 "This is probably due to a faulty installation\n"
				 "of Oregano. Please check your installation."));
		return 1;
	}

	oregano.clipboard = NULL;

	schematic = NULL;

	startup_files = (char **) poptGetArgs (ctx);
	if (startup_files)
		for (i = 0; startup_files[i]; i++) {
			Schematic *new_schematic;
			char *fname = startup_files[i];

			new_schematic = schematic_read (fname);
			if (new_schematic) {
				schematic_view = schematic_view_new (new_schematic);
				gtk_widget_show_all (schematic_view->toplevel);
				schematic_set_filename (new_schematic, fname);
				schematic_set_title (new_schematic, fname);
				while (gtk_events_pending ()) /* Show something. */
					gtk_main_iteration ();
				schematic = new_schematic;
			}
		}

	poptFreeContext (ctx);

	if (schematic == NULL){
		schematic = schematic_new ();
		schematic_view = schematic_view_new (schematic);
		gtk_widget_show_all (schematic_view->toplevel);
	}

	gtk_signal_add_emission_hook (
		gtk_signal_lookup ("last_schematic_destroyed", TYPE_SCHEMATIC),
		quit_hook,
		NULL);

	gtk_main ();
	cursors_shutdown ();
	gnome_config_drop_all ();
	return 0;
}

