/* pam_tmpdir module, based on pam_env which in turn was based on
   pam_mail */

/*
 * $Id: pam_env.c,v 1.3 1999/11/08 05:46:53 morgan Exp $
 * 
 * Written by Tollef Fog Heen <tollef@add.no> 2001-02-03
 *
 * Based on pam_env
 * Written by Dave Kinchlea <kinch@kinch.ark.com> 1997/01/31
 * Inspired by Andrew Morgan <morgan@parc.power.net, who also supplied the 
 * template for this file (via pam_mail)
 */


#include <ctype.h>
#include <errno.h>
#include <pwd.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

/*
 * here, we make a definition for the externally accessible function
 * in this file (this definition is required for static a module
 * but strongly encouraged generally) it is used to instruct the
 * modules include file to define the function prototypes.
 */

#define PAM_SM_SESSION      /* This is primarily a AUTH_SESSION module */
#define PAM_SM_AUTH         /* But I like to be friendly */
#define PAM_SM_PASSWORD     /*        ""                 */
#define PAM_SM_ACCOUNT      /*        ""                 */

#include <security/pam_modules.h>
#include <security/_pam_macros.h>

#define SYSUSRTMP "/tmp/user"

static int set_environment(pam_handle_t *pamh);
static int make_tmp_directory(pam_handle_t *pamh);

uid_t get_user_id(pam_handle_t *pamh) {
  char *username = malloc(25);
  struct passwd* user_entry;
  
  pam_get_item(pamh,PAM_USER,(void*)&username);
  user_entry = getpwnam(username);
  return user_entry->pw_uid;
}

gid_t get_group_id(pam_handle_t *pamh) {
  char *username = malloc(25);
  struct passwd* user_entry;
  
  pam_get_item(pamh,PAM_USER,(void*)&username);
  user_entry = getpwnam(username);
  return user_entry->pw_gid;
}


/* some syslogging */

static void _log_err(int err, const char *format, ...)
{
    va_list args;

    va_start(args, format);
    openlog("PAM_tmpdir", LOG_CONS|LOG_PID, LOG_AUTH);
    vsyslog(err, format, args);
    va_end(args);
    closelog();
}

/* argument parsing */

#define PAM_DEBUG_ARG       0x01
#define PAM_NEW_CONF_FILE   0x02
#define PAM_ENV_SILENT      0x04
#define PAM_NEW_ENV_FILE    0x10

static int set_environment(pam_handle_t *pamh) {
  char buf[512];
  char foo[24];
  char logbuf[1024];
  int ret;

  snprintf(buf,512,"TMP=%s/%d",SYSUSRTMP,get_user_id(pamh));
  ret = pam_putenv(pamh, buf);

  snprintf(buf,512,"TMPDIR=%s/%d",SYSUSRTMP,get_user_id(pamh));
  ret = pam_putenv(pamh, buf);
  return 0;
}

static int make_tmp_directory(pam_handle_t *pamh) {
  int ret;
  char buf[512];
  char logbuf[1024];
  struct stat statbuf;
  mode_t old_umask;

  ret = lstat(SYSUSRTMP,&statbuf);
  if (ret == -1 && errno != ENOENT) {
    snprintf(logbuf,sizeof logbuf,"lstat SYSUSRTMP failed: %s\n", strerror(errno));
    _log_err(LOG_NOTICE, "%s", logbuf);
    return -1;
  } else if (ret != -1 && statbuf.st_uid != geteuid()) {
    /* Somebody else than root has grabbed /tmp/user.  Bad, bad, bad. */
    snprintf(logbuf,sizeof logbuf,"%s is owned by uid %d instead of root. " 
	    "Failed to create safe $TMPDIR\n", SYSUSRTMP, statbuf.st_uid);
    _log_err(LOG_ERR, "%s", logbuf);
    return -1;
  } else if (ret != -1 && !S_ISDIR(statbuf.st_mode)) {
    snprintf(logbuf,sizeof logbuf,"%s is not a directory. Failed to create safe $TMPDIR\n", SYSUSRTMP);
    _log_err(LOG_NOTICE, "%s", logbuf);
    return -1;
  } else if (ret != -1 && ((statbuf.st_mode & S_IWGRP) || (statbuf.st_mode & S_IWOTH))) {
    snprintf(logbuf,sizeof logbuf,"%s is group or world writable. "
	     "Failed to create safe $TMPDIR\n", SYSUSRTMP);
    _log_err(LOG_NOTICE, "%s", logbuf);
    return -1;
  } else if (ret != -1 && !(statbuf.st_mode & S_IXOTH)) {
    snprintf(logbuf,sizeof logbuf,"%s is not world searchable. "
	     "Failed to create safe $TMPDIR\n", SYSUSRTMP); 
    _log_err(LOG_NOTICE, "%s", logbuf);
    return -1;
  } else if (ret == -1 && errno == ENOENT) {
    old_umask = umask(0000);
    if (mkdir(SYSUSRTMP,0711) == -1) {
      snprintf(logbuf,sizeof logbuf,"mkdir SYSUSRTMP failed: %s\n", strerror(errno));
      _log_err(LOG_NOTICE, "%s", logbuf);
      return -1;
    }
    umask(old_umask);
    if (chown(SYSUSRTMP,0,0) == -1) {
      snprintf(logbuf,sizeof logbuf,"chown 0:0 SYSUSRTMP failed: %s\n", strerror(errno));
      _log_err(LOG_NOTICE, "%s", logbuf);
      return -1;
    }
  }

  if (snprintf(buf, sizeof buf, "%s/%d",SYSUSRTMP,get_user_id(pamh)) == -1) {
    return -1;
  }
  ret = lstat(buf,&statbuf);
  if (ret == -1 && errno != ENOENT) {
    snprintf(logbuf,sizeof logbuf,"lstat %s failed: %s\n", buf, strerror(errno));
    return -1;
  } else if (ret != -1 && statbuf.st_uid != get_user_id(pamh)) {
    snprintf(logbuf,sizeof logbuf,"%s owned by uid %d instead of uid %d. "
	     "Failed to create safe $TMPDIR\n", buf, statbuf.st_uid, get_user_id(pamh));
    _log_err(LOG_NOTICE, "%s", logbuf);
    return -1;
  } else if (ret != -1 && !S_ISDIR(statbuf.st_mode)) {
    snprintf(logbuf,sizeof logbuf,"%s is not a directory. Failed to create safe $TMPDIR\n", buf);
    _log_err(LOG_NOTICE, "%s", logbuf);
    return -1;
  } else if (ret == -1 && errno == ENOENT) {
    if (mkdir(buf,0700) == -1) {
      snprintf(logbuf,sizeof logbuf,"mkdir %s failed: %s\n", buf, strerror(errno));
      _log_err(LOG_NOTICE, "%s", logbuf);
      return -1;
    }
    if (chown(buf,get_user_id(pamh),get_group_id(pamh)) == -1) {
      snprintf(logbuf,sizeof logbuf,"chown %d:%d %s failed: %s\n",
	       get_user_id(pamh),get_group_id(pamh),buf, strerror(errno));
      _log_err(LOG_NOTICE, "%s", logbuf);
      return -1;
    }
  }
  if (ret != -1 && ((statbuf.st_mode & S_IWGRP) || (statbuf.st_mode & S_IWOTH))) {
    snprintf(logbuf,sizeof logbuf,"Warning: %s is group or world writable. "
	     "Changing permissions to 0700\n", buf);
    _log_err(LOG_NOTICE, "%s", logbuf);
    if (chmod(buf, 0700) == -1) {
      snprintf(logbuf,sizeof logbuf,"chmod 0700 %s failed: %s\n",buf, strerror(errno));
      _log_err(LOG_NOTICE, "%s", logbuf);
      return -1;
    }
  }
  return 0;
}

/* --- authentication management functions (only) --- */

PAM_EXTERN
int pam_sm_authenticate(pam_handle_t *pamh, int flags, int argc,
			const char **argv)
{ 
  return PAM_IGNORE;
}

PAM_EXTERN 
int pam_sm_setcred(pam_handle_t *pamh, int flags, int argc, 
		   const char **argv)
{

  if (make_tmp_directory(pamh) == 0) {
    set_environment(pamh);
    return PAM_SUCCESS;
  } else {
    return PAM_ABORT;
  }
}

PAM_EXTERN 
int pam_sm_acct_mgmt(pam_handle_t *pamh, int flags, int argc, 
		     const char **argv)
{
  _log_err(LOG_NOTICE, "pam_sm_acct_mgmt called inappropriatly");
  return PAM_SERVICE_ERR;
}
 
PAM_EXTERN
int pam_sm_open_session(pam_handle_t *pamh,int flags,int argc, const char **argv)
{

  if (make_tmp_directory(pamh) == 0) {
    set_environment(pamh);
    return PAM_SUCCESS;
  } else {
    return PAM_ABORT;
  }
  
}

PAM_EXTERN
int pam_sm_close_session(pam_handle_t *pamh,int flags,int argc,
			 const char **argv)
{
  D(("Called and Exit"));
  return PAM_SUCCESS;
}

PAM_EXTERN 
int pam_sm_chauthtok(pam_handle_t *pamh, int flags, int argc, 
		     const char **argv)
{
  _log_err(LOG_NOTICE, "pam_sm_chauthtok called inappropriatly");
  return PAM_SERVICE_ERR;
}

#ifdef PAM_STATIC

/* static module data */

struct pam_module _pam_env_modstruct = {
     "pam_tmpdir",
     pam_sm_authenticate,
     pam_sm_setcred,
     pam_sm_acct_mgmt,
     pam_sm_open_session,
     pam_sm_close_session,
     pam_sm_chauthtok,
};

#endif

/* end of module definition */
