/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#ifdef ENABLE_PC98

#include <parted/parted.h>
#include <parted/endian.h>
#include <parted/disk_pc98.h>
#include <string.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

/* hacked from Linux/98 source: fs/partitions/nec98.h
 *
 * See also:
 *      http://people.FreeBSD.org/~kato/pc98.html
 *      http://www.kmc.kyoto-u.ac.jp/proj/linux98/index-english.html
 *
 * Partition types:
 * 
 *   id0(mid):
 *      bit 7: 1=bootable, 0=not bootable
 *        # Linux uses this flag to make a distinction between ext2 and swap.
 *      bit 6--0:
 *        00H      : N88-BASIC(data)?, PC-UX(data)?
 *        04H      : PC-UX(data)
 *        06H      : N88-BASIC
 *        10H      : N88-BASIC
 *        14H      : *BSD, PC-UX
 *        20H      : DOS(data), Windows95/98/NT, Linux
 *        21H..2FH : DOS(system#1 .. system#15)
 *        40H      : Minix
 *        
 *   id1(sid):
 *      bit 7: 1=active, 0=sleep(hidden)
 *        # PC-UX uses this flag to make a distinction between its filesystem
 *        # and its swap.
 *      bit 6--0:
 *        01H: FAT12
 *        11H: FAT16, <32MB [accessible to DOS 3.3]
 *        21H: FAT16, >=32MB [Large Partition]
 *        31H: NTFS
 *        28H: Windows NT (Volume/Stripe Set?)
 *        41H: Windows NT (Volume/Stripe Set?)
 *        48H: Windows NT (Volume/Stripe Set?)
 *        61H: FAT32
 *        04H: PC-UX
 *        06H: N88-BASIC
 *        44H: *BSD
 *        62H: ext2, linux-swap
 */

#define MAX_PART_COUNT 16
#define PC9800_EXTFMT_MAGIC 0xAA55

#define BIT(x) (1 << (x))
#define GET_BIT(n,bit) (((n) & BIT(bit)) != 0)
#define SET_BIT(n,bit,val) n = (val)?  (n | BIT(bit))  :  (n & ~BIT(bit))

typedef struct _PC98RawPartition	PC98RawPartition;
typedef struct _PC98RawTable		PC98RawTable;

/* ripped from Linux/98 source */
struct _PC98RawPartition {
	__u8	mid;		/* 0x80 - boot */
	__u8	sid;		/* 0x80 - active */
	__u8	dum1;		/* dummy for padding */
	__u8	dum2;		/* dummy for padding */
	__u8	ipl_sect;	/* IPL sector */
	__u8	ipl_head;	/* IPL head */
	__u16	ipl_cyl;	/* IPL cylinder */
	__u8	sector;		/* starting sector */
	__u8	head;		/* starting head */
	__u16	cyl;		/* starting cylinder */
	__u8	end_sector;	/* end sector */
	__u8	end_head;	/* end head */
	__u16	end_cyl;	/* end cylinder */
	__u8	name[16];
} __attribute__((packed));

struct _PC98RawTable {
	__u8			boot_code [510];
	__u16			magic;
	PC98RawPartition	partitions [MAX_PART_COUNT];
} __attribute__((packed));

/* this MBR boot code is dummy */
static char MBR_BOOT_CODE[] = {
	0xcb,			/* retf */
	0x00, 0x00, 0x00,	/* */
	0x49, 0x50, 0x4c, 0x31  /* "IPL1" */
};

void ped_disk_pc98_init ();
void ped_disk_pc98_done ();

static int pc98_probe (const PedDevice *dev);
static PedDisk* pc98_open (PedDevice* dev);
static PedDisk* pc98_create (PedDevice* dev);
static int pc98_clobber (PedDevice* dev);
static int pc98_close (PedDisk* disk);
static int pc98_read (PedDisk* disk);
static int pc98_write (PedDisk* disk);

static PedPartition* pc98_partition_new (
	const PedDisk* disk, PedPartitionType part_type,
	const PedFileSystemType* fs_type, PedSector start, PedSector end);
static void pc98_partition_destroy (PedPartition* part);
static int pc98_partition_set_flag (
	PedPartition* part, PedPartitionFlag flag, int state);
static int pc98_partition_get_flag (
	const PedPartition* part, PedPartitionFlag flag);
static int pc98_partition_is_flag_available (
	const PedPartition* part,
	PedPartitionFlag flag);
static void pc98_partition_set_name (PedPartition* part, const char* name);
static const char* pc98_partition_get_name (const PedPartition* part);
static int pc98_partition_align (PedPartition* part,
				 const PedConstraint* constraint);
static int pc98_partition_enumerate (PedPartition* part);

static int pc98_alloc_metadata (PedDisk* disk);
static int pc98_get_max_primary_partition_count (const PedDisk* disk);

static int is_unused_partition(const PC98RawPartition* raw_part);
static PedSector chs_to_sector (const PedDevice* dev, int c, int h, int s);
static void sector_to_chs (const PedDevice* dev, PedSector sector,
			   int* c, int* h, int* s);

static PedDiskOps pc98_disk_ops = {
	probe:			pc98_probe,
	open:			pc98_open,
	create:			pc98_create,
	clobber:		pc98_clobber,
	close:			pc98_close,
	read:			pc98_read,
	write:			pc98_write,
	
	partition_new:		pc98_partition_new,
	partition_destroy:	pc98_partition_destroy,
	partition_set_flag:	pc98_partition_set_flag,
	partition_get_flag:	pc98_partition_get_flag,
	partition_is_flag_available:	pc98_partition_is_flag_available,
	partition_set_name:	pc98_partition_set_name,
	partition_get_name:	pc98_partition_get_name,
	partition_align:	pc98_partition_align,
	partition_enumerate:	pc98_partition_enumerate,
	partition_set_extended_system:	NULL,

	alloc_metadata:		pc98_alloc_metadata,
	get_max_primary_partition_count:
				pc98_get_max_primary_partition_count
};

static PedDiskType pc98_disk_type = {
	next:		NULL,
	name:		"pc98",
	ops:		&pc98_disk_ops,
	features:	PED_DISK_TYPE_PARTITION_NAME
};

void
ped_disk_pc98_init ()
{
	PED_ASSERT (sizeof (PC98RawTable) == 512 * 2, return);
	ped_register_disk_type (&pc98_disk_type);
}

void
ped_disk_pc98_done ()
{
	ped_unregister_disk_type (&pc98_disk_type);
}

/* magic(?) check */
static int
pc98_check_magic (const PC98RawTable *part_table)
{
	/* check "extended-format" (have partition table?) */
	if (PED_LE16_TO_CPU(part_table->magic) != PC9800_EXTFMT_MAGIC)
		return 0;

	return 1;
}

static int
pc98_check_ipl_signature (const PC98RawTable *part_table)
{
	return !memcmp (part_table->boot_code + 4, "IPL1", 4);
}		

static int
check_partition_consistency (const PedDevice* dev,
	       		     const PC98RawPartition* raw_part)
{
	if (raw_part->ipl_sect >= dev->sectors
	   || raw_part->sector >= dev->sectors
	   || raw_part->end_sector >= dev->sectors
	   || raw_part->ipl_head >= dev->heads
	   || raw_part->head >= dev->heads
	   || raw_part->end_head >= dev->heads
	   || PED_LE16_TO_CPU(raw_part->ipl_cyl) >= dev->cylinders
	   || PED_LE16_TO_CPU(raw_part->cyl) >= dev->cylinders
	   || PED_LE16_TO_CPU(raw_part->end_cyl) >= dev->cylinders
#if 0
	   || !chs_to_sector(dev, PED_LE16_TO_CPU(raw_part->ipl_cyl),
			     raw_part->ipl_head, raw_part->ipl_sect)
	   || !chs_to_sector(dev, PED_LE16_TO_CPU(raw_part->cyl),
			     raw_part->head, raw_part->sector)
	   || !chs_to_sector(dev, PED_LE16_TO_CPU(raw_part->end_cyl),
			     raw_part->end_head, raw_part->end_sector)
#endif
	   || PED_LE16_TO_CPU(raw_part->end_cyl)
	  		< PED_LE16_TO_CPU(raw_part->cyl))
		return 0;

	return 1;
}

static int
pc98_probe (const PedDevice *dev)
{
	PC98RawTable		part_table;
	int			empty;
	const PC98RawPartition*	p;

	PED_ASSERT (dev != NULL, return 0);

	if (!ped_device_open ((PedDevice*) dev))
		return 0;
	if (!ped_device_read (dev, &part_table, 0, 2)) {
		ped_device_close ((PedDevice*) dev);
		return 0;
	}
	ped_device_close ((PedDevice*) dev);

	/* check magic */
	if (!pc98_check_magic(&part_table))
		return 0;

	/* check consistency */
	empty = 1;
	for (p = part_table.partitions;
	     p < part_table.partitions + MAX_PART_COUNT;
	     p++)
	{
		if (!is_unused_partition (p))
			empty = 0;
		if (p->mid == 0 && p->sid == 0)
			continue;
		if (!check_partition_consistency (dev, p))
			return 0;
	}

	/* check boot loader */
	if (pc98_check_ipl_signature (&part_table))
		return 1;
	else if (part_table.boot_code[0])	/* invalid boot loader */
		return 0;

	/* Not to mistake msdos disk map for PC-9800's empty disk map  */
	if (empty)
		return 0;

	return 1;
}

static PedDisk*
pc98_open (PedDevice* dev)
{
	PedDisk*	disk;

	PED_ASSERT (dev != NULL, return 0);

	if (!pc98_probe (dev))
		goto error;

	ped_device_open ((PedDevice*) dev);
	disk = ped_disk_alloc (dev, &pc98_disk_type);
	if (!disk)
		goto error;

	if (!pc98_read (disk))
		goto error_free_disk;

	return disk;

error_free_disk:
	ped_disk_free (disk);
error:
	return NULL;
}

static PedDisk*
pc98_create (PedDevice* dev)
{
	PC98RawTable		 table;

	PED_ASSERT (dev != NULL, return NULL);

	if (!ped_device_open (dev))
		goto error;
	if (!ped_device_read (dev, &table, 0, 2))
		goto error_close_dev;

	if (!pc98_check_ipl_signature (&table)) {
		memset (table.boot_code, 0, sizeof (table.boot_code));
		memcpy (table.boot_code, MBR_BOOT_CODE, sizeof(MBR_BOOT_CODE));
	}

	memset (table.partitions, 0, sizeof (table.partitions));
	table.magic = PED_CPU_TO_LE16(PC9800_EXTFMT_MAGIC);

	if (!ped_device_write (dev, (void*) &table, 0, 2))
		goto error_close_dev;
	if (!ped_device_sync (dev))
		goto error_close_dev;
	ped_device_close (dev);
	return pc98_open (dev);

error_close_dev:
	ped_device_close (dev);
error:
	return NULL;
}

static int
pc98_close (PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);

	ped_device_close (disk->dev);
	ped_disk_free (disk);
	return 1;
}

static int
pc98_clobber (PedDevice* dev)
{
	PC98RawTable	table;

	PED_ASSERT (dev != NULL, return 0);
	PED_ASSERT (pc98_probe (dev), return 0);

	if (!ped_device_open ((PedDevice*) dev))
		goto error;
	if (!ped_device_read (dev, &table, 0, 1))
		goto error_close_dev;

	memset (table.partitions, 0, sizeof (table.partitions));
	table.magic = PED_CPU_TO_LE16(0);

	if (pc98_check_ipl_signature (&table))
		memset (table.boot_code, 0, sizeof (table.boot_code));

	if (!ped_device_write (dev, (void*) &table, 0, 1))
		goto error_close_dev;
	if (!ped_device_sync (dev))
		goto error_close_dev;
	ped_device_close (dev);
	return 1;

error_close_dev:
	ped_device_close (dev);
error:
	return 0;
}

static PedSector
chs_to_sector (const PedDevice* dev, int c, int h, int s)
{
	PED_ASSERT (dev != NULL, return 0);
	return (c * dev->heads + h) * dev->sectors + s;
}

static void
sector_to_chs (const PedDevice* dev, PedSector sector, int* c, int* h, int* s)
{
	PED_ASSERT (dev != NULL, return);
	PED_ASSERT (c != NULL, return);
	PED_ASSERT (h != NULL, return);
	PED_ASSERT (s != NULL, return);
	*c = (sector) / (dev->heads * dev->sectors);
	*h = ((sector) % (dev->heads * dev->sectors)) / dev->sectors;
	*s = ((sector) % (dev->heads * dev->sectors)) % dev->sectors;
}

static PedSector
legacy_start (const PedDisk* disk, const PC98RawPartition* raw_part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (raw_part != NULL, return 0);

	return chs_to_sector (disk->dev, PED_LE16_TO_CPU(raw_part->cyl),
			      raw_part->head, raw_part->sector);
}

static PedSector
legacy_end (const PedDisk* disk, const PC98RawPartition* raw_part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (raw_part != NULL, return 0);

	if (raw_part->end_head == 0 && raw_part->end_sector == 0) {
		return chs_to_sector (disk->dev,
				      PED_LE16_TO_CPU(raw_part->end_cyl),
				      disk->dev->heads - 1,
				      disk->dev->sectors - 1);
	} else {
		return chs_to_sector (disk->dev,
				      PED_LE16_TO_CPU(raw_part->end_cyl),
				      raw_part->end_head,
				      raw_part->end_sector);
	}
}

static int
is_unused_partition(const PC98RawPartition* raw_part)
{
	if (raw_part->mid || raw_part->sid
	    || raw_part->ipl_sect
	    || raw_part->ipl_head
	    || PED_LE16_TO_CPU(raw_part->ipl_cyl)
	    || raw_part->sector
	    || raw_part->head
	    || PED_LE16_TO_CPU(raw_part->cyl)
	    || raw_part->end_sector
	    || raw_part->end_head
	    || PED_LE16_TO_CPU(raw_part->end_cyl))
		return 0;
	return 1;
}

static int
read_table (PedDisk* disk)
{
	int			i;
	PC98RawTable		table;
	PedConstraint*		constraint_any = ped_constraint_any (disk);

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (disk->dev != NULL, return 0);

	if (!ped_device_read (disk->dev, (void*) &table, 0, 2))
		goto error;

	if (!pc98_check_magic(&table)) {
		if (ped_exception_throw (
			PED_EXCEPTION_ERROR, PED_EXCEPTION_IGNORE_CANCEL,
			_("Invalid partition table on %s"),
			disk->dev->path))
			goto error;
	}

	for (i = 0; i < MAX_PART_COUNT; i++) {
		PC98RawPartition*	raw_part;
		PedPartition*		part;
		PC98PartitionData*	pc98_data;
		PedSector		part_start;
		PedSector		part_end;

		raw_part = &table.partitions [i];

		if (is_unused_partition(raw_part))
			continue;

		part_start = legacy_start (disk, raw_part);
		part_end   = legacy_end (disk, raw_part);

		part = ped_partition_new (disk, PED_PARTITION_PRIMARY, NULL,
					  part_start, part_end);
		if (!part)
			goto error;
		pc98_data = part->disk_specific;
		PED_ASSERT (pc98_data != NULL, goto error);

		pc98_data->system = (raw_part->mid << 8) | raw_part->sid;
		pc98_data->boot = GET_BIT(raw_part->mid, 7);
		pc98_data->hidden = !GET_BIT(raw_part->sid, 7);

		ped_partition_set_name (part, raw_part->name);

		pc98_data->ipl_sector = chs_to_sector (
			disk->dev,
			PED_LE16_TO_CPU(raw_part->ipl_cyl),
			raw_part->ipl_head,
			raw_part->ipl_sect);

		/* hack */
		if (pc98_data->ipl_sector == part->geom.start)
			pc98_data->ipl_sector = 0;

		part->num = i + 1;

		if (!ped_disk_add_partition (disk, part, constraint_any))
			goto error;

		if (part->geom.start != part_start
		    || part->geom.end != part_end) {
			ped_exception_throw (
				PED_EXCEPTION_NO_FEATURE,
				PED_EXCEPTION_CANCEL,
				_("Partition %d isn't aligned to cylinder "
				  "boundaries.  Need to add support for this."),
				part->num);
			goto error;
		}

		part->fs_type = ped_file_system_probe (&part->geom);
	}

	ped_constraint_destroy (constraint_any);
	return 1;

error:
	ped_disk_delete_all (disk);
	ped_constraint_destroy (constraint_any);
	return 0;
}

static int
pc98_read (PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (disk->dev != NULL, return 0);

	ped_disk_delete_all (disk);
	return read_table (disk);
}

static int
fill_raw_part (PC98RawPartition* raw_part, const PedPartition* part)
{
	PC98PartitionData*	pc98_data;
	int			c, h, s;
	const char*		name;

	PED_ASSERT (raw_part != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);
	PED_ASSERT (part->disk_specific != NULL, return 0);

	pc98_data = part->disk_specific;
	raw_part->mid = (pc98_data->system >> 8) & 0xFF;
	raw_part->sid = pc98_data->system & 0xFF;

	SET_BIT(raw_part->mid, 7, pc98_data->boot);
	SET_BIT(raw_part->sid, 7, !pc98_data->hidden);

	memset (raw_part->name, ' ', sizeof(raw_part->name));
	name = ped_partition_get_name (part);
	PED_ASSERT (name != NULL, return 0);
	PED_ASSERT (strlen (name) <= 16, return 0);
	if (!strlen (name) && part->fs_type)
		name = part->fs_type->name;
	memcpy (raw_part->name, name, strlen (name));

	sector_to_chs (part->geom.disk->dev, part->geom.start, &c, &h, &s);
	raw_part->cyl	 = PED_CPU_TO_LE16(c);
	raw_part->head	 = h;
	raw_part->sector = s;

	if (pc98_data->ipl_sector) {
		sector_to_chs (part->geom.disk->dev, pc98_data->ipl_sector,
			       &c, &h, &s);
		raw_part->ipl_cyl  = PED_CPU_TO_LE16(c);
		raw_part->ipl_head = h;
		raw_part->ipl_sect = s;
	} else {
		raw_part->ipl_cyl  = raw_part->cyl;
		raw_part->ipl_head = raw_part->head;
		raw_part->ipl_sect = raw_part->sector;
	}

	sector_to_chs (part->geom.disk->dev, part->geom.end, &c, &h, &s);
	if (h != part->geom.disk->dev->heads - 1
	    || s != part->geom.disk->dev->sectors - 1) {
		ped_exception_throw (
		    PED_EXCEPTION_NO_FEATURE,
		    PED_EXCEPTION_CANCEL,
		    _("Partition %d isn't aligned to cylinder "
		      "boundaries.  Need to add support for this."),
		    part->num);
		return 0;
	}
	raw_part->end_cyl    = PED_CPU_TO_LE16(c);
#if 0
	raw_part->end_head   = h;
	raw_part->end_sector = s;
#else
	raw_part->end_head   = 0;
	raw_part->end_sector = 0;
#endif

	return 1;
}

static int
pc98_write (PedDisk* disk)
{
	PC98RawTable		table;
	PedPartition*		part;
	int			i;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (disk->dev != NULL, return 0);

	ped_device_read (disk->dev, &table, 0, 2);

	if (!pc98_check_ipl_signature (&table)) {
		memset (table.boot_code, 0, sizeof(table.boot_code));
		memcpy (table.boot_code, MBR_BOOT_CODE, sizeof(MBR_BOOT_CODE));
	}

	memset (table.partitions, 0, sizeof (table.partitions));
	table.magic = PED_CPU_TO_LE16(PC9800_EXTFMT_MAGIC);

	for (i = 1; i <= MAX_PART_COUNT; i++) {
		part = ped_disk_get_partition (disk, i);
		if (!part)
			continue;

		if (!fill_raw_part (&table.partitions [i - 1], part))
			return 0;
	}

	if (!ped_device_write (disk->dev, (void*) &table, 0, 2))
		return 0;
	if (!ped_device_sync (disk->dev))
		return 0;
	return 1;
}

static PedPartition*
pc98_partition_new (
	const PedDisk* disk, PedPartitionType part_type,
	const PedFileSystemType* fs_type, PedSector start, PedSector end)
{
	PedPartition*		part;
	PC98PartitionData*	pc98_data;

	part = ped_partition_alloc (disk, part_type, fs_type, start, end);
	if (!part)
		goto error;

	if (ped_partition_is_active (part)) {
		part->disk_specific
			= pc98_data = ped_malloc (sizeof (PC98PartitionData));
		if (!pc98_data)
			goto error_free_part;
		pc98_data->ipl_sector = 0;
		pc98_data->hidden = 0;
		pc98_data->boot = 0;
		strcpy (pc98_data->name, "");
	} else {
		part->disk_specific = NULL;
	}
	return part;

error_free_pc98_data:
	ped_free (pc98_data);
error_free_part:
	ped_free (part);
error:
	return 0;
}

static void
pc98_partition_destroy (PedPartition* part)
{
	PED_ASSERT (part != NULL, return);

	if (ped_partition_is_active (part))
		ped_free (part->disk_specific);
	ped_free (part);
}

static int
pc98_partition_set_flag (PedPartition* part, PedPartitionFlag flag, int state)
{
	PedDisk*			disk;
	PC98PartitionData*		pc98_data;
	const PedFileSystemType*	fs_type;

	PED_ASSERT (part != NULL, return 0);
	PED_ASSERT (part->disk_specific != NULL, return 0);
	PED_ASSERT (part->geom.disk != NULL, return 0);

	pc98_data = part->disk_specific;
	disk = part->geom.disk;

	/* hack! */
	fs_type = part->fs_type ? part->fs_type
				: ped_file_system_type_get ("ext2");

	switch (flag) {
	case PED_PARTITION_HIDDEN:
		pc98_data->hidden = state;
		return ped_partition_set_system (part, fs_type);

	case PED_PARTITION_BOOT:
		pc98_data->boot = state;
		return ped_partition_set_system (part, fs_type);

	default:
		return 0;
	}
}

static int
pc98_partition_get_flag (const PedPartition* part, PedPartitionFlag flag)
{
	PC98PartitionData*	pc98_data;

	PED_ASSERT (part != NULL, return 0);
	PED_ASSERT (part->disk_specific != NULL, return 0);

	pc98_data = part->disk_specific;
	switch (flag) {
	case PED_PARTITION_HIDDEN:
		return pc98_data->hidden;

	case PED_PARTITION_BOOT:
		return pc98_data->boot;

	default:
		return 0;
	}
}

static int
pc98_partition_is_flag_available (
	const PedPartition* part, PedPartitionFlag flag)
{
	switch (flag) {
	case PED_PARTITION_HIDDEN:
	case PED_PARTITION_BOOT:
		return 1;

	default:
		return 0;
	}
}

static void
pc98_partition_set_name (PedPartition* part, const char* name)
{
	PC98PartitionData*	pc98_data;
	int			i;

	PED_ASSERT (part != NULL, return);
	PED_ASSERT (part->disk_specific != NULL, return);
	pc98_data = part->disk_specific;

	strncpy (pc98_data->name, name, 16);
	pc98_data->name [16] = 0;
	for (i = strlen (pc98_data->name) - 1; pc98_data->name[i] == ' '; i--)
		pc98_data->name [i] = 0;
}

static const char*
pc98_partition_get_name (const PedPartition* part)
{
	PC98PartitionData*	pc98_data;

	PED_ASSERT (part != NULL, return NULL);
	PED_ASSERT (part->disk_specific != NULL, return NULL);
	pc98_data = part->disk_specific;

	return pc98_data->name;
}

static PedConstraint*
_primary_constraint (PedDisk* disk)
{
	PedAlignment	start_align;
	PedAlignment	end_align;
	PedGeometry	max_geom;
	PedSector	cylinder_size;

	cylinder_size = disk->dev->sectors * disk->dev->heads;

	if (!ped_alignment_init (&start_align, 0, cylinder_size))
		return NULL;
	if (!ped_alignment_init (&end_align, -1, cylinder_size))
		return NULL;
	if (!ped_geometry_init (&max_geom, disk, cylinder_size,
			       	disk->dev->length - cylinder_size))
		return NULL;

	return ped_constraint_new (&start_align, &end_align, &max_geom,
				   &max_geom, 1);
}

/* argghh!  this should be somewhere else... but where?! */
static int
_try_constraint (PedPartition* part, const PedConstraint* external,
		 PedConstraint* internal)
{
	PedConstraint*		intersection;
	PedGeometry*		solution;

	intersection = ped_constraint_intersect (external, internal);
	ped_constraint_destroy (internal);
	if (!intersection)
		goto fail;

	solution = ped_constraint_solve_nearest (intersection, &part->geom);
	if (!solution)
		goto fail_free_intersection;
	ped_geometry_set (&part->geom, solution->start, solution->length);
	ped_geometry_destroy (solution);
	ped_constraint_destroy (intersection);
	return 1;

fail_free_intersection:
	ped_constraint_destroy (intersection);
fail:
	return 0;
}

static int
pc98_partition_align (PedPartition* part, const PedConstraint* constraint)
{
	PED_ASSERT (part != NULL, return 0);

	if (_try_constraint (part, constraint,
			     _primary_constraint (part->geom.disk)))
		return 1;

	ped_exception_throw (
		PED_EXCEPTION_ERROR,
		PED_EXCEPTION_CANCEL,
		_("Unable to align partition."));

	return 0;
}

static int
next_primary (PedDisk* disk)
{
	int	i;
	for (i=1; i<=MAX_PART_COUNT; i++) {
		if (!ped_disk_get_partition (disk, i))
			return i;
	}
	return 0;
}

static int
pc98_partition_enumerate (PedPartition* part)
{
	PedDisk*	disk;

	PED_ASSERT (part != NULL, return 0);
	PED_ASSERT (part->geom.disk != NULL, return 0);

	disk = part->geom.disk;

	/* don't re-number a partition */
	if (part->num != -1)
		return 1;

	PED_ASSERT (part->type == PED_PARTITION_PRIMARY, return 0);

	part->num = next_primary (disk);
	if (!part->num) {
		ped_exception_throw (PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("Can't add another partition."));
		return 0;
	}

	return 1;
}

static int
pc98_alloc_metadata (PedDisk* disk)
{
	PedPartition*		new_part;
	PedConstraint*		constraint_any = ped_constraint_any (disk);

	PED_ASSERT (disk != NULL, goto error);
	PED_ASSERT (disk->dev != NULL, goto error);

	new_part = ped_partition_new (
			disk,
			PED_PARTITION_PRIMARY | PED_PARTITION_METADATA,
			NULL,
			0, disk->dev->sectors * disk->dev->heads - 1);
	if (!new_part)
		goto error;

	if (!ped_disk_add_partition (disk, new_part, constraint_any)) {
		ped_partition_destroy (new_part);
		goto error;
	}

	ped_constraint_destroy (constraint_any);
	return 1;

error:
	ped_constraint_destroy (constraint_any);
	return 0;
}

static int
pc98_get_max_primary_partition_count (const PedDisk* disk)
{
	return MAX_PART_COUNT;
}

#endif /* ENABLE_PC98 */
