/* Copyright (c) 1997-1999 Guenter Geiger, Miller Puckette, Larry Troxler,
* Winfried Ritsch, Karl MacMillan, and others.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

/* this file implements the sys_ functions profiled in m_imp.h for
 audio and MIDI I/O.  In Linux there might be several APIs for doing the
 audio part; right now there are three (OSS, ALSA, RME); the third is
 for the RME 9652 driver by Winfried Ritsch.
 
 FUNCTION PREFIXES.
    sys_ -- functions which must be exported to Pd on all platforms
    linux_ -- linux-specific objects which don't depend on API,
    	mostly static but some exported.
    oss_, alsa_, rme_ -- API-specific functions, all of which are
    	static.
     
 For MIDI, we only offer the OSS API; ALSA has to emulate OSS for us.
*/

/* OSS include (whether we're doing OSS audio or not we need this for MIDI) */

#include <linux/soundcard.h>

#ifdef HAVE_ALSA   /* ALSA includes */
#include <sys/asoundlib.h>
#endif /* ALSA */

#include "m_imp.h"
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sched.h>
#include <sys/mman.h>


/* midi lib include */

#include "s_linux_mididefs.h"
#include "s_linux_midi_io.h"
#include "s_linux_midiparse.h"

/* local function prototypes */

static void linux_close_midi( void);

static int oss_open_audio(int inchans, int outchans, int rate);
static void oss_close_audio(void);
static int oss_send_dacs(void);
static void oss_reportidle(void);

#ifdef HAVE_ALSA
typedef int16_t t_alsa_sample16;
typedef int32_t t_alsa_sample32;
#define ALSA_SAMPLEWIDTH_16 sizeof(t_alsa_sample16)
#define ALSA_SAMPLEWIDTH_32 sizeof(t_alsa_sample32)
#define ALSA_XFERSIZE16  (signed int)(sizeof(t_alsa_sample16) * DACBLKSIZE)
#define ALSA_XFERSIZE32  (signed int)(sizeof(t_alsa_sample32) * DACBLKSIZE)
#define ALSA_MAXCHANS 16
#define ALSA_MAXDEV 1
#define ALSA_JITTER 1024
#define ALSA_EXTRABUFFER 2048

typedef struct _alsa_dev
{
    snd_pcm_t *handle;
    snd_pcm_channel_info_t info;
    snd_pcm_channel_setup_t setup;
} t_alsa_dev;

t_alsa_dev alsa_device[ALSA_MAXDEV];
static int n_alsa_dev;
static char *alsa_buf;
static int alsa_samplewidth;

static int alsa_mode;

static int alsa_open_audio(int inchans, int outchans, int rate);
static void alsa_close_audio(void);
static int alsa_send_dacs(void);
static void alsa_set_params(t_alsa_dev *dev, int dir, int rate, int voices);
static void alsa_reportidle(void);
#endif /* HAVE_ALSA */
 
#ifdef HAVE_RME
static int rme9652_open_audio(int inchans, int outchans, int rate);
static void rme9652_close_audio(void);
static int rme9652_send_dacs(void);
static void rme9652_reportidle(void);
#endif /* HAVE_RME */

/* Defines */
#define DEBUG(x) x
#define DEBUG2(x) {x;}

#define OSS_DEFAULTCH 2
#define RME_DEFAULTCH 8     /* need this even if RME undefined */
typedef int16_t t_oss_sample;
#define OSS_SAMPLEWIDTH sizeof(t_oss_sample)
#define OSS_BYTESPERCHAN (DACBLKSIZE * OSS_SAMPLEWIDTH) 
#define OSS_XFERSAMPS (OSSCHPERDEV*DACBLKSIZE)
#define OSS_XFERSIZE (OSS_SAMPLEWIDTH * OSS_XFERSAMPS)

#ifdef HAVE_RME
typedef int32_t t_rme_sample;
#define RME_SAMPLEWIDTH sizeof(t_rme_sample)
#define RME_BYTESPERCHAN (DACBLKSIZE * RME_SAMPLEWIDTH)
#endif /* HAVE_RME */

/* GLOBALS */
#define OSSCHPERDEV 2	    	/* channels per OSS device */
#define OSS_MAXDEV 4    	/* maximum number of input or output devices */

static int linux_whichapi = API_OSS;
static int linux_inchannels;
static int linux_outchannels;
static int linux_advance_samples; /* scheduler advance in samples */
static int linux_meters;    	/* true if we're metering */
static float linux_inmax;    	/* max input amplitude */
static float linux_outmax;    	/* max output amplitude */
#ifdef HAVE_ALSA
static int alsa_devno = 1;
#endif

/* our device handles */

typedef struct _dh {
     int fd;
     unsigned int space; 	/* bytes available for writing/reading  */
     int bufsize;   	/* total buffer size in blocks for this device */
     int dropcount; 	/* number of buffers to drop for resync (output only) */
} t_dac_handle;

static t_dac_handle linux_dacs[OSS_MAXDEV];
static t_dac_handle linux_adcs[OSS_MAXDEV];
static int linux_ndacs = 0;
static int linux_nadcs = 0;

    /* exported variables */
int sys_schedadvance = 50000; 	/* scheduler advance in microseconds */
float sys_dacsr;
int sys_hipriority = 0;
t_sample *sys_soundout;
t_sample *sys_soundin;

    /* OSS-specific private variables */
static int oss_oldaudio = 0;	/* flag to use "fragments" below  */
  /* log base 2 of audio "fragment size" in bytes */
static int oss_logfragsize = 0xb;
static int oss_nfragment = 4;
static char ossdsp[] = "/dev/dsp%d"; 

#ifndef INT32_MAX
#define INT32_MAX 0x7fffffff
#endif
#define CLIP32(x) (((x)>INT32_MAX)?INT32_MAX:((x) < -INT32_MAX)?-INT32_MAX:(x))


/* ------------- private routines for all APIS ------------------- */
static void linux_flush_all_underflows_to_zero(void)
{
/*
    TODO: Implement similar thing for linux (GGeiger) 

    One day we will figure this out, I hope, because it 
    costs CPU time dearly on Intel  - LT
  */
     /*    union fpc_csr f;
	   f.fc_word = get_fpc_csr();
	   f.fc_struct.flush = 1;
	   set_fpc_csr(f.fc_word);
     */
}

    /* set channels and sample rate.  */

static void linux_setchsr(int chin,int chout, int sr)
{
    int nblk;
    int inbytes = chin * (DACBLKSIZE*sizeof(float));
    int outbytes = chout * (DACBLKSIZE*sizeof(float));

    linux_inchannels = chin;
    linux_outchannels = chout;
    sys_dacsr = sr;
    linux_advance_samples = (sys_schedadvance * sys_dacsr) / (1000000.);
    if (linux_advance_samples < 3 * DACBLKSIZE)
    	linux_advance_samples = 3 * DACBLKSIZE;

    if (sys_soundin)
    	free(sys_soundin);
    sys_soundin = (t_float *)malloc(inbytes);
    memset(sys_soundin, 0, inbytes);

    if (sys_soundout)
    	free(sys_soundout);
    sys_soundout = (t_float *)malloc(outbytes);
    memset(sys_soundout, 0, outbytes);

    if (sys_verbose)
    	post("input channels = %d, output channels = %d",
    	    linux_inchannels, linux_outchannels);
}

/* ---------------- private MIDI routines -------------------------- */
#define NPORTS 16

void inmidi_rtin(int portno, int cmd);
void inmidi_mtc(int portno, int frame, int sec, int min, int hr);
void inmidi_noteon(int portno, int channel, int pitch, int velo);
void inmidi_controlchange(int portno, int channel, int ctlnumber, int value);
void inmidi_programchange(int portno, int channel, int value);
void inmidi_pitchbend(int portno, int channel, int value);
void inmidi_aftertouch(int portno, int channel, int value);
void inmidi_polyaftertouch(int portno, int channel, int pitch, int value);
void inmidi_byte(int portno, int byte);

static int  oss_nports;
MIDIParser *oss_mpa[NPORTS];
MIDIPort *oss_midiport[NPORTS];

static void inmidi_noteonoff(int portno, int channel, int pitch,
    int velon, int veloff)
{
    	/* ignore veloff, pd does not respect it */
    inmidi_noteon(portno, channel, pitch, velon);
}

void linux_open_midi(void)
{
    int i;
    	/* Look for MIDI devices */
    MIDIPort *mp;
    oss_nports = 0;
    if (sys_verbose)
        fprintf(stderr, "opening MIDI... ");
    if (midi_open()<=0)
    	goto bad;

    for (mp=midi_getfirstport(); oss_nports < NPORTS && mp != NULL;
    	mp = midi_getnextport())
    {
	oss_midiport[oss_nports] = mp;

	oss_mpa[oss_nports] = mp_parse_init(oss_nports);  /* .... */

	    /* initialize all reaction functions */
	oss_mpa[oss_nports]->raw = inmidi_byte;
	oss_mpa[oss_nports]->noteonoff = inmidi_noteonoff;
	oss_mpa[oss_nports]->noteon = NULL;
	oss_mpa[oss_nports]->noteoff = NULL;

	oss_mpa[oss_nports]->controlchange = inmidi_controlchange;
	oss_mpa[oss_nports]->programchange = inmidi_programchange;
	oss_mpa[oss_nports]->pitchbend = inmidi_pitchbend;
	oss_mpa[oss_nports]->aftertouch = inmidi_aftertouch;
	oss_mpa[oss_nports]->polyaftertouch = inmidi_polyaftertouch;
	    /* system messages */
	oss_mpa[oss_nports]->mode = NULL;
	oss_mpa[oss_nports]->quarterframe = NULL; /* inmidi_mtc */
	oss_mpa[oss_nports]->mode = NULL;
	oss_mpa[oss_nports]->songposition = NULL;
	oss_mpa[oss_nports]->songselect = NULL;
	oss_mpa[oss_nports]->tunerequest = NULL;
	oss_mpa[oss_nports]->sysex = NULL;
	oss_mpa[oss_nports]->realtime = NULL; /* inmidi_rtin; */

	oss_nports++;
    };

    if (sys_verbose)
    {
	if (oss_nports == 1) 
    	    fprintf(stderr,"MIDIlib: Found one MIDI port on %s\n",
	    oss_midiport[0]->devname);
	else 
    	    for(i=0;i<oss_nports;i++) 
    		fprintf(stderr,"MIDIlib Found MIDI port %d on %s\n",
    	    	    i,oss_midiport[i]->devname);		
    }
    return;

 bad:
    error("failed to open MIDI ports; continuing without MIDI.");
    oss_nports = 0;
    return;
}

void linux_close_midi()
{
     int i;
     for(i=0;i<oss_nports;i++)
     {
	  mp_parse_exit(oss_mpa[i]);
     }
     midi_close();
}

/* ----------------------- public routines ----------------------- */

void sys_open_audio_and_midi(int midiin, int midiout,
    int inchans, int outchans, int rate)
{
    int defaultchannels =
    	(linux_whichapi == API_RME ? RME_DEFAULTCH : OSS_DEFAULTCH);
    if (rate < 1) rate = 44100;
    if (inchans < 0) inchans = defaultchannels;
    if (outchans < 0) outchans = defaultchannels;
    linux_flush_all_underflows_to_zero();
#ifdef HAVE_ALSA 
     if (linux_whichapi == API_ALSA)
     	alsa_open_audio(inchans, outchans, rate);
     else 
#endif
#ifdef HAVE_RME
     if (linux_whichapi == API_RME)
     	rme9652_open_audio(inchans,outchans,rate);
     else 
#endif
    	oss_open_audio(inchans,outchans,rate);

    if (midiin || midiout) linux_open_midi();
}

void sys_close_audio_and_midi(void)
{
    /* set timeout to avoid hanging close() call */

    sys_setalarm(1000000);

#ifdef HAVE_ALSA 
    if (linux_whichapi == API_ALSA)
     	alsa_close_audio();
    else
#endif
#ifdef HAVE_RME 
    if (linux_whichapi == API_RME)
     	rme9652_close_audio();
    else
#endif
    oss_close_audio();

    linux_close_midi();

    sys_setalarm(0);
}

int sys_send_dacs(void)
{
    if (linux_meters)
    {
    	int i, n;
	float maxsamp;
	for (i = 0, n = linux_inchannels * DACBLKSIZE, maxsamp = linux_inmax;
	    i < n; i++)
	{
	    float f = sys_soundin[i];
	    if (f > maxsamp) maxsamp = f;
	    else if (-f > maxsamp) maxsamp = -f;
	}
	linux_inmax = maxsamp;
	for (i = 0, n = linux_outchannels * DACBLKSIZE, maxsamp = linux_outmax;
	    i < n; i++)
	{
	    float f = sys_soundout[i];
	    if (f > maxsamp) maxsamp = f;
	    else if (-f > maxsamp) maxsamp = -f;
	}
	linux_outmax = maxsamp;
    }
#ifdef HAVE_ALSA 
    if (linux_whichapi == API_ALSA)
    	return alsa_send_dacs();
#endif
#ifdef HAVE_RME
    if (linux_whichapi == API_RME)
    	return rme9652_send_dacs();
#endif
    return oss_send_dacs();
}

float sys_getsr(void)
{
     return (sys_dacsr);
}

int sys_get_outchannels(void)
{
     return (linux_outchannels); 
}

int sys_get_inchannels(void) 
{
     return (linux_inchannels);
}

void sys_audiobuf(int n)
{
     /* set the size, in milliseconds, of the audio FIFO */
     if (n < 5) n = 5;
     else if (n > 5000) n = 5000;
     sys_schedadvance = n * 1000;
}

void sys_getmeters(float *inmax, float *outmax)
{
    if (inmax)
    {
    	linux_meters = 1;
	*inmax = linux_inmax;
	*outmax = linux_outmax;
    }
    else
    	linux_meters = 0;
    linux_inmax = linux_outmax = 0;
}

void sys_reportidle(void)
{
}

void sys_putmidimess(int portno, int a, int b, int c)
{
    if (portno >= 0 && portno < oss_nports)
    {
       switch (md_msglen(a))
       {
       case 2:
	    midi_out(oss_midiport[portno],a);	 
	    midi_out(oss_midiport[portno],b);	 
	    midi_out(oss_midiport[portno],c);
	    return;
       case 1:
	    midi_out(oss_midiport[portno],a);	 
	    midi_out(oss_midiport[portno],b);	 
	    return;
       case 0:
	    midi_out(oss_midiport[portno],a);	 
	    return;
       };
    }
}

void sys_putmidibyte(int portno, int byte)
{
    if (portno >= 0 && portno < oss_nports)
    	midi_out(oss_midiport[portno], byte);	
}

void sys_poll_midi(void)
{
    int i, throttle = 100;

    for (i = 0; i < oss_nports; i++)
    {
	while ((throttle-- >= 0) && (midi_instat(oss_midiport[i]) > 0))
	{
	    mp_parse(oss_mpa[i],midi_in(oss_midiport[i]));
    	}
    }
}

void sys_set_priority(int higher) 
{
    struct sched_param par;
    int p1 ,p2, p3;
#ifdef _POSIX_PRIORITY_SCHEDULING

    p1 = sched_get_priority_min(SCHED_FIFO);
    p2 = sched_get_priority_max(SCHED_FIFO);
    p3 = (higher ? p2 - 1 : p2 - 3);
    par.sched_priority = p3;

    if (sched_setscheduler(0,SCHED_FIFO,&par) != -1)
       fprintf(stderr, "priority %d scheduling enabled.\n", p3);
#endif

#ifdef _POSIX_MEMLOCK
    if (mlockall(MCL_FUTURE) != -1) 
    	fprintf(stderr, "memory locking enabled.\n");
#endif
}

/* ------------ linux-specific command-line flags -------------- */

void linux_frags(int n) {
     oss_nfragment = n;
     oss_oldaudio = 1;
     if (sys_verbose)
     	post("fragment count specified; using old-fashioned audio I/O");
}

void linux_fragsize(int n) {
     oss_logfragsize = n;
     oss_oldaudio = 1;
     if (sys_verbose)
     	post("fragment count specified; using old-fashioned audio I/O");
}

void linux_set_sound_api(int which)
{
     linux_whichapi = which;
     if (sys_verbose)
     	post("linux_whichapi %d", linux_whichapi);
}

#ifdef HAVE_ALSA
void linux_alsa_devno(int devno)
{
    alsa_devno = devno;
}

#endif

/* -------------- Audio I/O using the OSS API ------------------ */ 

int oss_reset(int fd) {
     int err;
     if ((err = ioctl(fd,SNDCTL_DSP_RESET)) < 0)
	  error("OSS: Could not reset");
     return err;
}

void oss_configure(int dev, int fd, int srate, int dac)
{
    int orig, param, nblk;
    audio_buf_info ainfo;
    orig = param = srate; 

    /* samplerate */

    if (ioctl(fd,SNDCTL_DSP_SPEED,&param) == -1)
     fprintf(stderr,"OSS: Could not set sampling rate for device\n");
    else if( orig != param )
    	fprintf(stderr,"OSS: sampling rate: wanted %d, got %d\n",
	    orig, param );				

    	/* I don't know if you can really do this in general */
    if (oss_oldaudio)
    { 

    	/* setting fragment count and size */

    	param = (oss_nfragment<<16) + oss_logfragsize;

    	if (ioctl(fd,SNDCTL_DSP_SETFRAGMENT,&param) == -1)
    	    fprintf(stderr,"OSS: Could not set fragment size\n");
    }

    /* setting resolution */

    orig = param = AFMT_S16_NE;
    if (ioctl(fd,SNDCTL_DSP_SETFMT,&param) == -1)
    	fprintf(stderr,"OSS: Could not set DSP format\n");
    else if( orig != param )
    	fprintf(stderr,"OSS: DSP format: wanted %d, got %d\n",orig, param );

    /* setting channels */

    orig = param = OSSCHPERDEV;

    if (ioctl(fd,SNDCTL_DSP_CHANNELS,&param) == -1)
    	fprintf(stderr,"OSS: Could not set channels\n");
    else if( orig != param )
    	fprintf(stderr,"OSS: num channels: wanted %d, got %d\n",orig, param );

    if (dac)
    {
	/* use "free space" to learn the buffer size.  Normally you
	should set this to your own desired value; but this seems not
	to be implemented uniformly across different sound cards.  LATER
	we should figure out what to do if the requested scheduler advance
	is greater than this buffer size; for now, we just print something
	out.  */

    	int defect;
	if (ioctl(linux_dacs[dev].fd, SOUND_PCM_GETOSPACE,&ainfo) < 0)
	   fprintf(stderr,"OSS: ioctl on output device %d failed",dev);
	linux_dacs[dev].bufsize = ainfo.bytes;

    	defect = linux_advance_samples * (OSS_SAMPLEWIDTH * OSSCHPERDEV)
	    - linux_dacs[dev].bufsize - OSS_XFERSIZE;
	if (defect > 0)
	{
	    if (sys_verbose || defect > (linux_dacs[dev].bufsize >> 2))
	    	fprintf(stderr,
	    	    "OSS: requested audio buffer size %d limited to %d\n",
		    	linux_advance_samples * (OSS_SAMPLEWIDTH *OSSCHPERDEV),
		    	linux_dacs[dev].bufsize);
    	    linux_advance_samples =
	    	(linux_dacs[dev].bufsize - OSS_XFERSAMPS) /
		    (OSS_SAMPLEWIDTH *OSSCHPERDEV);
	}
    }
}

int oss_open_audio(int inchans, int outchans,int srate)
{  
    int orig;
    int tmp;
    int inchannels = 0,outchannels = 0;
    char devname[20];
    int i;
    char* buf[OSS_SAMPLEWIDTH*OSSCHPERDEV*DACBLKSIZE];
    int num_devs = 0;
    audio_buf_info ainfo;

    linux_nadcs = linux_ndacs = 0;

    /* First check if we can do full duplex */
    /* and open the write ports	*/

    for (num_devs=0; outchannels < outchans; num_devs++) {
	int channels = 2;
	if (num_devs)
	   sprintf(devname, ossdsp, num_devs);
	else
	   sprintf(devname,"/dev/dsp");
    	sys_setalarm(1000000);

	if ((tmp = open(devname, O_WRONLY)) == -1)
	{
	    post("%s (writeonly): %s",
	    	devname, strerror(errno));
	    break;
	}
    	else post("successfully opened %s write-only", devname);
	/* if (ioctl(tmp,SNDCTL_DSP_GETCAPS,&orig) == -1)
	    error("OSS: SNDCTL_DSP_GETCAPS failed %s",devname); */

	if (inchans > inchannels)
	{
	    if (1) /* used to test for DSP_CAP_DUPLEX; better just to try it */
	    {
		close(tmp);
    	    	sys_setalarm(1000000);
		if ((linux_dacs[linux_ndacs].fd = 
		   linux_adcs[linux_nadcs].fd = 
		   open(devname, O_RDWR)) < 0)
		{
		    post("%s (read/write): %s", devname, strerror(errno));
		    post("continuing without audio");
    	    	    linux_setchsr(2,2,44100);
    	    	    sys_setalarm(0);
		    return 0;
		}
		else
		{
		    post("successfully reopened %s read/write", devname);
		    linux_ndacs++;
		    linux_nadcs++;
		    outchannels += channels;
		    inchannels += channels;
		}
	    }
	    else
	    {
		linux_dacs[linux_ndacs].fd = tmp;
		outchannels += channels;
		linux_ndacs++;
	    }
	}
	else
	{
	   linux_dacs[linux_ndacs].fd = tmp;
	   outchannels += channels;
	   linux_ndacs++;
	}
     }

    for (; inchans > inchannels; linux_nadcs++, num_devs++)
    {
	int channels = 2;
	if (num_devs)
	   sprintf(devname,ossdsp,num_devs);
	else
	   sprintf(devname,"/dev/dsp");

    	sys_setalarm(1000000);
	if ((tmp = open(devname,O_RDONLY)) == -1)
	{
    	    post("%s (readonly): %s", devname, strerror(errno));
	    break;
	}
	else
	{
	    post("OSS: opened %s read-only", devname);
	    linux_adcs[linux_nadcs].fd = tmp;
	}
	inchannels += channels;
    }

    	/* set sample rate; this also calculates linux_advance_samples.  */
    linux_setchsr(inchannels, outchannels, srate);

    /* configure soundcards */

    for (i=0;i<linux_nadcs;i++)
	oss_configure(i, linux_adcs[i].fd, srate, 0);

    for (i=0; i<linux_ndacs; i++)
	oss_configure(i, linux_dacs[i].fd, srate, 1);

    
      /* We have to do a read to start the engine. This is 
	 necessary because sys_send_dacs waits until the input
	 buffer is filled and only reads on a filled buffer.
	 This is good, because it's a way to make sure that we
	 will not block */

    if (linux_nadcs) {
    	if (sys_verbose)
	    fprintf(stderr,("OSS: issuing first ADC 'read' ... "));
	read(linux_adcs[0].fd, buf, OSS_SAMPLEWIDTH*OSSCHPERDEV*DACBLKSIZE);
    	if (sys_verbose)
	    fprintf(stderr, "...done.\n");
    }
    sys_setalarm(0);
    return (0);
}

void oss_close_audio( void)
{
     int i;
     for (i=0;i<linux_ndacs;i++)
	  close(linux_dacs[i].fd);

     for (i=0;i<linux_nadcs;i++)
          close(linux_adcs[i].fd);
}

static int linux_dacs_write(int fd,void* buf,long bytes) {
    return write(fd, buf,bytes);
}

static int linux_adcs_read(int fd,void*  buf,long bytes) {
     return read(fd, buf,bytes);
}

    /* query audio devices for "available" data size. */
static void oss_calcspace(void)
{
    int dev;
    audio_buf_info ainfo;
    for (dev=0; dev < linux_ndacs; dev++)
    {
	if (ioctl(linux_dacs[dev].fd, SOUND_PCM_GETOSPACE,&ainfo) < 0)
	   fprintf(stderr,"OSS: ioctl on output device %d failed",dev);
	linux_dacs[dev].space = ainfo.bytes;
    }

    for (dev = 0; dev < linux_nadcs; dev++)
    {
	if (ioctl(linux_adcs[dev].fd, SOUND_PCM_GETISPACE,&ainfo) < 0)
	    fprintf(stderr, "OSS: ioctl on input device %d, fd %d failed",
    	    	dev, linux_adcs[dev].fd);
	linux_adcs[dev].space = ainfo.bytes;
    }
}

void linux_audiostatus(void)
{
    int dev;
    if (!oss_oldaudio)
    {
	oss_calcspace();
	for (dev=0; dev < linux_ndacs; dev++)
	    fprintf(stderr, "dac %d space %d\n", dev, linux_dacs[dev].space);

	for (dev = 0; dev < linux_nadcs; dev++)
	    fprintf(stderr, "adc %d space %d\n", dev, linux_adcs[dev].space);

    }
}

/* this call resyncs audio output and input which will cause discontinuities
in audio output and/or input. */ 

static void oss_doresync( void)
{
    int dev, zeroed = 0, wantsize;
    t_oss_sample buf[OSS_XFERSAMPS];
    audio_buf_info ainfo;

    	/* 1. if any input devices are ahead (have more than 1 buffer stored),
	    drop one or more buffers worth */
    for (dev = 0; dev < linux_nadcs; dev++)
    {
    	if (linux_adcs[dev].space == 0)
	{
	    linux_adcs_read(linux_adcs[dev].fd, buf, OSS_XFERSIZE);
	}
	else while (linux_adcs[dev].space > OSS_XFERSIZE)
	{
    	    linux_adcs_read(linux_adcs[dev].fd, buf, OSS_XFERSIZE);
	    if (ioctl(linux_adcs[dev].fd, SOUND_PCM_GETISPACE, &ainfo) < 0)
	    {
	    	fprintf(stderr, "OSS: ioctl on input device %d, fd %d failed",
    	    	    dev, linux_adcs[dev].fd);
	    	break;
	    }
	    linux_adcs[dev].space = ainfo.bytes;
	}
    }

    	/* 2. if any output devices are behind, feed them zeros to catch them
	    up */
    for (dev = 0; dev < linux_ndacs; dev++)
    {
    	while (linux_dacs[dev].space > linux_dacs[dev].bufsize - 
	    linux_advance_samples * (OSSCHPERDEV * OSS_SAMPLEWIDTH))
	{
    	    if (!zeroed)
	    {
	    	int i;
		for (i = 0; i < OSS_XFERSAMPS; i++) buf[i] = 0;
		zeroed = 1;
	    }
    	    linux_dacs_write(linux_dacs[dev].fd, buf, OSS_XFERSIZE);
	    if (ioctl(linux_dacs[dev].fd, SOUND_PCM_GETOSPACE, &ainfo) < 0)
	    {
	    	fprintf(stderr, "OSS: ioctl on output device %d, fd %d failed",
    	    	    dev, linux_dacs[dev].fd);
	    	break;
	    }
	    linux_dacs[dev].space = ainfo.bytes;
	}
    }
    	/* 3. if any DAC devices are too far ahead, plan to drop the
	    number of frames which will let the others catch up. */
    for (dev = 0; dev < linux_ndacs; dev++)
    {
    	if (linux_dacs[dev].space > linux_dacs[dev].bufsize - 
	    (linux_advance_samples - 1) * (OSSCHPERDEV * OSS_SAMPLEWIDTH))
	{
    	    linux_dacs[dev].dropcount = linux_advance_samples - 1 - 
	    	(linux_dacs[dev].space - linux_dacs[dev].bufsize) /
		     (OSSCHPERDEV * OSS_SAMPLEWIDTH) ;
	}
	else linux_dacs[dev].dropcount = 0;
    }
}

int oss_send_dacs(void)
{
    float *fp1, *fp2;
    long fill;
    int i, j, dev, rtnval = SENDDACS_YES;
    t_oss_sample buf[OSSCHPERDEV * DACBLKSIZE], *sp;
    	/* the maximum number of samples we should have in the ADC buffer */
    int idle = 0;
    double timeref, timenow;

    if (!linux_nadcs && !linux_ndacs)
    	return (SENDDACS_NO);

    if (!oss_oldaudio)
    	oss_calcspace();
    
    	/* determine whether we're idle.  This is true if either (1)
    	some input device has less than one buffer to read or (2) some
	output device has fewer than (linux_advance_samples) blocks buffered
	already. */
    for (dev=0; dev < linux_ndacs; dev++)
	if (linux_dacs[dev].dropcount ||
	    (linux_dacs[dev].bufsize - linux_dacs[dev].space >
	    	linux_advance_samples * (OSS_SAMPLEWIDTH * OSSCHPERDEV)))
    	    	    idle = 1;
    for (dev=0; dev < linux_nadcs; dev++)
	if (linux_adcs[dev].space < OSS_XFERSIZE)
	    idle = 1;

    if (idle && !oss_oldaudio)
    {
    	    /* sometimes---rarely---when the ADC available-byte-count is
	    zero, it's genuine, but usually it's because we're so
	    late that the ADC has overrun its entire kernel buffer.  We
	    distinguish between the two by waiting 2 msec and asking again.
	    There should be an error flag we could check instead; look for this
    	    someday... */
    	for (dev = 0;dev < linux_nadcs; dev++) 
	    if (linux_adcs[dev].space == 0)
	{
    	    audio_buf_info ainfo;
	    sys_microsleep(2000);
	    oss_calcspace();
	    if (linux_adcs[dev].space != 0) continue;

    	    	/* here's the bad case.  Give up and resync. */
	    sys_log_error(ERR_DATALATE);
	    oss_doresync();
	    return (SENDDACS_NO);
	}
    	    /* check for slippage between devices, either because
	    data got lost in the driver from a previous late condition, or
	    because the devices aren't synced.  When we're idle, no
	    input device should have more than one buffer readable and
	    no output device should have less than linux_advance_samples-1
	    */
	    
	for (dev=0; dev < linux_ndacs; dev++)
	    if (!linux_dacs[dev].dropcount &&
		(linux_dacs[dev].bufsize - linux_dacs[dev].space <
	    	    (linux_advance_samples - 2) *
		    	(OSS_SAMPLEWIDTH * OSSCHPERDEV)))
    	    		goto badsync;
	for (dev=0; dev < linux_nadcs; dev++)
	    if (linux_adcs[dev].space > 3 * OSS_XFERSIZE)
		goto badsync;

	    /* return zero to tell the scheduler we're idle. */
    	return (SENDDACS_NO);
    badsync:
	sys_log_error(ERR_RESYNC);
	oss_doresync();
	return (SENDDACS_NO);
    	
    }

	/* do output */

    timeref = sys_getrealtime();
    for (dev=0; dev < linux_ndacs; dev++)
    {
    	if (linux_dacs[dev].dropcount)
	    linux_dacs[dev].dropcount--;
	else
	{
	    for (i = DACBLKSIZE,  fp1 = sys_soundout +	
	    	OSSCHPERDEV*DACBLKSIZE*dev,
	    	sp = buf; i--; fp1++, sp += OSSCHPERDEV)
	    {
		for (j=0, fp2 = fp1; j<OSSCHPERDEV; j++, fp2 += DACBLKSIZE)
		{
	    		/* clip, don't wrap */
	    	    int s = *fp2 * 32767.;
		    if (s > 32767) s = 32767;
		    else if (s < -32767) s = -32767;
	    	    sp[j] = s;
		}
	    }
	    linux_dacs_write(linux_dacs[dev].fd, buf, OSS_XFERSIZE);
	    if ((timenow = sys_getrealtime()) - timeref > 0.002)
	    {
	    	if (!oss_oldaudio)
		    sys_log_error(ERR_DACSLEPT);
		else
		    rtnval = SENDDACS_SLEPT;
    	    }
	    timeref = timenow;
    	}
    }
    memset(sys_soundout, 0,
    	linux_ndacs * (sizeof(float) * OSSCHPERDEV* DACBLKSIZE));

    	/* do input */

    for (dev = 0; dev < linux_nadcs; dev++)
    {

    	linux_adcs_read(linux_adcs[dev].fd, buf, OSS_XFERSIZE);

	if ((timenow = sys_getrealtime()) - timeref > 0.002)
	{
	    if (!oss_oldaudio)
	    	sys_log_error(ERR_ADCSLEPT);
	    else
	    	rtnval = SENDDACS_SLEPT;
    	}
	timeref = timenow;

	for (i = DACBLKSIZE,fp1 = sys_soundin + OSSCHPERDEV*DACBLKSIZE*dev,
	    sp = buf; i--; fp1++,  sp += OSSCHPERDEV)
	{
    	    for (j=0;j<linux_inchannels;j++)
    	    	fp1[j*DACBLKSIZE] = (float)sp[j]*3.051850e-05;
	}	  	  
     }	
     return (rtnval);
}

/* ----------------- audio I/O using the ALSA native API ---------------- */

/* rewritten for ALSA pcmv2 api by Karl MacMillan<karlmac@peabody.jhu.edu> */ 

#ifdef HAVE_ALSA
static int alsa_open_audio(int wantinchans, int wantoutchans,
    int srate)
{
    int dir, voices, bsize;
    int err, id, rate, i;
    char *cardname;
    snd_ctl_hw_info_t hwinfo;
    snd_pcm_info_t pcminfo;
    snd_pcm_channel_info_t channelinfo;
    snd_ctl_t *handle;
    snd_pcm_sync_t sync;

    linux_inchannels = 0;
    linux_outchannels = 0;
    
    rate = 44100;
    alsa_samplewidth = 4;   	/* first try 4 byte samples */

    if (!wantinchans && !wantoutchans)
    	return (1);

    if (sys_verbose)
    {
	if ((err = snd_card_get_longname(alsa_devno-1, &cardname)) < 0)
	{
	    fprintf(stderr, "PD-ALSA: unable to get name of card number %d\n",
	    	alsa_devno);
	    return 1;
	}
	fprintf(stderr, "PD-ALSA: using card %s\n", cardname);
	free(cardname);
    }

    if ((err = snd_ctl_open(&handle, alsa_devno-1)) < 0)
    {
	fprintf(stderr, "PD-ALSA: unable to open control: %s\n",
    	    snd_strerror(err));
	return 1;
    }

    if ((err = snd_ctl_hw_info(handle, &hwinfo)) < 0)
    {
	fprintf(stderr, "PD-ALSA: unable to open get info: %s\n",
    	    snd_strerror(err));
	return 1;
    }
    if (hwinfo.pcmdevs < 1)
    {
	fprintf(stderr, "PD-ALSA: device %d doesn't support PCM\n",
	    alsa_devno);
    	snd_ctl_close(handle);
	return 1;
    }

    if ((err = snd_ctl_pcm_info(handle, 0, &pcminfo)) < 0)
    {
	fprintf(stderr, "PD-ALSA: unable to open get pcm info: %s\n",
	    snd_strerror(err));
    	snd_ctl_close(handle);
	return (1);
    }
    snd_ctl_close(handle);

    	/* find out if opening for input, output, or both and check that the
	device can handle it. */
    if (wantinchans && wantoutchans)
    {
	if (!(pcminfo.flags & SND_PCM_INFO_DUPLEX))
	{
	    fprintf(stderr, "PD-ALSA: device is not full duplex\n");
	    return (1);
	}
	dir = SND_PCM_OPEN_DUPLEX;
    }
    else if (wantoutchans)
    {
	if (!(pcminfo.flags & SND_PCM_INFO_PLAYBACK))
	{
	    fprintf(stderr, "PD-ALSA: device is not full duplex\n");
	    return (1);
	}
    	dir = SND_PCM_OPEN_PLAYBACK;
    }
    else
    {
	if (!(pcminfo.flags & SND_PCM_INFO_CAPTURE))
	{
	    fprintf(stderr, "PD-ALSA: device is not full duplex\n");
	    return (1);
	}
    	dir = SND_PCM_OPEN_CAPTURE;
    }

    	/* try to open the device */
    if ((err = snd_pcm_open(&alsa_device[0].handle, alsa_devno-1, 0, dir)) < 0)
    {
	fprintf(stderr, "PD-ALSA: error opening device: %s\n",
	    snd_strerror(err));
	return (1);
    }
    	/* get information from the handle */
    if (wantinchans)
    {
	channelinfo.channel = SND_PCM_CHANNEL_CAPTURE;
	channelinfo.subdevice = 0;
	if ((err = snd_pcm_channel_info(alsa_device[0].handle, &channelinfo))
	    < 0)
	{
	    fprintf(stderr, "PD-ALSA: snd_pcm_channel_info (input): %s\n",
		snd_strerror(err));
	    return (1);
	}
	if (sys_verbose)
	    post("input channels supported: %d-%d\n",
	    	channelinfo.min_voices, channelinfo.max_voices);

	if (wantinchans < channelinfo.min_voices)
	    post("increasing input channels to minimum of %d\n",
		wantinchans = channelinfo.min_voices);
	if (wantinchans > channelinfo.max_voices)
	    post("decreasing input channels to maximum of %d\n",
		wantinchans = channelinfo.max_voices);
	if (alsa_samplewidth == 4 &&
	    !(channelinfo.formats & (1<<SND_PCM_SFMT_S32_LE)))
	{
	    fprintf(stderr,
	    	"PD_ALSA: input doesn't support 32-bit samples; using 16\n");
	    alsa_samplewidth = 2;
	}
	if (alsa_samplewidth == 2 &&
	    !(channelinfo.formats & (1<<SND_PCM_SFMT_S16_LE)))
	{
	    fprintf(stderr,
	    	"PD_ALSA: can't find 4 or 2 byte format; giving up\n");
	    return (1);
	}
    }

    if (wantoutchans)
    {
	channelinfo.channel = SND_PCM_CHANNEL_PLAYBACK;
	channelinfo.subdevice = 0;
	if ((err = snd_pcm_channel_info(alsa_device[0].handle, &channelinfo))
	    < 0)
	{
	    fprintf(stderr, "PD-ALSA: snd_pcm_channel_info (output): %s\n",
		snd_strerror(err));
	    return (1);
	}
	if (sys_verbose)
	    post("output channels supported: %d-%d\n",
	    	channelinfo.min_voices, channelinfo.max_voices);
	if (wantoutchans < channelinfo.min_voices)
	    post("increasing output channels to minimum of %d\n",
		wantoutchans = channelinfo.min_voices);
	if (wantoutchans > channelinfo.max_voices)
	    post("decreasing output channels to maximum of %d\n",
		wantoutchans = channelinfo.max_voices);
	if (alsa_samplewidth == 4 &&
	    !(channelinfo.formats & (1<<SND_PCM_SFMT_S32_LE)))
	{
	    fprintf(stderr,
	    	"PD_ALSA: output doesn't support 32-bit samples; using 16\n");
	    alsa_samplewidth = 2;
	}
	if (alsa_samplewidth == 2 &&
	    !(channelinfo.formats & (1<<SND_PCM_SFMT_S16_LE)))
	{
	    fprintf(stderr,
	    	"PD_ALSA: can't find 4 or 2 byte format; giving up\n");
	    return (1);
	}
    }

    linux_setchsr(wantinchans, wantoutchans, rate);

    if (wantinchans)
    	alsa_set_params(&alsa_device[0], SND_PCM_CHANNEL_CAPTURE,
    	    srate, wantinchans);
    if (wantoutchans)
    	alsa_set_params(&alsa_device[0], SND_PCM_CHANNEL_PLAYBACK,
    	    srate, wantoutchans);

    n_alsa_dev = 1;

    /* check that all is as we think it should be */
    for (i = 0; i < n_alsa_dev; i++)
    {
	/* We need to handle if the rate is not the same for all
	* devices.  For now just hope. */
	rate = alsa_device[i].setup.format.rate;

    /* It turns out that this checking does not work on all of my cards
    * - in full duplex on my trident 4dwave the setup on the capture channel
    * shows a sampling rate of 0.  This is not true on my ess solo1.  Checking
    * the dac last helps the problem.  All of this needs to be much smarter
    * anyway (last minute hack).  A warning above is all I have time for.
    */
	if (rate != srate)
	{
	    post("PD-ALSA: unable to obtain rate %i using %i", srate, rate);
	    post("PD-ALSA: (despite this warning Pd might still work.)");
	}
    }
    bsize = alsa_samplewidth *
    	(linux_inchannels > linux_outchannels ? linux_inchannels :
	    linux_outchannels) * DACBLKSIZE;
    alsa_buf = malloc(bsize);
    if (!alsa_buf)
    	return (1);
    memset(alsa_buf, 0, bsize);
    return 0;
}

void alsa_set_params(t_alsa_dev *dev, int dir, int rate, int voices)
{
    int err;
    struct snd_pcm_channel_params params;

    memset(&dev->info, 0, sizeof(dev->info));
    dev->info.channel = dir;
    if ((err = snd_pcm_channel_info(dev->handle, &dev->info) < 0))
    {
	fprintf(stderr, "PD-ALSA: error getting channel info: %s\n",
	    snd_strerror(err));
    }
    memset(&params, 0, sizeof(params));
    params.format.interleave = 1;   /* may do non-interleaved later */
    	/* format is 2 or 4 bytes per sample depending on what was possible */
    params.format.format = 
    	(alsa_samplewidth == 4 ? SND_PCM_SFMT_S32_LE : SND_PCM_SFMT_S16_LE);

    	/*will check this further down -just try for now*/
    params.format.rate = rate; 
    params.format.voices = voices;
    params.start_mode = SND_PCM_START_GO;   /* seems most reliable */
    	                                    /*do not stop at overrun/underrun*/
    params.stop_mode = SND_PCM_STOP_ROLLOVER;

    params.channel = dir;                   /* playback|capture */
    params.buf.stream.queue_size =
	(ALSA_EXTRABUFFER + linux_advance_samples)
	    * alsa_samplewidth * voices;
    params.buf.stream.fill = SND_PCM_FILL_SILENCE_WHOLE;
    params.mode = SND_PCM_MODE_STREAM;

    if ((err = snd_pcm_channel_params(dev->handle, &params)) < 0)
    {
	printf("PD-ALSA: error setting parameters %s", snd_strerror(err));
    }

	/* This should clear the buffers but does not. There is often noise at
	startup that sounds like crap left in the buffers - maybe in the lib
	instead of the driver?  Some solution needs to be found.
	*/

    if ((err = snd_pcm_channel_prepare(dev->handle, dir)) < 0)
    {
	printf("PD-ALSA: error preparing channel %s", snd_strerror(err));
    }
    dev->setup.channel = dir;

    if ((err = snd_pcm_channel_setup(dev->handle, &dev->setup)) < 0)
    {
	printf("PD-ALSA: error getting setup %s", snd_strerror(err));
    }
    	/* for some reason, if you don't writesomething before starting the
	converters we get trash on startup */
    if (dir == SND_PCM_CHANNEL_PLAYBACK)
    {
    	char foo[1024];
	int xxx = 1024 - (1024 % (linux_outchannels * alsa_samplewidth));
	int i, r;
	for (i = 0; i < xxx; i++)
	    foo[i] = 0;
	if ((r = snd_pcm_write(dev->handle, foo, xxx)) < xxx)
	    fprintf(stderr, "alsa_write: %s\n", snd_strerror(errno));
    }
    snd_pcm_channel_go(dev->handle, dir);
}

void alsa_close_audio(void)
{
    int i;
    for(i = 0; i < n_alsa_dev; i++)
    	snd_pcm_close(alsa_device[i].handle);
}

/* #define DEBUG_ALSA_XFER */

int alsa_send_dacs(void)
{
    static int16_t *sp;
    t_sample *fp, *fp1, *fp2;
    int i, j, k, err, devno = 0;
    int inputcount = 0, outputcount = 0, inputlate = 0, outputlate = 0;
    int result; 
    snd_pcm_channel_status_t stat;
    static int callno = 0;
    static int xferno = 0;
    int countwas = 0;
    double timelast;
    static double timenow;
    int inchannels = linux_inchannels;
    int outchannels = linux_outchannels;
    int inbytesperframe = inchannels * alsa_samplewidth;
    int outbytesperframe = outchannels * alsa_samplewidth;
    int intransfersize = DACBLKSIZE * inbytesperframe;
    int outtransfersize = DACBLKSIZE * outbytesperframe;
    int alsaerror;
    int loggederror = 0;

    if (!inchannels && !outchannels)
    	return (SENDDACS_NO);
    timelast = timenow;
    timenow = sys_getrealtime();

#ifdef DEBUG_ALSA_XFER
    if (timenow - timelast > 0.050)
    	fprintf(stderr, "(%d)",
	    (int)(1000 * (timenow - timelast))), fflush(stderr);
#endif

    callno++;
    	    /* get input and output channel status */
    if (inchannels > 0)
    {
    	devno = 0;
    	stat.channel = SND_PCM_CHANNEL_CAPTURE;
	if (alsaerror = snd_pcm_channel_status(alsa_device[devno].handle,
	    &stat))
	{
	    fprintf(stderr, "snd_pcm_channel_status (input): %s\n",
	    	snd_strerror(alsaerror));
	    return (SENDDACS_NO);
	}
	inputcount = stat.count;
	inputlate = (stat.underrun > 0 || stat.overrun > 0);
    }
    if (outchannels > 0)
    {
	devno = 0;
    	stat.channel = SND_PCM_CHANNEL_PLAYBACK;
	if (alsaerror = snd_pcm_channel_status(alsa_device[devno].handle,
	    &stat))
	{
	    fprintf(stderr, "snd_pcm_channel_status (output): %s\n",
	    	snd_strerror(alsaerror));
	    return (SENDDACS_NO);
	}
	outputcount = stat.count;
    	outputlate = (stat.underrun > 0 || stat.overrun > 0);
    }

	/* check if input not ready */
    if (inputcount < intransfersize)
    {
	/* fprintf(stderr, "no adc; count %d, free %d, call %d, xfer %d\n",
	    stat.count, 
	    stat.free,
	    callno, xferno); */
	if (outchannels > 0)
	{
	    /* if there's no input but output is hungry, feed output. */
	    while (outputcount < (linux_advance_samples + ALSA_JITTER)
		* outbytesperframe)
	    {
	    	if (!loggederror)
		    sys_log_error(ERR_RESYNC), loggederror = 1;
		memset(alsa_buf, 0, outtransfersize);
		result = snd_pcm_write(alsa_device[devno].handle,
	    	    alsa_buf, outtransfersize);
		if (result < outtransfersize)
		{
#ifdef DEBUG_ALSA_XFER
		    if (result >= 0 || errno == EAGAIN)
			fprintf(stderr, "ALSA: write returned %d of %d\n",
			    result, outtransfersize);
		    else fprintf(stderr, "ALSA: write: %s\n",
	    		snd_strerror(errno));
		    fprintf(stderr,
		    	"inputcount %d, outputcount %d, outbufsize %d\n",
		    	inputcount, outputcount, 
	    	    	    (ALSA_EXTRABUFFER + linux_advance_samples)
	    	    	    	* alsa_samplewidth * outchannels);
#endif
		    return (SENDDACS_NO);
		}
    		stat.channel = SND_PCM_CHANNEL_PLAYBACK;
		if (alsaerror =
		    snd_pcm_channel_status(alsa_device[devno].handle,
		    	&stat))
		{
		    fprintf(stderr, "snd_pcm_channel_status (output): %s\n",
	    		snd_strerror(alsaerror));
		    return (SENDDACS_NO);
		}
		outputcount = stat.count;
	    }
	}

	return SENDDACS_NO;
    }

    	/* if output buffer has at least linux_advance_samples in it, we're
	not ready for this batch. */
    if (outputcount > linux_advance_samples * outbytesperframe)
    {
    	if (inchannels > 0)
	{
    	    while (inputcount > (DACBLKSIZE + ALSA_JITTER) * outbytesperframe)
	    {
	    	if (!loggederror)
		    sys_log_error(ERR_RESYNC), loggederror = 1;
	    	devno = 0;
		result = snd_pcm_read(alsa_device[devno].handle, alsa_buf,
		    intransfersize);
		if (result < intransfersize)
		{
#ifdef DEBUG_ALSA_XFER
		    if (result < 0)
    	    	    	fprintf(stderr,
	    		"snd_pcm_read %d %d: %s\n", 
	    		callno, xferno, snd_strerror(errno));
		    else fprintf(stderr,
	    		"snd_pcm_read %d %d returned only %d\n", 
	    		callno, xferno, result);
		    fprintf(stderr,
		    	"inputcount %d, outputcount %d, inbufsize %d\n",
		    	inputcount, outputcount, 
	    	    	    (ALSA_EXTRABUFFER + linux_advance_samples)
	    	    	    	* alsa_samplewidth * inchannels);
#endif
		    return (SENDDACS_NO);
		}
    		devno = 0;
    		stat.channel = SND_PCM_CHANNEL_CAPTURE;
		if (alsaerror =
		    snd_pcm_channel_status(alsa_device[devno].handle,
		    	&stat))
		{
		    fprintf(stderr, "snd_pcm_channel_status (input): %s\n",
	    		snd_strerror(alsaerror));
		    return (SENDDACS_NO);
		}
		inputcount = stat.count;
		inputlate = (stat.underrun > 0 || stat.overrun > 0);
	    }
	    return (SENDDACS_NO);
    	}
    }
    if (sys_getrealtime() - timenow > 0.002)
    {
#ifdef DEBUG_ALSA_XFER
    	fprintf(stderr, "check %d took %d msec\n",
	    callno, (int)(1000 * (timenow - timelast))), fflush(stderr);
#endif
    	sys_log_error(ERR_DACSLEPT);
    	timenow = sys_getrealtime();
    }
    if (inputlate || outputlate)
    	sys_log_error(ERR_DATALATE);    

    /* do output */
    	    /* this "for" loop won't work for more than one device. */
    for (devno = 0, fp = sys_soundout; devno < (outchannels > 0); devno++, 
       fp += 128)
    {
    	if (alsa_samplewidth == 4)
	{
	    for (i = 0, fp1 = fp; i < outchannels; i++, fp1 += DACBLKSIZE)
	    {
		for (j = i, k = DACBLKSIZE, fp2 = fp1; k--;
		    j += outchannels, fp2++)
	        {
		    float s1 = *fp2 * INT32_MAX;
		    ((t_alsa_sample32 *)alsa_buf)[j] = CLIP32(s1);
    	    	} 
	    }
	}
	else
	{
	    for (i = 0, fp1 = fp; i < outchannels; i++, fp1 += DACBLKSIZE)
	    {
		for (j = i, k = DACBLKSIZE, fp2 = fp1; k--;
		    j += outchannels, fp2++)
		{
		    int s = *fp2 * 32767.;
		    if (s > 32767)
			s = 32767;
		    else if (s < -32767)
			s = -32767;
		    ((t_alsa_sample16 *)alsa_buf)[j] = s;
		}
	    }
	}
	
	result = snd_pcm_write(alsa_device[devno].handle, alsa_buf,
	    outtransfersize);
	if (result < outtransfersize)
	{
#ifdef DEBUG_ALSA_XFER
	    if (result >= 0 || errno == EAGAIN)
		fprintf(stderr, "ALSA: write returned %d of %d\n",
		    result, outtransfersize);
	    else fprintf(stderr, "ALSA: write: %s\n",
	    	snd_strerror(errno));
	    fprintf(stderr,
		"inputcount %d, outputcount %d, outbufsize %d\n",
		inputcount, outputcount, 
	    	    (ALSA_EXTRABUFFER + linux_advance_samples)
	    	    	* alsa_samplewidth * outchannels);
#endif
    	    sys_log_error(ERR_DACSLEPT);
	    return (SENDDACS_NO);
	}
    }
    	/* zero out the output buffer */
    memset(sys_soundout, 0, DACBLKSIZE * sizeof(*sys_soundout) *
    	linux_outchannels);
    if (sys_getrealtime() - timenow > 0.002)
    {
#if DEBUG_ALSA_XFER
    	fprintf(stderr, "output %d took %d msec\n",
	    callno, (int)(1000 * (timenow - timelast))), fflush(stderr);
#endif
    	timenow = sys_getrealtime();
    	sys_log_error(ERR_DACSLEPT);
    }

    /* do input */
    for (devno = 0, fp = sys_soundin; devno < (linux_inchannels > 0); devno++, 
        fp += 128)
    {
	result = snd_pcm_read(alsa_device[devno].handle, alsa_buf,
	    intransfersize);
	if (result < intransfersize)
	{
#ifdef DEBUG_ALSA_XFER
	    if (result < 0)
    	    	fprintf(stderr,
	    	"snd_pcm_read %d %d: %s\n", 
	    	callno, xferno, snd_strerror(errno));
	    else fprintf(stderr,
	    	"snd_pcm_read %d %d returned only %d\n", 
	    	callno, xferno, result);
	    fprintf(stderr,
		"inputcount %d, outputcount %d, inbufsize %d\n",
		inputcount, outputcount, 
	    	    (ALSA_EXTRABUFFER + linux_advance_samples)
	    	    	* alsa_samplewidth * inchannels);
#endif
    	    sys_log_error(ERR_ADCSLEPT);
	    return (SENDDACS_NO);
	}
	if (alsa_samplewidth == 4)
	{
	    for (i = 0, fp1 = fp; i < inchannels; i++, fp1 += DACBLKSIZE)
	    {
		for (j = i, k = DACBLKSIZE, fp2 = fp1; k--;
		    j += inchannels, fp2++)
	    	    *fp2 = (float) ((t_alsa_sample32 *)alsa_buf)[j]
		    	* (1./ INT32_MAX);
	    }
	}
	else
	{
	    for (i = 0, fp1 = fp; i < inchannels; i++, fp1 += DACBLKSIZE)
	    {
		for (j = i, k = DACBLKSIZE, fp2 = fp1; k--; j += inchannels, fp2++)
	    	    *fp2 = (float) ((t_alsa_sample16 *)alsa_buf)[j]
		    	* 3.051850e-05;
	    }
    	}
    }
    xferno++;
    if (sys_getrealtime() - timenow > 0.002)
    {
#ifdef DEBUG_ALSA_XFER
    	fprintf(stderr, "routine took %d msec\n",
	    (int)(1000 * (sys_getrealtime() - timenow)));
#endif
    	sys_log_error(ERR_ADCSLEPT);
    }
    return SENDDACS_YES;
}

#endif				/* HAVE_ALSA */


/***************************************************
 *  Code using the RME_9652 API
 */ 

    /*
    trying native device for future use of native memory map:
    because of busmaster if you dont use the dac, you dont need 
    CPU Power und also no nearly no CPU-Power is used in device

    since always all DAs and ADs are synced (else they wouldnt work)
    we use linux_dacs[0], linux_adcs[0]
    */

#ifdef HAVE_RME

#define RME9652_MAX_CHANNELS 26

#define RME9652_CH_PER_NATIVE_DEVICE 1

static int rme9652_dac_devices[RME9652_MAX_CHANNELS];
static int rme9652_adc_devices[RME9652_MAX_CHANNELS];

static char rme9652_dsp_dac[] = "/dev/rme9652/C0da%d"; 
static char rme9652_dsp_adc[] = "/dev/rme9652/C0ad%d"; 

static int num_of_rme9652_dac = 0;
static int num_of_rme9652_adc = 0;

static int rme_soundindevonset = 1;
static int rme_soundoutdevonset = 1;

void rme_soundindev(int which)
{
    rme_soundindevonset = which;
}

void rme_soundoutdev(int which)
{
    rme_soundoutdevonset = which;
}

void rme9652_configure(int dev, int fd,int srate, int dac) {
  int orig, param, nblk;
  audio_buf_info ainfo;
  orig = param = srate; 

  /* samplerate */

  fprintf(stderr,"RME9652: configuring %d, fd=%d, sr=%d\n, dac=%d\n",
			 dev,fd,srate,dac);

  if (ioctl(fd,SNDCTL_DSP_SPEED,&param) == -1)
	 fprintf(stderr,"RME9652: Could not set sampling rate for device\n");
  else if( orig != param )
	 fprintf(stderr,"RME9652: sampling rate: wanted %d, got %d\n",
				orig, param );				
  
  // setting the correct samplerate (could be different than expected)     
  srate = param;


  /* setting resolution */

  /* use ctrlpanel to change, experiment, channels 1 */

  orig = param = AFMT_S16_NE;
  if (ioctl(fd,SNDCTL_DSP_SETFMT,&param) == -1)
	 fprintf(stderr,"RME9652: Could not set DSP format\n");
  else if( orig != param )
    fprintf(stderr,"RME9652: DSP format: wanted %d, got %d\n",orig, param );

  /* setting channels */
  orig = param = RME9652_CH_PER_NATIVE_DEVICE;

  if (ioctl(fd,SNDCTL_DSP_CHANNELS,&param) == -1)
	 fprintf(stderr,"RME9652: Could not set channels\n");
  else if( orig != param )
    fprintf(stderr,"RME9652: num channels: wanted %d, got %d\n",orig, param );

  if (dac)
    {

 /* use "free space" to learn the buffer size.  Normally you
	 should set this to your own desired value; but this seems not
	 to be implemented uniformly across different sound cards.  LATER
	 we should figure out what to do if the requested scheduler advance
	 is greater than this buffer size; for now, we just print something
	 out.  */

	if( ioctl(linux_dacs[0].fd, SOUND_PCM_GETOSPACE,&ainfo) < 0 )
	  fprintf(stderr,"RME: ioctl on output device %d failed",dev);

	linux_dacs[0].bufsize = ainfo.bytes;

	fprintf(stderr,"RME: ioctl SOUND_PCM_GETOSPACE says %d buffsize\n",
			  linux_dacs[0].bufsize);


	if (linux_advance_samples * (RME_SAMPLEWIDTH *
	    RME9652_CH_PER_NATIVE_DEVICE) 
		 > linux_dacs[0].bufsize - RME_BYTESPERCHAN)
	  {
		 fprintf(stderr,
		    "RME: requested audio buffer size %d limited to %d\n",
		    linux_advance_samples 
		    * (RME_SAMPLEWIDTH * RME9652_CH_PER_NATIVE_DEVICE),
		    linux_dacs[0].bufsize);
    	    	linux_advance_samples =
		    (linux_dacs[0].bufsize - RME_BYTESPERCHAN) 
		    / (RME_SAMPLEWIDTH *RME9652_CH_PER_NATIVE_DEVICE);
	  }
    }
}

 
int rme9652_open_audio(int inchans, int outchans,int srate)
{  
    int orig;
    int tmp;
    int inchannels = 0,outchannels = 0;
    char devname[20];
    int i;
    char buf[RME_SAMPLEWIDTH*RME9652_CH_PER_NATIVE_DEVICE*DACBLKSIZE];
    int num_devs = 0;
    audio_buf_info ainfo;

    linux_nadcs = linux_ndacs = 0;

    post("RME open");
    /* First check if we can  */
    /* open the write ports	*/

    for (num_devs=0; outchannels < outchans; num_devs++)
    {
	int channels = RME9652_CH_PER_NATIVE_DEVICE;

	sprintf(devname, rme9652_dsp_dac, num_devs + rme_soundoutdevonset);
	if ((tmp = open(devname,O_WRONLY)) == -1) 
	{
	    DEBUG(fprintf(stderr,"RME9652: failed to open %s writeonly\n",
    	    	devname);)
	    break;
	}
	DEBUG(fprintf(stderr,"RME9652: out device Nr. %d (%d) on %s\n",
    	    linux_ndacs+1,tmp,devname);)

	if (outchans > outchannels)
	{
	    rme9652_dac_devices[linux_ndacs] = tmp;
	    linux_ndacs++;
	    outchannels += channels;
	}
	else close(tmp);
    }
    if( linux_ndacs > 0)
	linux_dacs[0].fd =  rme9652_dac_devices[0];

    /* Second check if we can  */
    /* open the read ports	*/

    for (num_devs=0; inchannels < inchans; num_devs++)
    {
	int channels = RME9652_CH_PER_NATIVE_DEVICE;

	sprintf(devname, rme9652_dsp_adc, num_devs+rme_soundindevonset);

	if ((tmp = open(devname,O_RDONLY)) == -1) 
	{
	    DEBUG(fprintf(stderr,"RME9652: failed to open %s readonly\n",
	    	devname);)
	    break;
	}
	DEBUG(fprintf(stderr,"RME9652: in device Nr. %d (%d) on %s\n",
	    linux_nadcs+1,tmp,devname);)

	if (inchans > inchannels)
	{
	    rme9652_adc_devices[linux_nadcs] = tmp;
	    linux_nadcs++;
	    inchannels += channels;
	}
	else
	     close(tmp);
    }
    if(linux_nadcs > 0)
	linux_adcs[0].fd = rme9652_adc_devices[0];

    /* configure soundcards */

	 rme9652_configure(0, linux_adcs[0].fd,srate, 0);
	 rme9652_configure(0, linux_dacs[0].fd,srate, 1);
    
	 /* We have to do a read to start the engine. This is 
		 necessary because sys_send_dacs waits until the input
		 buffer is filled and only reads on a filled buffer.
		 This is good, because it's a way to make sure that we
		 will not block */
	 
    if (linux_nadcs)
    {
	fprintf(stderr,("RME9652: starting read engine ... "));


	for (num_devs=0; num_devs < linux_nadcs; num_devs++) 
	  read(rme9652_adc_devices[num_devs],
	    buf, RME_SAMPLEWIDTH* RME9652_CH_PER_NATIVE_DEVICE*
		DACBLKSIZE);


	for (num_devs=0; num_devs < linux_ndacs; num_devs++) 
	  write(rme9652_dac_devices[num_devs],
	    buf, RME_SAMPLEWIDTH* RME9652_CH_PER_NATIVE_DEVICE*
		DACBLKSIZE);

	if(linux_ndacs)
	  ioctl(rme9652_dac_devices[0],SNDCTL_DSP_SYNC);

	fprintf(stderr,"done\n");
    }

    linux_setchsr(linux_nadcs,linux_ndacs,44100);

    num_of_rme9652_dac = linux_ndacs;
    num_of_rme9652_adc = linux_nadcs;

    if(linux_ndacs)linux_ndacs=1;
    if(linux_nadcs)linux_nadcs=1;

    /* trick RME9652 behaves as one device fromread write pointers */
    return (0);
}

void rme9652_close_audio( void)
{
     int i;
     for (i=0;i<num_of_rme9652_dac;i++)
		 close(rme9652_dac_devices[i]);

     for (i=0;i<num_of_rme9652_adc;i++)
          close(rme9652_adc_devices[i]);
}


/* query audio devices for "available" data size. */
/* not needed because oss_calcspace does the same */
static int rme9652_calcspace(void)
{
    audio_buf_info ainfo;


    /* one for all */

    if (ioctl(linux_dacs[0].fd, SOUND_PCM_GETOSPACE,&ainfo) < 0)
	fprintf(stderr,
   "RME9652: calc ioctl SOUND_PCM_GETOSPACE on output device fd %d failed\n",
    	linux_dacs[0].fd);
    linux_dacs[0].space = ainfo.bytes;

    if (ioctl(linux_adcs[0].fd, SOUND_PCM_GETISPACE,&ainfo) < 0)
	fprintf(stderr, 
 "RME9652: calc ioctl SOUND_PCM_GETISPACE on input device fd %d failed\n", 
    	rme9652_adc_devices[0]);

    linux_adcs[0].space = ainfo.bytes;

    return 1;
}

/* this call resyncs audio output and input which will cause discontinuities
in audio output and/or input. */ 

static void rme9652_doresync( void)
{	
  if(linux_ndacs)
	 ioctl(rme9652_dac_devices[0],SNDCTL_DSP_SYNC);
}

static int mycount =0;

int rme9652_send_dacs(void)
{
    float *fp;
    long fill;
    int i, j, dev;
	 /* the maximum number of samples we should have in the ADC buffer */
    t_rme_sample buf[RME9652_CH_PER_NATIVE_DEVICE*DACBLKSIZE], *sp;

    double timeref, timenow;

    mycount++;

    if (!linux_nadcs && !linux_ndacs) return (0);

    rme9652_calcspace();
    	 
	/* do output */
	
   timeref = sys_getrealtime();

    if(linux_ndacs){

    	if (linux_dacs[0].dropcount)
    	    linux_dacs[0].dropcount--;
    	else{
	    /* fprintf(stderr,"output %d\n", linux_outchannels);*/

	    for(j=0;j<linux_outchannels;j++){

	      t_rme_sample *a,*b,*c,*d;
	      float *fp1,*fp2,*fp3,*fp4;

	      fp1 = sys_soundout + j*DACBLKSIZE-4;
	      fp2 = fp1 + 1;
	      fp3 = fp1 + 2;
	      fp4 = fp1 + 3;
	      a = buf-4;
	      b=a+1;
	      c=a+2;
	      d=a+3; 

	      for (i = DACBLKSIZE>>2;i--;)
		 { 
		    float s1 =  *(fp1+=4) * INT32_MAX;
		    float s2 =  *(fp2+=4) * INT32_MAX;
		    float s3 =  *(fp3+=4) * INT32_MAX;
		    float s4 =  *(fp4+=4) * INT32_MAX;

		    *(a+=4) = CLIP32(s1);
		    *(b+=4) = CLIP32(s2);
		    *(c+=4) = CLIP32(s3);
		    *(d+=4) = CLIP32(s4);
    	    	} 

		linux_dacs_write(rme9652_dac_devices[j],buf,RME_BYTESPERCHAN);
	    }
    	}

	  if ((timenow = sys_getrealtime()) - timeref > 0.02)
		 sys_log_error(ERR_DACSLEPT);
	  timeref = timenow;
    }

    memset(sys_soundout, 0,
    	linux_outchannels * (sizeof(float) * DACBLKSIZE));

	 /* do input */

    if(linux_nadcs) {

    	for(j=0;j<linux_inchannels;j++){

    	    linux_adcs_read(rme9652_adc_devices[j], buf, RME_BYTESPERCHAN);
		  
	    if ((timenow = sys_getrealtime()) - timeref > 0.02)
		 sys_log_error(ERR_ADCSLEPT);
	    timeref = timenow;
	    {
		t_rme_sample *a,*b,*c,*d;
		float *fp1,*fp2,*fp3,*fp4;

		fp1 = sys_soundin + j*DACBLKSIZE-4;
		fp2 = fp1 + 1;
		fp3 = fp1 + 2;
		fp4 = fp1 + 3;
		a = buf-4;
		b=a+1;
		c=a+2;
		d=a+3; 

		for (i = (DACBLKSIZE>>2);i--;)
		{ 
		    *(fp1+=4) = *(a+=4) * (float)(1./INT32_MAX);
		    *(fp2+=4) = *(b+=4) * (float)(1./INT32_MAX);
		    *(fp3+=4) = *(c+=4) * (float)(1./INT32_MAX);
		    *(fp4+=4) = *(d+=4) * (float)(1./INT32_MAX);
		}
	    }  
    	}	
    }
    /*	fprintf(stderr,"ready \n");*/

    return (1);
}

#endif /* HAVE_RME */
