#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: constant1d.c,v 1.7 2000/01/10 03:54:15 knepley Exp $";
#endif

/*
   Defines piecewise constant function space on a two dimensional 
   grid. Suitable for finite element type discretization of a PDE.
*/

#include "src/grid/discretization/discimpl.h"  /*I "discretization.h" I*/
#include "src/mesh/impls/triangular/triimpl.h" /* Just for MAX_CORNERS */

/* For precomputed integrals, the table is structured as follows:

     precompInt[op,i,j] = \int_{SE} <op \phi^i(\xi,\eta), \phi^j(\xi,\eta)> |J^{-1}|

   where we recall that |J| is a constant for linear affine maps,
   and the map of any node to the standard element is linear.
   The numbering of the nodes in the standard element is

          (1)----1----(2)
*/

#undef  __FUNCT__
#define __FUNCT__ "DiscDestroy_Triangular_1D_Constant"
static int DiscDestroy_Triangular_1D_Constant(Discretization disc) {
  PetscFunctionBegin;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscView_Triangular_1D_Constant_File"
static int DiscView_Triangular_1D_Constant_File(Discretization disc, PetscViewer viewer) {
  PetscFunctionBegin;
  PetscViewerASCIIPrintf(viewer, "Constant discretization\n");
  PetscViewerASCIIPrintf(viewer, "    %d shape functions per component\n", disc->funcs);
  PetscViewerASCIIPrintf(viewer, "    %d registered operators\n", disc->numOps);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscView_Triangular_1D_Constant"
static int DiscView_Triangular_1D_Constant(Discretization disc, PetscViewer viewer) {
  PetscTruth isascii;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_ASCII, &isascii);                            CHKERRQ(ierr);
  if (isascii == PETSC_TRUE) {
    ierr = DiscView_Triangular_1D_Constant_File(disc, viewer);                                            CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateFunctionGalerkin_Triangular_1D_Constant"
static int DiscEvaluateFunctionGalerkin_Triangular_1D_Constant(Discretization disc, Mesh mesh, PointFunction f, PetscScalar alpha,
                                                               int elem, PetscScalar *array, void *ctx)
{
  int          dim            = disc->dim;
  int          comp           = disc->comp;           /* The number of components in this field */
  int          funcs          = disc->funcs;          /* The number of shape functions per component */
  PetscScalar *funcVal        = disc->funcVal;        /* Function value at a quadrature point */
  int          numQuadPoints  = disc->numQuadPoints;  /* Number of points used for Gaussian quadrature */
  double      *quadPoints     = disc->quadPoints;     /* Points in the standard element for Gaussian quadrature */
  double      *quadWeights    = disc->quadWeights;    /* Weights in the standard element for Gaussian quadrature */
  double      *quadShapeFuncs = disc->quadShapeFuncs; /* Shape function evaluated at quadrature points */
  double       jac;                                   /* |J| for map to standard element */
  double       x;                                     /* The integration point */
  double       x11, x21;
  int          rank, node0, node1;
  int          i, j, k, p;
#ifdef PETSC_USE_BOPT_g
  Partition    part;
  int          numOverlapElements;
  PetscTruth   opt;
#endif
  int          ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);

  /* For dummy collective calls */
  if (array == PETSC_NULL) {
    for(p = 0; p < numQuadPoints; p++) {
      ierr = (*f)(0, 0, PETSC_NULL, PETSC_NULL, PETSC_NULL, PETSC_NULL, ctx);                             CHKERRQ(ierr);
    }
    PetscFunctionReturn(0);
  }

#ifdef PETSC_USE_BOPT_g
  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  ierr = PartitionGetNumOverlapElements(part, &numOverlapElements);                                       CHKERRQ(ierr);
  if ((elem < 0) || (elem >= numOverlapElements)) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE, "Invalid element");
#endif
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements */
  ierr = MeshGetNodeFromElement(mesh, elem, 0, &node0);                                                   CHKERRQ(ierr);
  ierr = MeshGetNodeFromElement(mesh, elem, 1, &node1);                                                   CHKERRQ(ierr);
  ierr = MeshGetNodeCoords(mesh, node0, &x11, PETSC_NULL, PETSC_NULL);                                    CHKERRQ(ierr);
  ierr = MeshGetNodeCoords(mesh, node1, &x,   PETSC_NULL, PETSC_NULL);                                    CHKERRQ(ierr);
  x21  = MeshPeriodicDiffX(mesh, x - x11);
  jac  = PetscAbsReal(x21);
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g\n", rank, elem, x21);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g jac: %g\n", rank, elem, x21, jac);
  }
#endif

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    x    = MeshPeriodicX(mesh, x21*quadPoints[p*dim] + x11);
    ierr = (*f)(1, comp, &x, PETSC_NULL, PETSC_NULL, funcVal, ctx);                                       CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p:%d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++) PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
  }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        array[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  array[%d]: %g\n", rank, k, PetscRealPart(array[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops(1 + (1 + 5*funcs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateOperatorGalerkin_Triangular_1D_Constant"
static int DiscEvaluateOperatorGalerkin_Triangular_1D_Constant(Discretization disc, Mesh mesh, int elemSize,
                                                               int rowStart, int colStart, int op, PetscScalar alpha,
                                                               int elem, PetscScalar *field, PetscScalar *mat, void *ctx)
{
  int              dim        = disc->dim;
  Operator         oper       = disc->operators[op]; /* The operator to discretize */
  Discretization   test       = oper->test;          /* The space of test functions */
  OperatorFunction opFunc     = oper->opFunc;        /* Integrals of operators which depend on J */
  PetscScalar     *precompInt = oper->precompInt;    /* Precomputed integrals of the operator on shape functions */
  int              rowSize    = test->size;          /* The number of shape functions per element */
  int              colSize    = disc->size;          /* The number of test  functions per element */
  double           x21;                              /* Coordinates of the element, with point 1 at the origin */
  double           jac;                              /* |J| for map to standard element */
  double           coords[MAX_CORNERS*2];            /* Coordinates of the element */
  double           x;
  int              rank, node0, node1;
  int              i, j, f;
  int              ierr; 

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
  /* Check for valid operator */
  if ((op < 0) || (op >= disc->numOps) || (!disc->operators[op])) SETERRQ(PETSC_ERR_ARG_WRONG, "Invalid operator");
#endif

  if (precompInt != PETSC_NULL) {
    /* Calculate the determinant of the inverse Jacobian of the map to the standard element
       which must be a constant for linear elements - 1/|x_{21} y_{31} - x_{31} y_{21}| */
    ierr = MeshGetNodeFromElement(mesh, elem, 0, &node0);                                                 CHKERRQ(ierr);
    ierr = MeshGetNodeFromElement(mesh, elem, 1, &node1);                                                 CHKERRQ(ierr);
    ierr = MeshGetNodeCoords(mesh, node0, &coords[0*dim+0], PETSC_NULL, PETSC_NULL);                      CHKERRQ(ierr);
    ierr = MeshGetNodeCoords(mesh, node1, &coords[1*dim+0], PETSC_NULL, PETSC_NULL);                      CHKERRQ(ierr);
    x21  = MeshPeriodicDiffX(mesh, coords[1*dim+0] - coords[0*dim+0]);
    jac  = PetscAbsReal(x21);
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]x21: %g jac: %g\n", rank, x21, jac);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }

    /* Calculate element matrix entries which are all precomputed */
    for(i = 0; i < rowSize; i++) {
      for(j = 0; j < colSize; j++) {
        mat[(i+rowStart)*elemSize + j+colStart] += alpha*precompInt[i*colSize + j]*jac;
      }
    }
    PetscLogFlops(1 + 2*rowSize*colSize);
  } else {
    if (opFunc == PETSC_NULL) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Invalid function");
    ierr = MeshGetNodeFromElement(mesh, elem, 0, &node0);                                                 CHKERRQ(ierr);
    ierr = MeshGetNodeCoords(mesh, node0, &coords[0*dim+0], PETSC_NULL, PETSC_NULL);                      CHKERRQ(ierr);
    for(f = 1; f < PetscMax(disc->funcs, test->funcs); f++) {
      ierr = MeshGetNodeFromElement(mesh, elem, f, &node1);                                               CHKERRQ(ierr);
      ierr = MeshGetNodeCoords(mesh, node1, &x, PETSC_NULL, PETSC_NULL);                                  CHKERRQ(ierr);
      coords[f*dim+0] = MeshPeriodicRelativeX(mesh, x, coords[0*dim+0]);
    }

    ierr = (*opFunc)(disc, test, rowSize, colSize, rowStart, colStart, elemSize, coords, alpha, field, mat, ctx);
    CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateNonlinearOperatorGalerkin_Triangular_1D_Constant"
static int DiscEvaluateNonlinearOperatorGalerkin_Triangular_1D_Constant(Discretization disc, Mesh mesh, NonlinearOperator f,
                                                                        PetscScalar alpha, int elem, int numArgs, PetscScalar **field,
                                                                        PetscScalar *vec, void *ctx)
{
  int              dim        = disc->dim;
  int              comp       = disc->comp;      /* The number of components in this field */
  int              funcs      = disc->funcs;     /* The number of shape functions per component */
  PetscScalar     *funcVal    = disc->funcVal;   /* Function value at a quadrature point */
  PetscScalar    **fieldVal   = disc->fieldVal;  /* Field value and derivatives at a quadrature point */
  double           jac;                          /* |J| for map to standard element */
  double           invjac;                       /* |J^{-1}| for map from standard element */
  int              numQuadPoints;                /* Number of points used for Gaussian quadrature */
  double          *quadPoints;                   /* Points in the standard element for Gaussian quadrature */
  double          *quadWeights;                  /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs;               /* Shape function evaluated at quadrature points */
  double          *quadShapeFuncDers;            /* Shape function derivatives evaluated at quadrature points */
  double           x;                            /* The integration point */
  double           dxix;                         /* \PartDer{\xi}{x}  */
  PetscScalar      dfxi;                         /* \PartDer{field}{\xi}  */
  double           x11, x21;
  int              rank, node0, node1;
  int              i, j, k, p, func, arg;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  if (numArgs > 2) SETERRQ(PETSC_ERR_SUP, "Only configured to handle two nonlinear arguments");
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);
  numQuadPoints     = disc->numQuadPoints;
  quadPoints        = disc->quadPoints;
  quadWeights       = disc->quadWeights;
  quadShapeFuncs    = disc->quadShapeFuncs;
  quadShapeFuncDers = disc->quadShapeFuncDers;
  
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements */
  ierr = MeshGetNodeFromElement(mesh, elem, 0, &node0);                                                   CHKERRQ(ierr);
  ierr = MeshGetNodeFromElement(mesh, elem, 1, &node1);                                                   CHKERRQ(ierr);
  ierr = MeshGetNodeCoords(mesh, node0, &x11, PETSC_NULL, PETSC_NULL);                                    CHKERRQ(ierr);
  ierr = MeshGetNodeCoords(mesh, node1, &x,   PETSC_NULL, PETSC_NULL);                                    CHKERRQ(ierr);
  x21  = MeshPeriodicDiffX(mesh, x - x11);
  jac  = PetscAbsReal(x21);
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g\n", rank, elem, x21);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g jac: %g\n", rank, elem, x21, jac);
  }
#endif

  /* These are the elements of the inverse matrix */
  invjac = 1/jac;
  dxix   = invjac;

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    x = MeshPeriodicX(mesh, x21*quadPoints[p*dim] + x11);
    /* Can this be simplified? */
    for(arg = 0; arg < numArgs; arg++) {
      for(j = 0; j < comp*3; j++) fieldVal[arg][j] = 0.0;
      for(func = 0; func < funcs; func++) {
        for(j = 0; j < comp; j++) {
          fieldVal[arg][j*(dim+1)]   += field[arg][func*comp+j]*quadShapeFuncs[p*funcs+func];
          fieldVal[arg][j*(dim+1)+1] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*dim+func*dim];
        }
      }
    }

    /* Convert the field derivatives to old coordinates */
    for(arg = 0; arg < numArgs; arg++) {
      for(j = 0; j < comp; j++) {
        dfxi                       = fieldVal[arg][j*(dim+1)+1];
        fieldVal[arg][j*(dim+1)+1] = dfxi*dxix;
      }
    }

    ierr = (*f)(1, comp, &x, PETSC_NULL, PETSC_NULL, numArgs, fieldVal, funcVal, ctx);                    CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p:%d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++)
        PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
    }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        vec[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  vec[%d]: %g\n", rank, k, PetscRealPart(vec[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops(2 + (1 + (4*numArgs + 5)*funcs*comp + numArgs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateALEOperatorGalerkin_Triangular_1D_Constant"
static int DiscEvaluateALEOperatorGalerkin_Triangular_1D_Constant(Discretization disc, Mesh mesh, int elemSize,
                                                                  int rowStart, int colStart, int op, PetscScalar alpha,
                                                                  int elem, PetscScalar *field, PetscScalar *ALEfield, PetscScalar *mat,
                                                                  void *ctx)
{
  int                 dim     = disc->dim;
  Operator            oper    = disc->operators[op]; /* The operator to discretize */
  Discretization      test    = oper->test;          /* The space of test functions */
  ALEOperatorFunction opFunc  = oper->ALEOpFunc;     /* Integrals of operators which depend on J */
  int                 rowSize = test->size;          /* The number of shape functions per element */
  int                 colSize = disc->size;          /* The number of test  functions per element */
  double              coords[MAX_CORNERS*2];         /* Coordinates of the element */
  double              x;
  int                 node0, node1;
  int                 f;
  int                 ierr; 

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  /* Check for valid operator */
  if ((op < 0) || (op >= disc->numOps) || (!disc->operators[op])) SETERRQ(PETSC_ERR_ARG_WRONG, "Invalid operator");
#endif

  if (opFunc == PETSC_NULL) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Invalid function");
  ierr = MeshGetNodeFromElement(mesh, elem, 0, &node0);                                                   CHKERRQ(ierr);
  ierr = MeshGetNodeCoords(mesh, node0, &coords[0*dim+0], PETSC_NULL, PETSC_NULL);                        CHKERRQ(ierr);
  for(f = 1; f < PetscMax(disc->funcs, test->funcs); f++) {
    ierr = MeshGetNodeFromElement(mesh, elem, f, &node1);                                                 CHKERRQ(ierr);
    ierr = MeshGetNodeCoords(mesh, node1, &x, PETSC_NULL, PETSC_NULL);                                    CHKERRQ(ierr);
    coords[f*dim+0] = MeshPeriodicRelativeX(mesh, x, coords[0*dim+0]);
  }

  ierr = (*opFunc)(disc, test, rowSize, colSize, rowStart, colStart, elemSize, coords, alpha, field, ALEfield, mat, ctx);
  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscEvaluateNonlinearALEOperatorGalerkin_Triangular_1D_Constant"
static int DiscEvaluateNonlinearALEOperatorGalerkin_Triangular_1D_Constant(Discretization disc, Mesh mesh, NonlinearOperator f,
                                                                           PetscScalar alpha, int elem, int numArgs, PetscScalar **field,
                                                                           PetscScalar *ALEfield, PetscScalar *vec, void *ctx)
{
  int              dim        = disc->dim;
  int              comp       = disc->comp;      /* The number of components in this field */
  int              funcs      = disc->funcs;     /* The number of shape functions per component */
  PetscScalar     *funcVal    = disc->funcVal;   /* Function value at a quadrature point */
  PetscScalar    **fieldVal   = disc->fieldVal;  /* Field value and derivatives at a quadrature point */
  double           jac;                          /* |J| for map to standard element */
  double           invjac;                       /* |J^{-1}| for map from standard element */
  int              numQuadPoints;                /* Number of points used for Gaussian quadrature */
  double          *quadPoints;                   /* Points in the standard element for Gaussian quadrature */
  double          *quadWeights;                  /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs;               /* Shape function evaluated at quadrature points */
  double          *quadShapeFuncDers;            /* Shape function derivatives evaluated at quadrature points */
  double           x;                            /* The integration point */
  double           dxix;                         /* \PartDer{\xi}{x}  */
  PetscScalar      dfxi;                         /* \PartDer{field}{\xi}  */
  double           x11, x21;
  int              rank, node0, node1;
  int              i, j, k, p, func, arg;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  if (numArgs > 2) SETERRQ(PETSC_ERR_SUP, "Only configured to handle two nonlinear arguments");
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);

  numQuadPoints     = disc->numQuadPoints;
  quadPoints        = disc->quadPoints;
  quadWeights       = disc->quadWeights;
  quadShapeFuncs    = disc->quadShapeFuncs;
  quadShapeFuncDers = disc->quadShapeFuncDers;
  
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements */
  ierr = MeshGetNodeFromElement(mesh, elem, 0, &node0);                                                   CHKERRQ(ierr);
  ierr = MeshGetNodeFromElement(mesh, elem, 1, &node1);                                                   CHKERRQ(ierr);
  ierr = MeshGetNodeCoords(mesh, node0, &x11, PETSC_NULL, PETSC_NULL);                                    CHKERRQ(ierr);
  ierr = MeshGetNodeCoords(mesh, node1, &x,   PETSC_NULL, PETSC_NULL);                                    CHKERRQ(ierr);
  x21  = MeshPeriodicDiffX(mesh, x - x11);
  jac  = PetscAbsReal(x21);
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g\n", rank, elem, x21);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]elem: %d x21: %g jac: %g\n", rank, elem, x21, jac);
  }
#endif

  /* These are the elements of the inverse matrix */
  invjac = 1/jac;
  dxix   = invjac;

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    x = MeshPeriodicX(mesh, x21*quadPoints[p*dim] + x11);
    /* Can this be simplified? */
    for(arg = 0; arg < numArgs; arg++) {
      for(j = 0; j < comp*(dim+1); j++) fieldVal[arg][j] = 0.0;
      for(func = 0; func < funcs; func++)
        for(j = 0; j < comp; j++) {
          fieldVal[arg][j*(dim+1)]   += (field[arg][func*comp+j] - ALEfield[func*comp+j])*quadShapeFuncs[p*funcs+func];
          fieldVal[arg][j*(dim+1)+1] += field[arg][func*comp+j]*quadShapeFuncDers[p*funcs*dim+func*dim];
        }
    }

    /* Convert the field derivatives to old coordinates */
    for(arg = 0; arg < numArgs; arg++) {
      for(j = 0; j < comp; j++) {
        dfxi                       = fieldVal[arg][j*(dim+1)+1];
        fieldVal[arg][j*(dim+1)+1] = dfxi*dxix;
      }
    }

    ierr = (*f)(1, comp, &x, PETSC_NULL, PETSC_NULL, numArgs, fieldVal, funcVal, ctx);                    CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p:%d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++)
        PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
    }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        vec[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  vec[%d]: %g\n", rank, k, PetscRealPart(vec[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops(2 + (1 + (5*numArgs + 5)*funcs*comp + numArgs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Laplacian_Triangular_1D_Constant"
int Laplacian_Triangular_1D_Constant(Discretization disc, Discretization test, int rowSize, int colSize,
                                   int globalRowStart, int globalColStart, int globalSize, double *coords,
                                   PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  double      x21;          /* Coordinates of the element, with point 1 at the origin */
  double      jac, invjac;  /* |J| and |J^{-1}| for map to standard element */
  PetscScalar entry;
  int         comp;         /* Number of components */
  int         i;

  PetscFunctionBegin;
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements - 1/|x_{21}| */
  x21 = coords[1] - coords[0];
  jac = PetscAbsReal(x21);
#ifdef PETSC_USE_BOPT_g
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "x21: %g jac: %g\n", x21, jac);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#endif
  invjac = 1.0/jac;

  comp = rowSize/disc->funcs;
  /* \alpha \PartDer{\phi}{x}^2 |J| = \alpha \PartDer{\xi}{x}^2 |J| = \alpha |J^{-1}|^2 |J| = \alpha |J^{-1}| */
  entry = alpha*invjac;
  for(i = 0; i < comp; i++) {
    /* \phi^1 \phi^1 */
    array[(0*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] = entry;
  }
  PetscLogFlops(4);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Weighted_Laplacian_Triangular_1D_Constant"
int Weighted_Laplacian_Triangular_1D_Constant(Discretization disc, Discretization test, int rowSize, int colSize,
                                            int globalRowStart, int globalColStart, int globalSize, double *coords,
                                            PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  double      x21;          /* Coordinates of the element, with point 1 at the origin */
  double      jac, invjac;  /* |J| and |J^{-1}| for map to standard element */
  PetscScalar entry;
  int         comp;         /* Number of components */
  int         i;

  PetscFunctionBegin;
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element
     which must be a constant for linear elements - 1/|x_{21}| */
  x21 = coords[1] - coords[0];
  jac = PetscAbsReal(x21);
#ifdef PETSC_USE_BOPT_g
  if (jac < 1.0e-14) {
    PetscPrintf(PETSC_COMM_SELF, "x21: %g jac: %g\n", x21, jac);
    SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
  }
#endif
  invjac = 1.0/jac;

  comp = rowSize/disc->funcs;
  /* \alpha \PartDer{\phi}{x}^2 = \alpha \PartDer{\xi}{x}^2 = \alpha |J^{-1}|^2 */
  entry = alpha*invjac*invjac;
  for(i = 0; i < comp; i++) {
    /* \phi^1 \phi^1 */
    array[(0*comp+i+globalRowStart)*globalSize + 0*comp+i+globalColStart] = entry;
  }
  PetscLogFlops(4);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Gradient_Triangular_1D_Constant"
int Gradient_Triangular_1D_Constant(Discretization disc, Discretization test, int rowSize, int colSize,
                                  int globalRowStart, int globalColStart, int globalSize, double *coords,
                                  PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  /* We are using the convention that

       \nabla \matrix{v_1 \cr v_2 \cr \vdots \cr v_n} =
         \matrix{v^{(1)}_1 \cr \vdots \cr v^{(d)}_1 \cr v^{(1)}_2 \cr \vdots \cr v^{(d)}_n}

     and

       \nabla \cdot \matrix{v^{(1)}_1 \cr \vdots \cr v^{(d)}_1 \cr v^{(1)}_2 \cr \vdots \cr v^{(d)}_n} =
         \matrix{v_1 \cr v_2 \cr \vdots \cr v_n}

     where $d$ is the number of space dimensions. This agrees with the convention which allows
     $\Delta \matrix{u_1 \cr u_2} = 0$ to denote a set of scalar equations. This also means that
     the dimension of the test function vector must be divisible by the number of space dimensions */
  int     numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double *quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double *quadShapeFuncs;    /* Shape functions evaluated at quadrature points */
  double *quadTestFuncDers;  /* Test function derivatives evaluated at quadrature points */
  double  dxxi;              /* \PartDer{x}{\xi}  */
  double  dxix;              /* \PartDer{\xi}{x}  */
  double  dphix;             /* \PartDer{\phi_i}{x} \times \PartDer{\phi_j}{x} */
  double  jac;               /* |J| for map to standard element */
  double  invjac;            /* |J^{-1}| for map from standard element */
  int     dim;               /* The problem dimension */
  int     comp;              /* The number of field components */
  int     tcomp;             /* The number of field components for the test field */
  int     funcs;             /* The number of shape functions */
  int     tfuncs;            /* The number of test functions */
  int     i, j, c, tc, f, p;

  PetscFunctionBegin;
  /* Calculate element matrix entries by Gaussian quadrature --
	   Since we integrate by parts here, the test and shape functions are switched */
  dim              = disc->dim;
  comp             = disc->comp;
  tcomp            = test->comp;
  funcs            = disc->funcs;
  tfuncs           = test->funcs;
  numQuadPoints    = disc->numQuadPoints;
  quadWeights      = disc->quadWeights;
  quadShapeFuncs   = disc->quadShapeFuncs;
  quadTestFuncDers = test->quadShapeFuncDers;
  for(p = 0; p < numQuadPoints; p++) {
    /* \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi} */
    dxxi = 0.0;
    for(f = 0; f < tfuncs; f++) {
      dxxi += coords[f*dim]*quadTestFuncDers[p*tfuncs*dim+f*dim];
    }
    jac = PetscAbsReal(dxxi);
#ifdef PETSC_USE_BOPT_g
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %g x2: %g\n", p, coords[0], coords[1]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#endif
    /* These are the elements of the inverse matrix */
    invjac =  1.0/jac;
    dxix   =  invjac;

    /* The rows are test functions */
    for(i = 0; i < tfuncs; i++) {
      /* We divide by the space dimension */
      for(tc = 0; tc < tcomp/dim; tc++) {
        /* The columns are shape functions */
        for(j = 0; j < funcs; j++) {
          for(c = 0; c < comp; c++) {
            dphix = quadTestFuncDers[p*tfuncs*dim+i*dim]*dxix;
            array[(i*tcomp+tc*dim+globalRowStart)*globalSize + j*comp+c+globalColStart] +=
              -alpha*dphix*quadShapeFuncs[p*funcs+j]*jac*quadWeights[p];
          }
        }
      }
    }
  }
  PetscLogFlops((2*tfuncs + 1 + 4*tfuncs*tcomp/dim*funcs*comp) * numQuadPoints);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Divergence_Triangular_1D_Constant"
int Divergence_Triangular_1D_Constant(Discretization disc, Discretization test, int rowSize, int colSize,
                                      int globalRowStart, int globalColStart, int globalSize, double *coords,
                                      PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
	/* We are using the convention that

		   \nabla \matrix{v_1 \cr v_2 \cr \vdots \cr v_n} =
			   \matrix{v^{(1)}_1 \cr \vdots \cr v^{(d)}_1 \cr v^{(1)}_2 \cr \vdots \cr v^{(d)}_n}

		 and

		   \nabla \cdot \matrix{v^{(1)}_1 \cr \vdots \cr v^{(d)}_1 \cr v^{(1)}_2 \cr \vdots \cr v^{(d)}_n} =
			   \matrix{v_1 \cr v_2 \cr \vdots \cr v_n}

		 where $d$ is the number of space dimensions. This agrees with the convention which allows
		 $\Delta \matrix{u_1 \cr u_2} = 0$ to denote a set of scalar equations	This also requires that
     the dimension of a vector must be divisible by the space dimension in order to be acted upon by
     the divergence operator */
  int     numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double *quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double *quadTestFuncs;     /* Test  functions evaluated at quadrature points */
  double *quadShapeFuncDers; /* Shape function derivatives evaluated at quadrature points */
  double *quadTestFuncDers;  /* Test function derivatives evaluated at quadrature points */
  double  dxxi;              /* \PartDer{x}{\xi}  */
  double  dxix;              /* \PartDer{\xi}{x}  */
  double  dphix;             /* \PartDer{\phi_i}{x} \times \PartDer{\phi_j}{x} */
  double  jac;               /* |J| for map to standard element */
  double  invjac;            /* |J^{-1}| for map from standard element */
  int     dim;               /* The problem dimension */
  int     comp;              /* The number of field components */
  int     tcomp;             /* The number of field components for the test field */
  int     funcs;             /* The number of shape functions */
  int     tfuncs;            /* The number of test functions */
  int     i, j, c, tc, f, p;

  PetscFunctionBegin;
  /* Calculate element matrix entries by Gaussian quadrature */
  dim              = disc->dim;
  comp              = disc->comp;
  tcomp             = test->comp;
  funcs             = disc->funcs;
  tfuncs            = test->funcs;
  numQuadPoints     = disc->numQuadPoints;
  quadWeights       = disc->quadWeights;
  quadTestFuncs     = test->quadShapeFuncs;
  quadShapeFuncDers = disc->quadShapeFuncDers;
  quadTestFuncDers  = test->quadShapeFuncDers;
  for(p = 0; p < numQuadPoints; p++) {
    /* \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi} */
    dxxi = 0.0;
    if (tfuncs >= funcs) {
      for(f = 0; f < tfuncs; f++) {
        dxxi += coords[f*dim]*quadTestFuncDers[p*tfuncs*dim+f*dim];
      }
    } else {
      for(f = 0; f < funcs; f++) {
        dxxi += coords[f*dim]*quadShapeFuncDers[p*funcs*dim+f*dim];
      }
    }
    jac = PetscAbsReal(dxxi);
#ifdef PETSC_USE_BOPT_g
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %g x2: %g\n", p, coords[0], coords[1]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#endif
    /* These are the elements of the inverse matrix */
    invjac = 1.0/jac;
    dxix   = invjac;

    /* The rows are test functions */
    for(i = 0; i < tfuncs; i++) {
      for(tc = 0; tc < tcomp; tc++) {
        /* The columns are shape functions */
        for(j = 0; j < funcs; j++) {
          dphix = quadShapeFuncDers[p*funcs*dim+j*dim]*dxix;
          /* We divide by the number of space dimensions */
          for(c = 0; c < comp/dim; c++) {
            array[(i*tcomp+tc+globalRowStart)*globalSize + j*comp+c*dim+globalColStart] +=
              alpha*dphix*quadTestFuncs[p*tfuncs+i]*jac*quadWeights[p];
          }
        }
      }
    }
  }
  PetscLogFlops((2*funcs + 1 + 4*tfuncs*tcomp*funcs*comp/dim) * numQuadPoints);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscInterpolateField_Triangular_1D_Constant"
int DiscInterpolateField_Triangular_1D_Constant(Discretization disc, Mesh oldMesh, int elem, double x, double y, double z,
                                              PetscScalar *oldFieldVal, PetscScalar *newFieldVal, InterpolationType type)
{
  int comp = disc->comp;
  int rank;
  int neighbor, corner, c;
  int ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);
  /* No scheme in place for boundary elements */
  for(corner = 0; corner < 2; corner++) {
    ierr = MeshGetElementNeighbor(oldMesh, elem, corner, &neighbor);                                      CHKERRQ(ierr);
    if (neighbor < 0) {
      type = INTERPOLATION_LOCAL;
      break;
    }
  }

  switch (type) {
  case INTERPOLATION_LOCAL:
    for(c = 0 ; c < comp; c++) {
      newFieldVal[c] = oldFieldVal[0*comp+c];
    }
    PetscLogFlops(4+3*comp);
    break;
  default:
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Unknown interpolation type %d", type);
  }
  
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscInterpolateElementVec_Triangular_1D_Constant"
int DiscInterpolateElementVec_Triangular_1D_Constant(Discretization disc, ElementVec vec, Discretization newDisc, ElementVec newVec) {
  int          comp  = disc->comp;
  PetscScalar *array, *newArray;
  int          f, c;
  int          ierr;

  PetscFunctionBegin;
  ierr = ElementVecGetArray(vec,    &array);                                                              CHKERRQ(ierr);
  ierr = ElementVecGetArray(newVec, &newArray);                                                           CHKERRQ(ierr);
  for(f = 0; f < newDisc->funcs; f++) {
    for(c = 0; c < comp; c++) {
      newArray[f*comp+c] = array[c];
    }
  }
  ierr = ElementVecRestoreArray(vec,    &array);                                                          CHKERRQ(ierr);
  ierr = ElementVecRestoreArray(newVec, &newArray);                                                       CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscSetupQuadrature_Triangular_1D_Constant"
/*
  DiscSetupQuadrature_Triangular_1D_Constant - Setup Gaussian quadrature with a 7 point integration rule

  Input Parameter:
. disc - The Discretization
*/
int DiscSetupQuadrature_Triangular_1D_Constant(Discretization disc) {
  int dim   = disc->dim;
  int funcs = disc->funcs;
  int p;
  int ierr;

  PetscFunctionBegin;
  disc->numQuadPoints = 7;
  ierr = PetscMalloc(disc->numQuadPoints*dim       * sizeof(double), &disc->quadPoints);                  CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints           * sizeof(double), &disc->quadWeights);                 CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs     * sizeof(double), &disc->quadShapeFuncs);              CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs*dim * sizeof(double), &disc->quadShapeFuncDers);           CHKERRQ(ierr);
  PetscLogObjectMemory(disc, (disc->numQuadPoints*(funcs*(dim+1) + dim+1)) * sizeof(double));
  disc->quadPoints[0]  = 0.0254460438286207377369052;
  disc->quadWeights[0] = 0.0647424830844348466353057;
  disc->quadPoints[1]  = 0.1292344072003027800680676;
  disc->quadWeights[1] = 0.1398526957446383339507339;
  disc->quadPoints[2]  = 0.29707742431130141654669679;
  disc->quadWeights[2] = 0.1909150252525594724751849;
  disc->quadPoints[3]  = 0.5000000000000000000000000;
  disc->quadWeights[3] = 0.2089795918367346938775510;
  disc->quadPoints[4]  = 0.70292257568869858345330321;
  disc->quadWeights[4] = disc->quadWeights[2];
  disc->quadPoints[5]  = 0.8707655927996972199319324;
  disc->quadWeights[5] = disc->quadWeights[1];
  disc->quadPoints[6]  = 0.9745539561713792622630948;
  disc->quadWeights[6] = disc->quadWeights[0];
  for(p = 0; p < disc->numQuadPoints; p++) {
    /* \phi^0: 1 */
    disc->quadShapeFuncs[p*funcs]              = 1.0;
    disc->quadShapeFuncDers[p*funcs*dim+0*dim] = 0.0;
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "DiscSetupOperators_Triangular_1D_Constant"
/*
  DiscSetupOperators_Triangular_1D_Constant - Setup the default operators

  Input Parameter:
. disc - The Discretization
*/
int DiscSetupOperators_Triangular_1D_Constant(Discretization disc) {
  int          comp = disc->comp;
  int          size = disc->size;
  PetscScalar *precompInt;
  int          newOp;
  int          c, i, j;
  int          ierr;

  PetscFunctionBegin;
  /* The Identity operator I -- the matrix is symmetric */
  ierr = PetscMalloc(size*size * sizeof(PetscScalar), &precompInt);                                       CHKERRQ(ierr);
  PetscLogObjectMemory(disc, size*size * sizeof(PetscScalar));
  ierr = PetscMemzero(precompInt, size*size * sizeof(PetscScalar));                                       CHKERRQ(ierr);
  for(c = 0; c < comp; c++) {
    precompInt[(0*comp+c)*size + 0*comp+c] = 1.0;
  }
  for(i = 0; i < size; i++) {
    for(j = 0; j < i; j++) {
      precompInt[i*size + j] = precompInt[j*size + i];
    }
  }
  ierr = DiscretizationRegisterPrecomputedOperator(disc, precompInt, &newOp);                             CHKERRQ(ierr);
  if (newOp != IDENTITY) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", IDENTITY);
  /* The Laplacian operator \Delta -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, Laplacian_Triangular_1D_Constant, &newOp);                  CHKERRQ(ierr);
  if (newOp != LAPLACIAN) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", LAPLACIAN);
  /* The Gradient operator \nabla -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, Gradient_Triangular_1D_Constant, &newOp);                   CHKERRQ(ierr);
  if (newOp != GRADIENT) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", GRADIENT);
  /* The Divergence operator \nabla\cdot -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, Divergence_Triangular_1D_Constant, &newOp);                 CHKERRQ(ierr);
  if (newOp != DIVERGENCE) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", DIVERGENCE);
  /* The weighted Laplacian operator -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, Weighted_Laplacian_Triangular_1D_Constant, &newOp);         CHKERRQ(ierr);
  if (newOp != WEIGHTED_LAP) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", WEIGHTED_LAP);
  PetscFunctionReturn(0);
}

static struct _DiscretizationOps DOps = {PETSC_NULL/* DiscretizationSetup */,
                                         DiscSetupOperators_Triangular_1D_Constant,
                                         PETSC_NULL/* DiscretizationSetFromOptions */,
                                         DiscView_Triangular_1D_Constant,
                                         DiscDestroy_Triangular_1D_Constant,
                                         DiscEvaluateFunctionGalerkin_Triangular_1D_Constant,
                                         DiscEvaluateOperatorGalerkin_Triangular_1D_Constant,
                                         DiscEvaluateALEOperatorGalerkin_Triangular_1D_Constant,
                                         DiscEvaluateNonlinearOperatorGalerkin_Triangular_1D_Constant,
                                         DiscEvaluateNonlinearALEOperatorGalerkin_Triangular_1D_Constant,
                                         DiscInterpolateField_Triangular_1D_Constant,
                                         DiscInterpolateElementVec_Triangular_1D_Constant};

EXTERN_C_BEGIN
#undef  __FUNCT__
#define __FUNCT__ "DiscCreate_Triangular_1D_Constant"
int DiscCreate_Triangular_1D_Constant(Discretization disc) {
  int arg;
  int ierr;

  PetscFunctionBegin;
  if (disc->comp <= 0) {
    SETERRQ(PETSC_ERR_ARG_WRONG, "Discretization must have at least 1 component. Call DiscretizationSetNumComponents() to set this.");
  }
  ierr = PetscMemcpy(disc->ops, &DOps, sizeof(struct _DiscretizationOps));                                CHKERRQ(ierr);
  disc->dim   = 1;
  disc->funcs = 1;
  disc->size  = disc->funcs*disc->comp;

  ierr = DiscretizationSetupDefaultOperators(disc);                                                       CHKERRQ(ierr);
  ierr = DiscSetupQuadrature_Triangular_1D_Constant(disc);                                                  CHKERRQ(ierr);

  /* Storage */
  ierr = PetscMalloc(disc->comp * sizeof(PetscScalar),   &disc->funcVal);                                 CHKERRQ(ierr);
  ierr = PetscMalloc(2          * sizeof(PetscScalar *), &disc->fieldVal);                                CHKERRQ(ierr);
  for(arg = 0; arg < 2; arg++) {
    ierr = PetscMalloc(disc->comp*(disc->dim+1) * sizeof(PetscScalar), &disc->fieldVal[arg]);             CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}
EXTERN_C_END
