#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: partreg.c,v 1.2 2000/01/10 03:54:25 knepley Exp $";
#endif

#include "src/mesh/meshimpl.h"  /*I "mesh.h"  I*/

PetscFList PartitionList                       = 0;
int        PartitionRegisterAllCalled          = 0;
PetscFList PartitionSerializeList              = 0;
int        PartitionSerializeRegisterAllCalled = 0;

#undef __FUNCT__  
#define __FUNCT__ "PartitionSetType"
/*@C
  PartitionSetType - Sets the creation method for the partition.

  Collective on Partition

  Input Parameters:
+ part   - The Partition context
- method - A known method

  Options Database Command:
. -part_type <method> - Sets the method; use -help for a list
                        of available methods (for instance, tri2d)

  Notes:
  See "petsc/include/mesh.h" for available methods (for instance)
. PARTITION_SER_TRIANGULAR_2D_BINARY - Triangular 2D to binary file

  Normally, it is best to use the PartitionSetFromOptions() command and
  then set the Partition type from the options database rather than by using
  this routine.  Using the options database provides the user with
  maximum flexibility in evaluating the many different solvers.
  The PartitionSetType() routine is provided for those situations
  where it is necessary to set the application ordering independently of the
  command line or options database.  This might be the case, for example,
  when the choice of solver changes during the execution of the
  program, and the user's application is taking responsibility for
  choosing the appropriate method.  In other words, this routine is
  not for beginners.

  Level: intermediate

.keywords: partition, set, type
.seealso PartitionSetSerializeType()
@*/
int PartitionSetType(Partition part, PartitionType method)
{
  int      (*r)(Partition);
  PetscTruth match;
  int        ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(part, PARTITION_COOKIE);
  ierr = PetscTypeCompare((PetscObject) part, method, &match);                                            CHKERRQ(ierr);
  if (match == PETSC_TRUE) PetscFunctionReturn(0);

  /* Get the function pointers for the method requested */
  if (!PartitionRegisterAllCalled) {
    ierr = PartitionRegisterAll(PETSC_NULL);                                                              CHKERRQ(ierr);
  }
  ierr = PetscFListFind(part->comm, PartitionList, method, (void (**)(void)) &r);                          CHKERRQ(ierr);
  if (!r) {
    SETERRQ1(PETSC_ERR_ARG_OUTOFRANGE, "Unknown method: %s", method);
  }

  if (part->ops->destroy) {
    ierr = (*part->ops->destroy)(part);                                                                   CHKERRQ(ierr);
  }
  ierr = (*r)(part);                                                                                      CHKERRQ(ierr);

  ierr = PetscObjectChangeTypeName((PetscObject) part, method);                                           CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "PartitionGetType"
/*@C
  PartitionGetType - Gets the Partition method type and name (as a string).

  Not collective

  Input Parameter:
. part - The partition

  Output Parameter:
. type - The name of Partition method

  Level: intermediate

.keywords: partition, get, type
.seealso PartitionSetType()
@*/
int PartitionGetType(Partition part, PartitionType *type)
{
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(part, PARTITION_COOKIE);
  PetscValidPointer(type);
  if (!PartitionRegisterAllCalled) {
    ierr = PartitionRegisterAll(PETSC_NULL);                                                             CHKERRQ(ierr);
  }
  *type = part->type_name;
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "PartitionSetSerializeType"
/*@C
  PartitionSetSerializeType - Sets the serialization method for the part.

  Collective on Partition

  Input Parameters:
+ part   - The Partition context
- method - A known method

  Options Database Command:
. -part_serialize_type <method> - Sets the method; use -help for a list
                                  of available methods (for instance, tri2d_binary)

  Notes:
  See "petsc/include/mesh.h" for available methods (for instance)
. PARTITION_SER_TRIANGULAR_2D_BINARY - Triangular 2D mesh partition to binary file

  Normally, it is best to use the PartitionSetFromOptions() command and
  then set the Partition type from the options database rather than by using
  this routine.  Using the options database provides the user with
  maximum flexibility in evaluating the many different solvers.
  The PartitionSetSerializeType() routine is provided for those situations
  where it is necessary to set the application ordering independently of the
  command line or options database.  This might be the case, for example,
  when the choice of solver changes during the execution of the
  program, and the user's application is taking responsibility for
  choosing the appropriate method.  In other words, this routine is
  not for beginners.

  Level: intermediate

.keywords: partition, set, type, serialization
.seealso PartitionSetType()
@*/
int PartitionSetSerializeType(Partition part, PartitionSerializeType method)
{
  int      (*r)(MPI_Comm, Partition *, PetscViewer, PetscTruth);
  PetscTruth match;
  int        ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(part, PARTITION_COOKIE);
  ierr = PetscSerializeCompare((PetscObject) part, method, &match);                                       CHKERRQ(ierr);
  if (match == PETSC_TRUE) PetscFunctionReturn(0);

  /* Get the function pointers for the method requested but do not call */
  if (!PartitionSerializeRegisterAllCalled) {
    ierr = PartitionSerializeRegisterAll(PETSC_NULL);                                                     CHKERRQ(ierr);
  }
  ierr = PetscFListFind(part->comm, PartitionSerializeList, method, (void (**)(void)) &r);                CHKERRQ(ierr);
  if (!r) {
    SETERRQ1(PETSC_ERR_ARG_OUTOFRANGE, "Unknown method: %s", method);
  }

  ierr = PetscObjectChangeSerializeName((PetscObject) part, method);                                      CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*-------------------------------------------------------------------------------------------------------------------*/
/*MC
  PartitionRegister - Adds a creation method to the partition package.

  Synopsis:

  PartitionRegister(char *name, char *path, char *func_name, int (*create_func)(Partition))

  Not Collective

  Input Parameters:
+ name        - The name of a new user-defined creation routine
. path        - The path (either absolute or relative) of the library containing this routine
. func_name   - The name of routine to create method context
- create_func - The creation routine itself

  Notes:
  PartitionRegister() may be called multiple times to add several user-defined meshes.

  If dynamic libraries are used, then the fourth input argument (create_func) is ignored.

  Sample usage:
.vb
  PartitionRegister("my_part", /home/username/my_lib/lib/libO/solaris/mylib.a, "MyFunc", MyFunc);
.ve

  Then, your mesh type can be chosen with the procedural interface via
$     PartitionSetType(vec, "my_part")
  or at runtime via the option
$     -part_type my_part

  Level: advanced

  $PETSC_ARCH and $BOPT occuring in pathname will be replaced with appropriate values.

.keywords: partition, register
.seealso: PartitionRegisterAll(), PartitionRegisterDestroy()
M*/
#undef __FUNCT__  
#define __FUNCT__ "PartitionRegister_Private"
int PartitionRegister_Private(const char sname[], const char path[], const char name[], int (*function)(Partition))
{
  char fullname[256];
  int  ierr;

  PetscFunctionBegin;
  ierr = PetscStrcpy(fullname, path);                                                                     CHKERRQ(ierr);
  ierr = PetscStrcat(fullname, ":");                                                                      CHKERRQ(ierr);
  ierr = PetscStrcat(fullname, name);                                                                     CHKERRQ(ierr);
  ierr = PetscFListAdd(&PartitionList, sname, fullname, (void (*)(void)) function);                       CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*MC
  PartitionSerializeRegister - Adds a serialization method to the partition package.

  Synopsis:

  PartitionSerializeRegister(char *serialize_name, char *path, char *serialize_func_name,
                             int (*serialize_func)(MPI_Comm, Partition *, PetscViewer, PetscTruth))

  Not Collective

  Input Parameters:
+ serialize_name      - The name of a new user-defined serialization routine
. path                - The path (either absolute or relative) of the library containing this routine
. serialize_func_name - The name of routine to create method context
- serialize_func      - The serialization routine itself

  Notes:
  PartitionSerializeRegister() may be called multiple times to add several user-defined partitioners.

  If dynamic libraries are used, then the fourth input argument (routine_create) is ignored.

  Sample usage:
.vb
  PartitionSerializeRegister("my_store", /home/username/my_lib/lib/libO/solaris/mylib.a, "MyStoreFunc", MyStoreFunc);
.ve

  Then, your serialization can be chosen with the procedural interface via
$     PartitionSetSerializeType(vec, "my_store")
  or at runtime via the option
$     -part_serialize_type my_store

  Level: advanced

  $PETSC_ARCH and $BOPT occuring in pathname will be replaced with appropriate values.

.keywords: partition, register
.seealso: PartitionSerializeRegisterAll(), PartitionSerializeRegisterDestroy()
M*/
#undef __FUNCT__  
#define __FUNCT__ "PartitionSerializeRegister_Private"
int PartitionSerializeRegister_Private(const char sname[], const char path[], const char name[],
                                  int (*function)(Mesh, Partition *, PetscViewer, PetscTruth))
{
  char fullname[256];
  int  ierr;

  PetscFunctionBegin;
  ierr = PetscStrcpy(fullname, path);                                                                     CHKERRQ(ierr);
  ierr = PetscStrcat(fullname, ":");                                                                      CHKERRQ(ierr);
  ierr = PetscStrcat(fullname, name);                                                                     CHKERRQ(ierr);
  ierr = PetscFListAdd(&PartitionSerializeList, sname, fullname, (void (*)(void)) function);              CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*-------------------------------------------------------------------------------------------------------------------*/
#undef __FUNCT__  
#define __FUNCT__ "PartitionRegisterDestroy"
/*@C
  PartitionRegisterDestroy - Frees the list of creation routines for
  partitions that were registered by FListAdd().

  Not collective

  Level: advanced

.keywords: partition, register, destroy
.seealso: PartitionRegisterAll(), PartitionSerializeRegisterDestroy()
@*/
int PartitionRegisterDestroy()
{
  int ierr;

  PetscFunctionBegin;
  if (PartitionList) {
    ierr = PetscFListDestroy(&PartitionList);                                                             CHKERRQ(ierr);
    PartitionList = PETSC_NULL;
  }
  PartitionRegisterAllCalled = 0;
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "PartitionSerializeRegisterDestroy"
/*@C
  PartitionSerializeRegisterDestroy - Frees the list of serialization routines for
  partitions that were registered by FListAdd().

  Not collective

  Level: advanced

.keywords: partition, serialization, register, destroy
.seealso: PartitionSerializeRegisterAll(), PartitionRegisterDestroy()
@*/
int PartitionSerializeRegisterDestroy()
{
  int ierr;

  PetscFunctionBegin;
  if (PartitionSerializeList) {
    ierr = PetscFListDestroy(&PartitionSerializeList);                                                    CHKERRQ(ierr);
    PartitionSerializeList = PETSC_NULL;
  }
  PartitionSerializeRegisterAllCalled = 0;
  PetscFunctionReturn(0);
}
