# -*- coding: UTF-8 -*-

# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

#---first test
import os, time, types

if __name__ == '__main__':
    import sys
    sys.path.extend(['lib','../core','../core/lib'])
    #test environment
    import gettext, sys
    gettext.install("test")
    sys.path.insert(0,os.path.dirname(os.getcwd()))
    
#---begin
import wx
from core import ct

#core.lib
import formField 
from core.message import send, ProgressReceiver
from core.pil import exif

#gui-dependent
import graphics, vlistTag, images, paint
from wildcard import wildcard_list, _wildcard_extension
from tag import Browser, ContentMixin
from pyWx.wxGlade   import dialogs
from imageInspector import ALL, SELECT

VLIST_ICON_SIZE = (48,48)

class BrowseMixin:
    def show_dir_dialog(self,defaultPath,message=_('Choose a folder'),
            style=wx.DEFAULT_DIALOG_STYLE): 
        dlg = wx.DirDialog(self,message,
            defaultPath = defaultPath,
            style       = wx.DEFAULT_DIALOG_STYLE)
        if dlg.ShowModal() == wx.ID_OK:
            path = dlg.GetPath()
        else:
            path = None
        dlg.Destroy()
        return path

class _IconDialog:
    if wx.Platform == '__WXGTK__':
        _icon_size  = (32,32)#(48,48)
    else:
        _icon_size  = (32,32)
    def _icon(self,name='information'):
        name    = 'ART_%s'%name.upper()
        #title icon
        bitmap  = graphics.bitmap(name,(16,16))
        _ic = wx.EmptyIcon()
        _ic.CopyFromBitmap(bitmap)
        self.SetIcon(_ic)
        #dialog icon
        bitmap  = graphics.bitmap(name,self._icon_size)
        self.icon.SetBitmap(bitmap)
        self.icon.Show(True)

class ErrorDialog(dialogs.ErrorDialog,_IconDialog):
    def __init__(self,parent,message,ignore=True,**keyw):
        super(ErrorDialog,self).__init__(parent,-1,**keyw)
        self._icon('error')
        self.message.SetLabel(message)
        if not ignore: 
            self.ignore.Hide()
            self.skip.SetDefault()
        self.GetSizer().Fit(self)
        self.Layout()
        
    #---events
    def on_skip(self,event):
        self.EndModal(wx.ID_FORWARD)
    
    def on_abort(self,event):
        self.EndModal(wx.ID_ABORT)
    
    def on_ignore(self,event):
        self.EndModal(wx.ID_IGNORE)
    
class ExecuteDialog(BrowseMixin,dialogs.ExecuteDialog):
    def __init__(self,parent,drop=False,**options):
        super(ExecuteDialog,self).__init__(parent,-1,**options)
        self.set_drop(drop)
        if wx.Platform == '__WXGTK__':
            width = 600
        else:
            width = 450
        self.SetSize((width,self.GetSize()[1]))
        
    def __do_layout(self,*args,**keyw):
        super(ExecuteDialog,self).__do_layout(*args,**keyw)
        self.save_metadata = wx.CheckBox(self, -1, 
            _("Save metadata")+" (exif && itpc)")
        self.options_sizer.Add(self.save_metadata, 0, wx.ALIGN_CENTER_VERTICAL, 6)
        self.Layout()
        
    def browse_files(self):
        style   = wx.OPEN | wx.MULTIPLE | wx.CHANGE_DIR
        if hasattr(wx,'FD_PREVIEW'):
            style   |= wx.FD_PREVIEW
        dlg = wx.FileDialog(
            self, message   = _("Choose File(s)"),
            defaultDir      = self.get_default_path(),
            defaultFile     = "",
            wildcard        = self.wildcard(),
            style           = style,
            )
        if dlg.ShowModal() == wx.ID_OK:
            self.path.SetValue(ct.PATH_DELIMITER.join(dlg.GetPaths()))
        dlg.Destroy()
            
    def browse_folder(self): 
        path    = self.show_dir_dialog(
            defaultPath = self.get_default_path(),
            message     = _("Choose an image folder")) 
        if path != None:
            self.path.SetValue(path)
        
    def get_default_path(self):
        path    = self.path.GetValue().split(ct.PATH_DELIMITER)[0] 
        return os.path.dirname(path)
    
    def export_settings(self,settings):
        settings['paths']      = self.path.GetValue().split(ct.PATH_DELIMITER)
        settings['extensions'] = [self.extensions.GetString(i) \
            for i in range(self.extensions.GetCount()) \
            if self.extensions.IsChecked(i)]
        settings['recursive']  = self.recursive.GetValue()
        settings['save_metadata']  = self.save_metadata.GetValue()
        settings['stop_for_errors'] = self.stop_for_errors.GetValue()
        settings['overwrite_existing_images'] = \
            self.overwrite_existing_images.GetValue() 
        settings['check_images_first'] = self.check_images_first.GetValue() 
        wx.GetApp()._saveSettings()
        
    def get_selected_extensions(self):
        result  = []
        exts    = formField.IMAGE_READ_EXTENSIONS
        for index, extension in enumerate(exts):
            if self.extensions.IsChecked(index):
                result.append(extension)
        return result
                
    def set_drop(self,drop):
        if drop:
            #change title
            self.SetTitle(_('Drag & Drop'))
            #radio box
            self.source.Hide()
            #hide browse & path
            self.browse.Hide()
            self.path.Hide()
            #layout (only allow vertical fit)
            grid_sizer  = self.GetSizer()
            size        = (self.GetSize()[0], grid_sizer.GetMinSize()[1])
            self.SetMinSize(size)
            self.Fit()
        
    def import_settings(self,settings):
        #path
        self.path.SetValue(ct.PATH_DELIMITER.join(settings['paths']))
        #extensions
        exts    = formField.IMAGE_READ_EXTENSIONS
        self.extensions.Set(exts)
        for index, extension in enumerate(exts):
            if extension in settings['extensions']:
                self.extensions.Check(index)
        #overwrite existing images
        self.overwrite_existing_images.SetValue(
            settings['overwrite_existing_images'] or\
            settings['overwrite_existing_images_forced'])
        #overwrite existing files
        self.check_images_first.SetValue(
            settings['check_images_first'])
        #recursive
        self.recursive.SetValue(settings['recursive'])
        #save_metadata
        self.save_metadata.SetValue(settings['save_metadata'])
        #errors
        self.stop_for_errors.SetValue(settings['stop_for_errors'])

    #---wildcard
    def wildcard(self):
        extensions  = self.get_selected_extensions()
        selected    = wildcard_list(_('All selected image types'),
                        extensions)
        default     = wildcard_list(_('All default image types'),
                        formField.IMAGE_EXTENSIONS)
        result      = [selected,default]
        result.extend([('%s '+_('images')+'|%s')\
                    %(ext,_wildcard_extension(ext))
                    for ext in extensions])
        return '|'.join(result)
        
    #---events
    def on_browse(self,event):
        if self.source.GetSelection() == 0:
            self.browse_folder()
        else:
            self.browse_files()
        
    def on_default(self,event):
        exts    = formField.IMAGE_READ_EXTENSIONS
        for index, extension in enumerate(exts):
            if extension in formField.IMAGE_EXTENSIONS:
                self.extensions.Check(index,True)
            else:
                self.extensions.Check(index,False)
                
    def on_source(self,event):
        self.browse.SetLabel(_('Browse')+' '+event.GetString())
        
class FilesDialog(dialogs.FilesDialog,_IconDialog):
    def __init__(self,parent,message,title,files,icon='warning',**keyw):
        super(FilesDialog,self).__init__(parent,-1,
            style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER|wx.MAXIMIZE_BOX,
            **keyw)
        self.SetTitle(title)
        self.message.SetLabel(message)
        self.list.InsertColumn(0, _("File"))
        self.list.InsertColumn(1, _("Folder"))
        for index, f in enumerate(files):
            index = self.list.InsertStringItem(index, os.path.basename(f))
            self.list.SetStringItem(index, 1, os.path.dirname(f))
        self.list.SetColumnWidth(0, wx.LIST_AUTOSIZE)
        self.list.SetColumnWidth(1, wx.LIST_AUTOSIZE)
        min = 100
        if self.list.GetColumnWidth(0) < min:
            self.list.SetColumnWidth(0, min)
        self._icon(icon)
        
class ProgressDialog(wx.ProgressDialog,ProgressReceiver):
    """+1 is added because eg opening a file is also an action"""
    def __init__(self,parent,title,parent_max=1,child_max=1,message=''):
        if message == '':
            message = '.'*80
        ProgressReceiver.__init__(self,parent_max,child_max)
        wx.ProgressDialog.__init__(self,
                title   = title,
                message = message,
                maximum = self.max,
                parent  = parent,
                style   = wx.PD_CAN_ABORT
                            | wx.PD_APP_MODAL
                            | wx.PD_REMAINING_TIME
                            | wx.PD_SMOOTH
            )
        self.Bind(wx.EVT_CLOSE,self.close,self)
        
    #---pubsub event methods
    def close(self,event=None):
        self.unsubscribe_all()
        self.Destroy()
        
    def update(self,result,value,**message):
        """Fix for wxPython2.6"""
        status = self.Update(value,**message)
        if type(status) == bool:
            result['keepgoing'] = result['skip'] = status
        else:
            result['keepgoing'], result['skip'] = status
        if result['keepgoing']:
            self.Refresh()
        else:
            self.close()

    def sleep(self):
        time.sleep(0.001)
        
class ActionListBox(ContentMixin,vlistTag.Box):
    #---vlist.Box obligatory overwritten
    def SetTag(self,tag=ALL):
        super(ActionListBox,self).SetTag(tag)
        #process tag
        self.tag                = tag
        if tag == SELECT:
            tag = _('default')
        #choose tag actions
        if tag == ALL:
            self.tag_actions    = self.all_actions
        else:
            self.tag_actions    = [a for a in self.all_actions 
                                    if tag in a.tags or _(a.label) == _('Save')]
        #take filter in account
        self.SetFilter(self.GetFilter().GetValue())
            
    def SetFilter(self,filter):
        if filter.strip():
            self.actions    = [a for a in self.tag_actions 
                                if filter in a.meta]
        else:
            self.actions    = self.tag_actions
        self.actions.sort(cmp=lambda x,y: cmp(_(x.label),_(y.label)))
        self.SetItemCount(len(self.actions))
        self.GetParent().CheckEmpty()
        self.RefreshAll()
        wx.GetTopLevelParent(self).ok.Enable(bool(self.actions))
        
    def _events(self):
        self.Bind(wx.EVT_CONTEXT_MENU, self.OnContextMenu)

    #---actions
    def SetActions(self,actions):
        self.all_actions    = actions.values()
        for action in self.all_actions:
            self.MetaAction(action)
        
    def MetaAction(self,action):
        action.tags = [_(tag) for tag in action.tags]
        action.meta = ' '.join([
            _(action.label).lower(),
            _(action.__doc__).lower(),
            ' '.join(action.tags)
        ])
        
    def OnContextMenu(self,event):
        # todo: does contextmenu always have to be recreated?
        #create id
        self.id_view_source = wx.NewId()
        #create menu
        menu = wx.Menu()
        item = wx.MenuItem(menu, self.id_view_source,_("View Source"))
        item.SetBitmap(graphics.bitmap('ART_FIND',(16,16)))
        self.Bind(wx.EVT_MENU, self.OnViewSource, id=self.id_view_source)
        menu.AppendItem(item)
        #show menu
        self.PopupMenu(menu)
        #destroy menu
        menu.Destroy()
        
    def OnViewSource(self,event):
        import actions
        action      = self.actions[self.list.GetSelection()]
        module      = action.__module__.split('.')[-1]
        filename    = os.path.join(ct.ACTIONS_PATH,'%s.py'%module)
        message     = open(filename).read()
        dir, base   = os.path.split(filename)
        send.frame_show_scrolled_message(message,'%s - %s'%(base,dir),
            size=(600,300))
            
    def RefreshList(self):
        self.actions.sort(cmp=lambda x,y: cmp(_(x.label),_(y.label)))
        self.Clear()
        self.SetItemCount(len(self.actions))
        self.RefreshAll()
        
    def GetItem(self, n):
        action  = self.actions[n]
        return (_(action.label),_(action.__doc__), 
            graphics.bitmap(action.icon,self.GetIconSize()))
    
    def GetStringSelection(self):
        return self.actions[self.GetSelection()].label
    
    def IsEmpty(self):
        return not (hasattr(self,'actions') and self.actions)
    
class ActionBrowser(Browser):
    ContentCtrl     = ActionListBox
    
    paint_message   = _("broaden your search")
    paint_colour    = images.LOGO_COLOUR
    #paint_logo      = images.LOGO
        
class ActionDialog(paint.Mixin,vlistTag.Dialog):
    ContentBrowser  = ActionBrowser
    
    def __init__(self,parent,actions,*args,**keyw):
        #extract tags
        tags    = self.ExtractTags(actions.values())
        #init dialog
        super(ActionDialog,self).__init__(parent,tags,*args,**keyw)
        #configure listbox
        list_box    = self.GetListBox()
        list_box.SetActions(actions)
        list_box.SetIconSize(VLIST_ICON_SIZE)
        list_box.SetTag(_('default'))
            
    def ExtractTags(self,actions):
        """Called by SetActions."""
        tags    = vlistTag.extract_tags(actions)
        tags.remove(_('default'))
        tags.sort()
        tags    = [SELECT,ALL]+tags
        return tags
        
    def GetListBox(self):
        return self.browser.content
        
    def GetStringSelection(self):
        return self.GetListBox().GetStringSelection()
    
class WritePluginDialog(dialogs.WritePluginDialog,_IconDialog):
    def __init__(self,parent,message,**keyw):
        super(WritePluginDialog,self).__init__(parent,-1,**keyw)
        path    = os.path.join(ct.PATH,'templates', 'action.py')
        self._icon('information')
        self.message.SetLabel(message)
        self.path.SetLabel('%s: %s'%(_('Path'),path))
        self._code(path)
        self.template_show(False)
        
    def _code(self,path):
        self.code.SetValue(open(path).read())
        self.code.SetMinSize((660,300))
        self.code.SetFont(wx.Font(10, wx.TELETYPE, wx.NORMAL, wx.NORMAL, 0, ""))

    #---events
    def on_help(self,event):
        import webbrowser
        url = ct.SEND_MAIL%('Writing an action plugin for Phatch',
                "As I know PIL, I'd like to write a plugin which does ... "
                "but I still have this question: ... "
            )
        webbrowser.open(url)
        
    def on_template(self,event):
        self.template_show(event.IsChecked())

    def template_show(self,show):
        sizer   = self.GetSizer()
        self.path.Show(show)
        self.code.Show(show)
        self.SetMinSize(sizer.GetMinSize())
        self.Fit()
        self.code.ShowPosition(0)
 
def test():
    class App(wx.App):
        def OnInit(self):
            wx.InitAllImageHandlers()
            frame = wx.Frame(None, -1, "")
            self.SetTopWindow(frame)
            frame.Show(False)
            wx.CallAfter(self.show_dialogs)
            return 1
        
        def show_dialogs(self):
##            self.show_error_dialog()
##            self.show_execute_dialog()
##            self.show_files_dialog()
##            self.show_progress_dialog()
            self.show_action_dialog()
            self.GetTopWindow().Destroy()
            
        def show_error_dialog(self):
            dlg = ErrorDialog(self.GetTopWindow(),'message')
            dlg.ShowModal()
            dlg.Destroy()
            
        def show_execute_dialog(self,drop=False):
            dlg = ExecuteDialog(self.GetTopWindow(),drop)
            dlg.ShowModal()
            dlg.Destroy()
            
        def show_files_dialog(self):
            dlg = FilesDialog(self.GetTopWindow(),'message','title',
                ['path/file'],'warning')
            dlg.ShowModal()
            dlg.Destroy()
            
        def show_progress_dialog(self):
            from core.lib.events import send
            import time
            n       = 5
            dlg     = ProgressDialog(self.GetTopWindow(),'title','messages',n)
            result  = {}
            for value in range(n):
                send.progress_update(result,value)
                if not result['keepgoing']: 
                    break
                time.sleep(1)
            dlg.Destroy()
            
        def show_action_dialog(self):
            from core import api
            api.init()
            dlg     = ActionDialog(self.GetTopWindow(),api.ACTIONS,
                size=(400,500))
            dlg.ShowModal()
            dlg.Destroy()
            
            
    app = App(0)
    app.MainLoop()
    
if __name__ == '__main__':
    test()