/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * Backend is the base abstract class to subclass in order to implement a
 * backend for an operating system. The backend is responsible for window and
 * context managements and for events retrieving. For instance, UNIX systems
 * need a backend to glue OpenGL with XWindow using GLX, and Windows systems
 * need another one to glue OpenGL with the Windows API using WGL.
 *
 * One of the task of the backend is to build and push the generated events to
 * the Viewport. Events are created with the API of PgmEvents, and pushed to
 * the Viewport using the pgm_viewport_push_event() method.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmbackend.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gl_backend_debug);
#define GST_CAT_DEFAULT pgm_gl_backend_debug

static GstObjectClass *parent_class = NULL;

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE_EXTENDED (PgmBackend, pgm_backend, GST_TYPE_OBJECT,
                                  G_TYPE_FLAG_ABSTRACT, {});

void
pgm_backend_register (GTypeModule *module)
{
  pgm_backend_register_type (module);
}

static void
pgm_backend_dispose (GObject *object)
{
  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_backend_class_init (PgmBackendClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  GST_DEBUG_CATEGORY_INIT (pgm_gl_backend_debug, "pgm_gl_backend", 0,
                           "OpenGL plugin: PgmBackend");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_backend_dispose);
}

static void
pgm_backend_class_finalize (PgmBackendClass *klass)
{
  return;
}

static void
pgm_backend_init (PgmBackend *backend)
{
  GST_DEBUG_OBJECT (backend, "init");

  backend->context = NULL;
}

/* Public methods */

gboolean
pgm_backend_create_window (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->create_window)
    return klass->create_window (backend);

  return FALSE;
}

gboolean
pgm_backend_destroy_window (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->destroy_window)
    return klass->destroy_window (backend);

  return FALSE;
}

void
pgm_backend_set_title (PgmBackend *backend,
                       const gchar *title)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->set_title)
    klass->set_title (backend, title);
}

gboolean
pgm_backend_set_decorated (PgmBackend *backend,
                           gboolean decorated)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->set_decorated)
    return klass->set_decorated (backend, decorated);

  return FALSE;
}

void
pgm_backend_swap_buffers (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->swap_buffers)
    klass->swap_buffers (backend);
}

gpointer
pgm_backend_get_proc_address (PgmBackend *backend,
                              const gchar *proc_name)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), NULL);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->get_proc_address)
    return klass->get_proc_address (backend, proc_name);

  return NULL;
}

gboolean
pgm_backend_set_size (PgmBackend *backend,
                      gint width,
                      gint height)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->set_size)
    return klass->set_size (backend, width, height);

  return FALSE;
}

gboolean
pgm_backend_set_fullscreen (PgmBackend *backend,
                            gboolean fullscreen)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->set_fullscreen)
    return klass->set_fullscreen (backend, fullscreen);

  return FALSE;
}

gboolean
pgm_backend_set_visibility (PgmBackend *backend,
                            gboolean visibility)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->set_visibility)
    klass->set_visibility (backend, visibility);

  return FALSE;
}

void
pgm_backend_get_screen_size_mm (PgmBackend *backend,
                                gint *width,
                                gint *height)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->get_screen_size_mm)
    klass->get_screen_size_mm (backend, width, height);
}

gboolean
pgm_backend_set_screen_resolution (PgmBackend *backend,
                                   gint width,
                                   gint height)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->set_screen_resolution)
    return klass->set_screen_resolution (backend, width, height);

  return FALSE;
}

void
pgm_backend_get_screen_resolution (PgmBackend *backend,
                                   gint *width,
                                   gint *height)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->get_screen_resolution)
    klass->get_screen_resolution (backend, width, height);
}

gboolean
pgm_backend_build_text_lists (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->build_text_lists)
    return klass->build_text_lists (backend);

  return FALSE;
}

gboolean
pgm_backend_destroy_text_lists (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->destroy_text_lists)
    return klass->destroy_text_lists (backend);

  return FALSE;
}

void
pgm_backend_raster_text (PgmBackend *backend,
                         const gchar *text,
                         gfloat x,
                         gfloat y,
                         gfloat r,
                         gfloat g,
                         gfloat b)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->raster_text)
    klass->raster_text (backend, text, x, y, r, g, b);
}

void
pgm_backend_wait_for_vblank (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->wait_for_vblank)
    klass->wait_for_vblank (backend);
}

void
pgm_backend_notify_startup_complete (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->notify_startup_complete)
    klass->notify_startup_complete (backend);
}

gboolean
pgm_backend_set_cursor (PgmBackend *backend,
                        PgmViewportCursor cursor)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->set_cursor)
    return klass->set_cursor (backend, cursor);

  return FALSE;
}

gboolean
pgm_backend_set_icon (PgmBackend *backend,
                      GdkPixbuf *icon)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->set_icon)
    return klass->set_icon (backend, icon);

  return FALSE;
}

void
pgm_backend_set_drag_status (PgmBackend *backend,
                             gboolean accept)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->set_drag_status)
    klass->set_drag_status (backend, accept);
}

gboolean
pgm_backend_is_accelerated (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->is_accelerated)
    return klass->is_accelerated (backend);

  return FALSE;
}

gboolean
pgm_backend_is_embeddable (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->is_embeddable)
    return klass->is_embeddable (backend);

  return FALSE;
}

void
pgm_backend_get_embedding_id (PgmBackend *backend,
                              gulong *embedding_id)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->get_embedding_id)
    klass->get_embedding_id (backend, embedding_id);
}

gboolean
pgm_backend_has_alpha_component (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->has_alpha_component)
    return klass->has_alpha_component (backend);

  return FALSE;
}

gboolean
pgm_backend_has_system_buffer (PgmBackend *backend)
{
  PgmBackendClass *klass;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), FALSE);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->has_system_buffer)
    return klass->has_system_buffer (backend);

  return FALSE;
}

gpointer
pgm_backend_create_system_buffer_object (PgmBackend *backend,
                                         gconstpointer system_buffer,
                                         PgmImagePixelFormat format)
{
  PgmBackendClass *klass;
  gpointer system_buffer_object = NULL;

  g_return_val_if_fail (PGM_IS_BACKEND (backend), system_buffer_object);

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->create_system_buffer_object)
    system_buffer_object =
      klass->create_system_buffer_object (backend, system_buffer, format);

  return system_buffer_object;
}

void
pgm_backend_destroy_system_buffer_object (PgmBackend *backend,
                                          gpointer system_buffer_object)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->destroy_system_buffer_object)
    klass->destroy_system_buffer_object (backend, system_buffer_object);
}

void
pgm_backend_bind_system_buffer_object (PgmBackend *backend,
                                       gconstpointer system_buffer_object)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->bind_system_buffer_object)
    klass->bind_system_buffer_object (backend, system_buffer_object);
}

void
pgm_backend_release_system_buffer_object (PgmBackend *backend,
                                          gconstpointer system_buffer_object)
{
  PgmBackendClass *klass;

  g_return_if_fail (PGM_IS_BACKEND (backend));

  klass = PGM_BACKEND_GET_CLASS (backend);

  if (klass->release_system_buffer_object)
    klass->release_system_buffer_object (backend, system_buffer_object);
}
