/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifndef __PGM_GLES_TEXTURE_H__
#define __PGM_GLES_TEXTURE_H__

#include <pgm/pgmimage.h>
#include "pgmglescontext.h"
#include "pgmglesdefs.h"

G_BEGIN_DECLS

/* Texture locking */
#define PGM_GLES_TEXTURE_LOCK(texture)   (g_mutex_lock (texture->mutex))
#define PGM_GLES_TEXTURE_UNLOCK(texture) (g_mutex_unlock (texture->mutex))

/* Type casting */
#define PGM_GLES_TEXTURE_CAST(obj) ((PgmGlesTexture*) (obj))

typedef struct _PgmGlesTexture PgmGlesTexture;

/* Function pointer types */
typedef void (*PgmGlesTextureUploadFunc) (PgmGlesTexture*, void*);
typedef void (*PgmGlesTextureCreateFunc) (PgmGlesTexture*);
typedef void (*PgmGlesTextureBindFunc)   (PgmGlesTexture*);
typedef void (*PgmGlesTextureUnbindFunc) (PgmGlesTexture*);

/* Texture flags */
typedef enum {
  PGM_GLES_TEXTURE_IDENTITY_MATRIX = (1 << 0)
} PgmGlesTextureFlags;

/* Data storage types */
typedef enum {
  PGM_GLES_TEXTURE_CLEAN,
  PGM_GLES_TEXTURE_BUFFER,
  PGM_GLES_TEXTURE_GST_BUFFER,
  PGM_GLES_TEXTURE_PIXBUF,
} PgmGlesTextureStorageType;

struct _PgmGlesTexture {
  PgmGlesTextureStorageType storage;

  union {
    gpointer   buffer;
    GstBuffer *gstbuffer;
    GdkPixbuf *pixbuf;
  } data;

  PgmImagePixelFormat csp; /* Color space */
  gint   width;            /* Width of the buffer */
  gint   height;           /* Height of the buffer */
  gint   width_pot;        /* Power-of-two width of the buffer */
  gint   height_pot;       /* Power-of-two height of the buffer */
  gfloat norm_width;       /* width / width_pot */
  gfloat norm_height;      /* height / height_pot */
  gfloat inv_norm_width;   /* (width / width_pot)^1 */
  gfloat inv_norm_height;  /* (height / height_pot)^1 */
  gint   stride;           /* Stride of the buffer (not used) */
  gint   size;             /* Size of the buffer in bytes */

  /* State flags */
  guint8 flags;

  /* Parameters */
  PgmMat4x4   *matrix;
  PgmGlesEnum  filter;
  PgmGlesEnum  wrap;
  PgmGlesEnum  format;

  /* Number of identifiant */
  gint count;

  /* Identifiant(s) */
  guint *id;
};

/* create a new texture */
PgmGlesTexture *pgm_gles_texture_new               (PgmGlesContext *glescontext);

/* Free a texture */
void            pgm_gles_texture_free              (PgmGlesTexture *glestexture);

/* Set a raw buffer */
void            pgm_gles_texture_set_buffer        (PgmGlesTexture *glestexture,
                                                    guchar *buffer,
                                                    PgmImagePixelFormat csp,
                                                    guint width,
                                                    guint height,
                                                    guint size,
                                                    guint stride);

/* Set a GdkPixbuf */
void            pgm_gles_texture_set_pixbuf        (PgmGlesTexture *glestexture,
                                                    GdkPixbuf *pixbuf);

/* Set a GstBuffer */
void            pgm_gles_texture_set_gst_buffer    (PgmGlesTexture *glestexture,
                                                    GstBuffer *gstbuffer,
                                                    PgmImagePixelFormat csp,
                                                    guint width,
                                                    guint height,
                                                    guint stride);

/* Update a GstBuffer */
void            pgm_gles_texture_update_gst_buffer (PgmGlesTexture *glestexture,
                                                    GstBuffer *gstbuffer);

/* Bind the texture */
void            pgm_gles_texture_bind              (PgmGlesTexture *glestexture);

/* Unbind the texture */
void            pgm_gles_texture_unbind            (PgmGlesTexture *glestexture);

/* Generate the texture */
void            pgm_gles_texture_generate          (PgmGlesTexture *glestexture);

/* Clean up the texture */
void            pgm_gles_texture_clean             (PgmGlesTexture *glestexture);

/* Upload the texture */
void            pgm_gles_texture_upload            (PgmGlesTexture *glestexture);

/* Update the texture */
void            pgm_gles_texture_update            (PgmGlesTexture *glestexture);

/* Set the texture matrix */
void            pgm_gles_texture_set_matrix        (PgmGlesTexture *glestexture,
                                                    PgmMat4x4 *matrix);

G_END_DECLS

#endif /* __PGM_GLES_TEXTURE_H__ */
