/*	Processing_Changes

PIRL CVS ID: Processing_Changes.java,v 1.13 2012/04/16 06:04:10 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor;

import	PIRL.PVL.Parameter;
import	PIRL.Configuration.Configuration;
import	PIRL.Configuration.Configuration_Exception;

import	java.util.Vector;

/**	<i>Processing_Changes</i> contains information provided by a
	Conductor about changes in its processing conditions.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.13
	@see		Conductor
	@see		Processing_Event
*/
public class Processing_Changes
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Processing_Changes ($Revison$ 2012/04/16 06:04:10)";


private Configuration
	Configuration			= null;

private int
	Processing_State		= 0;

private Vector<String>
	Source_Record			= null;

private Vector<String>
	Procedure_Record		= null;

private int
	Sequential_Failures		= -1;

private String
	Error_Condition			= null;

private boolean
	Sources_Refreshed		= false,
	Procedures_Changed		= false,
	Exiting					= false;


/**	The value to set for all {@link #Configuration(Configuration)
	configuration} parameters named "Password" (case insensitive).
*/
public static final String
	MASKED_PASSWORD			= "****";

private static final String
	NL						= System.getProperty ("line.separator");

/*==============================================================================
	Constructors
*/
/**	Construct Processing_Changes containing no changes.
*/
public Processing_Changes ()
{}

/*==============================================================================
	Accessors
*/
/**	Get the changed Configuration.
<p>
	@return	A copy of the changed Configuration. This will be null if
		the Configuration is unchanged.
	@see	#Configuration(Configuration)
*/
public Configuration Configuration ()
{return Configuration;}

/**	Set the changed Configuration.
<p>
	If the Configuration is non-null it is copied into this Processing_Changes.
	The copy is set to be {@link Configuration#Case_Sensitive(boolean)
	case insensitive} and all parameters named "PASSWORD" (case insensitive)
	have their values reset to {@link #MASKED_PASSWORD}.
<p>
	@param	configuration	A Configuration. May be null.
	@return	This Processing_Changes.
	@see	#Configuration(Parameter)
	@see	#Configuration()
*/
public Processing_Changes Configuration
	(
	Configuration	configuration
	)
	throws Configuration_Exception
{
if (configuration == null)
	Configuration = null;
else
	{
	Configuration = new Configuration (configuration);
	Configuration.Case_Sensitive (false);
	Configuration.Set_All ("PASSWORD", MASKED_PASSWORD);
	}
return this;
}

/**	Set the changed Configuration from a Parameter.
<p>
	If the Parameter is non-null it is copied into the Configuration of
	this Processing_Changes. The copy is set to be {@link
	Configuration#Case_Sensitive(boolean) case insensitive} and all
	parameters named "PASSWORD" (case insensitive) have their values
	reset to {@link #MASKED_PASSWORD}.
<p>
	@param	parameters	A Parameter. May be null.
	@return	This Processing_Changes.
	@see	#Configuration(Configuration)
	@see	#Configuration()
*/
public Processing_Changes Configuration
	(
	Parameter	parameters
	)
	throws Configuration_Exception
{
if (parameters == null)
	Configuration = null;
else
	{
	Configuration = new Configuration (parameters);
	Configuration.Case_Sensitive (false);
	Configuration.Set_All ("PASSWORD", MASKED_PASSWORD);
	}
return this;
}

/**	Get the processing state.
<p>
	The expected Conductor processing state values are:
<p>
<dl>
<dt>1 - {@link Conductor#RUNNING}
	<dd>Source records are being processing.
<dt>2 - {@link Conductor#POLLING}
	<dd>No source records are currently available for processing; the
	Conductor is polling for new source records to process.
<dt>-1 - {@link Conductor#WAITING}
	<dd>The Conductor is waiting to be told to being processing.
<dt>-2 - {@link Conductor#HALTED}
	<dd>A problem condition caused the Conductor to stop processing.
	The problem may be the result of the maximum number of sequential
	source records processing procedure failures having occured, a
	database access failure, or some other system error.
<dt>0 - Unchanged
	<dd>The processing state has not changed.
</dl>
<p>
	@return	A Conductor processing state value, or zero the state is
		unchanged.
	@see	#Processing_State(int)
*/
public int Processing_State ()
{return Processing_State;}

/**	Set the processing state.
<p>
	@param	processing_state	A Conductor processing state value, or
		zero the state is unchanged.
	@return	This Processing_Changes.
	@see	#Processing_State()
*/
public Processing_Changes Processing_State
	(
	int		processing_state
	)
{Processing_State = processing_state; return this;}

/**	Get the changed source record.
<p>
	@return	A Vector containing the changed source record, or null if
		the record is unchanged.
	@see	#Source_Record(Vector)
*/
public Vector<String> Source_Record ()
{return Source_Record;}

/**	Set the changed source record.
<p>
	If the record is non-null it is copied into this Processing_Changes.
<p>
	@param	source_record	A Vector containing the changed source
		record, or null if the record is unchanged.
	@return	This Processing_Changes.
	@see	#Source_Record()
*/
public Processing_Changes Source_Record
	(
	Vector<String>	source_record
	)
{
if (source_record == null)
	Source_Record = null;
else
	Source_Record = new Vector<String> (source_record);
return this;
}

/**	Test if the source records cache was refreshed with new records.
<p>
	@return	true if the source records cache was refreshed with new
		records; false if the cache is unchanged.
	@see	#Sources_Refreshed(boolean)
*/
public boolean Sources_Refreshed ()
{return Sources_Refreshed;}

/**	Set if the source records cache was refreshed with new records.
<p>
	@param	refreshed	true if the source records cache was refreshed
		with new records; false if the cache is unchanged.
	@return	This Processing_Changes.
	@see	#Sources_Refreshed()
*/
public Processing_Changes Sources_Refreshed
	(
	boolean	refreshed
	)
{Sources_Refreshed = refreshed; return this;}

/**	Get the changed procedure record.
<p>
	@return	A Vector containing the changed procedure record, or null if
		the record is unchanged.
	@see	#Procedure_Record(Vector)
*/
public Vector<String> Procedure_Record ()
{return Procedure_Record;}

/**	Set the changed prcedure record.
<p>
	If the record is non-null it is copied into this Processing_Changes.
<p>
	@param	procedure_record	A Vector containing the changed procedure
		record, or null if the record is unchanged.
	@return	This Processing_Changes.
	@see	#Procedure_Record()
*/
public Processing_Changes Procedure_Record
	(
	Vector<String>	procedure_record
	)
{
if (procedure_record == null)
	Procedure_Record = null;
else
	Procedure_Record = new Vector<String> (procedure_record);
return this;
}

/**	Test if the procedure records have changed.
<p>
	@return	true if the procedure records were replaced with new
		records; false if the records are unchanged.
	@see	#Procedures_Changed(boolean)
*/
public boolean Procedures_Changed ()
{return Procedures_Changed;}

/**	Set if the procedure records have changed.
<p>
	@param	changed	true if the procedure records were replaced with new
		records; false if the records are unchanged.
	@return	This Processing_Changes.
	@see	#Procedures_Changed()
*/
public Processing_Changes Procedures_Changed
	(
	boolean	changed
	)
{Procedures_Changed = changed; return this;}

/**	Get the changed sequential failures limit.
<p>
	@return	The sequential failures limit. This will be -1 if the limit
		is unchanged.
	@see	#Sequential_Failures(int)
*/
public int Sequential_Failures ()
{return Sequential_Failures;}

/**	Set the changed sequential failures limit.
<p>
	@return	The sequential failures limit. A negative value means the
		limit is unchanged.
	@return	This Processing_Changes.
	@see	#Sequential_Failures()
*/
public Processing_Changes Sequential_Failures
	(
	int		failure_count
	)
{
if (failure_count >= 0)
	Sequential_Failures = failure_count;
return this;
}

/**	Get an error condition report.
<p>
	@return	A String describing the error condition. This will be null
		if no error condition occurred.
	@see	#Error_Condition(String)
*/
public String Error_Condition ()
{return Error_Condition;}

/**	Set an error condition report.
<p>
	@return	A String describing the error condition. If null no error
		condition occurred.
	@return	This Processing_Changes.
	@see	#Error_Condition()
*/
public Processing_Changes Error_Condition
	(
	String	error_condition
	)
{Error_Condition = error_condition; return this;}

/**	Test if the Conductor is exiting.
<p>
	@return	true if the Conductor is exiting; false if the Conductor
		is still running.
	@see	#Exiting(boolean)
*/
public boolean Exiting ()
{return Exiting;}

/**	Set if the Conductor is exiting.
<p>
	@return	true if the Conductor is exiting; false if the Conductor
		is still running.
	@return	This Processing_Changes.
	@see	#Exiting()
*/
public Processing_Changes Exiting
	(
	boolean	state
	)
{Exiting = state; return this;}

/**	Get a description of this Processing_Changes state.
<p>
	The Processing_Changes object is described by its {@link #ID}
	followed by the name and values of each state variable.
<p>
	The {@link #Configuration() Configuration} {@link
	Configuration#Description() description} will occupy many lines and
	may include at the end warnings from the PVL Lister about illegal
	syntax in values; this is to be expected and is harmless. If the
	Configuration is malformed its description will begin with an empty
	line followed by a description of the problem.
<p>
	@return	A multi-line String describing the state of this
		Processing_Changes object.
*/
public String toString ()
{
return
	ID + NL
	+"    Configuration -" + NL
		+ ((Configuration == null) ?
			"null" :  Configuration.Description ()) + NL
	+"    Processing_State: " + Processing_State + NL
	+"    Sources_Refreshed: " + Sources_Refreshed + NL
	+"    Source_Record: " + Source_Record + NL
	+"    Procedures_Changed: " + Procedures_Changed + NL
	+"    Procedure_Record: " + Procedure_Record + NL
	+"    Sequential_Failures: " + Sequential_Failures + NL
	+"    Error_Condition -" + NL
		+ Error_Condition + NL
	+"    Exiting: " + Exiting + NL;
}

}
