/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001  Kevin P. Lawton
 *
 *  cgmain-mon.c: Main decode component of the DT engine.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_MONITOR_SPACE
#include "monitor.h"




  static unsigned
dtPassThru16(vm_t *, Bit8u *p, unsigned remain, Bit8u *iPtr,
             instruction_t *i);


  Bit32u
dtTranslateSequence(vm_t *vm, unsigned metaI, Bit32u guestOff,
                    Bit32u guestLinAddr, Bit32u monLinAddr)
{
  unsigned ret;
  instruction_t i; // xxx Needed?  Can we use one in vm_t?
  unsigned seg32, maxLen, opcodeAttr, tcodeLen, n;
  Bit32u pageRemain, pageOffCurr, pageOffOrig, eip;
  Bit8u *tcodePtr, *prevTcodePtr, *iPtr;
  unsigned tcodeSnippetsN = 0, tcodeRemain;
  dtPageMeta_t *dtPageMetaTable = vm->guest.addr.dtPageMetaTable;


  /* Calculate maximum tcode sequence length.  Since we can't add
   * any sequences longer than this, use this value to bound our
   * translations.
   */
  /* Start at beginning of buffer */
  tcodePtr = vm->tempTcodeChunk.raw;
  tcodeRemain = sizeof(tcodeChunk_t) -
      sizeof(vm->guest.addr.tcodeChunk[0].header);

  iPtr = (Bit8u *) monLinAddr;
  eip  = guestOff;
  pageOffCurr =
    pageOffOrig =
      (guestLinAddr & 0x00000fff);
  seg32 = vm->guest_cpu.desc_cache[SRegCS].desc.d_b;
  vm->relocN = 0; /* Clear relocation information. */
  // cpl = G_GetCPL(vm);

loop:
  /* If we have run out of elements of the snippet array, then
   * we have to virtualize the next instruction to force an end
   * to translation.
   */
  if (tcodeSnippetsN >= (MaxTcodeSnippets-1))
    goto virtualize;
  /* If there isn't enough room for at least a 1-byte instruction
   * and an extra byte to breakpoint after that, then we need to
   * stop.
   */
  if (tcodeRemain <= 1)
    goto virtualize;

  /* Determine the maximum length of an instruction to fetch.  Normally,
   * this is bounded by MaxIA32OpcodeLen, but if we are near a page
   * boundary, then we have to limit the fetch to the remaining number
   * of bytes in the page.
   */
  maxLen = MaxIA32OpcodeLen;
  // xxx Depending if last instruction is a short static branch,
  // xxx the last instruction in the page can align to the last
  // xxx offset in the page.  Otherwise, there must be at least
  // xxx one more start of an instruction to virtualize.
  // xxx pageRemain = 0x00001000 - pageOffCurr;
  pageRemain = 0x00000fff - pageOffCurr;
  /* fetchDecode needs at least 1 byte to function.  If there is
   * no room, then virtualize.
   */
  if (pageRemain == 0)
    goto virtualize;
  if (pageRemain < MaxIA32OpcodeLen)
    maxLen = pageRemain;

  ret = fetchDecode(iPtr, &i, maxLen, seg32);

  /* If fetch-decode ran up against a boundary and could not
   * fetch enough bytes, or if the opcode is invalid, then mark
   * the instruction as needing virtualization.
   */
  if ((ret == 0) || (i.attr & InvalidOpcode)) {
    goto virtualize;
    }
  if (i.attr & GroupN) {
    opcodeAttr = vm->vOpcodeMap->groups[GetGroupIndex(i.attr)][i.nnn];
    }
  else {
    opcodeAttr = vm->vOpcodeMap->standard[i.b1];
    }

// xxx Check for buffer overflows everywhere.
  if (opcodeAttr & VOpcodeRunNative) {
    /* Check for use of CS/GS over-rides.  Since we use these registers
     * for the tcode and ring3 handlers, we need to virtualize their use.
     */
    if ( (i.seg == SRegCS) || (i.seg == SRegGS) ) {
      /* If instruction uses CS or GS over-ride, then we need to virtualize
       * the instruction, since we modify these segments for use with tcode.
       */
      goto virtualize;
      }
    /* Emit pass-through code. */
    prevTcodePtr = tcodePtr; /* Save pointer to beginning of tcode sequence. */
    if (seg32) {
      /* There must be enough room to pass this instruction through
       * as-is, plus 1 byte to virtualize the end of sequence.
       */
      n = i.ilen;
      if (n < tcodeRemain) {
        mon_memcpy(tcodePtr, iPtr, n);
        }
      else
        goto virtualize;
if (n > (tcodeRemain-1)) monpanic(vm, "yucked 3.\n");
      }
    else {
      n = dtPassThru16(vm, tcodePtr, tcodeRemain-1, iPtr, &i);
      /* If there wasn't enough room in our scratch buffer for the
       * complemented 16-bit pass-through code, then we need to
       * virtualize.
       */
      if (n==0)
        goto virtualize;
if (n > (tcodeRemain-1)) monpanic(vm, "yucked 2: n=%u, remain=%u.\n",
n, tcodeRemain);
      }

addSnippet:

    tcodePtr    += n;
    tcodeRemain -= n;
    /* Register offsets for this instruction, both of the guest page offset
     * and the offset into the tcode buffer we will pass to dtAddTcode.
     */
    vm->tcodeSnippets[tcodeSnippetsN].pOff = pageOffCurr;
    vm->tcodeSnippets[tcodeSnippetsN].tcodeBuffOff =
        prevTcodePtr - vm->tempTcodeChunk.raw;
    vm->tcodeSnippets[tcodeSnippetsN].tcodeLen = tcodePtr - prevTcodePtr;
    tcodeSnippetsN++;
    pageOffCurr += i.ilen;
    iPtr += i.ilen;
    eip  += i.ilen;
    goto loop;
    }
  else if (opcodeAttr & VOpcodeTranslate) {
    prevTcodePtr = tcodePtr; /* Save pointer to beginning of tcode sequence. */
    if (i.attr & GroupN) {
      //opcodeAttr = vm->vOpcodeMap->groups[GetGroupIndex(i.attr)][i.nnn];
      monpanic(vm, "dtTranslateSeq: GrpIndex=%u, nnn=%u\n",
               GetGroupIndex(i.attr), i.nnn);
      }
    else {
      switch (i.b1) {
        case 0x70:   /* Jcc_Jb */
        case 0x71:
        case 0x72:
        case 0x73:
        case 0x74:
        case 0x75:
        case 0x76:
        case 0x77:
        case 0x78:
        case 0x79:
        case 0x7a:
        case 0x7b:
        case 0x7c:
        case 0x7d:
        case 0x7e:
        case 0x7f:
        case 0xe9:   /* Jmp_Jv */
        case 0xeb:   /* Jmp_Jb */
        case 0x180:  /* Jcc_Jv */
        case 0x181:
        case 0x182:
        case 0x183:
        case 0x184:
        case 0x185:
        case 0x186:
        case 0x187:
        case 0x188:
        case 0x189:
        case 0x18a:
        case 0x18b:
        case 0x18c:
        case 0x18d:
        case 0x18e:
        case 0x18f:
          n = dtGenJcc(vm, tcodePtr, tcodeRemain-1, &i, eip);
if (n > (tcodeRemain-1)) monpanic(vm, "yucked 1.\n"); // xxx Fix yucked
          if (n==0)
            goto virtualize;
          break;
        case 0xe2:  /* Loop_Jb */
          n = dtGenLoopJb(vm, tcodePtr, tcodeRemain-1, &i, eip);
if (n > (tcodeRemain-1)) monpanic(vm, "yucked 5.\n"); // xxx Fix yucked
          if (n==0)
            goto virtualize;
          break;
        case 0xe3:  /* JCXZ_Jb */
          n = dtGenJCXZ(vm, tcodePtr, tcodeRemain-1, &i, eip);
if (n > (tcodeRemain-1)) monpanic(vm, "yucked 4.\n"); // xxx Fix yucked
          if (n==0)
            goto virtualize;
          break;
        default:
          monpanic(vm, "dtTranslateSeq: i.b1=0x%x.\n", i.b1);
        }
      }
    goto addSnippet;
    }
  else {
virtualize:
    prevTcodePtr = tcodePtr; /* Save pointer to beginning of tcode sequence. */
    if (tcodeRemain < 1)
      monpanic(vm, "dtTranslateSeq: No room left to virtualize instruction.\n");
    *tcodePtr++  = INT3OP;
    tcodeRemain -= 1;
    vm->tcodeSnippets[tcodeSnippetsN].pOff = pageOffCurr;
    vm->tcodeSnippets[tcodeSnippetsN].tcodeBuffOff =
        prevTcodePtr - vm->tempTcodeChunk.raw;
    vm->tcodeSnippets[tcodeSnippetsN].tcodeLen = tcodePtr - prevTcodePtr;
    tcodeSnippetsN++;
    /* Set other stuff here if we want to continue fetching */
    goto finish;
    }

finish:
  /* Find length of translated sequence */
  tcodeLen = (tcodePtr - vm->tempTcodeChunk.raw);

  /* Add the fragment to the fragment cache */
  tcodePtr = dtAddTcode(vm, metaI, vm->tempTcodeChunk.raw, tcodeLen,
                        vm->tcodeSnippets, tcodeSnippetsN, guestOff);
  if (tcodePtr == (Bit8u *) TcodeOffsetNone)
    monpanic(vm, "dtTranslateSequence: TcodeOffsetNone.\n");

  /*
   *  Relocate branches.  Because branches use displacements relative
   *  to the EIP offset, they depend on the location where the tcode
   *  is actually placed.
   */

  for (n = 0; n < vm->relocN; n++) {
    /*
     *  Translate the address of the displacement to patch,
     *  from the original tcode buffer to the real tcode home
     */

    Bit32u offPtr, targetOffset;
    int o;

    /* Find a pointer to the offset in the tcode, given the whole
     * tcode chunk as been placed at a different location than the temp
     * buffer space that the tcode was generated in originally.
     */
    offPtr = (Bit32u) tcodePtr +
      ((Bit32u) vm->reloc[n].tcodePtr - (Bit32u) vm->tempTcodeChunk.raw);
    targetOffset = vm->reloc[n].tcodeOff;
    /* See if the target offset is in the tcode itself. */
    o = ((int) targetOffset) - (int) vm->tempTcodeChunk.raw;
    if ( (o >= 0) && (o < sizeof(vm->tempTcodeChunk)) ) {
      /* Target offset is in the tcode.  This means we are relocating
       * and offset used as a literal, not as a branch.  Relocate it
       * according to the actual placement of the tcode.
       */
      targetOffset += ( ((Bit32u) tcodePtr ) -
                        ((Bit32u) vm->tempTcodeChunk.raw) );
      *(Bit32u *) offPtr = targetOffset;
      }
    else {
      /* Target offset is outside of tcode.  This is used for branches only.
       * Relocate the branch offset.  Branch offsets are relative to
       * the EIP value at the _end_ of the instruction.
       */
      *(Bit32u *) offPtr =
          targetOffset - (offPtr + 4);
      }
    }

  dtPageMetaTable[metaI].ts.mon_prescan = vm_rdtsc();
  return( (Bit32u) tcodePtr );
}

#if 0
  /*
   *  Dynamically translate the guest code sequence at the given
   *  offset and linear address
   */

  int i;
  Bit32u pOff;
  Bit8u *iPtr, *isPtr;
  unsigned b0, b1, modrm, sib;
  Bit32u targetOffset32, gOffInstructionStart;
  unsigned tcodeLen;
  Bit8s displ8;

  pOff = gla & 0x00000fff;

  iPtr = (Bit8u *) gla;
  tcodePtr = tcode;

  /* clear the relocation records */
  relocN = 0;

  /* main decoder loop */
loop:
  isPtr = iPtr;
  b0 = *iPtr++;
  gOffInstructionStart = gOff;


  switch (b0) {
    case 0x0f: /* 2-byte primary opcode */
      b1 = *iPtr++;
      switch (b1) {
        case 0x84: /* JZ  Jd */
          gOff += 6;
          targetOffset32 = gOff + *(Bit32s *) iPtr;
          iPtr += 4;

          /* Emit code */
          tcodePtr = dtGenJzJd(tcodePtr, targetOffset32);
          goto loop;

        case 0x85: /* JNZ Jd */
          gOff += 6;
          targetOffset32 = gOff + *(Bit32s *) iPtr;
          iPtr += 4;

          /* Emit code */
          tcodePtr = dtGenJnzJd(tcodePtr, targetOffset32);
          goto loop;

        default:
          printf("dtTS: opcode 0x0f 0x%02x unhandled\n", b1);
          exit(1);
        }

      break;

    case 0x01: /* ADD EvGv */
      b1 = *iPtr++;
      if ((b1 & 0xc0) == 0xc0) {
        /* ADD Reg,Reg */
        gOff += 2;

        /* Emit code */
        tcodePtr = dtPass(tcodePtr, isPtr, 2);
        goto loop;
        }
      else {
        printf("dtTS: ADD not Reg,Reg\n");
        exit(1);
        }

    case 0x49: /* decl %ecx */
    case 0x51: /* pushl %ecx */
    case 0x59: /* popl %ecx */
    case 0x60: /* pusha */
    case 0x61: /* popa */
    case 0x90: /* NOP */
      *tcodePtr++ = b0; /* pass through */
      gOff++;
      goto loop;

    case 0xe2: /* LOOP Jb */
      gOff += 2;
      displ8 = *(Bit8s *) iPtr++;

      tcodePtr = dtGenLoopJb(tcodePtr, displ8);
      goto loop;

    case 0xb9: /* mov ECX,Id */
      gOff += 5;
      iPtr += 4;
      tcodePtr = dtPass(tcodePtr, isPtr, 5);
      goto loop;

    case 0xc3: /* RET */
      gOff++;

      /* Emit code */
      tcodePtr = dtGenRet(tcodePtr);
      goto finish;

    case 0xe9: /* JMP_Jd */
      gOff += 5;
      targetOffset32 = gOff + *(Bit32s *) iPtr;
      iPtr += 4;

      tcodePtr = dtGenJmpJd(tcodePtr, targetOffset32);
      goto finish;

    case 0xff: /* Group5 */
      modrm = *iPtr++;
      switch (modrm) {
        case 0x0d: /* DEC_Ed */
          gOff += 6;
          iPtr += 4;

          /* Emit pass through */
          tcodePtr = dtPass(tcodePtr, isPtr, 6);
          goto loop;

        case 0x24:
          sib = *iPtr++;
          if (sib == 0x8d) {    /* jmp *table(,%ecx,4) */
            gOff += 7;
            targetOffset32 = *(Bit32s *) iPtr;
            iPtr += 4;

            /* Emit code */
            tcodePtr = dtGenJmpEcx4(tcodePtr, targetOffset32);
            goto finish;
            }
          else {
            printf("dtTS: G5 sib=0x%x unhandled\n", sib);
            exit(1);
            }

        default:
          printf("dtTS: G5 modrm 0x%02x unhandled\n", modrm);
          exit(1);
        }

      break;

    default:
      printf("dtTS: opcode 0x%02x unhandled\n", b0);
      exit(1);
    }

  printf("dtTS: finished prematurely\n");
  exit(1);

#endif

  unsigned
dtPassThru16(vm_t *vm, Bit8u *p, unsigned remain, Bit8u *iPtr,
             instruction_t *i)
{
  /* Because we are running 16-bit code in a 32-bit segment,
   * we have to complement the instruction's OpSize and AddrSize
   * attributes to make it compatible.
   */

  /* Start with usual case as defaults. */
  unsigned useOSPrefix=1, useASPrefix=1;

  unsigned iRemain, b;
  Bit8u *pOrig = p;

  iRemain = i->ilen; /* Bytes remaining in instruction. */

  b = *iPtr; /* Get first byte of instruction. */
  while ( (OpcodeInfo[b] & Prefix) && iRemain ) {
    if ( b == 0x66 )
      useOSPrefix = 0; /* Instruction uses OpSize prefix; complement. */
    else if ( b == 0x67 )
      useASPrefix = 0; /* Instruction uses AddrSize prefix; complement. */
    else {
      /* Pass prefix through */
      if (remain < 1)
        return(0); /* Out of space. */
      *p++ = b;
      remain--;
      }
    iRemain--;
    iPtr++;
    b = *iPtr;
    }
  if (iRemain==0)
    monpanic(vm, "dtPassThru16: iRemain==0.\n");
  if (useOSPrefix) {
    if (remain < 1)
      return(0); /* Out of space. */
    *p++ = 0x66;
    remain--;
    }
  if (useASPrefix) {
    if (remain < 1)
      return(0); /* Out of space. */
    *p++ = 0x67;
    remain--;
    }
  if (remain < iRemain)
    return(0); /* Out of space. */
  mon_memcpy(p, iPtr, iRemain);
  p += iRemain;
  remain--;
  return( p - pOrig );
}
