//
// TypeHandler.h
//
// $Id: //poco/1.3/Data/include/Poco/Data/TypeHandler.h#2 $
//
// Library: Data
// Package: DataCore
// Module:  TypeHandler
//
// Definition of the TypeHandler class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Data_TypeHandler_INCLUDED
#define Data_TypeHandler_INCLUDED


#include "Poco/Data/Data.h"
#include "Poco/Data/AbstractBinder.h"
#include "Poco/Data/AbstractExtractor.h"
#include "Poco/Data/AbstractPreparation.h"
#include "Poco/Tuple.h"
#include <cstddef>


namespace Poco {
namespace Data {


template <class T>
class TypeHandler
	/// Converts Rows to a Type and the other way around. Provide template specializations to support your own complex types.
	///
	/// Take as example the following (simplified) class:
	///    class Person
	///    {
	///    private:
	///        std::string _lastName;
	///        std::string _firstName;
	///        int         _age;
	///        [....] // public set/get methods, a default constructor, optional < operator (for set, multiset) or function operator (for map, multimap)
	///    };
	///
	/// The TypeHandler must provide a costum bind, size, prepare and extract method:
	///    
	///    template <>
	///    class TypeHandler<struct Person>
	///    {
	///    public:
	///        static std::size_t size()
	///        {
	///            return 3; // lastName + firstname + age occupy three columns
	///        }
	///    
	///        static void bind(std::size_t pos, const Person& obj, AbstractBinder* pBinder)
	///        {
	///            // the table is defined as Person (LastName VARCHAR(30), FirstName VARCHAR, Age INTEGER(3))
	///            // Note that we advance pos by the number of columns the datatype uses! For string/int this is one.
	///            poco_assert_dbg (pBinder != 0);
	///            TypeHandler<std::string>::bind(pos++, obj.getLastName(), pBinder);
	///            TypeHandler<std::string>::bind(pos++, obj.getFirstName(), pBinder);
	///            TypeHandler<int>::bind(pos++, obj.getAge(), pBinder);
	///        }
	///    
	///        static void prepare(std::size_t pos, const Person& obj, AbstractPreparation* pPrepare)
	///        {
	///            // the table is defined as Person (LastName VARCHAR(30), FirstName VARCHAR, Age INTEGER(3))
	///            poco_assert_dbg (pPrepare != 0);
	///            TypeHandler<std::string>::prepare(pos++, obj.getLastName(), pPrepare);
	///            TypeHandler<std::string>::prepare(pos++, obj.getFirstName(), pPrepare);
	///            TypeHandler<int>::prepare(pos++, obj.getAge(), pPrepare);
	///        }
	///    
	///        static void extract(std::size_t pos, Person& obj, const Person& defVal, AbstractExtractor* pExt)
	///        {
	///            // defVal is the default person we should use if we encunter NULL entries, so we take the individual fields
	///            // as defaults. You can do more complex checking, ie return defVal if only one single entry of the fields is null etc...
	///            poco_assert_dbg (pExt != 0);
	///            std::string lastName;
	///            std::string firstName;
	///            int age = 0;
	///            // the table is defined as Person (LastName VARCHAR(30), FirstName VARCHAR, Age INTEGER(3))
	///            TypeHandler<std::string>::extract(pos++, lastName, defVal.getLastName(), pExt);
	///            TypeHandler<std::string>::extract(pos++, firstName, defVal.getFirstName(), pExt);
	///            TypeHandler<int>::extract(pos++, age, defVal.getAge(), pExt);
	///            obj.setLastName(lastName);
	///            obj.setFirstName(firstName);
	///            obj.setAge(age);
	///        }
	///    };
	///
	/// Note that the TypeHandler template specialization must always be declared in the namespace Poco::Data.
	/// Apart from that no further work is needed. One can now use Person with into and use clauses.
{
public:
	static void bind(std::size_t pos, const T& obj, AbstractBinder* pBinder)
	{
		poco_assert_dbg (pBinder != 0);
		pBinder->bind(pos, obj);
	}

	static std::size_t size()
	{
		return 1;
	}

	static void extract(std::size_t pos, T& obj, const T& defVal, AbstractExtractor* pExt)
	{
		poco_assert_dbg (pExt != 0);
		if (!pExt->extract(pos, obj))
			obj = defVal;
	}

	static void prepare(std::size_t pos, const T& obj, AbstractPreparation* pPrepare)
	{
		poco_assert_dbg (pPrepare != 0);
		pPrepare->prepare(pos, obj);
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator = (const TypeHandler&);
};


/// Poco::Tuple TypeHandler specializations


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15, 
	class T16, 
	class T17, 
	class T18, 
	class T19>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18,T19> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18,T19> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18,T19> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
		pBinder->bind(pos++, tuple.template get<11>());
		pBinder->bind(pos++, tuple.template get<12>());
		pBinder->bind(pos++, tuple.template get<13>());
		pBinder->bind(pos++, tuple.template get<14>());
		pBinder->bind(pos++, tuple.template get<15>());
		pBinder->bind(pos++, tuple.template get<16>());
		pBinder->bind(pos++, tuple.template get<17>());
		pBinder->bind(pos++, tuple.template get<18>());
		pBinder->bind(pos++, tuple.template get<19>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
		pPrepare->prepare(pos++, tuple.template get<11>());
		pPrepare->prepare(pos++, tuple.template get<12>());
		pPrepare->prepare(pos++, tuple.template get<13>());
		pPrepare->prepare(pos++, tuple.template get<14>());
		pPrepare->prepare(pos++, tuple.template get<15>());
		pPrepare->prepare(pos++, tuple.template get<16>());
		pPrepare->prepare(pos++, tuple.template get<17>());
		pPrepare->prepare(pos++, tuple.template get<18>());
		pPrepare->prepare(pos++, tuple.template get<19>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18,T19>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
		if (!pExt->extract(pos++, tuple.template get<11>()))
			tuple.template set<11>(defVal.template get<11>());
		if (!pExt->extract(pos++, tuple.template get<12>()))
			tuple.template set<12>(defVal.template get<12>());
		if (!pExt->extract(pos++, tuple.template get<13>()))
			tuple.template set<13>(defVal.template get<13>());
		if (!pExt->extract(pos++, tuple.template get<14>()))
			tuple.template set<14>(defVal.template get<14>());
		if (!pExt->extract(pos++, tuple.template get<15>()))
			tuple.template set<15>(defVal.template get<15>());
		if (!pExt->extract(pos++, tuple.template get<16>()))
			tuple.template set<16>(defVal.template get<16>());
		if (!pExt->extract(pos++, tuple.template get<17>()))
			tuple.template set<17>(defVal.template get<17>());
		if (!pExt->extract(pos++, tuple.template get<18>()))
			tuple.template set<18>(defVal.template get<18>());
		if (!pExt->extract(pos++, tuple.template get<19>()))
			tuple.template set<19>(defVal.template get<19>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15, 
	class T16, 
	class T17, 
	class T18>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
		pBinder->bind(pos++, tuple.template get<11>());
		pBinder->bind(pos++, tuple.template get<12>());
		pBinder->bind(pos++, tuple.template get<13>());
		pBinder->bind(pos++, tuple.template get<14>());
		pBinder->bind(pos++, tuple.template get<15>());
		pBinder->bind(pos++, tuple.template get<16>());
		pBinder->bind(pos++, tuple.template get<17>());
		pBinder->bind(pos++, tuple.template get<18>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
		pPrepare->prepare(pos++, tuple.template get<11>());
		pPrepare->prepare(pos++, tuple.template get<12>());
		pPrepare->prepare(pos++, tuple.template get<13>());
		pPrepare->prepare(pos++, tuple.template get<14>());
		pPrepare->prepare(pos++, tuple.template get<15>());
		pPrepare->prepare(pos++, tuple.template get<16>());
		pPrepare->prepare(pos++, tuple.template get<17>());
		pPrepare->prepare(pos++, tuple.template get<18>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17,T18>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
		if (!pExt->extract(pos++, tuple.template get<11>()))
			tuple.template set<11>(defVal.template get<11>());
		if (!pExt->extract(pos++, tuple.template get<12>()))
			tuple.template set<12>(defVal.template get<12>());
		if (!pExt->extract(pos++, tuple.template get<13>()))
			tuple.template set<13>(defVal.template get<13>());
		if (!pExt->extract(pos++, tuple.template get<14>()))
			tuple.template set<14>(defVal.template get<14>());
		if (!pExt->extract(pos++, tuple.template get<15>()))
			tuple.template set<15>(defVal.template get<15>());
		if (!pExt->extract(pos++, tuple.template get<16>()))
			tuple.template set<16>(defVal.template get<16>());
		if (!pExt->extract(pos++, tuple.template get<17>()))
			tuple.template set<17>(defVal.template get<17>());
		if (!pExt->extract(pos++, tuple.template get<18>()))
			tuple.template set<18>(defVal.template get<18>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15, 
	class T16, 
	class T17>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
		pBinder->bind(pos++, tuple.template get<11>());
		pBinder->bind(pos++, tuple.template get<12>());
		pBinder->bind(pos++, tuple.template get<13>());
		pBinder->bind(pos++, tuple.template get<14>());
		pBinder->bind(pos++, tuple.template get<15>());
		pBinder->bind(pos++, tuple.template get<16>());
		pBinder->bind(pos++, tuple.template get<17>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
		pPrepare->prepare(pos++, tuple.template get<11>());
		pPrepare->prepare(pos++, tuple.template get<12>());
		pPrepare->prepare(pos++, tuple.template get<13>());
		pPrepare->prepare(pos++, tuple.template get<14>());
		pPrepare->prepare(pos++, tuple.template get<15>());
		pPrepare->prepare(pos++, tuple.template get<16>());
		pPrepare->prepare(pos++, tuple.template get<17>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16,T17>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
		if (!pExt->extract(pos++, tuple.template get<11>()))
			tuple.template set<11>(defVal.template get<11>());
		if (!pExt->extract(pos++, tuple.template get<12>()))
			tuple.template set<12>(defVal.template get<12>());
		if (!pExt->extract(pos++, tuple.template get<13>()))
			tuple.template set<13>(defVal.template get<13>());
		if (!pExt->extract(pos++, tuple.template get<14>()))
			tuple.template set<14>(defVal.template get<14>());
		if (!pExt->extract(pos++, tuple.template get<15>()))
			tuple.template set<15>(defVal.template get<15>());
		if (!pExt->extract(pos++, tuple.template get<16>()))
			tuple.template set<16>(defVal.template get<16>());
		if (!pExt->extract(pos++, tuple.template get<17>()))
			tuple.template set<17>(defVal.template get<17>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15, 
	class T16>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
		pBinder->bind(pos++, tuple.template get<11>());
		pBinder->bind(pos++, tuple.template get<12>());
		pBinder->bind(pos++, tuple.template get<13>());
		pBinder->bind(pos++, tuple.template get<14>());
		pBinder->bind(pos++, tuple.template get<15>());
		pBinder->bind(pos++, tuple.template get<16>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
		pPrepare->prepare(pos++, tuple.template get<11>());
		pPrepare->prepare(pos++, tuple.template get<12>());
		pPrepare->prepare(pos++, tuple.template get<13>());
		pPrepare->prepare(pos++, tuple.template get<14>());
		pPrepare->prepare(pos++, tuple.template get<15>());
		pPrepare->prepare(pos++, tuple.template get<16>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15,T16>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
		if (!pExt->extract(pos++, tuple.template get<11>()))
			tuple.template set<11>(defVal.template get<11>());
		if (!pExt->extract(pos++, tuple.template get<12>()))
			tuple.template set<12>(defVal.template get<12>());
		if (!pExt->extract(pos++, tuple.template get<13>()))
			tuple.template set<13>(defVal.template get<13>());
		if (!pExt->extract(pos++, tuple.template get<14>()))
			tuple.template set<14>(defVal.template get<14>());
		if (!pExt->extract(pos++, tuple.template get<15>()))
			tuple.template set<15>(defVal.template get<15>());
		if (!pExt->extract(pos++, tuple.template get<16>()))
			tuple.template set<16>(defVal.template get<16>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14, 
	class T15>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
		pBinder->bind(pos++, tuple.template get<11>());
		pBinder->bind(pos++, tuple.template get<12>());
		pBinder->bind(pos++, tuple.template get<13>());
		pBinder->bind(pos++, tuple.template get<14>());
		pBinder->bind(pos++, tuple.template get<15>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
		pPrepare->prepare(pos++, tuple.template get<11>());
		pPrepare->prepare(pos++, tuple.template get<12>());
		pPrepare->prepare(pos++, tuple.template get<13>());
		pPrepare->prepare(pos++, tuple.template get<14>());
		pPrepare->prepare(pos++, tuple.template get<15>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14,T15>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
		if (!pExt->extract(pos++, tuple.template get<11>()))
			tuple.template set<11>(defVal.template get<11>());
		if (!pExt->extract(pos++, tuple.template get<12>()))
			tuple.template set<12>(defVal.template get<12>());
		if (!pExt->extract(pos++, tuple.template get<13>()))
			tuple.template set<13>(defVal.template get<13>());
		if (!pExt->extract(pos++, tuple.template get<14>()))
			tuple.template set<14>(defVal.template get<14>());
		if (!pExt->extract(pos++, tuple.template get<15>()))
			tuple.template set<15>(defVal.template get<15>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13, 
	class T14>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
		pBinder->bind(pos++, tuple.template get<11>());
		pBinder->bind(pos++, tuple.template get<12>());
		pBinder->bind(pos++, tuple.template get<13>());
		pBinder->bind(pos++, tuple.template get<14>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
		pPrepare->prepare(pos++, tuple.template get<11>());
		pPrepare->prepare(pos++, tuple.template get<12>());
		pPrepare->prepare(pos++, tuple.template get<13>());
		pPrepare->prepare(pos++, tuple.template get<14>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13,T14>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
		if (!pExt->extract(pos++, tuple.template get<11>()))
			tuple.template set<11>(defVal.template get<11>());
		if (!pExt->extract(pos++, tuple.template get<12>()))
			tuple.template set<12>(defVal.template get<12>());
		if (!pExt->extract(pos++, tuple.template get<13>()))
			tuple.template set<13>(defVal.template get<13>());
		if (!pExt->extract(pos++, tuple.template get<14>()))
			tuple.template set<14>(defVal.template get<14>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12, 
	class T13>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
		pBinder->bind(pos++, tuple.template get<11>());
		pBinder->bind(pos++, tuple.template get<12>());
		pBinder->bind(pos++, tuple.template get<13>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
		pPrepare->prepare(pos++, tuple.template get<11>());
		pPrepare->prepare(pos++, tuple.template get<12>());
		pPrepare->prepare(pos++, tuple.template get<13>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,T13>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
		if (!pExt->extract(pos++, tuple.template get<11>()))
			tuple.template set<11>(defVal.template get<11>());
		if (!pExt->extract(pos++, tuple.template get<12>()))
			tuple.template set<12>(defVal.template get<12>());
		if (!pExt->extract(pos++, tuple.template get<13>()))
			tuple.template set<13>(defVal.template get<13>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11, 
	class T12>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
		pBinder->bind(pos++, tuple.template get<11>());
		pBinder->bind(pos++, tuple.template get<12>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
		pPrepare->prepare(pos++, tuple.template get<11>());
		pPrepare->prepare(pos++, tuple.template get<12>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
		if (!pExt->extract(pos++, tuple.template get<11>()))
			tuple.template set<11>(defVal.template get<11>());
		if (!pExt->extract(pos++, tuple.template get<12>()))
			tuple.template set<12>(defVal.template get<12>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10, 
	class T11>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
		pBinder->bind(pos++, tuple.template get<11>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
		pPrepare->prepare(pos++, tuple.template get<11>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
		if (!pExt->extract(pos++, tuple.template get<11>()))
			tuple.template set<11>(defVal.template get<11>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, 
	class T1, 
	class T2, 
	class T3, 
	class T4, 
	class T5, 
	class T6, 
	class T7, 
	class T8, 
	class T9, 
	class T10>
class TypeHandler<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
		pBinder->bind(pos++, tuple.template get<10>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
		pPrepare->prepare(pos++, tuple.template get<10>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
		if (!pExt->extract(pos++, tuple.template get<10>()))
			tuple.template set<10>(defVal.template get<10>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8, class T9>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
		pBinder->bind(pos++, tuple.template get<9>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
		pPrepare->prepare(pos++, tuple.template get<9>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
		if (!pExt->extract(pos++, tuple.template get<9>()))
			tuple.template set<9>(defVal.template get<9>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5, class T6, class T7, class T8>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
		pBinder->bind(pos++, tuple.template get<8>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
		pPrepare->prepare(pos++, tuple.template get<8>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, T8, NullTypeList>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
		if (!pExt->extract(pos++, tuple.template get<8>()))
			tuple.template set<8>(defVal.template get<8>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5, class T6, class T7>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
		pBinder->bind(pos++, tuple.template get<7>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
		pPrepare->prepare(pos++, tuple.template get<7>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, T7, NullTypeList>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
		if (!pExt->extract(pos++, tuple.template get<7>()))
			tuple.template set<7>(defVal.template get<7>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5, class T6>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
		pBinder->bind(pos++, tuple.template get<6>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
		pPrepare->prepare(pos++, tuple.template get<6>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, T1, T2, T3, T4, T5, T6, NullTypeList>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
		if (!pExt->extract(pos++, tuple.template get<6>()))
			tuple.template set<6>(defVal.template get<6>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4, class T5>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, T5, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, T5, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
		pBinder->bind(pos++, tuple.template get<5>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
		pPrepare->prepare(pos++, tuple.template get<5>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, T1, T2, T3, T4, T5, NullTypeList>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
		if (!pExt->extract(pos++, tuple.template get<5>()))
			tuple.template set<5>(defVal.template get<5>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3, class T4>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, T4, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, T4, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
		pBinder->bind(pos++, tuple.template get<4>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
		pPrepare->prepare(pos++, tuple.template get<4>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, T1, T2, T3, T4, NullTypeList>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
		if (!pExt->extract(pos++, tuple.template get<4>()))
			tuple.template set<4>(defVal.template get<4>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2, class T3>
class TypeHandler<Poco::Tuple<T0, T1, T2, T3, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, T3, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
		pBinder->bind(pos++, tuple.template get<3>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
		pPrepare->prepare(pos++, tuple.template get<3>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, T1, T2, T3, NullTypeList>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
		if (!pExt->extract(pos++, tuple.template get<3>()))
			tuple.template set<3>(defVal.template get<3>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1, class T2>
class TypeHandler<Poco::Tuple<T0, T1, T2, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, T2, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
		pBinder->bind(pos++, tuple.template get<2>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
		pPrepare->prepare(pos++, tuple.template get<2>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, T1, T2, NullTypeList>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
		if (!pExt->extract(pos++, tuple.template get<2>()))
			tuple.template set<2>(defVal.template get<2>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0, class T1>
class TypeHandler<Poco::Tuple<T0, T1, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, T1, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
		pBinder->bind(pos++, tuple.template get<1>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
		pPrepare->prepare(pos++, tuple.template get<1>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, T1, NullTypeList>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
		if (!pExt->extract(pos++, tuple.template get<1>()))
			tuple.template set<1>(defVal.template get<1>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


template <class T0>
class TypeHandler<Poco::Tuple<T0, NullTypeList> >
{
public:
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, NullTypeList> >::CONSTREFTYPE TupleConstRef;
	typedef typename Poco::TypeWrapper<Poco::Tuple<T0, NullTypeList> >::REFTYPE      TupleRef;

	static void bind(std::size_t pos, TupleConstRef tuple, AbstractBinder* pBinder)
	{
		poco_assert (pBinder != 0);
		pBinder->bind(pos++, tuple.template get<0>());
	}

	static void prepare(std::size_t pos, TupleConstRef tuple, AbstractPreparation* pPrepare)
	{
		poco_assert (pPrepare != 0);
		pPrepare->prepare(pos++, tuple.template get<0>());
	}

	static std::size_t size()
	{
		return Poco::Tuple<T0, NullTypeList>::length;
	}

	static void extract(std::size_t pos, TupleRef tuple, TupleConstRef defVal, 
		AbstractExtractor* pExt)
	{
		poco_assert (pExt != 0);
		if (!pExt->extract(pos++, tuple.template get<0>()))
			tuple.template set<0>(defVal.template get<0>());
	}

private:
	TypeHandler();
	~TypeHandler();
	TypeHandler(const TypeHandler&);
	TypeHandler& operator=(const TypeHandler&);
};


} } // namespace Poco::Data


#endif // Data_TypeHandler_INCLUDED
