/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxcalc-calcmass.h"
#include "polyxcalc-ui-recorder.h"



/* These accounting functions return 1 upon success or 0 upon failure.
 * Note that some functions must be called ONLY if a polymer
 * definition is currently available to the definition. See the
 * function caller.
 */


gint
polxyxcalc_calcmass_account_atoms (PxmCalcCtxt *calcctxt)
{
  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;

  GString *gs = NULL;
  
  gchar *help = NULL;

  gint count = 0;
  gint error = 1;
  

  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);

  

  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  

  entry = g_object_get_data (G_OBJECT (calculator_wnd), "atoms_count_entry");
  g_assert (entry != NULL);
  
  /* Get the text from the widget and convert it to a gint with 
   * checking that it is not overflow.
   */
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }
  
  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == libpolyxmass_globals_strtoi (help, &count, 10))
    {
      g_critical (_("%s@%d: failed converting '%s' to gint\n"),
	     __FILE__, __LINE__, help);
      
      g_string_free (gs, TRUE);

      return 0;
    }
    
  /* Now count should contain a genuine gint. Note that this can be
   * preceded by a '+' or a '-'.
   */
  if (count == 0)
    return 1;
  
  /* Get the text from the widget of atom symbol.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), "atoms_combo_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }

  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == 
      pxmchem_formula_account_mass (help, calcctxt->atomGPA,
				    count, calcctxt->masspair_temp))
    {
      g_critical (_("%s@%d: failed accounting formula '%s'\n"),
	     __FILE__, __LINE__, help);
      
      g_string_append_printf (gs, 
			      _("Failed accounting formula '%s'\n"), help);

      error = 0;
    }
  else
    g_string_append_printf (gs, 
			    _("Accounted atom: '%s'; count: '%d' ==>"
			      " mono: %f ; avg: %f\n"),
			    help, count, 
			    calcctxt->masspair_temp->mono, 
			    calcctxt->masspair_temp->avg);
  
  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
					   gs->str);
  
  g_string_free (gs, TRUE);
  
  return error;
}


gint
polxyxcalc_calcmass_account_actform (PxmCalcCtxt *calcctxt)
{
  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;

  GString *gs = NULL;
  
  gchar *help = NULL;

  gint count = 0;
  gint error = 1;


  g_assert (calcctxt != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);
  

  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  
  entry = g_object_get_data (G_OBJECT (calculator_wnd),
			     "actform_count_entry");
  g_assert (entry != NULL);
  
  /* Get the text from the widget and convert it to a gint with 
   * checking that it is not overflow.
   */
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }
  
  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == libpolyxmass_globals_strtoi (help, &count, 10))
    {
      g_critical (_("%s@%d: failed converting '%s' to gint\n"),
	     __FILE__, __LINE__, help);
      
      g_string_free (gs, TRUE);

      return 0;
    }
    
  /* Now count should contain a genuine gint. Note that this can be
   * preceded by a '+' or a '-'.
   */
  if (count == 0)
    return 1;
  
  /* Get the text from the widget of actform entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), "actform_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }

  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == 
      pxmchem_actform_account_mass (help, calcctxt->atomGPA,
				    count, calcctxt->masspair_temp))
    {
      g_critical (_("%s@%d: failed accounting actform '%s'\n"),
	     __FILE__, __LINE__, help);
      
      g_string_append_printf (gs, 
			      _("Failed accounting actform '%s'\n"), help);

      error = 0;
    }
  else
    g_string_append_printf (gs, 
			    _("Accounted actform: '%s'; count: '%d' ==>"
			      " mono: %f ; avg: %f\n"),
			    help, count, 
			    calcctxt->masspair_temp->mono, 
			    calcctxt->masspair_temp->avg);

  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
				    gs->str);
  
  g_string_free (gs, TRUE);
  
  return error;
}

gint
polxyxcalc_calcmass_account_monomers (PxmCalcCtxt *calcctxt)
{
  PxmPolchemdef *polchemdef = NULL;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;

  GString *gs = NULL;
  
  gchar *help = NULL;

  gint count = 0;
  gint error = 1;


  g_assert (calcctxt != NULL);
  g_assert (calcctxt->polchemdefctxt != NULL);
  polchemdef = calcctxt->polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);


  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  

  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "monomers_count_entry");
  g_assert (entry != NULL);
  
  /* Get the text from the widget and convert it to a gint with 
   * checking that it is not overflow.
   */
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }
  
  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == libpolyxmass_globals_strtoi (help, &count, 10))
    {
      g_critical (_("%s@%d: failed converting '%s' to gint\n"),
	     __FILE__, __LINE__, help);
      
      g_string_free (gs, TRUE);

      return 0;
    }
    
  /* Now count should contain a genuine gint. Note that this can be
   * preceded by a '+' or a '-'.
   */
  if (count == 0)
    return 1;
  
  /* Get the text from the widget of monomer entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "monomers_combo_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }

  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == 
      pxmchem_monomer_account_mass_by_name (help,
					    polchemdef->monomerGPA,
					    calcctxt->atomGPA,
					    count,
					    calcctxt->masspair_temp))
    {
      g_critical (_("%s@%d: failed accounting monomer '%s'\n"),
	     __FILE__, __LINE__, help);
      
      g_string_append_printf (gs, 
			      _("Failed accounting monomer '%s'\n"), help);

      error = 0;
    }
  else
    g_string_append_printf (gs, 
			    _("Accounted monomer: '%s'; count: '%d' ==>"
			      " mono: %f ; avg: %f\n"),
			    help, count, 
			    calcctxt->masspair_temp->mono, 
			    calcctxt->masspair_temp->avg);
  
  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt, gs->str);
  
  g_string_free (gs, TRUE);
  
  return error;
}


gint
polxyxcalc_calcmass_account_modifs (PxmCalcCtxt *calcctxt)
{
  PxmPolchemdef *polchemdef = NULL;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;

  GString *gs = NULL;
  
  gchar *help = NULL;

  gint count = 0;
  gint error = 1;



  g_assert (calcctxt != NULL);
  g_assert (calcctxt->polchemdefctxt != NULL);
  polchemdef = calcctxt->polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);
  


  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  

  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "modifications_count_entry");
  g_assert (entry != NULL);
  
  /* Get the text from the widget and convert it to a gint with 
   * checking that it is not overflow.
   */
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }
  
  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == libpolyxmass_globals_strtoi (help, &count, 10))
    {
      g_critical (_("%s@%d: failed converting '%s' to gint\n"),
	     __FILE__, __LINE__, help);
      
      g_string_free (gs, TRUE);

      return 0;
    }
    
  /* Now count should contain a genuine gint. Note that this can be
   * preceded by a '+' or a '-'.
   */
  if (count == 0)
    return 1;
  
  /* Get the text from the widget of monomer entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "modifications_combo_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }

  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == pxmchem_modif_account_mass_by_name (help,
						   polchemdef->modifGPA,
						   calcctxt->atomGPA,
						   count,
						   calcctxt->masspair_temp))
    {
      g_critical (_("%s@%d: failed accounting modif '%s'\n"),
	     __FILE__, __LINE__, help);
      
      g_string_append_printf (gs, 
			      _("Failed accounting modif '%s'\n"), help);

      error = 0;
    }
  else
    g_string_append_printf (gs, 
			    _("Accounted modif: '%s'; count: '%d' ==>"
			      " mono: %f ; avg: %f\n"),
			    help, count, 
			    calcctxt->masspair_temp->mono, 
			    calcctxt->masspair_temp->avg);
  
  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt, gs->str);
  
  g_string_free (gs, TRUE);
  
  return error;
}



gint
polxyxcalc_calcmass_account_sequence (PxmCalcCtxt *calcctxt)
{
  PxmPolchemdef *polchemdef = NULL;

  GtkWidget *calculator_wnd = NULL;
  GtkWidget *entry = NULL;

  GString *gs = NULL;
  
  PxmMonomer *monomer = NULL;
  
  GPtrArray *fillGPA = NULL;
  
  gchar *help = NULL;

  gint result = -1;
  gint count = 0;
  gint iter = 0;


  g_assert (calcctxt != NULL);
  g_assert (calcctxt->polchemdefctxt != NULL);
  polchemdef = calcctxt->polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  calculator_wnd = calcctxt->calculator_wnd;
  g_assert (calculator_wnd != NULL);


  
  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  

  entry = g_object_get_data (G_OBJECT (calculator_wnd), 
			     "sequence_count_entry");
  g_assert (entry != NULL);
  
  /* Get the text from the widget and convert it to a gint with 
   * checking that it is not overflow.
   */
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }
  
  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == libpolyxmass_globals_strtoi (help, &count, 10))
    {
      g_critical (_("%s@%d: failed converting '%s' to gint\n"),
	     __FILE__, __LINE__, help);
      
      g_string_free (gs, TRUE);

      return 0;
    }
    
  /* Now count should contain a genuine gint. Note that this can be
   * preceded by a '+' or a '-'.
   */
  if (count == 0)
    return 1;
  

  /* Get the text from the widget of sequence entry.
   */
  entry = g_object_get_data (G_OBJECT (calculator_wnd), "sequence_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help == NULL)
    {
      g_string_free (gs, TRUE);

      return 0;
    }

  if (strlen (help) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  /* Allocate the array into which the allocated monomers are going
   * to go.
   */
  fillGPA = g_ptr_array_new ();
  
  /* We now want to convert each monomer code, in the string taken from
   * the entry, to a monomer object into an array of monomers:
   */
  result = pxmchem_monomer_fill_array_from_string (fillGPA, help,
						   polchemdef->codelen,
						   polchemdef->monomerGPA,
						   TRUE /*empty first*/);
  
  if (result == -1)
    {
      g_critical (_("%s@%d: failed filling array of monomers with sequence"
	       " '%s'\n"),
	     __FILE__, __LINE__, help);
      
      g_string_append_printf (gs, 
			      _("Failed filling array of monomers"
				" with sequence '%s'\n"), help);
      
      pxmchem_monomer_GPA_free (fillGPA);
      
      /* Put the operation on queue to the recorder text.
       */
      polyxcalc_ui_recorder_wnd_append_record (calcctxt, 
					gs->str);
      
      g_string_free (gs, TRUE);
      
      return 0;
    }

  /* Now that we have an array of monomer codes correctly set up, we
   * can iterate in it and for each monomer found ask that its formula
   * be taken into account for the mass calculation.
   */
  for (iter = 0; iter < fillGPA->len; iter++)
    {
      monomer = g_ptr_array_index (fillGPA, iter);
      g_assert (monomer != NULL);

      if (FALSE == 
	  pxmchem_monomer_account_mass_by_name (monomer->name,
						polchemdef->monomerGPA,
						calcctxt->atomGPA,
						count,
						calcctxt->masspair_temp))
	{
	  g_critical (_("%s@%d: failed accounting monomer '%s'\n"),
		 __FILE__, __LINE__, monomer->name);
	  
	  g_string_append_printf (gs, 
				  _("Failed accounting monomer '%s'\n"), 
				  monomer->name);
	  
	  /* Put the operation on queue to the recorder text.
	   */
	  polyxcalc_ui_recorder_wnd_append_record (calcctxt, 
					    gs->str);
	  
	  /* Now that we do not need the array of monomers anymore, we can 
	   * free it.
	   */
	  pxmchem_monomer_GPA_free (fillGPA);

	  g_string_free (gs, TRUE);
	  
	  return 0;
	}
  
    }
  

  /* Now that we do not need the array of monomers anymore, we can 
   * free it.
   */
  pxmchem_monomer_GPA_free (fillGPA);


  /* At this point, all went OK, apparently, so give the recording
   * feedback.
   */
  g_string_append_printf (gs, 
			  _("Accounted sequence: '%s'; count: '%d' ==>"
			    " mono: %f ; avg: %f\n"),
			  help, count,
			  calcctxt->masspair_temp->mono, 
			  calcctxt->masspair_temp->avg);
  
  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt, gs->str);

  g_string_free (gs, TRUE);

  return 1;
}


