/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-acidobasic.h"
#include "polyxedit-ui-pka-ph-pi.h"



gboolean
polyxedit_acidobasic_setup_framework (PxmEditCtxt *editctxt)
{
  GtkWidget *window = NULL;
  
  PxmProp *prop_mnm = NULL;
  PxmProp *prop_mdf = NULL;
  

  g_assert (editctxt != NULL);


  /*
    We are asked to perform an acido-basic computation for a given
    editctxt. We first have to ensure that acido-basic data have been
    already read from the xml file, otherwise we must first ensure
    that we do this.

    When we first read the acidobasic.xml file from disk, in order to
    create the two mnm_pka_GPA (monomers) and mdf_pka_GPA
    (modifications) arrays of pKa data, we set property objects to the
    array of the editctxt object. Let's try to get them.
  */
  prop_mnm = libpolyxmass_prop_find_prop (editctxt->propGPA,
					  NULL,
					  NULL,
					  "MNM_PKA_GPA",
					  NULL,
					  PXM_UNDEF_PROP_CMP);

  prop_mdf = libpolyxmass_prop_find_prop (editctxt->propGPA,
					  NULL,
					  NULL,
					  "MDF_PKA_GPA",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
  
  
  if (prop_mnm == NULL || prop_mdf == NULL)
    {
      /* One or both of the arrays were not filled with read data,
	 which means that all we have to do is re-load data from disk.
      */
      if (FALSE == polyxedit_acidobasic_reload_data (editctxt))
	{
	  g_critical (_("%s@%d: failed reloading the acido-basic "
		   "data from xml file\n"),
		 __FILE__, __LINE__);
	  
	  return FALSE;
	}
    }
  
  /* 
     At this point, we have the acido-basic data in their respective
     arrays of monomers/modifications, we can go on with the setting
     up of the GUI window.
  */

  window = polyxedit_pka_ph_pi_wnd_setup (editctxt);
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed setting up the window for acido-basic"
	       "computations\n"),
	     __FILE__, __LINE__);
      
      return FALSE;
    }

  return TRUE;
}



gboolean
polyxedit_acidobasic_reload_data (PxmEditCtxt *editctxt)
{
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  

  PxmProp *prop_mnm = NULL;
  PxmProp *prop_mdf = NULL;
  
  gchar *file = NULL;
  

  g_assert (editctxt != NULL);

  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  


  prop_mnm = libpolyxmass_prop_find_prop (editctxt->propGPA,
					  NULL,
					  NULL,
					  "MNM_PKA_GPA",
					  NULL,
					  PXM_UNDEF_PROP_CMP);

  if (prop_mnm != NULL)
    pxmchem_monomer_GPA_empty ((GPtrArray *) prop_mnm->data);
  else
    {
      prop_mnm = libpolyxmass_prop_new ();
      libpolyxmass_prop_set_name (prop_mnm, "MNM_PKA_GPA");
      
      prop_mnm->data = (gpointer) g_ptr_array_new ();

      prop_mnm->custom_free = pxmchem_monomer_GPA_prop_free;
						  
      g_ptr_array_add (editctxt->propGPA, prop_mnm);
    }
  

  
  prop_mdf = libpolyxmass_prop_find_prop (editctxt->propGPA,
					  NULL,
					  NULL,
					  "MDF_PKA_GPA",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
  
  if (prop_mdf != NULL)
    pxmchem_modif_GPA_empty ((GPtrArray *) prop_mdf->data);
  else
    {
      prop_mdf = libpolyxmass_prop_new ();
      libpolyxmass_prop_set_name (prop_mdf, "MDF_PKA_GPA");
      
      prop_mdf->data = (gpointer) g_ptr_array_new ();
      
      prop_mdf->custom_free = pxmchem_modif_GPA_prop_free;
      
      g_ptr_array_add (editctxt->propGPA, prop_mdf);
    }
  
  /* At this point we know we have empty arrays. We can do the parsing
     work. Construct the polymer chemistry definition directory.
  */
  file = g_strdup_printf ("%s/acidobasic.xml", polchemdef->dir);
  
  if (FALSE == 
      libpolyxmass_acidobasic_render_xml_file (file,
					       (GPtrArray *) prop_mnm->data,
					       (GPtrArray *) prop_mdf->data,
					       NULL))
    {
      g_critical (_("%s@%d: failed reloading the acido-basic "
		   "data from xml file\n"),
	     __FILE__, __LINE__);

      /* Because we failed to fill the array with data, we have to
	 fully remove the array, so that the system knows there is no
	 such stuff available.
      */
      /* The monomer array.
       */
      g_ptr_array_remove (editctxt->propGPA, prop_mnm);
      pxmchem_monomer_GPA_prop_free (prop_mnm);
            
      /* The modif array.
       */
      g_ptr_array_remove (editctxt->propGPA, prop_mdf);
      pxmchem_monomer_GPA_prop_free (prop_mdf);
      
      return FALSE;
    }
  
  return TRUE;
}


PxmMonomer *
polyxedit_acidobasic_get_monomer (gchar *code, PxmEditCtxt *editctxt)
{
  /* We have to first ensure that we can get to the monomer array.
     Next we look for a monomer by 'code'.
  */
  PxmProp *prop_mnm = NULL;

  g_assert (code != NULL);
  g_assert (editctxt != NULL);
  

  prop_mnm = libpolyxmass_prop_find_prop (editctxt->propGPA,
					  NULL,
					  NULL,
					  "MNM_PKA_GPA",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
  
  g_assert (prop_mnm != NULL);
  
  return pxmchem_monomer_get_ptr_by_code (code,
					  (GPtrArray *) prop_mnm->data);
}



PxmModif *
polyxedit_acidobasic_get_modif (gchar *name, PxmEditCtxt *editctxt)
{
  /* We have to first ensure that we can get to the modif array.
     Next we look for a modif by 'name'.
  */
  PxmProp *prop_mdf = NULL;

  g_assert (name != NULL);
  g_assert (editctxt != NULL);
  

  prop_mdf = libpolyxmass_prop_find_prop (editctxt->propGPA,
					  NULL,
					  NULL,
					  "MDF_PKA_GPA",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
  
  g_assert (prop_mdf != NULL);
  
  return pxmchem_modif_get_ptr_by_name (name,
					(GPtrArray *) prop_mdf->data);
}



/* CALCULATIONS FUNCTIONS
 */
/* Return -1 if the chemgroup was not dealt with, which is the calling
   function should continue processing as we return. Return 0 if we
   had not processed the chemgroup but it was dealt with (as it was
   eligible for left_end modification accounting). Return > 0 to tell
   the calling function how many charge computations were performed.
*/
gint
polyxedit_acidobasic_chemgroup_account_plm_end_modif (PxmChemgroup *chemgroup,
						      PxmPlmChement chement,
						      gdouble ph,
						      PxmCalcOpt *calcopt,
						      gint mnm_index,
						      PxmEditCtxt *editctxt,
						      gdouble *neg_charge,
						      gdouble *pos_charge)
{
  gdouble charge = 0;

  gint count = 0;
  gint iter = 0;
  
  PxmPolymer *polymer = NULL;

  gchar *modif_name = NULL;

  PxmChemgroupRule *cgr = NULL;
  PxmChemgroup *chemgroup_mdf = NULL;

  PxmModif *modif_cg = NULL;
  PxmProp *prop_mdf = NULL;

  g_assert (chemgroup != NULL);
  
  g_assert (editctxt != NULL);
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  
  g_assert (neg_charge != NULL);
  g_assert (pos_charge != NULL);
  

  /* Once we have the name of the polymer end modif (left or right),
     we have to check, in the chemgroup that is passed as parameter,
     if there is a chemgrouprule item that has the same 'entity' as
     the one of interest (either "LE_PLM_MODIF" or "RE_PLM_MODIF") and
     the same name as the name of the polymer end modif, that we get
     in the 'modif_name' variable below.

     If we find a chemgrouprule ('cgr'), then we'll have to study it
     and see what action we'll have to perform. If no cgr is found, we
     perform no action, we just return without applying any
     modification.

     This is because we only have this way to identify un-ambiguously
     that the current chemgroup object is effectively modifiable.
  */
  
  /* Below, modif_name can be NULL, as it can be NULL if the polymer
     is not left_end-modified.
  */
  if (chement == PXMCHEMENT_PLM_LEFT_MODIF)
    {
      modif_name = pxmchem_polymer_get_left_end_modif_name (polymer);
      
      cgr = libpolyxmass_chemgroup_get_chemgrouprule_ptr (chemgroup,
							  "LE_PLM_MODIF",
							  modif_name);
    }
  else if (chement == PXMCHEMENT_PLM_RIGHT_MODIF)
    {
      modif_name = pxmchem_polymer_get_right_end_modif_name (polymer);
      
      cgr = libpolyxmass_chemgroup_get_chemgrouprule_ptr (chemgroup,
							  "RE_PLM_MODIF",
							  modif_name);
    }
  else
    g_assert_not_reached ();
  
  
  if (modif_name == NULL)
    {
      /* The polymer is not right_end-modified. However, we still have
	 to make sure that the chemgroup that we were parsing is
	 actually a chemgroup suitable for a right_end modif.  If this
	 chemgroup was actually suitable for a right_end modif, but it
	 is not effectively modified, that means that we have to
	 compute the charge for the non-modified chemgroup...
      */
      if (cgr != NULL)
	{
	  charge = 
	    libpolyxmass_acidobasic_compute_charge_ratio 
	    (chemgroup->pka, ph, chemgroup->acidcharged);
	  
	  /*
	  debug_printf (("charge is %lf "
			 "for chemgroup pka %lf\n",
			 charge, chemgroup->pka));
	  */

	  if (charge < 0)
	    {
	      (*neg_charge) += charge;
	    }
	  else if (charge > 0)
	    {
	      (*pos_charge) += charge;
	    }
	  else
	    ;
	  
	  /* And now that we did take the non-modified chemgroup into
	     account, we can return the count variable, and thus let the
	     calling function know that the chemgroup was indeed processed
	     and that it thus should proceed to another chemgroup.
	  */
	  return ++count;
	}
      else
	/* The current chemgroup was both NOT modified, and not
	   eligible for a right end modification. This means that we
	   do not have to treat it, and we return -1 so that the
	   caller function knows we did not do anything and that this
	   chemgroup should continue to undergo analysis without
	   skipping it.
	*/
	return -1;
    }
  
  if (cgr == NULL)
    {
      /* This chemgroup was not "designed" to receive any left_end
	 modification, so we have nothing to do with it, and we return
	 -1 to let the caller know that its processing should be
	 continued in the caller's function space.
       */
      return -1;
    }

  /* At this point, we know that the chemgroup we are analyzing is
     eligible for a right_end modification and that it is indeed
     modified with a correct modification. So we have a rule for
     it. Let's continue the analysis:

     Apparently, a PxmChemgroupRule object has member data matching
     the ones we were looking for. At this point we should know what
     action to take for this chemgroup.
  */
  if (cgr->outcome == PXM_CHEMGROUP_RULE_LOST)
    {
      /* 
	 We do not use the current chemgroup, because the
	 modification has abolished it. 
      */
    }
  else if (cgr->outcome == PXM_CHEMGROUP_RULE_PRESERVED)
    {
      /*
	OK, we truly have to take this chemgroup into account in our
	calculation of the charge that the corresponding chemical
	group could bring to the polymer/oligomer.
      */
      charge = 
	libpolyxmass_acidobasic_compute_charge_ratio 
	(chemgroup->pka, ph, chemgroup->acidcharged);
	  
      /*
      debug_printf (("charge is %lf "
		     "for chemgroup pka %lf\n",
		     charge, chemgroup->pka));
      */
  
      if (charge < 0)
	{
	  (*neg_charge) += charge;
	}
      else if (charge > 0)
	{
	  (*pos_charge) += charge;
	}
      else
	;
	  
      count++;
    }
  else
    g_assert_not_reached ();
  

  /* Whatever we should do with the left/right end monomer's
     chemgroup, we should take into account the modification itself
     that might have brought chemgroup(s) worth computing their
     intrinsic charges!
     
     Find a modif object in the special acidobasic array of modif
     objects, that has the same name as the modification with which
     the left/right end of the polymer is modified. We'll see what
     chemgroup(s) this modification brings to the polymer sequence.
  */
  modif_cg = polyxedit_acidobasic_get_modif (modif_name, editctxt);
  
  if (modif_cg != NULL)
    {
      for (iter = 0; iter < modif_cg->propGPA->len; iter++)
	{
	  prop_mdf = g_ptr_array_index (modif_cg->propGPA, iter);
	  
	  /* Make sure we are iterating into a "CHEMGROUP"-named prop
	     instance.
	  */
	  if (0 != strcmp ("CHEMGROUP", prop_mdf->name))
	    continue;
		  
	  chemgroup_mdf = (PxmChemgroup *) prop_mdf->data;
	  g_assert (chemgroup_mdf != NULL);
			  
	  charge = 
	    libpolyxmass_acidobasic_compute_charge_ratio (chemgroup_mdf->
							  pka, ph, 
							  chemgroup_mdf->
							  acidcharged);
			
	  /*  
	  debug_printf (("charge is '%lf' "
			 "for chemgroup_mdf pka '%lf'\n",
			 charge, chemgroup_mdf->pka));
	  */
  
	  if (charge < 0)
	    {
	      (*neg_charge) += charge;
	    }
	  else if (charge > 0)
	    {
	      (*pos_charge) += charge;
	    }
	  else
	    ;
	      
	  count++;
	}
    }
  /* End of 
     if (modif_cg != NULL)
  */

  return count;
}



gint
polyxedit_acidobasic_chemgroup_account_monomer_modif (PxmMonomer *monomer,
						      PxmChemgroup *chemgroup,
						      gdouble ph,
						      PxmCalcOpt *calcopt,
						      gint mnm_index,
						      PxmEditCtxt *editctxt,
						      gdouble *neg_charge,
						      gdouble *pos_charge)
{
  gdouble charge = 0;

  gint count = 0;
  gint iter = 0;

  gchar *modif_name = NULL;
    
  PxmPolymer *polymer = NULL;

  PxmChemgroupRule *cgr = NULL;
  PxmChemgroup *chemgroup_mdf = NULL;

  PxmModif *modif_cg = NULL;
  PxmProp *prop_mdf = NULL;

  g_assert (monomer != NULL);
  g_assert (chemgroup != NULL);
  
  g_assert (editctxt != NULL);
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  
  g_assert (neg_charge != NULL);
  g_assert (pos_charge != NULL);
  

  /* We get a pointer to a monomer and a chemgroup of such monomer. We
     should check if there is in the chemgroup a chemgrouprule
     mentioning both the name of the modification and at least the
     entity "MNM_MODIF". This way we'll establish if this monomer was
     eligible for an account of the charges lost/gained by being
     modified.

     See the left/right end modif functions above for detailed
     explanations.
  */
  modif_name = pxmchem_monomer_get_modif_name (monomer);
  
  cgr = libpolyxmass_chemgroup_get_chemgrouprule_ptr (chemgroup,
						      "MNM_MODIF",
						      modif_name);
  if (modif_name == NULL)
    {
      /* The monomer does not seem to be modified. However, we still
	 have to make sure that the chemgroup that we were parsing is
	 actually a chemgroup suitable for a modif.  If this chemgroup
	 was actually suitable for a monomer modif, but it is not
	 effectively modified, that means that we have to compute the
	 charge for the non-modified chemgroup...
      */
      if (cgr != NULL)
	{
	  charge = 
	    libpolyxmass_acidobasic_compute_charge_ratio 
	    (chemgroup->pka, ph, chemgroup->acidcharged);
	  
	  /*
	  debug_printf (("charge is %lf "
			 "for chemgroup pka %lf\n",
			 charge, chemgroup->pka));
	  */

	  if (charge < 0)
	    {
	      (*neg_charge) += charge;
	    }
	  else if (charge > 0)
	    {
	      (*pos_charge) += charge;
	    }
	  else
	    ;
	  
	  /* And now that we did take the non-modified chemgroup into
	     account, we can return the count variable, and thus let the
	     calling function know that the chemgroup was indeed processed
	     and that it thus should proceed to another chemgroup.
	  */
	  return ++count;
	}
      else
	/* The current chemgroup was both NOT modified, and not
	   eligible for a monomer modification. This means that we do
	   not have to treat it, and we return -1 so that the caller
	   function knows we did not do anything and that this
	   chemgroup should continue to undergo analysis without
	   skipping it.
	*/
	return -1;
    }
  
  if (cgr == NULL)
    {
      /* This chemgroup was not "designed" to receive any
	 modification, so we have nothing to do with it, and we return
	 -1 to let the caller know that its processing should be
	 continued in the caller's function space.
       */
      return -1;
    }

  /* At this point, we know that the chemgroup we are analyzing is
     eligible for a modification and that we have a rule for it. Let's
     continue the analysis:

     Apparently, a PxmChemgroupRule object has member data matching
     the ones we were looking for. At this point we should know what
     action to take for this chemgroup.
  */
  if (cgr->outcome == PXM_CHEMGROUP_RULE_LOST)
    {
      /* 
	 We do not use the current chemgroup, because the
	 modification has abolished it. 
      */
    }
  else if (cgr->outcome == PXM_CHEMGROUP_RULE_PRESERVED)
    {
      /*
	OK, we truly have to take this chemgroup into account in our
	calculation of the charge that the corresponding chemical
	group could bring to the polymer/oligomer.
      */
      charge = 
	libpolyxmass_acidobasic_compute_charge_ratio 
	(chemgroup->pka, ph, chemgroup->acidcharged);
	  
      /*
      debug_printf (("charge is %lf "
		     "for chemgroup pka %lf\n",
		     charge, chemgroup->pka));
      */
  
      if (charge < 0)
	{
	  (*neg_charge) += charge;
	}
      else if (charge > 0)
	{
	  (*pos_charge) += charge;
	}
      else
	;
	  
      count++;
    }
  else
    g_assert_not_reached ();
  

  /* Whatever we should do with this monomer's chemgroup, we should
     take into account the modification itself that might have brought
     chemgroup(s) worth computing their intrinsic charges!
     
     Find a modif object in the special acidobasic array of modif
     objects, that has the same name as the modification with which
     the left/right end of the polymer is modified. We'll see what
     chemgroup(s) this modification brings to the polymer sequence.
  */
  modif_cg = polyxedit_acidobasic_get_modif (modif_name, editctxt);
  
  if (modif_cg != NULL)
    {
      for (iter = 0; iter < modif_cg->propGPA->len; iter++)
	{
	  prop_mdf = g_ptr_array_index (modif_cg->propGPA, iter);
	  
	  /* Make sure we are iterating into a "CHEMGROUP"-named prop
	     instance.
	  */
	  if (0 != strcmp ("CHEMGROUP", prop_mdf->name))
	    continue;
		  
	  chemgroup_mdf = (PxmChemgroup *) prop_mdf->data;
	  g_assert (chemgroup_mdf != NULL);
			  
	  charge = 
	    libpolyxmass_acidobasic_compute_charge_ratio (chemgroup_mdf->
							  pka, ph, 
							  chemgroup_mdf->
							  acidcharged);
			
	  /*  
	  debug_printf (("charge is '%lf' "
			 "for chemgroup_mdf pka '%lf'\n",
			 charge, chemgroup_mdf->pka));
	  */
  
	  if (charge < 0)
	    {
	      (*neg_charge) += charge;
	    }
	  else if (charge > 0)
	    {
	      (*pos_charge) += charge;
	    }
	  else
	    ;
	      
	  count++;
	}
    }
  /* End of 
     if (modif_cg != NULL)
  */

  return count;
}



gint
polyxedit_acidobasic_polseq_compute_net_charge (PxmEditCtxt *editctxt,
						gdouble ph,
						PxmCalcOpt *calcopt,
						gdouble *negative_charges,
						gdouble *positive_charges)
{
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolymer *polymer = NULL;
 

  PxmMonomer *monomer_seq = NULL;
  PxmMonomer *monomer_cg = NULL;


  PxmChemgroup *chemgroup_mnm = NULL;

  
  PxmProp *prop_mnm = NULL;
  
  gint iter = 0;
  gint jter = 0;
  gint result = 0;
  gint processed_chemgroups = 0;

  gdouble charge = 0;

  gboolean ignore_pka = FALSE;
    
  gchar *modif_name = NULL;
  


  g_assert (editctxt != NULL);
  
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  
  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  g_assert (calcopt != NULL);
  
  g_assert (negative_charges != NULL);
  g_assert (positive_charges != NULL);
  


  /*
    What we have to take into consideration is:

    - does the sequence interval comprise the left end monomer (index
    0), and if so, are we going to take into account the left end
    modif.

    - same for the right end monomer.

    - should we take into account the monomer modifications.
  */
  
  for (iter = calcopt->start_idx; iter < calcopt->end_idx ; iter++)
    {
      ignore_pka = FALSE;
      
      monomer_seq = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (monomer_seq != NULL);
      
      /* Now get the monomer from the array of monomer instances
	 related to chemgroup data.
      */
      monomer_cg = polyxedit_acidobasic_get_monomer (monomer_seq->code,
						     editctxt);
      
      if (monomer_cg == NULL)
	{
	  g_error (_("%s@%d: Found no corresponding monomer for code '%s'"
	       " in the array of chemgroup-containing monomers\n"),
	     __FILE__, __LINE__, monomer_seq->code);
	  
	  continue;
	}
      
      /* Now that we have the monomer containing the chemgroup data as
	 prop objects enshrined in the propGPA array of prop
	 instances, we can start working on it.
      */
      for (jter = 0; jter < monomer_cg->propGPA->len; jter++)
	{
	  prop_mnm = g_ptr_array_index (monomer_cg->propGPA, jter);
	  
	  /* Make sure we are iterating into a "CHEMGROUP"-named prop
	     instance.
	  */
	  if (0 != strcmp ("CHEMGROUP", prop_mnm->name))
	    continue;
	  
	  chemgroup_mnm = (PxmChemgroup *) prop_mnm->data;
	  
	  g_assert (chemgroup_mnm != NULL);
	  
	  /* 
	     OK we now have our chemgroup object. Let's start working
	     on it.
	  */
	  
	  /* 
	     Check if its PxmChemgroupPolRule makes this chemgroup
	     eligible for treatment or not depending on the position
	     of the current monomer in the polymer/oligomer sequence.
	  */
	  if (chemgroup_mnm->polrule & PXM_CHEMGROUP_GROUP_LEFT_TRAPPED)
	    {
	      /* The chemical group being dealt with is trapped when a
		 polymerization occurs at its left end. Thus, if this
		 monomer is not the left_end monomer, this chemgroup
		 must not be processed, as the group has necessarily
		 entered a chemical bond with the monomer located at
		 its left side.
	      */
	      if (iter > calcopt->start_idx)
		continue;
	    }
	  
	  if (chemgroup_mnm->polrule & PXM_CHEMGROUP_GROUP_RIGHT_TRAPPED)
	    {
	      /* The chemical group being dealt with is trapped when a
		 polymerization occurs at its right end. Thus, if this
		 monomer is not the right_end monomer, this chemgroup
		 must not be processed, as the group has necessarily
		 entered a chemical bond with the monomer located at
		 its right side.
	      */
	      if (iter < calcopt->end_idx - 1)
		continue;
	    }
	  

	  /* ******* POLYMER PERMANENT MODIFICATIONS ********
	   */
	  if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF
	      && iter == calcopt->start_idx)
	    {
	      result =
		polyxedit_acidobasic_chemgroup_account_plm_end_modif 
		(chemgroup_mnm, PXMCHEMENT_PLM_LEFT_MODIF,
		 ph, calcopt, iter, editctxt, 
		 negative_charges, positive_charges);
	      
	      /*
		debug_printf (("PLM_LEFT_MODIF new negative_charges is %lf"
		"and new positive_charges is %lf\n"
		"after calculation for "
		"left_end chemgroup_mnm pka %lf\n",
		*negative_charges,
		*positive_charges, 
		chemgroup_mnm->pka));
	      */

	      if (result >= 0)
		{
		  /* Evidently the computation was performed OK, which
		     means that we can continue to the next chemgroup
		     as the current one has been dealt with.
		  */
		  processed_chemgroups += result;
		  
		  continue;
		}
	    }

	  if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF
	      && iter == calcopt->end_idx - 1)
	    {
	      result = 
		polyxedit_acidobasic_chemgroup_account_plm_end_modif 
		(chemgroup_mnm, PXMCHEMENT_PLM_RIGHT_MODIF,
		 ph, calcopt, iter, editctxt, 
		 negative_charges, positive_charges);
	      
	      /*
		debug_printf (("PLM_RIGHT_MODIF new negative_charges is %lf"
		"and new positive_charges is %lf\n"
		"after calculation for "
		"right_end chemgroup_mnm pka %lf\n",
		*negative_charges,
		*positive_charges, 
		chemgroup_mnm->pka));
	      */

	      if (result >= 0)
		{
		  /* Evidently the computation was performed OK, which
		     means that we can continue to the next chemgroup
		     as the current one has been dealt with.
		  */
		  processed_chemgroups += result;
		  
		  continue;
		}
	    }
	  

	  /* ******* MONOMER MODIFICATIONS ********
	   */

	  /* We only bother with modifications in monomers if the
	     caller requires that modifications be taken into account.
	  */
	  if (calcopt->mnm_chement & PXMCHEMENT_MNM_MODIF)
	    {
	      modif_name = pxmchem_monomer_get_modif_name (monomer_seq);
	      
	      if (modif_name != NULL)
		{
		  result =
		    polyxedit_acidobasic_chemgroup_account_monomer_modif 
		    (monomer_seq, chemgroup_mnm, ph, calcopt, iter, editctxt, 
		     negative_charges, positive_charges);
		  
		  /*
		    debug_printf (("MNM_MODIF new negative_charges is %lf"
		    "and new positive_charges is %lf\n"
		    "after calculation for "
		    "monomer modif chemgroup_mnm pka %lf\n",
		    *negative_charges,
		    *positive_charges, 
		    chemgroup_mnm->pka));
		  */

		  if (result >= 0)
		    {
		      /* Evidently the computation was performed OK, which
			 means that we can continue to the next chemgroup
			 as the current one has been dealt with.
		      */
		      processed_chemgroups += result;
		      
		      continue;
		    }
		}
	    }
	  /* End of
	     if (calcopt->mnm_chement & PXMCHEMENT_MNM_MODIF)
	  */
	  
	  /* At this point, if we are still here, that means that we have
	     to calculate the charge for the current chemgroup, as it has
	     not been taken into account before.
	  */
	  charge = 
	    libpolyxmass_acidobasic_compute_charge_ratio 
	    (chemgroup_mnm->pka, ph, chemgroup_mnm->acidcharged);
      
	  /*
	    debug_printf (("charge is %lf "
	    "for chemgroup_mnm pka %lf\n",
	    charge, chemgroup_mnm->pka));
	  */

	  if (charge < 0)
	    {
	      *negative_charges += charge;
	    }
	  else if (charge > 0)
	    {
	      *positive_charges += charge;
	    }
	  else
	    ;
	  
	  processed_chemgroups++;
	}
      /* End of
	 for (jter = 0; jter < monomer_cg->propGPA->len; jter++)
      */
    }
  /* End of 
     for (iter = calcopt->start_idx; iter < calcopt->end_idx ; iter++)
  */
  

  /* 
     At this point we have finished iterating through all the monomers
     of the polymer/oligomer sequence. We can return the number of
     processed chemical groups.
  */
  
  return processed_chemgroups ;
}
