/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxedit-ui-seqed-kbd.h"
#include "polyxedit-ui-seqed-window-menu.h"
#include "polyxedit-ui-seqed-canvas-menu.h"
#include "polyxedit-ui-seqed-clipboard.h"
#include "polyxedit-monicon.h"
#include "polyxedit-rendering.h"
#include "polyxedit-ui-masses-display-wnd.h"
#include "polyxedit-editctxt.h"
#include "polyxedit-ui-self-speak.h"
#include "polyxmass-polchemdefctxt.h"


#define LEFT_MARGIN 64

#define WIN_WIDTH 400
#define WIN_HEIGHT 800


#define MONICON_SIZE 32
#define SEL_POLYGON_LINE_WIDTH 3.0


GtkWidget *
polyxedit_seqed_wnd_setup (PxmEditCtxt *editctxt)
{
  GtkWidget *widget = NULL;
  GtkWidget *hscale = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *basename = NULL;
  gchar *help = NULL;

  GdkColor green_color;
  GdkColor red_color;
  GdkColormap *colormap;

  gboolean success = FALSE;

  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  


  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  

  gui_file = 
    g_strdup_printf ("%s/polyxedit-seqeditor.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "polseq_editor_wnd", 
		       PACKAGE);
  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed loading the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  seqeditorctxt->sequence_editor_wnd = 
    glade_xml_get_widget (xml, "polseq_editor_wnd");
  
  if (seqeditorctxt->sequence_editor_wnd == NULL)
    {
      g_critical (_("%s@%d: failed creating the sequence editor window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (seqeditorctxt->sequence_editor_wnd),
		      "delete_event",
		    G_CALLBACK 
		      (polyxedit_seqed_wnd_delete_event), 
		      editctxt);

  g_signal_connect (G_OBJECT (seqeditorctxt->sequence_editor_wnd),
		      "focus_in_event",
		      G_CALLBACK  
		      (polyxedit_seqed_wnd_focus_in_event),
		      editctxt);


  /* The sequence editor window has its window menu in a menubar :
   */
  widget = polyxedit_seqed_window_menubar_setup (xml, editctxt);
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "seqed_menubar", widget);


  /* The canvas has its context menu that is popped up when
     the third mouse button is clicked. But we have to set it up 
     after we have set up the sequence_canvas. See below, then.
  */



  /* The frame where the polymer sequence data are displayed:
   */
  widget = glade_xml_get_widget (xml, "sequence_data_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "sequence_data_frame", widget);

  /* The vertical box where the polymer sequence data widgets are packed, 
     so that we make them invisible in one go.
  */
  widget = glade_xml_get_widget (xml, "polymer_sequence_data_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "polymer_sequence_data_vbox", widget);
  
  /* The button in the polymer sequence data frame, thanks to which we
     can hide/show the sequence data.
  */
  widget = glade_xml_get_widget (xml, "polymer_sequence_data_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "polymer_sequence_data_button", widget);

  /* Let's connect the button to a "clicked" signal.
  */
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_sequence_data_button_clicked), 
		    editctxt);
  
  /* The horizontal box where the editing feedback data widgets are
     packed, so that we make them invisible in one go.
  */
  widget = glade_xml_get_widget (xml, "editing_feedback_hbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "editing_feedback_hbox", widget);
  
  /* The button in the feedback data frame, thanks to which we
     can hide/show the feedback data.
  */
  widget = glade_xml_get_widget (xml, "editing_feedback_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "editing_feedback_button", widget);

  /* Let's connect the button to a "clicked" signal.
  */
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_editing_feedback_button_clicked), 
		    editctxt);
  
  /* Set the name of the protein to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "sequence_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "sequence_entry", widget);

  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->name);

  /* Set the title of the window itself:
   */
  basename = g_path_get_basename (editctxt->polymer->plminfo->file);
  help = g_strdup_printf ("polyxedit: %s", basename);
  gtk_window_set_title (GTK_WINDOW (seqeditorctxt->sequence_editor_wnd), 
			help);
  g_free (help);
  g_free (basename);
  
  /* The label that says if the sequence is modified or not.
   */
  widget = glade_xml_get_widget (xml, "polseq_status_label");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "polseq_status_label",
		     widget);

  /* We are going to have a flag in which we'll draw different colors
     depending on the currently "moused" monomer status (prop, note, modif).
  */
  widget = glade_xml_get_widget (xml, "flag_hbox");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "flag_hbox",
		     widget);
  g_assert (widget != NULL);

  /* We are going to connect flag drawing area so that whenever it
     needs redrawing we can deal with it.
  */
  g_signal_connect (G_OBJECT (widget), "expose_event",
		    G_CALLBACK (polyxedit_seqed_monomer_flag_expose_event), 
		    editctxt);

  widget = glade_xml_get_widget (xml, "prop_flag_drawingarea");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "prop_flag_drawingarea",
		     widget);
  g_assert (widget != NULL);

  widget = glade_xml_get_widget (xml, "note_flag_drawingarea");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "note_flag_drawingarea",
		     widget);
  g_assert (widget != NULL);

  widget = glade_xml_get_widget (xml, "modif_flag_drawingarea");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "modif_flag_drawingarea",
		     widget);
  g_assert (widget != NULL);

  
  /* For the moment the user cannot point any monomer, since the window
     is not displayed ! so all we want is to draw the background of the
     drawing area and also the black mask that will separate the
     different compartments.
  */
  polyxedit_seqed_wnd_setup_monomer_flag (editctxt);
  


  /* When created, this window contains a non modified polymer
   * sequence, we say the polymer is not modified.
   */
  polyxedit_seqed_wnd_set_polymer_modified (editctxt, FALSE);

  /* Set the identity number (actually the editctxt pointer) to 
   * its GtkEntry (this is useful when cleavages are made to identify
   * unambiguously the polymer from which results are displayed).
   */
  widget = glade_xml_get_widget (xml, "identity_entry");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "identity_entry",
		     widget);
  help = g_strdup_printf ("%p", editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);


  /* There is a frame where the left and right ends' modifications are
     displayed. That frame can be hidden/shown by clicking onto the button
     that contains its title. Like the polymer sequence data above.
  */

  widget = glade_xml_get_widget (xml, "left_right_ends_modif_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "left_right_ends_modif_button", widget);

  /* Let's connect the button to a "clicked" signal.
  */
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_lr_end_modif_button_clicked), 
		    editctxt);

  widget = glade_xml_get_widget (xml, "left_right_ends_modif_hbox");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "left_right_ends_modif_hbox",
		     widget);

  widget = glade_xml_get_widget (xml, "left_end_modif_entry");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "left_end_modif_entry",
		     widget);

  widget = glade_xml_get_widget (xml, "right_end_modif_entry");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "right_end_modif_entry",
		     widget);

  /* And finally the button that let's the user modify the modifications.
   */
  widget = glade_xml_get_widget (xml, 
				 "polymer_left_right_end_modifs_edit_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "polymer_left_right_end_modifs_edit_button", widget);
  
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxedit_seqed_window_menu_chemistry_modifs_polymer), 
		    editctxt);

  
  seqeditorctxt->messages_entry = 
    glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "messages_entry",
		     seqeditorctxt->messages_entry);
  

  /* Prepare the colors that we'll use to set the foreground to green
     and red respectively for the GtkEntry widgets below.
  */
  
  green_color.red = 5000;
  green_color.blue = 10000;
  green_color.green = 35000;
  
  red_color.red = 55000;
  red_color.blue = 10000;
  red_color.green = 5000;
  

  seqeditorctxt->elab_code_entry = glade_xml_get_widget (xml, 
							 "elab_code_entry");
  
  colormap = gdk_drawable_get_colormap (seqeditorctxt->
					elab_code_entry->window);

  gdk_colormap_alloc_colors (colormap,
			     &green_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);
  
  gtk_widget_modify_text (GTK_WIDGET (seqeditorctxt->elab_code_entry),
			GTK_STATE_NORMAL,
			&green_color);


  seqeditorctxt->error_code_entry = glade_xml_get_widget (xml, 
							  "error_code_entry");
  
  colormap = gdk_drawable_get_colormap (seqeditorctxt->
					error_code_entry->window);

  gdk_colormap_alloc_colors (colormap,
			     &red_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);
  

  gtk_widget_modify_text (GTK_WIDGET (seqeditorctxt->error_code_entry),
			GTK_STATE_NORMAL,
			&red_color);


  seqeditorctxt->monomer_pos_label = 
    glade_xml_get_widget (xml, "monomer_position_label");
  
  /* The hscale widget by which the user is to set the size of the 
     monicons to display the polymer sequence.
  */
  hscale = 
    glade_xml_get_widget (xml, "monicon_size_hscale");
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
		     "monicon_size_hscale",
		     hscale);

  gtk_range_set_range (GTK_RANGE (hscale), 15, 200);

  /* We do not need decimals for a value that represents the number
     of pixels!
  */
  gtk_scale_set_digits (GTK_SCALE (hscale), 0);
  

  seqeditorctxt->canvas_scroll_wnd = 
    glade_xml_get_widget (xml, "scrolled_window");
  g_assert (seqeditorctxt->canvas_scroll_wnd != NULL);
  
  if (seqeditorctxt->canvas_scroll_wnd == NULL)
    {
      g_critical (_("%s@%d: failed creating the scrolled window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW 
				  (seqeditorctxt->canvas_scroll_wnd),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);

  /* Because we have to know at each instant what is the size of 
   * the scroll window into which the canvas displays the monicons
   * (so that we can know where each monicon sits), we have to 
   * connect the size_allocate signal, which is emitted each time 
   * the window changes its geometry.
   */
  g_signal_connect (G_OBJECT (seqeditorctxt->canvas_scroll_wnd), 
		    "size_allocate",
		    G_CALLBACK (polyxedit_seqed_sw_size_allocate), 
		    editctxt);
  
  /* after the window has been resized, it is necessary to check that
   * the cursor and (if any) the selection be visible in the seqeditor
   * window. We cannot do all this stuff in
   * polyxedit_seqed_sw_size_allocate (), nor in
   * polyxedit_seqed_wnd_event () because in these functions the new
   * size of the window is not known. So we have to deal with these
   * calculations through a call to a function that is systematically
   * called after the first handlers.
   */
  g_signal_connect_after 
    (G_OBJECT (seqeditorctxt->canvas_scroll_wnd),
     "size_allocate", 
     G_CALLBACK (polyxedit_seqed_wnd_ensure_select_cursor_visible), 
     editctxt);
  

  /* Create the canvas in which sequence monicon's icons will be
   * displayed and configure the scroll region
   */
  seqeditorctxt->sequence_canvas = glade_xml_get_widget (xml, "canvas");

  if (seqeditorctxt->sequence_canvas == NULL)
    {
      g_critical (_("%s@%d: failed creating the canvas\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* The canvas has its context menu that is popped up when
     the third mouse button is clicked.
  */
  widget = polyxedit_seqed_canvas_menu_setup (editctxt);
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (seqeditorctxt->sequence_canvas), 
		     "canvas_menu",
		     widget);


  /* IN THE CANVAS COPY/CUT/PASTE/SELECTION stuff.
   */
  gtk_selection_add_target (GTK_WIDGET (seqeditorctxt->sequence_canvas),
			    GDK_SELECTION_PRIMARY,
			    GDK_SELECTION_TYPE_STRING,
			    1);
  gtk_selection_add_target (GTK_WIDGET (seqeditorctxt->sequence_canvas),
			    GDK_SELECTION_CLIPBOARD,
			    GDK_SELECTION_TYPE_STRING,
			    1);
  
  g_signal_connect (G_OBJECT (seqeditorctxt->sequence_canvas),
		    "selection_get",
		    G_CALLBACK (polyxedit_seqed_clipboard_selection_get),
		    editctxt);
  
  g_signal_connect (G_OBJECT (seqeditorctxt->sequence_canvas),
		    "selection_received",
		    G_CALLBACK (polyxedit_seqed_clipboard_selection_received),
		    editctxt);
  
  g_signal_connect (G_OBJECT (seqeditorctxt->sequence_canvas),
		    "selection_clear_event",
		    G_CALLBACK (polyxedit_seqed_clipboard_selection_clear),
		    editctxt);


  seqeditorctxt->main_canvas_group = 
    gnome_canvas_root (GNOME_CANVAS (seqeditorctxt->sequence_canvas));
  
  /* Set the left margin of the polymer sequence display inside the
   * scroll window, so that there is enough space for the labels
   *indicating the position of the first monicon in the row.
   */
  seqeditorctxt->left_margin = LEFT_MARGIN;
  
  /* Size / scrolling stuff for the main seqeditor window and canvas.
   */
  gtk_window_set_default_size (GTK_WINDOW (seqeditorctxt->
					   sequence_editor_wnd),
			       WIN_WIDTH, WIN_HEIGHT);
  
  gnome_canvas_set_scroll_region (GNOME_CANVAS (seqeditorctxt->
						sequence_canvas), 0, 0,
				  WIN_WIDTH, WIN_HEIGHT);
  
  /* Editing of the sequence, the keyboard handling here goes
   * directly to the canvas
   */
  g_signal_connect (G_OBJECT (seqeditorctxt->sequence_editor_wnd),
		      "key-press-event",
		      G_CALLBACK (polyxedit_seqed_kbd_key_press_event),
		      editctxt);
		      
  g_signal_connect (G_OBJECT (seqeditorctxt->sequence_editor_wnd),
		      "key-release-event",
		      G_CALLBACK (polyxedit_seqed_kbd_key_release_event), 
		      editctxt);


  /* The great many signals that we want to handle are relative to 
   * the canvas, so set a handler for the most general event category:
   * "event".
   */
  g_signal_connect (G_OBJECT (seqeditorctxt->sequence_canvas), "event",
		    G_CALLBACK (polyxedit_seqed_wnd_canvas_event), 
		    editctxt); 
  
  /* We have to tell GTK+ that we want to get these messages :
   */
  gtk_widget_add_events (seqeditorctxt->sequence_editor_wnd, 
			 GDK_KEY_RELEASE_MASK);

  /* We need to set these variables, because first display of the sequence
   * considers these variables as containing the effective window width
   * and icon width
   */
  seqeditorctxt->canvas_scroll_wnd_width = WIN_WIDTH;
  seqeditorctxt->monicon_size = MONICON_SIZE;

  /* Set the hscale widget to the value of the size of the monicons.
   */
  gtk_range_set_value (GTK_RANGE (hscale), 
		       (gdouble) seqeditorctxt->monicon_size);

  g_signal_connect (G_OBJECT (hscale),
		    "value-changed",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_monicon_size_value_changed),
		    editctxt);


  /* Now that the window has its dimensions, we can set the vertical
   * and horizontal GtkAdjustments to their holders. This way we'll
   * gain access to these very easily throughout the program.
   */
  seqeditorctxt->canvas_scroll_wnd_adjust_v = 
    gtk_scrolled_window_get_vadjustment 
    (GTK_SCROLLED_WINDOW (seqeditorctxt->canvas_scroll_wnd));
  /* Set the increment to the size of the monicons.
   */
  seqeditorctxt->canvas_scroll_wnd_adjust_v->step_increment = 
    seqeditorctxt->monicon_size;
  
  seqeditorctxt->canvas_scroll_wnd_adjust_h = 
    gtk_scrolled_window_get_hadjustment 
    (GTK_SCROLLED_WINDOW (seqeditorctxt->canvas_scroll_wnd));
  /* Set the increment to the size of the monicons.
   */
  seqeditorctxt->canvas_scroll_wnd_adjust_h->step_increment = 
    seqeditorctxt->monicon_size;



  /* Finally we can show the whole sequence editor.
   */
  gtk_widget_show_all (seqeditorctxt->sequence_editor_wnd);

  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));

  return seqeditorctxt->sequence_editor_wnd;
}


gint
polyxedit_seqed_wnd_render_polseq (PxmEditCtxt *editctxt,
				   gboolean rerender)
{
  gint count = 0;

  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  

  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  

  /* If the polymer sequence has left/right end modifications, then
   * we should mention this in the window gadgets ad hoc.
   */

  /* Both the left and right end modifs.
   */
  polyxedit_seqed_wnd_update_LR_end_modif_state (editctxt);
  
  /* The rendering of the polymer sequence is a multi-step process:
   */
  if (-1 == polyxedit_seqed_wnd_render_polseq_monicons (editctxt,
							FALSE))
    {
      g_warning (_("%s@%d: failed rendering monicons\n"),
		 __FILE__, __LINE__);
      
      return -1;
    }
      
  /* Rendering of the cursor cannot fail, or an error is issued with
   * following stop of the program.
   */
  polyxedit_rendering_cursor_monicon_render (editctxt, rerender);

  /* We are drawing the sequence for the very first time; we have to
     initialize the seqeditorctxt->selection_rect.x1 and
     seqeditorctxt->selection_rect.y1 to sane values, so that they
     correspond to the position of the cursor which is draw at
     position 0 the first time it is drawn.

     This is especially important, because if the user opens a
     sequence. The cursor is drawn left of the very first
     monomer. Now, the user clicks out of the sequence rendering area,
     which is that the mouse event handling function does not
     acknowledge that click. Thus, the sequence cursor (or point in
     Emacs terminology) does not change in any way. Now the user drags
     the mouse (button still depressed) to the sequence rendering
     area. The mouse event handling functions do take into account
     that movement. And if the user releases the mouse button, the
     selection polygon is going to be drawn. But if the location of
     the first point of the selection is not defined, CRASH. Thus this
     small calculation below. The coordinates that we ask are for the
     two '*' positions here (north west and south west):

     *----+
     |    |
     |    |
     *----+
     
  */
  polyxedit_seqed_wnd_monicon_get_pixel_coord 
    (0, 
     editctxt->seqeditorctxt, 
     &(seqeditorctxt->selection_rect), 
     COORDSYS_NW, COORDSYS_SW);

  return count;
}

  
gboolean
polyxedit_seqed_wnd_update_LR_end_modif_state (PxmEditCtxt *editctxt)
{
  GtkWidget *widget = NULL;
  
  PxmProp *prop = NULL;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  

  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  

  /* The left end modif.
   */
  prop = libpolyxmass_prop_find_prop (editctxt->polymer->propGPA,
				  NULL,
				  NULL,
				  "LEFT_END_MODIF",
				  NULL,
				  PXM_UNDEF_PROP_CMP);

  widget = g_object_get_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
			      "left_end_modif_entry");
  g_assert (widget != NULL);
  
  if (prop != NULL)
    {
      gtk_entry_set_text (GTK_ENTRY (widget), (gchar *) prop->data);
    }
  else
    {
      gtk_entry_set_text (GTK_ENTRY (widget), _("None set"));
    }

  
  /* The right end modif.
   */
  prop = libpolyxmass_prop_find_prop (editctxt->polymer->propGPA,
				  NULL,
				  NULL,
				  "RIGHT_END_MODIF",
				  NULL,
				  PXM_UNDEF_PROP_CMP);

  widget = g_object_get_data (G_OBJECT (seqeditorctxt->sequence_editor_wnd), 
			      "right_end_modif_entry");
  g_assert (widget != NULL);
  
  if (prop != NULL)
    {
      gtk_entry_set_text (GTK_ENTRY (widget), (gchar *) prop->data);
    }
  else
    {
      gtk_entry_set_text (GTK_ENTRY (widget), _("None set"));
    }

  return TRUE;
}






gint
polyxedit_seqed_wnd_render_polseq_monicons (PxmEditCtxt *editctxt,
					    gboolean rerender)
{
  /* We get a pointer to a polymer sequence context, and are asked
   * that the monicon rendering be performed for each monomer in the
   * sequence. If rerender is TRUE, then the array of monicons should
   * first be destroyed, so that the rendering is forced for EACH 
   * monomer in the sequence. This function returns the number of 
   * rendered pixbufs.
   *
   * The way the task is achieved is by running a loop in which (for
   * each monomer in editctxt->polymer->monomerGPA) three main
   * achievements are performed:
   *
   * 1. Allocate a new monicon instance.
   * 
   * 2. Render a pixbuf (or just find it in the GData of the
   * polchemdefctxt) for the monomer->code.
   *
   * 3. Use the pixbuf from step 1. to create with it a new
   * monicon->canvas_item object.
   */
  gint count = 0;
  gint iter = 0;

  PxmProp *prop = NULL;

  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  PxmPolymer *polymer = NULL;
  PxmMonomer *monomer = NULL;
  PxmMonicon *monicon = NULL;

  GdkPixbuf *pixbuf = NULL;
  

  g_assert (editctxt != NULL);
  

  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  if (rerender == TRUE)
    {
      /* If we are here because we have to rerender all the monicons
       * from scratch, typically if the user asked that the dimension
       * of the monicons be changed, then update also the scroll
       * adjustement's steps after having freed the array of monicons
       * instances.
       */

      /* First empty moniconGPA and free all the monicons that were
	 prepared to display the sequence in the editor.
      */
      polyxedit_monicon_GPA_empty (seqeditorctxt->moniconGPA);
      
      /* Clear the pxbfGData GData list. BUT before doing this
       * we have to unref one last time all the GdkPixbuf objects that
       * were made during the loading/editing of the polymer sequence.
       */
      g_datalist_foreach 
	(&editctxt->polchemdefctxt->pixbufGData,
	 polyxmass_polchemdefctxt_unref_last_pixbuf_from_gdata,
	 NULL);

      g_datalist_clear (&editctxt->polchemdefctxt->pixbufGData);

      /* And now we can initialize back the GData keyed datalist.
       */
      g_datalist_init (&editctxt->polchemdefctxt->pixbufGData);
      
      /* Set the increment to the size of the monicons.
       */
      seqeditorctxt->canvas_scroll_wnd_adjust_v->step_increment = 
	seqeditorctxt->monicon_size;
      seqeditorctxt->canvas_scroll_wnd_adjust_h->step_increment = 
	seqeditorctxt->monicon_size;

      /*
	debug_printf (("rerender was TRUE in "
	"polyxedit_seqed_wnd_render_polseq_monicons\n"));
      */
    }
  
  
  for (iter = 0; iter < polymer->monomerGPA->len; iter++)
    {
      monomer = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      /* First see if the monomer is modified. Depending on the result
       * create (or get from the GData in seqeditorctxt) a pixbuf for it.
       */
      prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				      NULL,
				      NULL,
				      "MODIF",
				      NULL,
				      PXM_UNDEF_PROP_CMP);
      if (prop == NULL)
	pixbuf = polyxedit_rendering_pixbuf_render_no_modif (monomer->code,
							    seqeditorctxt->
							    monicon_size,
							    editctxt);
      
      else
	pixbuf = polyxedit_rendering_pixbuf_render_modif (monomer->code,
							  (gchar *) prop->
							  data ,
							  seqeditorctxt->
							  monicon_size,
							  editctxt);
      
      if (pixbuf == NULL)
	{
	  g_critical (_("%s@%d: failed rendering monicon for monomer '%s' "
		   "at index '%d'\n"),
		 __FILE__, __LINE__, monomer->code, iter);
	  
	  /* Do not free it, because is will be freed later, when 
	     all the process of shutting down the editctxt is carried
	     out !!!
	     polyxedit_monicon_GPA_free (seqeditorctxt->moniconGPA);
	  */

	  return -1;
	}
      
      /* Apparently we could successfully render (or get a pointer to)
       * a pixbuf for the monomer being iterated. Go on with making a
       * new monicon for the monomer in question and then make the
       * canvas_item.  During the process of creating the canvas_item
       * the pixbuf is going to be reffed once.
       */
      monicon = polyxedit_monicon_new ();

      if (FALSE == 
	  polyxedit_rendering_monicon_make_canvas_item (monicon, 
							seqeditorctxt->
							monicon_size,
							pixbuf,
							iter,
							editctxt))
	{
	  g_critical (_("%s@%d: failed making canvas_item for monomer code '%s' "
		   "at index '%d'\n"),
		 __FILE__, __LINE__, monomer->code, iter);
	  
	  polyxedit_monicon_GPA_free (seqeditorctxt->moniconGPA);

	  return -1;
	}

      /* At this point we have a fully qualified monicon.
       */
      g_ptr_array_add (seqeditorctxt->moniconGPA, monicon);

      /*
	debug_printf (("added monicon %p for monomer %s - "
	"Array is of len %d\n", monicon, monomer->name,
	seqeditorctxt->moniconGPA->len));
      */

      count++;
    }

  /* Now, our seqeditorctxt object has a GPtrArray filled with monicon
   * objects faithfully representing the polymer sequence.
   */

  return count ;
}



void
polyxedit_seqed_wnd_set_polymer_modified (PxmEditCtxt *editctxt, 
					  gboolean modified)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  

  g_assert (editctxt != NULL);
  
  window = editctxt->seqeditorctxt->sequence_editor_wnd;
  g_assert (window != NULL);
  
  widget = g_object_get_data (G_OBJECT (window),  
			      "polseq_status_label");

  pxmchem_polymer_set_modified (editctxt->polymer, modified);
  
  (TRUE == editctxt->polymer->modified) ?
    gtk_label_set_text (GTK_LABEL (widget), _("Modified")) :
    gtk_label_set_text (GTK_LABEL (widget), _("Unmodified"));
   
}


gboolean 
polyxedit_seqed_wnd_ensure_region_visible (PxmEditCtxt *editctxt,
					   gint start_idx,
					   gint end_idx)
{
  /* This function is called by other functions when they need to make
   * sure that a given polymer sequence region is effectively visible
   * (at least its first elements if not in its entirety) in the
   * sequence editor window. We are handed the polymer sequence
   * context pointer, which is the gate to the editor window, and the
   * two indexes of the start monomer and of the end monomer which
   * both constitute the borders of the region to display.
   */
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  gint last_vis_line_pixel_y = 0;
  gint moment_idx = 0;
  gint hscrollbar_height = 0;

  PxmRect start_rect;
  
    


  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  /* We may check some sanity issues:
   */
  if (start_idx > end_idx)
    {
      moment_idx = start_idx;
      start_idx = end_idx;
      end_idx = moment_idx;
    }
  
  if (start_idx < 0)
    return FALSE;
  
  if (end_idx > editctxt->polymer->monomerGPA->len -1)
    return FALSE;
  
  /* Get the pixel y coordinate of the last visible line, we'll need it
   * later.
   */ 
  last_vis_line_pixel_y = seqeditorctxt->canvas_scroll_wnd_adjust_v->value 
    + seqeditorctxt->canvas_scroll_wnd_height;

  /* We should get the upper left (NorthWest) y coordinate of 
   * the initial monomer in the region (index start_idx):
   */
  polyxedit_seqed_wnd_monicon_get_pixel_coord (start_idx,
					       seqeditorctxt,
					       &start_rect,
					       COORDSYS_NW, COORDSYS_NW);
  
  /* start_rect.y1 represents now such y NW coordinate. If this 
   * y coordinate is not visible we should make a scroll, otherwise
   * do not bother. ATTENTION, the size of the scroll bar, the 
   * horizontal one (if any) is included in the computation of the 
   * scroll win height. This is why, in order to ENSURE that the region
   * which may be partially visible on the bottom of the seqed window is
   * indeed fully visible, I have asked that the north western 
   * coordinate of the monomer at index start_idx be less or equal to
   * the last visible line of the seqeditor window minus
   * the size of an icon and minus the horizontal scrollbar height.
   */
  hscrollbar_height = 
    GTK_WIDGET (GTK_SCROLLED_WINDOW 
		(seqeditorctxt->
		 canvas_scroll_wnd)->hscrollbar)->allocation.height;
  
  if (start_rect.y1 < seqeditorctxt->canvas_scroll_wnd_adjust_v->value
      || start_rect.y1 > (last_vis_line_pixel_y - 
			  seqeditorctxt->monicon_size -
			  hscrollbar_height))
    {
      seqeditorctxt->y_offset = start_rect.y1;
      
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      seqeditorctxt->y_offset);
      
      scroll_size_changed = FALSE;
      
      return TRUE;
    }

  return TRUE;
}


gboolean
polyxedit_seqed_wnd_ensure_select_cursor_visible (GtkWidget *window, 
						  GdkEventConfigure *event, 
						  gpointer data)
{
  /* Called after the handler seqed_sw_size_allocate() was
   * called, since we connected this function through a call to 
   * gtk_signal_connect_after () for "size_allocate" signal of 
   * seqeditorctxt->canvas_scroll_wnd. 
   *
   * However, we saw that the size_allocate () function is called even
   * when no change in the size of the scrolled window is made, thus
   * eliciting a call to _this_ function, which we do not want, since
   * the calculations dones here are ONLY for use when the size of the
   * window EFFECTIVELY changes. SO, we have a file-scope variable
   * (gboolean scrollwin_size_changed) that is set to TRUE in
   * size_allocate () function whenever a real size change has been
   * done to the scrolled window. If this variable equals TRUE in
   * _this_ function that means we have to do the computations,
   * otherwise just return.
   */
  /* This function is meant to make sure that
   * the cursor (and, if any, the biggest portion of the current 
   * selection) be visible in the seqeditor window. 
   * Typically this function is called
   * when the window is resized, which might hide the cursor 
   * (and selection if any) from the visible portion of the sequence.
   */
  PxmEditCtxt *editctxt = (PxmEditCtxt *) data;
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  gint last_vis_line_pixel_y = 0;
  gint hscrollbar_height = 0;


  g_assert (editctxt != NULL);

  /* Removed because it would inhibit the process when editing 
     a piece of sequence that would be below the bottom of the canvas.
     
     if (scroll_size_changed == FALSE) 
     return TRUE;
  */

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  
  /* There are two cases :
   *
   * 1. there is not a selection in the sequnece editor. In this case
   * we only want to make sure that the cursor is visible in the
   * sequence editor. Best would be to make it visible in the vertical
   * center of the sequence editor.
   *
   * 2. there is a selection. In this case there are two different
   * situations :
   *
   * 2.A. if the selection was done from top to bottom (sel_mnm_idx1 <
   * sel_mnm_idx2) the window should be scrolled so that the cursor
   * (at position sel_mnm_idx2) is displayed at the bottom of the
   * seqeditor, which is the last visible line.  This way we manage
   * that the largest portion possible of the selection is displayed
   * in the seqeditor window.
   *
   * 2.B. if the selection was done from bottom to top (sel_mnm_idx2 <
   * sel_mnm_idx1) the window should be scrolled so that the cursor
   * (at position sel_mnm_idx1) is displayed at the top of the
   * seqeditor, which is the first visible line. This way we manage
   * that the largest portion possible of the selection is displayed
   * in the seqeditor window.
   */

  /* Get the pixel y coordinate of the last visible line, we'll need
   * it later.
   */ 
  last_vis_line_pixel_y = seqeditorctxt->canvas_scroll_wnd_adjust_v->value
    + seqeditorctxt->canvas_scroll_wnd_height;
  
  /* Check if there is a selection.
   */
  if (seqeditorctxt->sel_mnm_idx1 != -1 && seqeditorctxt->sel_mnm_idx2 != -1)
    {
      /* An oligomer is selected. We have to test the two different cases
       * described above.
       */
      if (seqeditorctxt->sel_mnm_idx1 < seqeditorctxt->sel_mnm_idx2)
	{
	  /* The selection was done from top to bottom, so the scroll
	     has to be done so that the cursor is at the last visible
	     line.
	   */

	  /* ATTENTION, the size of the scroll bar, the horizontal one
	     (if any) is included in the computation of the scroll win
	     height.  This is why, in order to ENSURE that the cursor
	     be precisely located at the bottom of the visible scroll
	     window, I have asked that the window be scrolled
	     according to the computation below:
	   */
	  hscrollbar_height = 
	    GTK_WIDGET (GTK_SCROLLED_WINDOW 
			(seqeditorctxt->canvas_scroll_wnd)->
			hscrollbar)->allocation.height;
	  
	  seqeditorctxt->y_offset = seqeditorctxt->cursor->y  +
	    seqeditorctxt->monicon_size
	    - seqeditorctxt->canvas_scroll_wnd_height + hscrollbar_height + 
	    SEL_POLYGON_LINE_WIDTH ;
	}
      else
	  /* The selection was done from bottom to top, so the scroll
	   * has to be done so that the cursor is at first visible
	   * line.
	   */
	  seqeditorctxt->y_offset = seqeditorctxt->cursor->y;

      /* Now that the new y_offset is computed, make the scroll
	 happen.
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      seqeditorctxt->y_offset);

      scroll_size_changed = FALSE;

      return TRUE;
    }
  else
    /* No oligomer is currently selected, so the only task here is to
     * scroll the window to make sure the cursor is visible in it.
     */
    {
      /* Only perform the calculations and the y_scroll if the cursor
       * effectively lies either above the first visible line or below
       * the last visible line.
       */
      /* Test if cursor is not visible, in which case perform a scroll
       * of the canvas, so that the cursor is now visible on the first
       * visible line of the seqeditor.
       */
      if (seqeditorctxt->cursor->y < 
	  seqeditorctxt->canvas_scroll_wnd_adjust_v->value
	  || seqeditorctxt->cursor->y > last_vis_line_pixel_y)
	{
	  seqeditorctxt->y_offset = seqeditorctxt->cursor->y;

	  gnome_canvas_scroll_to 
	    (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
	     seqeditorctxt->x_offset,
	     seqeditorctxt->y_offset);

	  scroll_size_changed = FALSE;

	  return TRUE;
	}
    }

  scroll_size_changed = FALSE;

  return TRUE;
}







/*
****************************************************************
************* DRAWING FUNCTIONS THROUGHOUT THE FILE ************
****************************************************************
*/
gint 
polyxedit_seqed_wnd_redraw_sequence (PxmEditCtxt *editctxt)
{
  /* We get new size of the window, which means we can compute how
   * many icons we can set to the line:
   */
  gint iter = 0;
  gint count = 0;
  
  gdouble scroll_x = 0;
  gdouble scroll_y = 0;

  gchar *label = NULL;

  PxmMonicon *monicon = NULL;
  
  GnomeCanvasItem *label_item = NULL;

  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  g_assert (editctxt != NULL);
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  if (seqeditorctxt->monicon_size > 0)
    {
      seqeditorctxt->monicons_per_line = 
	(seqeditorctxt->canvas_scroll_wnd_width - seqeditorctxt->left_margin) 
	/ seqeditorctxt->monicon_size;
    }
  else
    {
      g_error (_("%s@%d: width of monomer icon is not > 0\n"),
	     __FILE__, __LINE__);
      
      return -1 ;
    }
  /*
  printf ("seqeditorctxt->monicons_per_line = %d\n", 
	  seqeditorctxt->monicons_per_line);
  */

  /* Empty the array of labels so that we can fill it with newly 
   * made ones.
   */
  g_assert (seqeditorctxt->labelGPA != NULL);
  
  while (seqeditorctxt->labelGPA->len)
    {
      label_item = g_ptr_array_index (seqeditorctxt->labelGPA, iter);
      g_ptr_array_remove_fast (seqeditorctxt->labelGPA, label_item);
      gtk_object_destroy (GTK_OBJECT (label_item));
    }
  
  g_ptr_array_free (seqeditorctxt->labelGPA, TRUE);
  seqeditorctxt->labelGPA = g_ptr_array_new ();


  /* Iter in the array of monicons and move them to their new place.
   */
  for (iter = 0; iter < seqeditorctxt->moniconGPA->len; iter++)
    {
      monicon = g_ptr_array_index (seqeditorctxt->moniconGPA, iter);

      polyxedit_seqed_wnd_monicon_get_pixel_coord (iter, 
						   seqeditorctxt, 
						   &monicon_rect, 
						   COORDSYS_NW, 
						   COORDSYS_NW);

      if (monicon_rect.x1 == seqeditorctxt->left_margin)
	{
	  /* We are at first position in a row, so we need to 
	   * create and display a label indicating the monomer
	   * current position in red in the margin
	   */
	  label = g_strdup_printf ("%d", iter + 1);

	  label_item =
	    gnome_canvas_item_new (seqeditorctxt->main_canvas_group,
				   gnome_canvas_text_get_type (),
				   "text", label,
				   "font", "12x24",
				   "x", monicon_rect.x1 -
				   (seqeditorctxt->left_margin/2),
				   "y", monicon_rect.y1 +
				   (seqeditorctxt->monicon_size/2),
				   "fill_color", "red",
				   "anchor", GTK_ANCHOR_CENTER,
				   NULL); 
	  
	  g_ptr_array_add (seqeditorctxt->labelGPA, label_item);
	  
	  g_free (label);
	}
      
      /* Now that we have computed the new position, we can ask a move
       * using the x y position that was previously stored in the item.
       */
      gnome_canvas_item_move (monicon->canvas_item, 
			      monicon_rect.x1 - monicon->x,
			      monicon_rect.y1 - monicon->y);

      /* Do not forget to update the x y position of the item now that
       * it has been moved (for next move, of course)!
       */
      monicon->x = monicon_rect.x1;
      monicon->y = monicon_rect.y1;

      count++;
    }

  /* Below, the y+seqeditorctxt->monicon_size is to take into account the
   * height of the icons, since scroll measurements are done according
   * to anchor point southwest and our measurements are according to
   * anchor point northwest.
   */
  scroll_x = (seqeditorctxt->monicons_per_line * seqeditorctxt->monicon_size) 
    + seqeditorctxt->left_margin;

  scroll_y = monicon_rect.y1 + seqeditorctxt->monicon_size;

  gnome_canvas_set_scroll_region (GNOME_CANVAS (seqeditorctxt->
						sequence_canvas), 
				  0, 0, scroll_x, scroll_y);



#if 0

  /* The sequence editor context (aka seqeditorctxt) has a GPtrArray of
   * very special prop instances: visualGPA. This array
   * contains prop instances which describe the way graphical elements
   * should be drawn onto the sequence canvas items made of all
   * the monicons.
   * 
   * For example, imagine you have searched for a sequence motif 
   * "KA" in the currently active polymer sequence and a full
   * list of found "KA" motifs is displayed to you. You may ask, by 
   * clicking onto an item (or more) of the list, that a visual clue
   * be given to you as to where the "KA" motif is found in the
   * sequence. This may be as simple as one black line at the bottom
   * of the two monicon objects that represent the "KA" sequence motif
   * present in the sequence editor.
   *
   * Now, imagine that you change the size of the seqence editor window.
   * The black line in question will no more be drawn at the proper
   * place, because it is stuck to coordinates which were calculated
   * for another size of the window. What we want here is to find
   * a way that will let the sequence redrawing function recalculate 
   * all graphical elements that need refreshing after the size of 
   * the window has changed or the sequence has been edited.
   *
   * In our example of the found motif, what we have is a 
   * prop named "FINDVISUAL_PTR" and of which the p_data member
   * points to a findvisual structure which holds absolutely all
   * the data required for full redraw of the graphical elements
   * for which it was created in the first place.
   * The strategy here is to do some kind of plugin work. We have 
   * to iterated in the array of prop instances and -for each one-
   * find a plugin function which can handle the redrawing work.
   */

  PxmProp *prop = NULL;
  
  gboolean render_result = FALSE;
  
  for (iter = 0 ; iter < seqeditorctxt->visualGPA->len ; iter++)
    {
      prop = g_ptr_array_index (seqeditorctxt->visualGPA, iter);
      g_assert (prop != NULL);


      visual_render_in_seqed_wnd_plugin_func = 
	visual_render_in_seqed_wnd_plugin_chooser (prop->str_name);
      
      if (visual_render_in_seqed_wnd_plugin_func != NULL)
	{
	  render_result = visual_render_in_seqed_wnd_plugin_func (prop);

	  if (render_result != PXM_ERR_NOERR)
	    {
	      mk_strerr (__FILE__, __LINE__);
	      g_critical ("could not find the rendering plugin for %s\n",
		     prop->str_name);
	    }
	}
      else
	{
	  mk_strerr (__FILE__, __LINE__);
	  g_critical ("the visual rendering property is not registered: %s\n",
		 prop->str_name);
	}
    }

#endif  

  /* Now that the mere polymer sequence has been drawn again, and 
     that all the other visuals are rendered, we
     can reselect the oligomer that was selected before.
  *
  polyxedit_seqed_wnd_update_sel_polygon (seqeditorctxt);
  */

  /* Draw the cursor.
   */
  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);

  return count;
}


void
polyxedit_seqed_wnd_draw_cursor (PxmSeqEditorCtxt *seqeditorctxt)
{
  /* The position of the cursor is governed by the last
   * pointed monomer index, which is stored in the variable 
   * last_point_1_idx each time something relevant is done
   * (mouse click, keyboard actions....)
   */
  PxmRect rect;

  g_assert (seqeditorctxt != NULL);

  if (seqeditorctxt->last_point_1_idx < 0)
    seqeditorctxt->last_point_1_idx = 0;

  /* We tolerate, for graphical purposes, that the "index" of the
   * last selectable monomer be > max index of polymer length, but
   * when accessing the polymer data, always check !
   */
  if (seqeditorctxt->last_point_1_idx > seqeditorctxt->moniconGPA->len)
    seqeditorctxt->last_point_1_idx = seqeditorctxt->moniconGPA->len;

  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
					       last_point_1_idx,
					       seqeditorctxt, 
					       &rect,
					       COORDSYS_NW, COORDSYS_NW);
  

  /* We do not want that the cursor be alone at position right of 
   * the LEFT_MARGIN because cursor position actually is at end of 
   * the line above the one where it would be without the correction
   * below. The exception is when the monicon index is 0,
   * since of course it is possible to set the cursor at left
   * position of the first monicon in the sequence.
   */
  if (rect.x1 == seqeditorctxt->left_margin && seqeditorctxt->
      last_point_1_idx > 0)
    {
      rect.x1 = seqeditorctxt->left_margin 
	+ (seqeditorctxt->monicons_per_line * seqeditorctxt->monicon_size);
      
      rect.y1 -= seqeditorctxt->monicon_size;
    } 
  
  gnome_canvas_item_move (seqeditorctxt->cursor->canvas_item,
			  rect.x1 - seqeditorctxt->cursor->x,
			  rect.y1 - seqeditorctxt->cursor->y);

  /* When we are redrawing the cursor because we have typed in a new
   * polymer sequence portion the newly monicon's will be on top of
   * the cursor, this is why we ask it to be on top of absolutely
   * everything !!!
   */
  gnome_canvas_item_raise_to_top (seqeditorctxt->cursor->canvas_item);

  seqeditorctxt->cursor->x = rect.x1;
  seqeditorctxt->cursor->y = rect.y1;
}
  

void
polyxedit_seqed_wnd_draw_sel_polygon (PxmSeqEditorCtxt *seqeditorctxt)
{
  /* We get a pointer to a polymer editing context structure that
   * contains a selection_rect member structure that contains the
   * coordinates of the (one or two) point(s) describing the pixel
   * position in the sequence editor canvas of the current selection
   * (if any !).  We use this (these) point(s) to draw a polygon by
   * computing all the necessary points around the selecton to draw a
   * "polyline" polygon.
   */
  gint sel_lines_count;

  gdouble end_line_EE_x = 0;


  g_assert (seqeditorctxt != NULL);

  /* Only consider selection is effective if the movement amplitude in
   * x or y is at least a third of the monicon width (in x).
   */
  if (abs (seqeditorctxt->selection_rect.x1 - 
	   seqeditorctxt->selection_rect.x2) 
      <= (0.3 * seqeditorctxt->monicon_size)
      && 
      abs (seqeditorctxt->selection_rect.y1 - 
	   seqeditorctxt->selection_rect.y2)
      <= (0.3 * seqeditorctxt->monicon_size))
    {
      /* the user has moved the pointer while pushing left button
       * over a too small distance to consider that it is a selection.
       * Thus we do not bother with drawing any specific selection
       * polygon, we just set the sel_monomerIdx(1,2) to -1, which
       * indicates that no real selection is there.
       */
      seqeditorctxt->sel_mnm_idx1 = -1;
      seqeditorctxt->sel_mnm_idx2 = -1;

      if (seqeditorctxt->canvas_selection_points)
	{
	  gnome_canvas_points_unref (seqeditorctxt->
				     canvas_selection_points);
	  seqeditorctxt->canvas_selection_points = NULL;
	}
      if (seqeditorctxt->canvas_selection_polygon)
	{
	  gtk_object_destroy (GTK_OBJECT (seqeditorctxt->
					  canvas_selection_polygon));
	  seqeditorctxt->canvas_selection_polygon = NULL;
	}
      return;
    }
  

  /* Apparently the user has effectively wanted to select a piece
   * of polymer sequence.
   */
  /* Get the indexes of the first and last selected monomers.
   */
  seqeditorctxt->sel_mnm_idx1 = 
    polyxedit_seqed_wnd_get_mnm_idx_with_xy (seqeditorctxt,
					     seqeditorctxt->
					     selection_rect.x1,
					     seqeditorctxt->
					     selection_rect.y1);
  
  seqeditorctxt->sel_mnm_idx2 = 
    polyxedit_seqed_wnd_get_mnm_idx_with_xy (seqeditorctxt,
					     seqeditorctxt->
					     selection_rect.x2,
					     seqeditorctxt->
					     selection_rect.y2);
  
  /* We have to set an ordered polygon construction mechanism because
   * the points at each polygon corner are in a very precise order.
   */
  if (seqeditorctxt->sel_mnm_idx1 <= seqeditorctxt->sel_mnm_idx2)
    {
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx1,
						   seqeditorctxt, 
						   &(seqeditorctxt->mnm1rect),
						   COORDSYS_NW, COORDSYS_SW);
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx2,
						   seqeditorctxt, 
						   &(seqeditorctxt->mnm2rect),
						   COORDSYS_NW, COORDSYS_SW);
    }

  if (seqeditorctxt->sel_mnm_idx1 > seqeditorctxt->sel_mnm_idx2)
    {
      /* We have to interchange the two monomers, since for polygon
       * drawing we need the first monomer to be the lowest index
       * Be shure to see the difference in the function calls this time !
       */
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx1,
						   seqeditorctxt, 
						   &(seqeditorctxt->mnm2rect),
						   COORDSYS_NW, COORDSYS_SW);
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx2,
						   seqeditorctxt, 
						   &(seqeditorctxt->mnm1rect),
						   COORDSYS_NW, COORDSYS_SW);
    }

  /* Now we can draw the polygon.
   */

  /* But first eliminate any trace of a previous polygon!
   */
  if (seqeditorctxt->canvas_selection_points)
    {
      gnome_canvas_points_unref (seqeditorctxt->canvas_selection_points);
      seqeditorctxt->canvas_selection_points = NULL;
    }

  if (seqeditorctxt->canvas_selection_polygon)
    {
      gtk_object_destroy (GTK_OBJECT (seqeditorctxt->
				      canvas_selection_polygon));
      seqeditorctxt->canvas_selection_polygon = NULL;
    }

  /* Compute the X position corresponding to EAST position of
   * last monomer icon in a ROW.
   */
  end_line_EE_x = seqeditorctxt->left_margin 
    + (seqeditorctxt->monicons_per_line * seqeditorctxt->monicon_size);

  /* Compute the number of lines that the selection encompasses.
   */
  sel_lines_count = (seqeditorctxt->mnm2rect.y1 - seqeditorctxt->mnm1rect.y1)
    / seqeditorctxt->monicon_size;
  
  /* We only draw the polygon if there is a real polygon,
   * which means something at least a square!
   */
  if (sel_lines_count == 0)
    {
      if (seqeditorctxt->mnm1rect.x1 == seqeditorctxt->mnm2rect.x1)
	{
	  /* Two points which should have different X coordinates
	   * are the same : there is no real square here.
	   */
	  if (seqeditorctxt->canvas_selection_points)
	    {
	      gnome_canvas_points_unref (seqeditorctxt->
					 canvas_selection_points);
	      seqeditorctxt->canvas_selection_points = NULL;
	    }
	  if (seqeditorctxt->canvas_selection_polygon)
	    {
	      gtk_object_destroy (GTK_OBJECT (seqeditorctxt->
					      canvas_selection_polygon));
	      seqeditorctxt->canvas_selection_polygon = NULL;
	    }
	  return;
	}
    }

  /*
    debug_printf (("SELECTION = monomer1 idx is %d -- monomer2 idx is %d\n",
    seqeditorctxt->sel_mnm_idx1, seqeditorctxt->sel_mnm_idx2)); 
  */

  /* We do not want that the selection polygon makes a fake
   * more-than-one-line selection pattern while at last line it does
   * not select anything because selection actually stops at right end
   * of line above.
   */
  if (sel_lines_count > 0 && seqeditorctxt->mnm2rect.x1 == 
      seqeditorctxt->left_margin)
    {
      /* We are right at the MARGIN, and we want to go up one line and
       * right to the last monicon.
       */
      sel_lines_count -= 1;
      seqeditorctxt->mnm2rect.x1 = end_line_EE_x;
      seqeditorctxt->mnm2rect.y1 -= seqeditorctxt->monicon_size;
      seqeditorctxt->mnm2rect.y2 -= seqeditorctxt->monicon_size;
    }  
  
  /* Now resume normal way of positioning the points of the selection
   * polygon.
   */
  if (sel_lines_count == 0)
    {
      seqeditorctxt->canvas_selection_points = gnome_canvas_points_new (4);

      seqeditorctxt->canvas_selection_points->coords[0] = 
	seqeditorctxt->mnm1rect.x1;
      seqeditorctxt->canvas_selection_points->coords[1] = 
	seqeditorctxt->mnm1rect.y1;

      seqeditorctxt->canvas_selection_points->coords[2] = 
	seqeditorctxt->mnm2rect.x1;
      seqeditorctxt->canvas_selection_points->coords[3] = 
	seqeditorctxt->mnm2rect.y1;

      seqeditorctxt->canvas_selection_points->coords[4] = 
	seqeditorctxt->mnm2rect.x1;
      seqeditorctxt->canvas_selection_points->coords[5] = 
	seqeditorctxt->mnm2rect.y2;

      seqeditorctxt->canvas_selection_points->coords[6] = 
	seqeditorctxt->mnm1rect.x1;
      seqeditorctxt->canvas_selection_points->coords[7] = 
	seqeditorctxt->mnm2rect.y2;

    }

  if (sel_lines_count > 0)
    {
      seqeditorctxt->canvas_selection_points = gnome_canvas_points_new (8);
      
      seqeditorctxt->canvas_selection_points->coords[0] = 
	seqeditorctxt->mnm1rect.x1;
      seqeditorctxt->canvas_selection_points->coords[1] = 
	seqeditorctxt->mnm1rect.y1;
      
      seqeditorctxt->canvas_selection_points->coords[2] = end_line_EE_x;
      seqeditorctxt->canvas_selection_points->coords[3] = 
	seqeditorctxt->mnm1rect.y1;
      
      seqeditorctxt->canvas_selection_points->coords[4] = end_line_EE_x;
      seqeditorctxt->canvas_selection_points->coords[5] = 
	seqeditorctxt->mnm1rect.y1 +
	(sel_lines_count * seqeditorctxt->monicon_size);
      
      seqeditorctxt->canvas_selection_points->coords[6] = 
	seqeditorctxt->mnm2rect.x1;
      seqeditorctxt->canvas_selection_points->coords[7] = 
	seqeditorctxt->mnm1rect.y1 +
	(sel_lines_count * seqeditorctxt->monicon_size);
      
      seqeditorctxt->canvas_selection_points->coords[8] = 
	seqeditorctxt->mnm2rect.x1;
      seqeditorctxt->canvas_selection_points->coords[9] = 
	seqeditorctxt->mnm1rect.y1 +
	(sel_lines_count * seqeditorctxt->monicon_size) 
	+ seqeditorctxt->monicon_size;
      
      seqeditorctxt->canvas_selection_points->coords[10] = LEFT_MARGIN;
      seqeditorctxt->canvas_selection_points->coords[11] = 
	seqeditorctxt->mnm1rect.y1 +
	(sel_lines_count * seqeditorctxt->monicon_size) 
	+ seqeditorctxt->monicon_size;

      seqeditorctxt->canvas_selection_points->coords[12] = LEFT_MARGIN;
      seqeditorctxt->canvas_selection_points->coords[13] = 
	seqeditorctxt->mnm1rect.y1
	+ seqeditorctxt->monicon_size;

      seqeditorctxt->canvas_selection_points->coords[14] = 
	seqeditorctxt->mnm1rect.x1;
      seqeditorctxt->canvas_selection_points->coords[15] = 
	seqeditorctxt->mnm1rect.y1
	+ seqeditorctxt->monicon_size;
    }

  seqeditorctxt->canvas_selection_polygon =
    gnome_canvas_item_new (seqeditorctxt->main_canvas_group,
			   gnome_canvas_polygon_get_type (),
			   "points", seqeditorctxt->canvas_selection_points,
			   "outline_color", "black",
			   "width_units", SEL_POLYGON_LINE_WIDTH,
			   NULL);

  /* PLEASE NOTE THAT LAST PARAM IS NON-NULL, it contains the
   * polymer editing context pointer.
   */
  g_signal_connect (G_OBJECT (seqeditorctxt->canvas_selection_polygon), 
		    "event",
		    G_CALLBACK (polyxedit_seqed_wnd_sel_polygon_event), 
				seqeditorctxt);
}


gboolean
polyxedit_seqed_wnd_update_sel_polygon (PxmSeqEditorCtxt *seqeditorctxt)
{
  /* We get the xy positions for the two points corresponding to
   * initial impact of mouse and to current mouse position.
   */
  gint sel_lines_count;

  gdouble end_line_EE_x = 0;


  g_assert (seqeditorctxt != NULL);


  /* This function returns TRUE if the selection polygon was 
     actually changed; FALSE if it remained unchanged.
  */


  /* If the window was moved or resized, the sequence is redrawn,
   * which is ok, but the selection is updated, hence we are here. BUT
   * if no selection was in operation before the moving or resizing of
   * the window, then we do not want any selection (if we let the
   * function do its work here with both values below == 0, the first
   * monomer of the sequence is selected, which we do not want)
   */
  if (seqeditorctxt->sel_mnm_idx1 == -1 && seqeditorctxt->sel_mnm_idx2 == -1)
    return FALSE;

  if (seqeditorctxt->sel_mnm_idx1 <= seqeditorctxt->sel_mnm_idx2)
    {
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx1,
						   seqeditorctxt, 
						   &(seqeditorctxt->mnm1rect),
						   COORDSYS_NW, COORDSYS_SW);
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx2,
						   seqeditorctxt, 
						   &(seqeditorctxt->mnm2rect),
						   COORDSYS_NW, COORDSYS_SW);
    }
  
  if (seqeditorctxt->sel_mnm_idx1 > seqeditorctxt->sel_mnm_idx2)
    {
      /* we have to interchange the two monomers, since for polygon
       * drawing we need the first monomer to be the lowest index
       * Be shure to see the difference in the function calls this time !
       */
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx1,
						   seqeditorctxt, 
						   &(seqeditorctxt->mnm2rect),
						   COORDSYS_NW, COORDSYS_SW);
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx2,
						   seqeditorctxt, 
						   &(seqeditorctxt->mnm1rect),
						   COORDSYS_NW, COORDSYS_SW);
    }
  
  /* Now we can draw the polygon.
   */
  
  /* But first eliminate any trace of a previous polygon!
   */
  if (seqeditorctxt->canvas_selection_points)
    {
      gnome_canvas_points_unref (seqeditorctxt->canvas_selection_points);
      seqeditorctxt->canvas_selection_points = NULL;
    }

  if (seqeditorctxt->canvas_selection_polygon)
    {
      gtk_object_destroy (GTK_OBJECT (seqeditorctxt->
				      canvas_selection_polygon));
      seqeditorctxt->canvas_selection_polygon = NULL;
    }

  /* Compute the X position corresponding to EAST position of
   * last monomer icon in a ROW.
   */
  end_line_EE_x = seqeditorctxt->left_margin 
    + (seqeditorctxt->monicons_per_line * seqeditorctxt->monicon_size);
  
  /* Compute the number of lines that the selection encompasses.
   */
  sel_lines_count = (seqeditorctxt->mnm2rect.y1 - seqeditorctxt->mnm1rect.y1)
    / seqeditorctxt->monicon_size;
  
  /* We only draw the polygon if there is a real polygon,
   * which means something at least a square!
   */
  if (sel_lines_count == 0)
    {
      if (seqeditorctxt->mnm1rect.x1 == seqeditorctxt->mnm2rect.x1)
	{
	  /* Two points which should have different X coordinates
	   * are the same : there is no real square here.
	   */
	  if (seqeditorctxt->canvas_selection_points)
	    {
	      gnome_canvas_points_unref (seqeditorctxt->
					 canvas_selection_points);
	      seqeditorctxt->canvas_selection_points = NULL;
	    }
	  if (seqeditorctxt->canvas_selection_polygon)
	    {
	      gtk_object_destroy (GTK_OBJECT (seqeditorctxt->
					      canvas_selection_polygon));
	      seqeditorctxt->canvas_selection_polygon = NULL;
	    }
	  return TRUE;
	}
    }

  /* We do not want that the selection polygon makes a fake
   * more-than-one-line selection pattern while at last line it does
   * not select anything because selection actually stops at right end
   * of line above.
   */
  if (sel_lines_count > 0 && seqeditorctxt->mnm2rect.x1 == 
      seqeditorctxt->left_margin)
    {
      /* We are right at the MARGIN, and we want to go up one line and
       * right to the last monicon.
       */
      sel_lines_count -= 1;
      seqeditorctxt->mnm2rect.x1 = end_line_EE_x;
      seqeditorctxt->mnm2rect.y1 -= seqeditorctxt->monicon_size;
      seqeditorctxt->mnm2rect.y2 -= seqeditorctxt->monicon_size;
    }  

  /* Now resume normal way of positioning the points of the selection
   * polygon.
   */
  if (sel_lines_count == 0)
    {
      seqeditorctxt->canvas_selection_points = gnome_canvas_points_new (4);

      seqeditorctxt->canvas_selection_points->coords[0] = 
	seqeditorctxt->mnm1rect.x1;
      seqeditorctxt->canvas_selection_points->coords[1] = 
	seqeditorctxt->mnm1rect.y1;

      seqeditorctxt->canvas_selection_points->coords[2] = 
	seqeditorctxt->mnm2rect.x1;
      seqeditorctxt->canvas_selection_points->coords[3] = 
	seqeditorctxt->mnm2rect.y1;

      seqeditorctxt->canvas_selection_points->coords[4] = 
	seqeditorctxt->mnm2rect.x1;
      seqeditorctxt->canvas_selection_points->coords[5] = 
	seqeditorctxt->mnm2rect.y2;

      seqeditorctxt->canvas_selection_points->coords[6] = 
	seqeditorctxt->mnm1rect.x1;
      seqeditorctxt->canvas_selection_points->coords[7] = 
	seqeditorctxt->mnm2rect.y2;
   }

  if (sel_lines_count > 0)
    {
      seqeditorctxt->canvas_selection_points = gnome_canvas_points_new (8);

      seqeditorctxt->canvas_selection_points->coords[0] = 
	seqeditorctxt->mnm1rect.x1;
      seqeditorctxt->canvas_selection_points->coords[1] = 
	seqeditorctxt->mnm1rect.y1;

      seqeditorctxt->canvas_selection_points->coords[2] = end_line_EE_x;
      seqeditorctxt->canvas_selection_points->coords[3] = 
	seqeditorctxt->mnm1rect.y1;

      seqeditorctxt->canvas_selection_points->coords[4] = end_line_EE_x;
      seqeditorctxt->canvas_selection_points->coords[5] = 
	seqeditorctxt->mnm1rect.y1 +
	(sel_lines_count * seqeditorctxt->monicon_size);

      seqeditorctxt->canvas_selection_points->coords[6] = 
	seqeditorctxt->mnm2rect.x1;
      seqeditorctxt->canvas_selection_points->coords[7] = 
	seqeditorctxt->mnm1rect.y1 +
	(sel_lines_count * seqeditorctxt->monicon_size);;

      seqeditorctxt->canvas_selection_points->coords[8] = 
	seqeditorctxt->mnm2rect.x1;
      seqeditorctxt->canvas_selection_points->coords[9] = 
	seqeditorctxt->mnm1rect.y1 +
	(sel_lines_count * seqeditorctxt->monicon_size) 
	+ seqeditorctxt->monicon_size;

      seqeditorctxt->canvas_selection_points->coords[10] = LEFT_MARGIN;
      seqeditorctxt->canvas_selection_points->coords[11] = 
	seqeditorctxt->mnm1rect.y1 +
	(sel_lines_count * seqeditorctxt->monicon_size) 
	+ seqeditorctxt->monicon_size;

      seqeditorctxt->canvas_selection_points->coords[12] = LEFT_MARGIN;
      seqeditorctxt->canvas_selection_points->coords[13] = 
	seqeditorctxt->mnm1rect.y1
	+ seqeditorctxt->monicon_size;

      seqeditorctxt->canvas_selection_points->coords[14] = 
	seqeditorctxt->mnm1rect.x1;
      seqeditorctxt->canvas_selection_points->coords[15] = 
	seqeditorctxt->mnm1rect.y1
	+ seqeditorctxt->monicon_size;
    }

  seqeditorctxt->canvas_selection_polygon =
    gnome_canvas_item_new (seqeditorctxt->main_canvas_group,
			   gnome_canvas_polygon_get_type (),
			   "points", seqeditorctxt->canvas_selection_points,
			   "outline_color", "black",
			   "width_units", SEL_POLYGON_LINE_WIDTH,
			   NULL);

  /* PLEASE NOTE THAT LAST PARAM IS NON-NULL, it contains the
   * polymer editing context pointer.
   */
  g_signal_connect (G_OBJECT (seqeditorctxt->canvas_selection_polygon), 
		    "event",
		    G_CALLBACK (polyxedit_seqed_wnd_sel_polygon_event), 
		    seqeditorctxt);

  return TRUE;
}


void 
polyxedit_seqed_wnd_remove_selection_polygon (PxmSeqEditorCtxt *seqeditorctxt)
{
  seqeditorctxt->sel_mnm_idx1 = -1;
  seqeditorctxt->sel_mnm_idx2 = -1;
  
  if (seqeditorctxt->canvas_selection_points)
    {
      gnome_canvas_points_unref (seqeditorctxt->canvas_selection_points);
      seqeditorctxt->canvas_selection_points = NULL;
    }
  
  if (seqeditorctxt->canvas_selection_polygon)
    {
      gtk_object_destroy (GTK_OBJECT (seqeditorctxt->
				      canvas_selection_polygon));
      seqeditorctxt->canvas_selection_polygon = NULL;
    }
  
  seqeditorctxt->selection_rect.x1 = seqeditorctxt->selection_rect.x2;
  seqeditorctxt->selection_rect.y1 = seqeditorctxt->selection_rect.y2;
  
  return;
}
  

void 
polyxedit_seqed_wnd_setup_monomer_flag (PxmEditCtxt *editctxt)
{
  GtkWidget *widget [4] = {NULL, NULL, NULL, NULL};

  GtkWidget *hbox = NULL;
  
  GdkColor green_color;
  GdkColor black_color;

  GdkColormap *colormap;
  GdkGC *gc;

  gboolean success = FALSE;

  gint iter = 0;
  
  g_assert (editctxt != NULL);
  
  
  /* First get a pointer to the hbox that contains the three flag
     element's drawing areas (prop, note, modif). We'll get the colormap
     out of it. If that works ?
   */
  hbox = 
    g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
				 sequence_editor_wnd),
		       "flag_hbox");
  g_assert (hbox != NULL);

  colormap = gdk_drawable_get_colormap (hbox->window);
  gc = gdk_gc_new (hbox->window);


  /* All the flag's elements are going to be drawn in green (resting
     position).
   */

  widget [0] = g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
					    sequence_editor_wnd),
		       "prop_flag_drawingarea");

  widget [1] = g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
					    sequence_editor_wnd),
		       "note_flag_drawingarea");

  widget [2] = g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
					    sequence_editor_wnd),
		       "modif_flag_drawingarea");


  
  /* The green background filling the whole surface.
   */
  green_color.red = 25000;
  green_color.blue = 25000;
  green_color.green = 65535;
  
  gdk_colormap_alloc_colors (colormap,
			     &green_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);

  black_color.red = 0;
  black_color.blue = 0;
  black_color.green = 0;
  
  gdk_colormap_alloc_colors (colormap,
			     &black_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);

  for (iter = 0; iter < 3 ; iter++)
    {
      /* We want to redraw the background to green.
       */
      gdk_gc_set_foreground (gc, &green_color);
      
      gdk_draw_rectangle (widget[iter]->window,
			  gc,
			  TRUE /*filled*/,
			  0 /*gint x*/,
			  0 /*gint y*/,
			  widget[iter]->allocation.width/*gint width*/,
			  widget[iter]->allocation.height/*gint height*/);

      /* The mask around each flag.
       */
      gdk_gc_set_foreground (gc, &black_color);
      
      gdk_draw_rectangle (widget[iter]->window,
			  gc,
			  FALSE /*filled*/,
			  0 /*gint x*/,
			  0 /*gint y*/,
			  widget[iter]->allocation.width-1/*gint width*/,
			  widget[iter]->allocation.height-1/*gint height*/);
    }

  g_object_unref (G_OBJECT (gc));
}





void 
polyxedit_seqed_wnd_update_monomer_flag (PxmEditCtxt *editctxt)
{
  PxmPolymer *polymer = NULL;
  PxmMonomer *monomer = NULL;

  GtkWidget *widget [4] = {NULL, NULL, NULL, NULL};
  GtkWidget *hbox = NULL;
  
  GdkColor black_color;
  GdkColor green_color;
  GdkColor red_color;

  GdkColormap *colormap;
  GdkGC *gc;

  gboolean success = FALSE;

  gint iter = 0;
  gint index = -1;

  PxmProp *prop = NULL;



  g_assert (editctxt != NULL);
  
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);

  
  /* The mouse cursor has moved over a monomer icon, and we want to
     update the monomer status flag so that the status of that
     specific monomer is updated.

     First flag element: prop status

     Second flag element: note status
     
     Third flag element: modif status.
  */

  /* First of all, prepare background.
   */
  polyxedit_seqed_wnd_setup_monomer_flag (editctxt);

  /* If no monomer is currently pointed to by the mouse cursor, or the
     sequence is empty, like if we had just created a new one, we just
     return.
  */
  index = editctxt->seqeditorctxt->last_mouse_idx;

  /* Do not forget that last_mouse_idx is one unit greater than the
     actual index of the moused monomer. Also, and that's very
     important, make sure that the sequence is not 0-length. This is
     because, this function may be called when a brand new polymer
     sequence is created (thus empty) or an empty polymer sequence
     file is loaded or all the monomers of a polymer sequence have
     been removed... Below, we ask to get the monomer at position
     'index' of the polymer sequence... if the sequence is emtpy,
     that's a segfault.
  */
  if (polymer->monomerGPA->len == 0 
      || index == -1 || index > polymer->monomerGPA->len - 1)
    {
      return;
    }
  
  /* Get the monomer that lies below the cursor.
   */
  monomer = g_ptr_array_index (polymer->monomerGPA, 
			       index);
  
  g_assert (monomer != NULL);


  /* First get a pointer to the hbox that contains the three flag
     element's drawing areas (prop, note, modif). We'll get the colormap
     out of it. If that works ?
   */
  hbox = 
    g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
				 sequence_editor_wnd),
		       "flag_hbox");
  g_assert (hbox != NULL);

  colormap = gdk_drawable_get_colormap (hbox->window);
  gc = gdk_gc_new (hbox->window);


  /* All the flag's elements are going to be drawn in green (resting
     position).
   */

  widget [0] = g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
					    sequence_editor_wnd),
		       "prop_flag_drawingarea");

  widget [1] = g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
					    sequence_editor_wnd),
		       "note_flag_drawingarea");

  widget [2] = g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
					    sequence_editor_wnd),
		       "modif_flag_drawingarea");


  
  /* The different colors we are going to need.
   */
  black_color.red = 0;
  black_color.blue = 0;
  black_color.green = 0;
  
  gdk_colormap_alloc_colors (colormap,
			     &black_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);

  green_color.red = 25000;
  green_color.blue = 25000;
  green_color.green = 65535;
  
  gdk_colormap_alloc_colors (colormap,
			     &green_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);

  red_color.red = 65535;
  red_color.blue = 0;
  red_color.green = 0;

  gdk_colormap_alloc_colors (colormap,
			     &red_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);




  for (iter = 0; iter < 3 ; iter++)
    {
      if (iter == 0)
	{
	  /* We are handling the flag element that pertains to the
	     presence of at least a prop object, in the monomer being
	     worked on. So we just ask this question here. If the array
	     of prop objects is empty, we draw the flag element in green,
	     otherwise in red.
	  */

	  if (monomer->propGPA->len <= 0)
	    {
	      /* We want to draw a green flag element.
	       */
	      gdk_gc_set_foreground (gc, &green_color);
	    }
	  else
	    {
	      /* We want to draw a green flag element.
	       */
	      gdk_gc_set_foreground (gc, &red_color);
	    }

	  /* Now that we have the colors in the graphical context,
	     just draw the colored rectangle. After that we'll draw
	     the black outline.
	  */
	  gdk_draw_rectangle 
	    (widget[iter]->window,
	     gc,
	     TRUE /*filled*/,
	     0 /*gint x*/,
	     0 /*gint y*/,
	     widget[iter]->allocation.width/*gint width*/,
	     widget[iter]->allocation.height/*gint height*/);
	  
	  /* The mask around each flag.
	   */
	  gdk_gc_set_foreground (gc, &black_color);
	  
	  gdk_draw_rectangle 
	    (widget[iter]->window,
	     gc,
	     FALSE /*filled*/,
	     0 /*gint x*/,
	     0 /*gint y*/,
	     widget[iter]->allocation.width-1/*gint width*/,
	     widget[iter]->allocation.height-1/*gint height*/);
	      
	  continue;
	}
      /* end of 
	 if (iter == 0)
      */

      if (iter == 1)
	{
	  /* We are handling the flag element that pertains to the
	     presence of at least a note object, in the monomer being
	     worked on. So we just ask this question here. If the
	     array of prop objects does not contain any note prop
	     object, we draw the flag element in green, otherwise in
	     red.
	  */
	  prop = libpolyxmass_prop_find_prop (monomer->propGPA,
					  NULL,
					  NULL,
					  "NOTE",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	  if (prop == NULL)
	    {
	      /* We want to draw a green flag element.
	       */
	      gdk_gc_set_foreground (gc, &green_color);
	    }
	  else
	    {
	      /* We want to draw a green flag element.
	       */
	      gdk_gc_set_foreground (gc, &red_color);
	    }

	  /* Now that we have the colors in the graphical context,
	     just draw the colored rectangle. After that we'll draw
	     the black outline.
	  */
	  gdk_draw_rectangle 
	    (widget[iter]->window,
	     gc,
	     TRUE /*filled*/,
	     0 /*gint x*/,
	     0 /*gint y*/,
	     widget[iter]->allocation.width/*gint width*/,
	     widget[iter]->allocation.height/*gint height*/);
	  
	  /* The mask around each flag.
	   */
	  gdk_gc_set_foreground (gc, &black_color);
	  
	  gdk_draw_rectangle 
	    (widget[iter]->window,
	     gc,
	     FALSE /*filled*/,
	     0 /*gint x*/,
	     0 /*gint y*/,
	     widget[iter]->allocation.width-1/*gint width*/,
	     widget[iter]->allocation.height-1/*gint height*/);
	      
	  continue;
	}
      /* end of 
	 if (iter == 1)
      */

      if (iter == 2)
	{
	  /* We are handling the flag element that pertains to the
	     presence of at least a modif object, in the monomer being
	     worked on. So we just ask this question here. If the
	     array of prop objects does not contain any modif prop
	     object, we draw the flag element in green, otherwise in
	     red.
	  */
	  prop = libpolyxmass_prop_find_prop (monomer->propGPA,
					  NULL,
					  NULL,
					  "MODIF",
					  NULL,
					  PXM_UNDEF_PROP_CMP);
	  if (prop == NULL)
	    {
	      /* We want to draw a green flag element.
	       */
	      gdk_gc_set_foreground (gc, &green_color);
	    }
	  else
	    {
	      /* We want to draw a green flag element.
	       */
	      gdk_gc_set_foreground (gc, &red_color);
	    }

	  /* Now that we have the colors in the graphical context,
	     just draw the colored rectangle. After that we'll draw
	     the black outline.
	  */
	  gdk_draw_rectangle 
	    (widget[iter]->window,
	     gc,
	     TRUE /*filled*/,
	     0 /*gint x*/,
	     0 /*gint y*/,
	     widget[iter]->allocation.width/*gint width*/,
	     widget[iter]->allocation.height/*gint height*/);
	  
	  /* The mask around each flag.
	   */
	  gdk_gc_set_foreground (gc, &black_color);
	  
	  gdk_draw_rectangle 
	    (widget[iter]->window,
	     gc,
	     FALSE /*filled*/,
	     0 /*gint x*/,
	     0 /*gint y*/,
	     widget[iter]->allocation.width-1/*gint width*/,
	     widget[iter]->allocation.height-1/*gint height*/);
	}
      /* end of 
	 if (iter == 2)
      */
    }
  /* end of
     for (iter = 0; iter < 3 ; iter++)
  */
  g_object_unref (G_OBJECT (gc));
  
  return;
}




/*
****************************************************************
****** HELPER FUNCTIONS FOR HANDLING THE SEQUENCE DISPLAY ******
****************************************************************
*/
gboolean
polyxedit_seqed_wnd_monicon_get_pixel_coord (gint idx, 
					     PxmSeqEditorCtxt *seqeditorctxt, 
					     PxmRect *rect, 
					     PxmCoordsys coordsys1, 
					     PxmCoordsys coordsys2)
{
  gint pos_line = 0;
  gint line_num = 0;
  


  g_assert (seqeditorctxt != NULL);
  
  g_assert (rect != NULL);
  g_assert (idx >= 0);
  
  seqeditorctxt->monicons_per_line =
    (seqeditorctxt->canvas_scroll_wnd_width - seqeditorctxt->left_margin) 
    / seqeditorctxt->monicon_size;
  
  line_num = idx / seqeditorctxt->monicons_per_line;
  
  pos_line = idx - (line_num * seqeditorctxt->monicons_per_line);
  
  /* First compute the first point's coordinates.
   */
  if (coordsys1 == COORDSYS_CN) 
    /* top vertically, center horizontally*/
    {
      rect->x1 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size)
	+ (seqeditorctxt->monicon_size / 2);
      rect->y1 = line_num * seqeditorctxt->monicon_size;
    }
  if (coordsys1 == COORDSYS_NW)
    {
      rect->x1 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size);
      rect->y1 = line_num * seqeditorctxt->monicon_size;
    }
  else if (coordsys1 == COORDSYS_SW)
    {
      rect->x1 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size);
      rect->y1 = (line_num * seqeditorctxt->monicon_size) 
	+ seqeditorctxt->monicon_size;
    }
  else if (coordsys1 == COORDSYS_NE)
    {
      rect->x1 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size)
	+ seqeditorctxt->monicon_size;
      rect->y1 = line_num * seqeditorctxt->monicon_size;
    }
  else if (coordsys1 == COORDSYS_SE)
    {
      rect->x1 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size)
	+ seqeditorctxt->monicon_size;
      rect->y1 = (line_num * seqeditorctxt->monicon_size) 
	+ seqeditorctxt->monicon_size;
    }
  else if (coordsys1 == COORDSYS_CW)
    {
      rect->x1 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size);
      rect->y1 = (line_num * seqeditorctxt->monicon_size)
	+ (seqeditorctxt->monicon_size / 2);
    }
  else if (coordsys1 == COORDSYS_CE)
    {
      rect->x1 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size)
	+ seqeditorctxt->monicon_size;
      rect->y1 = (line_num * seqeditorctxt->monicon_size)
	+ (seqeditorctxt->monicon_size / 2);
    }



  /* Second compute the second point's coordinates.
   */
  if (coordsys2 == COORDSYS_CN) 
    /* top vertically, center horizontally*/
    {
      rect->x2 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size)
	+ (seqeditorctxt->monicon_size / 2);
      rect->y2 = line_num * seqeditorctxt->monicon_size;
    }
  if (coordsys2 == COORDSYS_NW)
    {
      rect->x2 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size);
      rect->y2 = line_num * seqeditorctxt->monicon_size;
    }
  else if (coordsys2 == COORDSYS_SW)
    {
      rect->x2 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size);
      rect->y2 = (line_num * seqeditorctxt->monicon_size) 
	+ seqeditorctxt->monicon_size;
    }
  else if (coordsys2 == COORDSYS_NE)
    {
      rect->x2 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size)
	+ seqeditorctxt->monicon_size;
      rect->y2 = line_num * seqeditorctxt->monicon_size;
    }
  else if (coordsys2 == COORDSYS_SE)
    {
      rect->x2 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size)
	+ seqeditorctxt->monicon_size;
      rect->y2 = (line_num * seqeditorctxt->monicon_size) 
	+ seqeditorctxt->monicon_size;
    }
  else if (coordsys2 == COORDSYS_CW)
    {
      rect->x2 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size);
      rect->y2 = (line_num * seqeditorctxt->monicon_size)
	+ (seqeditorctxt->monicon_size / 2);
    }
  else if (coordsys2 == COORDSYS_CE)
    {
      rect->x2 = 
	seqeditorctxt->left_margin + (pos_line * seqeditorctxt->monicon_size)
	+ seqeditorctxt->monicon_size;
      rect->y2 = (line_num * seqeditorctxt->monicon_size)
	+ (seqeditorctxt->monicon_size / 2);
    }

  return TRUE;
}


gint
polyxedit_seqed_wnd_get_mnm_idx_with_xy (PxmSeqEditorCtxt *seqeditorctxt, 
					 gdouble mouse_x, gdouble mouse_y)
{
  gint length = 0;
  gint line_count = 0;


  /* monicon_numb is the number (with fraction part) of the
   * icon in the row we are sitting on . For example, 2.25 means
   * we are on the third icon in the fourth part of it. We also
   * can formulate this : "we are one fourth of icon forward after icon
   * number 2 in a row".
   */
  gdouble monicon_numb_x = 0;
  gdouble fractional_part = 0;
  gdouble interger_part = 0;

  gint res_idx = 0;

  /* Easy to understand from code below.
   */
  gint monicon_numb_y = 0;
  gint icons_per_line = 0;



  g_assert (seqeditorctxt != NULL);

  length = seqeditorctxt->moniconGPA->len;
  

  /* First of all note that if the mouse_x value is less than the
     margin, then we should return -1. That -1 value should be tested
     against by the caller function to inhibit dangerous actions. 

     Equally, if mouse_y is less than zero, we should also return -1.
  */

  if (mouse_x < seqeditorctxt->left_margin)
    return -1;
  if (mouse_y < 0)
    return -1;
  
  
  /* The y dimension is easier than the x dimension.
   */
  icons_per_line = 
    (seqeditorctxt->canvas_scroll_wnd_width - seqeditorctxt->left_margin) 
    / seqeditorctxt->monicon_size;

  line_count = mouse_y / seqeditorctxt->monicon_size;

  monicon_numb_y = line_count * icons_per_line;

  /* If the mouse pointer is clicked in the last horizontal tenth part
   * of a given icon, the cursor will be drawn on the right of this
   * monomer icon. And the monomer index incremented by one.
   *
   * Note that this is important to allow, because it is the only
   * way we can artificially set the currently pointed monomer index
   * (thus also draw the cursor) right of last monomer of a polymer
   * sequence so that user has the feeling that she will append
   * monomer's to the end of the polymer sequence. 
   */
  monicon_numb_x = (mouse_x - seqeditorctxt->left_margin) 
    / seqeditorctxt->monicon_size;

  fractional_part = modf (monicon_numb_x, &interger_part);

  if (fractional_part > 0.9)
    ++interger_part;

  res_idx = (gint) interger_part + monicon_numb_y;

  /* Some easy checks, again, a bit paranoid, but this ensures
     consistency.
   */
  if (res_idx < 0)
    res_idx = -1 ;
  if (res_idx > length)
    res_idx = -1;

  return res_idx;
}


/*
****************************************************************
**************** SEQUENCE MODIFYING  FUNCTIONS *****************
****************************************************************
*/
gboolean
polyxedit_seqed_wnd_remove_monomer (PxmEditCtxt *editctxt, gint idx)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmPolymer *polymer = NULL;
  PxmMonomer *monomer = NULL;
  PxmMonicon *monicon = NULL;

  gint length = 0;

  /* Attention, this function does not elicit recomputation of masses,
     it only ensure that a monomer is removed both from the polymer
     sequence monomerGPA monomer instances, and from the polymer
     editing context moniconGPA monicon instances. The caller function
     MUST ensure that the masses are recomputed (selection and or
     whole, appropriately).
  */

  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);

  length = polymer->monomerGPA->len;

  /* Some easy checks:
   */
  if (length <= 0)
    return TRUE;

  /* Removing a monomer from a polymer involves a number of different
   * steps :
   * 1. check that index is < length;
   * 2. remove/free the monomer object from the GPtrArray of monomer
   * objects of the polymer object.
   * 3. remove/free the monicon object from the GPtrArray of 
   * monicon objects of the seqeditorctxt object.
   */

  g_assert (idx < length || idx >= 0);
  
  /* Now remove the monomer from the sequence GPtrArray of monomers.
   */
  monomer = g_ptr_array_remove_index (polymer->monomerGPA, idx);
  g_assert (monomer != NULL);
  pxmchem_monomer_free (monomer);

  /* Now remove the monicon from the sequence editor's GPtrArray of
   * monicons.
   */
  monicon = g_ptr_array_remove_index (seqeditorctxt->moniconGPA, idx);
  g_assert (monicon != NULL);

  polyxedit_monicon_free (monicon);
  
  return TRUE;
}


gint
polyxedit_seqed_wnd_remove_sequence_indices (PxmEditCtxt *editctxt,
					     gint start, gint end)
{
  gint iter = 0;
  gint length = 0;
  gint first = 0;
  gint last = 0;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  PxmPolymer *polymer = NULL;
  PxmMonomer *monomer = NULL;
  PxmMonicon *monicon = NULL;
  
  /* This function returns the last_point_1_idx value.
   */

  g_assert (editctxt != NULL);
  
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  
  length = editctxt->polymer->monomerGPA->len;

  /* The start and end indices passed as parameter may not be ordered,
     so that we have to put them in order, and after that is done
     ensure that their value is compatible with the polymer sequence
     array boundaries. 

     In any way, none of the passed value can be higher than (length -
     1) because we are dealing with indices.

     Note that start and end may be the same if only one monomer is
     currently selected !!!
  */

  g_assert (start <= length - 1 && start >= 0);
  g_assert (end <= length - 1 && end >= 0 );
  
  /* First step: make sure that start is less than end.
   */
  first = start < end ? start : end;
  last = start < end ? end : start;

  /* Now we are certain that (first <= last).
   */

  /* Second, make sure that we can access the monomers that
     are at the two indices start and end.
  */
  monomer = g_ptr_array_index (polymer->monomerGPA, first);
  g_assert (monomer != NULL);
    
  monomer = g_ptr_array_index (polymer->monomerGPA, last);
  g_assert (monomer != NULL);
  
  
  /* AND NOW: removing monomers from a polymer sequence 
   * involves a number of different steps :
   *
   * 1. remove/free the monomer object from the GPtrArray of monomers
   * of the polymer object.
   *
   * 2. remove/free the monicon object from the GPtrArray of 
   * monicon objects of the seqeditorctxt object.
   */

  /* Now remove the monomer objects from the sequence's GPtrArray of
   * monomers.
   */
  iter = last;

  while (iter >= first)
    {
      /* First remove the monomer from the polymer sequence's array
       * of monomers.
       */
      monomer = g_ptr_array_remove_index (polymer->monomerGPA, iter);
      
      g_assert (monomer != NULL);

      pxmchem_monomer_free (monomer);

      /* Next, remove the monomer icon from the sequence editor's
       * array of monicons.
       */
      monicon = g_ptr_array_remove_index (seqeditorctxt->moniconGPA, iter);

      g_assert (monicon != NULL);

      /* polyxedit_monicon_free bothers to call the gtk_object_destroy ()
       * on the moniconCanvasItem, which means the GdkPixbuf
       * is unref'ed right away
       */
      polyxedit_monicon_free (monicon);

      iter--;
    }

  /* And finally we kill the selection polygon, and ask that the cursor
   * be drawn at index 'first'. The function below will set to -1
   * both sel_mnm_idx1 and sel_mnm_idx2.
   */
  polyxedit_seqed_wnd_remove_selection_polygon (seqeditorctxt);

  /* Set the cursor position at the new index:
   */
  seqeditorctxt->last_point_1_idx = first;
  
  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);
  
  /* Redraw the whole sequence.
   */
  if (- 1 == polyxedit_seqed_wnd_redraw_sequence (editctxt))
    {
      g_critical (_("%s@%d: failed redrawing the sequence\n"),
	     __FILE__, __LINE__);
    }
    
  /*
  if (gtk_events_pending())
    gtk_main_iteration();
  */

  /* Return the present index, so that if this function was 
   * called as the first step in a replacement operation, the caller
   * knows at which index it should insert the replacement. This is 
   * particularly interesting if this function was called with 
   * monomer pointers in the p_seqcoords, and not indexes.
   */

  return seqeditorctxt->last_point_1_idx;
}



gint
polyxedit_seqed_wnd_remove_selected_oligomer (PxmEditCtxt *editctxt)
{
  gint start = -1;
  gint end = -1;

  gboolean result = FALSE;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmPolymer *polymer = NULL;


  g_assert (editctxt != NULL);

  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  /* It is absolutely required, if we get to this place, that there is
     a selection. This is necessary to ensure the internal consistency
     of the program. Thus the g_assert () call below.
  */
  result = 
    polyxedit_seqed_wnd_get_selection_indices (editctxt, &start, &end);
  
  g_assert (result == TRUE);
  
  /* As usual it must be noted that when selecting a portion of the
     polymer sequence, the indexes that are returned are OK for the
     start index but BAD for the end index. Indeed, this latter index
     is one unit larger (behaving like a position and not like an
     index. So we have to decrement by one in order to get a true
     index.
   */
  if (end > 0)
    end--;

  /* Now everything should work OK. The function below returns the
     location of the cursor after the operation has taken place.
   */  
  return polyxedit_seqed_wnd_remove_sequence_indices (editctxt,
						      start, end);
}


gboolean
polyxedit_seqed_wnd_integrate_monomer_at_idx (PxmMonomer *monomer,
					      gint idx,
					      PxmEditCtxt *editctxt)
{
  gint length = 0;

  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmPolymer *polymer = NULL;

  PxmMonicon *monicon = NULL;
  
  GdkPixbuf *pixbuf = NULL;


  g_assert (editctxt != NULL);

  g_assert (idx >= 0);
  

  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  
  length = polymer->monomerGPA->len;
  
  /* Attention, this function does not elicit recomputation of masses,
     it only ensure that a monomer is integrate both in the polymer
     sequence monomerGPA monomer instances, and in the polymer
     editing context moniconGPA monicon instances. The caller function
     MUST ensure that the masses are recomputed (selection and or
     whole, appropriately).
  */

  /* First ADD/INSERT the new monomer pointer to its monomerGPA.
   */

  /* -------- ATTENTION --------- that the index might be higher than
     the upper bound of the polymer sequence GPtrArray because we want
     to allow the cursor in the editor to be drawn RIGHT TO THE LAST
     MONOMERICON, which means at an index = max_index_possible + 1
     !!!!!!!!!! IN THIS PRECISE CASE WE DO NOT INSERT A NEW MONOMER
     POINTER, WE ADD IT !!!!
   */

  if (idx >= length)
    g_ptr_array_add (polymer->monomerGPA, monomer);
  else
    g_ptr_array_insert_val (polymer->monomerGPA, idx, monomer);

  /* It is about time to tell the sequence editor window that the 
     sequence is dirty:
   */
  polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);

  /* Now that we have this new monomer in the sequence we should
     integrate it into the moniconGPA also, otherwise the colinearity
     between the polymer->monomerGPA and the moniconGPA goes dead.
   */

  /* Render (or get a reference to an existing one) a pixbuf for the
     monomer's code.
   */
  pixbuf = polyxedit_rendering_pixbuf_render_no_modif (monomer->code,
						       seqeditorctxt->
						       monicon_size,
						       editctxt);

  if (pixbuf == NULL)
    {
      g_critical (_("%s@%d: failed rendering monicon for monomer '%s' "
	       "at index '%d'\n"),
	     __FILE__, __LINE__, monomer->code, idx);
      
      return FALSE;
    }
    
  /* Allocate a new monicon.
   */
  monicon = polyxedit_monicon_new ();

  if (FALSE == 
      polyxedit_rendering_monicon_make_canvas_item (monicon, 
						    seqeditorctxt->
						    monicon_size,
						    pixbuf,
						    idx,
						    editctxt))
    {
      g_critical (_("%s@%d: failed making canvas_item for monomer code '%s' "
	       "at index '%d'\n"),
	     __FILE__, __LINE__, monomer->code, idx);
      
      polyxedit_monicon_free (monicon);

      return FALSE;
    }

  /* At this point we have a fully qualified monicon.
   *   
   * -------- ATTENTION ---------    SAME AS ABOVE
   */
  if (seqeditorctxt->last_point_1_idx >= length)
    g_ptr_array_add (seqeditorctxt->moniconGPA, monicon);
  else
    g_ptr_array_insert_val (seqeditorctxt->moniconGPA, idx, 
			    monicon);

  polyxedit_seqed_wnd_ensure_select_cursor_visible (editctxt->
						    seqeditorctxt->
						    sequence_editor_wnd, 
						    NULL, 
						    editctxt);
  
  return TRUE;
}


gboolean
polyxedit_seqed_wnd_integrate_monomer_at_point (PxmMonomer *monomer,
						PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;


  g_assert (monomer != NULL);
  
  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
    
  /* Attention, this function does not elicit recomputation of masses,
     it only ensure that a monomer is integrate both in the polymer
     sequence monomerGPA monomer instances, and in the polymer
     editing context moniconGPA monicon instances. The caller function
     MUST ensure that the masses are recomputed (selection and or
     whole, appropriately).
  */
 
  /* All we want is to insert/add the monomer at the current cursor point
     location:
  */
  if (FALSE == 
      polyxedit_seqed_wnd_integrate_monomer_at_idx (monomer,
						    seqeditorctxt->
						    last_point_1_idx,
						    editctxt))
    {
      g_critical (_("%s@%d: failed integrating monomer '%s' at point\n"),
	     __FILE__, __LINE__, monomer->code);
      
      return FALSE;
    }

  /* Since a monomer was inserted at point, the coordinates of
     that point have to be incremented by one.
  */
  seqeditorctxt->last_point_1_idx++;

  return TRUE;
}


gint
polyxedit_seqed_wnd_integrate_sequence_at_point (GPtrArray *GPA,
						 PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmMonomer *monomer = NULL;
  PxmMonomer *monomer_new = NULL;
  
  PxmRect rect;
  
  gint iter = 0;
  

  /* Attention, this function does not elicit recomputation of masses,
     it only ensure that a monomer is integrate both in the polymer
     sequence monomerGPA monomer instances, and in the polymer
     editing context moniconGPA monicon instances. The caller function
     MUST ensure that the masses are recomputed (selection and or
     whole, appropriately).
  */


  /* We get an array of PxmMonomer instances. That array should be
     copied into the polseqctx->polymer->monomerGPA array at the
     precise cursor point.
  */

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  



  g_assert (GPA != NULL);
  /* If the array of monomer that we should integrate in the sequence
     is empty, then we return the number of successfully integrated
     monomer: 0!
  */
  if (GPA->len <= 0) return 0;
  

  /* OK, let's first check if some sequence portion is selected right
     now. If so, that means that we'll first have to remove that
     sequence portion.
  */
  if (TRUE == 
      polyxedit_seqed_wnd_get_selection_indices (editctxt, NULL, NULL))
    {
      if (-1 == polyxedit_seqed_wnd_remove_selected_oligomer (editctxt))
	{
	  g_critical (_("%s@%d: failed removing the selected sequence portion.\n"),
		 __FILE__, __LINE__);
	  
	  return -1;
	}
    }
  

  /* OK, apparently everything is working, we can set each monomer
     object to its relevant place in the polymer sequence.
   */
  for (iter = 0; iter < GPA->len; iter++)
    {
      monomer = g_ptr_array_index (GPA, iter);
      g_assert (monomer != NULL);
  

      /* Make a copy of that monomer, because we do not know if the
	 monomers in the GPA were allocated for us or not. Note that
	 the duplication function is DEEP, i.e. it will duplicate also
	 the array of properties of the source monomer into the
	 destination monomer_new. The function below will make sure
	 that the variable describing the cursor point location is
	 incremented by one each time a new monomer is inserted in the
	 sequence.
      */
      monomer_new = pxmchem_monomer_dup (monomer, PXM_MONOMER_DUP_DEEP_YES);
      
      if (FALSE ==
	  polyxedit_seqed_wnd_integrate_monomer_at_point (monomer_new,
							  editctxt))
	{
	  g_critical (_("%s@%d: failed integrating the sequence element.\n"),
		 __FILE__, __LINE__);
	  
	  return -1;
	}
    }
  
  /* We should make sure that the insertion point is systematically
   * visible, even if we are inserting data on the lowest part of the
   * sequence editor window.
   */
  polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
					       last_point_1_idx,
					       seqeditorctxt,
					       &rect,
					       COORDSYS_NW, COORDSYS_NW);

  gnome_canvas_get_scroll_offsets (GNOME_CANVAS (seqeditorctxt->
						 sequence_canvas),
				   &(seqeditorctxt->x_offset),
				   &(seqeditorctxt->y_offset));
  
  if (rect.y1 
      >= seqeditorctxt->y_offset 
      + seqeditorctxt->monicon_size 
      + seqeditorctxt->canvas_scroll_wnd_height
      - - seqeditorctxt->monicon_size)
    {
      /* We are near the bottom of viewable sequence, so we ask
       * for a scroll to happen towards higher monomer indexes
       * (scroll downwards).
       */
      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->sequence_canvas),
			      seqeditorctxt->x_offset,
			      seqeditorctxt->y_offset +
			      seqeditorctxt->monicon_size);
    }
      
  polyxedit_seqed_wnd_redraw_sequence (editctxt);
  
  return iter + 1;
}






gint
polyxedit_seqed_wnd_export_sound_playlist (PxmEditCtxt *editctxt)
{
  GtkWidget *window = NULL;
  
  PxmPolchemdefCtxt *polchemdefctxt = NULL;

  /* We are asked to show a window that contains all the
     configurations for the user to tell what should be exported to
     the sound playlist file.
  */



  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);


  
  window = polyxedit_self_speak_wnd_setup (editctxt);
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed setting-up the self-speak config window\n"),
	     __FILE__, __LINE__);
      
      return -1;
    }

  return 1;
}



gboolean
polyxedit_seqed_wnd_select_sequence (PxmEditCtxt *editctxt,
				     gint start_idx, gint end_idx)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  PxmRect rect;

  gint length = 0;
  

  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  length = editctxt->polymer->monomerGPA->len;
  
  /* A polymer sequence must actually exist.
   */
  if (length < 1)
    return TRUE;

  /* Note that if any of the two index parameters are -1, then that
     means that the whole sequence should be selected.
  */
  if (start_idx == -1 || end_idx == -1)
    {
      seqeditorctxt->sel_mnm_idx1 = 0;
      seqeditorctxt->sel_mnm_idx2 = length;
    }
  else
    {
      /* Now we should make some tests to ensure that we are not
	 accessing monomers outside of the boundaries of the polymer
	 sequence. Remember that we have to increment the end_idx
	 value by one so that the selection routines actually work as
	 expected.
      */
      g_assert (start_idx < length);
      g_assert (end_idx + 1 <= length);
      
      seqeditorctxt->sel_mnm_idx1 = start_idx;
      seqeditorctxt->sel_mnm_idx2 = end_idx + 1;
    }

  if (TRUE == 
      polyxedit_seqed_wnd_update_sel_polygon (editctxt->seqeditorctxt))
    {
      /* Update the  selection masses, since the selection has changed.
       */
      polyxedit_ui_masses_display_wnd_update_selection_masses
	((PxmEditCtxt *) polyxedit_last_editctxt);


      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx1,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x1 = rect.x1;
      seqeditorctxt->selection_rect.y1 = rect.y1;
      
      polyxedit_seqed_wnd_monicon_get_pixel_coord (seqeditorctxt->
						   sel_mnm_idx2,
						   seqeditorctxt,
						   &rect,
						   COORDSYS_NW, 
						   COORDSYS_NW);
      
      seqeditorctxt->selection_rect.x2 = rect.x1;
      seqeditorctxt->selection_rect.y2 = rect.y1;  



      /* Reposition the cursor and redraw it !
       */
      seqeditorctxt->last_point_1_idx = end_idx + 1;
      polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);
  
      /* Since we decided above that we would select a whole portion
	 of the sequence, from top to bottom of the sequence, we must
	 make sure that the cursor is visible and that the horizontal
	 bottom line of the selection polygon is visible at the bottom
	 of the sequence display area.
      */
      polyxedit_seqed_wnd_ensure_select_cursor_visible (seqeditorctxt->
							sequence_editor_wnd,
							NULL,
							editctxt);


      /* And now make sure that we claim ownership of the X PRIMARY
	 selection.
      */
      polyxedit_seqed_clipboard_primary_copy (editctxt);      
    }
    
  return TRUE;
}










/*
****************************************************************
**************** VARIOUS  UTILITY   FUNCTIONS ******************
****************************************************************
*/
gboolean
polyxedit_seqed_wnd_get_selection_indices (PxmEditCtxt *editctxt,
					   gint *start, gint *end) 
     
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);


  /* A selection must actually exist.
   */

  /* Non-selection is stated in the editor by setting any of the
   * two members seqeditorctxt->sel_mnm_idx1 and seqeditorctxt->sel_mnm_idx2
   * to -1.
   */
  if (seqeditorctxt->sel_mnm_idx1 == -1 
       || seqeditorctxt->sel_mnm_idx2 == -1)
    return FALSE;

  /* We want that the selection be ordered, which means that we need
   * that the start of the selection be less than or equal to the end
   * of the selection.
   */
  if (start != NULL)
    {
      if (seqeditorctxt->sel_mnm_idx1 > seqeditorctxt->sel_mnm_idx2)
	*start = seqeditorctxt->sel_mnm_idx2 ;
      else
	*start = seqeditorctxt->sel_mnm_idx1 ;
    }
  if (end != NULL)
    {
      if (seqeditorctxt->sel_mnm_idx1 > seqeditorctxt->sel_mnm_idx2)
	*end = seqeditorctxt->sel_mnm_idx1;
      else
	*end = seqeditorctxt->sel_mnm_idx2;
    }
  
  /*
    if (start != NULL && end != NULL)
    debug_printf (("polyxedit_seqed_wnd_get_selection_indices:" "
    [%d--%d]", *start, *end));
  */
  
  return TRUE;
}


PxmMonomer *
polyxedit_seqed_wnd_get_selected_monomer (PxmEditCtxt *editctxt,
					  gint *idx)
{
  gint index = 0;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  /* We return a valid pointer ONLY if a single monomer is currently
   * selected. Otherwise we return NULL.
   */

  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  g_assert (editctxt->polymer->monomerGPA != NULL);
  
  if (editctxt->polymer->monomerGPA->len <= 0)
    return NULL;
    
  /* If no selection: return NULL.
   */
  if (seqeditorctxt->sel_mnm_idx1 == -1 || seqeditorctxt->sel_mnm_idx2 == -1)
    return NULL;
  
  /* If more than one monomer selected: return NULL.
   */
  if (abs (seqeditorctxt->sel_mnm_idx2 -seqeditorctxt->sel_mnm_idx1) != 1)
    return NULL;
  
  /* The index of the monomer that is valid is the lowest of the two
   * (we do not know if the selection was made from highest to lowest
   * monomer indexes).
   */
  index =
    seqeditorctxt->sel_mnm_idx1 < seqeditorctxt->sel_mnm_idx2 ?
    seqeditorctxt->sel_mnm_idx1 : seqeditorctxt->sel_mnm_idx2;
  
  /* Sanity check, the selected monomer cannot be out of boundaries of 
   * the array that contains the polymer sequence:
   */
  g_assert (index < editctxt->polymer->monomerGPA->len);
  
  if (idx != NULL)
    *idx = index;
  
  return g_ptr_array_index (editctxt->polymer->monomerGPA, index);
}

  





/*
****************************************************************
**************** E V E N T  HANDLING FUNCTIONS *****************
****************************************************************
*/

/* SEQEDITOR WINDOW ******************************
 */
void
polyxedit_seqed_wnd_sequence_data_button_clicked (GtkWidget *widget,
						  gpointer data)
{
  /* Parameter data is editctxt.
   */
  PxmEditCtxt *editctxt = data;
  
  GtkWidget *vbox = NULL;

  g_assert (widget != NULL);
  g_assert (editctxt != NULL);
  
  /* Get the vertical box where the polymer sequence data displaying
     widgets are packed.
  */
  vbox = g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
				      sequence_editor_wnd),
			    "polymer_sequence_data_vbox");
  g_assert (vbox != NULL);

  /* If the vbox is visible we make it invisible, and vice-versa.
   */
  if (GTK_WIDGET_VISIBLE (vbox))
    gtk_widget_hide (vbox);
  else
    gtk_widget_show (vbox);
  
  return ;
}


void
polyxedit_seqed_wnd_editing_feedback_button_clicked (GtkWidget *widget,
						     gpointer data)
{
  /* Parameter data is editctxt.
   */
  PxmEditCtxt *editctxt = data;
  
  GtkWidget *hbox = NULL;

  g_assert (widget != NULL);
  g_assert (editctxt != NULL);
  
  /* Get the horizontal box where the editing feedback data displaying
     widgets are packed.
  */
  hbox = g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
				      sequence_editor_wnd),
			    "editing_feedback_hbox");
  g_assert (hbox != NULL);
  
  /* If the hbox is visible we make it invisible, and vice-versa.
   */
  if (GTK_WIDGET_VISIBLE (hbox))
    gtk_widget_hide (hbox);
  else
    gtk_widget_show (hbox);
  
  return ;
}


void
polyxedit_seqed_wnd_lr_end_modif_button_clicked (GtkWidget *widget,
						  gpointer data)
{
  /* Parameter data is editctxt.
   */
  PxmEditCtxt *editctxt = data;
  
  GtkWidget *hbox = NULL;

  g_assert (widget != NULL);
  g_assert (editctxt != NULL);
  
  /* Get the vertical box where the polymer sequence data displaying
     widgets are packed.
  */
  hbox = g_object_get_data (G_OBJECT (editctxt->seqeditorctxt->
				      sequence_editor_wnd),
			      "left_right_ends_modif_hbox");
  g_assert (hbox != NULL);

  /* If the hbox is visible we make it invisible, and vice-versa.
   */
  if (GTK_WIDGET_VISIBLE (hbox))
    gtk_widget_hide (hbox);
  else
    gtk_widget_show (hbox);
  
  return ;
}


void
polyxedit_seqed_wnd_monicon_size_value_changed (GtkWidget *range,
						gpointer data)
{
  /* The user asks that the size of the monicons used to display the
     sequence be changed. We thus have to first destroy all the
     monicon objects in the array of monicon objects for the current
     sequence editor context and recreate them fully from disk.
  */

  PxmEditCtxt *editctxt = data;
  
  gint new_size = 0;

  g_assert (range != NULL);
  g_assert (editctxt != NULL);
  
  new_size = (gint) gtk_range_get_value (GTK_RANGE (range));
  
  editctxt->seqeditorctxt->monicon_size = new_size;
  
  polyxedit_seqed_wnd_render_polseq_monicons (editctxt, TRUE);
  
  /* Rendering of the cursor cannot fail, or an error is issued with
   * following stop of the program.
   */
  polyxedit_rendering_cursor_monicon_render (editctxt, TRUE);
  
  if (-1 == polyxedit_seqed_wnd_redraw_sequence (editctxt))
    {
      g_warning (_("%s@%d: failed redrawing the sequence\n"),
		 __FILE__, __LINE__);
      
      return;
    }
  
  polyxedit_seqed_wnd_update_sel_polygon (editctxt->seqeditorctxt);
  polyxedit_seqed_wnd_draw_cursor (editctxt->seqeditorctxt);
  polyxedit_seqed_wnd_ensure_select_cursor_visible (editctxt->
						    seqeditorctxt->
						    canvas_scroll_wnd, 
						    NULL, 
						    editctxt);
  return ;
}



gboolean 
polyxedit_seqed_sw_size_allocate (GtkWidget *widget, 
				  GtkAllocation *allocation,
				  gpointer data)
{
  /* This function is meant to provide to the sequence editor context
   * the dimensions of the scrolled window "real time". We cannot just
   * try to get the dimensions of the window, because the scrolled
   * window that contains the gnome canvas is not the only widget in
   * the window.
   */

  /* Strangely, this function seems to get called even when the mouse
   * is not resizing the window, but for example, simply going from
   * one monomer canvas item to another. This means that there is a huge
   * overhead I want to eliminate : I first check if the size
   * of the window effectively changed. 
   *
   * If not: I return.
   *
   * If yes: I grasp the new dimensions, put them to proper variables
   * and also set to TRUE a file-scope variable that will let the
   * "after-connected handler" know that the scrolled window size
   * effectively changed.  (see
   * polyxedit_seqed_wnd_ensure_select_cursor_visible () for details.
   */
  PxmEditCtxt *editctxt = (PxmEditCtxt *) data;
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  /* This function is called (I do not know why) each time the mouse
     cursor goes from a monicon to another one. Of course I don't want
     this, so I only update the seqeditorctxt->canvas_scroll_wndwin
     dimensions if they are changed.
   */
  if ((seqeditorctxt->canvas_scroll_wnd_width != allocation->width)
      || (seqeditorctxt->canvas_scroll_wnd_height != allocation->height))
    {
      /* If the width of the window goes less than
       * (seqeditorctxt->left_margin + seqeditorctxt->monicon_size) it makes
       * a segfault, so prevent this behaviour.
       */
      if (allocation->width > 
	  (seqeditorctxt->left_margin + seqeditorctxt->monicon_size))
	{
	  seqeditorctxt->canvas_scroll_wnd_width = allocation->width;
	}
      
      seqeditorctxt->canvas_scroll_wnd_height = allocation->height;

      scroll_size_changed = TRUE;

      if (-1 == polyxedit_seqed_wnd_redraw_sequence (editctxt))
	{
	  g_warning (_("%s@%d: failed redrawing the sequence\n"),
		     __FILE__, __LINE__);
	  
	  return TRUE;
	}

      polyxedit_seqed_wnd_update_sel_polygon (editctxt->seqeditorctxt);
      polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);
    }

  return TRUE;
}




gboolean
polyxedit_seqed_wnd_focus_in_event (GtkWidget *widget, 
				    GdkEventFocus *event,
				    gpointer data)
{
  /* The data parameter contains a pointer to the editctxt.  We just
   * want to store this pointer to the global
   * polyxedit_last_polseq_ctxt data, in order to know at each instant
   * what editctxt is having the focus.
   */
  polyxedit_last_editctxt = data;

  /* Since the window that has the focus has changed we want to elicit
     a refresh of the masses displayed in the massdata windows.  But
     first we want that the window knows what protein the masses are
     displayed of.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_data 
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  /* The polymer sequence.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_masses 
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  /* The selection.
   */
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);

  return TRUE;
}


gboolean
polyxedit_seqed_wnd_canvas_event (GtkWidget *widget,
				  GdkEvent *event,
				  gpointer data)
{
  GtkMenu *canvas_menu = NULL;
  
  gdouble click_x = 0;
  gdouble click_y = 0;

  PxmEditCtxt *editctxt =  (PxmEditCtxt *) data;

  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  gchar *label = NULL;

  gint idx = 0;
  
  PxmRect rect;
    
  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  /* store the position of the mouse event
   */
  click_x = event->button.x;
  click_y = event->button.y;

  gnome_canvas_window_to_world (GNOME_CANVAS (widget),
				event->button.x,
				event->button.y,
				&click_x,
				&click_y);


  /* Analyze the event and produce the effects if required.
   */
  switch (event->type)
    {
    case GDK_BUTTON_PRESS:
      
      switch (event->button.button)
	{
	case 1:
	  
	  if (kbd_shift_down == TRUE)
	    {
	      /* Compute the index of the monicon that was clicked, in
		 order to know at each instant where the "current
		 position" is in the polymer sequence. If -1 is
		 returned, that means that the mouse cursor was not
		 onto a monomer icon (in the margins of the sequence
		 drawing area, thus we just return).
	       */
	      idx = 
		polyxedit_seqed_wnd_get_mnm_idx_with_xy (seqeditorctxt,
							 click_x,
							 click_y);
	      if (-1 == idx)
		return FALSE;

	      seqeditorctxt->last_point_1_idx = idx;

	      /* Usually, in text editors, when the cursor is somewhere
	       * and that the user presses the shift key while clicking 
	       * the mouse in some other place, it is considered that 
	       * the user wanted to select the text between the first 
	       * cursor position and the newly pointed position. 
	       * This is what we do here.
	       */
	      seqeditorctxt->selection_rect.x2 = click_x;
	      seqeditorctxt->selection_rect.y2 = click_y;
	      
	      
	      /*
		debug_printf (("the 1_idx is %d\n", 
		seqeditorctxt->last_point_1_idx));
	      */
	    }
	  else
	    {
	      /* Compute the index of the monicon that was clicked, in
		 order to know at each instant where the "current
		 position" is in the polymer sequence. If -1 is
		 returned, that means that the mouse cursor was not
		 onto a monomer icon (in the margins of the sequence
		 drawing area, thus we just return).
	       */
	      idx = 
		polyxedit_seqed_wnd_get_mnm_idx_with_xy (seqeditorctxt,
							 click_x,
							 click_y);
	      if (-1 == idx)
		return FALSE;
	      
	      seqeditorctxt->last_point_1_idx = idx;
		
	      /* Left mouse button is clicked without ongoing
	       * selection, so store the impact point's coordinates
	       * into the seqeditorctxt's selrect first selpoint member
	       * data.
	       */
	      seqeditorctxt->selection_rect.x1 = click_x;
	      seqeditorctxt->selection_rect.y1 = click_y;
	      
	      seqeditorctxt->selection_rect.x2 = click_x;
	      seqeditorctxt->selection_rect.y2 = click_y;

	      /*
		debug_printf (("the 1_idx is %d\n", 
		seqeditorctxt->last_point_1_idx));
	      */
	    }
	  
	  /* The selection polygon drawing function below will compute
	     the indices of the currently selected first and last
	     monomers.
	  */
	  polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);
	    
	  /* Since we know where the impact occurred from a monicon
	   * index standpoint, we can ask that the cursor be drawn at
	   * this precise position. Drawing of the cursor can happen
	   * thanks to updating the seqeditorctxt->last_point_1_idx datum,
	   * which is then used to compute precise pixel position for
	   * the cursor.
	   */
	  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);
	  

	  /* The selection.
	   */
	  polyxedit_ui_masses_display_wnd_update_selection_masses
	    ((PxmEditCtxt *) polyxedit_last_editctxt);
	  

	  /* We might well be starting a mouse_dragging operation, if the
	   * user wants to select a portion of the sequence. Dragging
	   * will be set to FALSE as soon as the left button is
	   * released.
	   */
	  mouse_dragging = TRUE;
	  break;

	case 2:
	  /* We are asking that a paste be done with the contents of
	     the PRIMARY selection clipboard, the X traditional
	     selection. What we do here is acutally ask for that
	     data. We are not pasting that. The real pasting activity
	     will be performed when the data arrives to our process,
	     and at that time only the "selection received" signal
	     will be sent to our process. It is the handler of that
	     signal (that we have coded ourselves) that will actually
	     do the pasting with the received data. (Remember that the
	     selection/paste stuff in X (traditional) is asynchronous,
	     because X may be distributed across the network, which
	     may have some latency.)
	  */
	  polyxedit_seqed_clipboard_primary_paste (editctxt);
	  break;
	  
	case 3:
	  /* We'll maybe need the monomer index of the monomer that
	     lies beneath the mouse cursor right now, so compute that
	     right away.
	  */
	  rect.x1 = click_x;
	  rect.y1 = click_y;
	  
	  /* Compute the index of the monicon that was clicked, in
	     order to know at each instant where the "current
	     position" is in the polymer sequence. For this specific
	     case, we are OK if -1 is returned.
	  */
	  seqeditorctxt->last_point_3_idx =
	    polyxedit_seqed_wnd_get_mnm_idx_with_xy (seqeditorctxt,
						     rect.x1,
						     rect.y1);
	  
	  /*
	    debug_printf (("the 3_idx is %d\n", 
	    seqeditorctxt->last_point_3_idx));
	  */

	  /* We want to display a contextual menu now.
	   */
	  canvas_menu = 
	    (GtkMenu *) g_object_get_data (G_OBJECT (seqeditorctxt->
						     sequence_canvas), 
					   "canvas_menu");
	  g_assert (canvas_menu != NULL);
	  
	  /* The 1 below, as sixth param is to tell that the first button
	     of the mouse is going to be used to trigger the "activate"
	     signal of the menu item to be selected.
	  */
	  gtk_menu_popup (canvas_menu, NULL, NULL, NULL, NULL, 
			  1  /* ((GdkEventButton *)event)->button */, 
			  ((GdkEventButton *)event)->time);
	  break;
	}
      /* end of 
	 switch (event->button.button)
      */
      break;

      
    case GDK_MOTION_NOTIFY:
      /* It is interesting to trap this message:
       *
       * 1. if we are pressing the left mouse button, in which 
       * case the "mouse_dragging" variable was set to TRUE.
       *
       * 2. because we just want to update the index of the last
       * monomer onto which the mouse was passed over.
       */

      if (mouse_dragging && (event->motion.state & GDK_BUTTON1_MASK))
	{
	  /* Compute the monicon index where we are mouse_dragging the mouse
	     so that later the cursor drawing function can do its job
	     by computing precisely the cursor position from the
	     seqeditorctxt->last_point_1_idx value. If -1 is returned,
	     that means that the mouse cursor was not onto a monomer
	     icon (in the margins of the sequence drawing area, thus
	     we just return).
	   */
	  idx = 
	    polyxedit_seqed_wnd_get_mnm_idx_with_xy (seqeditorctxt,
						     click_x,
						     click_y);
	  if (-1 == idx)
	    return FALSE;
	  

	  seqeditorctxt->last_point_1_idx = idx;
	  
	  /* Update the mouse cursor position because we need to use
	   * new position to draw the selection polygon.
	   */
	  seqeditorctxt->selection_rect.x2 = click_x;
	  seqeditorctxt->selection_rect.y2 = click_y;

	  /* Draw the cursor, so that we see the movement of the mouse
	   * real time. This function takes as parameter the pointer
	   * to the polymer sequence context that we got in this
	   * function as param "data".
	   */
	  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);
	  
	  /* Draw the selection polygon, by specifying in the unique
	   * parameter a pointer to the polymer sequence context,
	   * where the called function can get values to guide its
	   * polygon drawing process.
	   */
	  polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);
	  
	  /* Deal with scrolling of the window if pointer approaches
	   * the top/bottom borders of the window and that scrolling
	   * is allowed.
	   */
	  gnome_canvas_get_scroll_offsets (GNOME_CANVAS (seqeditorctxt->
							 sequence_canvas),
					   &(seqeditorctxt->x_offset),
					   &(seqeditorctxt->y_offset));

	  if (click_y >= seqeditorctxt->y_offset
	      + seqeditorctxt->canvas_scroll_wnd_height - 
	      seqeditorctxt->monicon_size)
	    {
	      /* We are near the bottom of viewable sequence, so we
	       * ask for a scroll to happen toward higher monomer
	       * index (scroll downward).
	       */
	      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->
						    sequence_canvas),
				      seqeditorctxt->x_offset,
				      seqeditorctxt->y_offset +
				      seqeditorctxt->monicon_size);
	    }
	  else if (click_y <= seqeditorctxt->y_offset
		   + seqeditorctxt->monicon_size / 3)
	    {
	      /* We are near the top of viewable sequence, so we ask
	       * for a scroll to happen toward lower monomer index
	       * (scroll upward).
	       */
	      gnome_canvas_scroll_to (GNOME_CANVAS (seqeditorctxt->
						    sequence_canvas),
				      seqeditorctxt->x_offset,
				      seqeditorctxt->y_offset -
				      seqeditorctxt->monicon_size);
	    }

	  /* The selection.
	   */
	  polyxedit_ui_masses_display_wnd_update_selection_masses
	    ((PxmEditCtxt *) polyxedit_last_editctxt);

	}
      
      seqeditorctxt->last_mouse_idx =
	polyxedit_seqed_wnd_get_mnm_idx_with_xy (seqeditorctxt, 
						 click_x, click_y);

      /* Make sure that we have an update of the flags that show the
	 status of the moused monomer.
      */
      if (seqeditorctxt->last_mouse_idx != -1)
	polyxedit_seqed_wnd_update_monomer_flag (editctxt);

      break;
      
    case GDK_BUTTON_RELEASE:
      
      switch (event->button.button)
	{
	case 1:
	  
	  /* Compute the monicon index where we are unclicking the mouse
	   * so that later the cursor drawing function can do its job by
	   * computing precisely the cursor position from the
	   * seqeditorctxt->last_point_1_idx value. If -1 is returned,
	     that means that the mouse cursor was not onto a monomer
	     icon (in the margins of the sequence drawing area, thus
	     we just return).
	   */
	  idx =
	    polyxedit_seqed_wnd_get_mnm_idx_with_xy (seqeditorctxt,
						     click_x,
						     click_y);
	  if (-1 == idx)
	    return FALSE;

	  /* However, that is not enough. Imagine that the user left
	     clicked the mouse in the margins and dragged it still
	     clicked to the sequence. At that moment she would release
	     the mouse button. We are here. The value computed above
	     is not going to be -1 because we are IN the sequence
	     drawing area. But we started mouse_dragging the mouse out of
	     the sequence area. So the system is crippled because we
	     are going to ask that a selection polygon be drawn, but
	     we do not know from where !
	  */

	  seqeditorctxt->last_point_1_idx = idx;
	  
	  /* Store the position of the unclicking of the mouse button
	   * so that we always know where we are in the polymer sequence.
	   */
	  seqeditorctxt->selection_rect.x2 = click_x;
	  seqeditorctxt->selection_rect.y2 = click_y;


	  /*
	    debug_printf (("the 1_idx is %d\n", 
	    seqeditorctxt->last_point_1_idx));
	  */
	  
	  /* Draw the cursor, so that we see the last position of the
	   * mouse real time. This function takes as parameter the pointer
	   * to the polymer context that we got in this function as param
	   * "data".
	   */
	  polyxedit_seqed_wnd_draw_cursor (seqeditorctxt);

	  /* Draw the selection polygon, because this way, if one exists
	   * and we are clicking somewhere else, we can destroy it.
	   */
	  polyxedit_seqed_wnd_draw_sel_polygon (seqeditorctxt);
      
	  /* Set "mouse_dragging" to FALSE to clearly indicate that
	   * the mouse was released and mouse_dragging (if any) operation is
	   * terminated.
	   */
	  mouse_dragging = FALSE;


	  /* The selection.
	   */
	  polyxedit_ui_masses_display_wnd_update_selection_masses
	    ((PxmEditCtxt *) polyxedit_last_editctxt);


	  /* We can automatically make the selection owner...  This will
	     allow next putting on the primary clipboard the sequence
	     corresponding to the selection (if any only).
	  */
	  polyxedit_seqed_clipboard_primary_copy (editctxt) ;
	  break;

	default:
	  break;
	}
      /* end of 
	 switch (event->button.button)
      */


    default:
      break;
    }
  /* end 
     switch (event->type)
  */
      
  /* Set the monomer index that was "moused" (passed over with the
     mouse) to the lable sitting on top of the sequence
     editor. Attention we want monomer positions and not monomer
     indexes, which is why we add 1 before converting to string. Only
     if the index is > -1. (Remember that -1 indicates that the cursor
     is in the margin of the sequence display area.) That's true for
     all positions but not the virtual monomer that is right of the
     very last monomer of the sequence. This is why we do not display
     anything if the last_mouse_idx is greater than the index of the
     last monomer.
   */
  if (seqeditorctxt->last_mouse_idx > -1
      && seqeditorctxt->last_mouse_idx < editctxt->polymer->monomerGPA->len)
    label = g_strdup_printf ("%d", seqeditorctxt->last_mouse_idx + 1);
  else
    label = g_strdup (" ");

  gtk_label_set_text (GTK_LABEL(seqeditorctxt->monomer_pos_label),
		      label);
  g_free (label);
  
  /* Let the canvas do the actual pixel graphics redrawing work.
   */
  return FALSE;
}


gboolean
polyxedit_seqed_wnd_sel_polygon_event (GtkWidget *widget,
				       GdkEvent *event,
				       gpointer data)
{
  /*
    debug_printf (("polyxedit_seqed_wnd_sel_polygon_event\n"));
  */
  
  return FALSE;
}


gboolean
polyxedit_seqed_wnd_canvas_item_event (GnomeCanvasItem *canvas_item,
				       GdkEvent *event, gpointer data)
{
  /* This function handler will be used later to display data specific
   * of each canvas item (like a id card of the monomer that is 
   * represented by the canvas item, in the form of a tooltip, for 
   * example).
   */
  /*
   debug_printf (("polyxedit_seqed_wnd_canvas_item_event\n"));
  */
  
  return FALSE;
}


gboolean
polyxedit_seqed_monomer_flag_expose_event (GtkWidget *widget,
					   GdkEventExpose *event, 
					   gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  
  g_assert (editctxt != NULL);
  
  /* Widget is the triflag_box that contains the three flags (drawing
     areas).
  */
  polyxedit_seqed_wnd_update_monomer_flag (editctxt) ;
  
  return TRUE; 
}

















gint 
polyxedit_seqed_wnd_delete_event (GtkWidget *window, GdkEventAny *event, 
				  gpointer data)
{
  /* 
     Note that this function is ALSO called by the "Close" menu item
     of the contextual menu that pops up when the sequence editor is
     right-clicked in the sequence. The 'data' parameter is thus used
     when the 'window' parameter is NULL.

     If the window that is being closed contains a modified polymer
     sequence, then all we have to do is ask if the sequence should be
     saved or not. The dialog asking the question is then pointed,
     depending on the answer, to functions that perform the "no save
     and close" task or the "save and close if successful" task.
   */
  gint count = 0;

  PxmEditCtxt *editctxt = data;

  GtkWidget *dialog = NULL;
  GtkWidget *seq_wnd = NULL;
  
  gchar *help = NULL;
  gchar *basename = NULL;
  

  g_assert (editctxt != NULL);

  if (window == NULL)
    {
      seq_wnd = editctxt->seqeditorctxt->sequence_editor_wnd;
      
      g_assert (seq_wnd != NULL);
    }
  else
    seq_wnd = window;
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) seq_wnd);
  
  /*
    printf ("polyxedit_elemcompos_wnd_delete_event, count is %d\n",
    count);
  */
  

  
  if (TRUE == pxmchem_polymer_get_modified (editctxt->polymer))
    {
      /* The polymer sequence that is being edited in the window that
	 should be closed is dirty. Ask if it should be saved prior to
	 closing the window.
      */
      basename = g_path_get_basename (editctxt->polymer->plminfo->file);
      help = g_strdup_printf (_("%s (%p) modified \n Save sequence?"), 
			      basename, editctxt);
      g_free (basename);
      
      dialog = gtk_message_dialog_new (GTK_WINDOW (seq_wnd),
				       GTK_DIALOG_DESTROY_WITH_PARENT,
				       GTK_MESSAGE_QUESTION,
				       GTK_BUTTONS_YES_NO,
				       help);
      g_free (help);
      
      gtk_dialog_add_buttons (GTK_DIALOG (dialog),
			      GTK_STOCK_CANCEL,
			      GTK_RESPONSE_CANCEL,
			      NULL);

      /* Destroy the dialog when the user responds to it
       * (e.g. clicks a button) 
       */
      g_signal_connect (GTK_OBJECT (dialog), "response",
			G_CALLBACK 
			(polyxedit_seqed_wnd_save_question),
			editctxt);
      
      gtk_widget_show_all (GTK_WIDGET (dialog));
      
      return TRUE;
    }
  else
    {
      /* The polymer sequence is apparently in a correctly saved state.
       * Just do the work of the closing function in the proper function.
       */
      polyxedit_seqed_wnd_close (editctxt);
      
      return TRUE;
    }
  
  /* If we are here, that means that something is wrong. Do not let Gtk+
   * do anything.
   */
  return TRUE;
}


void 
polyxedit_seqed_wnd_save_question (GtkWidget *widget,
				   gint arg,
				   gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  /* The sequence edited in the sequence editor window was probably
     dirty and the program displayed a dialog message window asking if
     the sequence should be saved before closing the sequence editor
     window.
  */
  

  g_assert (widget != NULL);
  g_assert (editctxt != NULL);
  
  switch (arg)
    {
    case  GTK_RESPONSE_YES:
      polyxedit_seqed_wnd_save_and_close (editctxt);
      break;
      
    case  GTK_RESPONSE_NO:
      polyxedit_seqed_wnd_close (editctxt);
      break;
      
    case  GTK_RESPONSE_CANCEL:
      break;
    }
  
  gtk_widget_destroy (widget);

  return;
}


gboolean
polyxedit_seqed_wnd_save_and_close (PxmEditCtxt *editctxt)
{
  /* We have to first save the sequence to a file.
   */
  gchar *file = NULL;
  
  gint result = -1;

  gchar *xml = NULL;
      
  FILE *filep = NULL;



  g_assert (editctxt != NULL);



  file = editctxt->polymer->plminfo->file;
  g_assert (file != NULL);

  if (strlen (file) > 0)
    {
      /* All we do is save the sequence to its own file.
       */
      filep = fopen (file, "w");

      if (filep == NULL)
	{
	  g_critical (_("%s@%d: error with opening file %s\n"),
		 __FILE__, __LINE__, file);
      
	  return FALSE;
	}
  
      /* Construct a string with all the xml-formatted data pertaining
	 to the polymer sequence 'polymer'.
       */
      xml = 
	pxmchem_polymer_format_xml_string_polseqdata (editctxt->polymer, 
						      "  ", 0);
      g_assert (xml != NULL);

      /* Copy the xml data string to the file 'file'.
       */
      result = fputs (xml, filep);

      fclose (filep);
  
      g_free (xml);
  
      if (result == EOF || result < 0)
	{
	  g_critical (_("%s@%d: error with saving polymer sequence to file %s\n"),
		 __FILE__, __LINE__, file);
      
	  return FALSE;
	}
      
      /* By definition, a sequence that has been successfully saved to 
	 disk is no more dirty:
      */
      polyxedit_seqed_wnd_set_polymer_modified (editctxt, FALSE);
      
      polyxedit_seqed_wnd_close (editctxt);
      
      return TRUE;
    }
    
  /* At this point, the sequence could not be saved because no file
     name is available. We must first get a file name for the sequence
     to be saved properly.
  */
  polyxedit_seqed_window_menu_file_saveas (NULL, editctxt);
  
  return TRUE;
}

  
void 
polyxedit_seqed_wnd_close (PxmEditCtxt *editctxt)
{
  gboolean result = FALSE;
  


  g_assert (editctxt != NULL);


  /* First of all make sure we can remove the editctxt from 
     the array of such instances in the program:
  */
  result = g_ptr_array_remove (polyxedit_editctxtGPA, editctxt);
  
  g_assert (result == TRUE);
  
  if (polyxedit_last_editctxt == (gpointer) editctxt)
    {
      polyxedit_last_editctxt = NULL;

      /* We want to force the mass display window to reset all its
	 text entries to nothing, because we are closing the seqeditor
	 window for which the data were currently displayed (if the
	 mass-data-display window was opened of course).
      */
      polyxedit_ui_masses_display_wnd_reset_all ();
    }
  
  polyxedit_editctxt_free (editctxt);

  return;
}
