/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "pxmchem-masscalc.h"
#include "pxmchem-monomer.h"
#include "pxmchem-formula.h"


PxmMasscalcRes
pxmchem_masscalc_polymer (PxmPolymer *polymer,
			  PxmPolchemdef *polchemdef,
			  PxmCalcOpt *calcopt,
			  PxmIonizerule *ionizerule, /* can be set by
							the caller to
							the polchemdef->
							ionizerule */
			  PxmMasspair *masspair)
{
  PxmMasscalcRes masscalcres = PXM_MASSCALC_SUCCESS;
  
  g_assert (polymer != NULL);
  g_assert (polymer->monomerGPA != NULL);
  g_assert (masspair != NULL);
  g_assert (polchemdef != NULL);
  g_assert (calcopt != NULL);
  g_assert (ionizerule != NULL);

  
  /* Check that there is a sequence element for which to calculate a
     mass.  If not, this is not necessarily an error. We continue
     working because many other things have to be taken into
     account. But inform the user.
     *
     if (polymer->monomerGPA->pdata == NULL)
     
     g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
     _("%s@%d: the polymer sequence is empty.\n"),
     __FILE__, __LINE__);
  */
  
  /*********************** CAPPED OR NON CAPPED ************************/

  /* Now that we know we have a GPtrArray of monomer instances
     corresponding to the sequence of the polymer, we can continue the
     work, with the actual mass calculations based upon the
     composition of the monomer GPtrArray. 

     Note that if the caller has asked that the monomer entities (like
     modifications) be taken into account during the mass calculation,
     this will be done in the function call below.
  */
  if (TRUE != pxmchem_masscalc_noncapped_monomer_GPA (polymer->monomerGPA,
						      polchemdef, 
						      masspair, 
						      calcopt))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to calculate the non-capped masses of the polymer\n"),
	     __FILE__, __LINE__);

      masscalcres = PXM_MASSCALC_FAILURE;
    }

  /*
    debug_printf (("non-capped polymer - mono=%f and avg=%f\n",
    masspair->mono, masspair->avg));
  */
  
  /* We now have in the masspair structure the masses for the
     non-capped residual chain of monomers constituting the
     polymer. We now have to check if user wants masses for a CAPPED
     polymer or not. If yes we delegate this calculation to a helper
     function
  */
  if (calcopt->capping != PXM_CAPPING_NONE)
    {
      if (FALSE == 
	  pxmchem_masscalc_polymer_account_caps (masspair, 
						 calcopt->capping, 
						 polchemdef))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for caps of the polymer\n"),
		 __FILE__, __LINE__);

	  masscalcres = PXM_MASSCALC_FAILURE;
	}
      
      /*
	debug_printf (("capped polymer - mono=%f and avg=%f\n",
	masspair->mono, masspair->avg));
      */
    }


  /*** THE CHEMICAL ENTITIES THAT MAY BELONG TO THE POLYMER SEQUENCE ***/

  if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF)
    {
      /* Take into account the left end modif.
       */
      if (FALSE == 
	  pxmchem_masscalc_polymer_account_left_end_modif (polymer,
							   masspair,
							   polchemdef))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for the left"
		   " end modif of the polymer.\n"),
		 __FILE__, __LINE__);
	  
	  masscalcres = PXM_MASSCALC_FAILURE;
	}

      /*
	debug_printf (("accounted left end modif - mono=%f and avg=%f\n",
	masspair->mono, masspair->avg));
      */
    }
  
  
  if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF)
    {
      /* Take into account the right end modif.
       */
      if (FALSE == 
	  pxmchem_masscalc_polymer_account_right_end_modif (polymer,
							    masspair,
							    polchemdef))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for the"
		   " right end modif of the polymer.\n"),
		 __FILE__, __LINE__);
	  
	  masscalcres = PXM_MASSCALC_FAILURE;
	}
      
      /*
	debug_printf (("accounted right end modif - mono=%f and avg=%f\n",
	masspair->mono, masspair->avg));
      */
    }
  

  /******************** IONIZATION REQUIREMENTS**********************/

  /* For the moment the masses calculated are for a non-charged polymer:
     let's check if caller wants some kind of charge to be put on it.
     For this we also need to know what atom is responsible for the
     charge formation (proton or whatever).
  */
  if (FALSE ==
      pxmchem_masscalc_account_ionizerule (ionizerule,
					   masspair,
					   polchemdef->atomGPA))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for the ionizerule.\n"),
	     __FILE__, __LINE__);
      
      masscalcres = PXM_MASSCALC_FAILURE;
    }
  
  /*
    debug_printf (("accounted ionizerule - mono=%f and avg=%f\n",
    masspair->mono, masspair->avg));
  */
  
  return masscalcres;
}



gboolean
pxmchem_masscalc_noncapped_monomer_GPA (GPtrArray *GPA,
					PxmPolchemdef *polchemdef, 
					PxmMasspair *masspair, 
					PxmCalcOpt *calcopt)
{
  gint len = 0;
  gint iter = 0;

  PxmMonomer *monomer = NULL;

  /* We will iterate in the monomerGPA passed as parameter and for each
     monomer iterated we will add its mass contribution to the masspair
     object passed as parameter. Note that if calcopt states that
     the monomer chemical entities (modifs) must be taken into account,
     then, this is done.

     Note that the calcopt struct passed as param contains two members
     that will allow us to know which monomer stretch from the GPA
     should be taken into account during the calculatation.
  */

  /* Some easy checks to start with, in particular the array of
     monomers, GPA, cannot be NULL.
   */
  g_assert (GPA != NULL);
  g_assert (polchemdef != NULL);
  g_assert (masspair != NULL);

  len = GPA->len;

  if (len == 0)
    {
      /*
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	_("%s@%d: the monomer array is empty.\n"),
	__FILE__, __LINE__);
      */

      /* This is not a formal error.
       */
      return TRUE;
    }

  /* Now check some of the options passed as param pertaining to mass
   * calculatation of the non capped chain (make code FOOL PROOF).
   */

  /* Left end first. 
     ---------------

     It may happen that the user clicks and drags right of the last 
     polymer sequence's monicon and that this function is called with a 
     calcopt->start_idx = len. We just need to check this.
  */
  if (calcopt->start_idx > len)
    calcopt->start_idx = len;
  
  /* Special value: if -1, then that means the whole sequence should be
     taken into account. Thus set start_idx to 0.
  */
  if (calcopt->start_idx == -1)
    calcopt->start_idx = 0;

  /* And now make sure that start_idx does not have silly values apart
     from this special-meaning -1 value.
   */
  g_assert (calcopt->start_idx >= 0);
  

  /* Right end, second.
   */
  g_assert (calcopt->end_idx <= len);

  /* Special value: if -1, then that means the whole sequence should
     be taken into account. Thus set end_idx to len (because the for
     loop later will squeeze stop at iter == (len - 1).
  */
  if (calcopt->end_idx == -1)
    calcopt->end_idx = len;

  /* And now make sure that end_idx does not have silly values apart
     from this special-meaning -1 value.
   */
  g_assert (calcopt->end_idx >= 0);
  
  /* NOW start calculating the masses of the non-capped polymer by 
   * accumulating the masses calculated for the formula of each
   * iterated monomer in the GPtrArray of monomer passed as param.
   */
  for (iter = calcopt->start_idx; iter < calcopt->end_idx; iter++)
    {
      monomer = g_ptr_array_index (GPA, iter);

      if (FALSE ==
	  pxmchem_monomer_account_mass_by_name (monomer->name,
						polchemdef->monomerGPA,
						polchemdef->atomGPA,
						1, /*times*/
						masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for monomer: '%s'\n"),
		 __FILE__, __LINE__, monomer->name);

	  
	  return FALSE;
	}
      
      /* If monomer modifs are asked to be taken into account, do it:
       */
      if (calcopt->mnm_chement & PXMCHEMENT_MNM_MODIF)
	{
	  if (FALSE == 
	      pxmchem_monomer_account_mass_for_modif (monomer,
						      polchemdef->modifGPA,
						      polchemdef->atomGPA,
						      1, /*times*/
						      masspair))
	    {	  
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed to account for modif of "
		       "monomer: '%s'\n"),
		     __FILE__, __LINE__, monomer->name);


	      return FALSE;
	    }
	}	
      /* end of 
	 if (calcopt->mnm_chement == PXMCHEMENT_MNM_MODIF)
      */

    }
  /* end iterating in the GPtrArray of monomers */

  /* At this stage we have the NON CAPPED masses calculated for the
     given polymer stretch comprised of monomers [calcopt->start_idx -
     calcopt->end_idx]. If the options stated that the monomer modif's
     had to be taken into account, this has been done! Also, here both
     the masses avg and mono have been calculated.
   */

  return TRUE;
  
}


gboolean
pxmchem_masscalc_polymer_account_caps (PxmMasspair *masspair,
				       PxmCapping cap,
				       PxmPolchemdef *polchemdef)
{
  /* There are two kinds of caps: the left end cap and the right end cap.
     Each should be treated separately depending on the value of cap.
  */
  g_assert (masspair != NULL);
  g_assert (polchemdef != NULL);
  
  if (cap & PXM_CAPPING_NONE)
    return TRUE;
  
  if (FALSE != (cap & PXM_CAPPING_LEFT))
    {
      if (FALSE == 
	  pxmchem_actform_account_mass (polchemdef->leftcap, 
					polchemdef->atomGPA,
					1 /*times*/, masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for left cap: '%s'\n"),
		 __FILE__, __LINE__, polchemdef->leftcap);
	  return FALSE;
	}
    }
  
  if (FALSE != (cap & PXM_CAPPING_RIGHT))
    {
      if (FALSE == 
	  pxmchem_actform_account_mass (polchemdef->rightcap, 
					polchemdef->atomGPA, 
					1 /*times*/, masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for right cap: '%s'\n"),
		 __FILE__, __LINE__, polchemdef->rightcap);
	  return FALSE;
	}
    }
  
  return TRUE;
}

  
gboolean
pxmchem_masscalc_polymer_account_left_end_modif (PxmPolymer *polymer,
						 PxmMasspair *masspair,
						 PxmPolchemdef *polchemdef)
{
  PxmProp *prop = NULL;

  /* We need to get access to the LEFT_END_MODIF prop instance in 
     the polymer, if there is any.
  */
  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
				  NULL,
				  NULL,
				  "LEFT_END_MODIF",
				  NULL,
				  PXM_CMP_NO_DEEP);
  if (prop == NULL)
    return TRUE;
  
  if (FALSE == 
      pxmchem_modif_account_mass_by_name ((gchar *) prop->data,
					  polchemdef->modifGPA,
					  polchemdef->atomGPA,
					  1,
					  masspair))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for left end modif of the polymer: '%s'\n"),
	     __FILE__, __LINE__, (gchar *) prop->data);
      return FALSE;
    }
  
  return TRUE;
}

  
gboolean
pxmchem_masscalc_polymer_account_right_end_modif (PxmPolymer *polymer,
						  PxmMasspair *masspair,
						  PxmPolchemdef *polchemdef)
{
  PxmProp *prop = NULL;

  /* We need to get access to the LEFT_END_MODIF prop instance in 
     the polymer, if there is any.
  */
  prop = libpolyxmass_prop_find_prop (polymer->propGPA,
				  NULL,
				  NULL,
				  "RIGHT_END_MODIF",
				  NULL,
				  PXM_CMP_NO_DEEP);
  if (prop == NULL)
    return TRUE;
  
  if (FALSE == 
      pxmchem_modif_account_mass_by_name ((gchar *) prop->data,
					  polchemdef->modifGPA,
					  polchemdef->atomGPA,
					  1,
					  masspair))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for right end modif of polymer: '%s'\n"),
	     __FILE__, __LINE__, (gchar *) prop->data);
      return FALSE;
    }
  
  return TRUE;
}


gboolean
pxmchem_masscalc_account_ionizerule (PxmIonizerule *ionizerule,
				     PxmMasspair *masspair,
				     GPtrArray *atom_refGPA)
{
  g_assert (ionizerule != NULL);
  g_assert (ionizerule->actform != NULL);
  g_assert (atom_refGPA != NULL);
  g_assert (masspair != NULL);
  

  /* We get masses from the masspair parameter, and we need to modify
     them in order to comply to the ionization options asked in the 
     'ionizerule'.
   */
  /* We cannot have a charge == 0, because it will be in the
   * denominator in the following calculations, so return without
   * error, which simply means that the caller does not want to have
   * any ionization! Same for level. See below.
   */
  if (ionizerule->charge == 0 || ionizerule->level == 0)
    return TRUE;
  
  
  /* In the ionizerule argument, we have a member actform of the kind
     "+Cd1-H1", for example. This actionformula is to be taken into
     account in the masses stored in 'masspair' a number of times that
     is an unsigned (absolute) value of ionizerule->level. If 'level'
     were 2, for example, we would account the "+Cd1-H1" twice, which
     is add 2 Cadmium atoms and remove two protons. 
   */

  /*
  debug_printf (("prior: masspair->mono is %lf\n", masspair->mono));
  debug_printf (("prior: masspair->avg is %lf\n", masspair->avg));

  debug_printf (("ionizerule->charge is %d\n", abs (ionizerule->charge)));
  debug_printf (("ionizerule->level is %d\n", abs (ionizerule->level)));
  */

  if (FALSE == 
      pxmchem_actform_account_mass (ionizerule->actform, atom_refGPA,
				    + 1 * abs (ionizerule->level) /*times*/, 
				    masspair))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for the actform of"
	       " the ionizerule: '%s'\n"),
	     __FILE__, __LINE__, ionizerule->actform);
      
      return FALSE;
    }
 
  /* OK we have accounted the masspair for the ionizerule->actform by
     a number of times ionizerule->level. Now we have to calculate the
     new m/z ratio according to the charge ionizationrule's member
     along with the ionizerule->level member.
   */

  masspair->mono = masspair->mono / 
    (abs (ionizerule->charge) * abs (ionizerule->level));

  masspair->avg = masspair->avg / 
    (abs (ionizerule->charge) * abs (ionizerule->level));

  /*
  debug_printf (("after: masspair->mono is %lf\n", masspair->mono));
  debug_printf (("after: masspair->avg is %lf\n", masspair->avg));
  */

  return TRUE;
}



gboolean
pxmchem_masscalc_decount_ionizerule (PxmIonizerule *ionizerule,
				     PxmMasspair *masspair,
				     GPtrArray *atom_refGPA)
{
  g_assert (ionizerule != NULL);
  g_assert (ionizerule->actform != NULL);
  g_assert (atom_refGPA != NULL);
  g_assert (masspair != NULL);
  

  /* We get masses from the masspair parameter, and we need to modify
     them in order to revert the ionization options set in the
     'ionizerule'.
   */
  /* We cannot have a charge == 0, because it will be in the
     denominator in the following calculations, so return without
     error, which simply means that the caller does not want to have
     any ionization! Same for level. See below.
   */
  if (ionizerule->charge == 0 || ionizerule->level == 0)
    return TRUE;
  
  /* In the ionizerule argument, we have a member ionizerule->level
     that indicates the number of times the ionization was
     performed. Also, the re is another member, ionizerule->charge
     that reflects the charge that is brought to the analyte each time
     the ionization is performed.

     That means that if level=2 and charge=2, the total charge of the
     analyte will be 4.

     What we get is a 'masspair' parameter corresponding to the m/z
     ratio calculated for the 'charge' and 'level' pair of data. We
     have to revert this, by multiplying the m/z ratio present in
     'masspair' by the 'charge' * 'level' product. This way, will find
     the mass of the ion in its ionized state.
  */
  masspair->mono = masspair->mono *
    (abs (ionizerule->charge) * abs (ionizerule->level));

  masspair->avg = masspair->avg *
    (abs (ionizerule->charge) * abs (ionizerule->level));

  /* At this point the 'masspair' members do contain the mass of the
     ion. We still have to go back to the molecular mass by retrieving
     from that mass the mass of the ionization (chemically-wise).

     In the ionizerule argument, we have a member actform of the kind
     "+Cd1-H1", for example. That means that when the analyte was
     ionized, the "+Cd1-H1" actionformula was accounted on the
     masspair object. That accounting was repeated as many times as
     indicated in the ionizerule->level datum. See the _account_
     version of the function above for an explanation.

     Now, since we are un-doing this ionization, all we have to do is
     counteract this accounting of the actform by negating the
     ionizerule->level parameter.
  */
  if (FALSE == 
      pxmchem_actform_account_mass (ionizerule->actform, atom_refGPA,
				    -1 * abs (ionizerule->level) /*times*/, 
				    masspair))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to de-count for the actform of the ionization "
	       "rule:'%s'\n"),
	     __FILE__, __LINE__, ionizerule->actform);
      
      return FALSE;
    }
 
  /* Apparently this is OK now. We have now in 'masspair' masses
     corresponding to the molecular mass of the analyte, not charged,
     nothing. Just plain analyte.
  */
  
  return TRUE;
}







PxmMasscalcRes
pxmchem_masscalc_oligomer (PxmOligomer *oligomer,
			   PxmPolchemdef *polchemdef,
			   PxmCalcOpt *calcopt,
			   PxmCleaveOpt *cleaveopt,
			   PxmIonizerule *ionizerule,
			   PxmMasspair *masspair)
{
  gint len = 0;
  gint iter = 0;
  
  PxmPolymer *polymer = NULL;

  GPtrArray *clrGPA = NULL;
  


  g_assert (oligomer != NULL);
  polymer = oligomer->polymer;

  g_assert (polymer != NULL);
  g_assert (polymer->monomerGPA != NULL);
  
  g_assert (polchemdef != NULL);
  g_assert (calcopt != NULL);


  g_assert (ionizerule != NULL);
  g_assert (masspair != NULL);
  
  len = oligomer->polymer->monomerGPA->len;
  
  if (len <= 0)
    {
      /*
	g_log (G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE,
	_("%s@%d: the polymer sequence is empty.\n"),
	__FILE__, __LINE__);
      */
      
      return TRUE;
    }
  

  /* For the mass calculation process we must not take coordinates of
     the oligomer in the oligomer object itself since these have
     not been set for mass calculations. The values have been set 
     correctly for the mass calculation in the calcopt structure passed
     as parameter.
  */
  g_assert (calcopt->start_idx >=0 && calcopt->end_idx >= 0);
  g_assert (calcopt->start_idx < len);
  g_assert (calcopt->end_idx <= len);

  if (FALSE == 
      pxmchem_masscalc_noncapped_monomer_GPA (polymer->monomerGPA,
					      polchemdef, 
					      masspair, 
					      calcopt))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to calculate the masses of the oligomer\n"),
	     __FILE__, __LINE__);
      
      return PXM_MASSCALC_FAILURE;
    }
  

  /* At this stage we have the NON CAPPED masses calculated for the
     given polymer stretch comprised of monomers [calcopt->start_idx -
     calcopt->end_idx]. If the options stated that the monomer modif's
     had to be taken into account, this has been done! Also, here both
     the masses avg and mono have been calculated.
  */
  
  /* Here is one part that is a bit tricky with the calculating of
     oligomer masses: the cleaverule handling. Each oligomer may
     result from a cleavage that has as side effects the chemical
     modification of the monomers that were bordering the chemical
     bond that is cleaved. See the cyanogen bromide example in the
     proteins' world.

     The cleaveopt parameter has a cleavespec object in it, which, in
     turn has an array of cleaverules, that we are going to handle
     right now. Note that the array should exist, but may be empty.

     However, remember that the cleaveopt param may be NULL if the
     oligomer was not resulting from a chemical cleavage. See the case
     of searching masses in the polymer sequence.
  */
  if (cleaveopt != NULL)
    {
      PxmCleaveSpec *cleavespec = NULL;
      PxmCleaveRule *lr_rule = NULL;
      
      cleavespec = cleaveopt->cleavespec;
      g_assert (cleavespec != NULL);
     
      clrGPA = cleavespec->clrGPA;
      g_assert (clrGPA != NULL);
      
      
      for (iter = 0; iter < clrGPA->len; iter++)
	{
	  lr_rule = g_ptr_array_index (clrGPA, iter);
	  g_assert (lr_rule != NULL);
	  
	  /* Now account for this rule.
	   */
	  if (FALSE == pxmchem_masscalc_oligomer_account_cleaverule 
	      (lr_rule,
	       oligomer,
	       masspair,
	       polchemdef->atomGPA))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed to account for cleaverule.\n"),
		     __FILE__, __LINE__);
	      
	      return PXM_MASSCALC_FAILURE;
	    }
	}

      /* Done accounting for cleaverule objects (if any). OUF.
       */
    }
  
  /* We now have in the masspair structure the masses for the
     non-capped residual chain of monomers constituting the
     polymer. We now have to check if user wants masses for a CAPPED
     polymer or not. If yes we delegate this calculation to a helper
     function
  */
  if (calcopt->capping != PXM_CAPPING_NONE)
    {
      if (FALSE == 
	  pxmchem_masscalc_polymer_account_caps (masspair, 
						 calcopt->capping, 
						 polchemdef))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed to account for caps of the polymer\n"),
		 __FILE__, __LINE__);

	  return PXM_MASSCALC_FAILURE;
	}
      
      /*
	debug_printf (("capped oligomer - mono=%f and avg=%f\n",
	masspair->mono, masspair->avg));
      */
    }


  /*** THE CHEMICAL ENTITIES THAT MAY BELONG TO THE POLYMER SEQUENCE ***/

  if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF)
    {
      /* Take into account the left end modif. We only do this if
	  the current oligomer contains as its first monomer the 
	  monomer that was the left end monomer of the inital polymer 
	  sequence:
       */
      if (oligomer->start_idx == 0)
	{
	  if (FALSE == 
	      pxmchem_masscalc_polymer_account_left_end_modif (polymer,
							       masspair,
							       polchemdef))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed to account for the"
		       " left end modif of the polymer.\n"),
		     __FILE__, __LINE__);
	      
	      return PXM_MASSCALC_FAILURE;
	    }
	  /*
	    debug_printf (("accounted left end modif - mono=%f and avg=%f\n",
	    masspair->mono, masspair->avg));
	  */
	}
    }
    
  
  if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF)
    {
      /* Take into account the right end modif. We only do this if the
	  current oligomer contains as its last monomer the monomer
	  that was the right end monomer of the inital polymer
	  sequence:
       */
      if (oligomer->end_idx == len - 1)
	{
	  if (FALSE == 
	      pxmchem_masscalc_polymer_account_right_end_modif (polymer,
								masspair,
								polchemdef))
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: failed to account for the "
		       "right end modif of the polymer.\n"),
		     __FILE__, __LINE__);
	      
	      return PXM_MASSCALC_FAILURE;
	    }

	  /*
	    debug_printf (("accounted right end modif - "
	    "mono=%f and avg=%f\n",
	    masspair->mono, masspair->avg));
	  */
	}
    }
  

  /******************** IONIZATION REQUIREMENTS**********************/

  /* For the moment the masses calculated are for a non-charged polymer:
     let's check if caller wants some kind of charge to be put on it.
     For this we also need to know what atom is responsible for the
     charge formation (proton or whatever).
  */
  if (FALSE ==
      pxmchem_masscalc_account_ionizerule (ionizerule,
						   masspair,
						   polchemdef->atomGPA))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to account for the ionizerule.\n"),
	     __FILE__, __LINE__);
      
      return PXM_MASSCALC_FAILURE;
    }
  /*
    debug_printf (("accounted ionizerule - mono=%f and avg=%f\n",
    masspair->mono, masspair->avg));
  */
  
  return PXM_MASSCALC_SUCCESS;
}


gboolean
pxmchem_masscalc_oligomer_account_cleaverule (PxmCleaveRule *lr_rule,
						 PxmOligomer *oligomer,
						 PxmMasspair *masspair,
						 GPtrArray *atom_refGPA)
{
  gint len = 0;
  
  PxmMonomer *monomer = NULL;

  /* We get a masspair into which we are asked to account for the
     cleaverule that is passed as parameter. A cleaverule is
     nothing but a conditional actform. The condition exercises on two
     criteria: the identity of the monomer that is concerned and the
     location of this monomer (is it the left end monomer of the
     oligomer or is it the monomer at the right end of the oligomer.

     Note that any of the two monomer codes (even both actually) in
     the cleaverule may be NULL. If they are non-NULL, then their
     corresponding actform has to be also non-NULL. There is a
     specific case where the condition may be respected and the
     actform must not be enforced: when the monomer is at the
     left/right end of the oligomer, but it is there not by virtue of
     the cleavage of the initial polymer sequence. For example:

     ATJQFFGM/LQJDFQSDFUERM/QSDFIOPUFNQDFOPIERM is a sequence that
     gets cleaved right of each M (show with the '/' char.). The
     cleavage thus leads to the formation of three oligomers:

     ATJQFFGM   LQJDFQSDFUERM   QSDFIOPUFNQDFOPIERM

     The cleaverule states that if 'M' is found on the right end of
     the generated oligomers, then the actform must be enforced.

     We would quietly perform this chemical modification on the three
     oligomers. But this would be erroneous for the third one, because
     the 'M' was actually NEVER cleaved, since it corresponded to the
     right end monomer of the initial polymer. This is why we have to
     not only check if the monomer identity/location condition is
     verified but also if it makes sense to enforce the chemical
     modification.
  */
  g_assert (lr_rule != NULL);
  g_assert (masspair != NULL);
  g_assert (oligomer != NULL);
  g_assert (atom_refGPA != NULL);

  len = oligomer->polymer->monomerGPA->len;


  /* The left couple of data.
   */
  if (lr_rule->left_code != NULL)
    {
      /* If there is a left code, then the actform must also be
	 there. Assert this.
      */
      g_assert (lr_rule->left_actform != NULL);
	  
      /* Get the monomer that is at the left end of the current
	 oligomer.
      */     
      monomer = g_ptr_array_index (oligomer->polymer->monomerGPA,
				   oligomer->start_idx);
      g_assert (monomer != NULL);
	  
      /* Now check if the monomer's code matches the code that
	 is specified in the cleaverule.
      */
      if (0 == strcmp (lr_rule->left_code, monomer->code))
	{
	  if (oligomer->start_idx != 0)
	    {
	      /* The monomer at left end of current monomer matches!
		 And monomer was not the left end monomer of the
		 polymer sequence. Now do the chemical work proper.
	      */
	      if (FALSE == 
		  pxmchem_actform_account_mass (lr_rule->left_actform, 
						atom_refGPA,
						1 /*times*/, masspair))
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
			 _("%s@%d: failed to account for actform of cleaverule: '%s'\n"),
			 __FILE__, __LINE__, lr_rule->left_actform);
		  
		  return PXM_MASSCALC_FAILURE;
		}
	    }
	}
    }
  /* end of
     if (lr_rule->left_code != NULL)
  */
      
  /* The right couple of data.
   */
  if (lr_rule->right_code != NULL)
    {
      /* If there is a right code, then the actform must also be
	 there. Assert this.
      */
      g_assert (lr_rule->right_actform != NULL);
	  
      /* Get the monomer that is at the right end of the current
	 oligomer.
      */     
      monomer = g_ptr_array_index (oligomer->polymer->monomerGPA,
				   oligomer->end_idx);
      g_assert (monomer != NULL);
	  
      /* Now check if the monomer's code matches the code that
	 is specified in the cleaverule.
      */
      if (0 == strcmp (lr_rule->right_code, monomer->code))
	{
	  if (oligomer->end_idx == len - 1)
	    {
	      /* The monomer at right end of current monomer matches!
		 And monomer was not the right end monomer of the
		 polymer sequence. Now do the chemical work proper.
	      */
	      if (FALSE == 
		  pxmchem_actform_account_mass (lr_rule->right_actform, 
						atom_refGPA,
						1 /*times*/, masspair))
		{
		  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
			 _("%s@%d: failed to account for actform of cleaverule: '%s'\n"),
			 __FILE__, __LINE__, lr_rule->right_actform);
		  
		  return PXM_MASSCALC_FAILURE;
		}
	    }
	  
	}
    }
  /* end of
     if (lr_rule->right_code != NULL)
  */
  return PXM_MASSCALC_SUCCESS;
}
