/*
 * powernowd.c: (c) 2003-2004 John Clemens <clemej@alum.rpi.edu>
 *
 * Released under the GNU General Public License. See the LICENSE file
 * included with this file.
 *
 * * Changelog:
 *   v0.75, initial release.
 *   v0.80, add syslog support, -s,-p,-u,-l options, clean up error reporting
 *          some (still some more work needed), made packaging better, removed
 *          find_mount() code.. /sys is the blessed place, so be it.. no sense
 *          having unnecessary code and the maintenance thereof.  fixed bug 
 *          where mode was actually defaulting to SINE, not AGGRESSIVE. 
 *   v0.85, Minor memory init fixes, add clearer error messages, check for 
 *   	    root, added -b and -n options, add pause support (SIGUSR1/2, -b 
 *   	    option), added #ifdef'd-out code to handle buggy athlons which
 *   	    might not be necessary anymore, fixed some help text bugs, added
 *   	    a few more comments.
 *   v0.90, Support drivers that report speed in Mhz instead of Khz. Removed
 *   	    buggy athlon workaround, as it's not needed. Removed pause/unpause
 *   	    code. Added sample powernowd init script to show how to emulate
 *   	    old pause/unpause behavior a cleaner way. Added LEAPS mode and 
 *   	    verbosity patches by Hans Ulrich Niedermann. Cleaned up verbosity
 *   	    handling more.
 *   	    
 * * Contributions from:
 * 	Warren Togami <warren@togami.com>
 * 	Michael Schreiter <Michael_Schreiter@gmx.de>
 * 	Wolfgang Tremmel <tremmel@garf.de>
 * 	Hans Ulrich Niedermann <kdev@n-dimensional.de>
 */


#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>
#include <time.h>

#define pprintf(level, ...) do { \
	if (level <= verbosity) { \
		if (daemonize) \
			syslog(LOG_INFO, __VA_ARGS__); \
		else \
			printf(__VA_ARGS__); \
	} \
} while(0)

typedef struct cpustats {
	unsigned long long user;
	unsigned long long mynice;
	unsigned long long system;
	unsigned long long idle;
} cpustats_t;

typedef struct cpuinfo {
	unsigned int cpuid;
	unsigned int nspeeds;
	unsigned int max_speed;
	unsigned int min_speed;
	unsigned int current_speed;
	int fd;
	char *sysfs_dir;
	cpustats_t *last_reading;
	cpustats_t *reading;
	int in_mhz; /* 0 = speed in kHz, 1 = speed in mHz */  
} cpuinfo_t;

/* 
 * Global pointer to the beginning of all our info.  This is so we can tear 
 * stuff down later in the signal handler.
 */

cpuinfo_t **cpuinfo;

/* idea stolen from procps */
static char buf[1024];

enum function {
	SINE,
	AGGRESSIVE,
	PASSIVE,
	LEAPS
} func = AGGRESSIVE; 

enum modes {
	RAISE,
	LOWER
};

/* for a daemon as simple as this, global data is ok. */
/* settings */
int daemonize = 1;
int ignore_nice = 1;
int verbosity = 0;
unsigned int step = 100000;  /* in kHz */
unsigned int poll = 1000; /* in msecs */
unsigned int highwater = 80;
unsigned int lowwater = 20;

/* statistics */
unsigned int change_speed_count = 0;
time_t start_time = 0;

#define SYSFS_TREE "/sys/devices/system/cpu/"
#define SYSFS_SETSPEED "scaling_setspeed"
#define SYSFS_GOVERNOR "scaling_governor"

#define VERSION	"0.90"

void help(void)
{
	
	printf("PowerNow Daemon v%s, (c) 2003-2004 John Clemens\n", VERSION);
	printf("Daemon to control the speed and voltage of cpus.\n\n");
	printf("This is a simple client to the CPUFreq driver, and uses\n");
	printf("linux kernel v2.5 sysfs interface.  You need a supported\n");
	printf("cpu, and a kernel that supports sysfs to run this daemon.\n");
	printf("\nAvailable Options:\n");
	printf(" 	-h	Print this help message\n");
	printf("	-d	Don't detach from terminal (default is to\n");
	printf("		detach and run in the background)\n");
	printf("	-v	Increase output verbosity, can be used more than once.\n");
	printf("	-q	Quiet mode, only emergency output.\n");
	printf("	-n	Include 'nice'd processes in calculations\n");
	printf("	-m #	Modes of operation, can be 0, 1, 2, or 3:\n");
	printf("		0 = SINE, 1 = AGGRESSIVE (default),\n");
	printf("		2 = PASSIVE, 3 = LEAPS\n");
	printf("	-s #	Frequency step in kHz (default = 100000)\n");
	printf("	-p #	Polling frequency in msecs (default = 1000)\n");
	printf("	-u #	CPU usage upper limit percentage [0 .. 100, default 80]\n");
	printf("	-l #    CPU usage lower limit percentage [0 .. 100, default 20]\n");

	printf("\n");
	return;
}

/* 
 * Open a file and copy it's first 1024 bytes into the global "buf".
 * Zero terminate the buffer. 
 */
int read_file(const char *file, int fd, int new)
{
	int n, err;
	
	if (new) {
		if ((fd = open(file, O_RDONLY)) == -1) {
			err = errno;
			perror("Couldn't open file");
			return err;
		}
	}
	
	lseek(fd, 0, SEEK_SET);
	if ((n = read(fd, buf, sizeof(buf)-1)) < 0) {
		err = errno;
		perror("Couldn't read file");
		close(fd);
		return err;
	}
	buf[n] = '\0';

	if (new)
		close(fd);
	
	return 0;
}

/*
 * Reads /proc/stat into buf, and parses the output.
 *
 * Format of line:
 * ...
 * cpu<id> <user> <nice> <system> <idle> <iowait>
 */
int get_stat(cpuinfo_t *cpu)
{
	char *p1, *p2, searchfor[10];
	int err;
	
	if ((err = read_file("/proc/stat", cpu->fd, 0)) < 0) {
		perror("Error reading /proc/stat");
		return err;
	}

	sprintf(searchfor, "cpu%d ", cpu->cpuid);

	p1 = strstr(buf, searchfor);
	if (p1 == NULL) {
		perror("Error parsing /proc/stat");
		return ENOENT;
	}
	
	p2 = p1+strlen(searchfor);

	memcpy(cpu->last_reading, cpu->reading, sizeof(cpustats_t));
	
	cpu->reading->user = strtol(p2, &p2, 10);
	cpu->reading->mynice = strtol(p2, &p2, 10);
	cpu->reading->system = strtol(p2, &p2, 10);
	cpu->reading->idle = strtol(p2, &p2, 10);

	return 0;
}

/*
 * Once a decision is made, change the speed.
 */

int change_speed(cpuinfo_t *cpu, enum modes mode)
{
	int fd, len, err;
	char writestr[100];
	
	if (mode == RAISE) {
		if ((func == AGGRESSIVE) || (func == LEAPS)) {
			cpu->current_speed = cpu->max_speed;
		} else {
			cpu->current_speed += step;
			if (cpu->current_speed > cpu->max_speed)
				cpu->current_speed = cpu->max_speed;
		} 
	} else {
		if ((func == PASSIVE) || (func == LEAPS)) {
			cpu->current_speed = cpu->min_speed;
		} else {
			cpu->current_speed -= step;
			if (cpu->current_speed < cpu->min_speed)
				cpu->current_speed = cpu->min_speed;
		}
	}

	pprintf(3,"Setting speed to %d\n", cpu->current_speed);

	change_speed_count++;
	
	strncpy(writestr, cpu->sysfs_dir, 50);
	strncat(writestr, SYSFS_SETSPEED, 20);
	
	if ((fd = open(writestr, O_WRONLY)) < 0) {
		err = errno;
		perror("Can't open scaling_setspeed");
		return err;
	}

	lseek(fd, 0, SEEK_CUR);
	
	sprintf(writestr, "%d\n", (cpu->in_mhz) ?
			(cpu->current_speed / 1000) : cpu->current_speed); 

	pprintf(4,"mode=%d, str=%s", mode, writestr);
	
	if ((len = write(fd, writestr, strlen(writestr))) < 0) {
		err = errno;
		perror("Couldn't write to scaling_setspeed\n");
		return err;
	}

	if (len != strlen(writestr)) {
		printf("Could not write scaling_setspeed\n");
		close(fd);
		return EPIPE;
	}
	close(fd);

	return 0;
}

/*
 * The heart of the program... decide to raise or lower the speed.
 */
int inline decide_speed(cpuinfo_t *cpu)
{
	int err;
	float pct;
	unsigned long long usage, total;
	
	if ((err = get_stat(cpu)) < 0) {
		perror("Can't get stats");
		return err;
	}

	total = (cpu->reading->user - cpu->last_reading->user) +
		(cpu->reading->system - cpu->last_reading->system) +
		(cpu->reading->mynice - cpu->last_reading->mynice) +
		(cpu->reading->idle - cpu->last_reading->idle);

	if (ignore_nice) { 
		usage = (cpu->reading->user - cpu->last_reading->user) +
			(cpu->reading->system - cpu->last_reading->system);
	} else {
		usage = (cpu->reading->user - cpu->last_reading->user) +
			(cpu->reading->mynice - cpu->last_reading->mynice) +
			(cpu->reading->system - cpu->last_reading->system);
	}
	
	pct = ((float)usage)/((float)total);
	
	pprintf(4,"PCT = %f\n", pct);
	
	if ((pct > ((float)highwater/100.0)) && 
			(cpu->current_speed != cpu->max_speed)) {
		/* raise speed to next level */
		change_speed(cpu, RAISE);
	} else if ((pct < ((float)lowwater/100.0)) && 
			(cpu->current_speed != cpu->min_speed)) {
		/* lower speed */
		change_speed(cpu, LOWER);
	}
	
	return 0;
}

/*
 * Allocates and initialises the per-cpu data structures.
 */
int get_per_cpu_info(cpuinfo_t *cpu, int cpuid)
{
	char cpustr[100], scratch[100], tmp[10];
	int fd, err;
	
	cpu->cpuid = cpuid;
	cpu->sysfs_dir = (char *)malloc(50*sizeof(char));
	if (cpu->sysfs_dir == NULL) {
		perror("Couldn't allocate per-cpu sysfs_dir");
		return ENOMEM;
	}
	memset(cpu->sysfs_dir, 0, (50*sizeof(char)));

	strncpy(cpu->sysfs_dir, SYSFS_TREE, 30);
	sprintf(cpustr, "cpu%d/cpufreq/", cpuid);
	strncat(cpu->sysfs_dir, cpustr, 20);
	
	strncpy(scratch, cpu->sysfs_dir, 50);
	strncat(scratch, "cpuinfo_max_freq", 18);
	if ((err = read_file(scratch, 0, 1)) < 0) {
		perror("couldn't open max freq file");
		return err;
	}
	
	cpu->max_speed = strtol(buf, NULL, 10);

	strncpy(scratch, cpu->sysfs_dir, 50);
	strncat(scratch, "cpuinfo_min_freq", 18);

	if ((err = read_file(scratch, 0, 1)) < 0) {
		perror("couldn't open min freq file");
		return err;
	}

	cpu->min_speed = strtol(buf, NULL, 10);
	cpu->current_speed = cpu->max_speed;

	strncpy(scratch, cpu->sysfs_dir, 50);
	strncat(scratch, SYSFS_GOVERNOR, 20);

	if ((err = read_file(scratch, 0, 1)) < 0) {
		perror("couldn't open scaling_governors file");
		return err;
	}

	if (strncmp(buf, "userspace", 9) != 0) {
		if ((fd = open(scratch, O_RDWR)) < 0) {
			err = errno;
			perror("couldn't open govn's file for writing");
			return err;
		}
		strncpy(tmp, "userspace\n", 11);
		if (write(fd, tmp, 11*sizeof(char)) < 0) {
			err = errno;
			perror("Error writing file governor");
			close(fd);
			return err;
		}
		if ((err = read_file(scratch, fd, 0)) < 0) {
			perror("Error reading back governor file");
			close(fd);
			return err;
		}
		close(fd);
		if (strncmp(buf, "userspace", 9) != 0) {
			perror("Can't set to userspace governor, exiting");
			return EPIPE;
		}
	}
	
	cpu->last_reading = (cpustats_t *)malloc(sizeof(cpustats_t));
	cpu->reading = (cpustats_t *)malloc(sizeof(cpustats_t));
	memset(cpu->last_reading, 0, sizeof(cpustats_t));
	memset(cpu->reading, 0, sizeof(cpustats_t));
	
	/*
	 * Some cpufreq drivers (longhaul) report speeds in MHz instead
	 * of KHz.  Assume for now that any currently supported cpufreq 
	 * processor will a) not be faster then 10GHz, and b) not be slower
	 * then 10MHz. Therefore, is the number for max_speed is less than
	 * 10000, assume the driver is reporting speeds in MHz, not KHz,
	 * and adjust accordingly.
	 */
	cpu->in_mhz = 0;
	if (cpu->max_speed <= 10000) {
		cpu->in_mhz = 1;
		cpu->max_speed *= 1000;
		cpu->min_speed *= 1000;
		cpu->current_speed *= 1000;
	}
	
	if ((cpu->fd = open("/proc/stat", O_RDONLY)) < 0) {
		err = errno;
		perror("can't open /proc/stat");
		return err;
	}
	
	if ((err = get_stat(cpu)) < 0) {
		perror("can't read /proc/stat");
		return err;
	}
	
	return 0;
}

/*
 * Signal handler for SIGTERM/SIGINT... clean up after ourselves
 */
void terminate(int signum)
{
	int ncpus, i;
	cpuinfo_t *cpu;
	
	ncpus = sysconf(_SC_NPROCESSORS_CONF);
	
	for(i = ncpus; i > 0; i--) {
		cpu = cpuinfo[i-1];
		/* force ourselves back up to the highest Mhz speed */
		func = LEAPS;
		change_speed(cpu, RAISE);
		/* close the /proc/stat fd */
		close(cpu->fd);
		/* deallocate everything */
		free(cpu->sysfs_dir);
		free(cpu->last_reading);
		free(cpu->reading);
		free(cpu);
	}
	time_t duration = time(NULL) - start_time;
	pprintf(1,"Statistics:\n");
	pprintf(1,"  %d speed changes in %d seconds\n",
			change_speed_count, (unsigned int) duration);
	pprintf(0,"PowerNow Daemon Exiting.\n");

	closelog();

	exit(0);
}

const char *str_func(void)
{
	switch (func) {
		case SINE: return "SINE";
		case AGGRESSIVE: return "AGGRESSIVE";
		case PASSIVE: return "PASSIVE";
		case LEAPS: return "LEAPS";
		default: return "UNKNOWN";
	}
}

/* 
 * Main program loop.. parse arguments, sanity chacks, setup signal handlers
 * and then enter main loop
 */
int main(int argc, char **argv)
{
	cpuinfo_t *cpu;
	int ncpus, i, err;

	/* Parse command line args */
	while(1) {
		int c;

		c = getopt(argc, argv, "dnvqm:s:p:u:l:h");
		if (c == -1)
			break;

		switch(c) {
			case 'd':
				daemonize = 0;
				break;
			case 'n':
				ignore_nice = 0;
				break;
 			case 'v':
 				verbosity++;
				if (verbosity > 10) verbosity = 10;
 				break;
 			case 'q':
 				verbosity = -1;
 				break;
			case 'm':
				func = strtol(optarg, NULL, 10);
				if ((func < 0) || (func > 3)) {
					printf("Invalid mode specified");
					help();
					exit(ENOTSUP);
				}
				pprintf(2,"Using %s mode.\n", str_func());
				break;
			case 's':
				step = strtol(optarg, NULL, 10);
				if (step < 0) {
					printf("step must be non-negative");
					help();
					exit(ENOTSUP);
				}
				pprintf(2,"Using %dHz step.\n", step);
				break;
			case 'p':
				poll = strtol(optarg, NULL, 10);
				if (poll < 0) {
					printf("poll must be non-negative");
					help();
					exit(ENOTSUP);
				}
				pprintf(2,"Polling every %d msecs\n", poll);
				break;
			case 'u':
				highwater = strtol(optarg, NULL, 10);
				if ((highwater < 0) || (highwater > 100)) {
					printf("upper limit must be between 0 and 100\n");
					help();
					exit(ENOTSUP);
				}
				pprintf(2,"Using upper limit of %d%%\n",highwater);
				break;
			case 'l':
				lowwater = strtol(optarg, NULL, 10);
				if ((lowwater < 0) || (lowwater > 100)) {
					printf("lower limit must be between 0 and 100\n");
					help();
					exit(ENOTSUP);
				}
				pprintf(2,"Using lower limit of %d%%\n",lowwater);
				break;
			case 'h':
			default:
				help();
				return 0;
		}
	}

	/* one last thing to check... */
	if (lowwater > highwater) {
		printf("Invalid: lower limit higher then upper limit!\n");
		help();
		exit(ENOTSUP);
	}
	
	/* so we don't interfere with anything, including ourself */
	nice(5);
	
	if (daemonize)
		openlog("powernowd", LOG_AUTHPRIV|LOG_PERROR, LOG_DAEMON);
	
	/* My ego's pretty big... */
	pprintf(0,"PowerNow Daemon v%s, (c) 2003-2004 John Clemens\n", 
			VERSION);

	/* are we root?? */
	if (getuid() != 0) {
		printf("Go away, you are not root. Only root can run me.\n");
		exit(EPERM);
	}

	pprintf(1,"Settings:\n");
	pprintf(1,"  verbosity:     %4d\n", verbosity);
	pprintf(1,"  mode:          %4d     (%s)\n", func, str_func());
	pprintf(1,"  step:          %4d MHz (%d kHz)\n", step/1000, step);
	pprintf(1,"  lowwater:      %4d %%\n", lowwater);
	pprintf(1,"  highwater:     %4d %%\n", highwater);
	pprintf(1,"  poll interval: %4d ms\n", poll);
	
	ncpus = sysconf(_SC_NPROCESSORS_CONF);
	pprintf(0,"Found %d cpu%s:\n", ncpus, (ncpus==1)?"":"s");

	/* Malloc, initialise data structs */
	cpu = malloc(sizeof(cpuinfo_t)*ncpus);
	
	if (cpu == (cpuinfo_t *)NULL) {
		perror("Couldn't malloc cpuinfo");
		return ENOMEM;
	}
	memset(cpu, 0, (sizeof(cpuinfo_t)*ncpus));
	cpuinfo = &cpu;

	for (i=0;i<ncpus;i++) {
		if ((err = get_per_cpu_info(&cpu[i], i)) != 0) {
			perror("Couldn't get per-cpu data");
			goto out;
		}
		pprintf(0,"  cpu%d: %dMhz - %dMhz\n", cpu->cpuid, 
				cpu->min_speed / 1000, 
				cpu->max_speed / 1000);
	}
	
	/* now that everything's all set up, lets set up a exit handler */
	signal(SIGTERM, terminate);
	signal(SIGINT, terminate);
	
	if (daemonize)
		daemon(0, 0);

	start_time = time(NULL);

	while(1) {
		usleep(poll*1000);
		for(i=0; i<ncpus; i++) {
			decide_speed(&cpu[i]);
		}
	}

out:
	printf("PowerNowd encountered and error and could not start.\n");
	printf("Please make sure that:\n");
	printf(" - You are running a v2.5/v2.6 kernel or later\n");
	printf(" - That you have sysfs mounted /sys\n");
	printf(" - That you have the core cpufreq and cpufreq-userspace\n");
	printf("   modules loaded into your kernel\n");
	printf(" - That you have the cpufreq driver for your cpu loaded,\n");
	printf("   and that it works. (check dmesg for errors)\n");
	printf("If all of the above are true, and you still have problems,\n");
	printf("please email the author: clemej@alum.rpi.edu\n");
	
	/* should free more here.. will get to that later.... */
	/* or we can just be lazy and let the OS do it for us... */
	free(cpu);
	return err;
}

