/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef POWERSAVE_DBUS_SERVER_H
#define POWERSAVE_DBUS_SERVER_H
#define DBUS_API_SUBJECT_TO_CHANGE 1

#define DBUS_HAL_MANAGER_INTERFACE	"org.freedesktop.Hal.Manager"
#define DBUS_HAL_DEVICE_INTERFACE	"org.freedesktop.Hal.Device"
#define DBUS_HAL_UDI_COMPUTER		"/org/freedesktop/Hal/devices/computer"
#define DBUS_HAL_INTERFACE		"org.freedesktop.Hal"
#define DBUS_HAL_CPUFREQ_INTERFACE	"org.freedesktop.Hal.Device.CPUFreq"
#define DBUS_SYSTEM_PM_INTERFACE	"org.freedesktop.Policy.Power"
#define DBUS_HAL_SYSTEM_POWER_INTERFACE	"org.freedesktop.Hal.Device.SystemPowerManagement"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <string>

class PM_Interface;

/** 
 * @defgroup DBus Server class
 *
 * @brief The DBus Server sets up the dbus
 * connection and registers our name with it. It also contains the filter
 * function and cares about reconnections.
 */
class DBus_Server {
public:
	DBus_Server();
	~DBus_Server();

	/**
	 * @brief Opens a new connection to the system bus,
	 *        tries to register the service and returns
	 *        a filedescriptor to watch for
	 *
	 * @return    filedescriptor to watch on or negative value on failure         
	 */
	static DBusConnection *openSystemConnection();

	/** @brief emits a signal over the system bus
	 *
	 * @param signal the signal to emit
	 * @param value the value of the signal
	 * 
	 * @return DBUS_ERROR
	 */
	static int emitSignal(const char *signal, const char *value);

	/** @brief send a method call over the bus
	 *
	 * @param interface the interface to sent to
	 * @param path the object path to sent to
	 * @param object the object to sent to
	 * @param method the method to sent
	 * @param first argument type
	 * @param first argument
	 *
	 * @return true if successfull, false otherwise
	 */
	static bool sendMethodCall(const char *interface, const char *path,
				   const char* object, const char *method,
				   int first_arg_type, ...);

	static bool sendMethodCallAndBlock(const char *interface, const char *path,
					   const char* object, const char *method,
					   int first_arg_type, ...);

	/** @brief sends a method call with a return value
	 *
	 * @param interface the interface to sent to
	 * @param path the object path to sent to
	 * @param object the object to sent to
	 * @param method the method to sent
	 * @param arg_type the argument type to get from the message
	 * @param value to store the value
	 *
	 * @return true if successfull, false otherwise
	 */
	static bool sendMethodCallWithReturn(const char *interface, const char *path,
					     const char* object ,const char *method,
					     int arg_type, void *value);

	/** @brief check if a service is registered on the bus
	 *
	 * @param service the service to check for
	 *
	 * @return true if successfull, false otherwise
	 */
	static bool serviceHasOwner(const std::string &service);
	
	/** @brief check if another PolicyManager is available */
	static bool haveClient();

	/** @brief tries to reconnect to the dbus service 
	 *
	 * declared static to be used as callback function in main loop
	 */
	static gboolean dbus_reconnect();
	static DBusConnection* _connection;
private:
	static DBusHandlerResult filter_function(DBusConnection * conn, DBusMessage * msg, void *data);
	
	static void sendReply(DBusMessage * reply, int error_id);
};

#endif // POWERSAVE_DBUS_SERVER_H
