/*****
*
* Copyright (C) 2001, 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Yoann Vandoorselaere <yoann@prelude-ids.org>
*
*****/


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <inttypes.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <assert.h>

#include <rpc/rpc.h>
#include <rpc/xdr.h>
#include <rpc/rpc_msg.h>

#include <libprelude/prelude-log.h>
#include <libprelude/extract.h>

#include "protocol.h"
#include "rpc-prv.h"
#include "rpc-plugin.h"



extern int rpc_plugin_id;
extern port_list_t *port_list;



int match_rpc_rule(packet_container_t *packet, void *data) 
{
        uint32_t tmp;
	rpc_hdr_t *hdr;
	rpc_test_t *test;
        
        if ( packet->protocol_plugin_id != rpc_plugin_id ) 
                return -1;
        
	test = data;
	hdr = packet->protocol_plugin_data;
        
        tmp = extract_uint32(&hdr->program);
        if ( test->flags & RPC_CHECK_PROGRAM && test->program != tmp ) 
                return -1;

        tmp = extract_uint32(&hdr->version);
        if ( test->flags & RPC_CHECK_VERSION && test->version != tmp )
                return -1;

        tmp = extract_uint32(&hdr->procedure);
        if ( test->flags & RPC_CHECK_PROCEDURE && test->procedure != tmp )
                return -1;
        
        return 0;
}




static int decode_rpc_request(packet_container_t *pc, unsigned char *data) 
{
        uint32_t tmp;
 	rpc_hdr_t *hdr;
        
	hdr = (rpc_hdr_t *) data;

	/*
         * Read direction : CALL or REPLY, only look at calls.
         */
        tmp = extract_uint32(&hdr->call);
        if ( tmp != CALL ) 
		return -1;
        
        /*
         * Read the RPC message version, fail if it is not right.
         */
        tmp = extract_uint32(&hdr->rpc_version);
        if ( tmp != RPC_MSG_VERSION ) 
		return -1;
        
        pc->protocol_plugin_data = data;
  	pc->protocol_plugin_id = rpc_plugin_id;

        return 0;
}




/*
 * From RFC 1831 :
 *
 * 10. RECORD MARKING STANDARD 
 * When RPC messages are passed on top of a byte stream transport protocol (like TCP),
 * it is necessary to delimit one message from another in order to detect and possibly
 * recover from protocol errors. This is called record marking (RM). One RPC message
 * fits into one RM record. 
 *
 * A record is composed of one or more record fragments. A record fragment is a four-byte
 * header followed by 0 to (2**31) - 1 bytes of fragment data.
 *
 * The bytes encode an unsigned
 * binary number; as with XDR integers, the byte order is from highest to lowest. The number
 * encodes two values -- a boolean which indicates whether the fragment is the last fragment
 * of the record (bit value 1 implies the fragment is the last fragment) and a 31-bit
 * unsigned binary value which is the length in bytes of the fragment's data. The boolean
 * value is the highest-order bit of the header; the length is the 31 low-order bits. (Note
 * that this record specification is NOT in XDR standard form!) 
 */
static int reasm_rpc_fragments(packet_container_t *pc, unsigned char *data, int len) 
{
        uint32_t record, rlen, totlen = 0;
        uint8_t last_fragment, *write_ptr, *ptr;
        
        ptr = data;
        write_ptr = data + 4;
        
        while ( ptr < (data + len) ) {
                
                if ( (ptr + 4) > (data + len) )
                        return -1;
                
                record = extract_uint32(ptr);
                
                rlen = record & 0x7fffffff;
             
                if ( (ptr + 4 + rlen) > (data + len) ) {
                        log(LOG_ERR, "error calculating record length (%d > %d).\n", rlen, len - 4);
                        return -1;
                }
                
                ptr += 4;
                totlen += rlen;
                memcpy(write_ptr, ptr, rlen);

                ptr += rlen;
                write_ptr += rlen;

                last_fragment = (record & 0x80000000) >> 31;

#if 0
                printf("len=%d, rlen=%d, last=%d\n", len, rlen, last_fragment);
#endif                           
                if ( last_fragment )
                        break;
                
        }

        /*
         * Rebuild the fragment record. 
         */
        
        data[0] = 0x80 ^ (totlen >> 8);
        data[1] = totlen >> 16;
        data[2] = totlen >> 24;
        data[3] = totlen;
        
#if 0
        printf("hexval=%x totlen=%x, skip=%d\n", (*(uint32_t*)data), totlen, len - (totlen + 4));
#endif
        /*
         * If we encounter several record fragment,
         * we now have only one fragment record for all the reassembled
         * fragments. Adjust len.
         */
        return len - (totlen + 4);
}




int decode_rpc(packet_container_t *pc, unsigned char *data, int len) 
{
        int ret, skip;
        uint16_t dport;
        int depth = pc->transport_layer_depth;

        if ( depth < 0 )
                return -1;
	
        else if ( pc->packet[depth].proto == p_tcp ) {
		if ( len < RPC_HDR_SIZE + 4 )
                        return -1;

                dport = extract_uint16(&pc->packet[depth].p.tcp->th_dport);
		
                ret = protocol_plugin_is_port_ok(port_list, dport);
                if ( ret < 0 )
                        return -1;
                
                skip = reasm_rpc_fragments(pc, data, len);
                if ( skip < 0 )
                        return -1;
                
                ret = decode_rpc_request(pc, data + 4);
                if ( ret < 0 )
                        return -1;

                return skip;
        }
	
        else if ( pc->packet[depth].proto == p_udp ) {
		if ( len < RPC_HDR_SIZE )
			return -1;

                dport = extract_uint16(&pc->packet[depth].p.udp_hdr->uh_dport);
                
                ret = protocol_plugin_is_port_ok(port_list, dport);
                if ( ret < 0 )
                        return -1;
                
                return decode_rpc_request(pc, data);
        }
	
	return -1;
}
