/*****
*
* Copyright (C) 2000, 2002, 2003 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

/*
 * INET		An implementation of the TCP/IP protocol suite for the LINUX
 *		operating system.  INET is implemented using the  BSD Socket
 *		interface as the means of communication with the user level.
 *
 *		The IP fragmentation functionality.
 *		
 *
 * Authors:	Fred N. van Kempen <waltje@uWalt.NL.Mugnet.ORG>
 *		Alan Cox <Alan.Cox@linux.org>
 *
 * Fixes:
 *		Alan Cox	:	Split from ip.c , see ip_input.c for history.
 *		David S. Miller :	Begin massive cleanup...
 *		Andi Kleen	:	Add sysctls.
 *		xxxx		:	Overlapfrag bug.
 *		Ultima          :       ip_expire() kernel panic.
 *		Bill Hawes	:	Frag accounting and evictor fixes.
 *		John McDonald	:	0 length frag bug.
 *		Alexey Kuznetsov:	SMP races, threading, cleanup.
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <inttypes.h>

#include "packet.h"

#include <libprelude/list.h>
#include <libprelude/prelude-log.h>
#include <libprelude/plugin-common.h>
#include <libprelude/plugin-common-prv.h>
#include <libprelude/extract.h>
#include <libprelude/timer.h>

#include "nids-alert.h"
#include "packet-decode.h"
#include "ip-fragment.h"
#include "hostdb.h"



/*
 * "Fragment Offset" part
 */
#define IP_OFFSET       0x1FFF



typedef struct frag_item_t
{
        int offset; /* offset of frag in Ip datagram */
        int end;    /* last byte of data in datagram */
        int len;    /* lenght of this fragment       */
        unsigned char *data;
        struct frag_item_t *next;
} frag_item_t;


/*
 * Describe an entry in the "incomplete datagrams" queue.
 */
struct ipq {
        packet_container_t *packet;
        iphdr_t *ip;
	uint8_t		last_in;
        
#define FIRST_IN		2
#define LAST_IN			1

	frag_item_t	*fragments;	/* linked list of received fragments */
	int		len;		/* total length of original datagram */
	int		meat;
	prelude_timer_t         timer;	        /* when will this queue expire?	     */
};


static int my_id = 0;


#define setup_frag_alert(class) \
        setup_alert(class, sizeof(class))


static nids_alert_t *setup_alert(const char *class, int clen) 
{
        static int initialized = 0;
        static nids_alert_t alert;
        static idmef_impact_t impact;

        if ( ! initialized ) {
                initialized = 1;
                nids_alert_init(&alert);
                alert.impact = &impact;
                impact.type = dos;
                impact.completion = 0; /* we don't know */
                impact.severity = impact_low;
                idmef_string_set_constant(&impact.description, "Might result in an operating system crash");
        }
        
        alert.classification.name.len = clen;
        alert.classification.name.string = class;
        
        return &alert;
}





/*
 * Destruction primitives.
 */

/*
 * Complete destruction of ipq.
 */
static void ip_frag_destroy(struct ipq *qp)
{
	frag_item_t *fp, *next;
        
	/*
         * Release all fragment data.
         */
	fp = qp->fragments;
	while (fp) {

                next = fp->next;

                free(fp->data);
                free(fp);

                fp = next;
	}

        packet_release(qp->packet);
        
	/*
         * Finally, release the queue descriptor itself.
         */
        free(qp);
}



/*
 * Kill ipq entry.
 */
static __inline__ void ipq_kill(hostdb_t *hdb, struct ipq *ipq)
{
        hostdb_del(hdb, my_id);
        timer_destroy(&ipq->timer);
        ip_frag_destroy(ipq);
}




/*
 * Oops, a fragment queue timed out.  Kill it.
 */
static void ip_expire(void *arg)
{
        struct ipq *qp;
        hostdb_t *hdb = arg;

        qp = (struct ipq *) hostdb_get_plugin_data(hdb, my_id);

        if( ! qp->fragments ) {
                
                nids_alert(NULL, qp->packet, setup_frag_alert("IP Fragmentation Expire attack"), 
                           "A packet with the fragment bit turned on was received but no "
                           "fragment were sent by this host");
        }
        
        ipq_kill(hdb, qp);
}



/*
 * Add an entry to the 'ipq' queue for a newly received IP datagram.
 */
static struct ipq *ip_frag_create(packet_container_t *packet, hostdb_t *hdb, iphdr_t *iph)
{
	struct ipq *qp;

        qp = malloc(sizeof(struct ipq));
        if ( ! qp ) {
                log(LOG_ERR,"ip_frag_create(): no memory left.\n");
                return NULL;
        }
        
        qp->ip = iph;
        qp->packet = packet;
        qp->len = 0;
        qp->last_in = 0;
	qp->meat = 0;
	qp->fragments = NULL;
        
	/*
         * Initialize a timer for this entry.
         */
        timer_set_data(&qp->timer, hdb);
        timer_set_expire(&qp->timer, 30);
        timer_set_callback(&qp->timer, ip_expire);
        timer_init(&qp->timer);

        packet_lock(packet);
        
	return qp;
}




/*
 * Add new segment to existing queue.
 */
static int ip_frag_queue(packet_container_t *packet, struct ipq *qp, const iphdr_t *ip, frag_item_t *frag)
{        
        int ihl, end, flags;
        uint16_t len, offset;
	frag_item_t *prev, *next;

        offset = extract_uint16(&ip->ip_off);
        
        flags = offset & ~IP_OFFSET;
	offset &= IP_OFFSET;
	offset <<= 3;		/* offset is in 8-byte chunks */
	ihl = IP_HL(ip) * 4;

        frag->offset = offset;
        
	/*
         * Determine the position of this fragment.
         */
        len = extract_uint16(&ip->ip_len);
	end = offset + (len - ihl);
        
	/*
         * Is this the final fragment ?
         */
	if ((flags & IP_MF) == 0) {
                /*
                 * If we already have some bits beyond end
		 * or have different end, the segment is corrrupted.
		 */
		if (end < qp->len ) {
                        nids_alert(NULL, packet, setup_frag_alert("Ip Fragmentation attack"),
                                   "Received last fragment of a fragmented stream with end of data "
                                   "at byte %d, but current received data len (%d) is beyond that end", end, qp->len);
                        return -1;
                }

                if ( (qp->last_in & LAST_IN) && end != qp->len ) {
                        nids_alert(NULL, packet, setup_frag_alert("Ip fragmentation attack"),
                                   "Received last fragment of a fragmented stream with end of data "
                                   "at byte %d; then received another last fragment for the same stream "
                                   "with end of data at %d, which isn't equal to first end of data gathered",
                                   qp->len, end);
                        return -1;
               }
                
		qp->last_in |= LAST_IN;
		qp->len = end;
	} else {
		if (end & 7) 
			end &= ~7;
		
		if (end > qp->len) {
			/*
                         * Some bits beyond end -> corruption.
                         */
			if (qp->last_in & LAST_IN) {
                                nids_alert(NULL, packet, setup_frag_alert("Ip fragmentation attack"),
                                           "Received last fragment of a fragmented stream with end pf data "
                                           "at byte %d; then received another packet with MF flag set and "
                                           "end of data at byte %d (beyond the end)", qp->len, end);
                                return -1;
                        }
                        
			qp->len = end;
		}
	}
        
	if (end == offset) 
		return -1;
        
	
	/*
         * Find out which fragments are in front and at the back of us
	 * in the chain of fragments so far.  We must know where to put
	 * this fragment, right?
	 */
        prev = NULL;
        
        for ( next = qp->fragments; next != NULL; next = next->next ) {
                if ( next->offset >= offset )
			break;	/* bingo! */

                prev = next;
	}
        

	/*
         * We found where to put this one.  Check for overlap with
	 * preceding fragment, and, if needed, align things so that
	 * any overlaps are eliminated.
	 */
	if ( prev ) {
		int i = (prev->offset + prev->len) - offset;
                
		if ( i > 0 ) {
			offset += i;
                        nids_alert(NULL, packet, setup_frag_alert("Ip Fragmentation attack"),
                                   "IP fragment overlaping attack, %d byte", i);
			if ( end <= offset )
				return -1;
		}
	}

	while ( next && next->offset < end ) {
		int i = end - next->offset; /* overlap is 'i' bytes */
                
		if (i < next->len) {
			/*
                         * Eat head of the next overlapped fragment
			 * and leave the loop. The next ones cannot overlap.
			 */
			next->offset += i;
			qp->meat -= i;
                        nids_alert(NULL, packet, setup_frag_alert("Ip Fragmentation attack"),
                                   "IP fragment overlaping attack, %d byte", i);
                        break;
		} else {
			frag_item_t *free_it = next;

			/*
                         * Old fragment is completely overridden with
			 * new one drop it.
			 */
                        nids_alert(NULL, packet, setup_frag_alert("Ip Fragmentation attack"),
                                   "IP fragment overlaping attack, %d byte", i);
			next = next->next;

			if (prev)
				prev->next = next;
			else
				qp->fragments = next;

			qp->meat -= free_it->len;

                        free(free_it->data);
                        free(free_it);
		}
	}

        /*
         * Insert this fragment in the chain of fragments.
         */
              
	frag->next = next;
	if (prev)
		prev->next = frag;
	else
		qp->fragments = frag;

        qp->meat += frag->len;
        if (offset == 0) 
		qp->last_in |= FIRST_IN;
        
        return 0;
}




/*
 * Build a new IP datagram from all its fragments.
 *
 * FIXME: We copy here because we lack an effective way of handling lists
 * of bits on input. Until the new skb data handling is in I'm not going
 * to touch this with a bargepole. 
 */
static int ip_frag_reasm(hostdb_t *hdb, struct ipq *qp, unsigned char **dp)
{
        unsigned char *p;
        int ihlen, len, nlen;
        iphdr_t *iph = qp->ip;
        frag_item_t *fp, *bkp;
        
        /*
         * Allocate a new buffer for the datagram.
         */

        ihlen = IP_HL(iph) * 4;
	len = ihlen + qp->len;

	if (len > 65535) {                        
                nids_alert(NULL, qp->packet, setup_frag_alert("Ip Fragmentation Attack"), 
                           "Oversized packet (len=%d > 65535) while doing reassembly", len);
                return -1;
        }
        
        *dp = p = malloc(len);
	if ( ! p ) {
                log(LOG_ERR,"malloc(%d).\n", len);
		return -1;
        }

        hostdb_del(hdb, my_id);
        timer_destroy(&qp->timer);

        /*
         * Fixup the new IP headers,
         * we have to use memcpy to avoid alignment issue.
         */
        nlen = htons(len);
        memcpy(&iph->ip_len, &nlen, sizeof(iph->ip_len));

        nlen = htons(0);
        memcpy(&iph->ip_off, &nlen, sizeof(iph->ip_len));
        
        /*
         * Copy the original IP headers into the new buffer.
         */
        memcpy(p, iph, ihlen);
        p += ihlen;
        
	/*
         * Copy the data portions of all fragments into the new buffer.
         */
	for (fp = qp->fragments; fp; fp = bkp) {
                
                memcpy(p, fp->data, fp->len);
                p += fp->len;
                
                free(fp->data);

                bkp = fp->next;
                free(fp);
        }

        packet_release(qp->packet);
        free(qp);
        
	return len;
}




static frag_item_t *nfrag(const unsigned char *data, int len) 
{
        frag_item_t *f = NULL;

        f = (frag_item_t *) malloc(sizeof(frag_item_t));
        if ( ! f ) {
                log(LOG_ERR,"malloc(%d).\n", sizeof(frag_item_t));
                return NULL;
        }

        f->len = len;

        f->data = (unsigned char *) malloc(len);
        if ( ! f->data ) {
                log(LOG_ERR,"malloc(%d).\n", len);
                free(f);
                return NULL;
        }
        
        memcpy(f->data, data, len);
        
        f->next = NULL;
        
        return f;
}




/* Process an incoming IP datagram fragment. */
int ip_defrag(packet_container_t *packet, iphdr_t *ip, const unsigned char *data,
              int len, unsigned char **dp)
{
        int ret;
        hostdb_t *hdb;
        frag_item_t *f;
        struct ipq *qp = NULL;

        timer_lock_critical_region();
        
        hdb = hostdb_search(ip);
        if ( ! hdb ) {
                hdb = hostdb_new(packet, ip);
                if (! hdb )
                        goto err;
                
        } else 
                qp = (struct ipq *) hostdb_get_plugin_data(hdb, my_id);
                
        if ( ! qp ) {
                qp = ip_frag_create(packet, hdb, ip);
                if ( ! qp ) {
                        hostdb_del(hdb, my_id);
                        goto err;
                }
                
                hostdb_set_plugin_data(hdb, my_id, qp);
        } else 
                timer_reset(&qp->timer);
                
        /*
         * Lookup (or create) queue header
         */
        f = nfrag(data, len);        
        if ( ! f ) {
                ipq_kill(hdb, qp);
                goto err;
        }
                
        ret = ip_frag_queue(packet, qp, ip, f);
        if ( ret < 0 ) {
                ipq_kill(hdb, qp);
                free(f->data);
                free(f);
                goto err;
        }
        
        if (qp->last_in == (FIRST_IN|LAST_IN) && qp->meat == qp->len) {
                ret = ip_frag_reasm(hdb, qp, dp);
                if ( ret < 0 )
                        ipq_kill(hdb, qp);
        }

        timer_unlock_critical_region();
        return ret;
        
 err:
        timer_unlock_critical_region();
        return -1;
}




void ip_defrag_init(void) 
{
        my_id = plugin_request_new_id();
}

