/*****
*
* Copyright (C) 1998, 1999, 2000, 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <stdlib.h>
#include <signal.h>
#include <inttypes.h>

#include "config.h"

#include "packet.h"
#include "nids-alert.h"

#include <libprelude/list.h>
#include <libprelude/timer.h>
#include <libprelude/plugin-common.h>
#include <libprelude/plugin-common-prv.h>
#include <libprelude/prelude-log.h>
#include <libprelude/sensor.h>

#include "pconfig.h"
#include "capture.h"
#include "plugin-detect.h"
#include "plugin-protocol.h"
#include "pcap.h"
#include "packet-decode.h"
#include "rules.h"
#include "rules-default.h"
#include "ip-fragment.h"
#include "hostdb.h"


Pconfig_t config;


static void cleanup(void) 
{
        capture_stats();
        
        capture_stop();

        plugins_print_stats();
}



static void sighandler(int sig)
{
	fprintf(stderr, "\n\nCaught signal %d.\n", sig);

        signal(sig, SIG_DFL);

        cleanup();
        
        /*
         * Currently, exit with status 2 for gprof analysis.
         */
        exit(2);
}




int main(int argc, char **argv)
{
        int ret;
                
        if (getuid() != 0 || geteuid() != 0) {
                fprintf(stderr, "\n\nPrelude need root access in order to run.\n\n");
		exit(1);
	}
        
        ret = protocol_plugins_init(PROTOCOL_PLUGIN_DIR, argc, argv);
        if ( ret < 0 ) {
                log(LOG_INFO, "error initializing protocols plugins.\n");
                return -1;
        }
        log(LOG_INFO, "- Initialized %d protocols plugins.\n", ret);
        
        ret = detect_plugins_init(DETECT_PLUGIN_DIR, argc, argv);
        if ( ret < 0 ) {
                log(LOG_INFO, "error initializing detection plugins.\n");
                return -1;
        }
        log(LOG_INFO, "- Initialized %d detections plugins.\n\n", ret);

        /*
	 * Rules engine initialization
	 */
	signature_engine_init();
        
        if ( pconfig_set(argc, argv) < 0 )
                exit(1);

        
        nids_alert_init_subsystem();
        hostdb_init();
        ip_defrag_init();
        
        /*
         * Creation of signal handler.
         */
	signal(SIGTERM, sighandler);
	signal(SIGINT, sighandler);
        signal(SIGQUIT, sighandler);
        signal(SIGABRT, sighandler);

        /*
         * Prelude capture begin here.
         */
        prelude_log_set_prefix(NULL);

        log(LOG_INFO, "\n- Initializing packet capture.\n");
        ret = capture_start();

        timer_flush();
        
        cleanup();

        exit(ret);
}




