/* Copyright (C) 1999-2000 Bernhard Trummer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: imagenode.cc,v $
 * Revision 1.3  2001/04/27 20:02:50  slash
 * Make use of xmlChildrenNode and xmlRootNode.
 *
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:25  slash
 * Initial release.
 *
 */

#include "main.h"
#include "parameters.h"
#include "imgnode.h"
#include "imagenode.h"


// The following macro is used to provide an easy handling,
// if the apropriate property isn't given in the XML-file.
#define ALIGN_        (align_ != UNDEFINED ? align_ : CENTER)
#define LATEX_SCALE_  (latex_scale_ != -1 ? latex_scale_ : 1)

//---------------------------------------------------------------------------
ImageNode::ImageNode(xmlNodePtr node) : EnvironmentNode()
{
    align_ = UNDEFINED;
    latex_scale_ = -1;

    // Extract the properties of the image-node.
    char *align = (char*)xmlGetProp(node, (xmlChar*)"align");
    if (align) {
        if      (!strcmp(align, "left"))   align_ = LEFT;
        else if (!strcmp(align, "center")) align_ = CENTER;
        else if (!strcmp(align, "right"))  align_ = RIGHT;
    }
    char *latex_scale = (char*)xmlGetProp(node, (xmlChar*)"latex_scale");
    if (latex_scale){
        if      (!strcmp(latex_scale, "yes")) latex_scale_ = 1;
        else if (!strcmp(latex_scale, "no"))  latex_scale_ = 0;
    }

    // Extract the images.
    for (node = node->xmlChildrenNode; node; node = node->next) {
        if (!strcmp((char*)node->name, "img")) {
            child_nodes_.push_back(new ImgNode(node, LATEX_SCALE_));
        } else {
            Number_Of_Errors_++;
        }
    }
}

//---------------------------------------------------------------------------
ImageNode::~ImageNode()
{
}


//---------------------------------------------------------------------------
void ImageNode::writeHTML(int which_image) const
{
    if (which_image < 0) {
        which_image = 1;
    } else if ((unsigned)which_image > child_nodes_.size()) {
        which_image = child_nodes_.size();
    }

    // If the 'align' attribute is given in the <IMG> tag, it will lead to
    // strange results. So the image has to be embedded in a table.
    Output_ << "<TABLE"
            << " summary=\"image-table\""
            << " width=\"100%\""
            << " border=\"0\""
            << " cellpadding=\"0\""
            << " cellspacing=\"0\""
            << "><TR><TD";
    switch (ALIGN_) {
      case UNDEFINED: break;
      case LEFT:   Output_ << " align=\"left\""; break;
      case CENTER: Output_ << " align=\"center\""; break;
      case RIGHT:  Output_ << " align=\"right\""; break;
    }
    Output_ << ">" << endl;

    child_nodes_[which_image-1]->writeHTML();

    Output_ << "</TD></TR></TABLE>" << endl;
}

//---------------------------------------------------------------------------
void ImageNode::writeLaTeX() const
{
    if (Parameters_.isOutlineEnabled()) {
        for (unsigned int i=0; i<child_nodes_.size(); i++) {
            if (child_nodes_.size() > 2) {
                Output_ << "\\parbox[t]{" << 0.46 << "\\textwidth}{" << endl;
                child_nodes_[i]->writeLaTeX();
                Output_ << "}" << endl;
                if ((i%2 == 0) && (i < child_nodes_.size())) {
                    Output_ << "\\\\" << endl;
                }
            } else {
                child_nodes_[i]->writeLaTeX();
            }
        }

    } else {
        Output_ << "\\begin{figure}[ht]" << endl;
        for (unsigned int i=0; i<child_nodes_.size(); i++) {
            if (child_nodes_.size() > 2) {
                // Scale the images down using minipages.
                Output_ << "\\begin{minipage}{0.46\\linewidth}" << endl;
                child_nodes_[i]->writeLaTeX();
                Output_ << "\\end{minipage}" << endl;
                if ((i%2 == 0) && (i < child_nodes_.size())) {
                    Output_ << "\\hspace{0.04\\linewidth}" << endl;
                }
            } else {
                child_nodes_[i]->writeLaTeX();
            }
        }
        Output_ << "\\end{figure}" << endl;
    }
}

//---------------------------------------------------------------------------
void ImageNode::writeXML() const
{
    Output_ << "<image";
    switch (align_) {
      case UNDEFINED: break;
      case LEFT:   Output_ << " align=\"left\""; break;
      case CENTER: Output_ << " align=\"center\""; break;
      case RIGHT:  Output_ << " align=\"right\""; break;
    }
    if (latex_scale_ != -1) {
        Output_ << " latex_scale=\"" << (latex_scale_ == 1 ? "yes" : "no") << "\"";
    }
    Output_ << ">" << endl;
    
    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeXML();
    }

    Output_ << "</image>" << endl;
}

//---------------------------------------------------------------------------
int ImageNode::isHTMLStepped() const
{
    return (child_nodes_.size() > 1 ? 1 : 0);
}

//-----------------------------------------------------------------------
unsigned int ImageNode::getNumberOfHTMLSteps() const
{
    return child_nodes_.size();
}
