/****************************************************************************
** common.cpp - contains all the common variables and functions for Psi
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"common.h"
#include"profiles.h"

QString PROG_NAME = "Psi";
QString PROG_VERSION = "0.8.5";

#include<qregexp.h>
#include<qdir.h>
#include<qfile.h>
#include<qapplication.h>

#include<stdio.h>

#ifdef Q_WS_X11
#include<sys/types.h>
#include<sys/stat.h>
#endif

#ifdef Q_WS_WIN
#include<windows.h>
#endif

#ifdef Q_WS_MAC
#include <Carbon/Carbon.h> // for HIToolbox/InternetConfig
#include <CoreServices/CoreServices.h>
#endif

QString activeProfile;

TransportIconSet tic[5];

QPixmap *pix_online,
	*pix_offline,
	*pix_away,
	*pix_xa,
	*pix_dnd,
	*pix_blank,
	*pix_arrow[3],
	*pix_add,
	*pix_remove,
	*pix_send,
	*pix_history,
	*pix_info,
	*pix_url,
	*pix_ssl_yes,
	*pix_ssl_no,
	*pix_logo,
	*pix_changeacc,
	*pix_main,
	*pix_bigIcon,
	*pix_chatsend1,
	*pix_chatsend2,
	*pix_chatclear,
	*pix_account,
	*pix_icon_48,
	*pix_ft_back,
	*pix_ft_file,
	*pix_ft_folder;

QImage  *qim_online,
	*qim_offline,
	*qim_away,
	*qim_xa,
	*qim_dnd;

char *pixdat_ft_back, *pixdat_ft_file, *pixdat_ft_folder;
int pixlen_ft_back, pixlen_ft_file, pixlen_ft_folder;

Anim *anim_message, *anim_system, *anim_chat, *anim_headline;

LogWindow *debug_window = 0;
Jabber *debug_jabber = 0;
Options option;
PsiGlobal g;
bool useSound;


QString qstrlower(QString str)
{
	for(unsigned int n = 0; n < str.length(); ++n) {
		str.at(n) = str.at(n).lower();
	}

	return str;
}

int qstrcasecmp(const QString &str1, const QString &str2)
{
        if(str1.length() != str2.length())
                return 1;

        for(unsigned int n = 0; n < str1.length(); ++n) {
                if(str1.at(n).lower() != str2.at(n).lower())
                        return 1;
        }

        return 0;
}

int qstringlistmatch(QStringList &list, const QString &str)
{
	int n = 0;

        for(QStringList::Iterator i = list.begin(); i != list.end(); ++i, ++n) {
                if(*i == str)
                        return n;
        }

        return -1;
}

QString qstringlistlookup(QStringList &list, int x)
{
	int n = 0;
	QStringList::Iterator i = list.begin();
	for(;i != list.end() && n < x; ++i, ++n);
	if(n != x)
		return "";

	return *i;
}

QString CAP(const QString &str)
{
	return QString("%1: %2").arg(PROG_NAME).arg(str);
}

void pdb(int depth, const QString &str)
{
	//fprintf(stderr, "%s", str.latin1());

	if(debug_window)
		debug_window->append(depth, str);
}


QString eatptag(QString txt)
{
	if(txt.left(3) == "<p>")
                txt = txt.mid(3, txt.length() - 7);  // strlen("<p>") + strlen("</p>") = 7

	return txt;
}

QString plain2rich(const QString &plain)
{
	QString rich;
	int col = 0;

	for(int i = 0; i < (int)plain.length(); ++i) {
		if(plain[i] == '\n') {
			rich += "<br>\n";
			col = 0;
		}
		else if(plain[i] == '\t') {
			rich += QChar::nbsp;
			while(col % 4) {
				rich += QChar::nbsp;
				++col;
			}
		}
		else if(plain[i].isSpace()) {
			if(i > 0 && plain[i-1] == ' ')
				rich += QChar::nbsp;
			else
				rich += ' ';
		}
		else if(plain[i] == '<')
			rich += "&lt;";
		else if(plain[i] == '>')
			rich += "&gt;";
		else if(plain[i] == '\"')
			rich += "&quot;";
		else if(plain[i] == '\'')
			rich += "&apos;";
		else if(plain[i] == '&')
			rich += "&amp;";
		else
			rich += plain[i];
		++col;
	}

	return rich;
}

// clips plain text
QString clipStatus(const QString &str, int width, int height)
{
	QString out = "";
	int at = 0;
	int len = str.length();
	if(len == 0)
		return out;

	// only take the first "height" lines
	for(int n2 = 0; n2 < height; ++n2) {
		// only take the first "width" chars
		QString line;
		for(int n = 0; str.at(at) != '\n' && at < len; ++n, ++at)
			line += str.at(at);
		++at;
		if((int)line.length() > width) {
			line.truncate(width-3);
			line += "...";
		}
		out += line + '\n';

		if(at >= len)
			break;
	}

	return out;
}

QString expandEntities(const QString &in)
{
	QString out;
	int col = 0;

	for(int i = 0; i < (int)in.length(); ++i) {
		if(in[i] == '<')
			out += "&lt;";
		else if(in[i] == '>')
			out += "&gt;";
		else if(in[i] == '\"')
			out += "&quot;";
		else if(in[i] == '\'')
			out += "&apos;";
		else if(in[i] == '&')
			out += "&amp;";
		else
			out += in[i];
		++col;
	}

	return out;
}

QString resolveEntities(const QString &in)
{
	QString out;

	for(int i = 0; i < (int)in.length(); ++i) {
		if(in[i] == '&') {
			// find a semicolon
			++i;
			int n = in.find(';', i);
			if(n == -1)
				break;
			QString type = in.mid(i, (n-i));
			i = n; // should be n+1, but we'll let the loop increment do it

			if(type == "amp")
				out += '&';
			else if(type == "lt")
				out += '<';
			else if(type == "gt")
				out += '>';
			else if(type == "quot")
				out += '\"';
			else if(type == "apos")
				out += '\'';
		}
		else {
			out += in[i];
		}
	}

	return out;
}


static bool linkify_pmatch(const QString &str1, int at, const QString &str2)
{
	if(str2.length() > (str1.length()-at))
		return FALSE;

	for(int n = 0; n < (int)str2.length(); ++n) {
		if(str1.at(n+at).lower() != str2.at(n).lower())
			return FALSE;
	}

	return TRUE;
}

static bool linkify_isOneOf(const QChar &c, const QString &charlist)
{
	for(int i = 0; i < (int)charlist.length(); ++i) {
		if(c == charlist.at(i))
			return TRUE;
	}

	return FALSE;
}

// encodes a few dangerous html characters
static QString linkify_htmlsafe(const QString &in)
{
	QString out;

	for(unsigned int n = 0; n < in.length(); ++n) {
		if(linkify_isOneOf(in.at(n), "\"\'<>&")) {
			// hex encode
			QString hex;
			hex.sprintf("%%%02X", in.at(n).latin1());
			out.append(hex);
		}
		else {
			out.append(in.at(n));
		}
	}

	return out;
}

static bool linkify_okEmail(const QString &addy)
{
	// this makes sure that there is an '@' and a '.' after it, and that there is
	// at least one char for each of the three sections
	int n = addy.find('@');
	if(n == -1 || n == 0)
		return FALSE;
	int d = addy.find('.', n+1);
	if(d == -1 || d == 0)
		return FALSE;
	if((addy.length()-1) - d <= 0)
		return FALSE;

	return TRUE;
}

QString linkify(const QString &in)
{
	QString out = in;
	int x1, x2;
	bool isUrl, isEmail;
	QString linked, link, href;

	for(int n = 0; n < (int)out.length(); ++n) {
		isUrl = FALSE;
		isEmail = FALSE;
		x1 = n;

		if(linkify_pmatch(out, n, "http://")) {
			n += 7;
			isUrl = TRUE;
			href = "";
		}
		else if(linkify_pmatch(out, n, "ftp://")) {
			n += 6;
			isUrl = TRUE;
			href = "";
		}
		else if(linkify_pmatch(out, n, "news://")) {
			n += 7;
			isUrl = TRUE;
			href = "";
		}
		else if(linkify_pmatch(out, n, "www.")) {
			isUrl = TRUE;
			href = "http://";
		}
		else if(linkify_pmatch(out, n, "ftp.")) {
			isUrl = TRUE;
			href = "ftp://";
		}
		else if(linkify_pmatch(out, n, "@")) {
			isEmail = TRUE;
			href = "mailto:";
		}

		if(isUrl) {
			// make sure the previous char is not alphanumeric
			if(x1 > 0 && out.at(x1-1).isLetterOrNumber())
				continue;

			// find whitespace (or end)
			for(x2 = n; x2 < (int)out.length(); ++x2) {
				if(out.at(x2).isSpace() || out.at(x2) == '<')
					break;
			}
			--x2;

			// go backward hacking off unwanted punctuation
			for(; x2 >= n; --x2) {
				if(!linkify_isOneOf(out.at(x2), "!?,.()\""))
					break;
			}
			++x2;

			int len = x2-x1;
			link = out.mid(x1, len);
			href += resolveEntities(link);
			href = linkify_htmlsafe(href);
			//printf("link: [%s], href=[%s]\n", link.latin1(), href.latin1());
			linked = QString("<a href=\"%1\">").arg(href) + link + "</a>";
			out.replace(x1, len, linked);
			n = x1 + linked.length() - 1;
		}
		else if(isEmail) {
			// go backward till we find the beginning
			if(x1 == 0)
				continue;
			--x1;
			for(; x1 >= 0; --x1) {
				if(!linkify_isOneOf(out.at(x1), "_.-") && !out.at(x1).isLetterOrNumber())
					break;
			}
			++x1;

			// go forward till we find the end
			x2 = n + 1;
			for(; x2 < (int)out.length(); ++x2) {
				if(!linkify_isOneOf(out.at(x2), "_.-") && !out.at(x2).isLetterOrNumber())
					break;
			}

			int len = x2-x1;
			link = out.mid(x1, len);
			//link = resolveEntities(link);

			if(!linkify_okEmail(link)) {
				n = x1 + link.length();
				continue;
			}

			href += link;
			//printf("link: [%s], href=[%s]\n", link.latin1(), href.latin1());
			linked = QString("<a href=\"%1\">").arg(href) + link + "</a>";
			out.replace(x1, len, linked);
			n = x1 + linked.length() - 1;
		}
	}

	return out;
}

QString encodePassword(const QString &pass, const QString &key)
{
	QString result;
	unsigned int n1, n2;

	if(key.length() == 0)
		return pass;

	for(n1 = 0, n2 = 0; n1 < pass.length(); ++n1) {
		ushort x = pass.at(n1).unicode() ^ key.at(n2++).unicode();
		QString hex;
		hex.sprintf("%04x", x);
		result += hex;
		if(n2 >= key.length())
			n2 = 0;
	}
	return result;
}

QString decodePassword(const QString &pass, const QString &key)
{
	QString result;
	unsigned int n1, n2;

	if(key.length() == 0)
		return pass;

	for(n1 = 0, n2 = 0; n1 < pass.length(); n1 += 4) {
		ushort x = 0;
		if(n1 + 4 > pass.length())
			break;
		x += hexChar2int(pass.at(n1))*4096;
		x += hexChar2int(pass.at(n1+1))*256;
		x += hexChar2int(pass.at(n1+2))*16;
		x += hexChar2int(pass.at(n1+3));
		QChar c(x ^ key.at(n2++).unicode());
		result += c;
		if(n2 >= key.length())
			n2 = 0;
	}
	return result;
}

QPixmap & status2pix(int status)
{
        switch(status) {
                case STATUS_OFFLINE:    return *pix_offline;
                case STATUS_ONLINE:     return *pix_online;
                case STATUS_AWAY:       return *pix_away;
                case STATUS_XA:         return *pix_xa;
                case STATUS_DND:        return *pix_dnd;

                default:                return *pix_online;
        }
}

QImage & status2qim(int status)
{
        switch(status) {
                case STATUS_OFFLINE:    return *qim_offline;
                case STATUS_ONLINE:     return *qim_online;
                case STATUS_AWAY:       return *qim_away;
                case STATUS_XA:         return *qim_xa;
                case STATUS_DND:        return *qim_dnd;

                default:                return *qim_online;
        }
}

QString status2txt(int status)
{
        switch(status) {
                case STATUS_OFFLINE:    return QObject::tr("Offline");
                case STATUS_ONLINE:     return QObject::tr("Online");
                case STATUS_AWAY:       return QObject::tr("Away");
                case STATUS_XA:         return QObject::tr("Not Available");
                case STATUS_DND:        return QObject::tr("Do not Disturb");

                default:                return QObject::tr("Online");
        }
}


QString RecognizedTransportNames[] = {
        "ICQ",
        "AIM",
        "MSN",
        "Yahoo!",
};

const QString & transport2str(const QString &name)
{
        QString service(name);

        if(service.mid(0,4) == "icq.")
                return RecognizedTransportNames[0];
        else if(service.mid(0,4) == "aim.")
                return RecognizedTransportNames[1];
        else if(service.mid(0,4) == "msn.")
                return RecognizedTransportNames[2];
        else if(service.mid(0,6) == "yahoo.")
                return RecognizedTransportNames[3];

        return name;
}

QPixmap & transport2icon(const QString &jid, int status)
{
	int trans, pix;

	if(jid.mid(0,4) == "icq.")
		trans = 1;
	else if(jid.mid(0,4) == "aim.")
		trans = 2;
	else if(jid.mid(0,4) == "msn.")
		trans = 3;
	else if(jid.mid(0,6) == "yahoo.")
		trans = 4;
	else
		trans = 0;

	switch(status) {
		case STATUS_OFFLINE:    pix = 0; break;
		case STATUS_ONLINE:     pix = 1; break;
		case STATUS_AWAY:       pix = 2; break;
		case STATUS_XA:         pix = 3; break;
		case STATUS_DND:        pix = 4; break;

		default:                pix = 1; break;
	}

	if(!tic[trans].p[pix])
		trans = 0;

	return *tic[trans].p[pix];
}

QPixmap *loadImage(const QString &fname, const QStringList &_dirs)
{
	QImage image;
	QStringList dirs = _dirs;

	for(QStringList::Iterator it = dirs.begin(); it != dirs.end(); ++it) {
		QString str = (*it) + "/" + fname;
		if(image.load(str) == TRUE) {
			QPixmap *p = new QPixmap;
			// this is weird, but it gets the mask right on Windows
			p->convertFromImage(image.copy(0,0,image.width(),image.height()));
			return p;
		}
	}

        return 0;
}

char *loadImageData(const QString &fname, const QStringList &_dirs, int *size)
{
	char *p;
	QStringList dirs = _dirs;

	for(QStringList::Iterator it = dirs.begin(); it != dirs.end(); ++it) {
		QString str = (*it) + "/" + fname;
		QFileInfo info(str);
		if(info.exists()) {
			*size = info.size();
			QFile f(str);
			if(!f.open(IO_ReadOnly))
				continue;
			QByteArray a = f.readAll();
			p = new char[(*size)];
			memcpy(p, a.data(), *size);

			return p;
		}
	}

	return 0;
}

Anim *loadAnim(const QString &fname, const QStringList &dirs)
{
	QPixmap *p = loadImage(fname, dirs);
	if(!p)
		return 0;

	Anim *a = new Anim(*p);
	delete p;

	return a;
}

void bringToFront(QWidget *w)
{
        w->show();
        w->raise();
        w->setActiveWindow();
}

#ifdef Q_WS_WIN
void win_raiseNoFocus(QWidget *w)
{
	w->show();
	w->raise();
	//SetWindowPos(w->winId(),HWND_TOP,0,0,0,0, SWP_NOACTIVATE|SWP_NOMOVE|SWP_NOSIZE|SWP_SHOWWINDOW);
}
#endif

int hexChar2int(char c)
{
	if(c >= 'A' && c <= 'F')
		return c - 'A' + 10;
	else if(c >= 'a' && c <= 'f')
		return c - 'a' + 10;
	else if(c >= '0' && c <= '9')
		return c - '0';

	return 0;
}

char int2hexChar(int x)
{
	if(x < 10)
		return (char)x + '0';
	else
		return (char)x - 10 + 'a';
}

QString jidEncode(const QString &jid)
{
        QString jid2;

        for(unsigned int n = 0; n < jid.length(); ++n) {
                if(jid.at(n) == '@') {
                        jid2.append("_at_");
                }
                else if(jid.at(n) == '.') {
                        jid2.append('.');
                }
                else if(!jid.at(n).isLetterOrNumber()) {
                        // hex encode
                        QString hex;
                        hex.sprintf("%%%02X", jid.at(n).latin1());
                        jid2.append(hex);
                }
                else {
                        jid2.append(jid.at(n));
                }
        }

        return jid2;
}

QString jidDecode(const QString &jid)
{
	QString jid2;
	int n;

	for(n = 0; n < (int)jid.length(); ++n) {
		if(jid.at(n) == '%' && (jid.length() - n - 1) >= 2) {
			QString str = jid.mid(n+1,2);
			bool ok;
			char c = str.toInt(&ok, 16);
			if(!ok)
				continue;

			QChar a(c);
			jid2.append(a);
			n += 2;
		}
		else {
			jid2.append(jid.at(n));
		}
	}

	// search for the _at_ backwards, just in case
	for(n = (int)jid2.length(); n >= 3; --n) {
		if(jid2.mid(n, 4) == "_at_") {
			jid2.replace(n, 4, "@");
			break;
		}
	}

	return jid2;
}

QString jidnick(const QString &jid, const QString &nick)
{
	if(nick.isEmpty())
		return jid;
	else
		return nick;
}

QString logencode(QString str)
{
        str.replace(QRegExp("\\\\"), "\\\\");   // backslash to double-backslash
        str.replace(QRegExp("\\|"), "\\p");     // pipe to \p
        str.replace(QRegExp("\n"), "\\n");      // newline to \n
        return str;
}

QString logdecode(const QString &str)
{
        QString ret;

        for(unsigned int n = 0; n < str.length(); ++n) {
                if(str.at(n) == '\\') {
                        ++n;
                        if(n >= str.length())
                                break;

                        if(str.at(n) == 'n')
                                ret.append('\n');
                        if(str.at(n) == 'p')
                                ret.append('|');
                        if(str.at(n) == '\\')
                                ret.append('\\');
                }
                else {
                        ret.append(str.at(n));
                }
        }

        return ret;
}

void unfinishedSoftware()
{
        QMessageBox::information(0, CAP(QObject::tr("Warning")), QObject::tr(
         "<h3>*** Warning! ***</h3>\n"
         "This is unfinished software.  It has been released so that interested users "
         "can try it out and perhaps follow its development.  It is not guaranteed to be "
         "bugfree, although it is quite stable in my experience.<br><br>\n"
         "One day it will be finished.  Until then, don't count on every "
         "Jabber feature to work.  If you want to assist me in making this program "
         "better, feel free to email me bugreports/patches/wishes.<br><br>\n"
	 "Note: quite a few options/features appear \"grayed out\".  They "
	 "indicate planned features that are not in the program yet.<br><br>"
         "Now that this is out of the way, enjoy Psi!\n"
         ));
}

bool loadPsiIconSet(const QString name)
{
	QStringList dirs;
	QString subdir = "/iconsets/" + name;

	dirs += "." + subdir;
	dirs += g.pathHome + subdir;
	dirs += g.pathBase + subdir;
	dirs += g.pathBase + "/iconsets/stellar";


	// load animations - this needs to be implemented better
	anim_message = loadAnim("message.png", dirs);
	anim_system = loadAnim("system.png", dirs);
	anim_chat = loadAnim("chat.png", dirs);
	anim_headline = loadAnim("headline.png", dirs);
	if(!anim_message || !anim_system || !anim_chat || !anim_headline)
		return FALSE;

	// load pixmaps
	pix_online      = loadImage("online.png", dirs);
	pix_offline     = loadImage("offline.png", dirs);
	pix_away        = loadImage("away.png", dirs);
	pix_xa          = loadImage("xa.png", dirs);
	pix_dnd         = loadImage("dnd.png", dirs);
	pix_arrow[0]    = loadImage("groupclose.png", dirs);
	pix_arrow[1]    = loadImage("groupopen.png", dirs);
	pix_arrow[2]    = loadImage("groupempty.png", dirs);
	pix_add         = loadImage("add.png", dirs);
	pix_remove      = loadImage("remove.png", dirs);
	pix_send        = loadImage("send.png", dirs);
	pix_history     = loadImage("history.png", dirs);
	pix_info	= loadImage("info.png", dirs);
	pix_changeacc   = loadImage("changeacc.png", dirs);
	pix_url         = loadImage("url.png", dirs);

	if(!pix_online || !pix_offline || !pix_away || !pix_xa || !pix_dnd || !pix_arrow[0] ||
	   !pix_arrow[1] || !pix_arrow[2] || !pix_add || !pix_remove || !pix_send || !pix_history ||
	   !pix_info || !pix_changeacc || !pix_url)
	   	return FALSE;

	qim_online      = new QImage(pix_online->convertToImage());
	qim_offline     = new QImage(pix_offline->convertToImage());
	qim_away        = new QImage(pix_away->convertToImage());
	qim_xa          = new QImage(pix_xa->convertToImage());
	qim_dnd         = new QImage(pix_dnd->convertToImage());

	return TRUE;
}

void unloadPsiIconSet()
{
	return;
	delete anim_message;
	delete anim_system;
	delete anim_chat;
	delete anim_headline;

	delete pix_online;
	delete pix_offline;
	delete pix_away;
	delete pix_xa;
	delete pix_dnd;
	delete pix_arrow[0];
	delete pix_arrow[1];
	delete pix_arrow[2];
	delete pix_add;
	delete pix_remove;
	delete pix_send;
	delete pix_history;
	delete pix_changeacc;
	delete pix_url;

	delete qim_online;
	delete qim_offline;
	delete qim_away;
	delete qim_xa;
	delete qim_dnd;
}

void getPsiIconSets(QStringList &names, QStringList &descriptions)
{
	QStringList dirs;
	QString subdir = "/iconsets";

	dirs += "." + subdir;
	dirs += g.pathHome + subdir;
	dirs += g.pathBase + subdir;

	for(QStringList::Iterator it = dirs.begin(); it != dirs.end(); ++it) {
		if(!QFile::exists(*it))
			continue;
		QDir d(*it);
		QStringList entries = d.entryList();
		for(QStringList::Iterator it2 = entries.begin(); it2 != entries.end(); ++it2) {
			// make sure we don't have it
			bool found = FALSE;
			for(QStringList::Iterator it3 = names.begin(); it3 != names.end(); ++it3) {
				if(*it3 == *it2) {
					found = TRUE;
					break;
				}
			}
			if(found)
				continue;

			QFile f(*it + "/" + *it2 + "/desc");
			if(!f.open(IO_ReadOnly))
				continue;
			QCString line;
			line.resize(256);
			int n = f.readLine(line.data(), 255);
			if(n == -1)
				continue;
			line[n] = 0;
			if(line.at(line.length()-1) == '\n')
				line.at(line.length()-1) = 0;

			QString desc = QString::fromUtf8(line);

			names += *it2;
			descriptions += line;
		}
	}
}


void openURL(const QString &url)
{
	//fprintf(stderr, "openURL: [%s]\n", url.latin1());
	bool useCustom = TRUE;

#ifdef Q_WS_WIN
	if(option.browser == 0)
		useCustom = FALSE;
#endif
#ifdef Q_WS_X11
	if(option.browser == 0)
		useCustom = FALSE;
#endif
#ifdef Q_WS_MAC
	useCustom = FALSE;
#endif

	if(useCustom) {
		bool isMail = FALSE;
		QString s = url;
		if(url.left(7) == "mailto:") {
			s.remove(0, 7);
			isMail = TRUE;
		}

		QStringList args;

		if(isMail) {
			if(option.customMailer.isEmpty()) {
				QMessageBox::critical(0, CAP(QObject::tr("URL error")), QObject::tr("Unable to open the URL. You have not selected a mailer (see Options)."));
				return;
			}
			args += option.customMailer;
		}
		else {
			if(option.customBrowser.isEmpty()) {
				QMessageBox::critical(0, CAP(QObject::tr("URL error")), QObject::tr("Unable to open the URL. You have not selected a browser (see Options)."));
				return;
			}
			args += option.customBrowser;
		}

		args += s;
		QProcess cmd(args);
		if(!cmd.start()) {
			QMessageBox::critical(0, CAP(QObject::tr("URL error")), QObject::tr("Unable to open the URL. Ensure that your custom browser/mailer exists (see Options)."));
		}
	}
	else {
#ifdef Q_WS_WIN
		if ((unsigned int)::ShellExecuteA(NULL,NULL,url.latin1(),NULL,NULL,SW_SHOW) <= 32) {
			QMessageBox::critical(0, CAP(QObject::tr("URL error")), QObject::tr("Unable to open the URL. Ensure that you have a web browser installed."));
		}
#endif
#ifdef Q_WS_X11
		QStringList args;
		args += "kfmclient";
		args += "exec";
		args += url;
		QProcess cmd(args);
		if(!cmd.start()) {
			QMessageBox::critical(0, CAP(QObject::tr("URL error")), QObject::tr("Unable to open the URL. Ensure that you have KDE installed."));
		}
#endif
#ifdef Q_WS_MAC
    // Use Internet Config to hand the URL to the appropriate application, as
    // set by the user in the Internet Preferences pane.
    // NOTE: ICStart could be called once at Psi startup, saving the
    //       ICInstance in a global variable, as a minor optimization.
    //       ICStop should then be called at Psi shutdown if ICStart succeeded.
    ICInstance icInstance;
    OSType psiSignature = 'psi ';
    OSStatus error = ::ICStart( &icInstance, psiSignature );
    if ( error == noErr ) {
      ConstStr255Param hint( 0x0 );
      const char* data = url.latin1();
      long length = url.length();
      long start( 0 );
      long end( length );
      // Don't bother testing return value (error); launched application will report problems.
      ::ICLaunchURL( icInstance, hint, data, length, &start, &end );
      ICStop( icInstance );
    }
#endif
	}
}

Anim *messagetype2anim(int type)
{
	if(type == MESSAGE_SYS || type == MESSAGE_AUTHREQ || type == MESSAGE_ERROR)
		return anim_system;
	else if(type == MESSAGE_CHAT)
		return anim_chat;
	else if(type == MESSAGE_HEADLINE)
		return anim_headline;

	return anim_message;
}

QString messagetype2str(int type)
{
        if(type == MESSAGE_SYS)
                return QString(QObject::tr("System"));
        else if(type == MESSAGE_AUTHREQ)
                return QString(QObject::tr("Auth Request"));
        else if(type == MESSAGE_CHAT)
		return QString(QObject::tr("Chat"));
	else if(type == MESSAGE_ERROR)
		return QString(QObject::tr("Error"));
	else if(type == MESSAGE_HEADLINE)
		return QString(QObject::tr("Headline"));
	else
                return QString(QObject::tr("Normal"));
}

static bool sysinfo_done = FALSE;
static int timezone_offset = 0;
static QString timezone_str = "N/A";
static QString os_str = "Unknown";

#if defined(Q_WS_X11) || defined(Q_WS_MAC)
#include<time.h>
#include<stdlib.h>
#include<string.h>
#include<sys/utsname.h>
#endif

static void getSysInfo()
{
#if defined(Q_WS_X11) || defined(Q_WS_MAC)
	time_t x;
	time(&x);
	char str[256];
	char *fmt = "%z";
	strftime(str, 256, fmt, localtime(&x));
	if(strcmp(fmt, str)) {
		QString s = str;
		if(s.at(0) == '+')
			s.remove(0,1);
		s.truncate(s.length()-2);
		timezone_offset = s.toInt();
	}
	fmt = "%Z";
	strftime(str, 256, fmt, localtime(&x));
	if(strcmp(fmt, str))
		timezone_str = str;
	struct utsname u;
	uname(&u);
	os_str.sprintf("%s", u.sysname);
#elif defined(Q_WS_WIN)
	TIME_ZONE_INFORMATION i;
	GetTimeZoneInformation(&i);
	timezone_offset = (-i.Bias) / 60;
	timezone_str = "";
	for(int n = 0; n < 32; ++n) {
		uint w = i.StandardName[n];
		if(w == 0)
			break;
		timezone_str += QChar(w);
	}

	Qt::WindowsVersion v = QApplication::winVersion();
	if(v == Qt::WV_95)
		os_str = "Windows 95";
	else if(v == Qt::WV_98)
		os_str = "Windows 98";
	else if(v == Qt::WV_Me)
		os_str = "Windows ME";
	else if(v == Qt::WV_NT)
		os_str = "Windows NT 4.x";
	else if(v == Qt::WV_2000)
		os_str = "Windows 2000 (NT5)";
	else if(v == Qt::WV_XP)
		os_str = "Windows XP";
#endif
	sysinfo_done = TRUE;
}

QString getOSName()
{
	if(!sysinfo_done)
		getSysInfo();

	return os_str;
}

int getTZOffset()
{
	if(!sysinfo_done)
		getSysInfo();

	return timezone_offset;
}

QString getTZString()
{
	if(!sysinfo_done)
		getSysInfo();

	return timezone_str;
}


#ifdef Q_WS_X11
QString getResourcesDir()
{
	return "/usr/local/psi";
}

QString getHomeDir()
{
	QDir proghome(QDir::homeDirPath() + "/.psi");
	if(!proghome.exists()) {
		QDir home = QDir::home();
		home.mkdir(".psi");
		chmod(proghome.path().latin1(), 0700);
	}

	return proghome.path();
}
#endif

#ifdef Q_WS_WIN
QString getResourcesDir()
{
	return ".";
}

QString getHomeDir()
{
	QString base;

	// Windows 9x
	if(QDir::homeDirPath() == QDir::rootDirPath())
		base = ".";
	// Windows NT/2K/XP variant
	else
		base = QDir::homeDirPath();

	// no trailing slash
	if(base.at(base.length()-1) == '/')
		base.truncate(base.length()-1);

	QDir proghome(base + "/PsiData");
	if(!proghome.exists()) {
		QDir home(base);
		home.mkdir("PsiData");
	}

	return proghome.path();
}
#endif

#ifdef Q_WS_MAC
/******************************************************************************/
/* Get path to Resources directory as a string.                               */
/* Return an empty string if can't find it.                                   */
/******************************************************************************/
QString getResourcesDir()
{
  // System routine locates resource files. We "know" that Psi.icns is
  // in the Resources directory.
  QString resourcePath;
  CFBundleRef mainBundle = CFBundleGetMainBundle();
  CFStringRef resourceCFStringRef
      = CFStringCreateWithCString( NULL, "Psi.icns",
                                   kCFStringEncodingASCII );
  CFURLRef resourceURLRef = CFBundleCopyResourceURL( mainBundle,
                                                     resourceCFStringRef,
                                                     NULL,
                                                     NULL );
  if ( resourceURLRef ) {
    CFStringRef resourcePathStringRef =
    CFURLCopyFileSystemPath( resourceURLRef, kCFURLPOSIXPathStyle );
    const char* resourcePathCString =
      CFStringGetCStringPtr( resourcePathStringRef, kCFStringEncodingASCII );
    if ( resourcePathCString ) {
      resourcePath.setLatin1( resourcePathCString );
    } else { // CFStringGetCStringPtr failed; use fallback conversion
      CFIndex bufferLength = CFStringGetLength( resourcePathStringRef ) + 1;
      char* resourcePathCString = new char[ bufferLength ];
      Boolean conversionSuccess =
        CFStringGetCString( resourcePathStringRef,
                            resourcePathCString, bufferLength,
                            kCFStringEncodingASCII );
      if ( conversionSuccess ) {
        resourcePath = resourcePathCString;
      }
      delete [] resourcePathCString;  // I own this
    }
    CFRelease( resourcePathStringRef ); // I own this
  }
  // Remove the tail component of the path
  if ( ! resourcePath.isNull() ) {
    QFileInfo fileInfo( resourcePath );
    resourcePath = fileInfo.dirPath( true );
  }
  return resourcePath;
}

QString getHomeDir()
{
	QDir proghome(QDir::homeDirPath() + "/.psi");
	if(!proghome.exists()) {
		QDir home = QDir::home();
		home.mkdir(".psi");
	}

	return proghome.path();
}
#endif


QString getHistoryDir()
{
	return pathToProfile(activeProfile) + "/history";
}

bool fileCopy(const QString &src, const QString &dest)
{
	QFile in(src);
	QFile out(dest);

	if(!in.open(IO_ReadOnly))
		return FALSE;
	if(!out.open(IO_WriteOnly))
		return FALSE;

	char *dat = new char[16384];
	int n = 0;
	while(!in.atEnd()) {
		n = in.readBlock(dat, 16384);
		if(n == -1) {
			delete dat;
			return FALSE;
		}
		out.writeBlock(dat, n);
	}
	delete dat;

	out.close();
	in.close();

	return TRUE;
}

#ifdef Q_WS_MAC
static bool isDockTileBouncing = false;
static NMRec bounceRec;

void stopDockTileBounce()
{
	if(isDockTileBouncing) {
		NMRemove(&bounceRec);
		isDockTileBouncing = false;
	}
}

void bounceDockTile()
{
	stopDockTileBounce();
	bounceRec.qType = nmType;
	bounceRec.nmMark = 1;
	NMInstall(&bounceRec);
	isDockTileBouncing = true;
}
#endif

