/****************************************************************************
** jabsess.cpp - keeps track of Jabber accounts/sessions
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"jabsess.h"
#include"accountdlg.h"
#include<qinputdialog.h>


/****************************************************************************
  JabSession
****************************************************************************/
JabSession::JabSession(JabSessionManager *par)
:QObject(par)
{
	v_serv.setOfflineEnabled(TRUE);

	// remap the Jabber signals
	connect(&v_serv, SIGNAL(connected()), SIGNAL(connected()));
	connect(&v_serv, SIGNAL(disconnected()), SLOT(slotDisconnected()));
	connect(&v_serv, SIGNAL(statusUpdate(JabUpdate *)), SIGNAL(statusUpdate(JabUpdate *)));
	connect(&v_serv, SIGNAL(error(JabError *)), SIGNAL(error(JabError *)));
	connect(&v_serv, SIGNAL(messageReceived(const JabMessage &)), SIGNAL(messageReceived(const JabMessage &)));
	connect(&v_serv, SIGNAL(resourceAvailable(const Jid &, const JabResource &)), SIGNAL(resourceAvailable(const Jid &, const JabResource &)));
	connect(&v_serv, SIGNAL(resourceUnavailable(const Jid &)), SIGNAL(resourceUnavailable(const Jid &)));
	connect(&v_serv, SIGNAL(contactChanged(JabRosterEntry *)), SIGNAL(contactChanged(JabRosterEntry *)));
	connect(&v_serv, SIGNAL(contactNew(JabRosterEntry *)), SIGNAL(contactNew(JabRosterEntry *)));
	connect(&v_serv, SIGNAL(contactRemove(JabRosterEntry *)), SIGNAL(contactRemove(JabRosterEntry *)));
	connect(&v_serv, SIGNAL(authRequest(const Jid &)), SIGNAL(authRequest(const Jid &)));
	connect(&v_serv, SIGNAL(authGrant(const Jid &)), SIGNAL(authGrant(const Jid &)));
	connect(&v_serv, SIGNAL(authRemove(const Jid &)), SIGNAL(authRemove(const Jid &)));

	// init
	v_cp = 0;

	// setup
	usingAutoStatus = FALSE;
	isDisconnecting = FALSE;
	onEventOnlineOk = FALSE;

	loginAs.type = STATUS_ONLINE;
	loginAs.str = "";
	localStatus = STATUS_OFFLINE;
}

JabSession::~JabSession()
{
}

void JabSession::setAccount(const UserAccount &_acc)
{
	v_acc = _acc;

	serv()->setCurrentRoster(&v_acc.roster);
	serv()->setOLR(v_acc.olr_string);
	v_acc.olr_string = "";

	update(FALSE);
}

void JabSession::setContactProfile(ContactProfile *_cp)
{
	v_cp = _cp;
}

void JabSession::update(bool newUser)
{
        // different user?
        if(newUser) {
                // nuke the offline contact list (this will clear cvlist thru signal-chain)
                serv()->reset();

                // there might be some "not in list" contacts left, so this gets them too
                cp()->clear();
        }

	if(v_acc.opt_keepAlive)
		v_serv.setNoop(180000);  // prevent NAT timeouts every 3 minutes
	else
		v_serv.setNoop(0);

	v_serv.setSSLEnabled(v_acc.opt_ssl);

	JabSessionManager *jsm = (JabSessionManager *)parent();
	jsm->childUpdate(this);
}

void JabSession::conn()
{
	if(!serv()->isActive()) {
		isDisconnecting = FALSE;
		onEventOnlineOk = FALSE;

		serv()->setHost(acc()->host, acc()->port);
		serv()->setAccount(acc()->user, acc()->pass, acc()->resource);
		serv()->login(loginAs.type, loginAs.str, acc()->priority, acc()->opt_plain);
	}
}

void JabSession::disc(bool fast)
{
	// disconnect
	if(serv()->isActive()) {
		isDisconnecting = TRUE;
		serv()->disc(fast);
		//mainwin->setUsingSSL(FALSE);
	}
}

void JabSession::setStatus(const StatusInfo &info)
{
	usingAutoStatus = FALSE;

	loginAs.type = info.type;
	loginAs.str = info.str;

	// if the Jabber (serv) class is not busy, then attempt to login
	// if it is busy then change the status
	if(!serv()->isActive()) {
		if(acc()->user.isEmpty() || acc()->host.isEmpty() || acc()->resource.isEmpty()) {
			QMessageBox::information(0, CAP(tr("Error")), tr("Unable to login.  Ensure your account information is filled out."));
			JabSessionManager *jsm = (JabSessionManager *)parent();
			jsm->modify(this);
			return;
		}
		if(!acc()->opt_pass) {
			bool ok = FALSE;
			QString text = QInputDialog::getText(
				tr("Need Password"),
				QString(tr("Please enter the password for <b>%1</b>")).arg(acc()->jid()),
				QLineEdit::Password, QString::null, &ok, 0);
			if(ok && !text.isEmpty())
				acc()->pass = text;
			else
				return;
		}

		conn();
	}
	// change status
	else {
		serv()->setPresence(info.type, info.str, acc()->priority);
	}
}

void JabSession::slotDisconnected()
{
	isDisconnecting = TRUE;

	disconnected();
}

void JabSession::secondsIdle(int x)
{
	static int lastIdle = 0;
	static int lastStatus = STATUS_OFFLINE;

	if(serv()->isActive()) {
		// no longer idle?
		if(lastIdle > x) {
			if(localStatus != STATUS_ONLINE && usingAutoStatus) {
				serv()->setPresence(STATUS_ONLINE, "", acc()->priority);
				lastStatus = STATUS_ONLINE;
				usingAutoStatus = FALSE;
			}
		}
		else if( !(localStatus != STATUS_ONLINE && !usingAutoStatus) ) {
			int minutes = x / 60;

			if(option.asOffline > 0 && minutes >= option.asOffline) {
				isDisconnecting = TRUE;
				lastStatus = STATUS_OFFLINE;
				usingAutoStatus = FALSE;
				disc();
			}
			else if(option.asXa > 0 && minutes >= option.asXa) {
				if(localStatus != STATUS_XA && lastStatus != STATUS_XA) {
					QString str = tr("Auto Status (idle)");
					serv()->setPresence(STATUS_XA, str, acc()->priority);
					lastStatus = STATUS_XA;
					usingAutoStatus = TRUE;
				}
			}
			else if(option.asAway > 0 && minutes >= option.asAway) {
				if(localStatus != STATUS_AWAY && lastStatus != STATUS_AWAY) {
					QString str = tr("Auto Status (idle)");
					serv()->setPresence(STATUS_AWAY, str, acc()->priority);
					lastStatus = STATUS_AWAY;
					usingAutoStatus = TRUE;
				}
			}
		}
	}

	lastIdle = x;
}


/****************************************************************************
  JabSessionManager
****************************************************************************/
JabSessionManager::JabSessionManager(QObject *par)
:QObject(par)
{
	list.setAutoDelete(TRUE);
}

JabSessionManager::~JabSessionManager()
{
}

void JabSessionManager::add(JabSession *s)
{
	list.append(s);

	connect(s, SIGNAL(connected()), SLOT(slotConnected()));
	connect(s, SIGNAL(disconnected()), SLOT(slotDisconnected()));
	connect(s, SIGNAL(statusUpdate(JabUpdate *)), SLOT(slotStatusUpdate(JabUpdate *)));
	connect(s, SIGNAL(error(JabError *)), SLOT(slotError(JabError *)));
	connect(s, SIGNAL(messageReceived(const JabMessage &)), SLOT(slotMessageReceived(const JabMessage &)));
	connect(s, SIGNAL(resourceAvailable(const Jid &, const JabResource &)), SLOT(slotResourceAvailable(const Jid &, const JabResource &)));
	connect(s, SIGNAL(resourceUnavailable(const Jid &)), SLOT(slotResourceUnavailable(const Jid &)));
	connect(s, SIGNAL(contactChanged(JabRosterEntry *)), SLOT(slotContactChanged(JabRosterEntry *)));
	connect(s, SIGNAL(contactNew(JabRosterEntry *)), SLOT(slotContactNew(JabRosterEntry *)));
	connect(s, SIGNAL(contactRemove(JabRosterEntry *)), SLOT(slotContactRemove(JabRosterEntry *)));
	connect(s, SIGNAL(authRequest(const Jid &)), SLOT(slotAuthRequest(const Jid &)));
	connect(s, SIGNAL(authGrant(const Jid &)), SLOT(slotAuthGrant(const Jid &)));
	connect(s, SIGNAL(authRemove(const Jid &)), SLOT(slotAuthRemove(const Jid &)));
}

void JabSessionManager::clear()
{
	list.clear();
}

void JabSessionManager::slotConnected()
{
	JabSession *s = (JabSession *)sender();
	jab_connected(s);
}

void JabSessionManager::slotDisconnected()
{
	JabSession *s = (JabSession *)sender();
	jab_disconnected(s);
}

void JabSessionManager::slotStatusUpdate(JabUpdate *x)
{
	JabSession *s = (JabSession *)sender();
	jab_statusUpdate(s, x);

	sessionUpdate(s);
}

void JabSessionManager::slotError(JabError *x)
{
	JabSession *s = (JabSession *)sender();
	jab_error(s, x);
}

void JabSessionManager::slotMessageReceived(const JabMessage &msg)
{
	JabSession *s = (JabSession *)sender();
	jab_messageReceived(s, msg);
}

void JabSessionManager::slotResourceAvailable(const Jid &j, const JabResource &r)
{
	JabSession *s = (JabSession *)sender();
	jab_resourceAvailable(s, j, r);
}

void JabSessionManager::slotResourceUnavailable(const Jid &j)
{
	JabSession *s = (JabSession *)sender();
	jab_resourceUnavailable(s, j);
}

void JabSessionManager::slotContactChanged(JabRosterEntry *e)
{
	JabSession *s = (JabSession *)sender();
	jab_contactChanged(s, e);
}

void JabSessionManager::slotContactNew(JabRosterEntry *e)
{
	JabSession *s = (JabSession *)sender();
	jab_contactNew(s, e);
}

void JabSessionManager::slotContactRemove(JabRosterEntry *e)
{
	JabSession *s = (JabSession *)sender();
	jab_contactRemove(s, e);
}

void JabSessionManager::slotAuthRequest(const Jid &j)
{
	JabSession *s = (JabSession *)sender();
	jab_authRequest(s, j);
}

void JabSessionManager::slotAuthGrant(const Jid &j)
{
	JabSession *s = (JabSession *)sender();
	jab_authGrant(s, j);
}

void JabSessionManager::slotAuthRemove(const Jid &j)
{
	JabSession *s = (JabSession *)sender();
	jab_authRemove(s, j);
}

JabSession *JabSessionManager::find(const QString &name)
{
	QString val = name.lower();
	QPtrListIterator<JabSession> it(list);
	for(JabSession *s; (s = it.current()); ++it) {
		if(s->acc()->name.lower() == val)
			return s;
	}
	return 0;
}

void JabSessionManager::manage()
{
	AccountManageDlg *w = AccountManageDlg::find();
	if(w)
		bringToFront(w);
	else {
		w = new AccountManageDlg(this);
		connect(w, SIGNAL(signalModify(const QString &)), SLOT(modify(const QString &)));
		w->show();
	}
}

void JabSessionManager::modify(const QString &accname)
{
	JabSession *s = find(accname);
	if(!s)
		return;
	modify(s);
}

void JabSessionManager::modify(JabSession *s)
{
	AccountModifyDlg *w = AccountModifyDlg::find(s->name());
	if(w)
		bringToFront(w);
	else {
		w = new AccountModifyDlg(s, s->serv()->isSSLSupported(), 0);
		w->show();
	}
}

void JabSessionManager::childUpdate(JabSession *s)
{
	// jabcon should respond to this and save settings
	accountSettingsChanged();

	// notify the world that this session was updated
	sessionUpdate(s);
}
