/****************************************************************************
** jabtasks.cpp - basic tasks for use with the JabIO task system
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"jabtasks.h"
#include"common.h"


/****************************************************************************
  JT_Auth
****************************************************************************/
JT_Auth::JT_Auth(JabTask *parent, const QString &user, const QString &pass, const QString &sid, const QString &resource, bool plain)
:JabTask(parent)
{
	iq = createIQ("set", "", id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:auth");
	iq.appendChild(query);
	query.appendChild(textTag("username", user));
	if(plain)
		query.appendChild(textTag("password", pass));
	else {
		QString hash = SHA1::digest(sid + pass);
		query.appendChild(textTag("digest", hash));
	}
	query.appendChild(textTag("resource", resource));
}

void JT_Auth::go()
{
	send(iq);
}

bool JT_Auth::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		// login success
		setSuccess(TRUE);
	}
	else {
		// login fail
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}


/****************************************************************************
  JT_Register
****************************************************************************/
JT_Register::JT_Register(JabTask *parent)
:JabTask(parent)
{
}

void JT_Register::reg(const QString &user, const QString &pass)
{
	iq = createIQ("set", "", id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:register");
	iq.appendChild(query);
	query.appendChild(textTag("username", user));
	query.appendChild(textTag("password", pass));
}

void JT_Register::go()
{
	send(iq);
}

bool JT_Register::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		setSuccess(TRUE);
	}
	else {
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}


/****************************************************************************
  JT_Roster
****************************************************************************/
JT_Roster::JT_Roster(JabTask *parent)
:JabTask(parent)
{
	type = -1;
}

void JT_Roster::get()
{
	type = 0;

	iq = createIQ("get", "", id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:roster");
	iq.appendChild(query);
}

void JT_Roster::set(const QString &_jid, const QString &_nick, const QStringList &_groups)
{
	type = 1;
	jid = _jid;
	nick = _nick;
	groups = _groups;

	iq = createIQ("set", "", id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:roster");
	iq.appendChild(query);

	QDomElement item = doc()->createElement("item");
	item.setAttribute("jid", jid);
	if(!nick.isEmpty())
		item.setAttribute("name", nick);
	for(QStringList::Iterator it = groups.begin(); it != groups.end(); ++it)
		item.appendChild(textTag("group", *it));
	query.appendChild(item);
}

void JT_Roster::remove(const QString &jid)
{
	type = 2;

	iq = createIQ("set", "", id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:roster");
	iq.appendChild(query);

	QDomElement item = doc()->createElement("item");
	item.setAttribute("jid", jid);
	item.setAttribute("subscription", "remove");
	query.appendChild(item);
}

void JT_Roster::go()
{
	send(iq);
}

QString JT_Roster::toString()
{
	if(type != 1)
		return "";

	QStringList list;
	list += "JT_Roster";
	list += jid;
	list += nick;
	list += lineEncodeList(groups);
	return lineEncodeList(list);
}

bool JT_Roster::fromString(const QString &str)
{
	QStringList args = lineDecodeList(str);

	if(args.count() != 4)
		return FALSE;

	QStringList::Iterator i = args.begin();
	QString arg0;
	arg0 = *(i++);

	if(arg0 == "JT_Roster") {
		QString jid = *(i++);
		QString nick = *(i++);
		QStringList groups = lineDecodeList(*(i++));
		set(jid, nick, groups);
		return TRUE;
	}

	return FALSE;
}

bool JT_Roster::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(type == 0) {
		if(x.attribute("type") == "result") {
			QDomElement q = queryTag(x);
			roster = xmlReadRoster(q);
			setSuccess(TRUE);
		}
		else {
			setError(getErrorString(x));
			setSuccess(FALSE);
		}

		return TRUE;
	}
	else if(type == 1) {
		if(x.attribute("type") == "result")
			setSuccess(TRUE);
		else {
			setError(getErrorString(x));
			setSuccess(FALSE);
		}

		return TRUE;
	}
	else if(type == 2) {
                pdb(DEBUG_JABBER, "[Remove successful]\n");

		setSuccess(TRUE);
		return TRUE;
	}

	return FALSE;
}


/****************************************************************************
  JT_Presence
****************************************************************************/
JT_Presence::JT_Presence(JabTask *parent)
:JabTask(parent)
{
}

void JT_Presence::pres(int _status, const QString &_statusString, int _priority)
{
	type = 0;

	status = _status;
	statusString = _statusString;
	priority = _priority;

	tag = doc()->createElement("presence");
	if(status == STATUS_OFFLINE) {
		tag.setAttribute("type", "unavailable");
		if(!statusString.isEmpty())
			tag.appendChild(textTag("status", statusString));
	}
	else {
		if(status != STATUS_ONLINE)
			tag.appendChild(textTag("show", status2jabprestxt(status)));
		if(!statusString.isEmpty())
			tag.appendChild(textTag("status", statusString));
		tag.appendChild(textTag("priority", QString("%1").arg(priority)));
	}
}

void JT_Presence::sub(const Jid &to, int subType)
{
	type = 1;

	tag = doc()->createElement("presence");
	tag.setAttribute("to", to.full());
	if(subType == JABSUB_SUBSCRIBE)
		tag.setAttribute("type", "subscribe");
	else if(subType == JABSUB_SUBSCRIBED)
		tag.setAttribute("type", "subscribed");
	else if(subType == JABSUB_UNSUBSCRIBE)
		tag.setAttribute("type", "unsubscribe");
	else if(subType == JABSUB_UNSUBSCRIBED)
		tag.setAttribute("type", "unsubscribed");
}

void JT_Presence::go()
{
	send(tag);
	if(type == 0 && status != STATUS_OFFLINE)
		receivePresence(status, statusString);
	setSuccess(TRUE);
}


/****************************************************************************
  JT_Login
****************************************************************************/
JT_Login::JT_Login(JabTask *parent)
:JabTask(parent)
{
	reg = 0;
	auth = 0;
	rost = 0;
	pres = 0;
	type = -1;
}

void JT_Login::login(const QString &_user, const QString &_pass, const QString &_sid, const QString &_resource, int _status, const QString &_statusString, int _priority, bool plain)
{
	user = _user;
	pass = _pass;
	sid = _sid;
	resource = _resource;
	status = _status;
	statusString = _statusString;
	priority = _priority;
	usePlain = plain;
	type = 0;
}

void JT_Login::create(const QString &_user, const QString &_pass, const QString &_sid, const QString &_resource, int _status, const QString &_statusString, int _priority, bool plain)
{
	user = _user;
	pass = _pass;
	sid = _sid;
	resource = _resource;
	status = _status;
	statusString = _statusString;
	priority = _priority;
	usePlain = plain;
	type = 1;
}

void JT_Login::go()
{
	if(type == 0) {
		auth = new JT_Auth(this, user, pass, sid, resource, usePlain);
		stepChanged();
		auth->go();
	}
	else if(type == 1) {
		reg = new JT_Register(this);
		reg->reg(user, pass);
		stepChanged();
		reg->go();
	}
}

void JT_Login::done(JabTask *p)
{
	if(reg && p->id() == reg->id()) {
		if(reg->success()) {
			//printf("JT_Login: Created.\n");
			delete reg;
			reg = 0;

			auth = new JT_Auth(this, user, pass, sid, resource, usePlain);
			stepChanged();
			auth->go();
		}
		else {
			setError(reg->errorString());
			setSuccess(FALSE);
			return;
		}
	}
	else if(auth && p->id() == auth->id()) {
		if(auth->success()) {
			//printf("JT_Login: Authorized.\n");
			receiveAuth();
			delete auth;
			auth = 0;

			rost = new JT_Roster(this);
			rost->get();
			stepChanged();
			rost->go();
		}
		else {
			setError(auth->errorString());
			setSuccess(FALSE);
			return;
		}
	}
	else if(rost && p->id() == rost->id()) {
		//printf("JT_Login: Got roster.\n");
		receiveRoster(rost->roster);
		delete rost;
		rost = 0;

		pres = new JT_Presence(this);
		pres->pres(status, statusString, priority);
		pres->go();
	}
	else if(pres && p->id() == pres->id()) {
		//printf("JT_Login: Sent presence.\n");
		receivePresence(pres->status, pres->statusString);
		delete pres;
		pres = 0;

		setSuccess(TRUE);
	}
}


/****************************************************************************
  JT_PushPresence
****************************************************************************/
JT_PushPresence::JT_PushPresence(JabTask *parent)
:JabTask(parent)
{
}

bool JT_PushPresence::take(const QDomElement &e)
{
	if(e.tagName() != "presence")
		return FALSE;

	Info p;
	p.jid = e.attribute("from");
	p.status = STATUS_ONLINE;
	p.priority = 0;
	p.statusString = "";
	p.songTitle = ""; // Gabber music
	p.timeStamp = QDateTime::currentDateTime();

	if(e.hasAttribute("type")) {
		QString type = e.attribute("type");
		if(type == "unavailable") {
			p.status = STATUS_OFFLINE;
		}
		else {
			int subType = -1;
			if(type == "subscribe") {
				subType = JABSUB_SUBSCRIBE;
			}
			else if(type == "subscribed") {
				subType = JABSUB_SUBSCRIBED;
			}
			else if(type == "unsubscribe") {
				subType = JABSUB_UNSUBSCRIBE;
			}
			else if(type == "unsubscribed") {
				subType = JABSUB_UNSUBSCRIBED;
			}
			if(subType == -1)
				return TRUE;

			subscription(p.jid, subType);
			return TRUE;
		}
	}

	QDomElement tag;
	bool found;

	tag = findSubTag(e, "status", &found);
	if(found)
		p.statusString = tagContent(tag);
	tag = findSubTag(e, "show", &found);
	if(found)
		p.status = jabprestxt2status(tagContent(tag));
	tag = findSubTag(e, "priority", &found);
	if(found)
		p.priority = tagContent(tag).toInt();

	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;

		if(i.tagName() == "x" && i.attribute("xmlns") == "jabber:x:delay") {
			if(i.hasAttribute("stamp")) {
				if(stamp2TS(i.attribute("stamp"), &p.timeStamp))
					p.timeStamp = p.timeStamp.addSecs(getTZOffset() * 3600);
			}
		}

		if(i.tagName() == "x" && i.attribute("xmlns") == "gabber:x:music:info") {
			QDomElement t;
			bool found;
			QString title, state;

			t = findSubTag(i, "title", &found);
			if(found)
				title = tagContent(t);
			t = findSubTag(i, "state", &found);
			if(found)
				state = tagContent(t);

			if(!title.isEmpty() && state == "playing")
				p.songTitle = tagContent(t);
		}
	}

	presence(p);

	return TRUE;
}


/****************************************************************************
  JT_PushRoster
****************************************************************************/
JT_PushRoster::JT_PushRoster(JabTask *parent)
:JabTask(parent)
{
}

bool JT_PushRoster::take(const QDomElement &e)
{
	// must be an iq tag
	if(e.tagName() != "iq")
		return FALSE;

	// let's not steal non-push responses
	if(e.hasAttribute("id"))
		return FALSE;

	if(queryNS(e) != "jabber:iq:roster")
		return FALSE;

	QDomElement q = queryTag(e);
	JabRoster r = xmlReadRoster(q, TRUE);
	roster(r);

	return TRUE;
}


/****************************************************************************
  JT_Message
****************************************************************************/
JT_Message::JT_Message(JabTask *parent, const JabMessage &_jmsg)
:JabTask(parent)
{
	jmsg = _jmsg;
}

void JT_Message::go()
{
	QDomElement message = doc()->createElement("message");
	message.setAttribute("to", jmsg.to.full());
	if(jmsg.type == JABMESSAGE_CHAT)
		message.setAttribute("type", "chat");
	if(!jmsg.subject.isEmpty())
		message.appendChild(textTag("subject", jmsg.subject));
	message.appendChild(textTag("body", jmsg.body));
	if(!jmsg.url.isEmpty()) {
		QDomElement x = doc()->createElement("x");
		x.setAttribute("xmlns", "jabber:x:oob");
		x.appendChild(textTag("url", jmsg.url));
		x.appendChild(textTag("desc", jmsg.url_desc));
		message.appendChild(x);
	}
	send(message);
	setSuccess(TRUE);
}


/****************************************************************************
  JT_PushMessage
****************************************************************************/
JT_PushMessage::JT_PushMessage(JabTask *parent)
:JabTask(parent)
{
}

bool JT_PushMessage::take(const QDomElement &e)
{
	if(e.tagName() != "message")
		return FALSE;

	QDomElement tag;
	bool found;
	JabMessage msg;
	msg.late = FALSE;
	msg.timeStamp = QDateTime::currentDateTime();
	QString errorString;

	QString type = e.attribute("type");
	QString from = e.attribute("from");
	msg.from = from;
	msg.type = JABMESSAGE_NORM;

	if(type == "chat")
		msg.type = JABMESSAGE_CHAT;
	else if(type == "error")
		msg.type = JABMESSAGE_ERROR;
	else if(type == "headline")
		msg.type = JABMESSAGE_HEADLINE;

	tag = findSubTag(e, "body", &found);
	if(found)
		msg.body = tagContent(tag);

	// no body tag?  no message!
	if(!found && msg.type != JABMESSAGE_ERROR)
		return FALSE;

	tag = findSubTag(e, "subject", &found);
	if(found)
		msg.subject = tagContent(tag);
	tag = findSubTag(e, "error", &found);
	if(found)
		errorString = tagContent(tag);
	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;

		if(i.tagName() == "x" && i.attribute("xmlns") == "jabber:x:delay") {
			if(i.hasAttribute("stamp")) {
				if(stamp2TS(i.attribute("stamp"), &msg.timeStamp)) {
					msg.timeStamp = msg.timeStamp.addSecs(getTZOffset() * 3600);
					msg.late = TRUE;
				}
			}
		}

		if(i.tagName() == "x" && i.attribute("xmlns") == "jabber:x:oob") {
			QDomElement tag;
			bool found;

			tag = findSubTag(i, "url", &found);
			if(found)
				msg.url = tagContent(tag);
			tag = findSubTag(i, "desc", &found);
			if(found)
				msg.url_desc = tagContent(tag);
		}
	}

	msg.originLocal = FALSE;
	if(msg.type == JABMESSAGE_ERROR)
		msg.body = QString("%1\n------\n%2").arg(errorString).arg(msg.body);

	message(msg);

	return TRUE;
}


/****************************************************************************
  JT_Queue
****************************************************************************/
JT_Queue::JT_Queue(JabTask *parent)
:JabTask(parent)
{
	active = 0;
	going = FALSE;
}

void JT_Queue::addTask(JabTask *j)
{
	list.append(j);

	//printf("JT_Queue: count=%d\n", count());
	countChanged();

	if(!active)
		doNext();
}

void JT_Queue::done(JabTask *j)
{
	itemDone(j);
	QTimer::singleShot(0, this, SLOT(afterDone()));
}

void JT_Queue::go()
{
	going = TRUE;
	if(!active)
		doNext();
}

void JT_Queue::stop()
{
	going = FALSE;

	// check the active
	if(active) {
		if(active->isDone()) {
			list.remove(active);
			delete active;
		}
		else
			active->stop();
		active = 0;
	}
}

void JT_Queue::afterDone()
{
	if(active) {
		list.remove(active);

		// should be safe to delete now
		delete active;
		active = 0;
	}
	countChanged();
	doNext();
}

void JT_Queue::doNext()
{
	if(going) {
		//printf("JT_Queue: count=%d\n", count());
		active = list.getFirst();
		if(active)
			active->go();
	}
}

int JT_Queue::count()
{
	return list.count();
}

QString JT_Queue::toString()
{
	QStringList all;
	QPtrListIterator<JabTask> it(list);
	for(JabTask *j; (j = it.current()); ++it) {
		QString s = j->toString();
		if(!s.isEmpty())
			all.append(s);
	}
	return lineEncodeList(all);
}

bool JT_Queue::fromString(const QString &str)
{
	QStringList all = lineDecodeList(str);

	for(QStringList::Iterator it = all.begin(); it != all.end(); ++it) {
		// only type that is supported is roster
		JT_Roster *r = new JT_Roster(this);
		//printf("trying to add: [%s]\n", (*it).latin1());
		if(!r->fromString(*it)) {
			delete r;
			continue;
		}
		addTask(r);
		//printf("success\n");
	}
	//printf("JT_Queue: count=%d\n", count());
	doNext();
	return TRUE;
}


/****************************************************************************
  JT_GetServices
****************************************************************************/
JT_GetServices::JT_GetServices(JabTask *parent, const QString &host)
:JabTask(parent)
{
	iq = createIQ("get", host, id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:agents");
	iq.appendChild(query);
}

void JT_GetServices::go()
{
	send(iq);
}

bool JT_GetServices::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		QDomElement q = queryTag(x);

		// agents
		for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
			QDomElement i = n.toElement();
			if(i.isNull())
				continue;

			if(i.tagName() == "agent") {
				JabRosterEntry entry;

				entry.jid = i.attribute("jid");

				QDomElement tag;
				bool found;

				tag = findSubTag(i, "name", &found);
				if(found)
					entry.nick = tagContent(tag);
				tag = findSubTag(i, "register", &found);
				if(found)
					entry.isRegisterable = TRUE;
				tag = findSubTag(i, "search", &found);
				if(found)
					entry.isSearchable = TRUE;
				tag = findSubTag(i, "groupchat", &found);
				if(found)
					entry.isGCCapable = TRUE;
				tag = findSubTag(i, "transport", &found);
				if(found)
					entry.isTransport = TRUE;

				JabRosterEntry *tmp = new JabRosterEntry(entry);
				services.add(tmp);
			}
		}

		setSuccess(TRUE);
	}
	else {
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}


/****************************************************************************
  JT_VCard
****************************************************************************/
JT_VCard::JT_VCard(JabTask *parent)
:JabTask(parent)
{
	type = -1;
}

void JT_VCard::get(const QString &_jid)
{
	type = 0;
	jid = _jid;
	iq = createIQ("get", jid, id());
	QDomElement v = doc()->createElement("vCard");
	v.setAttribute("xmlns", "vcard-temp");
	v.setAttribute("version", "3.0");
	v.setAttribute("prodid", "-//HandGen//NONSGML vGen v1.0//EN");
	iq.appendChild(v);
}

void JT_VCard::set(VCard &card)
{
	type = 1;
	iq = createIQ("set", "", id());
	iq.appendChild(card.toXml());
}

void JT_VCard::go()
{
	send(iq);
}

bool JT_VCard::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		if(type == 0) {
			QDomElement q = x.firstChild().toElement();
			if(q.tagName() == "vcard") {
				if(vcard.fromXml(q)) {
					setSuccess(TRUE);
					return TRUE;
				}
			}

			setSuccess(FALSE);
			return TRUE;
		}
		else {
			setSuccess(TRUE);
			return TRUE;
		}
	}
	else {
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}


/****************************************************************************
  JT_RegForm
****************************************************************************/
JT_RegForm::JT_RegForm(JabTask *parent)
:JabTask(parent)
{
	type = -1;
}

void JT_RegForm::get(const QString &_jid)
{
	type = 0;
	jid = _jid;
	iq = createIQ("get", jid, id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:register");
	iq.appendChild(query);
}

void JT_RegForm::set(const JabForm &form)
{
	type = 1;
	jid = form.jid.full();
	iq = createIQ("set", form.jid.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:register");
	iq.appendChild(query);

	// key?
	if(!form.key.isEmpty())
		query.appendChild(textTag("key", form.key));
	// fields
	QPtrListIterator<JabFormField> it(form);
	for(JabFormField *f; (f = it.current()); ++it)
		query.appendChild(textTag(f->realName(), f->value()));
}

void JT_RegForm::go()
{
	send(iq);
}

bool JT_RegForm::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		if(type == 0) {
			form.jid = x.attribute("from");
			QDomElement q = queryTag(x);
			for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
				QDomElement i = n.toElement();
				if(i.isNull())
					continue;

				if(i.tagName() == "instructions")
					form.instructions = tagContent(i);
				else if(i.tagName() == "key")
					form.key = tagContent(i);
				else {
					JabFormField *f = new JabFormField;
					if(f->setType(i.tagName())) {
						f->setValue(tagContent(i));
						form.append(f);
					}
					else {
						delete f;
					}
				}
			}
			setSuccess(TRUE);
			return TRUE;
		}
		else {
			jid = x.attribute("from");
			setSuccess(TRUE);
			return TRUE;
		}
	}
	else {
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}


/****************************************************************************
  JT_Search
****************************************************************************/
JT_Search::JT_Search(JabTask *parent)
:JabTask(parent)
{
	type = -1;
}

void JT_Search::get(const QString &_jid)
{
	type = 0;
	jid = _jid;
	iq = createIQ("get", jid, id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:search");
	iq.appendChild(query);
}

void JT_Search::set(const JabForm &form)
{
	type = 1;
	jid = form.jid.full();
	iq = createIQ("set", form.jid.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:search");
	iq.appendChild(query);

	// key?
	if(!form.key.isEmpty())
		query.appendChild(textTag("key", form.key));
	// fields
	QPtrListIterator<JabFormField> it(form);
	for(JabFormField *f; (f = it.current()); ++it) {
		if(f->value().isEmpty())
			continue;

		query.appendChild(textTag(f->realName(), f->value()));
	}
}

void JT_Search::go()
{
	send(iq);
}

bool JT_Search::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		if(type == 0) {
			form.jid = x.attribute("from");
			QDomElement q = queryTag(x);
			for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
				QDomElement i = n.toElement();
				if(i.isNull())
					continue;

				if(i.tagName() == "instructions")
					form.instructions = tagContent(i);
				else if(i.tagName() == "key")
					form.key = tagContent(i);
				else {
					JabFormField *f = new JabFormField;
					if(f->setType(i.tagName())) {
						f->setValue(tagContent(i));
						form.append(f);
					}
					else {
						delete f;
					}
				}
			}
			setSuccess(TRUE);
			return TRUE;
		}
		else {
			jid = x.attribute("from");
			QDomElement q = queryTag(x);
			for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
				QDomElement i = n.toElement();
				if(i.isNull())
					continue;

				if(i.tagName() == "item") {
					JabRosterEntry entry;

					entry.jid = i.attribute("jid");

					QDomElement tag;
					bool found;

					tag = findSubTag(i, "nick", &found);
					if(found)
						entry.nick = tagContent(tag);
					tag = findSubTag(i, "first", &found);
					if(found)
						entry.first = tagContent(tag);
					tag = findSubTag(i, "last", &found);
					if(found)
						entry.last = tagContent(tag);
					tag = findSubTag(i, "email", &found);
					if(found)
						entry.email = tagContent(tag);

					JabRosterEntry *tmp = new JabRosterEntry(entry);
					roster.add(tmp);
				}
			}
			setSuccess(TRUE);
			return TRUE;
		}
	}
	else {
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}


/****************************************************************************
  JT_ClientVersion
****************************************************************************/
JT_ClientVersion::JT_ClientVersion(JabTask *parent, const Jid &_j)
:JabTask(parent)
{
	j = _j;
	iq = createIQ("get", j.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:version");
	iq.appendChild(query);
}

void JT_ClientVersion::go()
{
	send(iq);
}

bool JT_ClientVersion::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		bool found;
		QDomElement q = queryTag(x);
		QDomElement tag;
		tag = findSubTag(x, "name", &found);
		if(found)
			name = tagContent(tag);
		tag = findSubTag(x, "version", &found);
		if(found)
			version = tagContent(tag);
		tag = findSubTag(x, "os", &found);
		if(found)
			os = tagContent(tag);

		setSuccess(FALSE);
	}
	else {
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}


/****************************************************************************
  JT_ClientTime
****************************************************************************/
JT_ClientTime::JT_ClientTime(JabTask *parent, const Jid &_j)
:JabTask(parent)
{
	j = _j;
	iq = createIQ("get", j.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:time");
	iq.appendChild(query);
}

void JT_ClientTime::go()
{
	send(iq);
}

bool JT_ClientTime::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		bool found;
		QDomElement q = queryTag(x);
		QDomElement tag;
		tag = findSubTag(x, "utc", &found);
		if(found)
			stamp2TS(tagContent(tag), &utc);
		tag = findSubTag(x, "tz", &found);
		if(found)
			timezone = tagContent(tag);
		tag = findSubTag(x, "display", &found);
		if(found)
			display = tagContent(tag);

		setSuccess(FALSE);
	}
	else {
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}


/****************************************************************************
  JT_ServInfo
****************************************************************************/
JT_ServInfo::JT_ServInfo(JabTask *parent)
:JabTask(parent)
{
}

bool JT_ServInfo::take(const QDomElement &e)
{
	if(e.tagName() != "iq")
		return FALSE;

	if(e.attribute("type") != "get")
		return FALSE;

	QString ns = queryNS(e);
	if(ns == "jabber:iq:version") {
		QDomElement iq = createIQ("result", e.attribute("from"), e.attribute("id"));
		QDomElement query = doc()->createElement("query");
		query.setAttribute("xmlns", "jabber:iq:version");
		iq.appendChild(query);
		query.appendChild(textTag("name", PROG_NAME));
		query.appendChild(textTag("version", PROG_VERSION));
		query.appendChild(textTag("os", getOSName()));
		send(iq);
		return TRUE;
	}
	else if(ns == "jabber:iq:time") {
		QDomElement iq = createIQ("result", e.attribute("from"), e.attribute("id"));
		QDomElement query = doc()->createElement("query");
		query.setAttribute("xmlns", "jabber:iq:time");
		iq.appendChild(query);
		QDateTime local = QDateTime::currentDateTime();
		QDateTime utc = local.addSecs(-getTZOffset() * 3600);
		QString str = getTZString();
		query.appendChild(textTag("utc", TS2stamp(utc)));
		query.appendChild(textTag("tz", str));
		query.appendChild(textTag("display", QString("%1 %2").arg(local.toString()).arg(str)));
		send(iq);
		return TRUE;
	}
	else
		return FALSE;
}


/****************************************************************************
  JT_Search
****************************************************************************/
JT_Gateway::JT_Gateway(JabTask *parent)
:JabTask(parent)
{
	type = -1;
}

void JT_Gateway::get(const QString &_jid)
{
	type = 0;
	jid = _jid;
	iq = createIQ("get", jid, id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:gateway");
	iq.appendChild(query);
}

void JT_Gateway::set(const QString &_jid, const QString &prompt)
{
	type = 1;
	jid = _jid;
	iq = createIQ("set", jid, id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:gateway");
	iq.appendChild(query);
	query.appendChild(textTag("prompt", prompt));
}

void JT_Gateway::go()
{
	send(iq);
}

bool JT_Gateway::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		if(type == 0) {
			QDomElement query = queryTag(x);
			bool found;
			QDomElement tag;
			tag = findSubTag(query, "desc", &found);
			if(found)
				desc = tagContent(tag);
			tag = findSubTag(query, "prompt", &found);
			if(found)
				prompt = tagContent(tag);
		}
		else {
			QDomElement query = queryTag(x);
			bool found;
			QDomElement tag;
			tag = findSubTag(query, "prompt", &found);
			if(found)
				prompt = tagContent(tag);
		}

		setSuccess(TRUE);
	}
	else {
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}
