/****************************************************************************
** sslfilter.h - simple OpenSSL encryption I/O
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#ifndef SSLFILTER_H
#define SSLFILTER_H

#include<qobject.h>
#include<qcstring.h>
#include<qlibrary.h>


/****************************************************************************
  SSLFilter

  This class handles an OpenSSL session.  It works independent of fds and
  sockets.  Pass incoming socket data to it via putIncomingSSLData().  When
  OpenSSL wants to send data out the channel, outgoingSSLDataReady() will
  be emitted and you can get the data to be sent with getOutgoingSSLData().

  Once you have these hooks set up, the rest is easy.  Use send() to encrypt
  data and use recv() to retrieve unencrypted data.

  When you call send(), the data is encrypted and this class will then signal
  for you to do the actual sending (as described above).

  HOWTO:

  Initialize:

    SSLFilter ssl;

    connect(&ssl, SIGNAL(readyRead()), SLOT(ssl_readyRead()));
    connect(&ssl, SIGNAL(outgoingSSLDataReady()), SLOT(ssl_outgoingReady()));

  When you are ready to begin negotiation, use begin().  You would probably
  always do this immediately after you establish a socket connection.  The
  negotiation happens transparently, and if you send data before the SSL
  channel is ready then it will be queued.  So it is OK to do something
  like this:

    // slot that gets called when QSocket emits connected()
    void sock_connected()
    {
      if(!ssl.begin())
        return; // error initializing ssl

      // login (or something)
      sock->send(array.data(), array.size());
    }

  Receiving data requires one extra step than before.  Instead of
  processing the socket data from your readyRead slot, you need to pass
  the data on to SSLFilter.  SSLFilter will then emit its own readyRead for
  real data that you can process.

    // slot that gets called when QSocket emits readyRead()
    void sock_readyRead()
    {
      QByteArray buf;
      int size = sock->bytesAvailable();
      buf.resize(size);
      sock->readBlock(buf.data(), size);
      ssl.putIncomingSSLData(buf);
    }

    // slot that gets called when SSLFilter emits readyRead()
    void ssl_readyRead()
    {
      QByteArray buf = ssl->recv();

      // do something
      processIncomingData(buf);
    }

  Often, SSLFilter will want to send data.  This will happen generally
  after you call SSLFilter::send(), but OpenSSL may have other reasons
  sometimes.  It will emit outgoingSSLDataReady() when data is ready
  to be sent.  Simple!

    void ssl_outgoingReady()
    {
      QByteArray buf = ssl->getOutgoingSSLData();
      sock->writeBlock(buf.data(), buf.size());
    }

  To send data, use SSLFilter::send().  SSLFilter will perform the
  encryption, and signal you to send the encrypted data when ready.

    QByteArray buf;
    ssl.send(buf);

  The OpenSSL library is used dynamically during runtime, and is thus not
  required to compile or run your program.  Be sure to use the isSupported()
  function to make sure that the library was loaded properly.

  On error, SSLFilter will emit error() and reset the class.

****************************************************************************/

class SSLFilter : public QObject
{
	Q_OBJECT

public:
	SSLFilter();
	~SSLFilter();

	bool isSupported();
	void reset();
	bool begin();

	// send data
	void send(const QByteArray &);
	// check/recv data
	bool isRecvData();
	QByteArray recv();

	// pass incoming socket data to this function
	void putIncomingSSLData(const QByteArray &);
	// check/read outgoing socket data with this function
	bool isOutgoingSSLData();
	QByteArray getOutgoingSSLData();

signals:
	void readyRead();
	void outgoingSSLDataReady();
	void error();

private slots:
	void sslUpdate();

private:
	enum { SSL_CONNECT, SSL_ACTIVE };
	int sslAction;

	bool supported;
	void *dp;

	QByteArray sendQueue, recvQueue;
	void processSendQueue();
	void sslReadAll();
	void doError();
};

#endif
