#include "opt_presence.h"
#include "common.h"
#include "iconwidget.h"

#include <qbuttongroup.h>
#include <qwhatsthis.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qspinbox.h>
#include <qtextedit.h>
#include <qinputdialog.h>

#include "opt_presence_auto.h"
#include "opt_presence_presets.h"
#include "opt_presence_misc.h"

//----------------------------------------------------------------------------
// OptionsTabPresence
//----------------------------------------------------------------------------

OptionsTabPresence::OptionsTabPresence(QObject *parent)
: MetaOptionsTab(parent, "presence", "", tr("Presence"), tr("Configure the presence options"), "status/online")
{
	addTab( new OptionsTabPresenceAuto(this) );
	addTab( new OptionsTabPresencePresets(this) );
	addTab( new OptionsTabPresenceMisc(this) );
}

//----------------------------------------------------------------------------
// OptionsTabPresenceAuto
//----------------------------------------------------------------------------

OptionsTabPresenceAuto::OptionsTabPresenceAuto(QObject *parent)
: OptionsTab(parent, "presence_auto", "presence", tr("Auto status"), tr("Configure your auto-idle status"))
{
	w = 0;
}

QWidget *OptionsTabPresenceAuto::widget()
{
	if ( w )
		return 0;

	w = new PresenceAutoUI;
	PresenceAutoUI *d = (PresenceAutoUI *)w;

	QString s = tr("Makes Psi automatically set your status to \"away\" if your"
		" computer is idle for the specified amount of time.");
	QWhatsThis::add(d->ck_asAway, s);
	QWhatsThis::add(d->sb_asAway, s);
	s = tr("Makes Psi automatically set your status to \"extended away\" if your"
		" computer is idle for the specified amount of time.");
	QWhatsThis::add(d->ck_asXa, s);
	QWhatsThis::add(d->sb_asXa, s);
	s = tr("Makes Psi automatically set your status to \"offline\" if your"
		" computer is idle for the specified amount of time."
		"  This will disconnect you from the Jabber server.");
	QWhatsThis::add(d->ck_asOffline, s);
	QWhatsThis::add(d->sb_asOffline, s);

	QWhatsThis::add(d->te_asMessage,
		tr("Specifies an extended message to use if you allow Psi"
		" to set your status automatically.  See options above."));

	return w;
}

void OptionsTabPresenceAuto::applyOptions(Options *opt)
{
	if ( !w )
		return;

	PresenceAutoUI *d = (PresenceAutoUI *)w;
	opt->asAway = d->sb_asAway->value();
	opt->asXa = d->sb_asXa->value();
	opt->asOffline = d->sb_asOffline->value();
	opt->use_asAway = d->ck_asAway->isChecked();
	opt->use_asXa = d->ck_asXa->isChecked();
	opt->use_asOffline = d->ck_asOffline->isChecked();
	opt->asMessage = d->te_asMessage->text();
}

void OptionsTabPresenceAuto::restoreOptions(const Options *opt)
{
	if ( !w )
		return;

	PresenceAutoUI *d = (PresenceAutoUI *)w;
	d->sb_asAway->setMinValue(0);
	d->sb_asAway->setValue( opt->asAway );
	d->sb_asXa->setMinValue(0);
	d->sb_asXa->setValue( opt->asXa );
	d->sb_asOffline->setMinValue(0);
	d->sb_asOffline->setValue( opt->asOffline );
	/*if (opt->asAway <= 0 )
		opt->use_asAway = FALSE;
	if (opt->asXa <= 0 )
		opt->use_asXa = FALSE;
	if(d->opt.asOffline <= 0)
		opt->use_asOffline = FALSE;*/
	d->ck_asAway->setChecked( opt->use_asAway );
	d->ck_asXa->setChecked( opt->use_asXa );
	d->ck_asOffline->setChecked( opt->use_asOffline );
	d->te_asMessage->setText( opt->asMessage );
}

//----------------------------------------------------------------------------
// OptionsTabPresencePresets
//----------------------------------------------------------------------------

OptionsTabPresencePresets::OptionsTabPresencePresets(QObject *parent)
: OptionsTab(parent, "presence_presets", "presence", tr("Presets"), tr("Set up custom status presets"))
{
	w = 0;
	o = new Options;
}

OptionsTabPresencePresets::~OptionsTabPresencePresets()
{
	delete o;
}

QWidget *OptionsTabPresencePresets::widget()
{
	if ( w )
		return 0;

	w = new PresencePresetsUI;
	PresencePresetsUI *d = (PresencePresetsUI *)w;

	d->pb_spNew->setEnabled(TRUE);
	d->pb_spDelete->setEnabled(FALSE);
	d->te_sp->setEnabled(FALSE);
	connect(d->pb_spNew, SIGNAL(clicked()), SLOT(newStatusPreset()));
	connect(d->pb_spDelete, SIGNAL(clicked()), SLOT(removeStatusPreset()));
	connect(d->lb_sp, SIGNAL(highlighted(int)), SLOT(selectStatusPreset(int)));
	connect(d->te_sp, SIGNAL(textChanged()), SLOT(changeStatusPreset()));

	QWhatsThis::add(d->pb_spNew,
		tr("Press this button to create a new status message preset."));
	QWhatsThis::add(d->pb_spDelete,
		tr("Press this button to delete a status message preset."));
	QWhatsThis::add(d->lb_sp,
		tr("Use this list to select a status message preset"
		" to view or edit in the box to the right."));
	QWhatsThis::add(d->te_sp,
		tr("You may edit the message here for the currently selected"
		" status message preset in the list to the left."));

	return w;
}

void OptionsTabPresencePresets::setData(PsiCon *, QWidget *parentDialog)
{
	parentWidget = parentDialog;
}

void OptionsTabPresencePresets::applyOptions(Options *opt)
{
	if ( !w )
		return;

	opt->sp = o->sp;
}

void OptionsTabPresencePresets::restoreOptions(const Options *opt)
{
	if ( !w )
		return;

	PresencePresetsUI *d = (PresencePresetsUI *)w;

	o->sp = opt->sp;
	d->lb_sp->insertStringList(o->sp.varsToStringList());
	if(d->lb_sp->count() >= 1)
		d->lb_sp->setSelected(0, TRUE);
}

void OptionsTabPresencePresets::selectStatusPreset(int x)
{
	PresencePresetsUI *d = (PresencePresetsUI *)w;

	//noDirty = TRUE;
	disconnect(d->te_sp, SIGNAL(textChanged()), 0, 0);
	if ( x == -1 ) {
		d->pb_spDelete->setEnabled(false);
		d->te_sp->setText("");
		d->te_sp->setEnabled(false);

		//noDirty = FALSE;
		connect(d->te_sp, SIGNAL(textChanged()), SLOT(changeStatusPreset()));
		return;
	}

	d->pb_spDelete->setEnabled(true);

	d->te_sp->setText( o->sp.get( d->lb_sp->text(x)) );
	d->te_sp->setEnabled(true);

	//noDirty = FALSE;
	connect(d->te_sp, SIGNAL(textChanged()), SLOT(changeStatusPreset()));
}

void OptionsTabPresencePresets::newStatusPreset()
{
	PresencePresetsUI *d = (PresencePresetsUI *)w;

	QString text;

	while(1) {
		bool ok = FALSE;
		text = QInputDialog::getText(
			CAP(tr("New Status Preset")),
			tr("Please enter a name for the new status preset:"),
			QLineEdit::Normal, text, &ok, parentWidget);
		if(!ok)
			return;

		if(text.isEmpty())
			QMessageBox::information(parentWidget, tr("Error"), tr("Can't create a blank preset!"));
		else if(o->sp.findByKey(text) != o->sp.end())
			QMessageBox::information(parentWidget, tr("Error"), tr("You already have a preset with that name!"));
		else
			break;
	}

	o->sp.set(text, "");
	d->lb_sp->insertItem(text);
	d->lb_sp->setSelected(d->lb_sp->count()-1, TRUE);
	d->te_sp->setFocus();

	emit dataChanged();
}

void OptionsTabPresencePresets::removeStatusPreset()
{
	PresencePresetsUI *d = (PresencePresetsUI *)w;
	int id = d->lb_sp->currentItem();
	if(id == -1)
		return;

	emit dataChanged();

	o->sp.unset(d->lb_sp->text(id));
	d->lb_sp->removeItem(id);

	// select a new entry if possible
	if(d->lb_sp->count() == 0) {
		selectStatusPreset(-1);
		return;
	}

	if(id >= (int)d->lb_sp->count())
		id = d->lb_sp->count()-1;

	d->lb_sp->setSelected(id, TRUE);
	selectStatusPreset(id);
}

void OptionsTabPresencePresets::changeStatusPreset()
{
	PresencePresetsUI *d = (PresencePresetsUI *)w;
	int id = d->lb_sp->currentItem();
	if(id == -1)
		return;

	o->sp.set(d->lb_sp->text(id), d->te_sp->text());
	emit dataChanged();
}

//----------------------------------------------------------------------------
// OptionsTabPresenceMisc
//----------------------------------------------------------------------------

OptionsTabPresenceMisc::OptionsTabPresenceMisc(QObject *parent)
: OptionsTab(parent, "presence_misc", "presence", tr("Misc"), tr("Miscellaneous settings"))
{
	w = 0;
}

QWidget *OptionsTabPresenceMisc::widget()
{
	if ( w )
		return 0;

	w = new PresenceMiscUI;
	PresenceMiscUI *d = (PresenceMiscUI *)w;

	QWhatsThis::add(d->ck_askOnline,
		tr("Jabber allows you to put extended status messages on"
		" all status types.  Normally, Psi does not prompt you for"
		" an extended message when you set your status to \"online\"."
		"  Check this option if you want to have this prompt."));
	QWhatsThis::add(d->ck_rosterAnim,
		tr("Makes Psi animate contact names in the main window when they come online."));
	QWhatsThis::add(d->ck_autoVersion,
		tr("Automatically sends iq:version query to contact when he becomes online. This allows you to determine what clien he is using to get online. Note: results in increased traffic."));
	QWhatsThis::add(d->ck_autoVCardOnLogin,
		tr("By default, Psi always checks your vCard on login. If you want to save some traffic, you can uncheck this option."));

	return w;
}

void OptionsTabPresenceMisc::applyOptions(Options *opt)
{
	if ( !w )
		return;

	PresenceMiscUI *d = (PresenceMiscUI *)w;
	opt->askOnline = d->ck_askOnline->isChecked();
	opt->rosterAnim = d->ck_rosterAnim->isChecked();
	opt->autoVersion = d->ck_autoVersion->isChecked();
	opt->autoVCardOnLogin = d->ck_autoVCardOnLogin->isChecked();
}

void OptionsTabPresenceMisc::restoreOptions(const Options *opt)
{
	if ( !w )
		return;

	PresenceMiscUI *d = (PresenceMiscUI *)w;
	d->ck_askOnline->setChecked( opt->askOnline );
	d->ck_rosterAnim->setChecked( opt->rosterAnim );
	d->ck_autoVersion->setChecked( opt->autoVersion );
	d->ck_autoVCardOnLogin->setChecked( opt->autoVCardOnLogin );
}
