# -*- coding: iso-8859-1 -*-
#
# pyinotify.py - high level python interface to inotify
# Copyright (C) 2005-2006 Sbastien Martini <sebastien.martini@gmail.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
# 02111-1307, USA.

"""
Interface for inotify.

@var DEBUG: debug mode, print extra informations
@type DEBUG: boolean

@author: Sebastien Martini
@license: GPL 2
@contact: sebastien.martini@gmail.com
"""
# check version
import sys
if sys.version < '2.3':
    sys.stderr.write('This module requires Python 2.3 or later.\n')
    sys.exit(1)

# import statements
import threading, os, select, struct
from inotify import inotify_init, inotify_add_watch, inotify_rm_watch

__all__ = ["DEBUG", "Event", "EventsCodes", "ProcessEventException",
           "ProcessEvent", "INotifyException", "SimpleINotify",
           "ThreadedINotify"]


# debug mode, turn on for extra messages
DEBUG = False


class Queue(list):
    """
    Queue container class (internal use).
    """
    def __init__(self):
        """
        Queue initialization.
        """
        super(Queue, self).__init__()

    def pop(self):
        """
        Get the first element and remove it from the queue.

        @return: Front queue element.
        @rtype: queue element, None if the queue is empty
        """
        try:
            return list.pop(self)
        except IndexError:
            return None

    def push(self, new_e):
        """
        Insert back one element.

        @param new_e: Queue element.
        @type new_e: instance of class
        """
        self.insert(0, new_e)



class Event(object):
    """
    Observed event structure, contains useful informations about the
    observed event.

    @ivar isdir: flag telling if the event occured against a directory.
    @type isdir: bool
    """
    def __init__(self, wd, mask, cookie, path, name=None):
        """
        As additional member there is a boolean variable isdir,
        which tell if the event occured against a directory.

        @param wd: Watch Descriptor.
        @type wd: int
        @param mask: Bitmask of events.
        @type mask: int
        @param cookie: Cookie.
        @type cookie: int
        @param path: Path of the file or directory being watched.
                     If it is a file and that's file is directly watched,
                     the basename is included. ex: path=/apath/foo.txt
        @type path: string
        @param name: Basename of the file if the event is raised on
                     a watched directory, None if it is the file itself
                     who is directly watched.
        @type name: string or None
        """
        self.wd = wd
        self.mask = mask
        self.cookie = cookie
        self.path = path
        self.name = name.rstrip('\0') # remove trailing \0
        self.isdir = bool(self.mask & EventsCodes.IN_ISDIR)


    def __repr__(self):
        """
        @return: String representation.
        @rtype: string
        """
        return "wd: %d, mask: %d, cookie: %d, path: %s, name: %s" % \
               (self.wd, self.mask, self.cookie, self.path, self.name)



class EventsCodes(object):
    """
    Events codes corresponding to events which can be watched.

    @cvar IN_ACCESS: File was accessed.
    @type IN_ACCESS: int
    @cvar IN_MODIFY: File was modified.
    @type IN_MODIFY: int
    @cvar IN_ATTRIB: Metadata changed.
    @type IN_ATTRIB: int
    @cvar IN_CLOSE_WRITE: Writtable file was closed.
    @type IN_CLOSE_WRITE: int
    @cvar IN_CLOSE_NOWRITE: Unwrittable file closed.
    @type IN_CLOSE_NOWRITE: int
    @cvar IN_OPEN: File was opened.
    @type IN_OPEN: int
    @cvar IN_MOVED_FROM: File was moved from X.
    @type IN_MOVED_FROM: int
    @cvar IN_MOVED_TO: File was moved to Y.
    @type IN_MOVED_TO: int
    @cvar IN_CREATE: Subfile was created.
    @type IN_CREATE: int
    @cvar IN_DELETE: Subfile was deleted.
    @type IN_DELETE: int
    @cvar IN_DELETE_SELF: Self was deleted.
    @type IN_DELETE_SELF: int
    @cvar IN_UNMOUNT: Backing fs was unmounted.
    @type IN_UNMOUNT: int
    @cvar IN_Q_OVERFLOW: Event queued overflowed.
    @type IN_Q_OVERFLOW: int
    @cvar IN_IGNORED: File was ignored.
    @type IN_IGNORED: int
    @cvar IN_ISDIR: Event occurred against dir.
    @type IN_ISDIR: int
    @cvar IN_ONESHOT: Only send event once.
    @type IN_ONESHOT: int
    @cvar ALL_EVENTS: Alias for considering all of the events.
    @type ALL_EVENTS: int
    """
    IN_ACCESS = 0x00000001 # File was accessed
    IN_MODIFY = 0x00000002 # File was modified
    IN_ATTRIB = 0x00000004 # Metadata changed
    IN_CLOSE_WRITE = 0x00000008	# Writtable file was closed
    IN_CLOSE_NOWRITE = 0x00000010 # Unwrittable file closed
    IN_OPEN = 0x00000020 # File was opened
    IN_MOVED_FROM = 0x00000040 # File was moved from X
    IN_MOVED_TO = 0x00000080 # File was moved to Y
    IN_CREATE = 0x00000100 # Subfile was created
    IN_DELETE = 0x00000200 # Subfile was deleted
    IN_DELETE_SELF = 0x00000400 # Self was deleted

    IN_UNMOUNT = 0x00002000 # Backing fs was unmounted
    IN_Q_OVERFLOW = 0x00004000 # Event queued overflowed
    IN_IGNORED = 0x00008000 # File was ignored

    # special flags
    IN_ISDIR = 0x40000000 # event occurred against dir
    IN_ONESHOT = 0x80000000 # only send event once

    # all events
    ALL_EVENTS = IN_ACCESS | IN_MODIFY | IN_ATTRIB | IN_CLOSE_WRITE | \
                 IN_CLOSE_NOWRITE | IN_OPEN | IN_MOVED_FROM | \
                 IN_MOVED_TO | IN_DELETE | IN_CREATE | IN_DELETE_SELF



class ProcessEventException(Exception):
    """
    ProcessEvent Exception. Raised on ProcessEvent error.
    """
    def __init__(self, msg):
        """
        @param msg: Exception string's description.
        @type msg: string
        """
        Exception.__init__(self, msg)



class ProcessEvent(object):
    """
    Class for processing event objects, can be specialized via subclassing,
    thus its behavior can be overrided:

      1. Process individual event with individual method, e.g.
         process_IN_DELETE method will process IN_DELETE events.
      2. Process aliased events, e.g. process_IN_CLOSE method will
         process both IN_CLOSE_WRITE and IN_CLOSE_NOWRITE events if
         process_IN_CLOSE_WRITE and process_IN_CLOSE_NOWRITE aren't
         defined.
      3. process_default method overriden if it is redifined.
    """

    def __call__(self, event_k):
        """
        To behave like a functor the object must be callable.
        This method is a dispatch method. Look-up order:
        1- look for process_EVENT_NAME method, 2- look for
        process_ALIAS_EVENT_NAME, 3- otherwise call process_default.

        @param event_k: Event to be processed.
        @type event_k: Event object
        @raise ProcessEventException: Event object undispatchable,
                                      event is unknown.
        """
        mask = event_k.mask
        for ev in dir(EventsCodes):
            if ev.startswith('IN_') and ev != 'IN_ISDIR' and \
                   (mask & getattr(EventsCodes, ev)):
                # 1- look for process_EVENT_NAME
                if hasattr(self, 'process_%s' % ev):
                    getattr(self, 'process_%s' % ev)(event_k)
                # 2- look for process_ALIAS_EVENT_NAME
                elif hasattr(self, 'process_IN_%s' % ev.split('_')[1]):
                    getattr(self, 'process_IN_%s' % ev.split('_')[1])(event_k)
                # 3- default method process_default
                else:
                    self.process_default(event_k, ev)
                # FIXME: is it really safe like this ?
                break
        else:
            raise ProcessEventException("Couldn't extract event" + \
                                        " from mask %d" % mask)


    def process_default(self, event_k, event):
        """
        Default processing event method. Print event description on
        standart output.

        @param event_k: Event to be processed.
        @type event_k: Event object
        @param event: Event's name, e.g. 'IN_DELETE'.
        @type event: string
        """
        if event_k.name:
            f = "%s" % os.path.join(event_k.path, event_k.name)
        else:
            f = "%s" % event_k.path
        print "WD: %d, Event: %s (0x%08x), flag_dir: %s, %s" % \
              (event_k.wd, event, getattr(EventsCodes, event),
               event_k.isdir, f)



class INotifyException(Exception):
    """
    INotify Exception. Raised on inotify error.
    """
    def __init__(self, msg):
        """
        @param msg: Exception string's description.
        @type msg: string
        """
        Exception.__init__(self, msg)



class SimpleINotify(object):
    """
    Simple (non-threaded) INotify class, provides tools for watching files
    or directories, for reading notifications of events, and for processing
    these events.
    """
    def __init__(self):
        """
        initialization.
        """
        self._eventq = Queue() # event queue
        self._fd = inotify_init() # wrapped init
        self._wdd_lock = threading.Lock() # wdd lock
        # watch dict {wd: [path, proc_fun, active], ...} proc_fun and
        # active are mutable, path is immutable. wd is key for fast lookup.
        self._wdd = {}


    def __add_watch(self, path, mask, proc_fun):
        """
        add a watch on path, stores the wd and the processing
        function, then returns the wd.
        """
        wd = inotify_add_watch(self._fd, path, mask)
        if wd < 0:
            sys.stderr.write('add_watch error: cannot add watch to %s (WD=%d)\n' % \
                             (path,wd))
            return wd
        self._wdd_lock.acquire()
        self._wdd[wd] = [os.path.normpath(path), proc_fun, True]
        self._wdd_lock.release()
        if DEBUG:
            print "WD=%d assigned to %s" % (wd, path)
        return wd


    def add_watch(self, path, mask, proc_fun=ProcessEvent(), rec=False):
        """
        Add watch(es) on given path(s) with the specified mask and
        optionnally with a processing function.

        @param path: path to watch, the path can either be a file or a
                     directory. Also accepts a sequence (list) of paths.
        @type path: string or list of string
        @param mask: Bitmask of events.
        @type mask: int
        @param proc_fun: Optionnal processing function.
        @type proc_fun: function or ProcessEvent instance or instance of
                        one of its subclasses or callable object.
        @param rec: Recursively add watches from path on all its
                    subdirectories, set to False by default (doesn't
                    follows symlinks).
        @type rec: bool
        @return: dict of paths associated to watch descriptors. A wd value
                 is positive if the watch has been sucessfully added,
                 otherwise the value is negative.
        @rtype: dict of str: int
        """
        lpath = self.__format_param(path)

        wdd = {} # returned wd dict
        for apath in lpath:
            for rpath in self.__walk_rec(apath, rec):
                wdd[rpath] = self.__add_watch(rpath, mask, proc_fun)
	return wdd


    def __get_sub_rec(self, lpath):
        """
        Get every wd from self._wdd if its path is under the path of
        one (at least) of those in l. Doesn't follow symlinks.

        @param lpath:
        @type lpath: list of string or list of int
        @return:
        @rtype: list of string or list of int
        """
        if len(lpath):
            ltype = type(lpath[0])

        newl = {}
        for d in lpath:
            # if invalid, don't start recursion
            if ltype is int:
                root = self.get_path(d)
                if not root:
                    continue
            elif ltype is str:
                wd = self.get_wd(d)
                if not wd:
                    continue
                root = d
            else:
                continue
            # always keep root
            newl[d] = 1
            if not os.path.isdir(root):
                continue
            # normalization
            root = os.path.normpath(root)
            # recursion
            lend = len(root)
            self._wdd_lock.acquire()
            for iwd in self._wdd.iteritems():
                if not iwd[1][2]: # check activity
                    continue
                cur = iwd[1][0] # path
                pref = os.path.commonprefix([root, cur])
                if root == os.sep or (len(pref) == lend and \
                                      len(cur) > lend and \
                                      cur[lend] == os.sep):
                    if ltype is int:
                        newl[iwd[0]] = 1
                    else:
                        newl[cur] = 1
            self._wdd_lock.release()
        return newl.keys()


    def update_watch(self, wd, mask=None, proc_fun=None, rec=False):
        """
        Update existing watch(es). Both the mask and the processing
        function can be modified.

        @param wd: watch descriptor to update. Also accepts a list of
                     watch descriptors.
        @type wd: int or list of int
        @param mask: Optional new bitmask of events.
        @type mask: int
        @param proc_fun: Optional new processing function.
        @type proc_fun: function or ProcessEvent instance or instance of
                        one of its subclasses or callable object.
        @param rec: Recursively update watches on every already watched
                    subdirectories and subfiles.
        @type rec: bool
        @return: dict of watch descriptors associated to booleans values.
                 True if the corresponding wd has been successfully
                 updated, False otherwise.
        @rtype: dict of int: bool
        """
        lwd = self.__format_param(wd)
        if rec:
            lwd = self.__get_sub_rec(lwd)

        wdd = {} # wd dict
        for awd in lwd:
            apath = self.get_path(awd)
            self._wdd_lock.acquire()
            try:
                if not apath or awd < 0 or not self._wdd[awd][2]:
                    sys.stderr.write('update_watch error: invalid WD=%d\n'%\
                                     awd)
                    wdd[awd] = False
                    continue
            finally:
                self._wdd_lock.release()
            if mask:
                nwd = inotify_add_watch(self._fd, apath, mask)
                if nwd < 0:
                    sys.stderr.write('update_watch error: cannot update WD=%d (%s)\n'% (nwd, apath))
                    wdd[awd] = False
                    continue
                # fixme: not too strict ?
                assert(awd == nwd)

            if proc_fun:
                self._wdd_lock.acquire()
                try:
                    self._wdd[awd][1] = proc_fun
                finally:
                    self._wdd_lock.release()
            wdd[awd] = True
            if DEBUG:
                print "WD=%d (%s) updated" % (awd, apath)
	return wdd


    def __format_param(self, param):
        """
        @param param: Parameter.
        @type param: string or int
        @return: wrap param.
        @rtype: list of type(param)
        """
        if isinstance(param, list):
            return param
        return [param]


    def get_wd(self, path):
        """
        Returns the watch descriptor associated to path, if path couldn't
        be retrieved or is inactive None is returned.

        @param path: pathname.
        @type path: string
        @return: WD or None.
        @rtype: int or None
        """
        path = os.path.normpath(path)
        self._wdd_lock.acquire()
        try:
            for iwd in self._wdd.items():
                if iwd[1][0] == path and iwd[1][2]:
                    return iwd[0]
        finally:
            self._wdd_lock.release()
        if DEBUG:
            print  'get_wd: unknown path %s' % path
        return None


    def get_path(self, wd):
        """
        Returns the path associated to WD, if WD doesn't exist or is
        inactive None is returned.

        @param wd: watch descriptor.
        @type wd: int
        @return: path or None.
        @rtype: string or None
        """
        self._wdd_lock.acquire()
        try:
            try:
                if self._wdd[wd][2]:
                    return self._wdd[wd][0]
            finally:
                self._wdd_lock.release()
        except KeyError:
            pass
        if DEBUG:
            print  'get_path: unknown WD %d' % wd
        return None


    def __walk_rec(self, top, rec):
        """
        Yields each subdirectories of top, doesn't follow symlinks.
        If rec is false, only yield top.

        @param top: root directory.
        @type top: string
        @param rec: recursive flag.
        @type rec: bool
        @return: path of one subdirectory.
        @rtype: string
        """
        if not rec or os.path.islink(top) or not os.path.isdir(top):
            yield top
        else:
            for root, dirs, files in os.walk(top):
                yield root


    def rm_watch(self, wd, rec=False):
        """
        Removes watch(es).

        @param wd: Watch descriptor of the file or directory to unwatch.
                   Also accepts a list of WDs.
        @type wd: int or list of int.
        @param rec: Recursively removes watches on every already watched
                    subdirectories and subfiles.
        @type rec: bool
        @return: dict of watch descriptors associated to booleans values.
                 True if the corresponding wd has been successfully
                 removed, False otherwise.
        @rtype: dict of int: bool
        """
        lwd = self.__format_param(wd)
        if rec:
            lwd = self.__get_sub_rec(lwd)

        wdd = {} # wd dict
        for awd in lwd:
            if DEBUG:
                apath = self.get_path(awd)
            # remove watch
            ret = inotify_rm_watch(self._fd, awd)
            if ret < 0:
                sys.stderr.write('rm_watch error: cannot remove WD=%d\n' % awd)
                wdd[awd] = False
                continue

            # Never delete wd keys in self._wdd, it seems not to
            # be a wrong choice, at worst a same new wd will overwrite
            # the old one. Moreover it seems to be neccessary in order
            # to process enqueued events even after inotify_remove has
            # been called. Howewer a flag of inactivy is set.
            self._wdd_lock.acquire()
            try:
                self._wdd[awd][2] = False
            finally:
                self._wdd_lock.release()

            wdd[awd] = True
            if DEBUG:
                print "watch WD=%d (%s) removed" % (awd, apath)
        return wdd


    def event_check(self, timeout=4):
        """
        Check for inotify events

        @param timeout: The timeout is passed on to select.select(),
                        thus timeout==0 means 'return immediately',
                        timeout>0 means 'wait up to timeout seconds'
                        and timeout==None means 'wait eternally',
                        i.e. block.
        @type timeout: int
        """
        s = select.select([self._fd],[],[], timeout)
        return len(s[0]) > 0


    def read_events(self):
        """
        Read events from device and enqueue them.
        """
        maxbuf = 16384 # max size buffer
        try:
            r = os.read(self._fd, maxbuf)
        except Exception, msg:
            raise INotifyException(msg)
        lenbuf = len(r) # buffer length
        if DEBUG:
            print "read %d bit" % lenbuf
        sumbuf = 0 # buffer iterator
        while sumbuf < lenbuf:
            s_size = 16
            # retrieve wd, mask, cookie
            s_ = struct.unpack('iIII', r[sumbuf:sumbuf+s_size])
            # length of name
            fname_len = s_[3]
            # field 'length' useless
            s_ = s_[:-1]
            self._wdd_lock.acquire()
            try:
                # append path to s_
                s_ += (self._wdd[int(s_[0])][0],)
            finally:
                self._wdd_lock.release()
            # retrieve name
            s_ += struct.unpack('%ds' % fname_len,
                                r[sumbuf+s_size:sumbuf+s_size+fname_len])
            self._eventq.push(Event(*s_))
            sumbuf += s_size + fname_len


    def process_events(self):
        """
        Process events from queue by calling their associated
        proccessing function (or functor).
        """
        el = len(self._eventq)
        for i in range(el):
            e = self._eventq.pop()
            self._wdd_lock.acquire()
            try:
                proc_fun = self._wdd[e.wd][1]
            finally:
                self._wdd_lock.release()
            proc_fun(e)


    def close(self):
        """
        Close the inotify's instance (close its file descriptor).
        It destroys all existing watches, pending events,...
        """
        os.close(self._fd)



class ThreadedINotify(threading.Thread, SimpleINotify):
    """
    Threaded INotify class, inherits from threading.Thread to instantiate
    a separate thread and from SimpleINotify. This is a threaded version
    of SimpleINotify.
    """
    def __init__(self):
        """
        Initialization, initialize parents classes.
        """
        threading.Thread.__init__(self)
        self._stopevent = threading.Event() # stop condition
        SimpleINotify.__init__(self)


    def stop(self):
        """
        Stop the thread, close the inotify's instance.
        """
        self._stopevent.set()
        threading.Thread.join(self)
        self.close()


    def run(self):
        """
        Start the thread's task: Read and process events forever
        (until the method stop() is called).
        Don't call this method directly, instead call the start()
        method inherited from threading.Thread.
        """
        while not self._stopevent.isSet():
            self.process_events()
            if self.event_check():
                self.read_events()



if __name__ == '__main__':
    #
    # Demonstrates the use of the pyinotify module.
    #
    # - Default monitoring: watch for all events and only print
    #   them on standart output (this is the default processing
    #   behavior).
    # - The watched path is '/tmp' (by default) or the first
    #   command line argument if given.
    # - The monitoring execution block and serve forever, type c^c
    #   to stop it.
    #
    import sys

    path = '/tmp' # default watched path
    if sys.argv[1:]:
        path = sys.argv[1]

    # watched events (all)
    mask = EventsCodes.ALL_EVENTS

    # class instance and init
    ino = SimpleINotify()

    # watch path for events handled by mask
    ino.add_watch(path, mask)

    print 'start monitoring %s, (press c^c to halt pyinotify)' % path
    # read and process events
    while True:
        try:
            ino.process_events()
            if ino.event_check():
                ino.read_events()
        except KeyboardInterrupt:
            # ...until c^c signal
            print 'stop monitoring...'
            # close inotify's instance
            ino.close()
            break
