# W3C XSLT source and stylesheet

from Xslt import test_harness

expected = """<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN">
<html lang='EN'><head><meta http-equiv='Content-Type' content='text/html; charset=ISO-8859-1'><meta content='text/html; charset=ISO-8859-1' http-equiv='Content-Type'><title>XSL Transformations (XSLT) Version 1.0</title><style type='text/css'>
code           { font-family: monospace; }

div.constraint,
div.issue,
div.note,
div.notice     { margin-left: 2em; }

dt.label       { display: run-in; }

li p           { margin-top: 0.3em;
                 margin-bottom: 0.3em; }
      </style><link type='text/css' rel='stylesheet' href='http://www.w3.org/StyleSheets/TR/W3C-NOTE.css'></head><body>
<div class='head'><p><a href='http://www.w3.org/'><img height='48' src='http://www.w3.org/Icons/w3c_home' width='72' alt='W3C'></a></p><h1>XSL Transformations (XSLT) Version 1.0</h1><h2>W3C Recommendation 16 November 1999</h2><dl><dt>This version:</dt><dd>
<a href='http://www.w3.org/TR/1999/REC-xslt-19991116'>http://www.w3.org/TR/1999/REC-xslt-19991116</a>
<a href='http://www.w3.org/TR/1999/REC-xslt-19991116.xml'>XML</a>
<a href='http://www.w3.org/TR/1999/REC-xslt-19991116.html'>HTML</a>

</dd><dt>Latest version:</dt><dd>
<a href='http://www.w3.org/TR/xslt'>http://www.w3.org/TR/xslt</a>
</dd><dt>Previous versions:</dt><dd>
<a href='http://www.w3.org/TR/1999/PR-xslt-19991008'>http://www.w3.org/TR/1999/PR-xslt-19991008</a>
<a href='http://www.w3.org/1999/08/WD-xslt-19990813'>http://www.w3.org/1999/08/WD-xslt-19990813</a>
<a href='http://www.w3.org/1999/07/WD-xslt-19990709'>http://www.w3.org/1999/07/WD-xslt-19990709</a>
<a href='http://www.w3.org/TR/1999/WD-xslt-19990421'>http://www.w3.org/TR/1999/WD-xslt-19990421</a>
<a href='http://www.w3.org/TR/1998/WD-xsl-19981216'>http://www.w3.org/TR/1998/WD-xsl-19981216</a>
<a href='http://www.w3.org/TR/1998/WD-xsl-19980818'>http://www.w3.org/TR/1998/WD-xsl-19980818</a>
</dd><dt>Editor:</dt>
<dd>
James Clark
 <a href='mailto:jjc@jclark.com'>&lt;jjc@jclark.com></a>
</dd>
</dl><p class='copyright'><a href='http://www.w3.org/Consortium/Legal/ipr-notice#Copyright'>Copyright</a>&nbsp;&copy;&nbsp;1999&nbsp;<a href='http://www.w3.org/'><abbr title='World Wide Web Consortium'>W3C</abbr></a><sup>&reg;</sup> (<a href='http://www.lcs.mit.edu/'><abbr title='Massachusetts Institute of Technology'>MIT</abbr></a>, <a href='http://www.inria.fr/'><abbr lang='fr' title='Institut National de Recherche en Informatique et Automatique'>INRIA</abbr></a>, <a href='http://www.keio.ac.jp/'>Keio</a>), All Rights Reserved. W3C <a href='http://www.w3.org/Consortium/Legal/ipr-notice#Legal_Disclaimer'>liability</a>, <a href='http://www.w3.org/Consortium/Legal/ipr-notice#W3C_Trademarks'>trademark</a>, <a href='http://www.w3.org/Consortium/Legal/copyright-documents-19990405'>document use</a>, and <a href='http://www.w3.org/Consortium/Legal/copyright-software-19980720'>software licensing</a> rules apply.</p></div><hr><div id='abstract'><h2><a name='abstract'>Abstract</a></h2>

<p>This specification defines the syntax and semantics of XSLT, which
is a language for transforming XML documents into other XML
documents.</p>

<p>XSLT is designed for use as part of XSL, which is a stylesheet
language for XML. In addition to XSLT, XSL includes an XML vocabulary
for specifying formatting.  XSL specifies the styling of an XML
document by using XSLT to describe how the document is transformed
into another XML document that uses the formatting vocabulary.</p>

<p>XSLT is also designed to be used independently of XSL.  However,
XSLT is not intended as a completely general-purpose XML
transformation language.  Rather it is designed primarily for the
kinds of transformations that are needed when XSLT is used as part of
XSL.</p>

</div><div id='status'><h2><a name='status'>Status of this Document</a></h2>

<p>This document has been reviewed by W3C Members and other interested
parties and has been endorsed by the Director as a W3C <a href='http://www.w3.org/Consortium/Process/#RecsW3C'>Recommendation</a>. It
is a stable document and may be used as reference material or cited as
a normative reference from other documents. W3C's role in making the
Recommendation is to draw attention to the specification and to
promote its widespread deployment. This enhances the functionality and
interoperability of the Web.</p>

<p>The list of known errors in this specification is available at
<a href='http://www.w3.org/1999/11/REC-xslt-19991116-errata'>http://www.w3.org/1999/11/REC-xslt-19991116-errata</a>.</p>

<p>Comments on this specification may be sent to <a href='mailto:xsl-editors@w3.org'>xsl-editors@w3.org</a>; <a href='http://lists.w3.org/Archives/Public/xsl-editors'>archives</a>
of the comments are available.  Public discussion of XSL, including
XSL Transformations, takes place on the <a href='http://www.mulberrytech.com/xsl/xsl-list/index.html'>XSL-List</a>
mailing list.</p>

<p>The English version of this specification is the only normative
version. However, for translations of this document, see <a href='http://www.w3.org/Style/XSL/translations.html'>http://www.w3.org/Style/XSL/translations.html</a>.</p>

<p>A list of current W3C Recommendations and other technical documents
can be found at <a href='http://www.w3.org/TR'>http://www.w3.org/TR</a>.</p>

<p>This specification has been produced as part of the <a href='http://www.w3.org/Style/Activity'>W3C Style activity</a>.</p>

</div>
<div class='toc'><h2><a name='contents'>Table of Contents</a></h2><p class='toc'>1 <a href='#id1'>Introduction</a><br>2 <a href='#id2'>Stylesheet Structure</a><br>&nbsp;&nbsp;&nbsp;&nbsp;2.1 <a href='#xslt-namespace'>XSLT Namespace</a><br>&nbsp;&nbsp;&nbsp;&nbsp;2.2 <a href='#stylesheet-element'>Stylesheet Element</a><br>&nbsp;&nbsp;&nbsp;&nbsp;2.3 <a href='#result-element-stylesheet'>Literal Result Element as Stylesheet</a><br>&nbsp;&nbsp;&nbsp;&nbsp;2.4 <a href='#qname'>Qualified Names</a><br>&nbsp;&nbsp;&nbsp;&nbsp;2.5 <a href='#forwards'>Forwards-Compatible Processing</a><br>&nbsp;&nbsp;&nbsp;&nbsp;2.6 <a href='#id3'>Combining Stylesheets</a><br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;2.6.1 <a href='#include'>Stylesheet Inclusion</a><br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;2.6.2 <a href='#import'>Stylesheet Import</a><br>&nbsp;&nbsp;&nbsp;&nbsp;2.7 <a href='#id4'>Embedding Stylesheets</a><br>3 <a href='#data-model'>Data Model</a><br>&nbsp;&nbsp;&nbsp;&nbsp;3.1 <a href='#root-node-children'>Root Node Children</a><br>&nbsp;&nbsp;&nbsp;&nbsp;3.2 <a href='#base-uri'>Base URI</a><br>&nbsp;&nbsp;&nbsp;&nbsp;3.3 <a href='#unparsed-entities'>Unparsed Entities</a><br>&nbsp;&nbsp;&nbsp;&nbsp;3.4 <a href='#strip'>Whitespace Stripping</a><br>4 <a href='#id5'>Expressions</a><br>5 <a href='#rules'>Template Rules</a><br>&nbsp;&nbsp;&nbsp;&nbsp;5.1 <a href='#id6'>Processing Model</a><br>&nbsp;&nbsp;&nbsp;&nbsp;5.2 <a href='#patterns'>Patterns</a><br>&nbsp;&nbsp;&nbsp;&nbsp;5.3 <a href='#id7'>Defining Template Rules</a><br>&nbsp;&nbsp;&nbsp;&nbsp;5.4 <a href='#id8'>Applying Template Rules</a><br>&nbsp;&nbsp;&nbsp;&nbsp;5.5 <a href='#conflict'>Conflict Resolution for Template Rules</a><br>&nbsp;&nbsp;&nbsp;&nbsp;5.6 <a href='#apply-imports'>Overriding Template Rules</a><br>&nbsp;&nbsp;&nbsp;&nbsp;5.7 <a href='#modes'>Modes</a><br>&nbsp;&nbsp;&nbsp;&nbsp;5.8 <a href='#built-in-rule'>Built-in Template Rules</a><br>6 <a href='#named-templates'>Named Templates</a><br>7 <a href='#id9'>Creating the Result Tree</a><br>&nbsp;&nbsp;&nbsp;&nbsp;7.1 <a href='#id10'>Creating Elements and Attributes</a><br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;7.1.1 <a href='#literal-result-element'>Literal Result Elements</a><br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;7.1.2 <a href='#id11'>Creating Elements with xsl:element</a><br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;7.1.3 <a href='#creating-attributes'>Creating Attributes with xsl:attribute</a><br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;7.1.4 <a href='#attribute-sets'>Named Attribute Sets</a><br>&nbsp;&nbsp;&nbsp;&nbsp;7.2 <a href='#id12'>Creating Text</a><br>&nbsp;&nbsp;&nbsp;&nbsp;7.3 <a href='#id13'>Creating Processing Instructions</a><br>&nbsp;&nbsp;&nbsp;&nbsp;7.4 <a href='#id14'>Creating Comments</a><br>&nbsp;&nbsp;&nbsp;&nbsp;7.5 <a href='#copying'>Copying</a><br>&nbsp;&nbsp;&nbsp;&nbsp;7.6 <a href='#id15'>Computing Generated Text</a><br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;7.6.1 <a href='#value-of'>Generating Text with xsl:value-of</a><br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;7.6.2 <a href='#attribute-value-templates'>Attribute Value Templates</a><br>&nbsp;&nbsp;&nbsp;&nbsp;7.7 <a href='#number'>Numbering</a><br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;7.7.1 <a href='#convert'>Number to String Conversion Attributes</a><br>8 <a href='#for-each'>Repetition</a><br>9 <a href='#id16'>Conditional Processing</a><br>&nbsp;&nbsp;&nbsp;&nbsp;9.1 <a href='#id17'>Conditional Processing with xsl:if</a><br>&nbsp;&nbsp;&nbsp;&nbsp;9.2 <a href='#id18'>Conditional Processing with xsl:choose</a><br>10 <a href='#sorting'>Sorting</a><br>11 <a href='#variables'>Variables and Parameters</a><br>&nbsp;&nbsp;&nbsp;&nbsp;11.1 <a href='#id19'>Result Tree Fragments</a><br>&nbsp;&nbsp;&nbsp;&nbsp;11.2 <a href='#variable-values'>Values of Variables and Parameters</a><br>&nbsp;&nbsp;&nbsp;&nbsp;11.3 <a href='#copy-of'>Using Values of Variables and Parameters with
xsl:copy-of</a><br>&nbsp;&nbsp;&nbsp;&nbsp;11.4 <a href='#top-level-variables'>Top-level Variables and Parameters</a><br>&nbsp;&nbsp;&nbsp;&nbsp;11.5 <a href='#local-variables'>Variables and Parameters within Templates</a><br>&nbsp;&nbsp;&nbsp;&nbsp;11.6 <a href='#id20'>Passing Parameters to Templates</a><br>12 <a href='#add-func'>Additional Functions</a><br>&nbsp;&nbsp;&nbsp;&nbsp;12.1 <a href='#document'>Multiple Source Documents</a><br>&nbsp;&nbsp;&nbsp;&nbsp;12.2 <a href='#key'>Keys</a><br>&nbsp;&nbsp;&nbsp;&nbsp;12.3 <a href='#format-number'>Number Formatting</a><br>&nbsp;&nbsp;&nbsp;&nbsp;12.4 <a href='#misc-func'>Miscellaneous Additional Functions</a><br>13 <a href='#message'>Messages</a><br>14 <a href='#extension'>Extensions</a><br>&nbsp;&nbsp;&nbsp;&nbsp;14.1 <a href='#extension-element'>Extension Elements</a><br>&nbsp;&nbsp;&nbsp;&nbsp;14.2 <a href='#id21'>Extension Functions</a><br>15 <a href='#fallback'>Fallback</a><br>16 <a href='#output'>Output</a><br>&nbsp;&nbsp;&nbsp;&nbsp;16.1 <a href='#id22'>XML Output Method</a><br>&nbsp;&nbsp;&nbsp;&nbsp;16.2 <a href='#id23'>HTML Output Method</a><br>&nbsp;&nbsp;&nbsp;&nbsp;16.3 <a href='#id24'>Text Output Method</a><br>&nbsp;&nbsp;&nbsp;&nbsp;16.4 <a href='#disable-output-escaping'>Disabling Output Escaping</a><br>17 <a href='#conformance'>Conformance</a><br>18 <a href='#notation'>Notation</a><br></p><h3>Appendices</h3><p class='toc'>A <a href='#id25'>References</a><br>&nbsp;&nbsp;&nbsp;&nbsp;A.1 <a href='#id26'>Normative References</a><br>&nbsp;&nbsp;&nbsp;&nbsp;A.2 <a href='#id27'>Other References</a><br>B <a href='#element-syntax-summary'>Element Syntax Summary</a><br>C <a href='#dtd'>DTD Fragment for XSLT Stylesheets</a> (Non-Normative)<br>D <a href='#id28'>Examples</a> (Non-Normative)<br>&nbsp;&nbsp;&nbsp;&nbsp;D.1 <a href='#id29'>Document Example</a><br>&nbsp;&nbsp;&nbsp;&nbsp;D.2 <a href='#data-example'>Data Example</a><br>E <a href='#id30'>Acknowledgements</a> (Non-Normative)<br>F <a href='#id31'>Changes from Proposed Recommendation</a> (Non-Normative)<br>G <a href='#id32'>Features under Consideration for Future Versions of XSLT</a> (Non-Normative)<br></p></div><hr><div class='body'>
<div class='div1'>
<h2><a name='id1'></a>1 Introduction</h2>

<p>This specification defines the syntax and semantics of the XSLT
language.  A transformation in the XSLT language is expressed as a
well-formed XML document <a href='#XML'>[XML]</a> conforming to the
Namespaces in XML Recommendation <a href='#XMLNAMES'>[XML Names]</a>, which may
include both elements that are defined by XSLT and elements that are
not defined by XSLT.  [<a name='dt-xslt-namespace' title='XSLT Namespace'>Definition</a>: XSLT-defined elements are distinguished by belonging to a
specific XML namespace (see <a href='#xslt-namespace'><b>2.1 XSLT Namespace</b></a>), which is
referred to in this specification as the <b>XSLT
namespace</b>.] Thus this specification is a definition of
the syntax and semantics of the XSLT namespace.</p>

<p>A transformation expressed in XSLT describes rules for transforming
a source tree into a result tree.  The transformation is achieved by
associating patterns with templates.  A pattern is matched against
elements in the source tree.  A template is instantiated to create
part of the result tree.  The result tree is separate from the source
tree.  The structure of the result tree can be completely different
from the structure of the source tree. In constructing the result
tree, elements from the source tree can be filtered and reordered, and
arbitrary structure can be added.</p>

<p>A transformation expressed in XSLT is called a stylesheet.  This is
because, in the case when XSLT is transforming into the XSL formatting
vocabulary, the transformation functions as a stylesheet.</p>

<p>This document does not specify how an XSLT stylesheet is associated
with an XML document.  It is recommended that XSL processors support
the mechanism described in <a href='#XMLSTYLE'>[XML Stylesheet]</a>.  When this or any
other mechanism yields a sequence of more than one XSLT stylesheet to
be applied simultaneously to a XML document, then the effect
should be the same as applying a single stylesheet that imports each
member of the sequence in order (see <a href='#import'><b>2.6.2 Stylesheet Import</b></a>).</p>

<p>A stylesheet contains a set of template rules.  A template rule has
two parts: a pattern which is matched against nodes in the source tree
and a template which can be instantiated to form part of the result
tree.  This allows a stylesheet to be applicable to a wide class of
documents that have similar source tree structures.</p>

<p>A template is instantiated for a particular source element
to create part of the result tree. A template can contain elements
that specify literal result element structure.  A template can also
contain elements from the XSLT namespace
that are instructions for creating result tree
fragments.  When a template is instantiated, each instruction is
executed and replaced by the result tree fragment that it creates.
Instructions can select and process descendant source elements.  Processing a
descendant element creates a result tree fragment by finding the
applicable template rule and instantiating its template. Note
that elements are only processed when they have been selected by the
execution of an instruction.  The result tree is constructed by
finding the template rule for the root node and instantiating
its template.</p>

<p>In the process of finding the applicable template rule, more
than one template rule may have a pattern that matches a given
element. However, only one template rule will be applied. The
method for deciding which template rule to apply is described
in <a href='#conflict'><b>5.5 Conflict Resolution for Template Rules</b></a>.</p>

<p>A single template by itself has considerable power: it can create
structures of arbitrary complexity; it can pull string values out of
arbitrary locations in the source tree; it can generate structures
that are repeated according to the occurrence of elements in the
source tree.  For simple transformations where the structure of the
result tree is independent of the structure of the source tree, a
stylesheet can often consist of only a single template, which
functions as a template for the complete result tree.  Transformations
on XML documents that represent data are often of this kind (see
<a href='#data-example'><b>D.2 Data Example</b></a>). XSLT allows a simplified syntax for
such stylesheets (see <a href='#result-element-stylesheet'><b>2.3 Literal Result Element as Stylesheet</b></a>).</p>

<p>When a template is instantiated, it is always instantiated with
respect to a [<a name='dt-current-node' title='Current Node'>Definition</a>: <b>current node</b>] and a [<a name='dt-current-node-list' title='Current Node List'>Definition</a>: <b>current node
list</b>]. The current node is always a member of the
current node list.  Many operations in XSLT are relative to the
current node. Only a few instructions change the current node list or
the current node (see <a href='#rules'><b>5 Template Rules</b></a> and <a href='#for-each'><b>8 Repetition</b></a>); during the instantiation of one of these
instructions, the current node list changes to a new list of nodes and
each member of this new list becomes the current node in turn; after
the instantiation of the instruction is complete, the current node and
current node list revert to what they were before the instruction was
instantiated.</p>

<p>XSLT makes use of the expression language defined by <a href='#XPATH'>[XPath]</a> for selecting elements for processing, for conditional
processing and for generating text.</p>

<p>XSLT provides two "hooks" for extending the language,
one hook for extending the set of instruction elements used in
templates and one hook for extending the set of functions used in
XPath expressions.  These hooks are both based on XML namespaces.
This version of XSLT does not define a mechanism for implementing the
hooks. See <a href='#extension'><b>14 Extensions</b></a>.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>The XSL WG intends to define such a mechanism in a future
version of this specification or in a separate
specification.</p></div>

<p>The element syntax summary notation used to describe the syntax of
XSLT-defined elements is described in <a href='#notation'><b>18 Notation</b></a>.</p>

<p>The MIME media types <code>text/xml</code> and
<code>application/xml</code> <a href='#RFC2376'>[RFC2376]</a> should be used
for XSLT stylesheets.  It is possible that a media type will be
registered specifically for XSLT stylesheets; if and when it is, that
media type may also be used.</p>

</div>

<div class='div1'>
<h2><a name='id2'></a>2 Stylesheet Structure</h2>

<div class='div2'>
<h3><a name='xslt-namespace'></a>2.1 XSLT Namespace</h3>

<p>The XSLT namespace has the URI <code>http://www.w3.org/1999/XSL/Transform</code>.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>The <code>1999</code> in the URI indicates the year in which
the URI was allocated by the W3C.  It does not indicate the version of
XSLT being used, which is specified by attributes (see <a href='#stylesheet-element'><b>2.2 Stylesheet Element</b></a> and <a href='#result-element-stylesheet'><b>2.3 Literal Result Element as Stylesheet</b></a>).</p></div>

<p>XSLT processors must use the XML namespaces mechanism <a href='#XMLNAMES'>[XML Names]</a> to recognize elements and attributes from this
namespace. Elements from the XSLT namespace are recognized only in the
stylesheet not in the source document. The complete list of
XSLT-defined elements is specified in <a href='#element-syntax-summary'><b>B Element Syntax Summary</b></a>.  Vendors must not extend the XSLT
namespace with additional elements or attributes. Instead, any
extension must be in a separate namespace.  Any namespace that is used
for additional instruction elements must be identified by means of the
extension element mechanism specified in <a href='#extension-element'><b>14.1 Extension Elements</b></a>.</p>

<p>This specification uses a prefix of <code>xsl:</code> for referring
to elements in the XSLT namespace. However, XSLT stylesheets are free
to use any prefix, provided that there is a namespace declaration that
binds the prefix to the URI of the XSLT namespace.</p>

<p>An element from the XSLT namespace may have any attribute not from
the XSLT namespace, provided that the <a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a> of the
attribute has a non-null namespace URI.  The presence of such
attributes must not change the behavior of XSLT elements and functions
defined in this document. Thus, an XSLT processor is always free to
ignore such attributes, and must ignore such attributes without giving
an error if it does not recognize the namespace URI. Such attributes
can provide, for example, unique identifiers, optimization hints, or
documentation.</p>

<p>It is an error for an element from the XSLT namespace to have
attributes with expanded-names that have null namespace URIs
(i.e. attributes with unprefixed names) other than attributes defined
for the element in this document.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>The conventions used for the names of XSLT elements,
attributes and functions are that names are all lower-case, use
hyphens to separate words, and use abbreviations only if they already
appear in the syntax of a related language such as XML or
HTML.</p></div>


</div>

<div class='div2'>
<h3><a name='stylesheet-element'></a>2.2 Stylesheet Element</h3>


  
    
  
  
    
  
  
    
  
  
    
  
  
    
    
  



  
    
  
  
    
  
  
    
  
  
    
  
  
    
    
  


<p>A stylesheet is represented by an <code>xsl:stylesheet</code>
element in an XML document.  <code>xsl:transform</code> is allowed as
a synonym for <code>xsl:stylesheet</code>.</p>

<p>An <code>xsl:stylesheet</code> element must have a
<code>version</code> attribute, indicating the version of XSLT that
the stylesheet requires.  For this version of XSLT, the value should
be <code>1.0</code>.  When the value is not equal to <code>1.0</code>,
forwards-compatible processing mode is enabled (see <a href='#forwards'><b>2.5 Forwards-Compatible Processing</b></a>).</p>

<p>The <code>xsl:stylesheet</code> element may contain the following types
of elements:</p>
<ul>
<li><p><code>xsl:import</code></p></li>
<li><p><code>xsl:include</code></p></li>
<li><p><code>xsl:strip-space</code></p></li>
<li><p><code>xsl:preserve-space</code></p></li>
<li><p><code>xsl:output</code></p></li>
<li><p><code>xsl:key</code></p></li>
<li><p><code>xsl:decimal-format</code></p></li>
<li><p><code>xsl:namespace-alias</code></p></li>
<li><p><code>xsl:attribute-set</code></p></li>
<li><p><code>xsl:variable</code></p></li>
<li><p><code>xsl:param</code></p></li>
<li><p><code>xsl:template</code></p></li>
</ul>

<p>[<a name='dt-top-level' title='Top-level'>Definition</a>: An element occurring as
a child of an <code>xsl:stylesheet</code> element is called a
<b>top-level</b> element.]</p>

<p>This example shows the structure of a stylesheet.  Ellipses
(<code>...</code>) indicate where attribute values or content have
been omitted.  Although this example shows one of each type of allowed
element, stylesheets may contain zero or more of each of these
elements.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
  &lt;xsl:import href="..."/>

  &lt;xsl:include href="..."/>

  &lt;xsl:strip-space elements="..."/>
  
  &lt;xsl:preserve-space elements="..."/>

  &lt;xsl:output method="..."/>

  &lt;xsl:key name="..." match="..." use="..."/>

  &lt;xsl:decimal-format name="..."/>

  &lt;xsl:namespace-alias stylesheet-prefix="..." result-prefix="..."/>

  &lt;xsl:attribute-set name="...">
    ...
  &lt;/xsl:attribute-set>

  &lt;xsl:variable name="...">...&lt;/xsl:variable>

  &lt;xsl:param name="...">...&lt;/xsl:param>

  &lt;xsl:template match="...">
    ...
  &lt;/xsl:template>

  &lt;xsl:template name="...">
    ...
  &lt;/xsl:template>

&lt;/xsl:stylesheet></pre></td></tr></table>

<p>The order in which the children of the <code>xsl:stylesheet</code>
element occur is not significant except for <code>xsl:import</code>
elements and for error recovery.  Users are free to order the elements
as they prefer, and stylesheet creation tools need not provide control
over the order in which the elements occur.</p>

<p>In addition, the <code>xsl:stylesheet</code> element may contain
any element not from the XSLT namespace, provided that the
expanded-name of the element has a non-null namespace URI.  The presence of
such top-level elements must not change the behavior of XSLT elements
and functions defined in this document; for example, it would not be
permitted for such a top-level element to specify that
<code>xsl:apply-templates</code> was to use different rules to resolve
conflicts. Thus, an XSLT processor is always free to ignore such
top-level elements, and must ignore a top-level element without giving
an error if it does not recognize the namespace URI. Such elements can
provide, for example,</p>

<ul>

<li><p>information used by extension elements or extension functions
(see <a href='#extension'><b>14 Extensions</b></a>),</p></li>

<li><p>information about what to do with the result tree,</p></li>

<li><p>information about how to obtain the source tree,</p></li>

<li><p>metadata about the stylesheet,</p></li>

<li><p>structured documentation for the stylesheet.</p></li>

</ul>

</div>

<div class='div2'>
<h3><a name='result-element-stylesheet'></a>2.3 Literal Result Element as Stylesheet</h3>

<p>A simplified syntax is allowed for stylesheets that consist of only
a single template for the root node.  The stylesheet may consist of
just a literal result element (see <a href='#literal-result-element'><b>7.1.1 Literal Result Elements</b></a>).  Such a stylesheet is equivalent to a
stylesheet with an <code>xsl:stylesheet</code> element containing a
template rule containing the literal result element; the template rule
has a match pattern of <code>/</code>. For example</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;html xsl:version="1.0"
      xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
      xmlns="http://www.w3.org/TR/xhtml1/strict">
  &lt;head>
    &lt;title>Expense Report Summary&lt;/title>
  &lt;/head>
  &lt;body>
    &lt;p>Total Amount: &lt;xsl:value-of select="expense-report/total"/>&lt;/p>
  &lt;/body>
&lt;/html></pre></td></tr></table>

<p>has the same meaning as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns="http://www.w3.org/TR/xhtml1/strict">
&lt;xsl:template match="/">
&lt;html>
  &lt;head>
    &lt;title>Expense Report Summary&lt;/title>
  &lt;/head>
  &lt;body>
    &lt;p>Total Amount: &lt;xsl:value-of select="expense-report/total"/>&lt;/p>
  &lt;/body>
&lt;/html>
&lt;/xsl:template>
&lt;/xsl:stylesheet></pre></td></tr></table>

<p>A literal result element that is the document element of a
stylesheet must have an <code>xsl:version</code> attribute, which
indicates the version of XSLT that the stylesheet requires.  For this
version of XSLT, the value should be <code>1.0</code>; the value must
be a <a href='http://www.w3.org/TR/xpath#NT-Number'>Number</a>.  Other literal result
elements may also have an <code>xsl:version</code> attribute. When the
<code>xsl:version</code> attribute is not equal to <code>1.0</code>,
forwards-compatible processing mode is enabled (see <a href='#forwards'><b>2.5 Forwards-Compatible Processing</b></a>).</p>

<p>The allowed content of a literal result element when used as a
stylesheet is no different from when it occurs within a
stylesheet. Thus, a literal result element used as a stylesheet cannot
contain <a title='Top-level' href='#dt-top-level'>top-level</a> elements.</p>

<p>In some situations, the only way that a system can recognize that an
XML document needs to be processed by an XSLT processor as an XSLT
stylesheet is by examining the XML document itself.  Using the
simplified syntax makes this harder.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>For example, another XML language (AXL) might also use an
<code>axl:version</code> on the document element to indicate that an
XML document was an AXL document that required processing by an AXL
processor; if a document had both an <code>axl:version</code>
attribute and an <code>xsl:version</code> attribute, it would be
unclear whether the document should be processed by an XSLT processor
or an AXL processor.</p></div>

<p>Therefore, the simplified syntax should not be used for XSLT
stylesheets that may be used in such a situation.  This situation can,
for example, arise when an XSLT stylesheet is transmitted as a message
with a MIME media type of <code>text/xml</code> or
<code>application/xml</code> to a recipient that will use the MIME
media type to determine how the message is processed.</p>

</div>

<div class='div2'>
<h3><a name='qname'></a>2.4 Qualified Names</h3>

<p>The name of an internal XSLT object, specifically a named template
(see <a href='#named-templates'><b>6 Named Templates</b></a>), a mode (see <a href='#modes'><b>5.7 Modes</b></a>), an attribute set (see <a href='#attribute-sets'><b>7.1.4 Named Attribute Sets</b></a>), a key (see <a href='#key'><b>12.2 Keys</b></a>), a
decimal-format (see <a href='#format-number'><b>12.3 Number Formatting</b></a>), a variable or a
parameter (see <a href='#variables'><b>11 Variables and Parameters</b></a>) is specified as a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>.  If it has a prefix, then the
prefix is expanded into a URI reference using the namespace
declarations in effect on the attribute in which the name occurs.  The
<a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a>
consisting of the local part of the name and the possibly null URI
reference is used as the name of the object.  The default namespace is
<em>not</em> used for unprefixed names.</p>

</div>

<div class='div2'>
<h3><a name='forwards'></a>2.5 Forwards-Compatible Processing</h3>

<p>An element enables forwards-compatible mode for itself, its
attributes, its descendants and their attributes if either it is an
<code>xsl:stylesheet</code> element whose <code>version</code>
attribute is not equal to <code>1.0</code>, or it is a literal result
element that has an <code>xsl:version</code> attribute whose value is
not equal to <code>1.0</code>, or it is a literal result element that
does not have an <code>xsl:version</code> attribute and that is the
document element of a stylesheet using the simplified syntax (see
<a href='#result-element-stylesheet'><b>2.3 Literal Result Element as Stylesheet</b></a>).  A literal result element
that has an <code>xsl:version</code> attribute whose value is equal to
<code>1.0</code> disables forwards-compatible mode for itself, its
attributes, its descendants and their attributes.</p>

<p>If an element is processed in forwards-compatible mode, then:</p>

<ul>

<li><p>if it is a <a title='Top-level' href='#dt-top-level'>top-level</a>
element and XSLT 1.0 does not allow such elements as top-level
elements, then the element must be ignored along with its
content;</p></li>

<li><p>if it is an element in a template and XSLT 1.0 does not allow
such elements to occur in templates, then if the element is not
instantiated, an error must not be signaled, and if the element is
instantiated, the XSLT must perform fallback for the element as
specified in <a href='#fallback'><b>15 Fallback</b></a>;</p></li>

<li><p>if the element has an attribute that XSLT 1.0 does not allow
the element to have or if the element has an optional attribute with a
value that the XSLT 1.0 does not allow the attribute to have, then the
attribute must be ignored.</p></li>

</ul>

<p>Thus, any XSLT 1.0 processor must be able to process the following
stylesheet without error, although the stylesheet includes elements
from the XSLT namespace that are not defined in this
specification:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet version="1.1"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
  &lt;xsl:template match="/">
    &lt;xsl:choose>
      &lt;xsl:when test="system-property('xsl:version') >= 1.1">
        &lt;xsl:exciting-new-1.1-feature/>
      &lt;/xsl:when>
      &lt;xsl:otherwise>
        &lt;html>
        &lt;head>
          &lt;title>XSLT 1.1 required&lt;/title>
        &lt;/head>
        &lt;body>
          &lt;p>Sorry, this stylesheet requires XSLT 1.1.&lt;/p>
        &lt;/body>
        &lt;/html>
      &lt;/xsl:otherwise>
    &lt;/xsl:choose>
  &lt;/xsl:template>
&lt;/xsl:stylesheet></pre></td></tr></table>

<div class='note'><p class='prefix'><b>Note:</b></p><p>If a stylesheet depends crucially on a top-level element
introduced by a version of XSL after 1.0, then the stylesheet can use
an <code>xsl:message</code> element with <code>terminate="yes"</code>
(see <a href='#message'><b>13 Messages</b></a>) to ensure that XSLT processors
implementing earlier versions of XSL will not silently ignore the
top-level element. For example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet version="1.5"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  &lt;xsl:important-new-1.1-declaration/>

  &lt;xsl:template match="/">
    &lt;xsl:choose>
      &lt;xsl:when test="system-property('xsl:version') &amp;lt; 1.1">
        &lt;xsl:message terminate="yes">
          &lt;xsl:text>Sorry, this stylesheet requires XSLT 1.1.&lt;/xsl:text>
        &lt;/xsl:message>
      &lt;/xsl:when>
      &lt;xsl:otherwise>
        ...
      &lt;/xsl:otherwise>
    &lt;/xsl:choose>
  &lt;/xsl:template>
  ...
&lt;/xsl:stylesheet></pre></td></tr></table>
</div>

<p>If an <a title='Expression' href='#dt-expression'>expression</a> occurs in
an attribute that is processed in forwards-compatible mode, then an
XSLT processor must recover from errors in the expression as
follows:</p>

<ul>

<li><p>if the expression does not match the syntax allowed by the
XPath grammar, then an error must not be signaled unless the
expression is actually evaluated;</p></li>

<li><p>if the expression calls a function with an unprefixed name
that is not part of the XSLT library, then an error must not be
signaled unless the function is actually called;</p></li>

<li><p>if the expression calls a function with a number of arguments
that XSLT does not allow or with arguments of types that XSLT does not
allow, then an error must not be signaled unless the function is
actually called.</p></li>

</ul>


</div>

<div class='div2'>
<h3><a name='id3'></a>2.6 Combining Stylesheets</h3>

<p>XSLT provides two mechanisms to combine stylesheets:</p>

<blockquote><p>

an inclusion mechanism that allows stylesheets to be combined
without changing the semantics of the stylesheets being combined,
and

an import mechanism that allows stylesheets to override each
other.

</p></blockquote>

<div class='div3'>
<h4><a name='include'></a>2.6.1 Stylesheet Inclusion</h4>


  
  
    
  
  


<p>An XSLT stylesheet may include another XSLT stylesheet using an
<code>xsl:include</code> element. The <code>xsl:include</code> element
has an <code>href</code> attribute whose value is a URI reference
identifying the stylesheet to be included.  A relative URI is resolved
relative to the base URI of the <code>xsl:include</code> element (see
<a href='#base-uri'><b>3.2 Base URI</b></a>).</p>

<p>The <code>xsl:include</code> element is only allowed as a <a title='Top-level' href='#dt-top-level'>top-level</a> element.</p>

<p>The inclusion works at the XML tree level.  The resource located by
the <code>href</code> attribute value is parsed as an XML document,
and the children of the <code>xsl:stylesheet</code> element in this
document replace the <code>xsl:include</code> element in the including
document.  The fact that template rules or definitions are included
does not affect the way they are processed.</p>

<p>The included stylesheet may use the simplified syntax described in
<a href='#result-element-stylesheet'><b>2.3 Literal Result Element as Stylesheet</b></a>.  The included stylesheet
is treated the same as the equivalent <code>xsl:stylesheet</code>
element.</p>

<p>It is an error if a stylesheet directly or indirectly includes
itself.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Including a stylesheet multiple times can cause errors
because of duplicate definitions.  Such multiple inclusions are less
obvious when they are indirect. For example, if stylesheet
<var>B</var> includes stylesheet <var>A</var>, stylesheet <var>C</var>
includes stylesheet <var>A</var>, and stylesheet <var>D</var> includes
both stylesheet <var>B</var> and stylesheet <var>C</var>, then
<var>A</var> will be included indirectly by <var>D</var> twice.  If
all of <var>B</var>, <var>C</var> and <var>D</var> are used as
independent stylesheets, then the error can be avoided by separating
everything in <var>B</var> other than the inclusion of <var>A</var>
into a separate stylesheet <var>B'</var> and changing <var>B</var> to
contain just inclusions of <var>B'</var> and <var>A</var>, similarly
for <var>C</var>, and then changing <var>D</var> to include
<var>A</var>, <var>B'</var>, <var>C'</var>.</p></div>

</div>

<div class='div3'>
<h4><a name='import'></a>2.6.2 Stylesheet Import</h4>


  
    
  
  


<p>An XSLT stylesheet may import another XSLT stylesheet using an
<code>xsl:import</code> element.  Importing a stylesheet is the same
as including it (see <a href='#include'><b>2.6.1 Stylesheet Inclusion</b></a>) except that definitions
and template rules in the importing stylesheet take precedence over
template rules and definitions in the imported stylesheet; this is
described in more detail below.  The <code>xsl:import</code> element
has an <code>href</code> attribute whose value is a URI reference
identifying the stylesheet to be imported.  A relative URI is resolved
relative to the base URI of the <code>xsl:import</code> element (see
<a href='#base-uri'><b>3.2 Base URI</b></a>).</p>

<p>The <code>xsl:import</code> element is only allowed as a <a title='Top-level' href='#dt-top-level'>top-level</a> element.  The
<code>xsl:import</code> element children must precede all other
element children of an <code>xsl:stylesheet</code> element, including
any <code>xsl:include</code> element children.  When
<code>xsl:include</code> is used to include a stylesheet, any
<code>xsl:import</code> elements in the included document are moved up
in the including document to after any existing
<code>xsl:import</code> elements in the including document.</p>

<p>For example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
  &lt;xsl:import href="article.xsl"/>
  &lt;xsl:import href="bigfont.xsl"/>
  &lt;xsl:attribute-set name="note-style">
    &lt;xsl:attribute name="font-style">italic&lt;/xsl:attribute>
  &lt;/xsl:attribute-set>
&lt;/xsl:stylesheet></pre></td></tr></table>

<p>[<a name='dt-import-tree' title='Import Tree'>Definition</a>: The
<code>xsl:stylesheet</code> elements encountered during processing of
a stylesheet that contains <code>xsl:import</code> elements are
treated as forming an <b>import tree</b>.  In the import tree,
each <code>xsl:stylesheet</code> element has one import child for each
<code>xsl:import</code> element that it contains. Any
<code>xsl:include</code> elements are resolved before constructing the
import tree.] [<a name='dt-import-precedence' title='Import Precedence'>Definition</a>: An <code>xsl:stylesheet</code> element in the import tree
is defined to have lower <b>import precedence</b> than another
<code>xsl:stylesheet</code> element in the import tree if it would be
visited before that <code>xsl:stylesheet</code> element in a
post-order traversal of the import tree (i.e. a traversal of the
import tree in which an <code>xsl:stylesheet</code> element is visited
after its import children).] Each definition and template
rule has import precedence determined by the
<code>xsl:stylesheet</code> element that contains it.</p>

<p>For example, suppose</p>

<ul>

<li><p>stylesheet <var>A</var> imports stylesheets <var>B</var>
and <var>C</var> in that order;</p></li>

<li><p>stylesheet <var>B</var> imports stylesheet
<var>D</var>;</p></li>

<li><p>stylesheet <var>C</var> imports stylesheet
<var>E</var>.</p></li>

</ul>

<p>Then the order of import precedence (lowest first) is
<var>D</var>, <var>B</var>, <var>E</var>, <var>C</var>,
<var>A</var>.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Since <code>xsl:import</code> elements are required to occur
before any definitions or template rules, an implementation that
processes imported stylesheets at the point at which it encounters the
<code>xsl:import</code> element will encounter definitions and
template rules in increasing order of import precedence.</p></div>

<p>In general, a definition or template rule with higher import
precedence takes precedence over a definition or template rule with
lower import precedence.  This is defined in detail for each kind of
definition and for template rules.</p>

<p>It is an error if a stylesheet directly or indirectly imports
itself. Apart from this, the case where a stylesheet with a particular
URI is imported in multiple places is not treated specially. The
<a title='Import Tree' href='#dt-import-tree'>import tree</a> will have a
separate <code>xsl:stylesheet</code> for each place that it is
imported.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>If <code>xsl:apply-imports</code> is used (see <a href='#apply-imports'><b>5.6 Overriding Template Rules</b></a>), the behavior may be different from the
behavior if the stylesheet had been imported only at the place with
the highest <a title='Import Precedence' href='#dt-import-precedence'>import
precedence</a>.</p></div>

</div>

</div>

<div class='div2'>
<h3><a name='id4'></a>2.7 Embedding Stylesheets</h3>

<p>Normally an XSLT stylesheet is a complete XML document with the
<code>xsl:stylesheet</code> element as the document element. However,
an XSLT stylesheet may also be embedded in another resource. Two forms
of embedding are possible:</p>

<blockquote><p>

the XSLT stylesheet may be textually embedded in a non-XML
resource, or

the <code>xsl:stylesheet</code> element may occur in an XML
document other than as the document element.

</p></blockquote>

<p>To facilitate the second form of embedding, the
<code>xsl:stylesheet</code> element is allowed to have an ID attribute
that specifies a unique identifier.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>In order for such an attribute to be used with the XPath
id function, it must actually be declared in
the DTD as being an ID.</p></div>

<p>The following example shows how the <code>xml-stylesheet</code>
processing instruction <a href='#XMLSTYLE'>[XML Stylesheet]</a> can be used to allow a
document to contain its own stylesheet.  The URI reference uses a
relative URI with a fragment identifier to locate the
<code>xsl:stylesheet</code> element:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;?xml-stylesheet type="text/xml" href="#style1"?>
&lt;!DOCTYPE doc SYSTEM "doc.dtd">
&lt;doc>
&lt;head>
&lt;xsl:stylesheet id="style1"
                version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:fo="http://www.w3.org/1999/XSL/Format">
&lt;xsl:import href="doc.xsl"/>
&lt;xsl:template match="id('foo')">
  &lt;fo:block font-weight="bold">&lt;xsl:apply-templates/>&lt;/fo:block>
&lt;/xsl:template>
&lt;xsl:template match="xsl:stylesheet">
  &lt;!-- ignore -->
&lt;/xsl:template>
&lt;/xsl:stylesheet>
&lt;/head>
&lt;body>
&lt;para id="foo">
...
&lt;/para>
&lt;/body>
&lt;/doc>
</pre></td></tr></table>

<div class='note'><p class='prefix'><b>Note:</b></p><p>A stylesheet that is embedded in the document to which it is
to be applied or that may be included or imported into an stylesheet
that is so embedded typically needs to contain a template rule that
specifies that <code>xsl:stylesheet</code> elements are to be
ignored.</p></div>

</div>

</div>

<div class='div1'>
<h2><a name='data-model'></a>3 Data Model</h2>

<p>The data model used by XSLT is the same as that used by <a href='http://www.w3.org/TR/xpath#data-model'>XPath</a> with the additions
described in this section.  XSLT operates on source, result and
stylesheet documents using the same data model.  Any two XML documents
that have the same tree will be treated the same by XSLT.</p>

<p>Processing instructions and comments in the stylesheet are ignored:
the stylesheet is treated as if neither processing instruction nodes
nor comment nodes were included in the tree that represents the
stylesheet.</p>

<div class='div2'>
<h3><a name='root-node-children'></a>3.1 Root Node Children</h3>

<p>The normal restrictions on the children of the root node are
relaxed for the result tree.  The result tree may have any sequence of
nodes as children that would be possible for an element node. In
particular, it may have text node children, and any number of element
node children. When written out using the XML output method (see
<a href='#output'><b>16 Output</b></a>), it is possible that a result tree will not
be a well-formed XML document; however, it will always be a
well-formed external general parsed entity.</p>

<p>When the source tree is created by parsing a well-formed XML
document, the root node of the source tree will automatically satisfy
the normal restrictions of having no text node children and exactly
one element child.  When the source tree is created in some other way,
for example by using the DOM, the usual restrictions are relaxed for
the source tree as for the result tree.</p>

</div>

<div class='div2'>
<h3><a name='base-uri'></a>3.2 Base URI</h3>

<p>Every node also has an associated URI called its base URI, which is
used for resolving attribute values that represent relative URIs into
absolute URIs.  If an element or processing instruction occurs in an
external entity, the base URI of that element or processing
instruction is the URI of the external entity; otherwise, the base URI
is the base URI of the document.  The base URI of the document node is
the URI of the document entity.  The base URI for a text node, a
comment node, an attribute node or a namespace node is the base URI of
the parent of the node.</p>

</div>

<div class='div2'>
<h3><a name='unparsed-entities'></a>3.3 Unparsed Entities</h3>

<p>The root node has a mapping that gives the URI for each unparsed
entity declared in the document's DTD.  The URI is generated from the
system identifier and public identifier specified in the entity
declaration. The XSLT processor may use the public identifier to
generate a URI for the entity instead of the URI specified in the
system identifier.  If the XSLT processor does not use the public
identifier to generate the URI, it must use the system identifier; if
the system identifier is a relative URI, it must be resolved into an
absolute URI using the URI of the resource containing the entity
declaration as the base URI <a href='#RFC2396'>[RFC2396]</a>.</p>

</div>

<div class='div2'>
<h3><a name='strip'></a>3.4 Whitespace Stripping</h3>

<p>After the tree for a source document or stylesheet document has
been constructed, but before it is otherwise processed by XSLT,
some text nodes are stripped.  A text node is never stripped
unless it contains only whitespace characters.  Stripping the text
node removes the text node from the tree.  The stripping process takes
as input a set of element names for which whitespace must be
preserved.  The stripping process is applied to both stylesheets and
source documents, but the set of whitespace-preserving element names
is determined differently for stylesheets and for source
documents.</p>

<p>A text node is preserved if any of the following apply:</p>

<ul>

<li><p>The element name of the parent of the text node is in the set
of whitespace-preserving element names.</p></li>

<li><p>The text node contains at least one non-whitespace character.
As in XML, a whitespace character is #x20, #x9, #xD or #xA.</p></li>

<li><p>An ancestor element of the text node has an
<code>xml:space</code> attribute with a value of
<code>preserve</code>, and no closer ancestor element has
<code>xml:space</code> with a value of
<code>default</code>.</p></li>

</ul>

<p>Otherwise, the text node is stripped.</p>

<p>The <code>xml:space</code> attributes are not stripped from the
tree.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>This implies that if an <code>xml:space</code> attribute is
specified on a literal result element, it will be included in the
result.</p></div>

<p>For stylesheets, the set of whitespace-preserving element names
consists of just <code>xsl:text</code>.</p>


  
  
    
  
  



  
  
    
  
  


<p>For source documents, the set of whitespace-preserving element
names is specified by <code>xsl:strip-space</code> and
<code>xsl:preserve-space</code> <a title='Top-level' href='#dt-top-level'>top-level</a> elements.  These elements each
have an <code>elements</code> attribute whose value is a
whitespace-separated list of <a href='http://www.w3.org/TR/xpath#NT-NameTest'>NameTest</a>s.  Initially, the
set of whitespace-preserving element names contains all element names.
If an element name matches a <a href='http://www.w3.org/TR/xpath#NT-NameTest'>NameTest</a> in an
<code>xsl:strip-space</code> element, then it is removed from the set
of whitespace-preserving element names.  If an element name matches a
<a href='http://www.w3.org/TR/xpath#NT-NameTest'>NameTest</a> in an
<code>xsl:preserve-space</code> element, then it is added to the set
of whitespace-preserving element names.  An element matches a <a href='http://www.w3.org/TR/xpath#NT-NameTest'>NameTest</a> if and only if the
<a href='http://www.w3.org/TR/xpath#NT-NameTest'>NameTest</a> would be true
for the element as an <a href='http://www.w3.org/TR/xpath#node-tests'>XPath node
test</a>.  Conflicts between matches to
<code>xsl:strip-space</code> and <code>xsl:preserve-space</code>
elements are resolved the same way as conflicts between template rules
(see <a href='#conflict'><b>5.5 Conflict Resolution for Template Rules</b></a>).  Thus, the applicable match for a
particular element name is determined as follows:</p>

<ul>

<li><p>First, any match with lower <a title='Import Precedence' href='#dt-import-precedence'>import precedence</a> than another
match is ignored.</p></li>

<li><p>Next, any match with a <a href='http://www.w3.org/TR/xpath#NT-NameTest'>NameTest</a> that has a lower
<a title='Default Priority' href='#dt-default-priority'>default priority</a> than the
<a title='Default Priority' href='#dt-default-priority'>default priority</a> of the
<a href='http://www.w3.org/TR/xpath#NT-NameTest'>NameTest</a> of another
match is ignored.</p></li>

</ul>

<p>It is an error if this leaves more than one match.  An XSLT
processor may signal the error; if it does not signal the error, it
must recover by choosing, from amongst the matches that are left, the
one that occurs last in the stylesheet.</p>

</div>

</div>

<div class='div1'>
<h2><a name='id5'></a>4 Expressions</h2>

<p>XSLT uses the expression language defined by XPath <a href='#XPATH'>[XPath]</a>.  Expressions are used in XSLT for a variety of purposes
including:</p>

<blockquote><p>
selecting nodes for processing;
specifying conditions for different ways of processing a node;
generating text to be inserted in the result tree.
</p></blockquote>

<p>[<a name='dt-expression' title='Expression'>Definition</a>: An
<b>expression</b> must match the XPath production <a href='http://www.w3.org/TR/xpath#NT-Expr'>Expr</a>.]</p>

<p>Expressions occur as the value of certain attributes on
XSLT-defined elements and within curly braces in <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute value
template</a>s.</p>

<p>In XSLT, an outermost expression (i.e. an expression that is not
part of another expression) gets its context as follows:</p>

<ul>

<li><p>the context node comes from the <a title='Current Node' href='#dt-current-node'>current node</a></p></li>

<li><p>the context position comes from the position of the <a title='Current Node' href='#dt-current-node'>current node</a> in the <a title='Current Node List' href='#dt-current-node-list'>current node list</a>; the first
position is 1</p></li>

<li><p>the context size comes from the size of the <a title='Current Node List' href='#dt-current-node-list'>current node list</a></p></li>

<li><p>the variable bindings are the bindings in scope on the
element which has the attribute in which the expression occurs (see
<a href='#variables'><b>11 Variables and Parameters</b></a>)</p></li>

<li><p>the set of namespace declarations are those in scope on the
element which has the attribute in which the expression occurs;
this includes the implicit declaration of the prefix <code>xml</code>
required by the the XML Namespaces Recommendation <a href='#XMLNAMES'>[XML Names]</a>;
the default
namespace (as declared by <code>xmlns</code>) is not part of this
set</p></li>

<li><p>the function library consists of the core function library
together with the additional functions defined in <a href='#add-func'><b>12 Additional Functions</b></a> and extension functions as described in <a href='#extension'><b>14 Extensions</b></a>; it is an error for an expression to include a call
to any other function</p></li>

</ul>

</div>

<div class='div1'>
<h2><a name='rules'></a>5 Template Rules</h2>

<div class='div2'>
<h3><a name='id6'></a>5.1 Processing Model</h3>

<p>A list of source nodes is processed to create a result tree
fragment.  The result tree is constructed by processing a list
containing just the root node.  A list of source nodes is processed by
appending the result tree structure created by processing each of the
members of the list in order.  A node is processed by finding all the
template rules with patterns that match the node, and choosing the
best amongst them; the chosen rule's template is then instantiated
with the node as the <a title='Current Node' href='#dt-current-node'>current
node</a> and with the list of source nodes as the <a title='Current Node List' href='#dt-current-node-list'>current node list</a>.  A template
typically contains instructions that select an additional list of
source nodes for processing.  The process of matching, instantiation
and selection is continued recursively until no new source nodes are
selected for processing.</p>

<p>Implementations are free to process the source document in any way
that produces the same result as if it were processed using this
processing model.</p>

</div>

<div class='div2'>
<h3><a name='patterns'></a>5.2 Patterns</h3>

<p>[<a name='dt-pattern' title='Pattern'>Definition</a>: Template rules identify the
nodes to which they apply by using a <b>pattern</b>.  As well as
being used in template rules, patterns are used for numbering (see
<a href='#number'><b>7.7 Numbering</b></a>) and for declaring keys (see <a href='#key'><b>12.2 Keys</b></a>).  A pattern specifies a set of conditions on a node.  A
node that satisfies the conditions matches the pattern; a node that
does not satisfy the conditions does not match the pattern.  The
syntax for patterns is a subset of the syntax for expressions. In
particular, location paths that meet certain restrictions can be used
as patterns.  An expression that is also a pattern always evaluates to
an object of type node-set.  A node matches a pattern if the node is a
member of the result of evaluating the pattern as an expression with
respect to some possible context; the possible contexts are those
whose context node is the node being matched or one of its
ancestors.]</p>

<p>Here are some examples of patterns:</p>

<ul>

<li><p><code>para</code> matches any <code>para</code> element</p></li>

<li><p><code>*</code> matches any element</p></li>

<li><p><code>chapter|appendix</code> matches any
<code>chapter</code> element and any <code>appendix</code>
element</p></li>

<li><p><code>olist/item</code> matches any <code>item</code> element with
an <code>olist</code> parent</p></li>

<li><p><code>appendix//para</code> matches any <code>para</code> element with
an <code>appendix</code> ancestor element</p></li>

<li><p><code>/</code> matches the root node</p></li>

<li><p><code>text()</code> matches any text node</p></li>

<li><p><code>processing-instruction()</code> matches any processing
instruction</p></li>

<li><p><code>node()</code> matches any node other than an attribute
node and the root node</p></li>

<li><p><code>id("W11")</code> matches the element with unique ID
<code>W11</code></p></li>

<li><p><code>para[1]</code> matches any <code>para</code> element
that is the first <code>para</code> child element of its
parent</p></li>

<li><p><code>*[position()=1 and self::para]</code> matches any
<code>para</code> element that is the first child element of its
parent</p></li>

<li><p><code>para[last()=1]</code> matches any <code>para</code>
element that is the only <code>para</code> child element of its
parent</p></li>

<li><p><code>items/item[position()>1]</code> matches any
<code>item</code> element that has a <code>items</code> parent and
that is not the first <code>item</code> child of its parent</p></li>

<li><p><code>item[position() mod 2 = 1]</code> would be true for any
<code>item</code> element that is an odd-numbered <code>item</code>
child of its parent.</p></li>

<li><p><code>div[@class="appendix"]//p</code> matches any
<code>p</code> element with a <code>div</code> ancestor element that
has a <code>class</code> attribute with value
<code>appendix</code></p></li>

<li><p><code>@class</code> matches any <code>class</code> attribute
(<em>not</em> any element that has a <code>class</code>
attribute)</p></li>

<li><p><code>@*</code> matches any attribute</p></li>

</ul>

<p>A pattern must match the grammar for <a href='#NT-Pattern'>Pattern</a>.  A <a href='#NT-Pattern'>Pattern</a> is
a set of location path patterns separated by <code>|</code>.  A
location path pattern is a location path whose steps all use only the
<code>child</code> or <code>attribute</code> axes.  Although patterns
must not use the <code>descendant-or-self</code> axis, patterns may
use the <code>//</code> operator as well as the <code>/</code>
operator.  Location path patterns can also start with an
id or <code>key</code> function call
with a literal argument.  Predicates in a pattern can use arbitrary
expressions just like predicates in a location path.</p>

<h5>Patterns</h5><table class='scrap' summary='Scrap'><tbody>
<tr valign='baseline'><td><a name='NT-Pattern'></a>[1]&nbsp;&nbsp;&nbsp;</td><td><code>Pattern</code></td><td>&nbsp;&nbsp;&nbsp;::=&nbsp;&nbsp;&nbsp;</td><td><code><a href='#NT-LocationPathPattern'>LocationPathPattern</a></code></td></tr><tr valign='baseline'><td></td><td></td><td></td><td><code>| <a href='#NT-Pattern'>Pattern</a> '|' <a href='#NT-LocationPathPattern'>LocationPathPattern</a></code></td></tr>
<tr valign='baseline'><td><a name='NT-LocationPathPattern'></a>[2]&nbsp;&nbsp;&nbsp;</td><td><code>LocationPathPattern</code></td><td>&nbsp;&nbsp;&nbsp;::=&nbsp;&nbsp;&nbsp;</td><td><code>'/' <a href='#NT-RelativePathPattern'>RelativePathPattern</a>?</code></td></tr><tr valign='baseline'><td></td><td></td><td></td><td><code>| <a href='#NT-IdKeyPattern'>IdKeyPattern</a> (('/' | '//') <a href='#NT-RelativePathPattern'>RelativePathPattern</a>)?</code></td></tr><tr valign='baseline'><td></td><td></td><td></td><td><code>| '//'? <a href='#NT-RelativePathPattern'>RelativePathPattern</a></code></td></tr>
<tr valign='baseline'><td><a name='NT-IdKeyPattern'></a>[3]&nbsp;&nbsp;&nbsp;</td><td><code>IdKeyPattern</code></td><td>&nbsp;&nbsp;&nbsp;::=&nbsp;&nbsp;&nbsp;</td><td><code>'id' '(' <a href='http://www.w3.org/TR/xpath#NT-Literal'>Literal</a> ')'</code></td></tr><tr valign='baseline'><td></td><td></td><td></td><td><code>| 'key' '(' <a href='http://www.w3.org/TR/xpath#NT-Literal'>Literal</a> ',' <a href='http://www.w3.org/TR/xpath#NT-Literal'>Literal</a> ')'</code></td></tr>
<tr valign='baseline'><td><a name='NT-RelativePathPattern'></a>[4]&nbsp;&nbsp;&nbsp;</td><td><code>RelativePathPattern</code></td><td>&nbsp;&nbsp;&nbsp;::=&nbsp;&nbsp;&nbsp;</td><td><code><a href='#NT-StepPattern'>StepPattern</a></code></td></tr><tr valign='baseline'><td></td><td></td><td></td><td><code>| <a href='#NT-RelativePathPattern'>RelativePathPattern</a> '/' <a href='#NT-StepPattern'>StepPattern</a></code></td></tr><tr valign='baseline'><td></td><td></td><td></td><td><code>| <a href='#NT-RelativePathPattern'>RelativePathPattern</a> '//' <a href='#NT-StepPattern'>StepPattern</a></code></td></tr>
<tr valign='baseline'><td><a name='NT-StepPattern'></a>[5]&nbsp;&nbsp;&nbsp;</td><td><code>StepPattern</code></td><td>&nbsp;&nbsp;&nbsp;::=&nbsp;&nbsp;&nbsp;</td><td><code>
<a href='#NT-ChildOrAttributeAxisSpecifier'>ChildOrAttributeAxisSpecifier</a>
<a href='http://www.w3.org/TR/xpath#NT-NodeTest'>NodeTest</a>
<a href='http://www.w3.org/TR/xpath#NT-Predicate'>Predicate</a>*
</code></td></tr>
<tr valign='baseline'><td><a name='NT-ChildOrAttributeAxisSpecifier'></a>[6]&nbsp;&nbsp;&nbsp;</td><td><code>ChildOrAttributeAxisSpecifier</code></td><td>&nbsp;&nbsp;&nbsp;::=&nbsp;&nbsp;&nbsp;</td><td><code><a href='http://www.w3.org/TR/xpath#NT-AbbreviatedAxisSpecifier'>AbbreviatedAxisSpecifier</a></code></td></tr><tr valign='baseline'><td></td><td></td><td></td><td><code>| ('child' | 'attribute') '::'</code></td></tr>
</tbody></table>

<p>A pattern is defined to match a node if and only if there is
possible context such that when the pattern is evaluated as an
expression with that context, the node is a member of the resulting
node-set.  When a node is being matched, the possible contexts have a
context node that is the node being matched or any ancestor of that
node, and a context node list containing just the context node.</p>

<p>For example, <code>p</code> matches any <code>p</code> element,
because for any <code>p</code> if the expression <code>p</code> is
evaluated with the parent of the <code>p</code> element as context the
resulting node-set will contain that <code>p</code> element as one of
its members.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>This matches even a <code>p</code> element that is the
document element, since the document root is the parent of the
document element.</p></div>

<p>Although the semantics of patterns are specified indirectly in
terms of expression evaluation, it is easy to understand the meaning
of a pattern directly without thinking in terms of expression
evaluation.  In a pattern, <code>|</code> indicates alternatives; a
pattern with one or more <code>|</code> separated alternatives matches
if any one of the alternative matches.  A pattern that consists of a
sequence of <a href='#NT-StepPattern'>StepPattern</a>s separated by
<code>/</code> or <code>//</code> is matched from right to left.  The
pattern only matches if the rightmost <a href='#NT-StepPattern'>StepPattern</a> matches and a suitable element
matches the rest of the pattern; if the separator is <code>/</code>
then only the parent is a suitable element; if the separator is
<code>//</code>, then any ancestor is a suitable element.  A <a href='#NT-StepPattern'>StepPattern</a> that uses the child axis matches
if the <a href='http://www.w3.org/TR/xpath#NT-NodeTest'>NodeTest</a> is true for the
node and the node is not an attribute node.  A <a href='#NT-StepPattern'>StepPattern</a> that uses the attribute axis
matches if the <a href='http://www.w3.org/TR/xpath#NT-NodeTest'>NodeTest</a> is true
for the node and the node is an attribute node.  When <code>[]</code>
is present, then the first <a href='http://www.w3.org/TR/xpath#NT-PredicateExpr'>PredicateExpr</a> in a <a href='#NT-StepPattern'>StepPattern</a> is evaluated with the node being
matched as the context node and the siblings of the context node that
match the <a href='http://www.w3.org/TR/xpath#NT-NodeTest'>NodeTest</a> as the
context node list, unless the node being matched is an attribute node,
in which case the context node list is all the attributes that have
the same parent as the attribute being matched and that match the <a href='http://www.w3.org/TR/xpath#NT-NameTest'>NameTest</a>.</p>

<p>For example</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>appendix//ulist/item[position()=1]</pre></td></tr></table>

<p>matches a node if and only if all of the following are true:</p>

<ul>

<li><p>the <a href='http://www.w3.org/TR/xpath#NT-NodeTest'>NodeTest</a> <code>item</code> is
true for the node and the node is not an attribute; in other words the
node is an <code>item</code> element</p></li>

<li><p>evaluating the <a href='http://www.w3.org/TR/xpath#NT-PredicateExpr'>PredicateExpr</a>
<code>position()=1</code> with the node as context node and the
siblings of the node that are <code>item</code> elements as the
context node list yields true</p></li>

<li><p>the node has a parent that matches
<code>appendix//ulist</code>; this will be true if the parent is a
<code>ulist</code> element that has an <code>appendix</code> ancestor
element.</p></li>

</ul>

</div>

<div class='div2'>
<h3><a name='id7'></a>5.3 Defining Template Rules</h3>


  
  
    
  
  
    
  
  
    
  
  
    
  
  
    
    
  


<p>A template rule is specified with the <code>xsl:template</code>
element. The <code>match</code> attribute is a <a href='#NT-Pattern'>Pattern</a> that identifies the source node or nodes
to which the rule applies.  The <code>match</code> attribute is
required unless the <code>xsl:template</code> element has a
<code>name</code> attribute (see <a href='#named-templates'><b>6 Named Templates</b></a>).
It is an error for the value of the <code>match</code> attribute to
contain a <a href='http://www.w3.org/TR/xpath#NT-VariableReference'>VariableReference</a>. The
content of the <code>xsl:template</code> element is the template that
is instantiated when the template rule is applied.</p>

<p>For example, an XML document might contain:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>This is an &lt;emph>important&lt;/emph> point.</pre></td></tr></table>

<p>The following template rule matches <code>emph</code> elements and
produces a <code>fo:inline-sequence</code> formatting object with a
<code>font-weight</code> property of <code>bold</code>.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="emph">
  &lt;fo:inline-sequence font-weight="bold">
    &lt;xsl:apply-templates/>
  &lt;/fo:inline-sequence>
&lt;/xsl:template>
</pre></td></tr></table>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Examples in this document use the <code>fo:</code> prefix for
the namespace <code>http://www.w3.org/1999/XSL/Format</code>, which is
the namespace of the formatting objects defined in <a href='#XSL'>[XSL]</a>.</p></div>

<p>As described next, the <code>xsl:apply-templates</code> element
recursively processes the children of the source element.</p>

</div>

<div class='div2'>
<h3><a name='id8'></a>5.4 Applying Template Rules</h3>


  
  
    
  
  
    
  
  
    
    
  


<p>This example creates a block for a <code>chapter</code> element and
then processes its immediate children.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="chapter">
  &lt;fo:block>
    &lt;xsl:apply-templates/>
  &lt;/fo:block>
&lt;/xsl:template></pre></td></tr></table>

<p>In the absence of a <code>select</code> attribute, the
<code>xsl:apply-templates</code> instruction processes all of the
children of the current node, including text nodes.  However, text
nodes that have been stripped as specified in <a href='#strip'><b>3.4 Whitespace Stripping</b></a>
will not be processed.  If stripping of whitespace nodes has not been
enabled for an element, then all whitespace in the content of the
element will be processed as text, and thus whitespace
between child elements will count in determining the position of a
child element as returned by the position
function.</p>

<p>A <code>select</code> attribute can be used to process nodes
selected by an expression instead of processing all children.  The
value of the <code>select</code> attribute is an <a title='Expression' href='#dt-expression'>expression</a>.  The expression must
evaluate to a node-set.  The selected set of nodes is processed in
document order, unless a sorting specification is present (see
<a href='#sorting'><b>10 Sorting</b></a>).  The following example processes all of the
<code>author</code> children of the <code>author-group</code>:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="author-group">
  &lt;fo:inline-sequence>
    &lt;xsl:apply-templates select="author"/>
  &lt;/fo:inline-sequence>
&lt;/xsl:template></pre></td></tr></table>

<p>The following example processes all of the <code>given-name</code>s
of the <code>author</code>s that are children of
<code>author-group</code>:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="author-group">
  &lt;fo:inline-sequence>
    &lt;xsl:apply-templates select="author/given-name"/>
  &lt;/fo:inline-sequence>
&lt;/xsl:template></pre></td></tr></table>

<p>This example processes all of the <code>heading</code> descendant
elements of the <code>book</code> element.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="book">
  &lt;fo:block>
    &lt;xsl:apply-templates select=".//heading"/>
  &lt;/fo:block>
&lt;/xsl:template></pre></td></tr></table>

<p>It is also possible to process elements that are not descendants of
the current node.  This example assumes that a <code>department</code>
element has <code>group</code> children and <code>employee</code>
descendants. It finds an employee's department and then processes
the <code>group</code> children of the <code>department</code>.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="employee">
  &lt;fo:block>
    Employee &lt;xsl:apply-templates select="name"/> belongs to group
    &lt;xsl:apply-templates select="ancestor::department/group"/>
  &lt;/fo:block>
&lt;/xsl:template></pre></td></tr></table>

<p>Multiple <code>xsl:apply-templates</code> elements can be used within a
single template to do simple reordering.  The following example
creates two HTML tables. The first table is filled with domestic sales
while the second table is filled with foreign sales.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="product">
  &lt;table>
    &lt;xsl:apply-templates select="sales/domestic"/>
  &lt;/table>
  &lt;table>
    &lt;xsl:apply-templates select="sales/foreign"/>
  &lt;/table>
&lt;/xsl:template></pre></td></tr></table>

<div class='note'><p class='prefix'><b>Note:</b></p>

<p>It is possible for there to be two matching descendants where one
is a descendant of the other.  This case is not treated specially:
both descendants will be processed as usual. For example, given a
source document</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;doc>&lt;div>&lt;div>&lt;/div>&lt;/div>&lt;/doc></pre></td></tr></table>

<p>the rule</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="doc">
  &lt;xsl:apply-templates select=".//div"/>
&lt;/xsl:template></pre></td></tr></table>

<p>will process both the outer <code>div</code> and inner <code>div</code>
elements.</p>

</div>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Typically, <code>xsl:apply-templates</code> is used to
process only nodes that are descendants of the current node.  Such use
of <code>xsl:apply-templates</code> cannot result in non-terminating
processing loops.  However, when <code>xsl:apply-templates</code> is
used to process elements that are not descendants of the current node,
the possibility arises of non-terminating loops. For example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="foo">
  &lt;xsl:apply-templates select="."/>
&lt;/xsl:template></pre></td></tr></table>

<p>Implementations may be able to detect such loops in some cases, but
the possibility exists that a stylesheet may enter a non-terminating
loop that an implementation is unable to detect. This may present a
denial of service security risk.</p></div>

</div>

<div class='div2'>
<h3><a name='conflict'></a>5.5 Conflict Resolution for Template Rules</h3>

<p>It is possible for a source node to match more than one template
rule. The template rule to be used is determined as follows:</p>

<ol>

<li><p>First, all matching template rules that have lower <a title='Import Precedence' href='#dt-import-precedence'>import precedence</a> than the
matching template rule or rules with the highest import precedence are
eliminated from consideration.</p></li>

<li><p>Next, all matching template rules that have lower priority
than the matching template rule or rules with the highest priority are
eliminated from consideration.  The priority of a template rule is
specified by the <code>priority</code> attribute on the template rule.
The value of this must be a real number (positive or negative),
matching the production <a href='http://www.w3.org/TR/xpath#NT-Number'>Number</a>
with an optional leading minus sign (<code>-</code>).  [<a name='dt-default-priority' title='Default Priority'>Definition</a>: The <b>default
priority</b> is computed as follows:]</p>

<ul>

<li><p>If the pattern contains multiple alternatives separated by
<code>|</code>, then it is treated equivalently to a set of template
rules, one for each alternative.</p></li>

<li><p>If the pattern has the form of a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> preceded by a <a href='#NT-ChildOrAttributeAxisSpecifier'>ChildOrAttributeAxisSpecifier</a>
or has the form
<code>processing-instruction(</code><a href='http://www.w3.org/TR/xpath#NT-Literal'>Literal</a><code>)</code> preceded by a <a href='#NT-ChildOrAttributeAxisSpecifier'>ChildOrAttributeAxisSpecifier</a>,
then the priority is 0.</p></li>

<li><p>If the pattern has the form <a href='http://www.w3.org/TR/REC-xml-names#NT-NCName'>NCName</a><code>:*</code> preceded by a
<a href='#NT-ChildOrAttributeAxisSpecifier'>ChildOrAttributeAxisSpecifier</a>,
then the priority is -0.25.</p></li>

<li><p>Otherwise, if the pattern consists of just a <a href='http://www.w3.org/TR/xpath#NT-NodeTest'>NodeTest</a> preceded by a <a href='#NT-ChildOrAttributeAxisSpecifier'>ChildOrAttributeAxisSpecifier</a>,
then the priority is -0.5.</p></li>

<li><p>Otherwise, the priority is 0.5.</p></li>

</ul>

<p>Thus, the most common kind of pattern (a pattern that tests for a
node with a particular type and a particular expanded-name) has
priority 0. The next less specific kind of pattern (a pattern that
tests for a node with a particular type and an expanded-name with a
particular namespace URI) has priority -0.25.  Patterns less specific
than this (patterns that just tests for nodes with particular types)
have priority -0.5.  Patterns more specific than the most common kind
of pattern have priority 0.5.</p>

</li>

</ol>

<p>It is an error if this leaves more than one matching template
rule.  An XSLT processor may signal the error; if it does not signal
the error, it must recover by choosing, from amongst the matching
template rules that are left, the one that occurs last in the
stylesheet.</p>

</div>

<div class='div2'>
<h3><a name='apply-imports'></a>5.6 Overriding Template Rules</h3>


  
  


<p>A template rule that is being used to override a template rule in
an imported stylesheet (see <a href='#conflict'><b>5.5 Conflict Resolution for Template Rules</b></a>) can use the
<code>xsl:apply-imports</code> element to invoke the overridden
template rule.</p>

<p>[<a name='dt-current-template-rule' title='Current Template Rule'>Definition</a>: At any point in the processing of a stylesheet, there is a
<b>current template rule</b>.  Whenever a template rule is
chosen by matching a pattern, the template rule becomes the current
template rule for the instantiation of the rule's template. When an
<code>xsl:for-each</code> element is instantiated, the current
template rule becomes null for the instantiation of the content of the
<code>xsl:for-each</code> element.]</p>

<p><code>xsl:apply-imports</code> processes the current node using
only template rules that were imported into the stylesheet element
containing the current template rule; the node is processed in the
current template rule's mode.  It is an error if
<code>xsl:apply-imports</code> is instantiated when the current
template rule is null.</p>

<p>For example, suppose the stylesheet <code>doc.xsl</code> contains a
template rule for <code>example</code> elements:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="example">
  &lt;pre>&lt;xsl:apply-templates/>&lt;/pre>
&lt;/xsl:template></pre></td></tr></table>

<p>Another stylesheet could import <code>doc.xsl</code> and modify the
treatment of <code>example</code> elements as follows:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:import href="doc.xsl"/>

&lt;xsl:template match="example">
  &lt;div style="border: solid red">
     &lt;xsl:apply-imports/>
  &lt;/div>
&lt;/xsl:template></pre></td></tr></table>

<p>The combined effect would be to transform an <code>example</code>
into an element of the form:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;div style="border: solid red">&lt;pre>...&lt;/pre>&lt;/div></pre></td></tr></table>

</div>

<div class='div2'>
<h3><a name='modes'></a>5.7 Modes</h3>

<p>Modes allow an element to be processed multiple times, each time
producing a different result.</p>

<p>Both <code>xsl:template</code> and <code>xsl:apply-templates</code>
have an optional <code>mode</code> attribute.  The value of the
<code>mode</code> attribute is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>. If <code>xsl:template</code> does not have
a <code>match</code> attribute, it must not have a <code>mode</code>
attribute.  If an <code>xsl:apply-templates</code> element has a
<code>mode</code> attribute, then it applies only to those template
rules from <code>xsl:template</code> elements that have a
<code>mode</code> attribute with the same value; if an
<code>xsl:apply-templates</code> element does not have a
<code>mode</code> attribute, then it applies only to those template
rules from <code>xsl:template</code> elements that do not have a
<code>mode</code> attribute.</p>

</div>

<div class='div2'>
<h3><a name='built-in-rule'></a>5.8 Built-in Template Rules</h3>

<p>There is a built-in template rule to allow recursive processing to
continue in the absence of a successful pattern match by an explicit
template rule in the stylesheet.  This template rule applies to both
element nodes and the root node.  The following shows the equivalent
of the built-in template rule:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="*|/">
  &lt;xsl:apply-templates/>
&lt;/xsl:template></pre></td></tr></table>

<p>There is also a built-in template rule for each mode, which allows
recursive processing to continue in the same mode in the absence of a
successful pattern match by an explicit template rule in the
stylesheet.  This template rule applies to both element nodes and the
root node.  The following shows the equivalent of the built-in
template rule for mode <code><var>m</var></code>.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="*|/" mode="<var>m</var>">
  &lt;xsl:apply-templates mode="<var>m</var>"/>
&lt;/xsl:template></pre></td></tr></table>

<p>There is also a built-in template rule for text and attribute nodes
that copies text through:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="text()|@*">
  &lt;xsl:value-of select="."/>
&lt;/xsl:template></pre></td></tr></table>

<p>The built-in template rule for processing instructions and comments
is to do nothing.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="processing-instruction()|comment()"/></pre></td></tr></table>

<p>The built-in template rule for namespace nodes is also to do
nothing. There is no pattern that can match a namespace node; so, the
built-in template rule is the only template rule that is applied for
namespace nodes.</p>

<p>The built-in template rules are treated as if they were imported
implicitly before the stylesheet and so have lower <a title='Import Precedence' href='#dt-import-precedence'>import precedence</a> than all other
template rules.  Thus, the author can override a built-in template
rule by including an explicit template rule.</p>

</div>


</div>

<div class='div1'>
<h2><a name='named-templates'></a>6 Named Templates</h2>


  
  
    
  
  


<p>Templates can be invoked by name.  An <code>xsl:template</code>
element with a <code>name</code> attribute specifies a named template.
The value of the <code>name</code> attribute is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>. If an <code>xsl:template</code> element has
a <code>name</code> attribute, it may, but need not, also have a
<code>match</code> attribute.  An <code>xsl:call-template</code>
element invokes a template by name; it has a required
<code>name</code> attribute that identifies the template to be
invoked.  Unlike <code>xsl:apply-templates</code>,
<code>xsl:call-template</code> does not change the current node or the
current node list.</p>

<p>The <code>match</code>, <code>mode</code> and <code>priority</code> attributes on an
<code>xsl:template</code> element do not affect whether the template
is invoked by an <code>xsl:call-template</code> element.  Similarly,
the <code>name</code> attribute on an <code>xsl:template</code>
element does not affect whether the template is invoked by an
<code>xsl:apply-templates</code> element.</p>

<p>It is an error if a stylesheet contains more than one template with
the same name and same <a title='Import Precedence' href='#dt-import-precedence'>import
precedence</a>.</p>

</div>


<div class='div1'>
<h2><a name='id9'></a>7 Creating the Result Tree</h2>

<p>This section describes instructions that directly create nodes in
the result tree.</p>

<div class='div2'>
<h3><a name='id10'></a>7.1 Creating Elements and Attributes</h3>

<div class='div3'>
<h4><a name='literal-result-element'></a>7.1.1 Literal Result Elements</h4>

<p>In a template, an element in the stylesheet that does not belong to
the XSLT namespace and that is not an extension element (see <a href='#extension-element'><b>14.1 Extension Elements</b></a>) is instantiated to create an element node
with the same <a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a>.  The content
of the element is a template, which is instantiated to give the
content of the created element node. The created element node will
have the attribute nodes that were present on the element node in the
stylesheet tree, other than attributes with names in the XSLT
namespace.</p>

<p>The created element node will also have a copy of the namespace
nodes that were present on the element node in the stylesheet tree
with the exception of any namespace node whose string-value is the
XSLT namespace URI (<code>http://www.w3.org/1999/XSL/Transform</code>), a
namespace URI declared as an extension namespace (see <a href='#extension-element'><b>14.1 Extension Elements</b></a>), or a namespace URI designated as an
excluded namespace.  A namespace URI is designated as an excluded
namespace by using an <code>exclude-result-prefixes</code> attribute
on an <code>xsl:stylesheet</code> element or an
<code>xsl:exclude-result-prefixes</code> attribute on a literal result
element.  The value of both these attributes is a whitespace-separated
list of namespace prefixes. The namespace bound to each of the
prefixes is designated as an excluded namespace.  It is an error if
there is no namespace bound to the prefix on the element bearing the
<code>exclude-result-prefixes</code> or
<code>xsl:exclude-result-prefixes</code> attribute.  The default
namespace (as declared by <code>xmlns</code>) may be designated as an
excluded namespace by including <code>#default</code> in the list of
namespace prefixes.  The designation of a namespace as an excluded
namespace is effective within the subtree of the stylesheet rooted at
the element bearing the <code>exclude-result-prefixes</code> or
<code>xsl:exclude-result-prefixes</code> attribute;
a subtree rooted at an <code>xsl:stylesheet</code> element
does not include any stylesheets imported or included by children
of that <code>xsl:stylesheet</code> element.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>When a stylesheet uses a namespace declaration only for the
purposes of addressing the source tree, specifying the prefix in the
<code>exclude-result-prefixes</code> attribute will avoid superfluous
namespace declarations in the result tree.</p></div>

<p>The value of an attribute of a literal result element is
interpreted as an <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute
value template</a>: it can contain expressions contained
in curly braces (<code>{}</code>).</p>

<p>[<a name='dt-literal-namespace-uri' title='Literal Namespace URI'>Definition</a>: A namespace URI in the stylesheet tree that is being used to
specify a namespace URI in the result tree is called a <b>literal
namespace URI</b>.] This applies to:</p>

<ul>

<li><p>the namespace URI in the expanded-name of a literal
result element in the stylesheet</p></li>

<li><p>the namespace URI in the expanded-name of an attribute
specified on a literal result element in the stylesheet</p></li>

<li><p>the string-value of a namespace node on a literal result
element in the stylesheet</p></li>

</ul>


  
  
    
    
  
  
    
    
  
  


<p>[<a name='dt-alias' title='Alias'>Definition</a>: A stylesheet can use the
<code>xsl:namespace-alias</code> element to declare that one namespace
URI is an <b>alias</b> for another namespace URI.] When
a <a title='Literal Namespace URI' href='#dt-literal-namespace-uri'>literal namespace
URI</a> has been declared to be an alias for another namespace
URI, then the namespace URI in the result tree will be the namespace
URI that the literal namespace URI is an alias for, instead of the
literal namespace URI itself.  The <code>xsl:namespace-alias</code>
element declares that the namespace URI bound to the prefix specified
by the <code>stylesheet-prefix</code> attribute is an alias for the
namespace URI bound to the prefix specified by the
<code>result-prefix</code> attribute.  Thus, the
<code>stylesheet-prefix</code> attribute specifies the namespace URI
that will appear in the stylesheet, and the
<code>result-prefix</code> attribute specifies the corresponding
namespace URI that will appear in the result tree.  The default
namespace (as declared by <code>xmlns</code>) may be specified by
using <code>#default</code> instead of a prefix.  If a namespace URI
is declared to be an alias for multiple different namespace URIs, then
the declaration with the highest <a title='Import Precedence' href='#dt-import-precedence'>import precedence</a> is used. It is
an error if there is more than one such declaration.  An XSLT
processor may signal the error; if it does not signal the error, it
must recover by choosing, from amongst the declarations with the
highest import precedence, the one that occurs last in the
stylesheet.</p>

<p>When literal result elements are being used to create element,
attribute, or namespace nodes that use the XSLT namespace URI, the
stylesheet must use an alias.  For example, the stylesheet</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet
  version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:fo="http://www.w3.org/1999/XSL/Format"
  xmlns:axsl="http://www.w3.org/1999/XSL/TransformAlias">

&lt;xsl:namespace-alias stylesheet-prefix="axsl" result-prefix="xsl"/>

&lt;xsl:template match="/">
  &lt;axsl:stylesheet>
    &lt;xsl:apply-templates/>
  &lt;/axsl:stylesheet>
&lt;/xsl:template>

&lt;xsl:template match="block">
  &lt;axsl:template match="{.}">
     &lt;fo:block>&lt;axsl:apply-templates/>&lt;/fo:block>
  &lt;/axsl:template>
&lt;/xsl:template>

&lt;/xsl:stylesheet></pre></td></tr></table>

<p>will generate an XSLT stylesheet from a document of the form:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;elements>
&lt;block>p&lt;/block>
&lt;block>h1&lt;/block>
&lt;block>h2&lt;/block>
&lt;block>h3&lt;/block>
&lt;block>h4&lt;/block>
&lt;/elements></pre></td></tr></table>

<div class='note'><p class='prefix'><b>Note:</b></p><p>It may be necessary also to use aliases for namespaces other
than the XSLT namespace URI.  For example, literal result elements
belonging to a namespace dealing with digital signatures might cause
XSLT stylesheets to be mishandled by general-purpose security
software; using an alias for the namespace would avoid the possibility
of such mishandling.</p></div>

</div>

<div class='div3'>
<h4><a name='id11'></a>7.1.2 Creating Elements with <code>xsl:element</code></h4>


  
  
    
      
    
  
  
    
      
    
  
  
    
  
  


<p>The <code>xsl:element</code> element allows an element to be
created with a computed name.  The <a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a> of the
element to be created is specified by a required <code>name</code>
attribute and an optional <code>namespace</code> attribute.  The
content of the <code>xsl:element</code> element is a template for the
attributes and children of the created element.</p>

<p>The <code>name</code> attribute is interpreted as an <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute value template</a>.
It is an error if the string that results from instantiating the
attribute value template is not a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>.  An XSLT processor may signal
the error; if it does not signal the error, then it must recover
by making the the result of instantiating the <code>xsl:element</code>
element be the sequence of nodes created by instantiating
the content of the  <code>xsl:element</code> element, excluding
any initial attribute nodes. If the <code>namespace</code> attribute is
not present then the <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> is
expanded into an expanded-name using the namespace declarations in
effect for the <code>xsl:element</code> element, including any default
namespace declaration.</p>

<p>If the <code>namespace</code> attribute is present, then it also is
interpreted as an <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute
value template</a>. The string that results from instantiating
the attribute value template should be a URI reference.  It is not an
error if the string is not a syntactically legal URI reference.  If
the string is empty, then the expanded-name of the element has a null
namespace URI.  Otherwise, the string is used as the namespace URI of
the expanded-name of the element to be created. The local part of the
<a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> specified by the
<code>name</code> attribute is used as the local part of the
expanded-name of the element to be created.</p>

<p>XSLT processors may make use of the prefix of the <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> specified in the
<code>name</code> attribute when selecting the prefix used for
outputting the created element as XML; however, they are not required
to do so.</p>

</div>

<div class='div3'>
<h4><a name='creating-attributes'></a>7.1.3 Creating Attributes with <code>xsl:attribute</code></h4>


  
  
    
      
    
  
  
    
      
    
  
  


<p>The <code>xsl:attribute</code> element can be used to add
attributes to result elements whether created by literal result
elements in the stylesheet or by instructions such as
<code>xsl:element</code>. The <a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a> of the
attribute to be created is specified by a required <code>name</code>
attribute and an optional <code>namespace</code> attribute.
Instantiating an <code>xsl:attribute</code> element adds an attribute
node to the containing result element node. The content of the
<code>xsl:attribute</code> element is a template for the value of the
created attribute.</p>

<p>The <code>name</code> attribute is interpreted as an <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute value template</a>.
It is an error if the string that results from instantiating the
attribute value template is not a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> or is the string
<code>xmlns</code>.  An XSLT processor may signal the error; if it
does not signal the error, it must recover by not adding the attribute
to the result tree. If the <code>namespace</code> attribute is not
present, then the <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> is
expanded into an expanded-name using the namespace declarations in
effect for the <code>xsl:attribute</code> element, <em>not</em>
including any default namespace declaration.</p>

<p>If the <code>namespace</code> attribute is present, then it also is
interpreted as an <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute
value template</a>. The string that results from instantiating
it should be a URI reference.  It is not an error if the string is not
a syntactically legal URI reference.  If the string is empty, then the
expanded-name of the attribute has a null namespace URI.  Otherwise,
the string is used as the namespace URI of the expanded-name of the
attribute to be created. The local part of the <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> specified by the
<code>name</code> attribute is used as the local part of the
expanded-name of the attribute to be created.</p>

<p>XSLT processors may make use of the prefix of the <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> specified in the
<code>name</code> attribute when selecting the prefix used for
outputting the created attribute as XML; however, they are not
required to do so and, if the prefix is <code>xmlns</code>, they must
not do so. Thus, although it is not an error to do:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:attribute name="xmlns:xsl" namespace="whatever">http://www.w3.org/1999/XSL/Transform&lt;/xsl:attribute></pre></td></tr></table>

<p>it will not result in a namespace declaration being output.</p>

<p>Adding an attribute to an element replaces any existing attribute
of that element with the same expanded-name.</p>

<p>The following are all errors:</p>

<ul>

<li><p>Adding an attribute to an element after children have been
added to it; implementations may either signal the error or ignore the
attribute.</p></li>

<li><p>Adding an attribute to a node that is not an element;
implementations may either signal the error or ignore the
attribute.</p></li>

<li><p>Creating nodes other than text nodes during the
instantiation of the content of the <code>xsl:attribute</code>
element; implementations may either signal the error or ignore the
offending nodes.</p></li>

</ul>

<div class='note'><p class='prefix'><b>Note:</b></p><p>When an <code>xsl:attribute</code> contains a text node with
a newline, then the XML output must contain a character reference.
For example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:attribute name="a">x
y&lt;/xsl:attribute></pre></td></tr></table>

<p>will result in the output</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>a="x&amp;#xA;y"</pre></td></tr></table>

<p>(or with any equivalent character reference). The XML output cannot
be</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>a="x
y"</pre></td></tr></table>

<p>This is because XML 1.0 requires newline characters in attribute
values to be normalized into spaces but requires character references
to newline characters not to be normalized.  The attribute values in
the data model represent the attribute value after normalization.  If
a newline occurring in an attribute value in the tree were output as a
newline character rather than as character reference, then the
attribute value in the tree created by reparsing the XML would contain
a space not a newline, which would mean that the tree had not been
output correctly.</p></div>

</div>

<div class='div3'>

<h4><a name='attribute-sets'></a>7.1.4 Named Attribute Sets</h4>


  
  
    
  
  
    
  
  


<p>The <code>xsl:attribute-set</code> element defines a named set of
attributes.  The <code>name</code> attribute specifies the name of the
attribute set.  The value of the <code>name</code> attribute is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>. The content of the <code>xsl:attribute-set</code>
element consists of zero or more <code>xsl:attribute</code> elements
that specify the attributes in the set.</p>

<p>Attribute sets are used by specifying a
<code>use-attribute-sets</code> attribute on <code>xsl:element</code>,
<code>xsl:copy</code> (see <a href='#copying'><b>7.5 Copying</b></a>) or
<code>xsl:attribute-set</code> elements.  The value of the
<code>use-attribute-sets</code> attribute is a whitespace-separated
list of names of attribute sets.  Each name is specified as a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>.  Specifying a
<code>use-attribute-sets</code> attribute is equivalent to adding
<code>xsl:attribute</code> elements for each of the attributes in each
of the named attribute sets to the beginning of the content of the
element with the <code>use-attribute-sets</code> attribute, in the
same order in which the names of the attribute sets are specified in
the <code>use-attribute-sets</code> attribute.  It is an error if use
of <code>use-attribute-sets</code> attributes on
<code>xsl:attribute-set</code> elements causes an attribute set to
directly or indirectly use itself.</p>

<p>Attribute sets can also be used by specifying an
<code>xsl:use-attribute-sets</code> attribute on a literal result
element.  The value of the <code>xsl:use-attribute-sets</code>
attribute is a whitespace-separated list of names of attribute sets.
The <code>xsl:use-attribute-sets</code> attribute has the same effect
as the <code>use-attribute-sets</code> attribute on
<code>xsl:element</code> with the additional rule that attributes
specified on the literal result element itself are treated as if they
were specified by <code>xsl:attribute</code> elements before any
actual <code>xsl:attribute</code> elements but after any
<code>xsl:attribute</code> elements implied by the
<code>xsl:use-attribute-sets</code> attribute.  Thus, for a literal
result element, attributes from attribute sets named in an
<code>xsl:use-attribute-sets</code> attribute will be added first, in
the order listed in the attribute; next, attributes specified on the
literal result element will be added; finally, any attributes
specified by <code>xsl:attribute</code> elements will be added.  Since
adding an attribute to an element replaces any existing attribute of
that element with the same name, this means that attributes specified
in attribute sets can be overridden by attributes specified on the
literal result element itself.</p>

<p>The template within each <code>xsl:attribute</code> element in an
<code>xsl:attribute-set</code> element is instantiated each time the
attribute set is used; it is instantiated using the same current node
and current node list as is used for instantiating the element bearing
the <code>use-attribute-sets</code> or
<code>xsl:use-attribute-sets</code> attribute. However, it is the
position in the stylesheet of the <code>xsl:attribute</code> element
rather than of the element bearing the <code>use-attribute-sets</code>
or <code>xsl:use-attribute-sets</code> attribute that determines which
variable bindings are visible (see <a href='#variables'><b>11 Variables and Parameters</b></a>); thus,
only variables and parameters declared by <a title='Top-level' href='#dt-top-level'>top-level</a> <code>xsl:variable</code> and
<code>xsl:param</code> elements are visible.</p>

<p>The following example creates a named attribute set
<code>title-style</code> and uses it in a template rule.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="chapter/heading">
  &lt;fo:block quadding="start" xsl:use-attribute-sets="title-style">
    &lt;xsl:apply-templates/>
  &lt;/fo:block>
&lt;/xsl:template>

&lt;xsl:attribute-set name="title-style">
  &lt;xsl:attribute name="font-size">12pt&lt;/xsl:attribute>
  &lt;xsl:attribute name="font-weight">bold&lt;/xsl:attribute>
&lt;/xsl:attribute-set></pre></td></tr></table>

<p>Multiple definitions of an attribute set with the same
expanded-name are merged.  An attribute from a definition that has
higher <a title='Import Precedence' href='#dt-import-precedence'>import precedence</a>
takes precedence over an attribute from a definition that has lower
<a title='Import Precedence' href='#dt-import-precedence'>import precedence</a>.  It
is an error if there are two attribute sets that have the same
expanded-name and equal import precedence and that both contain
the same attribute, unless there is a definition of the attribute set
with higher <a title='Import Precedence' href='#dt-import-precedence'>import
precedence</a> that also contains the attribute.  An XSLT
processor may signal the error; if it does not signal the error, it
must recover by choosing from amongst the definitions that specify the
attribute that have the highest import precedence the one that was
specified last in the stylesheet.  Where the attributes in an
attribute set were specified is relevant only in merging the
attributes into the attribute set; it makes no difference when the
attribute set is used.</p>

</div>

</div>

<div class='div2'>

<h3><a name='id12'></a>7.2 Creating Text</h3>

<p>A template can also contain text nodes.  Each text node in a
template remaining after whitespace has been stripped as specified in
<a href='#strip'><b>3.4 Whitespace Stripping</b></a> will create a text node with the same
string-value in the result tree.  Adjacent text nodes in the result
tree are automatically merged.</p>

<p>Note that text is processed at the tree level. Thus, markup of
<code>&amp;lt;</code> in a template will be represented in the
stylesheet tree by a text node that includes the character
<code>&lt;</code>. This will create a text node in the result tree
that contains a <code>&lt;</code> character, which will be represented
by the markup <code>&amp;lt;</code> (or an equivalent character
reference) when the result tree is externalized as an XML document
(unless output escaping is disabled as described in <a href='#disable-output-escaping'><b>16.4 Disabling Output Escaping</b></a>).</p>


  
  
    
    
  
    


<p>Literal data characters may also be wrapped in an
<code>xsl:text</code> element.  This wrapping may change what
whitespace characters are stripped (see <a href='#strip'><b>3.4 Whitespace Stripping</b></a>) but
does not affect how the characters are handled by the XSLT processor
thereafter.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>The <code>xml:lang</code> and <code>xml:space</code>
attributes are not treated specially by XSLT. In particular,</p>

<ul>
<li><p>it is the responsibility of the stylesheet author explicitly
to generate any <code>xml:lang</code> or <code>xml:space</code>
attributes that are needed in the result;</p></li>

<li><p>specifying an <code>xml:lang</code> or <code>xml:space</code>
attribute on an element in the XSLT namespace will not cause any
<code>xml:lang</code> or <code>xml:space</code> attributes to appear
in the result.</p></li>
</ul>
</div>

</div>


<div class='div2'>
<h3><a name='id13'></a>7.3 Creating Processing Instructions</h3>



  
  
    
      
    
  
  


<p>The <code>xsl:processing-instruction</code> element is instantiated
to create a processing instruction node.  The content of the
<code>xsl:processing-instruction</code> element is a template for the
string-value of the processing instruction node.  The
<code>xsl:processing-instruction</code> element has a required
<code>name</code> attribute that specifies the name of the processing
instruction node.  The value of the <code>name</code> attribute is
interpreted as an <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute
value template</a>.</p>

<p>For example, this</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:processing-instruction name="xml-stylesheet">href="book.css" type="text/css"&lt;/xsl:processing-instruction></pre></td></tr></table>

<p>would create the processing instruction</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;?xml-stylesheet href="book.css" type="text/css"?></pre></td></tr></table>

<p>It is an error if the string that results from instantiating the
<code>name</code> attribute is not both an <a href='http://www.w3.org/TR/REC-xml-names#NT-NCName'>NCName</a> and a <a href='http://www.w3.org/TR/REC-xml#NT-PITarget'>PITarget</a>.  An XSLT processor may signal
the error; if it does not signal the error, it must recover by not
adding the processing instruction to the result tree.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>This means that <code>xsl:processing-instruction</code>
cannot be used to output an XML declaration.  The
<code>xsl:output</code> element should be used instead (see <a href='#output'><b>16 Output</b></a>).</p></div>

<p>It is an error if instantiating the content of
<code>xsl:processing-instruction</code> creates nodes other than
text nodes.  An XSLT processor may signal the error; if it does not
signal the error, it must recover by ignoring the offending nodes
together with their content.</p>

<p>It is an error if the result of instantiating the content of the
<code>xsl:processing-instruction</code> contains the string
<code>?></code>.  An XSLT processor may signal the error; if it does
not signal the error, it must recover by inserting a space after any
occurrence of <code>?</code> that is followed by a <code>></code>.</p>

</div>

<div class='div2'>
<h3><a name='id14'></a>7.4 Creating Comments</h3>


  
  


<p>The <code>xsl:comment</code> element is instantiated to create a
comment node in the result tree.  The content of the
<code>xsl:comment</code> element is a template for the string-value of
the comment node.</p>

<p>For example, this</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:comment>This file is automatically generated. Do not edit!&lt;/xsl:comment></pre></td></tr></table>

<p>would create the comment</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;!--This file is automatically generated. Do not edit!--></pre></td></tr></table>

<p>It is an error if instantiating the content of
<code>xsl:comment</code> creates nodes other than text nodes.  An
XSLT processor may signal the error; if it does not signal the error,
it must recover by ignoring the offending nodes together with their
content.</p>

<p>It is an error if the result of instantiating the content of the
<code>xsl:comment</code> contains the string <code>--</code> or ends
with <code>-</code>.  An XSLT processor may signal the error; if it
does not signal the error, it must recover by inserting a space after
any occurrence of <code>-</code> that is followed by another
<code>-</code> or that ends the comment.</p>

</div>

<div class='div2'>
<h3><a name='copying'></a>7.5 Copying</h3>


  
  
    
  
  


<p>The <code>xsl:copy</code> element provides an easy way of copying
the current node. Instantiating the <code>xsl:copy</code> element
creates a copy of the current node.  The namespace nodes of the
current node are automatically copied as well, but the attributes and
children of the node are not automatically copied.  The content of the
<code>xsl:copy</code> element is a template for the attributes and
children of the created node; the content is instantiated only for
nodes of types that can have attributes or children (i.e. root
nodes and element nodes).</p>

<p>The <code>xsl:copy</code> element may have a
<code>use-attribute-sets</code> attribute (see <a href='#attribute-sets'><b>7.1.4 Named Attribute Sets</b></a>). This is used only when copying element
nodes.</p>

<p>The root node is treated specially because the root node of the
result tree is created implicitly.  When the current node is the root
node, <code>xsl:copy</code> will not create a root node, but will just
use the content template.</p>

<p>For example, the identity transformation can be written using
<code>xsl:copy</code> as follows:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="@*|node()">
  &lt;xsl:copy>
    &lt;xsl:apply-templates select="@*|node()"/>
  &lt;/xsl:copy>
&lt;/xsl:template></pre></td></tr></table>

<p>When the current node is an attribute, then if it would be an error
to use <code>xsl:attribute</code> to create an attribute with the same
name as the current node, then it is also an error to use
<code>xsl:copy</code> (see <a href='#creating-attributes'><b>7.1.3 Creating Attributes with xsl:attribute</b></a>).</p>

<p>The following example shows how <code>xml:lang</code> attributes
can be easily copied through from source to result. If a stylesheet
defines the following named template:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template name="apply-templates-copy-lang">
 &lt;xsl:for-each select="@xml:lang">
   &lt;xsl:copy/>
 &lt;/xsl:for-each>
 &lt;xsl:apply-templates/>
&lt;/xsl:template></pre></td></tr></table>

<p>then it can simply do</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:call-template name="apply-templates-copy-lang"/></pre></td></tr></table>

<p>instead of</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:apply-templates/></pre></td></tr></table>

<p>when it wants to copy the <code>xml:lang</code> attribute.</p>

</div>

<div class='div2'>
<h3><a name='id15'></a>7.6 Computing Generated Text</h3>

<p>Within a template, the <code>xsl:value-of</code> element can be
used to compute generated text, for example by extracting text from
the source tree or by inserting the value of a variable.  The
<code>xsl:value-of</code> element does this with an <a title='Expression' href='#dt-expression'>expression</a> that is specified as the
value of the <code>select</code> attribute.  Expressions can
also be used inside attribute values of literal result elements by
enclosing the expression in curly braces (<code>{}</code>).</p>

<div class='div3'>
<h4><a name='value-of'></a>7.6.1 Generating Text with <code>xsl:value-of</code></h4>


  
  
    
  
  
    
    
  
  


<p>The <code>xsl:value-of</code> element is instantiated to create a
text node in the result tree.  The required <code>select</code>
attribute is an <a title='Expression' href='#dt-expression'>expression</a>;
this expression is evaluated and the resulting object is converted to
a string as if by a call to the string
function. The string specifies the string-value of the created text
node.  If the string is empty, no text node will be created.  The
created text node will be merged with any adjacent text nodes.</p>

<p>The <code>xsl:copy-of</code> element can be used to copy a node-set
over to the result tree without converting it to a string. See <a href='#copy-of'><b>11.3 Using Values of Variables and Parameters with
xsl:copy-of</b></a>.</p>

<p>For example, the following creates an HTML paragraph from a
<code>person</code> element with <code>given-name</code> and
<code>family-name</code> attributes.  The paragraph will contain the value
of the <code>given-name</code> attribute of the current node followed
by a space and the value of the <code>family-name</code> attribute of the
current node.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="person">
  &lt;p>
   &lt;xsl:value-of select="@given-name"/>
   &lt;xsl:text> &lt;/xsl:text>
   &lt;xsl:value-of select="@family-name"/>
  &lt;/p>
&lt;/xsl:template></pre></td></tr></table>

<p>For another example, the following creates an HTML paragraph from a
<code>person</code> element with <code>given-name</code> and
<code>family-name</code> children elements.  The paragraph will
contain the string-value of the first <code>given-name</code> child
element of the current node followed by a space and the string-value
of the first <code>family-name</code> child element of the current
node.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="person">
  &lt;p>
   &lt;xsl:value-of select="given-name"/>
   &lt;xsl:text> &lt;/xsl:text>
   &lt;xsl:value-of select="family-name"/>
  &lt;/p>
&lt;/xsl:template></pre></td></tr></table>

<p>The following precedes each <code>procedure</code> element with a
paragraph containing the security level of the procedure.  It assumes
that the security level that applies to a procedure is determined by a
<code>security</code> attribute on the procedure element or on an
ancestor element of the procedure. It also assumes that if more than
one such element has a <code>security</code> attribute then the
security level is determined by the element that is closest to the
procedure.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="procedure">
  &lt;fo:block>
    &lt;xsl:value-of select="ancestor-or-self::*[@security][1]/@security"/>
  &lt;/fo:block>
  &lt;xsl:apply-templates/>
&lt;/xsl:template></pre></td></tr></table>

</div>

<div class='div3'>
<h4><a name='attribute-value-templates'></a>7.6.2 Attribute Value Templates</h4>

<p>[<a name='dt-attribute-value-template' title='Attribute Value Template'>Definition</a>: In an attribute value that is interpreted as an
<b>attribute value template</b>, such as an attribute of a
literal result element, an <a title='Expression' href='#dt-expression'>expression</a> can be used by surrounding
the expression with curly braces (<code>{}</code>)].  The
attribute value template is instantiated by replacing the expression
together with surrounding curly braces by the result of evaluating the
expression and converting the resulting object to a string as if by a
call to the string function.  Curly braces are
not recognized in an attribute value in an XSLT stylesheet unless the
attribute is specifically stated to be one that is interpreted as an
attribute value template; in an element syntax summary, the value
of such attributes is surrounded by curly braces.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Not all attributes are interpreted as attribute value
templates.  Attributes whose value is an expression or pattern,
attributes of <a title='Top-level' href='#dt-top-level'>top-level</a> elements
and attributes that refer to named XSLT objects are not interpreted as
attribute value templates. In addition, <code>xmlns</code> attributes
are not interpreted as attribute value templates; it would not be
conformant with the XML Namespaces Recommendation to do
this.</p></div>

<p>The following example creates an <code>img</code> result element
from a <code>photograph</code> element in the source; the value of the
<code>src</code> attribute of the <code>img</code> element is computed
from the value of the <code>image-dir</code> variable and the
string-value of the <code>href</code> child of the
<code>photograph</code> element; the value of the <code>width</code>
attribute of the <code>img</code> element is computed from the value
of the <code>width</code> attribute of the <code>size</code> child of
the <code>photograph</code> element:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:variable name="image-dir">/images&lt;/xsl:variable>

&lt;xsl:template match="photograph">
&lt;img src="{$image-dir}/{href}" width="{size/@width}"/>
&lt;/xsl:template></pre></td></tr></table>

<p>With this source</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;photograph>
  &lt;href>headquarters.jpg&lt;/href>
  &lt;size width="300"/>
&lt;/photograph></pre></td></tr></table>

<p>the result would be</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;img src="/images/headquarters.jpg" width="300"/></pre></td></tr></table>

<p>When an attribute value template is instantiated, a double left or
right curly brace outside an expression will be replaced by a single
curly brace.  It is an error if a right curly brace occurs in an
attribute value template outside an expression without being followed
by a second right curly brace.  A right curly brace inside a <a href='http://www.w3.org/TR/xpath#NT-Literal'>Literal</a> in an expression is not
recognized as terminating the expression.</p>

<p>Curly braces are <em>not</em> recognized recursively inside
expressions.  For example:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;a href="#{id({@ref})/title}"></pre></td></tr></table>

<p>is <em>not</em> allowed.  Instead, use simply:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;a href="#{id(@ref)/title}"></pre></td></tr></table>

</div>

</div>

<div class='div2'>
<h3><a name='number'></a>7.7 Numbering</h3>


  
  
    
    
    
  
  
    
  
  
    
  
  
    
  
  
    
      
    
  
  
    
      
    
  
  
    
      
      
    
  
  
    
      
    
  
  
    
      
    
  
  


<p>The <code>xsl:number</code> element is used to insert a formatted
number into the result tree.  The number to be inserted may be
specified by an expression. The <code>value</code> attribute contains
an <a title='Expression' href='#dt-expression'>expression</a>.  The expression
is evaluated and the resulting object is converted to a number as if
by a call to the number function.  The number is
rounded to an integer and then converted to a string using the
attributes specified in <a href='#convert'><b>7.7.1 Number to String Conversion Attributes</b></a>; in this
context, the value of each of these attributes is
interpreted as an <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute
value template</a>.  After conversion, the resulting string is
inserted in the result tree. For example, the following example
numbers a sorted list:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="items">
  &lt;xsl:for-each select="item">
    &lt;xsl:sort select="."/>
    &lt;p>
      &lt;xsl:number value="position()" format="1. "/>
      &lt;xsl:value-of select="."/>
    &lt;/p>
  &lt;/xsl:for-each>
&lt;/xsl:template></pre></td></tr></table>

<p>If no <code>value</code> attribute is specified, then the
<code>xsl:number</code> element inserts a number based on the position
of the current node in the source tree. The following attributes
control how the current node is to be numbered:</p>

<ul>

<li><p>The <code>level</code> attribute specifies what levels of the
source tree should be considered; it has the values
<code>single</code>, <code>multiple</code> or <code>any</code>. The
default is <code>single</code>.</p></li>

<li><p>The <code>count</code> attribute is a pattern that specifies
what nodes should be counted at those levels.  If <code>count</code>
attribute is not specified, then it defaults to the pattern that
matches any node with the same node type as the current node and, if
the current node has an expanded-name, with the same expanded-name as
the current node.</p></li>

<li><p>The <code>from</code> attribute is a pattern that specifies
where counting starts.</p></li>

</ul>

<p>In addition, the attributes specified in <a href='#convert'><b>7.7.1 Number to String Conversion Attributes</b></a>
are used for number to string conversion, as in the case when the
<code>value</code> attribute is specified.</p>

<p>The <code>xsl:number</code> element first constructs a list of
positive integers using the <code>level</code>, <code>count</code> and
<code>from</code> attributes:</p>

<ul>

<li><p>When <code>level="single"</code>, it goes up to the first
node in the ancestor-or-self axis that matches
the <code>count</code> pattern, and constructs a list of length one
containing one plus the number of preceding siblings of that ancestor
that match the <code>count</code> pattern. If there is no such
ancestor, it constructs an empty list.  If the <code>from</code>
attribute is specified, then the only ancestors that are searched are
those that are descendants of the nearest ancestor that matches the
<code>from</code> pattern. Preceding siblings has the same meaning
here as with the <code>preceding-sibling</code> axis.</p></li>

<li><p>When <code>level="multiple"</code>, it constructs a list of all
ancestors of the current node in document order followed by the
element itself; it then selects from the list those nodes that match
the <code>count</code> pattern; it then maps each node in the list to
one plus the number of preceding siblings of that node that match the
<code>count</code> pattern.  If the <code>from</code> attribute is
specified, then the only ancestors that are searched are those that
are descendants of the nearest ancestor that matches the
<code>from</code> pattern. Preceding siblings has the same meaning
here as with the <code>preceding-sibling</code> axis.</p></li>

<li><p>When <code>level="any"</code>, it constructs a list of length
one containing the number of nodes that match the <code>count</code>
pattern and belong to the set containing the current node and all
nodes at any level of the document that are before the current node in
document order, excluding any namespace and attribute nodes (in other
words the union of the members of the <code>preceding</code> and
<code>ancestor-or-self</code> axes). If the <code>from</code>
attribute is specified, then only nodes after the first node before
the current node that match the <code>from</code> pattern are
considered.</p></li>

</ul>

<p>The list of numbers is then converted into a string using the
attributes specified in <a href='#convert'><b>7.7.1 Number to String Conversion Attributes</b></a>; in this
context, the value of each of these attributes is
interpreted as an <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute
value template</a>.  After conversion, the resulting string is
inserted in the result tree.</p>

<p>The following would number the items in an ordered list:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="ol/item">
  &lt;fo:block>
    &lt;xsl:number/>&lt;xsl:text>. &lt;/xsl:text>&lt;xsl:apply-templates/>
  &lt;/fo:block>
&lt;xsl:template></pre></td></tr></table>

<p>The following two rules would number <code>title</code> elements.
This is intended for a document that contains a sequence of chapters
followed by a sequence of appendices, where both chapters and
appendices contain sections, which in turn contain subsections.
Chapters are numbered 1, 2, 3; appendices are numbered A, B, C;
sections in chapters are numbered 1.1, 1.2, 1.3; sections in
appendices are numbered A.1, A.2, A.3.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="title">
  &lt;fo:block>
     &lt;xsl:number level="multiple"
                 count="chapter|section|subsection"
                 format="1.1 "/>
     &lt;xsl:apply-templates/>
  &lt;/fo:block>
&lt;/xsl:template>

&lt;xsl:template match="appendix//title" priority="1">
  &lt;fo:block>
     &lt;xsl:number level="multiple"
                 count="appendix|section|subsection"
                 format="A.1 "/>
     &lt;xsl:apply-templates/>
  &lt;/fo:block>
&lt;/xsl:template></pre></td></tr></table>

<p>The following example numbers notes sequentially within a
chapter:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="note">
  &lt;fo:block>
     &lt;xsl:number level="any" from="chapter" format="(1) "/>
     &lt;xsl:apply-templates/>
  &lt;/fo:block>
&lt;/xsl:template></pre></td></tr></table>

<p>The following example would number <code>H4</code> elements in HTML
with a three-part label:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="H4">
 &lt;fo:block>
   &lt;xsl:number level="any" from="H1" count="H2"/>
   &lt;xsl:text>.&lt;/xsl:text>
   &lt;xsl:number level="any" from="H2" count="H3"/>
   &lt;xsl:text>.&lt;/xsl:text>
   &lt;xsl:number level="any" from="H3" count="H4"/>
   &lt;xsl:text> &lt;/xsl:text>
   &lt;xsl:apply-templates/>
 &lt;/fo:block>
&lt;/xsl:template></pre></td></tr></table>

<div class='div3'>
<h4><a name='convert'></a>7.7.1 Number to String Conversion Attributes</h4>

<p>The following attributes are used to control conversion of a list
of numbers into a string. The numbers are integers greater than
0. The attributes are all optional.</p>

<p>The main attribute is <code>format</code>.  The default value for
the <code>format</code> attribute is <code>1</code>.  The
<code>format</code> attribute is split into a sequence of tokens where
each token is a maximal sequence of alphanumeric characters or a
maximal sequence of non-alphanumeric characters.  Alphanumeric means
any character that has a Unicode category of Nd, Nl, No, Lu, Ll, Lt,
Lm or Lo.  The alphanumeric tokens (format tokens) specify the format
to be used for each number in the list.  If the first token is a
non-alphanumeric token, then the constructed string will start with
that token; if the last token is non-alphanumeric token, then the
constructed string will end with that token.  Non-alphanumeric tokens
that occur between two format tokens are separator tokens that are
used to join numbers in the list.  The <var>n</var>th format token
will be used to format the <var>n</var>th number in the list.  If
there are more numbers than format tokens, then the last format token
will be used to format remaining numbers.  If there are no format
tokens, then a format token of <code>1</code> is used to format all
numbers.  The format token specifies the string to be used to
represent the number 1.  Each number after the first will be separated
from the preceding number by the separator token preceding the format
token used to format that number, or, if there are no separator
tokens, then by <code>.</code> (a period character).</p>

<p>Format tokens are a superset of the allowed values for the
<code>type</code> attribute for the <code>OL</code> element in HTML
4.0 and are interpreted as follows:</p>

<ul>

<li><p>Any token where the last character has a decimal digit value
of 1 (as specified in the Unicode character property database),
and the Unicode value of preceding characters is one less than the
Unicode value of the last character generates a decimal
representation of the number where each number is at least as long as
the format token.  Thus, a format token <code>1</code> generates the
sequence <code>1 2 ... 10 11 12 ...</code>, and a format token
<code>01</code> generates the sequence <code>01 02 ... 09 10 11 12
... 99 100 101</code>.</p></li>

<li><p>A format token <code>A</code> generates the sequence <code>A
B C ... Z AA AB AC...</code>.</p></li>

<li><p>A format token <code>a</code> generates the sequence <code>a
b c ... z aa ab ac...</code>.</p></li>

<li><p>A format token <code>i</code> generates the sequence <code>i
ii iii iv v vi vii viii ix x ...</code>.</p></li>

<li><p>A format token <code>I</code> generates the sequence <code>I
II III IV V VI VII VIII IX X ...</code>.</p></li>

<li><p>Any other format token indicates a numbering sequence that
starts with that token.  If an implementation does not support a
numbering sequence that starts with that token, it must use a format
token of <code>1</code>.</p></li>

</ul>

<p>When numbering with an alphabetic sequence, the <code>lang</code>
attribute specifies which language's alphabet is to be used; it has
the same range of values as <code>xml:lang</code> <a href='#XML'>[XML]</a>;
if no <code>lang</code> value is specified, the language should be
determined from the system environment.  Implementers should document
for which languages they support numbering.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Implementers should not make any assumptions about how
numbering works in particular languages and should properly research
the languages that they wish to support.  The numbering conventions of
many languages are very different from English.</p></div>

<p>The <code>letter-value</code> attribute disambiguates between
numbering sequences that use letters.  In many languages there are two
commonly used numbering sequences that use letters.  One numbering
sequence assigns numeric values to letters in alphabetic sequence, and
the other assigns numeric values to each letter in some other manner
traditional in that language.  In English, these would correspond to
the numbering sequences specified by the format tokens <code>a</code>
and <code>i</code>.  In some languages, the first member of each
sequence is the same, and so the format token alone would be
ambiguous.  A value of <code>alphabetic</code> specifies the
alphabetic sequence; a value of <code>traditional</code> specifies the
other sequence.  If the <code>letter-value</code> attribute is not
specified, then it is implementation-dependent how any ambiguity is
resolved.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>It is possible for two conforming XSLT processors not to
convert a number to exactly the same string.  Some XSLT processors may not
support some languages.  Furthermore, there may be variations possible
in the way conversions are performed for any particular language that
are not specifiable by the attributes on <code>xsl:number</code>.
Future versions of XSLT may provide additional attributes to provide
control over these variations.  Implementations may also use
implementation-specific namespaced attributes on
<code>xsl:number</code> for this.</p></div>

<p>The <code>grouping-separator</code> attribute gives the separator
used as a grouping (e.g. thousands) separator in decimal numbering
sequences, and the optional <code>grouping-size</code> specifies the
size (normally 3) of the grouping.  For example,
<code>grouping-separator=","</code> and <code>grouping-size="3"</code>
would produce numbers of the form <code>1,000,000</code>.  If only one
of the <code>grouping-separator</code> and <code>grouping-size</code>
attributes is specified, then it is ignored.</p>

<p>Here are some examples of conversion specifications:</p>

<ul>

<li><p><code>format="&amp;#x30A2;"</code> specifies Katakana
numbering</p></li>

<li><p><code>format="&amp;#x30A4;"</code> specifies Katakana
numbering in the "iroha" order</p></li>

<li><p><code>format="&amp;#x0E51;"</code> specifies numbering with
Thai digits</p></li>

<li><p><code>format="&amp;#x05D0;" letter-value="traditional"</code>
specifies "traditional" Hebrew numbering</p></li>

<li><p><code>format="&amp;#x10D0;" letter-value="traditional"</code>
specifies Georgian numbering</p></li>

<li><p><code>format="&amp;#x03B1;" letter-value="traditional"</code>
specifies "classical" Greek numbering</p></li>

<li><p><code>format="&amp;#x0430;" letter-value="traditional"</code>
specifies Old Slavic numbering</p></li>

</ul>

</div>
</div>
</div>

<div class='div1'>

<h2><a name='for-each'></a>8 Repetition</h2>


  
  
    
  
  
    
    
  


<p>When the result has a known regular structure, it is useful to be
able to specify directly the template for selected nodes.  The
<code>xsl:for-each</code> instruction contains a template, which is
instantiated for each node selected by the <a title='Expression' href='#dt-expression'>expression</a> specified by the
<code>select</code> attribute. The <code>select</code> attribute is
required.  The expression must evaluate to a node-set.  The template
is instantiated with the selected node as the <a title='Current Node' href='#dt-current-node'>current node</a>, and with a list of all
of the selected nodes as the <a title='Current Node List' href='#dt-current-node-list'>current node list</a>.  The nodes are
processed in document order, unless a sorting specification is present
(see <a href='#sorting'><b>10 Sorting</b></a>).</p>

<p>For example, given an XML document with this structure</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;customers>
  &lt;customer>
    &lt;name>...&lt;/name>
    &lt;order>...&lt;/order>
    &lt;order>...&lt;/order>
  &lt;/customer>
  &lt;customer>
    &lt;name>...&lt;/name>
    &lt;order>...&lt;/order>
    &lt;order>...&lt;/order>
  &lt;/customer>
&lt;/customers></pre></td></tr></table>

<p>the following would create an HTML document containing a table with
a row for each <code>customer</code> element</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="/">
  &lt;html>
    &lt;head>
      &lt;title>Customers&lt;/title>
    &lt;/head>
    &lt;body>
      &lt;table>
	&lt;tbody>
	  &lt;xsl:for-each select="customers/customer">
	    &lt;tr>
	      &lt;th>
		&lt;xsl:apply-templates select="name"/>
	      &lt;/th>
	      &lt;xsl:for-each select="order">
		&lt;td>
		  &lt;xsl:apply-templates/>
		&lt;/td>
	      &lt;/xsl:for-each>
	    &lt;/tr>
	  &lt;/xsl:for-each>
	&lt;/tbody>
      &lt;/table>
    &lt;/body>
  &lt;/html>
&lt;/xsl:template></pre></td></tr></table>

</div>

<div class='div1'>
<h2><a name='id16'></a>9 Conditional Processing</h2>

<p>There are two instructions in XSLT that support conditional
processing in a template: <code>xsl:if</code> and
<code>xsl:choose</code>. The <code>xsl:if</code> instruction provides
simple if-then conditionality; the <code>xsl:choose</code> instruction
supports selection of one choice when there are several
possibilities.</p>

<div class='div2'>
<h3><a name='id17'></a>9.1 Conditional Processing with <code>xsl:if</code></h3>


  
  
    
  
  


<p>The <code>xsl:if</code> element has a <code>test</code> attribute,
which specifies an <a title='Expression' href='#dt-expression'>expression</a>.
The content is a template.  The expression is evaluated and the
resulting object is converted to a boolean as if by a call to the
boolean function.  If the result is true, then
the content template is instantiated; otherwise, nothing is created.
In the following example, the names in a group of names are formatted
as a comma separated list:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="namelist/name">
  &lt;xsl:apply-templates/>
  &lt;xsl:if test="not(position()=last())">, &lt;/xsl:if>
&lt;/xsl:template></pre></td></tr></table>

<p>The following colors every other table row yellow:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="item">
  &lt;tr>
    &lt;xsl:if test="position() mod 2 = 0">
       &lt;xsl:attribute name="bgcolor">yellow&lt;/xsl:attribute>
    &lt;/xsl:if>
    &lt;xsl:apply-templates/>
  &lt;/tr>
&lt;/xsl:template></pre></td></tr></table>

</div>


<div class='div2'>
<h3><a name='id18'></a>9.2 Conditional Processing with <code>xsl:choose</code></h3>


  
  
    
    
  



  
    
  
  



  


<p>The <code>xsl:choose</code> element selects one among a number of
possible alternatives. It consists of a sequence of
<code>xsl:when</code> elements followed by an optional
<code>xsl:otherwise</code> element.  Each <code>xsl:when</code>
element has a single attribute, <code>test</code>, which specifies an
<a title='Expression' href='#dt-expression'>expression</a>. The content of the
<code>xsl:when</code> and <code>xsl:otherwise</code> elements is a
template.  When an <code>xsl:choose</code> element is processed, each
of the <code>xsl:when</code> elements is tested in turn, by evaluating
the expression and converting the resulting object to a boolean as if
by a call to the boolean function.  The content
of the first, and only the first, <code>xsl:when</code> element whose
test is true is instantiated.  If no <code>xsl:when</code> is true,
the content of the <code>xsl:otherwise</code> element is
instantiated. If no <code>xsl:when</code> element is true, and no
<code>xsl:otherwise</code> element is present, nothing is created.</p>

<p>The following example enumerates items in an ordered list using
arabic numerals, letters, or roman numerals depending on the depth to
which the ordered lists are nested.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="orderedlist/listitem">
  &lt;fo:list-item indent-start='2pi'>
    &lt;fo:list-item-label>
      &lt;xsl:variable name="level"
                    select="count(ancestor::orderedlist) mod 3"/>
      &lt;xsl:choose>
        &lt;xsl:when test='$level=1'>
          &lt;xsl:number format="i"/>
        &lt;/xsl:when>
        &lt;xsl:when test='$level=2'>
          &lt;xsl:number format="a"/>
        &lt;/xsl:when>
        &lt;xsl:otherwise>
          &lt;xsl:number format="1"/>
        &lt;/xsl:otherwise>
      &lt;/xsl:choose>
      &lt;xsl:text>. &lt;/xsl:text>
    &lt;/fo:list-item-label>
    &lt;fo:list-item-body>
      &lt;xsl:apply-templates/>
    &lt;/fo:list-item-body>
  &lt;/fo:list-item>
&lt;/xsl:template></pre></td></tr></table>

</div>
</div>

<div class='div1'>
<h2><a name='sorting'></a>10 Sorting</h2>


  
    
  
  
    
      
    
  
  
    
       
       
       
    
  
  
    
       
       
    
  
  
    
       
       
    
  
  


<p>Sorting is specified by adding <code>xsl:sort</code> elements as
children of an <code>xsl:apply-templates</code> or
<code>xsl:for-each</code> element.  The first <code>xsl:sort</code>
child specifies the primary sort key, the second <code>xsl:sort</code>
child specifies the secondary sort key and so on.  When an
<code>xsl:apply-templates</code> or <code>xsl:for-each</code> element
has one or more <code>xsl:sort</code> children, then instead of
processing the selected nodes in document order, it sorts the nodes
according to the specified sort keys and then processes them in sorted
order.  When used in <code>xsl:for-each</code>, <code>xsl:sort</code>
elements must occur first.  When a template is instantiated by
<code>xsl:apply-templates</code> and <code>xsl:for-each</code>, the
<a title='Current Node List' href='#dt-current-node-list'>current node list</a> list
consists of the complete list of nodes being processed in sorted
order.</p>

<p><code>xsl:sort</code> has a <code>select</code> attribute whose
value is an <a title='Expression' href='#dt-expression'>expression</a>. For
each node to be processed, the expression is evaluated with that node
as the current node and with the complete list of nodes being
processed in unsorted order as the current node list.
The resulting object is converted to a string as
if by a call to the string function; this string
is used as the sort key for that node. The default value of the
<code>select</code> attribute is <code>.</code>, which will cause the
string-value of the current node to be used as the sort key.</p>

<p>This string serves as a sort key for the node.  The following
optional attributes on <code>xsl:sort</code> control how the list of
sort keys are sorted; the values of all of these attributes are
interpreted as <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute
value templates</a>.</p>

<ul>

<li><p><code>order</code> specifies whether the strings should be
sorted in ascending or descending order; <code>ascending</code>
specifies ascending order; <code>descending</code> specifies
descending order; the default is <code>ascending</code></p></li>

<li><p><code>lang</code> specifies the language of the sort keys; it
has the same range of values as <code>xml:lang</code> <a href='#XML'>[XML]</a>; if no <code>lang</code> value is specified, the language
should be determined from the system environment</p></li>

<li><p><code>data-type</code> specifies the data type of the
strings; the following values are allowed:</p>

<ul>

<li><p><code>text</code> specifies that the sort keys should be
sorted lexicographically in the culturally correct manner for the
language specified by <code>lang</code></p></li>

<li><p><code>number</code> specifies that the sort keys should be
converted to numbers and then sorted according to the numeric value;
the sort key is converted to a number as if by a call to the
number function; the <code>lang</code>
attribute is ignored</p></li>

<li><p>a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> with a prefix
is expanded into an <a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a> as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>; the expanded-name identifies the data-type;
the behavior in this case is not specified by this document</p></li>

</ul>

<p>The default value is <code>text</code>.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>The XSL Working Group plans that future versions of XSLT will
leverage XML Schemas to define further values for this
attribute.</p></div>

</li>

<li><p><code>case-order</code> has the value
<code>upper-first</code> or <code>lower-first</code>; this applies
when <code>data-type="text"</code>, and specifies that upper-case
letters should sort before lower-case letters or vice-versa
respectively. For example, if <code>lang="en"</code>, then <code>A a B
b</code> are sorted with <code>case-order="upper-first"</code> and
<code>a A b B</code> are sorted with
<code>case-order="lower-first"</code>. The default value is language
dependent.</p></li>

</ul>

<div class='note'><p class='prefix'><b>Note:</b></p><p>It is possible for two conforming XSLT processors not to sort
exactly the same.  Some XSLT processors may not support some
languages.  Furthermore, there may be variations possible in the
sorting of any particular language that are not specified by the
attributes on <code>xsl:sort</code>, for example, whether Hiragana or
Katakana is sorted first in Japanese.  Future versions of XSLT may
provide additional attributes to provide control over these
variations.  Implementations may also use implementation-specific
namespaced attributes on <code>xsl:sort</code> for this.</p></div>

<div class='note'><p class='prefix'><b>Note:</b></p><p>It is recommended that implementers consult <a href='#UNICODE-TR10'>[UNICODE TR10]</a> for information on internationalized
sorting.</p></div>

<p>The sort must be stable: in the sorted list of nodes, any sub list
that has sort keys that all compare equal must be in document
order.</p>

<p>For example, suppose an employee database has the form</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;employees>
  &lt;employee>
    &lt;name>
      &lt;given>James&lt;/given>
      &lt;family>Clark&lt;/family>
    &lt;/name>
    ...
  &lt;/employee>
&lt;/employees>
</pre></td></tr></table>
  
<p>Then a list of employees sorted by name could be generated
using:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template match="employees">
  &lt;ul>
    &lt;xsl:apply-templates select="employee">
      &lt;xsl:sort select="name/family"/>
      &lt;xsl:sort select="name/given"/>
    &lt;/xsl:apply-templates>
  &lt;/ul>
&lt;/xsl:template>

&lt;xsl:template match="employee">
  &lt;li>
    &lt;xsl:value-of select="name/given"/>
    &lt;xsl:text> &lt;/xsl:text>
    &lt;xsl:value-of select="name/family"/>
  &lt;/li>
&lt;/xsl:template></pre></td></tr></table>

</div>

<div class='div1'>
<h2><a name='variables'></a>11 Variables and Parameters</h2>


  
  
  
    
  
  
    
  
  



  
  
    
  
  
    
  
  


<p>A variable is a name that may be bound to a value.  The value to
which a variable is bound (the <b>value</b> of the variable) can
be an object of any of the types that can be returned by expressions.
There are two elements that can be used to bind variables:
<code>xsl:variable</code> and <code>xsl:param</code>. The difference
is that the value specified on the <code>xsl:param</code> variable is
only a default value for the binding; when the template or stylesheet
within which the <code>xsl:param</code> element occurs is invoked,
parameters may be passed that are used in place of the default
values.</p>

<p>Both <code>xsl:variable</code> and <code>xsl:param</code> have a
required <code>name</code> attribute, which specifies the name of the
variable.  The value of the <code>name</code> attribute is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>.</p>

<p>For any use of these variable-binding elements, there is a region
of the stylesheet tree within which the binding is visible; within
this region, any binding of the variable that was visible on the
variable-binding element itself is hidden.  Thus, only the innermost
binding of a variable is visible.  The set of variable bindings in
scope for an expression consists of those bindings that are visible at
the point in the stylesheet where the expression occurs.</p>

<div class='div2'>
<h3><a name='id19'></a>11.1 Result Tree Fragments</h3>

<p>Variables introduce an additional data-type into the expression
language.  [<a name='dt-result-tree-fragment' title='Result Tree Fragment'>Definition</a>: This additional data type is called <b>result tree
fragment</b>.  A variable may be bound to a result tree fragment
instead of one of the four basic XPath data-types (string, number,
boolean, node-set).  A result tree fragment represents a fragment of
the result tree. A result tree fragment is treated equivalently to a
node-set that contains just a single root node.] However, the
operations permitted on a result tree fragment are a subset of those
permitted on a node-set.  An operation is permitted on a result tree
fragment only if that operation would be permitted on a string (the
operation on the string may involve first converting the string to a
number or boolean). In particular, it is not permitted to use the
<code>/</code>, <code>//</code>, and <code>[]</code> operators on
result tree fragments.  When a permitted operation is performed on a
result tree fragment, it is performed exactly as it would be on the
equivalent node-set.</p>

<p>When a result tree fragment is copied into the result tree (see
<a href='#copy-of'><b>11.3 Using Values of Variables and Parameters with
xsl:copy-of</b></a>), then all the nodes that are children of the
root node in the equivalent node-set are added in sequence to the
result tree.</p>

<p>Expressions can only return values of type result tree fragment by
referencing variables of type result tree fragment or calling
extension functions that return a result tree fragment or getting a
system property whose value is a result tree fragment.</p>

</div>

<div class='div2'>
<h3><a name='variable-values'></a>11.2 Values of Variables and Parameters</h3>

<p>A variable-binding element can specify the value of the variable in
three alternative ways.</p>

<ul>

<li><p>If the variable-binding element has a <code>select</code>
attribute, then the value of the attribute must be an <a title='Expression' href='#dt-expression'>expression</a> and the value of the variable
is the object that results from evaluating the expression.  In this
case, the content must be empty.</p></li>

<li>

<p>If the variable-binding element does not have a <code>select</code>
attribute and has non-empty content (i.e. the variable-binding element
has one or more child nodes), then the content of the
variable-binding element specifies the value. The content of the
variable-binding element is a template, which is instantiated to give
the value of the variable. The value is a result tree fragment
equivalent to a node-set containing just a single root node having as
children the sequence of nodes produced by instantiating the template.
The base URI of the nodes in the result tree fragment is the base URI
of the variable-binding element.</p>

<p>It is an error if a member of the sequence of nodes created by
instantiating the template is an attribute node or a namespace node,
since a root node cannot have an attribute node or a namespace node as
a child. An XSLT processor may signal the error; if it does not signal
the error, it must recover by not adding the attribute node or
namespace node.</p>

</li>

<li>

<p>If the variable-binding element has empty content and does not have
a <code>select</code> attribute, then the value of the variable is an
empty string. Thus</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:variable name="x"/></pre></td></tr></table>

<p>is equivalent to</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:variable name="x" select="''"/></pre></td></tr></table>

</li>

</ul>

<div class='note'><p class='prefix'><b>Note:</b></p><p>When a variable is used to select nodes by position, be careful
not to do:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:variable name="n">2&lt;/xsl:variable>
...
&lt;xsl:value-of select="item[$n]"/></pre></td></tr></table>

<p>This will output the value of the first item element, because the
variable <code>n</code> will be bound to a result tree fragment, not a
number. Instead, do either</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:variable name="n" select="2"/>
...
&lt;xsl:value-of select="item[$n]"/></pre></td></tr></table>

<p>or</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:variable name="n">2&lt;/xsl:variable>
...
&lt;xsl:value-of select="item[position()=$n]"/></pre></td></tr></table>
</div>

<div class='note'><p class='prefix'><b>Note:</b></p><p>One convenient way to specify the empty node-set as the default
value of a parameter is:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:param name="x" select="/.."/></pre></td></tr></table>
</div>

</div>

<div class='div2'>
<h3><a name='copy-of'></a>11.3 Using Values of Variables and Parameters with
<code>xsl:copy-of</code></h3>


  
  
    
  
  


<p>The <code>xsl:copy-of</code> element can be used to insert a result
tree fragment into the result tree, without first converting it to a
string as <code>xsl:value-of</code> does (see <a href='#value-of'><b>7.6.1 Generating Text with xsl:value-of</b></a>).  The required <code>select</code> attribute
contains an <a title='Expression' href='#dt-expression'>expression</a>.  When
the result of evaluating the expression is a result tree fragment, the
complete fragment is copied into the result tree.  When the result is
a node-set, all the nodes in the set are copied in document order into
the result tree; copying an element node copies the attribute nodes,
namespace nodes and children of the element node as well as the
element node itself; a root node is copied by copying its children.
When the result is neither a node-set nor a result tree fragment, the
result is converted to a string and then inserted into the result
tree, as with <code>xsl:value-of</code>.</p>

</div>

<div class='div2'>
<h3><a name='top-level-variables'></a>11.4 Top-level Variables and Parameters</h3>

<p>Both <code>xsl:variable</code> and <code>xsl:param</code> are
allowed as <a title='Top-level' href='#dt-top-level'>top-level</a> elements.
A top-level variable-binding element declares a global variable that
is visible everywhere.  A top-level <code>xsl:param</code> element
declares a parameter to the stylesheet; XSLT does not define the
mechanism by which parameters are passed to the stylesheet.  It is an
error if a stylesheet contains more than one binding of a top-level
variable with the same name and same <a title='Import Precedence' href='#dt-import-precedence'>import precedence</a>. At the
top-level, the expression or template specifying the variable value is
evaluated with the same context as that used to process the root node
of the source document: the current node is the root node of the
source document and the current node list is a list containing just
the root node of the source document.  If the template or expression
specifying the value of a global variable <var>x</var> references a
global variable <var>y</var>, then the value for <var>y</var> must
be computed before the value of <var>x</var>.  It is an error if it
is impossible to do this for all global variable definitions; in other
words, it is an error if the definitions are circular.</p>

<p>This example declares a global variable <code>para-font-size</code>,
which it references in an attribute value template.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:variable name="para-font-size">12pt&lt;/xsl:variable>

&lt;xsl:template match="para">
 &lt;fo:block font-size="{$para-font-size}">
   &lt;xsl:apply-templates/>
 &lt;/fo:block>
&lt;/xsl:template>
</pre></td></tr></table>

</div>

<div class='div2'>
<h3><a name='local-variables'></a>11.5 Variables and Parameters within Templates</h3>

<p>As well as being allowed at the top-level, both
<code>xsl:variable</code> and <code>xsl:param</code> are also
allowed in templates.  <code>xsl:variable</code> is allowed anywhere
within a template that an instruction is allowed.  In this case, the
binding is visible for all following siblings and their descendants.
Note that the binding is not visible for the <code>xsl:variable</code>
element itself.  <code>xsl:param</code> is allowed as a child
at the beginning of an <code>xsl:template</code> element.  In this
context, the binding is visible for all following siblings and their
descendants.  Note that the binding is not visible for the
<code>xsl:param</code> element itself.</p>

<p>[<a name='dt-shadows' title='Shadows'>Definition</a>: A binding
<b>shadows</b> another binding if the binding occurs at a point
where the other binding is visible, and the bindings have the same
name.] It is an error if a binding established by an
<code>xsl:variable</code> or <code>xsl:param</code> element within a
template <a title='Shadows' href='#dt-shadows'>shadows</a> another binding
established by an <code>xsl:variable</code> or <code>xsl:param</code>
element also within the template.  It is not an error if a binding
established by an <code>xsl:variable</code> or <code>xsl:param</code>
element in a template <a title='Shadows' href='#dt-shadows'>shadows</a>
another binding established by an <code>xsl:variable</code> or
<code>xsl:param</code> <a title='Top-level' href='#dt-top-level'>top-level</a>
element.  Thus, the following is an error:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template name="foo">
&lt;xsl:param name="x" select="1"/>
&lt;xsl:variable name="x" select="2"/>
&lt;/xsl:template></pre></td></tr></table>

<p>However, the following is allowed:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:param name="x" select="1"/>
&lt;xsl:template name="foo">
&lt;xsl:variable name="x" select="2"/>
&lt;/xsl:template></pre></td></tr></table>

<div class='note'><p class='prefix'><b>Note:</b></p><p>The nearest equivalent in Java to an <code>xsl:variable</code>
element in a template is a final local variable declaration with an
initializer.  For example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:variable name="x" select="'value'"/></pre></td></tr></table>

<p>has similar semantics to</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>final Object x = "value";</pre></td></tr></table>

<p>XSLT does not provide an equivalent to the Java assignment operator</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>x = "value";</pre></td></tr></table>

<p>because this would make it harder to create an implementation that
processes a document other than in a batch-like way, starting at the
beginning and continuing through to the end.</p></div>

</div>

<div class='div2'>
<h3><a name='id20'></a>11.6 Passing Parameters to Templates</h3>


  
    
  
  
    
  
  


<p>Parameters are passed to templates using the
<code>xsl:with-param</code> element.  The required <code>name</code>
attribute specifies the name of the parameter (the variable the value
of whose binding is to be replaced).  The value of the
<code>name</code> attribute is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>.  <code>xsl:with-param</code> is allowed
within both <code>xsl:call-template</code> and
<code>xsl:apply-templates</code>.  The value of the parameter is
specified in the same way as for <code>xsl:variable</code> and
<code>xsl:param</code>.  The current node and current node list used
for computing the value specified by <code>xsl:with-param</code>
element is the same as that used for the
<code>xsl:apply-templates</code> or <code>xsl:call-template</code>
element within which it occurs.  It is not an error to pass a
parameter <var>x</var> to a template that does not have an
<code>xsl:param</code> element for <var>x</var>; the parameter is
simply ignored.</p>

<p>This example defines a named template for a
<code>numbered-block</code> with an argument to control the format of
the number.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:template name="numbered-block">
  &lt;xsl:param name="format">1. &lt;/xsl:param>
  &lt;fo:block>
    &lt;xsl:number format="{$format}"/>
    &lt;xsl:apply-templates/>
  &lt;/fo:block>
&lt;/xsl:template>

&lt;xsl:template match="ol//ol/li">
  &lt;xsl:call-template name="numbered-block">
    &lt;xsl:with-param name="format">a. &lt;/xsl:with-param>
  &lt;/xsl:call-template>
&lt;/xsl:template></pre></td></tr></table>

</div>

</div>

<div class='div1'>
<h2><a name='add-func'></a>12 Additional Functions</h2>

<p>This section describes XSLT-specific additions to the core XPath
function library.  Some of these additional functions also make use of
information specified by <a title='Top-level' href='#dt-top-level'>top-level</a>
elements in the stylesheet; this section also describes these
elements.</p>

<div class='div2'>

<h3><a name='document'></a>12.1 Multiple Source Documents</h3>

<p><em>node-set</em> <b>document</b>(<var>object</var>
, <var>node-set</var>?)</p>

<p>The <code>document</code> function allows
access to XML documents other than the main source document.</p>

<p>When the <code>document</code> function has exactly one
argument and the argument is a node-set, then the result is the union,
for each node in the argument node-set, of the result of calling the
<code>document</code> function with the first argument being
the <a href='http://www.w3.org/TR/xpath#dt-string-value'>string-value</a>
of the node, and the second argument being a node-set with the node as
its only member. When the <code>document</code> function has
two arguments and the first argument is a node-set, then the result is
the union, for each node in the argument node-set, of the result of
calling the <code>document</code> function with the first
argument being the <a href='http://www.w3.org/TR/xpath#dt-string-value'>string-value</a> of the node,
and with the second argument being the second argument passed to the
<code>document</code> function.</p>

<p>When the first argument to the <code>document</code>
function is not a node-set, the first argument is converted to a
string as if by a call to the string function.
This string is treated as a URI reference; the resource identified by
the URI is retrieved. The data resulting from the retrieval action is
parsed as an XML document and a tree is constructed in accordance with
the data model (see <a href='#data-model'><b>3 Data Model</b></a>).  If there is an
error retrieving the resource, then the XSLT processor may signal an
error; if it does not signal an error, it must recover by returning an
empty node-set.  One possible kind of retrieval error is that the XSLT
processor does not support the URI scheme used by the URI.  An XSLT
processor is not required to support any particular URI schemes.  The
documentation for an XSLT processor should specify which URI schemes
the XSLT processor supports.</p>

<p>If the URI reference does not contain a fragment identifier, then a
node-set containing just the root node of the document is returned.
If the URI reference does contain a fragment identifier, the function
returns a node-set containing the nodes in the tree identified by the
fragment identifier of the URI reference. The semantics of the
fragment identifier is dependent on the media type of the result of
retrieving the URI.  If there is an error in processing the fragment
identifier, the XSLT processor may signal the error; if it does not
signal the error, it must recover by returning an empty node-set.
Possible errors include:</p>

<ul>

<li><p>The fragment identifier identifies something that cannot be
represented by an XSLT node-set (such as a range of characters within
a text node).</p></li>

<li><p>The XSLT processor does not support fragment identifiers for
the media-type of the retrieval result.  An XSLT processor is not
required to support any particular media types.  The documentation for
an XSLT processor should specify for which media types the XSLT
processor supports fragment identifiers.</p></li>

</ul>

<p>The data resulting from the retrieval action is parsed as an XML
document regardless of the media type of the retrieval result; if the
top-level media type is <code>text</code>, then it is parsed in the
same way as if the media type were <code>text/xml</code>; otherwise,
it is parsed in the same way as if the media type were
<code>application/xml</code>.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Since there is no top-level <code>xml</code> media type, data
with a media type other than <code>text/xml</code> or
<code>application/xml</code> may in fact be XML.</p></div>

<p>The URI reference may be relative. The base URI (see <a href='#base-uri'><b>3.2 Base URI</b></a>) of the node in the second argument node-set that is
first in document order is used as the base URI for resolving the
relative URI into an absolute URI.  If the second argument is omitted,
then it defaults to the node in the stylesheet that contains the
expression that includes the call to the <code>document</code>
function.  Note that a zero-length URI reference is a reference to the
document relative to which the URI reference is being resolved; thus
<code>document("")</code> refers to the root node of the stylesheet;
the tree representation of the stylesheet is exactly the same as if
the XML document containing the stylesheet was the initial source
document.</p>

<p>Two documents are treated as the same document if they are
identified by the same URI. The URI used for the comparison is the
absolute URI into which any relative URI was resolved and does not
include any fragment identifier.  One root node is treated as the same
node as another root node if the two nodes are from the same document.
Thus, the following expression will always be true:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>generate-id(document("foo.xml"))=generate-id(document("foo.xml"))</pre></td></tr></table>

<p>The <code>document</code> function gives rise to the
possibility that a node-set may contain nodes from more than one
document.  With such a node-set, the relative document order of two
nodes in the same document is the normal <a href='http://www.w3.org/TR/xpath#dt-document-order'>document order</a> defined by
XPath <a href='#XPATH'>[XPath]</a>.  The relative document order of two nodes
in different documents is determined by an implementation-dependent
ordering of the documents containing the two nodes.  There are no
constraints on how the implementation orders documents other than that
it must do so consistently: an implementation must always use the same
order for the same set of documents.</p>

</div>

<div class='div2'>
<h3><a name='key'></a>12.2 Keys</h3>

<p>Keys provide a way to work with documents that contain an implicit
cross-reference structure.  The <code>ID</code>, <code>IDREF</code>
and <code>IDREFS</code> attribute types in XML provide a mechanism to
allow XML documents to make their cross-reference explicit.  XSLT
supports this through the XPath id function.
However, this mechanism has a number of limitations:</p>

<ul>

<li><p>ID attributes must be declared as such in the DTD.  If an ID
attribute is declared as an ID attribute only in the external DTD
subset, then it will be recognized as an ID attribute only if the XML
processor reads the external DTD subset.  However, XML does not require
XML processors to read the external DTD, and they may well choose not
to do so, especially if the document is declared
<code>standalone="yes"</code>.</p></li>

<li><p>A document can contain only a single set of unique IDs.
There cannot be separate independent sets of unique IDs.</p></li>

<li><p>The ID of an element can only be specified in an attribute;
it cannot be specified by the content of the element, or by a child
element.</p></li>

<li><p>An ID is constrained to be an XML name.  For example, it
cannot contain spaces.</p></li>

<li><p>An element can have at most one ID.</p></li>

<li><p>At most one element can have a particular ID.</p></li>

</ul>

<p>Because of these limitations XML documents sometimes contain a
cross-reference structure that is not explicitly declared by
ID/IDREF/IDREFS attributes.</p>

<p>A key is a triple containing:</p>

<ol>

<li><p>the node which has the key</p></li>

<li><p>the name of the key (an <a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a>)</p></li>

<li><p>the value of the key (a string)</p></li>

</ol>

<p>A stylesheet declares a set of keys for each document using the
<code>xsl:key</code> element.  When this set of keys contains a member
with node <var>x</var>, name <var>y</var> and value
<var>z</var>, we say that node <var>x</var> has a key with name
<var>y</var> and value <var>z</var>.</p>

<p>Thus, a key is a kind of generalized ID, which is not subject to the
same limitations as an XML ID:</p>

<ul>

<li><p>Keys are declared in the stylesheet using
<code>xsl:key</code> elements.</p></li>

<li><p>A key has a name as well as a value; each key name may be
thought of as distinguishing a separate, independent space of
identifiers.</p></li>

<li><p>The value of a named key for an element may be specified in
any convenient place; for example, in an attribute, in a child element
or in content.  An XPath expression is used to specify where to find
the value for a particular named key.</p></li>

<li><p>The value of a key can be an arbitrary string; it is not
constrained to be a name.</p></li>

<li><p>There can be multiple keys in a document with the same node,
same key name, but different key values.</p></li>

<li><p>There can be multiple keys in a document with the same key
name, same key value, but different nodes.</p></li>

</ul>


  
  
    
  
  
    
  
  
    
  
  


<p>The <code>xsl:key</code> element is used to declare keys.  The
<code>name</code> attribute specifies the name of the key.  The value
of the <code>name</code> attribute is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>. The <code>match</code> attribute is a <a href='#NT-Pattern'>Pattern</a>; an <code>xsl:key</code> element gives
information about the keys of any node that matches the pattern
specified in the match attribute.  The <code>use</code> attribute is
an <a title='Expression' href='#dt-expression'>expression</a> specifying the
values of the key; the expression is evaluated once for each node that
matches the pattern.  If the result is a node-set, then for each node
in the node-set, the node that matches the pattern has a key of the
specified name whose value is the string-value of the node in the
node-set; otherwise, the result is converted to a string, and the node
that matches the pattern has a key of the specified name with value
equal to that string.  Thus, a node <var>x</var> has a key with name
<var>y</var> and value <var>z</var> if and only if there is an
<code>xsl:key</code> element such that:</p>

<ul>

<li><p><var>x</var> matches the pattern specified in the
<code>match</code> attribute of the <code>xsl:key</code> element;</p></li>

<li><p>the value of the <code>name</code> attribute of the
<code>xsl:key</code> element is equal to <var>y</var>;
and</p></li>

<li><p>when the expression specified in the <code>use</code>
attribute of the <code>xsl:key</code> element is evaluated with
<var>x</var> as the current node and with a node list containing
just <var>x</var> as the current node list resulting in an object
<var>u</var>, then either <var>z</var> is equal to the result of
converting <var>u</var> to a string as if by a call to the
string function, or <var>u</var> is a
node-set and <var>z</var> is equal to the string-value of one or
more of the nodes in <var>u</var>.</p></li>

</ul>

<p>Note also that there may be more than one <code>xsl:key</code>
element that matches a given node; all of the matching
<code>xsl:key</code> elements are used, even if they do not have the
same <a title='Import Precedence' href='#dt-import-precedence'>import
precedence</a>.</p>

<p>It is an error for the value of either the <code>use</code>
attribute or the <code>match</code> attribute to contain a <a href='http://www.w3.org/TR/xpath#NT-VariableReference'>VariableReference</a>.</p>

<p><em>node-set</em> <b>key</b>(<var>string</var>, <var>object</var>)</p>

<p>The <code>key</code> function does for keys what the
id function does for IDs.  The first argument
specifies the name of the key. The value of the argument must be a
<a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as
described in <a href='#qname'><b>2.4 Qualified Names</b></a>. When the second argument to the
<code>key</code> function is of type node-set, then the result
is the union of the result of applying the <code>key</code>
function to the string <a href='http://www.w3.org/TR/xpath#dt-value'>value</a> of each of the nodes in the
argument node-set.  When the second argument to
<code>key</code> is of any other type, the argument is
converted to a string as if by a call to the
string function; it returns a node-set
containing the nodes in the same document as the context node that
have a value for the named key equal to this string.</p>

<p>For example, given a declaration</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:key name="idkey" match="div" use="@id"/></pre></td></tr></table>

<p>an expression <code>key("idkey",@ref)</code> will return the same
node-set as <code>id(@ref)</code>, assuming that the only ID attribute
declared in the XML source document is:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;!ATTLIST div id ID #IMPLIED></pre></td></tr></table>

<p>and that the <code>ref</code> attribute of the current node
contains no whitespace.</p>

<p>Suppose a document describing a function library uses a
<code>prototype</code> element to define functions</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;prototype name="key" return-type="node-set">
&lt;arg type="string"/>
&lt;arg type="object"/>
&lt;/prototype></pre></td></tr></table>

<p>and a <code>function</code> element to refer to function names</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;function>key&lt;/function></pre></td></tr></table>

<p>Then the stylesheet could generate hyperlinks between the
references and definitions as follows:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:key name="func" match="prototype" use="@name"/>

&lt;xsl:template match="function">
&lt;b>
  &lt;a href="#{generate-id(key('func',.))}">
    &lt;xsl:apply-templates/>
  &lt;/a>
&lt;/b>
&lt;/xsl:template>

&lt;xsl:template match="prototype">
&lt;p>&lt;a name="{generate-id()}">
&lt;b>Function: &lt;/b>
...
&lt;/a>&lt;/p>
&lt;/xsl:template></pre></td></tr></table>

<p>The <code>key</code> can be used to retrieve a key from a
document other than the document containing the context node.  For
example, suppose a document contains bibliographic references in the
form <code>&lt;bibref>XSLT&lt;/bibref></code>, and there is a
separate XML document <code>bib.xml</code> containing a bibliographic
database with entries in the form:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;entry name="XSLT">...&lt;/entry></pre></td></tr></table>

<p>Then the stylesheet could use the following to transform the
<code>bibref</code> elements:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:key name="bib" match="entry" use="@name"/>

&lt;xsl:template match="bibref">
  &lt;xsl:variable name="name" select="."/>
  &lt;xsl:for-each select="document('bib.xml')">
    &lt;xsl:apply-templates select="key('bib',$name)"/>
  &lt;/xsl:for-each>
&lt;/xsl:template></pre></td></tr></table>

</div>

<div class='div2'>
<h3><a name='format-number'></a>12.3 Number Formatting</h3>

<p><em>string</em> <b>format-number</b>(<var>number</var>, <var>string</var>, <var>string</var>?)</p>

<p>The <code>format-number</code> function converts its first
argument to a string using the format pattern string specified by the
second argument and the decimal-format named by the third argument, or
the default decimal-format, if there is no third argument.  The format
pattern string is in the syntax specified by the JDK 1.1 <a href='http://java.sun.com/products/jdk/1.1/docs/api/java.text.DecimalFormat.html'>DecimalFormat</a> class. The format pattern string is in a
localized notation: the decimal-format determines what characters have
a special meaning in the pattern (with the exception of the quote
character, which is not localized).  The format pattern must not
contain the currency sign (#x00A4); support for this feature was added
after the initial release of JDK 1.1.  The decimal-format name must be
a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as
described in <a href='#qname'><b>2.4 Qualified Names</b></a>.  It is an error if the stylesheet
does not contain a declaration of the decimal-format with the specified
<a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a>.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Implementations are not required to use the JDK 1.1
implementation, nor are implementations required to be implemented in
Java.</p></div>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Stylesheets can use other facilities in XPath to control
rounding.</p></div>


  
  
  
    
  
  
    
  
  
    
  
  
    
  
  
    
  
  
    
  
  
    
  
  
    
  
  
    
  
  
    
  
  
    
  
  


<p>The <code>xsl:decimal-format</code> element declares a
decimal-format, which controls the interpretation of a format pattern
used by the <code>format-number</code> function.  If there is
a <code>name</code> attribute, then the element declares a named
decimal-format; otherwise, it declares the default decimal-format.
The value of the <code>name</code> attribute is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>, which is expanded as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>.  It is an error to declare either the
default decimal-format or a decimal-format with a given name more than
once (even with different <a title='Import Precedence' href='#dt-import-precedence'>import
precedence</a>), unless it is declared every time with the same
value for all attributes (taking into account any default values).</p>

<p>The other attributes on <code>xsl:decimal-format</code> correspond
to the methods on the JDK 1.1 <a href='http://java.sun.com/products/jdk/1.1/docs/api/java.text.DecimalFormatSymbols.html'>DecimalFormatSymbols</a> class.  For each
<code>get</code>/<code>set</code> method pair there is an attribute
defined for the <code>xsl:decimal-format</code> element.</p>

<p>The following attributes both control the interpretation of
characters in the format pattern and specify characters that may
appear in the result of formatting the number:</p>

<ul>

<li><p><code>decimal-separator</code> specifies the character used
for the decimal sign; the default value is the period character
(<code>.</code>)</p></li>

<li><p><code>grouping-separator</code> specifies the character used
as a grouping (e.g. thousands) separator; the default value is the
comma character (<code>,</code>)</p></li>

<li><p><code>percent</code> specifies the character used as a
percent sign; the default value is the percent character
(<code>%</code>)</p></li>

<li><p><code>per-mille</code> specifies the character used as a per
mille sign; the default value is the Unicode per-mille character
(#x2030)</p></li>

<li><p><code>zero-digit</code> specifies the character used as the
digit zero; the default value is the digit zero
(<code>0</code>)</p></li>

</ul>

<p>The following attributes control the interpretation of characters
in the format pattern:</p>

<ul>

<li><p><code>digit</code> specifies the character used for a digit
in the format pattern; the default value is the number sign character
(<code>#</code>)</p></li>

<li><p><code>pattern-separator</code> specifies the character used
to separate positive and negative sub patterns in a pattern; the
default value is the semi-colon character (<code>;</code>)</p></li>

</ul>

<p>The following attributes specify characters or strings that may
appear in the result of formatting the number:</p>

<ul>

<li><p><code>infinity</code> specifies the string used to represent
infinity; the default value is the string
<code>Infinity</code></p></li>

<li><p><code>NaN</code> specifies the string used to represent the
NaN value; the default value is the string <code>NaN</code></p></li>

<li><p><code>minus-sign</code> specifies the character used as the
default minus sign; the default value is the hyphen-minus character
(<code>-</code>, #x2D)</p></li>

</ul>

</div>

<div class='div2'>
<h3><a name='misc-func'></a>12.4 Miscellaneous Additional Functions</h3>

<p><em>node-set</em> <b>current</b>()</p>

<p>The <code>current</code> function returns a node-set that
has the <a title='Current Node' href='#dt-current-node'>current node</a> as its
only member.  For an outermost expression (an expression not occurring
within another expression), the current node is always the same as the
context node.  Thus,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:value-of select="current()"/></pre></td></tr></table>

<p>means the same as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:value-of select="."/></pre></td></tr></table>

<p>However, within square brackets the current node is usually
different from the context node. For example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:apply-templates select="//glossary/item[@name=current()/@ref]"/></pre></td></tr></table>

<p>will process all <code>item</code> elements that have a
<code>glossary</code> parent element and that have a <code>name</code>
attribute with value equal to the value of the current node's
<code>ref</code> attribute. This is different from</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:apply-templates select="//glossary/item[@name=./@ref]"/></pre></td></tr></table>

<p>which means the same as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:apply-templates select="//glossary/item[@name=@ref]"/></pre></td></tr></table>

<p>and so would process all <code>item</code> elements that have a
<code>glossary</code> parent element and that have a <code>name</code>
attribute and a <code>ref</code> attribute with the same value.</p>

<p>It is an error to use the <code>current</code> function in
a <a title='Pattern' href='#dt-pattern'>pattern</a>.</p>

<p><em>string</em> <b>unparsed-entity-uri</b>(<var>string</var>)</p>

<p>The <code>unparsed-entity-uri</code> returns the URI of the
unparsed entity with the specified name in the same document as the
context node (see <a href='#unparsed-entities'><b>3.3 Unparsed Entities</b></a>).  It returns the
empty string if there is no such entity.</p>

<p><em>string</em> <b>generate-id</b>(<var>node-set</var>?)</p>

<p>The <code>generate-id</code> function returns a string that
uniquely identifies the node in the argument node-set that is first in
document order.  The unique identifier must consist of ASCII
alphanumeric characters and must start with an alphabetic character.
Thus, the string is syntactically an XML name.  An implementation is
free to generate an identifier in any convenient way provided that it
always generates the same identifier for the same node and that
different identifiers are always generated from different nodes. An
implementation is under no obligation to generate the same identifiers
each time a document is transformed.  There is no guarantee that a
generated unique identifier will be distinct from any unique IDs
specified in the source document.  If the argument node-set is empty,
the empty string is returned. If the argument is omitted, it defaults
to the context node.</p>

<p><em>object</em> <b>system-property</b>(<var>string</var>)</p>

<p>The argument must evaluate to a string that is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>.  The <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> is expanded into a name using
the namespace declarations in scope for the expression. The
<code>system-property</code> function returns an object
representing the value of the system property identified by the name.
If there is no such system property, the empty string should be
returned.</p>

<p>Implementations must provide the following system properties, which
are all in the XSLT namespace:</p>

<blockquote><p>

<code>xsl:version</code>, a number giving the version of XSLT
implemented by the processor; for XSLT processors implementing the
version of XSLT specified by this document, this is the number
1.0

<code>xsl:vendor</code>, a string identifying the vendor of the
XSLT processor

<code>xsl:vendor-url</code>, a string containing a URL
identifying the vendor of the XSLT processor; typically this is the
host page (home page) of the vendor's Web site.

</p></blockquote>

</div>

</div>

<div class='div1'>
<h2><a name='message'></a>13 Messages</h2>


  
  
    
    
  
  


<p>The <code>xsl:message</code> instruction sends a message in a way
that is dependent on the XSLT processor.  The content of the
<code>xsl:message</code> instruction is a template.  The
<code>xsl:message</code> is instantiated by instantiating the content
to create an XML fragment.  This XML fragment is the content of the
message.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>An XSLT processor might implement <code>xsl:message</code> by
popping up an alert box or by writing to a log file.</p></div>

<p>If the <code>terminate</code> attribute has the value
<code>yes</code>, then the XSLT processor should terminate processing
after sending the message.  The default value is <code>no</code>.</p>

<p>One convenient way to do localization is to put the localized
information (message text, etc.) in an XML document, which becomes an
additional input file to the stylesheet.  For example, suppose
messages for a language <code><var>L</var></code> are stored in an XML
file <code>resources/<var>L</var>.xml</code> in the form:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;messages>
  &lt;message name="problem">A problem was detected.&lt;/message>
  &lt;message name="error">An error was detected.&lt;/message>
&lt;/messages>
</pre></td></tr></table>

<p>Then a stylesheet could use the following approach to localize
messages:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:param name="lang" select="en"/>
&lt;xsl:variable name="messages"
  select="document(concat('resources/', $lang, '.xml'))/messages"/>

&lt;xsl:template name="localized-message">
  &lt;xsl:param name="name"/>
  &lt;xsl:message>
    &lt;xsl:value-of select="$messages/message[@name=$name]"/>
  &lt;/xsl:message>
&lt;/xsl:template>

&lt;xsl:template name="problem">
  &lt;xsl:call-template name="localized-message"/>
    &lt;xsl:with-param name="name">problem&lt;/xsl:with-param>
  &lt;/xsl:call-template>
&lt;/xsl:template></pre></td></tr></table>

</div>

<div class='div1'>
<h2><a name='extension'></a>14 Extensions</h2>

<p>XSLT allows two kinds of extension, extension elements and
extension functions.</p>

<p>This version of XSLT does not provide a mechanism for defining
implementations of extensions.  Therefore, an XSLT stylesheet that must
be portable between XSLT implementations cannot rely on particular
extensions being available.  XSLT provides mechanisms that allow an
XSLT stylesheet to determine whether the XSLT processor by which it is
being processed has implementations of particular extensions
available, and to specify what should happen if those extensions are
not available.  If an XSLT stylesheet is careful to make use of these
mechanisms, it is possible for it to take advantage of extensions and
still work with any XSLT implementation.</p>

<div class='div2'>
<h3><a name='extension-element'></a>14.1 Extension Elements</h3>

<p>[<a name='dt-extension-namespace' title='Extension Namespace'>Definition</a>: The
element extension mechanism allows namespaces to be designated as
<b>extension namespace</b>s. When a namespace is designated as
an extension namespace and an element with a name from that namespace
occurs in a template, then the element is treated as an instruction
rather than as a literal result element.] The namespace
determines the semantics of the instruction.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Since an element that is a child of an
<code>xsl:stylesheet</code> element is not occurring <em>in a
template</em>, non-XSLT <a title='Top-level' href='#dt-top-level'>top-level</a> elements are not extension
elements as defined here, and nothing in this section applies to
them.</p></div>

<p>A namespace is designated as an extension namespace by using an
<code>extension-element-prefixes</code> attribute on an
<code>xsl:stylesheet</code> element or an
<code>xsl:extension-element-prefixes</code> attribute on a literal
result element or extension element.
The value of both these attributes is a
whitespace-separated list of namespace prefixes. The namespace bound
to each of the prefixes is designated as an extension namespace.  It
is an error if there is no namespace bound to the prefix on the
element bearing the <code>extension-element-prefixes</code> or
<code>xsl:extension-element-prefixes</code> attribute.  The default
namespace (as declared by <code>xmlns</code>) may be designated as an
extension namespace by including <code>#default</code> in the list of
namespace prefixes.  The designation of a namespace as an extension
namespace is effective within the subtree of the stylesheet rooted at
the element bearing the <code>extension-element-prefixes</code> or
<code>xsl:extension-element-prefixes</code> attribute;
a subtree rooted at an <code>xsl:stylesheet</code> element
does not include any stylesheets imported or included by children
of that <code>xsl:stylesheet</code> element.</p>

<p>If the XSLT processor does not have an implementation of a
particular extension element available, then the
<code>element-available</code> function must return false for
the name of the element.  When such an extension element is
instantiated, then the XSLT processor must perform fallback for the
element as specified in <a href='#fallback'><b>15 Fallback</b></a>.  An XSLT processor
must not signal an error merely because a template contains an
extension element for which no implementation is available.</p>

<p>If the XSLT processor has an implementation of a particular
extension element available, then the
<code>element-available</code> function must return true for
the name of the element.</p>

</div>

<div class='div2'>
<h3><a name='id21'></a>14.2 Extension Functions</h3>

<p>If a <a href='http://www.w3.org/TR/xpath#NT-FunctionName'>FunctionName</a> in a
<a href='http://www.w3.org/TR/xpath#NT-FunctionCall'>FunctionCall</a> expression is
not an <a href='http://www.w3.org/TR/REC-xml-names#NT-NCName'>NCName</a> (i.e. if it
contains a colon), then it is treated as a call to an extension
function.  The <a href='http://www.w3.org/TR/xpath#NT-FunctionName'>FunctionName</a>
is expanded to a name using the namespace declarations from the
evaluation context.</p>

<p>If the XSLT processor does not have an implementation of an
extension function of a particular name available, then the
<code>function-available</code> function must return false for
that name.  If such an extension function occurs in an expression and
the extension function is actually called, the XSLT processor must
signal an error.  An XSLT processor must not signal an error merely
because an expression contains an extension function for which no
implementation is available.</p>

<p>If the XSLT processor has an implementation of an extension
function of a particular name available, then the
<code>function-available</code> function must return
true for that name. If such an extension is called, then the XSLT
processor must call the implementation passing it the function call
arguments; the result returned by the implementation is returned as
the result of the function call.</p>

</div>

</div>

<div class='div1'>
<h2><a name='fallback'></a>15 Fallback</h2>


  
  


<p>Normally, instantiating an <code>xsl:fallback</code> element does
nothing.  However, when an XSLT processor performs fallback for an
instruction element, if the instruction element has one or more
<code>xsl:fallback</code> children, then the content of each of the
<code>xsl:fallback</code> children must be instantiated in sequence;
otherwise, an error must be signaled. The content of an
<code>xsl:fallback</code> element is a template.</p>

<p>The following functions can be used with the
<code>xsl:choose</code> and <code>xsl:if</code> instructions to
explicitly control how a stylesheet should behave if particular
elements or functions are not available.</p>

<p><em>boolean</em> <b>element-available</b>(<var>string</var>)</p>

<p>The argument must evaluate to a string that is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>.  The <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> is expanded into an <a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a> using the
namespace declarations in scope for the expression. The
<code>element-available</code> function returns true if and
only if the expanded-name is the name of an instruction.  If the
expanded-name has a namespace URI equal to the XSLT namespace URI,
then it refers to an element defined by XSLT.  Otherwise, it refers to
an extension element. If the expanded-name has a null namespace URI,
the <code>element-available</code> function will return
false.</p>

<p><em>boolean</em> <b>function-available</b>(<var>string</var>)</p>

<p>The argument must evaluate to a string that is a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>.  The <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> is expanded into an <a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a> using the
namespace declarations in scope for the expression. The
<code>function-available</code> function returns true if and
only if the expanded-name is the name of a function in the function
library. If the expanded-name has a non-null namespace URI, then it
refers to an extension function; otherwise, it refers to a function
defined by XPath or XSLT.</p>

</div>

<div class='div1'>
<h2><a name='output'></a>16 Output</h2>


  
  
    
    
    
    
  
  
    
  
  
    
  
  
    
    
  
  
    
    
  
  
    
  
  
    
  
  
    
  
  
    
    
  
  
    
  
  


<p>An XSLT processor may output the result tree as a sequence of
bytes, although it is not required to be able to do so (see <a href='#conformance'><b>17 Conformance</b></a>). The <code>xsl:output</code> element allows
stylesheet authors to specify how they wish the result tree to be
output. If an XSLT processor outputs the result tree, it should do so
as specified by the <code>xsl:output</code> element; however, it is
not required to do so.</p>

<p>The <code>xsl:output</code> element is only allowed as a <a title='Top-level' href='#dt-top-level'>top-level</a> element.</p>

<p>The <code>method</code> attribute on <code>xsl:output</code>
identifies the overall method that should be used for outputting the
result tree.  The value must be a <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>.  If the <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> does not have a prefix, then it
identifies a method specified in this document and must be one of
<code>xml</code>, <code>html</code> or <code>text</code>.  If the <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> has a prefix, then the <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> is expanded into an <a href='http://www.w3.org/TR/xpath#dt-expanded-name'>expanded-name</a> as described
in <a href='#qname'><b>2.4 Qualified Names</b></a>; the expanded-name identifies the output
method; the behavior in this case is not specified by this
document.</p>

<p>The default for the <code>method</code> attribute is chosen as
follows.  If</p>

<ul>

<li><p>the root node of the result tree has an element
child,</p></li>

<li><p>the expanded-name of the first element child of the root node
(i.e. the document element) of the result tree has local part
<code>html</code> (in any combination of upper and lower case) and a
null namespace URI, and</p></li>

<li><p>any text nodes preceding the first element child of the root
node of the result tree contain only whitespace characters,</p></li>

</ul>

<p>then the default output method is <code>html</code>; otherwise, the
default output method is <code>xml</code>.  The default output method
should be used if there are no <code>xsl:output</code> elements or if
none of the <code>xsl:output</code> elements specifies a value for the
<code>method</code> attribute.</p>

<p>The other attributes on <code>xsl:output</code> provide parameters
for the output method.  The following attributes are allowed:</p>

<ul>

<li><p><code>version</code> specifies the version of the output
method</p></li>

<li><p><code>indent</code> specifies whether the XSLT processor may
add additional whitespace when outputting the result tree; the value
must be <code>yes</code> or <code>no</code></p></li>

<li><p><code>encoding</code> specifies the preferred character
encoding that the XSLT processor should use to encode sequences of
characters as sequences of bytes; the value of the attribute should be
treated case-insensitively; the value must contain only characters in
the range #x21 to #x7E (i.e. printable ASCII characters); the value
should either be a <code>charset</code> registered with the Internet
Assigned Numbers Authority <a href='#IANA'>[IANA]</a>, <a href='#RFC2278'>[RFC2278]</a> or start with <code>X-</code></p></li>

<li><p><code>media-type</code> specifies the media type (MIME
content type) of the data that results from outputting the result
tree; the <code>charset</code> parameter should not be specified
explicitly; instead, when the top-level media type is
<code>text</code>, a <code>charset</code> parameter should be added
according to the character encoding actually used by the output
method</p></li>

<li><p><code>doctype-system</code> specifies the system identifier
to be used in the document type declaration</p></li>

<li><p><code>doctype-public</code> specifies the public identifier
to be used in the document type declaration</p></li>

<li><p><code>omit-xml-declaration</code> specifies whether the XSLT
processor should output an XML declaration; the value must be
<code>yes</code> or <code>no</code></p></li>

<li><p><code>standalone</code> specifies whether the XSLT processor
should output a standalone document declaration; the value must be
<code>yes</code> or <code>no</code></p></li>

<li><p><code>cdata-section-elements</code> specifies a list of the
names of elements whose text node children should be output using
CDATA sections</p></li>

</ul>

<p>The detailed semantics of each attribute will be described
separately for each output method for which it is applicable.  If the
semantics of an attribute are not described for an output method, then
it is not applicable to that output method.</p>

<p>A stylesheet may contain multiple <code>xsl:output</code> elements
and may include or import stylesheets that also contain
<code>xsl:output</code> elements.  All the <code>xsl:output</code>
elements occurring in a stylesheet are merged into a single effective
<code>xsl:output</code> element. For the
<code>cdata-section-elements</code> attribute, the effective value is
the union of the specified values.  For other attributes, the
effective value is the specified value with the highest <a title='Import Precedence' href='#dt-import-precedence'>import precedence</a>. It is an error
if there is more than one such value for an attribute.  An XSLT
processor may signal the error; if it does not signal the error, if
should recover by using the value that occurs last in the stylesheet.
The values of attributes are defaulted after the
<code>xsl:output</code> elements have been merged; different output
methods may have different default values for an attribute.</p>

<div class='div2'>
<h3><a name='id22'></a>16.1 XML Output Method</h3>

<p>The <code>xml</code> output method outputs the result tree as a
well-formed XML external general parsed entity. If the root node of
the result tree has a single element node child and no text node
children, then the entity should also be a well-formed XML document
entity. When the entity is referenced within a trivial XML document
wrapper like this</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>
&lt;!DOCTYPE doc [
&lt;!ENTITY e SYSTEM "<var>entity-URI</var>">
]>
&lt;doc>&amp;e;&lt;/doc></pre></td></tr></table>

<p>where <code><var>entity-URI</var></code> is a URI for the entity,
then the wrapper
document as a whole should be a well-formed XML document conforming to
the XML Namespaces Recommendation <a href='#XMLNAMES'>[XML Names]</a>.  In
addition, the output should be such that if a new tree was constructed
by parsing the wrapper as an XML document as specified in <a href='#data-model'><b>3 Data Model</b></a>, and then removing the document element, making its
children instead be children of the root node, then the new tree would
be the same as the result tree, with the following possible
exceptions:</p>

<ul>

<li><p>The order of attributes in the two trees may be
different.</p></li>

<li><p>The new tree may contain namespace nodes that were not
present in the result tree.</p>
<div class='note'><p class='prefix'><b>Note:</b></p><p>An XSLT processor may need to add
namespace declarations in the course of outputting the result tree as
XML.</p></div>
</li>

</ul>

<p>If the XSLT processor generated a document type declaration because
of the <code>doctype-system</code> attribute, then the above
requirements apply to the entity with the generated document type
declaration removed.</p>

<p>The <code>version</code> attribute specifies the version of XML to
be used for outputting the result tree.  If the XSLT processor does
not support this version of XML, it should use a version of XML that
it does support.  The version output in the XML declaration (if an XML
declaration is output) should correspond to the version of XML that
the processor used for outputting the result tree. The value of the
<code>version</code> attribute should match the <a href='http://www.w3.org/TR/REC-xml#NT-VersionNum'>VersionNum</a> production of the XML
Recommendation <a href='#XML'>[XML]</a>. The default value is
<code>1.0</code>.</p>

<p>The <code>encoding</code> attribute specifies the preferred
encoding to use for outputting the result tree.  XSLT processors are
required to respect values of <code>UTF-8</code> and
<code>UTF-16</code>.  For other values, if the XSLT processor does not
support the specified encoding it may signal an error; if it does not
signal an error it should use <code>UTF-8</code> or
<code>UTF-16</code> instead.  The XSLT processor must not use an
encoding whose name does not match the <a href='http://www.w3.org/TR/REC-xml#NT-EncName'>EncName</a> production of the XML
Recommendation <a href='#XML'>[XML]</a>.  If no <code>encoding</code>
attribute is specified, then the XSLT processor should use either
<code>UTF-8</code> or <code>UTF-16</code>.  It is possible that the
result tree will contain a character that cannot be represented in the
encoding that the XSLT processor is using for output.  In this case,
if the character occurs in a context where XML recognizes character
references (i.e. in the value of an attribute node or text node), then
the character should be output as a character reference; otherwise
(for example if the character occurs in the name of an element) the
XSLT processor should signal an error.</p>

<p>If the <code>indent</code> attribute has the value
<code>yes</code>, then the <code>xml</code> output method may output
whitespace in addition to the whitespace in the result tree (possibly
based on whitespace stripped from either the source document or the
stylesheet) in order to indent the result nicely; if the
<code>indent</code> attribute has the value <code>no</code>, it should
not output any additional whitespace. The default value is
<code>no</code>.  The <code>xml</code> output method should use an
algorithm to output additional whitespace that ensures that the result
if whitespace were to be stripped from the output using the process
described in <a href='#strip'><b>3.4 Whitespace Stripping</b></a> with the set of
whitespace-preserving elements consisting of just
<code>xsl:text</code> would be the same when additional whitespace is
output as when additional whitespace is not output.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>It is usually not safe to use <code>indent="yes"</code> with
document types that include element types with mixed content.</p></div>

<p>The <code>cdata-section-elements</code> attribute contains a
whitespace-separated list of <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>s.  Each <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> is expanded into an
expanded-name using the namespace declarations in effect on the
<code>xsl:output</code> element in which the <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a> occurs; if there is a default
namespace, it is used for <a href='http://www.w3.org/TR/REC-xml-names#NT-QName'>QName</a>s
that do not have a prefix.  The expansion is performed before the
merging of multiple <code>xsl:output</code> elements into a single
effective <code>xsl:output</code> element. If the expanded-name of the
parent of a text node is a member of the list, then the text node
should be output as a CDATA section. For example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:output cdata-section-elements="example"/></pre></td></tr></table>

<p>would cause a literal result element written in the stylesheet as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;example>&amp;lt;foo>&lt;/example></pre></td></tr></table>

<p>or as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;example>&lt;![CDATA[&lt;foo>]]&gt;&lt;/example></pre></td></tr></table>

<p>to be output as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;example>&lt;![CDATA[&lt;foo>]]&gt;&lt;/example></pre></td></tr></table>

<p>If the text node contains the sequence of characters
<code>]]&gt;</code>, then the currently open CDATA section should be
closed following the <code>]]</code> and a new CDATA section opened
before the <code>></code>. For example, a literal result element
written in the stylesheet as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;example>]]&amp;gt;&lt;/example></pre></td></tr></table>

<p>would be output as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;example>&lt;![CDATA[]]]]&gt;&lt;![CDATA[>]]&gt;&lt;/example></pre></td></tr></table>

<p>If the text node contains a character that is not representable in
the character encoding being used to output the result tree, then the
currently open CDATA section should be closed before the character,
the character should be output using a character reference or entity
reference, and a new CDATA section should be opened for any further
characters in the text node.</p>

<p>CDATA sections should not be used except for text nodes that the
<code>cdata-section-elements</code> attribute explicitly specifies
should be output using CDATA sections.</p>

<p>The <code>xml</code> output method should output an XML declaration
unless the <code>omit-xml-declaration</code> attribute has the value
<code>yes</code>. The XML declaration should include both version
information and an encoding declaration. If the
<code>standalone</code> attribute is specified, it should include a
standalone document declaration with the same value as the value as
the value of the <code>standalone</code> attribute.  Otherwise, it
should not include a standalone document declaration; this ensures
that it is both a XML declaration (allowed at the beginning of a
document entity) and a text declaration (allowed at the beginning of
an external general parsed entity).</p>

<p>If the <code>doctype-system</code> attribute is specified, the
<code>xml</code> output method should output a document type
declaration immediately before the first element.  The name following
<code>&lt;!DOCTYPE</code> should be the name of the first element.  If
<code>doctype-public</code> attribute is also specified, then the
<code>xml</code> output method should output <code>PUBLIC</code>
followed by the public identifier and then the system identifier;
otherwise, it should output <code>SYSTEM</code> followed by the system
identifier.  The internal subset should be empty.  The
<code>doctype-public</code> attribute should be ignored unless the
<code>doctype-system</code> attribute is specified.</p>

<p>The <code>media-type</code> attribute is applicable for the
<code>xml</code> output method.  The default value for the
<code>media-type</code> attribute is <code>text/xml</code>.</p>

</div>

<div class='div2'>
<h3><a name='id23'></a>16.2 HTML Output Method</h3>

<p>The <code>html</code> output method outputs the result tree as
HTML; for example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

&lt;xsl:output method="html"/>

&lt;xsl:template match="/">
  &lt;html>
   &lt;xsl:apply-templates/>
  &lt;/html>
&lt;/xsl:template>

...

&lt;/xsl:stylesheet></pre></td></tr></table>

<p>The <code>version</code> attribute indicates the version of the
HTML.  The default value is <code>4.0</code>, which specifies that the
result should be output as HTML conforming to the HTML 4.0
Recommendation <a href='#HTML'>[HTML]</a>.</p>

<p>The <code>html</code> output method should not output an element
differently from the <code>xml</code> output method unless the
expanded-name of the element has a null namespace URI; an element
whose expanded-name has a non-null namespace URI should be output as
XML.  If the expanded-name of the element has a null namespace URI,
but the local part of the expanded-name is not recognized as the name
of an HTML element, the element should output in the same way as a
non-empty, inline element such as <code>span</code>.</p>

<p>The <code>html</code> output method should not output an end-tag
for empty elements.  For HTML 4.0, the empty elements are
<code>area</code>, <code>base</code>, <code>basefont</code>,
<code>br</code>, <code>col</code>, <code>frame</code>,
<code>hr</code>, <code>img</code>, <code>input</code>,
<code>isindex</code>, <code>link</code>, <code>meta</code> and
<code>param</code>. For example, an element written as
<code>&lt;br/></code> or <code>&lt;br>&lt;/br></code> in the
stylesheet should be output as <code>&lt;br></code>.</p>

<p>The <code>html</code> output method should recognize the names of
HTML elements regardless of case.  For example, elements named
<code>br</code>, <code>BR</code> or <code>Br</code> should all be
recognized as the HTML <code>br</code> element and output without an
end-tag.</p>

<p>The <code>html</code> output method should not perform escaping for
the content of the <code>script</code> and <code>style</code>
elements. For example, a literal result element written in the
stylesheet as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;script>if (a &amp;lt; b) foo()&lt;/script></pre></td></tr></table>

<p>or</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;script>&lt;![CDATA[if (a &lt; b) foo()]]&gt;&lt;/script></pre></td></tr></table>

<p>should be output as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;script>if (a &lt; b) foo()&lt;/script></pre></td></tr></table>

<p>The <code>html</code> output method should not escape
<code>&lt;</code> characters occurring in attribute values.</p>

<p>If the <code>indent</code> attribute has the value
<code>yes</code>, then the <code>html</code> output method may add or
remove whitespace as it outputs the result tree, so long as it does
not change how an HTML user agent would render the output.  The
default value is <code>yes</code>.</p>

<p>The <code>html</code> output method should escape non-ASCII
characters in URI attribute values using the method recommended in
<a href='http://www.w3.org/TR/REC-html40/appendix/notes.html#h-B.2.1'>Section
B.2.1</a> of the HTML 4.0 Recommendation.</p>

<p>The <code>html</code> output method may output a character using a
character entity reference, if one is defined for it in the version of
HTML that the output method is using.</p>

<p>The <code>html</code> output method should terminate processing
instructions with <code>></code> rather than
<code>?></code>.</p>

<p>The <code>html</code> output method should output boolean
attributes (that is attributes with only a single allowed value that
is equal to the name of the attribute) in minimized form. For example,
a start-tag written in the stylesheet as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;OPTION selected="selected"></pre></td></tr></table>

<p>should be output as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;OPTION selected></pre></td></tr></table>

<p>The <code>html</code> output method should not escape a
<code>&amp;</code> character occurring in an attribute value
immediately followed by a <code>{</code> character (see <a href='http://www.w3.org/TR/REC-html40/appendix/notes.html#h-B.7.1.1'>Section
B.7.1</a> of the HTML 4.0 Recommendation). For example, a start-tag
written in the stylesheet as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;BODY bgcolor='&amp;amp;{{randomrbg}};'></pre></td></tr></table>

<p>should be output as</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;BODY bgcolor='&amp;{randomrbg};'></pre></td></tr></table>

<p>The <code>encoding</code> attribute specifies the preferred
encoding to be used. If there is a <code>HEAD</code> element, then the
<code>html</code> output method should add a <code>META</code> element
immediately after the start-tag of the <code>HEAD</code> element
specifying the character encoding actually used. For example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;HEAD>
&lt;META http-equiv="Content-Type" content="text/html; charset=EUC-JP">
...</pre></td></tr></table>

<p>It is possible that the result tree will contain a character that
cannot be represented in the encoding that the XSLT processor is using
for output.  In this case, if the character occurs in a context where
HTML recognizes character references, then the character should be
output as a character entity reference or decimal numeric character
reference; otherwise (for example, in a
<code>script</code> or <code>style</code> element or in a comment),
the XSLT processor should signal an error.</p>

<p>If the <code>doctype-public</code> or <code>doctype-system</code>
attributes are specified, then the <code>html</code> output method
should output a document type declaration immediately before the first
element.  The name following <code>&lt;!DOCTYPE</code> should be
<code>HTML</code> or <code>html</code>.  If the
<code>doctype-public</code> attribute is specified, then the output
method should output <code>PUBLIC</code> followed by the specified
public identifier; if the <code>doctype-system</code> attribute is
also specified, it should also output the specified system identifier
following the public identifier.  If the <code>doctype-system</code>
attribute is specified but the <code>doctype-public</code> attribute
is not specified, then the output method should output
<code>SYSTEM</code> followed by the specified system identifier.</p>

<p>The <code>media-type</code> attribute is applicable for the
<code>html</code> output method.  The default value is
<code>text/html</code>.</p>

</div>

<div class='div2'>
<h3><a name='id24'></a>16.3 Text Output Method</h3>

<p>The <code>text</code> output method outputs the result tree by
outputting the string-value of every text node in the result tree in
document order without any escaping.</p>

<p>The <code>media-type</code> attribute is applicable for the
<code>text</code> output method.  The default value for the
<code>media-type</code> attribute is <code>text/plain</code>.</p>

<p>The <code>encoding</code> attribute identifies the encoding that
the <code>text</code> output method should use to convert sequences of
characters to sequences of bytes.  The default is system-dependent. If
the result tree contains a character that cannot be represented in the
encoding that the XSLT processor is using for output, the XSLT
processor should signal an error.</p>

</div>

<div class='div2'>
<h3><a name='disable-output-escaping'></a>16.4 Disabling Output Escaping</h3>

<p>Normally, the <code>xml</code> output method escapes &amp; and &lt;
(and possibly other characters) when outputting text nodes.  This
ensures that the output is well-formed XML. However, it is sometimes
convenient to be able to produce output that is almost, but not quite
well-formed XML; for example, the output may include ill-formed
sections which are intended to be transformed into well-formed XML by
a subsequent non-XML aware process.  For this reason, XSLT provides a
mechanism for disabling output escaping. An <code>xsl:value-of</code>
or <code>xsl:text</code> element may have a
<code>disable-output-escaping</code> attribute; the allowed values are
<code>yes</code> or <code>no</code>; the default is <code>no</code>;
if the value is <code>yes</code>, then a text node generated by
instantiating the <code>xsl:value-of</code> or <code>xsl:text</code>
element should be output without any escaping. For example,</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:text disable-output-escaping="yes">&amp;lt;&lt;/xsl:text></pre></td></tr></table>

<p>should generate the single character <code>&lt;</code>.</p>

<p>It is an error for output escaping to be disabled for a text node
that is used for something other than a text node in the result tree.
Thus, it is an error to disable output escaping for an
<code>xsl:value-of</code> or <code>xsl:text</code> element that is
used to generate the string-value of a comment, processing instruction
or attribute node; it is also an error to convert a <a title='Result Tree Fragment' href='#dt-result-tree-fragment'>result tree fragment</a> to a
number or a string if the result tree fragment contains a text node for
which escaping was disabled.  In both cases, an XSLT processor may
signal the error; if it does not signal the error, it must recover by
ignoring the <code>disable-output-escaping</code> attribute.</p>

<p>The <code>disable-output-escaping</code> attribute may be used with
the <code>html</code> output method as well as with the
<code>xml</code> output method.  The <code>text</code> output method
ignores the <code>disable-output-escaping</code> attribute, since it
does not perform any output escaping.</p>

<p>An XSLT processor will only be able to disable output escaping if
it controls how the result tree is output. This may not always be the
case.  For example, the result tree may be used as the source tree for
another XSLT transformation instead of being output.  An XSLT
processor is not required to support disabling output escaping.  If an
<code>xsl:value-of</code> or <code>xsl:text</code> specifies that
output escaping should be disabled and the XSLT processor does not
support this, the XSLT processor may signal an error; if it does not
signal an error, it must recover by not disabling output escaping.</p>

<p>If output escaping is disabled for a character that is not
representable in the encoding that the XSLT processor is using for
output, then the XSLT processor may signal an error; if it does not
signal an error, it must recover by not disabling output escaping.</p>

<p>Since disabling output escaping may not work with all XSLT
processors and can result in XML that is not well-formed, it should be
used only when there is no alternative.</p>


</div>

</div>

<div class='div1'>
<h2><a name='conformance'></a>17 Conformance</h2>

<p>A conforming XSLT processor must be able to use a stylesheet to
transform a source tree into a result tree as specified in this
document.  A conforming XSLT processor need not be able to output the
result in XML or in any other form.</p>

<div class='note'><p class='prefix'><b>Note:</b></p><p>Vendors of XSLT processors are strongly encouraged to provide
a way to verify that their processor is behaving conformingly by
allowing the result tree to be output as XML or by providing access to
the result tree through a standard API such as the DOM or
SAX.</p></div>

<p>A conforming XSLT processor must signal any errors except for those
that this document specifically allows an XSLT processor not to
signal. A conforming XSLT processor may but need not recover from any
errors that it signals.</p>

<p>A conforming XSLT processor may impose limits on the processing
resources consumed by the processing of a stylesheet.</p>

</div>

<div class='div1'>
<h2><a name='notation'></a>18 Notation</h2>

<p>The specification of each XSLT-defined element type is preceded by
a summary of its syntax in the form of a model for elements of that
element type.  The meaning of syntax summary notation is as
follows:</p>

<ul>

<li><p>An attribute is required if and only if its name is in
bold.</p></li>

<li><p>The string that occurs in the place of an attribute value
specifies the allowed values of the attribute.  If this is surrounded
by curly braces, then the attribute value is treated as an <a title='Attribute Value Template' href='#dt-attribute-value-template'>attribute value template</a>,
and the string occurring within curly braces specifies the allowed
values of the result of instantiating the attribute value template.
Alternative allowed values are separated by <code>|</code>.  A quoted
string indicates a value equal to that specific string. An unquoted,
italicized name specifies a particular type of value.</p></li>

<li><p>If the element is allowed not to be empty, then the element
contains a comment specifying the allowed content.  The allowed
content is specified in a similar way to an element type declaration
in XML; <em>template</em> means that any mixture of text nodes,
literal result elements, extension elements, and XSLT elements from
the <code>instruction</code> category is allowed;
<em>top-level-elements</em> means that any mixture of XSLT
elements from the <code>top-level-element</code> category is
allowed.</p></li>

<li><p>The element is prefaced by comments indicating if it belongs
to the <code>instruction</code> category or
<code>top-level-element</code> category or both.  The category of an
element just affects whether it is allowed in the content of elements
that allow a <em>template</em> or
<em>top-level-elements</em>.</p></li>

</ul>

</div>

</div>

<div class='back'>
<div class='div1'>
<h2><a name='id25'></a>A References</h2>
<div class='div2'>
<h3><a name='id26'></a>A.1 Normative References</h3>

<dl>

<dt class='label'><a name='XML'></a>XML</dt><dd>World Wide Web Consortium. <em>Extensible
Markup Language (XML) 1.0.</em> W3C Recommendation. See <a href='http://www.w3.org/TR/1998/REC-xml-19980210'>http://www.w3.org/TR/1998/REC-xml-19980210</a></dd>

<dt class='label'><a name='XMLNAMES'></a>XML Names</dt><dd>World Wide Web
Consortium. <em>Namespaces in XML.</em> W3C Recommendation. See
<a href='http://www.w3.org/TR/REC-xml-names'>http://www.w3.org/TR/REC-xml-names</a></dd>

<dt class='label'><a name='XPATH'></a>XPath</dt><dd>World Wide Web Consortium. <em>XML Path
Language.</em> W3C Recommendation. See <a href='http://www.w3.org/TR/xpath'>http://www.w3.org/TR/xpath</a></dd>

</dl>
</div>
<div class='div2'>
<h3><a name='id27'></a>A.2 Other References</h3>

<dl>

<dt class='label'><a name='CSS2'></a>CSS2</dt><dd>World Wide Web Consortium.  <em>Cascading
Style Sheets, level 2 (CSS2)</em>.  W3C Recommendation.  See <a href='http://www.w3.org/TR/1998/REC-CSS2-19980512'>http://www.w3.org/TR/1998/REC-CSS2-19980512</a></dd>

<dt class='label'><a name='DSSSL'></a>DSSSL</dt><dd>International Organization
for Standardization, International Electrotechnical Commission.
<em>ISO/IEC 10179:1996.  Document Style Semantics and Specification
Language (DSSSL)</em>.  International Standard.</dd>

<dt class='label'><a name='HTML'></a>HTML</dt><dd>World Wide Web Consortium. <em>HTML 4.0
specification</em>. W3C Recommendation. See <a href='http://www.w3.org/TR/REC-html40'>http://www.w3.org/TR/REC-html40</a></dd>

<dt class='label'><a name='IANA'></a>IANA</dt><dd>Internet Assigned Numbers
Authority. <em>Character Sets</em>. See <a href='ftp://ftp.isi.edu/in-notes/iana/assignments/character-sets'>ftp://ftp.isi.edu/in-notes/iana/assignments/character-sets</a>.</dd>

<dt class='label'><a name='RFC2278'></a>RFC2278</dt><dd>N. Freed, J. Postel.  <em>IANA
Charset Registration Procedures</em>.  IETF RFC 2278. See <a href='http://www.ietf.org/rfc/rfc2278.txt'>http://www.ietf.org/rfc/rfc2278.txt</a>.</dd>

<dt class='label'><a name='RFC2376'></a>RFC2376</dt><dd>E. Whitehead, M. Murata.  <em>XML
Media Types</em>. IETF RFC 2376. See <a href='http://www.ietf.org/rfc/rfc2376.txt'>http://www.ietf.org/rfc/rfc2376.txt</a>.</dd>

<dt class='label'><a name='RFC2396'></a>RFC2396</dt><dd>T. Berners-Lee, R. Fielding, and
L. Masinter.  <em>Uniform Resource Identifiers (URI): Generic
Syntax</em>. IETF RFC 2396. See <a href='http://www.ietf.org/rfc/rfc2396.txt'>http://www.ietf.org/rfc/rfc2396.txt</a>.</dd>

<dt class='label'><a name='UNICODE-TR10'></a>UNICODE TR10</dt><dd>Unicode Consortium.
<em>Unicode Technical Report #10. Unicode Collation
Algorithm</em>.  Unicode Technical Report.  See <a href='http://www.unicode.org/unicode/reports/tr10/index.html'>http://www.unicode.org/unicode/reports/tr10/index.html</a>.</dd>

<dt class='label'><a name='XHTML'></a>XHTML</dt><dd>World Wide Web Consortium. <em>XHTML
1.0: The Extensible HyperText Markup Language.</em> W3C Proposed
Recommendation. See <a href='http://www.w3.org/TR/xhtml1'>http://www.w3.org/TR/xhtml1</a></dd>

<dt class='label'><a name='XPTR'></a>XPointer</dt><dd>World Wide Web
Consortium. <em>XML Pointer Language (XPointer).</em> W3C Working
Draft. See <a href='http://www.w3.org/TR/xptr'>http://www.w3.org/TR/xptr</a></dd>

<dt class='label'><a name='XMLSTYLE'></a>XML Stylesheet</dt><dd>World Wide Web
Consortium. <em>Associating stylesheets with XML documents.</em>
W3C Recommendation. See <a href='http://www.w3.org/TR/xml-stylesheet'>http://www.w3.org/TR/xml-stylesheet</a></dd>

<dt class='label'><a name='XSL'></a>XSL</dt><dd>World Wide Web Consortium.  <em>Extensible
Stylesheet Language (XSL).</em>  W3C Working Draft.  See <a href='http://www.w3.org/TR/WD-xsl'>http://www.w3.org/TR/WD-xsl</a></dd>

</dl>

</div>
</div>

<div class='div1'>
<h2><a name='element-syntax-summary'></a>B Element Syntax Summary</h2>



</div>

<div class='div1'>
<h2><a name='dtd'></a>C DTD Fragment for XSLT Stylesheets (Non-Normative)</h2>

<div class='note'><p class='prefix'><b>Note:</b></p><p>This DTD Fragment is not normative because XML 1.0 DTDs do
not support XML Namespaces and thus cannot correctly describe the
allowed structure of an XSLT stylesheet.</p></div>

<p>The following entity can be used to construct a DTD for XSLT
stylesheets that create instances of a particular result DTD.  Before
referencing the entity, the stylesheet DTD must define a
<code>result-elements</code> parameter entity listing the allowed
result element types.  For example:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;!ENTITY % result-elements "
  | fo:inline-sequence
  | fo:block
"></pre></td></tr></table>

<p>Such result elements should be declared to have
<code>xsl:use-attribute-sets</code> and
<code>xsl:extension-element-prefixes</code> attributes.  The following
entity declares the <code>result-element-atts</code> parameter for
this purpose. The content that XSLT allows for result elements is the
same as it allows for the XSLT elements that are declared in the
following entity with a content model of <code>%template;</code>.  The
DTD may use a more restrictive content model than
<code>%template;</code> to reflect the constraints of the result
DTD.</p>

<p>The DTD may define the <code>non-xsl-top-level</code> parameter
entity to allow additional top-level elements from namespaces other
than the XSLT namespace.</p>

<p>The use of the <code>xsl:</code> prefix in this DTD does not imply
that XSLT stylesheets are required to use this prefix.  Any of the
elements declared in this DTD may have attributes whose name starts
with <code>xmlns:</code> or is equal to <code>xmlns</code> in addition
to the attributes declared in this DTD.</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;!ENTITY % char-instructions "
  | xsl:apply-templates
  | xsl:call-template
  | xsl:apply-imports
  | xsl:for-each
  | xsl:value-of
  | xsl:copy-of
  | xsl:number
  | xsl:choose
  | xsl:if
  | xsl:text
  | xsl:copy
  | xsl:variable
  | xsl:message
  | xsl:fallback
">

&lt;!ENTITY % instructions "
  %char-instructions;
  | xsl:processing-instruction
  | xsl:comment
  | xsl:element
  | xsl:attribute
">

&lt;!ENTITY % char-template "
 (#PCDATA
  %char-instructions;)*
">

&lt;!ENTITY % template "
 (#PCDATA
  %instructions;
  %result-elements;)*
">

&lt;!-- Used for the type of an attribute value that is a URI reference.-->
&lt;!ENTITY % URI "CDATA">

&lt;!-- Used for the type of an attribute value that is a pattern.-->
&lt;!ENTITY % pattern "CDATA">

&lt;!-- Used for the type of an attribute value that is an
     attribute value template.-->
&lt;!ENTITY % avt "CDATA">

&lt;!-- Used for the type of an attribute value that is a QName; the prefix
     gets expanded by the XSLT processor. -->
&lt;!ENTITY % qname "NMTOKEN">

&lt;!-- Like qname but a whitespace-separated list of QNames. -->
&lt;!ENTITY % qnames "NMTOKENS">

&lt;!-- Used for the type of an attribute value that is an expression.-->
&lt;!ENTITY % expr "CDATA">

&lt;!-- Used for the type of an attribute value that consists
     of a single character.-->
&lt;!ENTITY % char "CDATA">

&lt;!-- Used for the type of an attribute value that is a priority. -->
&lt;!ENTITY % priority "NMTOKEN">

&lt;!ENTITY % space-att "xml:space (default|preserve) #IMPLIED">

&lt;!-- This may be overridden to customize the set of elements allowed
at the top-level. -->

&lt;!ENTITY % non-xsl-top-level "">

&lt;!ENTITY % top-level "
 (xsl:import*,
  (xsl:include
  | xsl:strip-space
  | xsl:preserve-space
  | xsl:output
  | xsl:key
  | xsl:decimal-format
  | xsl:attribute-set
  | xsl:variable
  | xsl:param
  | xsl:template
  | xsl:namespace-alias
  %non-xsl-top-level;)*)
">

&lt;!ENTITY % top-level-atts '
  extension-element-prefixes CDATA #IMPLIED
  exclude-result-prefixes CDATA #IMPLIED
  id ID #IMPLIED
  version NMTOKEN #REQUIRED
  xmlns:xsl CDATA #FIXED "http://www.w3.org/1999/XSL/Transform"
  %space-att;
'>

&lt;!-- This entity is defined for use in the ATTLIST declaration
for result elements. -->

&lt;!ENTITY % result-element-atts '
  xsl:extension-element-prefixes CDATA #IMPLIED
  xsl:exclude-result-prefixes CDATA #IMPLIED
  xsl:use-attribute-sets %qnames; #IMPLIED
  xsl:version NMTOKEN #IMPLIED
'>

&lt;!ELEMENT xsl:stylesheet %top-level;>
&lt;!ATTLIST xsl:stylesheet %top-level-atts;>

&lt;!ELEMENT xsl:transform %top-level;>
&lt;!ATTLIST xsl:transform %top-level-atts;>

&lt;!ELEMENT xsl:import EMPTY>
&lt;!ATTLIST xsl:import href %URI; #REQUIRED>

&lt;!ELEMENT xsl:include EMPTY>
&lt;!ATTLIST xsl:include href %URI; #REQUIRED>

&lt;!ELEMENT xsl:strip-space EMPTY>
&lt;!ATTLIST xsl:strip-space elements CDATA #REQUIRED>

&lt;!ELEMENT xsl:preserve-space EMPTY>
&lt;!ATTLIST xsl:preserve-space elements CDATA #REQUIRED>

&lt;!ELEMENT xsl:output EMPTY>
&lt;!ATTLIST xsl:output
  method %qname; #IMPLIED
  version NMTOKEN #IMPLIED
  encoding CDATA #IMPLIED
  omit-xml-declaration (yes|no) #IMPLIED
  standalone (yes|no) #IMPLIED
  doctype-public CDATA #IMPLIED
  doctype-system CDATA #IMPLIED
  cdata-section-elements %qnames; #IMPLIED
  indent (yes|no) #IMPLIED
  media-type CDATA #IMPLIED
>

&lt;!ELEMENT xsl:key EMPTY>
&lt;!ATTLIST xsl:key
  name %qname; #REQUIRED
  match %pattern; #REQUIRED
  use %expr; #REQUIRED
>

&lt;!ELEMENT xsl:decimal-format EMPTY>
&lt;!ATTLIST xsl:decimal-format
  name %qname; #IMPLIED
  decimal-separator %char; "."
  grouping-separator %char; ","
  infinity CDATA "Infinity"
  minus-sign %char; "-"
  NaN CDATA "NaN"
  percent %char; "%"
  per-mille %char; "&amp;#x2030;"
  zero-digit %char; "0"
  digit %char; "#"
  pattern-separator %char; ";"
>

&lt;!ELEMENT xsl:namespace-alias EMPTY>
&lt;!ATTLIST xsl:namespace-alias
  stylesheet-prefix CDATA #REQUIRED
  result-prefix CDATA #REQUIRED
>

&lt;!ELEMENT xsl:template
 (#PCDATA
  %instructions;
  %result-elements;
  | xsl:param)*
>

&lt;!ATTLIST xsl:template
  match %pattern; #IMPLIED
  name %qname; #IMPLIED
  priority %priority; #IMPLIED
  mode %qname; #IMPLIED
  %space-att;
>

&lt;!ELEMENT xsl:value-of EMPTY>
&lt;!ATTLIST xsl:value-of
  select %expr; #REQUIRED
  disable-output-escaping (yes|no) "no"
>

&lt;!ELEMENT xsl:copy-of EMPTY>
&lt;!ATTLIST xsl:copy-of select %expr; #REQUIRED>

&lt;!ELEMENT xsl:number EMPTY>
&lt;!ATTLIST xsl:number
   level (single|multiple|any) "single"
   count %pattern; #IMPLIED
   from %pattern; #IMPLIED
   value %expr; #IMPLIED
   format %avt; '1'
   lang %avt; #IMPLIED
   letter-value %avt; #IMPLIED
   grouping-separator %avt; #IMPLIED
   grouping-size %avt; #IMPLIED
>

&lt;!ELEMENT xsl:apply-templates (xsl:sort|xsl:with-param)*>
&lt;!ATTLIST xsl:apply-templates
  select %expr; "node()"
  mode %qname; #IMPLIED
>

&lt;!ELEMENT xsl:apply-imports EMPTY>

&lt;!-- xsl:sort cannot occur after any other elements or
any non-whitespace character -->

&lt;!ELEMENT xsl:for-each
 (#PCDATA
  %instructions;
  %result-elements;
  | xsl:sort)*
>

&lt;!ATTLIST xsl:for-each
  select %expr; #REQUIRED
  %space-att;
>

&lt;!ELEMENT xsl:sort EMPTY>
&lt;!ATTLIST xsl:sort
  select %expr; "."
  lang %avt; #IMPLIED
  data-type %avt; "text"
  order %avt; "ascending"
  case-order %avt; #IMPLIED
>

&lt;!ELEMENT xsl:if %template;>
&lt;!ATTLIST xsl:if
  test %expr; #REQUIRED
  %space-att;
>

&lt;!ELEMENT xsl:choose (xsl:when+, xsl:otherwise?)>
&lt;!ATTLIST xsl:choose %space-att;>

&lt;!ELEMENT xsl:when %template;>
&lt;!ATTLIST xsl:when
  test %expr; #REQUIRED
  %space-att;
>

&lt;!ELEMENT xsl:otherwise %template;>
&lt;!ATTLIST xsl:otherwise %space-att;>

&lt;!ELEMENT xsl:attribute-set (xsl:attribute)*>
&lt;!ATTLIST xsl:attribute-set
  name %qname; #REQUIRED
  use-attribute-sets %qnames; #IMPLIED
>

&lt;!ELEMENT xsl:call-template (xsl:with-param)*>
&lt;!ATTLIST xsl:call-template
  name %qname; #REQUIRED
>

&lt;!ELEMENT xsl:with-param %template;>
&lt;!ATTLIST xsl:with-param
  name %qname; #REQUIRED
  select %expr; #IMPLIED
>

&lt;!ELEMENT xsl:variable %template;>
&lt;!ATTLIST xsl:variable 
  name %qname; #REQUIRED
  select %expr; #IMPLIED
>

&lt;!ELEMENT xsl:param %template;>
&lt;!ATTLIST xsl:param 
  name %qname; #REQUIRED
  select %expr; #IMPLIED
>

&lt;!ELEMENT xsl:text (#PCDATA)>
&lt;!ATTLIST xsl:text
  disable-output-escaping (yes|no) "no"
>

&lt;!ELEMENT xsl:processing-instruction %char-template;>
&lt;!ATTLIST xsl:processing-instruction 
  name %avt; #REQUIRED
  %space-att;
>

&lt;!ELEMENT xsl:element %template;>
&lt;!ATTLIST xsl:element 
  name %avt; #REQUIRED
  namespace %avt; #IMPLIED
  use-attribute-sets %qnames; #IMPLIED
  %space-att;
>

&lt;!ELEMENT xsl:attribute %char-template;>
&lt;!ATTLIST xsl:attribute 
  name %avt; #REQUIRED
  namespace %avt; #IMPLIED
  %space-att;
>

&lt;!ELEMENT xsl:comment %char-template;>
&lt;!ATTLIST xsl:comment %space-att;>

&lt;!ELEMENT xsl:copy %template;>
&lt;!ATTLIST xsl:copy
  %space-att;
  use-attribute-sets %qnames; #IMPLIED
>

&lt;!ELEMENT xsl:message %template;>
&lt;!ATTLIST xsl:message
  %space-att;
  terminate (yes|no) "no"
>

&lt;!ELEMENT xsl:fallback %template;>
&lt;!ATTLIST xsl:fallback %space-att;></pre></td></tr></table>

</div>

<div class='div1'>
<h2><a name='id28'></a>D Examples (Non-Normative)</h2>

<div class='div2'>
<h3><a name='id29'></a>D.1 Document Example</h3>

<p>This example is a stylesheet for transforming documents that
conform to a simple DTD into XHTML <a href='#XHTML'>[XHTML]</a>.  The DTD
is:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;!ELEMENT doc (title, chapter*)>
&lt;!ELEMENT chapter (title, (para|note)*, section*)>
&lt;!ELEMENT section (title, (para|note)*)>
&lt;!ELEMENT title (#PCDATA|emph)*>
&lt;!ELEMENT para (#PCDATA|emph)*>
&lt;!ELEMENT note (#PCDATA|emph)*>
&lt;!ELEMENT emph (#PCDATA|emph)*></pre></td></tr></table>

<p>The stylesheet is:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns="http://www.w3.org/TR/xhtml1/strict">

&lt;xsl:strip-space elements="doc chapter section"/>
&lt;xsl:output
   method="xml"
   indent="yes"
   encoding="iso-8859-1"
/>

&lt;xsl:template match="doc">
 &lt;html>
   &lt;head>
     &lt;title>
       &lt;xsl:value-of select="title"/>
     &lt;/title>
   &lt;/head>
   &lt;body>
     &lt;xsl:apply-templates/>
   &lt;/body>
 &lt;/html>
&lt;/xsl:template>

&lt;xsl:template match="doc/title">
  &lt;h1>
    &lt;xsl:apply-templates/>
  &lt;/h1>
&lt;/xsl:template>

&lt;xsl:template match="chapter/title">
  &lt;h2>
    &lt;xsl:apply-templates/>
  &lt;/h2>
&lt;/xsl:template>

&lt;xsl:template match="section/title">
  &lt;h3>
    &lt;xsl:apply-templates/>
  &lt;/h3>
&lt;/xsl:template>

&lt;xsl:template match="para">
  &lt;p>
    &lt;xsl:apply-templates/>
  &lt;/p>
&lt;/xsl:template>

&lt;xsl:template match="note">
  &lt;p class="note">
    &lt;b>NOTE: &lt;/b>
    &lt;xsl:apply-templates/>
  &lt;/p>
&lt;/xsl:template>

&lt;xsl:template match="emph">
  &lt;em>
    &lt;xsl:apply-templates/>
  &lt;/em>
&lt;/xsl:template>

&lt;/xsl:stylesheet></pre></td></tr></table>

<p>With the following input document</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;!DOCTYPE doc SYSTEM "doc.dtd">
&lt;doc>
&lt;title>Document Title&lt;/title>
&lt;chapter>
&lt;title>Chapter Title&lt;/title>
&lt;section>
&lt;title>Section Title&lt;/title>
&lt;para>This is a test.&lt;/para>
&lt;note>This is a note.&lt;/note>
&lt;/section>
&lt;section>
&lt;title>Another Section Title&lt;/title>
&lt;para>This is &lt;emph>another&lt;/emph> test.&lt;/para>
&lt;note>This is another note.&lt;/note>
&lt;/section>
&lt;/chapter>
&lt;/doc></pre></td></tr></table>

<p>it would produce the following result</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;?xml version="1.0" encoding="iso-8859-1"?>
&lt;html xmlns="http://www.w3.org/TR/xhtml1/strict">
&lt;head>
&lt;title>Document Title&lt;/title>
&lt;/head>
&lt;body>
&lt;h1>Document Title&lt;/h1>
&lt;h2>Chapter Title&lt;/h2>
&lt;h3>Section Title&lt;/h3>
&lt;p>This is a test.&lt;/p>
&lt;p class="note">
&lt;b>NOTE: &lt;/b>This is a note.&lt;/p>
&lt;h3>Another Section Title&lt;/h3>
&lt;p>This is &lt;em>another&lt;/em> test.&lt;/p>
&lt;p class="note">
&lt;b>NOTE: &lt;/b>This is another note.&lt;/p>
&lt;/body>
&lt;/html></pre></td></tr></table>

</div>

<div class='div2'>
<h3><a name='data-example'></a>D.2 Data Example</h3>

<p>This is an example of transforming some data represented in XML
using three different XSLT stylesheets to produce three different
representations of the data, HTML, SVG and VRML.</p>

<p>The input data is:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;sales>

        &lt;division id="North">
                &lt;revenue>10&lt;/revenue>
                &lt;growth>9&lt;/growth>
                &lt;bonus>7&lt;/bonus>
        &lt;/division>

        &lt;division id="South">
                &lt;revenue>4&lt;/revenue>
                &lt;growth>3&lt;/growth>
                &lt;bonus>4&lt;/bonus>
        &lt;/division>

        &lt;division id="West">
                &lt;revenue>6&lt;/revenue>
                &lt;growth>-1.5&lt;/growth>
                &lt;bonus>2&lt;/bonus>
        &lt;/division>

&lt;/sales></pre></td></tr></table>

<p>The following stylesheet, which uses the simplified syntax
described in <a href='#result-element-stylesheet'><b>2.3 Literal Result Element as Stylesheet</b></a>, transforms
the data into HTML:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;html xsl:version="1.0"
      xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
      lang="en">
    &lt;head>
	&lt;title>Sales Results By Division&lt;/title>
    &lt;/head>
    &lt;body>
	&lt;table border="1">
	    &lt;tr>
		&lt;th>Division&lt;/th>
		&lt;th>Revenue&lt;/th>
		&lt;th>Growth&lt;/th>
		&lt;th>Bonus&lt;/th>
	    &lt;/tr>
	    &lt;xsl:for-each select="sales/division">
		&lt;!-- order the result by revenue -->
		&lt;xsl:sort select="revenue"
			  data-type="number"
			  order="descending"/>
		&lt;tr>
		    &lt;td>
			&lt;em>&lt;xsl:value-of select="@id"/>&lt;/em>
		    &lt;/td>
		    &lt;td>
			&lt;xsl:value-of select="revenue"/>
		    &lt;/td>
		    &lt;td>
			&lt;!-- highlight negative growth in red -->
			&lt;xsl:if test="growth &amp;lt; 0">
			     &lt;xsl:attribute name="style">
				 &lt;xsl:text>color:red&lt;/xsl:text>
			     &lt;/xsl:attribute>
			&lt;/xsl:if>
			&lt;xsl:value-of select="growth"/>
		    &lt;/td>
		    &lt;td>
			&lt;xsl:value-of select="bonus"/>
		    &lt;/td>
		&lt;/tr>
	    &lt;/xsl:for-each>
	&lt;/table>
    &lt;/body>
&lt;/html></pre></td></tr></table>

<p>The HTML output is:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;html lang="en">
&lt;head>
&lt;meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">
&lt;title>Sales Results By Division&lt;/title>
&lt;/head>
&lt;body>
&lt;table border="1">
&lt;tr>
&lt;th>Division&lt;/th>&lt;th>Revenue&lt;/th>&lt;th>Growth&lt;/th>&lt;th>Bonus&lt;/th>
&lt;/tr>
&lt;tr>
&lt;td>&lt;em>North&lt;/em>&lt;/td>&lt;td>10&lt;/td>&lt;td>9&lt;/td>&lt;td>7&lt;/td>
&lt;/tr>
&lt;tr>
&lt;td>&lt;em>West&lt;/em>&lt;/td>&lt;td>6&lt;/td>&lt;td style="color:red">-1.5&lt;/td>&lt;td>2&lt;/td>
&lt;/tr>
&lt;tr>
&lt;td>&lt;em>South&lt;/em>&lt;/td>&lt;td>4&lt;/td>&lt;td>3&lt;/td>&lt;td>4&lt;/td>
&lt;/tr>
&lt;/table>
&lt;/body>
&lt;/html></pre></td></tr></table>

<p>The following stylesheet transforms the data into SVG:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns="http://www.w3.org/Graphics/SVG/SVG-19990812.dtd">

&lt;xsl:output method="xml" indent="yes" media-type="image/svg"/>

&lt;xsl:template match="/">

&lt;svg width = "3in" height="3in">
    &lt;g style = "stroke: #000000"> 
        &lt;!-- draw the axes -->
        &lt;line x1="0" x2="150" y1="150" y2="150"/>
        &lt;line x1="0" x2="0" y1="0" y2="150"/>
        &lt;text x="0" y="10">Revenue&lt;/text>
        &lt;text x="150" y="165">Division&lt;/text>
        &lt;xsl:for-each select="sales/division">
	    &lt;!-- define some useful variables -->

	    &lt;!-- the bar's x position -->
	    &lt;xsl:variable name="pos"
	                  select="(position()*40)-30"/>

	    &lt;!-- the bar's height -->
	    &lt;xsl:variable name="height"
	                  select="revenue*10"/>

	    &lt;!-- the rectangle -->
	    &lt;rect x="{$pos}" y="{150-$height}"
                  width="20" height="{$height}"/>

	    &lt;!-- the text label -->
	    &lt;text x="{$pos}" y="165">
	        &lt;xsl:value-of select="@id"/>
	    &lt;/text> 

	    &lt;!-- the bar value -->
	    &lt;text x="{$pos}" y="{145-$height}">
	        &lt;xsl:value-of select="revenue"/>
	    &lt;/text>
        &lt;/xsl:for-each>
    &lt;/g>
&lt;/svg>

&lt;/xsl:template>
&lt;/xsl:stylesheet></pre></td></tr></table>

<p>The SVG output is:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;svg width="3in" height="3in"
     xmlns="http://www.w3.org/Graphics/SVG/svg-19990412.dtd">
    &lt;g style="stroke: #000000">
	&lt;line x1="0" x2="150" y1="150" y2="150"/>
	&lt;line x1="0" x2="0" y1="0" y2="150"/>
	&lt;text x="0" y="10">Revenue&lt;/text>
	&lt;text x="150" y="165">Division&lt;/text>
	&lt;rect x="10" y="50" width="20" height="100"/>
	&lt;text x="10" y="165">North&lt;/text>
	&lt;text x="10" y="45">10&lt;/text>
	&lt;rect x="50" y="110" width="20" height="40"/>
	&lt;text x="50" y="165">South&lt;/text>
	&lt;text x="50" y="105">4&lt;/text>
	&lt;rect x="90" y="90" width="20" height="60"/>
	&lt;text x="90" y="165">West&lt;/text>
	&lt;text x="90" y="85">6&lt;/text>
    &lt;/g>
&lt;/svg></pre></td></tr></table>

<p>The following stylesheet transforms the data into VRML:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>&lt;xsl:stylesheet version="1.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

&lt;!-- generate text output as mime type model/vrml, using default charset -->
&lt;xsl:output method="text" encoding="UTF-8" media-type="model/vrml"/>  

        &lt;xsl:template match="/">#VRML V2.0 utf8 
 
# externproto definition of a single bar element 
EXTERNPROTO bar [ 
  field SFInt32 x  
  field SFInt32 y  
  field SFInt32 z  
  field SFString name  
  ] 
  "http://www.vrml.org/WorkingGroups/dbwork/barProto.wrl" 
 
# inline containing the graph axes 
Inline {  
        url "http://www.vrml.org/WorkingGroups/dbwork/barAxes.wrl" 
        } 
        
                &lt;xsl:for-each select="sales/division">
bar {
        x &lt;xsl:value-of select="revenue"/>
        y &lt;xsl:value-of select="growth"/>
        z &lt;xsl:value-of select="bonus"/>
        name "&lt;xsl:value-of select="@id"/>" 
        }
                &lt;/xsl:for-each>
        
        &lt;/xsl:template> 
 
&lt;/xsl:stylesheet></pre></td></tr></table>

<p>The VRML output is:</p>

<table class='eg' border='1' summary='Example' width='100%' cellpadding='5' bgcolor='#99ffff'><tr><td><pre>#VRML V2.0 utf8 
 
# externproto definition of a single bar element 
EXTERNPROTO bar [ 
  field SFInt32 x  
  field SFInt32 y  
  field SFInt32 z  
  field SFString name  
  ] 
  "http://www.vrml.org/WorkingGroups/dbwork/barProto.wrl" 
 
# inline containing the graph axes 
Inline {  
        url "http://www.vrml.org/WorkingGroups/dbwork/barAxes.wrl" 
        } 
        
                
bar {
        x 10
        y 9
        z 7
        name "North" 
        }
                
bar {
        x 4
        y 3
        z 4
        name "South" 
        }
                
bar {
        x 6
        y -1.5
        z 2
        name "West" 
        }</pre></td></tr></table>

</div>

</div>

<div class='div1'>
<h2><a name='id30'></a>E Acknowledgements (Non-Normative)</h2>
<p>The following have contributed to authoring this draft:</p>
<blockquote><p>
Daniel Lipkin, Saba
Jonathan Marsh, Microsoft
Henry Thompson, University of Edinburgh
Norman Walsh, Arbortext
Steve Zilles, Adobe
</p></blockquote>

<p>This specification was developed and approved for publication by the
W3C XSL Working Group (WG). WG approval of this specification does not
necessarily imply that all WG members voted for its approval. The
current members of the XSL WG are:</p>

<ul>
<li>
Sharon Adler
, IBM
 (<i>Co-Chair</i>) 
</li>
<li>
Anders Berglund
, IBM
</li>
<li>
Perin Blanchard
, Novell
</li>
<li>
Scott Boag
, Lotus 
</li>
<li>
Larry Cable
, Sun
</li>
<li>
Jeff Caruso
, Bitstream
</li>
<li>
James Clark
</li>
<li>
Peter Danielsen
, Bell Labs
</li>
<li>
Don Day
, IBM
</li>
<li>
Stephen Deach
, Adobe
</li>
<li>
Dwayne Dicks
, SoftQuad
</li>
<li>
Andrew Greene
, Bitstream
</li>
<li>
Paul Grosso
, Arbortext
</li>
<li>
Eduardo Gutentag
, Sun
</li>
<li>
Juliane Harbarth
, Software AG
</li>
<li>
Mickey Kimchi
, Enigma
</li>
<li>
Chris Lilley
, W3C
</li>
<li>
Chris Maden
, Exemplary Technologies
</li>
<li>
Jonathan Marsh
, Microsoft
</li>
<li>
Alex Milowski 
, Lexica
</li>
<li>
Steve Muench
, Oracle
</li>
<li>
Scott Parnell
, Xerox
</li>
<li>
Vincent Quint
, W3C
</li>
<li>
Dan Rapp
, Novell
</li>
<li>
Gregg Reynolds
, Datalogics
</li>
<li>
Jonathan Robie
, Software AG
</li>
<li>
Mark Scardina
, Oracle
</li>
<li>
Henry Thompson
, University of Edinburgh
</li>
<li>
Philip Wadler
, Bell Labs
</li>
<li>
Norman Walsh
, Arbortext
</li>
<li>
Sanjiva Weerawarana
, IBM
</li>
<li>
Steve Zilles
, Adobe
 (<i>Co-Chair</i>) 
</li>
</ul>

</div>

<div class='div1'>
<h2><a name='id31'></a>F Changes from Proposed Recommendation (Non-Normative)</h2>

<p>The following are the changes since the Proposed Recommendation:</p>

<ul>

<li><p>The <code>xsl:version</code> attribute is required on a
literal result element used as a stylesheet (see <a href='#result-element-stylesheet'><b>2.3 Literal Result Element as Stylesheet</b></a>).</p></li>

<li><p>The <code>data-type</code> attribute on <code>xsl:sort</code>
can use a prefixed name to specify a data-type not defined by
XSLT (see <a href='#sorting'><b>10 Sorting</b></a>).</p></li>

</ul>

</div>

<div class='div1'>
<h2><a name='id32'></a>G Features under Consideration for Future Versions of XSLT (Non-Normative)</h2>

<p>The following features are under consideration for versions of XSLT
after XSLT 1.0:</p>

<ul>

<li><p>a conditional expression;</p></li>

<li><p>support for XML Schema datatypes and archetypes;</p></li>

<li><p>support for something like style rules in the original XSL
submission;</p></li>

<li><p>an attribute to control the default namespace for names
occurring in XSLT attributes;</p></li>

<li><p>support for entity references;</p></li>

<li><p>support for DTDs in the data model;</p></li>

<li><p>support for notations in the data model;</p></li>

<li><p>a way to get back from an element to the elements that
reference it (e.g. by IDREF attributes);</p></li>

<li><p>an easier way to get an ID or key in another document;</p></li>

<li><p>support for regular expressions for matching against any or
all of text nodes, attribute values, attribute names, element type
names;</p></li>

<li><p>case-insensitive comparisons;</p></li>

<li><p>normalization of strings before comparison, for example for
compatibility characters;</p></li>

<li><p>a function <code>string resolve(node-set)</code> function
that treats the value of the argument as a relative URI and turns it
into an absolute URI using the base URI of the node;</p></li>

<li><p>multiple result documents;</p></li>

<li><p>defaulting the <code>select</code> attribute on
<code>xsl:value-of</code> to the current node;</p></li>

<li><p>an attribute on <code>xsl:attribute</code> to control how the
attribute value is normalized;</p></li>

<li><p>additional attributes on <code>xsl:sort</code> to provide
further control over sorting, such as relative order of
scripts;</p></li>

<li><p>a way to put the text of a resource identified by a URI into
the result tree;</p></li>

<li><p>allow unions in steps (e.g. <code>foo/(bar|baz)</code>);</p></li>

<li><p>allow for result tree fragments all operations that are
allowed for node-sets;</p></li>

<li><p>a way to group together consecutive nodes having duplicate
subelements or attributes;</p></li>

<li><p>features to make handling of the HTML <code>style</code>
attribute more convenient.</p></li>

</ul>

</div>

</div>
</body></html>"""

#"

# We need to have generate-id return the same id in different runs
from xml.xslt import XsltException, Error, XsltFunctions
from xml.xpath import Util
g_ctr = 1
g_map = {}
def GenerateId(context, nodeSet=None):
    gen_id = XsltFunctions.GenerateId(context, nodeSet)
    if g_map.has_key(gen_id):
        return g_map[gen_id]
    global g_ctr
    id = 'id%d' % g_ctr
    g_ctr = g_ctr + 1
    g_map[gen_id] = id
    return id
    
ExtFunctions = {('', 'generate-id') : GenerateId}

def Test(tester):
    source = test_harness.FileInfo(uri='Xslt/Borrowed/REC-xslt-19991116.xml')
    sty = test_harness.FileInfo(uri='Xslt/Borrowed/w3c-xmlspec.xslt')
    xtest = test_harness.XsltTester(source,
                                    [sty],
                                    expected,
                                    extensionModules=[__name__],
                                    )

    xtest.test(tester)
    return
