
/*****************************************************************************
*
* Copyright (c) 2003-2016 by The University of Queensland
* http://www.uq.edu.au
*
* Primary Business: Queensland, Australia
* Licensed under the Apache License, version 2.0
* http://www.apache.org/licenses/LICENSE-2.0
*
* Development until 2012 by Earth Systems Science Computational Center (ESSCC)
* Development 2012-2013 by School of Earth Sciences
* Development from 2014 by Centre for Geoscience Computing (GeoComp)
*
*****************************************************************************/


/****************************************************************************

  Finley: Reference elements

*****************************************************************************/

#include "ReferenceElements.h"

#include <algorithm> // std::max
#include <cstring>

namespace finley {

/// this list has been generated by generateReferenceElementList.py
const ReferenceElementInfo ReferenceElement_InfoList[] = {
{ Point1, "Point1", 1, 1, 1, { 0, 1 }, Point1,
    { 0 }, PointQuad, Point1Shape, Point1Shape,
    { 0 },
  1, { 0 },
  1, { 0 },
    { 0 },
    { -1 } },
{ Line2, "Line2", 2, 1, 1, { 0, 2 }, Line2,
    { 0, 1 }, LineQuad, Line2Shape, Line2Shape,
    { 0, 1 },
  2, { 0, 1 },
  2, { 0, 1 },
    { 1, 0 },
    { -1 } },
{ Line3, "Line3", 3, 1, 1, { 0, 3 }, Line2,
    { 0, 1 }, LineQuad, Line3Shape, Line3Shape,
    { 0, 1, 2 },
  3, { 0, 1, 2 },
  3, { 0, 1, 2 },
    { 1, 0, 2 },
    { -1 } },
{ Line4, "Line4", 4, 1, 1, { 0, 4 }, Line2,
    { 0, 1 }, LineQuad, Line4Shape, Line4Shape,
    { 0, 1, 2, 3 },
  4, { 0, 1, 2, 3 },
  4, { 0, 1, 2, 3 },
    { 1, 0, 3, 2 },
    { -1 } },
{ Tri3, "Tri3", 3, 1, 1, { 0, 3 }, Tri3,
    { 0, 1, 2 }, TriQuad, Tri3Shape, Tri3Shape,
    { 0, 1, 2 },
  3, { 0, 1, 2 },
  3, { 0, 1, 2 },
    { 1, 2, 0 },
    { 0, 2, 1 } },
{ Tri6, "Tri6", 6, 1, 1, { 0, 6 }, Tri3,
    { 0, 1, 2 }, TriQuad, Tri6Shape, Tri6Shape,
    { 0, 1, 2, 3, 4, 5 },
  6, { 0, 1, 2, 3, 4, 5 },
  6, { 0, 1, 2, 3, 4, 5 },
    { 1, 2, 0, 4, 5, 3 },
    { 0, 2, 1, 5, 4, 3 } },
{ Tri9, "Tri9", 9, 1, 1, { 0, 9 }, Tri3,
    { 0, 1, 2 }, TriQuad, Tri9Shape, Tri9Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
  9, { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
  9, { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
    { 1, 2, 0, 5, 6, 7, 8, 3, 4 },
    { 0, 2, 1, 8, 7, 6, 5, 4, 3 } },
{ Tri10, "Tri10", 10, 1, 1, { 0, 10 }, Tri3,
    { 0, 1, 2 }, TriQuad, Tri10Shape, Tri10Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  10, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  10, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
    { 1, 2, 0, 5, 6, 7, 8, 3, 4, 9 },
    { 0, 2, 1, 8, 7, 6, 5, 4, 3, 9 } },
{ Rec4, "Rec4", 4, 1, 1, { 0, 4 }, Rec4,
    { 0, 1, 2, 3 }, RecQuad, Rec4Shape, Rec4Shape,
    { 0, 1, 2, 3 },
  4, { 0, 1, 2, 3 },
  4, { 0, 1, 2, 3 },
    { 1, 2, 3, 0 },
    { 0, 3, 2, 1 } },
{ Rec8, "Rec8", 8, 1, 1, { 0, 8 }, Rec4,
    { 0, 1, 2, 3 }, RecQuad, Rec8Shape, Rec8Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7 },
  8, { 0, 1, 2, 3, 4, 5, 6, 7 },
  8, { 0, 1, 2, 3, 4, 5, 6, 7 },
    { 1, 2, 3, 0, 5, 6, 7, 4 },
    { 0, 3, 2, 1, 7, 6, 5, 4 } },
{ Rec9, "Rec9", 9, 1, 1, { 0, 9 }, Rec4,
    { 0, 1, 2, 3 }, RecQuad, Rec9Shape, Rec9Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
  9, { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
  9, { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
    { 1, 2, 3, 0, 5, 6, 7, 4, 8 },
    { 0, 3, 2, 1, 7, 6, 5, 4, 8 } },
{ Rec12, "Rec12", 12, 1, 1, { 0, 12 }, Rec4,
    { 0, 1, 2, 3 }, RecQuad, Rec12Shape, Rec12Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 },
  12, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 },
  12, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 },
    { 1, 2, 3, 0, 6, 7, 8, 9, 10, 11, 4, 5 },
    { 0, 3, 2, 1, 11, 10, 9, 8, 7, 6, 5, 4 } },
{ Rec16, "Rec16", 16, 1, 1, { 0, 16 }, Rec4,
    { 0, 1, 2, 3 }, RecQuad, Rec16Shape, Rec16Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  16, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  16, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
    { 1, 2, 3, 0, 6, 7, 8, 9, 10, 11, 4, 5, 13, 14, 15, 12 },
    { 0, 3, 2, 1, 11, 10, 9, 8, 7, 6, 5, 4, 12, 15, 14, 13 } },
{ Tet4, "Tet4", 4, 1, 1, { 0, 4 }, Tet4,
    { 0, 1, 2, 3 }, TetQuad, Tet4Shape, Tet4Shape,
    { 0, 1, 2, 3 },
  4, { 0, 1, 2, 3 },
  4, { 0, 1, 2, 3 },
    { -1 },
    { -1 } },
{ Tet10, "Tet10", 10, 1, 1, { 0, 10 }, Tet4,
    { 0, 1, 2, 3 }, TetQuad, Tet10Shape, Tet10Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  10, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  10, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
    { -1 },
    { -1 } },
{ Tet16, "Tet16", 16, 1, 1, { 0, 16 }, Tet4,
    { 0, 1, 2, 3 }, TetQuad, Tet16Shape, Tet16Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  16, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  16, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
    { -1 },
    { -1 } },
{ Hex8, "Hex8", 8, 1, 1, { 0, 8 }, Hex8,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, HexQuad, Hex8Shape, Hex8Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7 },
  8, { 0, 1, 2, 3, 4, 5, 6, 7 },
  8, { 0, 1, 2, 3, 4, 5, 6, 7 },
    { -1 },
    { -1 } },
{ Hex20, "Hex20", 20, 1, 1, { 0, 20 }, Hex8,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, HexQuad, Hex20Shape, Hex20Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 },
  20, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 },
  20, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 },
    { -1 },
    { -1 } },
{ Hex27, "Hex27", 27, 1, 1, { 0, 27 }, Hex8,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, HexQuad, Hex27Shape, Hex27Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26 },
  27, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26 },
  27, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26 },
    { -1 },
    { -1 } },
{ Hex32, "Hex32", 32, 1, 1, { 0, 32 }, Hex8,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, HexQuad, Hex32Shape, Hex32Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31 },
  32, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31 },
  32, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31 },
    { -1 },
    { -1 } },
{ Line2Face, "Line2Face", 2, 1, 1, { 0, 2 }, Line2Face,
    { 0, 1 }, PointQuad, Line2Shape, Line2Shape,
    { 0, 1 },
  1, { 0 },
  1, { 0 },
    { 0, 1, 2 },
    { -1 } },
{ Line3Face, "Line3Face", 3, 1, 1, { 0, 3 }, Line2Face,
    { 0, 1 }, PointQuad, Line3Shape, Line3Shape,
    { 0, 1, 2 },
  1, { 0 },
  1, { 0 },
    { 0, 1, 2 },
    { -1 } },
{ Line4Face, "Line4Face", 4, 1, 1, { 0, 4 }, Line2Face,
    { 0, 1 }, PointQuad, Line4Shape, Line4Shape,
    { 0, 1, 2, 3 },
  1, { 0 },
  1, { 0 },
    { 0, 1, 2 },
    { -1 } },
{ Tri3Face, "Tri3Face", 3, 1, 1, { 0, 3 }, Tri3Face,
    { 0, 1, 2 }, LineQuad, Tri3Shape, Tri3Shape,
    { 0, 1, 2 },
  2, { 0, 1 },
  2, { 0, 1 },
    { 1, 0, 2 },
    { -1 } },
{ Tri6Face, "Tri6Face", 6, 1, 1, { 0, 6 }, Tri3Face,
    { 0, 1, 2 }, LineQuad, Tri6Shape, Tri6Shape,
    { 0, 1, 2, 3, 4, 5 },
  3, { 0, 1, 3 },
  3, { 0, 1, 3 },
    { 1, 0, 2, 3, 5, 4 },
    { -1 } },
{ Tri9Face, "Tri9Face", 9, 1, 1, { 0, 9 }, Tri3Face,
    { 0, 1, 2 }, LineQuad, Tri9Shape, Tri9Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
  4, { 0, 1, 3, 4 },
  4, { 0, 1, 3, 4 },
    { 1, 0, 2, 4, 3, 7, 8, 6, 5 },
    { -1 } },
{ Tri10Face, "Tri10Face", 10, 1, 1, { 0, 10 }, Tri3Face,
    { 0, 1, 2 }, LineQuad, Tri10Shape, Tri10Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  4, { 0, 1, 3, 4 },
  4, { 0, 1, 3, 4 },
    { 1, 0, 2, 4, 3, 7, 8, 6, 5, 9 },
    { -1 } },
{ Rec4Face, "Rec4Face", 4, 1, 1, { 0, 4 }, Rec4Face,
    { 0, 1, 2, 3 }, LineQuad, Rec4Shape, Rec4Shape,
    { 0, 1, 2, 3 },
  2, { 0, 1 },
  2, { 0, 1 },
    { 1, 0, 3, 2 },
    { -1 } },
{ Rec8Face, "Rec8Face", 8, 1, 1, { 0, 8 }, Rec4Face,
    { 0, 1, 2, 3 }, LineQuad, Rec8Shape, Rec8Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7 },
  3, { 0, 1, 4 },
  3, { 0, 1, 4 },
    { 1, 0, 3, 2, 4, 7, 6, 5 },
    { -1 } },
{ Rec9Face, "Rec9Face", 9, 1, 1, { 0, 9 }, Rec4Face,
    { 0, 1, 2, 3 }, LineQuad, Rec9Shape, Rec9Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
  3, { 0, 1, 4 },
  3, { 0, 1, 4 },
    { 1, 0, 3, 2, 4, 7, 6, 5, 8 },
    { -1 } },
{ Rec12Face, "Rec12Face", 12, 1, 1, { 0, 12 }, Rec4Face,
    { 0, 1, 2, 3 }, LineQuad, Rec12Shape, Rec12Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 },
  4, { 0, 1, 4, 5 },
  4, { 0, 1, 4, 5 },
    { 1, 0, 3, 2, 5, 4, 11, 10, 9, 8, 7, 6 },
    { -1 } },
{ Rec16Face, "Rec16Face", 16, 1, 1, { 0, 16 }, Rec4Face,
    { 0, 1, 2, 3 }, LineQuad, Rec16Shape, Rec16Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  4, { 0, 1, 4, 5 },
  4, { 0, 1, 4, 5 },
    { 1, 0, 3, 2, 5, 4, 11, 10, 9, 8, 7, 6, 13, 12, 15, 14 },
    { -1 } },
{ Tet4Face, "Tet4Face", 4, 1, 1, { 0, 4 }, Tet4Face,
    { 0, 1, 2, 3 }, TriQuad, Tet4Shape, Tet4Shape,
    { 0, 1, 2, 3 },
  3, { 0, 1, 2 },
  4, { 0, 1, 2, 3 },
    { 1, 2, 0, 3 },
    { 0, 2, 1, 3 } },
{ Tet10Face, "Tet10Face", 10, 1, 1, { 0, 10 }, Tet4Face,
    { 0, 1, 2, 3 }, TriQuad, Tet10Shape, Tet10Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  6, { 0, 1, 2, 4, 5, 6 },
  6, { 0, 1, 2, 4, 5, 6 },
    { 1, 2, 0, 3, 5, 6, 4, 8, 9, 7 },
    { 0, 2, 1, 3, 6, 7, 9, 8 } },
{ Tet16Face, "Tet16Face", 16, 1, 1, { 0, 16 }, Tet4Face,
    { 0, 1, 2, 3 }, TriQuad, Tet16Shape, Tet16Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  9, { 0, 1, 2, 4, 5, 6, 7, 8, 9 },
  9, { 0, 1, 2, 4, 5, 6, 7, 8, 9 },
    { 1, 2, 0, 3, 6, 7, 8, 9, 4, 5, 11, 12, 10, 14, 15, 13 },
    { 0, 2, 1, 3, 9, 8, 7, 6, 5, 4, 9, 8, 7, 6, 10, 12, 11, 13, 15, 14 } },
{ Hex8Face, "Hex8Face", 8, 1, 1, { 0, 8 }, Hex8Face,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, RecQuad, Hex8Shape, Hex8Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7 },
  4, { 0, 1, 2, 3 },
  4, { 0, 1, 2, 3 },
    { 1, 2, 3, 0, 5, 6, 7, 4 },
    { 0, 3, 2, 1, 4, 7, 6, 5 } },
{ Hex20Face, "Hex20Face", 20, 1, 1, { 0, 20 }, Hex8Face,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, RecQuad, Hex20Shape, Hex20Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 },
  8, { 0, 1, 2, 3, 8, 9, 10, 11 },
  8, { 0, 1, 2, 3, 8, 9, 10, 11 },
    { 1, 2, 3, 0, 5, 6, 7, 4, 9, 10, 11, 8, 13, 14, 15, 12, 17, 18, 19, 16 },
    { 0, 3, 2, 1, 4, 7, 6, 5, 11, 10, 9, 8, 12, 15, 14, 13, 19, 18, 17, 16 } },
{ Hex27Face, "Hex27Face", 27, 1, 1, { 0, 27 }, Hex8Face,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, RecQuad, Hex27Shape, Hex27Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26 },
  9, { 0, 1, 2, 3, 8, 9, 10, 11, 20 },
  9, { 0, 1, 2, 3, 8, 9, 10, 11, 20 },
    { 1, 2, 3, 0, 5, 6, 7, 4, 9, 10, 11, 8, 13, 14, 15, 12, 17, 18, 19, 16, 20, 22, 23, 24, 22, 25, 26 },
    { 0, 3, 2, 1, 4, 7, 6, 5, 11, 10, 9, 8, 12, 15, 14, 13, 19, 18, 17, 16, 20, 24, 23, 22, 21, 25, 26 } },
{ Hex32Face, "Hex32Face", 32, 1, 1, { 0, 32 }, Hex8Face,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, RecQuad, Hex32Shape, Hex32Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31 },
  12, { 0, 1, 2, 3, 8, 9, 10, 11, 12, 13, 14, 15 },
  12, { 0, 1, 2, 3, 8, 9, 10, 11, 12, 13, 14, 15 },
    { 1, 2, 3, 0, 5, 6, 7, 4, 10, 11, 12, 13, 14, 15, 8, 9, 17, 18, 19, 16, 21, 22, 23, 20, 26, 27, 28, 29, 30, 31, 34, 25 },
    { 0, 3, 2, 1, 4, 7, 6, 5, 15, 14, 13, 12, 11, 10, 9, 8, 16, 19, 18, 17, 20, 23, 22, 21, 31, 30, 29, 28, 27, 26, 25, 24 } },
{ Point1_Contact, "Point1_Contact", 2, 1, 2, { 0, 1, 2 }, Point1_Contact,
    { 0, 1 }, PointQuad, Point1Shape, Point1Shape,
    { 0, 1 },
  1, { 0 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Line2_Contact, "Line2_Contact", 4, 1, 2, { 0, 2, 4 }, Line2_Contact,
    { 0, 1, 2, 3 }, LineQuad, Line2Shape, Line2Shape,
    { 0, 1, 2, 3 },
  2, { 0, 1 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Line3_Contact, "Line3_Contact", 6, 1, 2, { 0, 3, 6 }, Line2_Contact,
    { 0, 1, 3, 4 }, LineQuad, Line3Shape, Line3Shape,
    { 0, 1, 2, 3, 4, 5 },
  3, { 0, 1, 2 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Line4_Contact, "Line4_Contact", 8, 1, 2, { 0, 4, 8 }, Line2_Contact,
    { 0, 1, 4, 5 }, LineQuad, Line4Shape, Line4Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7 },
  4, { 0, 1, 2, 3 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tri3_Contact, "Tri3_Contact", 6, 1, 2, { 0, 3, 6 }, Tri3_Contact,
    { 0, 1, 2, 3, 4, 5 }, TriQuad, Tri3Shape, Tri3Shape,
    { 0, 1, 2, 3, 4, 5 },
  3, { 0, 1, 2 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tri6_Contact, "Tri6_Contact", 12, 1, 2, { 0, 6, 12 }, Tri3_Contact,
    { 0, 1, 2, 6, 7, 8 }, TriQuad, Tri6Shape, Tri6Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 },
  6, { 0, 1, 2, 3, 4, 5 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tri9_Contact, "Tri9_Contact", 18, 1, 2, { 0, 9, 18 }, Tri3_Contact,
    { 0, 1, 2, 9, 10, 11 }, TriQuad, Tri9Shape, Tri9Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17 },
  9, { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tri10_Contact, "Tri10_Contact", 20, 1, 2, { 0, 10, 20 }, Tri3_Contact,
    { 0, 1, 2, 10, 11, 12 }, TriQuad, Tri10Shape, Tri10Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 },
  10, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec4_Contact, "Rec4_Contact", 8, 1, 2, { 0, 4, 8 }, Rec4_Contact,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, RecQuad, Rec4Shape, Rec4Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7 },
  4, { 0, 1, 2, 3 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec8_Contact, "Rec8_Contact", 16, 1, 2, { 0, 8, 16 }, Rec4_Contact,
    { 0, 1, 2, 3, 8, 9, 10, 11 }, RecQuad, Rec8Shape, Rec8Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  8, { 0, 1, 2, 3, 4, 5, 6, 7 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec9_Contact, "Rec9_Contact", 18, 1, 2, { 0, 9, 18 }, Rec4_Contact,
    { 0, 1, 2, 3, 9, 10, 11, 12 }, RecQuad, Rec9Shape, Rec9Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17 },
  9, { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec12_Contact, "Rec12_Contact", 24, 1, 2, { 0, 12, 24 }, Rec4_Contact,
    { 0, 1, 2, 3, 12, 13, 14, 15 }, RecQuad, Rec12Shape, Rec12Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23 },
  12, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec16_Contact, "Rec16_Contact", 32, 1, 2, { 0, 16, 32 }, Rec4_Contact,
    { 0, 1, 2, 3, 16, 17, 18, 19 }, RecQuad, Rec16Shape, Rec16Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31 },
  16, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Line2Face_Contact, "Line2Face_Contact", 4, 1, 2, { 0, 2, 4 }, Line2Face_Contact,
    { 0, 1, 2, 3 }, PointQuad, Line2Shape, Line2Shape,
    { 0, 1, 2, 3 },
  1, { 0 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Line3Face_Contact, "Line3Face_Contact", 6, 1, 2, { 0, 3, 6 }, Line2Face_Contact,
    { 0, 1, 3, 4 }, PointQuad, Line3Shape, Line3Shape,
    { 0, 1, 2, 3, 4, 5 },
  1, { 0 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Line4Face_Contact, "Line4Face_Contact", 8, 1, 2, { 0, 4, 8 }, Line2Face_Contact,
    { 0, 1, 4, 5 }, PointQuad, Line4Shape, Line4Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7 },
  1, { 0 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tri3Face_Contact, "Tri3Face_Contact", 6, 1, 2, { 0, 3, 6 }, Tri3Face_Contact,
    { 0, 1, 2, 3, 4, 5 }, LineQuad, Tri3Shape, Tri3Shape,
    { 0, 1, 2, 3, 4, 5 },
  2, { 0, 1 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tri6Face_Contact, "Tri6Face_Contact", 12, 1, 2, { 0, 6, 12 }, Tri3Face_Contact,
    { 0, 1, 2, 6, 7, 8 }, LineQuad, Tri6Shape, Tri6Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11 },
  3, { 0, 1, 3 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tri9Face_Contact, "Tri9Face_Contact", 18, 1, 2, { 0, 9, 18 }, Tri3Face_Contact,
    { 0, 1, 2, 9, 10, 11 }, LineQuad, Tri9Shape, Tri9Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17 },
  4, { 0, 1, 3, 4 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tri10Face_Contact, "Tri10Face_Contact", 20, 1, 2, { 0, 10, 20 }, Tri3Face_Contact,
    { 0, 1, 2, 10, 11, 12 }, LineQuad, Tri10Shape, Tri10Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 },
  4, { 0, 1, 3, 4 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec4Face_Contact, "Rec4Face_Contact", 8, 1, 2, { 0, 4, 8 }, Rec4Face_Contact,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, LineQuad, Rec4Shape, Rec4Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7 },
  2, { 0, 1 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec8Face_Contact, "Rec8Face_Contact", 16, 1, 2, { 0, 8, 16 }, Rec4Face_Contact,
    { 0, 1, 2, 3, 8, 9, 10, 11 }, LineQuad, Rec8Shape, Rec8Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  3, { 0, 1, 4 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec9Face_Contact, "Rec9Face_Contact", 18, 1, 2, { 0, 9, 18 }, Rec4Face_Contact,
    { 0, 1, 2, 3, 9, 10, 11, 12 }, LineQuad, Rec9Shape, Rec9Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17 },
  3, { 0, 1, 4 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec12Face_Contact, "Rec12Face_Contact", 24, 1, 2, { 0, 12, 24 }, Rec4Face_Contact,
    { 0, 1, 2, 3, 12, 13, 14, 15 }, LineQuad, Rec12Shape, Rec12Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23 },
  4, { 0, 1, 4, 5 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Rec16Face_Contact, "Rec16Face_Contact", 32, 1, 2, { 0, 16, 32 }, Rec4Face_Contact,
    { 0, 1, 2, 3, 16, 17, 18, 19 }, LineQuad, Rec16Shape, Rec16Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31 },
  4, { 0, 1, 4, 5 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tet4Face_Contact, "Tet4Face_Contact", 8, 1, 2, { 0, 4, 8 }, Tet4Face_Contact,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, TriQuad, Tet4Shape, Tet4Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7 },
  3, { 0, 1, 2 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tet10Face_Contact, "Tet10Face_Contact", 20, 1, 2, { 0, 10, 20 }, Tet4Face_Contact,
    { 0, 1, 2, 3, 10, 11, 12, 13 }, TriQuad, Tet10Shape, Tet10Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19 },
  6, { 0, 1, 2, 4, 5, 6 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Tet16Face_Contact, "Tet16Face_Contact", 32, 1, 2, { 0, 16, 32 }, Tet4Face_Contact,
    { 0, 1, 2, 3, 16, 17, 18, 19 }, TriQuad, Tet16Shape, Tet16Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31 },
  9, { 0, 1, 2, 4, 5, 6, 7, 8, 9 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Hex8Face_Contact, "Hex8Face_Contact", 16, 1, 2, { 0, 8, 16 }, Hex8Face_Contact,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 }, RecQuad, Hex8Shape, Hex8Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
  4, { 0, 1, 2, 3 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Hex20Face_Contact, "Hex20Face_Contact", 40, 1, 2, { 0, 20, 40 }, Hex8Face_Contact,
    { 0, 1, 2, 3, 4, 5, 6, 7, 20, 21, 22, 23, 24, 25, 26, 27 }, RecQuad, Hex20Shape, Hex20Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39 },
  8, { 0, 1, 2, 3, 8, 9, 10, 11 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Hex27Face_Contact, "Hex27Face_Contact", 54, 1, 2, { 0, 27, 54 }, Hex8Face_Contact,
    { 0, 1, 2, 3, 4, 5, 6, 7, 27, 28, 29, 30, 31, 32, 33, 34 }, RecQuad, Hex27Shape, Hex27Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53 },
  9, { 0, 1, 2, 3, 8, 9, 10, 11, 20 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Hex32Face_Contact, "Hex32Face_Contact", 64, 1, 2, { 0, 32, 64 }, Hex8Face_Contact,
    { 0, 1, 2, 3, 4, 5, 6, 7, 32, 33, 34, 35, 36, 37, 38, 39 }, RecQuad, Hex32Shape, Hex32Shape,
    { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63 },
  12, { 0, 1, 2, 3, 8, 9, 10, 11, 12, 13, 14, 15 },
  -1, { -1 },
    { -1 },
    { -1 } },
{ Line3Macro, "Line3Macro", 3, 2, 1, { 0, 3 }, Line2,
    { 0, 1 }, LineQuad, Line3Shape, Line2Shape,
    { 0, 2, 2, 1 },
  3, { 0, 1, 2 },
  3, { 0, 1, 2 },
    { 1, 0, 2 },
    { -1 } },
{ Tri6Macro, "Tri6Macro", 6, 4, 1, { 0, 6 }, Tri3,
    { 0, 1, 2 }, TriQuad, Tri6Shape, Tri3Shape,
    { 0, 3, 5, 5, 4, 2, 3, 1, 4, 4, 5, 3},
  6, { 0, 1, 2, 3, 4, 5 },
  6, { 0, 1, 2, 3, 4, 5 },
    { 1, 2, 0, 4, 5, 3 },
    { 0, 2, 1, 5, 4, 3 } },
{ Rec9Macro, "Rec9Macro", 9, 4, 1, { 0, 9 }, Rec4,
    { 0, 1, 2, 3 }, RecQuad, Rec9Shape, Rec4Shape,
    { 0, 4, 8, 7, 7, 8, 6, 3, 4, 1, 5, 8, 8, 5, 2, 6 },
  9, { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
  9, { 0, 1, 2, 3, 4, 5, 6, 7, 8 },
    { 1, 2, 3, 0, 5, 6, 7, 4, 8 },
    { 0, 3, 2, 1, 7, 6, 5, 4, 8 } },
{ Tet10Macro, "Tet10Macro", 10, 8, 1, { 0, 10 }, Tet4,
    { 0, 1, 2, 3 }, TetQuad, Tet10Shape, Tet4Shape,
    { 0, 4, 6, 7, 4, 1, 5, 8, 6, 5, 2, 9, 7, 8, 9, 3, 4, 5, 6, 8, 5, 9, 8, 6, 9, 7, 6, 8, 7, 4, 6, 8 },
  10, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  10, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
    { -1 },
    { -1 } },
{ Hex27Macro, "Hex27Macro", 27, 8, 1, { 0, 27 }, Hex8,
    { 0, 1, 2, 3, 4, 5, 6, 7 }, HexQuad, Hex27Shape, Hex8Shape,
    { 0, 8, 20, 11, 12, 21, 26, 24, 8, 1, 9, 20, 21, 13, 22, 26, 11, 20, 10, 3, 24, 26, 23, 15, 20, 9, 2, 10, 26, 22, 14, 23, 12, 21, 26, 24, 4, 16, 25, 19, 21, 13, 22, 26, 16, 5, 17, 25, 24, 26, 23, 15, 19, 25, 18, 7, 26, 22, 14, 23, 25, 17, 6, 18 },
  27, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26 },
  27, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26 },
    { -1 },
    { -1 } },
{ NoRef, "noElement", 0, 0, 0, { 0 }, NoRef,
    { 0 }, NoQuad, NoShape, NoShape,
    { 0 },
  -1, { 0 },
  -1, { 0 },
    { 0 },
    { 0 } }
};


/// creates a ReferenceElement of type id and a given integration order
ReferenceElement::ReferenceElement(ElementTypeId id, int order) :
    integrationOrder(-1),
    DBasisFunctionDv(NULL),
    DBasisFunctionDvShared(true)

{
    Type = getInfo(id);
    if (!Type) {
        throw escript::ValueError("ReferenceElement: unable to identify element type.");
    }
    LinearType = getInfo(Type->LinearTypeId);
    if (!LinearType) {
        throw escript::ValueError("ReferenceElement: unable to identify linear element type.");
    }

    const QuadInfo* quadscheme=QuadInfo_getInfo(Type->Quadrature);
    const ShapeFunctionInfo* parametrization=ShapeFunction::getInfo(Type->Parametrization);
    const ShapeFunctionInfo* basisfunction=ShapeFunction::getInfo(Type->BasisFunctions);
    const ShapeFunctionInfo* linearbasisfunction=ShapeFunction::getInfo(ReferenceElement_InfoList[Type->LinearTypeId].BasisFunctions);
    const int nsub = Type->numSubElements;

    // set up the basic integration scheme.
    // Note that quadscheme->numDim is not necessarily the dimension of the
    // element
    if (order < 0)
        order=std::max(2*basisfunction->numOrder, 0);

    integrationOrder=order;
    numLocalDim=quadscheme->numDim;
    numNodes=Type->numNodes;
    numLinearNodes=LinearType->numNodes;

    const int numQuadNodes=quadscheme->getNumQuadNodes(order);
    std::vector<double> quadNodes(numQuadNodes*quadscheme->numDim*nsub);
    std::vector<double> quadWeights(numQuadNodes*nsub);

    quadscheme->getQuadNodes(numQuadNodes, quadNodes, quadWeights);

    // Set the basis functions on the quadrature points.
    // Note: ShapeFunction constructor will introduce 0 if the dimensions of
    // the quadrature scheme and the dimensions of the element don't match.
    // Before we can set the shape function for the parametrization the
    // quadrature scheme needs to be replicated:
    if (nsub > 1) {
        DBasisFunctionDv=new double[numQuadNodes*nsub*(basisfunction->numShapes) * (basisfunction->numDim)];
        DBasisFunctionDvShared=false;
        BasisFunctions.reset(new ShapeFunction(basisfunction->TypeId, quadscheme->numDim, numQuadNodes, quadNodes, quadWeights));
        std::vector<double> quadNodes2(numQuadNodes*quadscheme->numDim*nsub);
        std::vector<double> quadWeights2(numQuadNodes*nsub);
        int numQuadNodes2=quadscheme->getMacro(nsub,
                BasisFunctions->numQuadNodes,
                &BasisFunctions->QuadNodes[0],
                &BasisFunctions->QuadWeights[0],
                BasisFunctions->Type->numShapes,
                &BasisFunctions->dSdv[0],
                numQuadNodes*nsub, &quadNodes2[0], &quadWeights2[0],
                DBasisFunctionDv);
        Parametrization.reset(new ShapeFunction(parametrization->TypeId, quadscheme->numDim, numQuadNodes2, quadNodes2, quadWeights2));
        LinearBasisFunctions.reset(new ShapeFunction(linearbasisfunction->TypeId, quadscheme->numDim, numQuadNodes2, quadNodes2, quadWeights2));
    } else {
        Parametrization.reset(new ShapeFunction(parametrization->TypeId, quadscheme->numDim, numQuadNodes*nsub, quadNodes, quadWeights));
        BasisFunctions.reset(new ShapeFunction(basisfunction->TypeId, quadscheme->numDim, numQuadNodes, quadNodes, quadWeights));
        LinearBasisFunctions.reset(new ShapeFunction(linearbasisfunction->TypeId, quadscheme->numDim, numQuadNodes, quadNodes, quadWeights));
        if (BasisFunctions->dSdv.size() > 0)
            DBasisFunctionDv=const_cast<double*>(&BasisFunctions->dSdv[0]);
        DBasisFunctionDvShared=true;
    }
}

ReferenceElement::~ReferenceElement()
{
    if (!DBasisFunctionDvShared)
        delete[] DBasisFunctionDv;
}

ElementTypeId ReferenceElement::getTypeId(const char* element_type)
{
    int ptr=0;
    ElementTypeId out=NoRef;
    while (ReferenceElement_InfoList[ptr].TypeId!=NoRef && out==NoRef) {
        if (strcmp(element_type, ReferenceElement_InfoList[ptr].Name)==0)
            out=ReferenceElement_InfoList[ptr].TypeId;
        ptr++;
    }
    return out;
}

const ReferenceElementInfo* ReferenceElement::getInfo(ElementTypeId id)
{
    int ptr=0;
    const ReferenceElementInfo* out=NULL;
    while (ReferenceElement_InfoList[ptr].TypeId!=NoRef && out==NULL) {
        if (ReferenceElement_InfoList[ptr].TypeId==id)
            out=&ReferenceElement_InfoList[ptr];
        ptr++;
    }
    if (out==NULL) {
        throw escript::ValueError("ReferenceElement::getInfo: cannot find requested reference element.");
    }
    return out;
}

} // namespace finley

