/* KInterbasDB Python Package - Implementation of Services Manager Support
**
** Version 3.1
**
** The following contributors hold Copyright (C) over their respective
** portions of code (see license.txt for details):
**
** [Original Author (maintained through version 2.0-0.3.1):]
**   1998-2001 [alex]  Alexander Kuznetsov   <alexan@users.sourceforge.net>
** [Maintainers (after version 2.0-0.3.1):]
**   2001-2002 [maz]   Marek Isalski         <kinterbasdb@maz.nu>
**   2002-2004 [dsr]   David Rushby          <woodsplitter@rocketmail.com>
** [Contributors:]
**   2001      [eac]   Evgeny A. Cherkashin  <eugeneai@icc.ru>
**   2001-2002 [janez] Janez Jere            <janez.jere@void.si>
*/

#include "_kiservices.h"

/* DSR:2002.10.14:begin block */
#include "pythread.h"

/* include "_kilock.h"
** is not allowed because it refers to a thread lock that's not in this
** compilation unit.  A reference to that lock must be acquired in a roundabout
** manner similar to the way references to kinterbasdb's exception classes
** are acquired (see the init_kiservices function). */
#include "_ki_brute_threadmodule.h"

#define SPB_BOILERPLATE_SIZE 2
#define INFINITE_TIMEOUT -1

/* Set the following to a very small value if you want to test result buffer
** truncation handling. */
#define SERVICE_RESULT_BUFFER_INITIAL_SIZE (unsigned short) 1024

/* Types of query output handling that pyob_query_base can be instructed to
** perform (these constants are made accessible to Python in function
** _init_kiservices_ibase_header_constants): */
#define QUERY_TYPE_PLAIN_STRING          1
#define QUERY_TYPE_PLAIN_INTEGER         2
#define QUERY_TYPE_RAW                   3

/* The following is a version of the ibase.h-standard macro ADD_SPB_NUMERIC
** that doesn't generate compiler warnings (compiler warnings drive me
** crazy(er)).
** This custom version is probably a little slower, but this macro really
** isn't used in performance-intensive areas anyway. */

#define ADD_SPB_NUMERIC_DSR(buf_pos, data) \
  memcpy( buf_pos, &data, sizeof(unsigned long) ); \
  buf_pos += sizeof(unsigned long);
/*
#define ADD_SPB_STRING_DSR(buf_pos, string, string_length) \
    ADD_SPB_LENGTH(buf_pos, string_length); \
    strncpy(buf_pos, string, string_length); \
    buf_pos += string_length;
*/

/******************** GLOBAL VARIABLES:BEGIN ***********************/
static PyObject *this_module = NULL;

/* DSR:2002.10.14:begin block */
#if SHOULD_MANAGE_GIL
static PyThread_type_lock module_thread_lock = NULL;
#endif /* SHOULD_MANAGE_GIL */
/* DSR:2002.10.14:end block */

/* 2003.01.01: */
#include "_kilock.h"
#include "_kinterbasdb_exception_functions.c"

/*static PyObject *UNSIGNED_LONG_MAX_As_PyObject = NULL;*/

/* Global references to the DB API exception objects from _kinterbasdb.
** References to the exception objects are transferred here by the
** pyob_initialize_from function so that code in this module can access these
** exceptions in order to raise them. */
static PyObject *Warning                   = NULL;
static PyObject *Error                     = NULL;
static PyObject *InterfaceError            = NULL;
static PyObject *DatabaseError             = NULL;
static PyObject *DataError                 = NULL;
static PyObject *OperationalError          = NULL;
static PyObject *IntegrityError            = NULL;
static PyObject *InternalError             = NULL;
static PyObject *ProgrammingError          = NULL;
static PyObject *NotSupportedError         = NULL;
/******************** GLOBAL VARIABLES:END ***********************/

/******************** PRIVATE FUNCTION PROTOTYPES:BEGIN ********************/
static PyObject *pyob_initialize_from(PyObject *self, PyObject *args);

static PyObject *pyob_connection_connect(PyObject *self, PyObject *args);
static PyObject *pyob_connection_close(PyObject *self, PyObject *args);
static int connection_close(
    ServicesConnectionObject *conn, boolean allowed_to_raise_exception
  );
static void pyob_connection_del(PyObject *conn);

static PyObject *pyob_isc_vax_integer( PyObject *self, PyObject *args );
static PyObject *pyob_query_base( PyObject *self, PyObject *args );
/******************** PRIVATE FUNCTION PROTOTYPES:END ********************/

/****** SERCIVES MANAGER CONNECTION CREATION/DELETION FUNCTIONS:BEGIN *******/
static PyObject *pyob_connection_connect(PyObject *self, PyObject *args) {
  ServicesConnectionObject *conn;

  char *service_manager_name = NULL;
  int service_manager_name_len = -1;
  char *username = NULL;
  int username_len = -1;
  char *password = NULL;
  int password_len = -1;

  char *spb, *spb_walk;
  unsigned short spb_length;

  if ( !PyArg_ParseTuple(args, "z#z#z#",
          &service_manager_name, &service_manager_name_len,
          &username, &username_len,
          &password, &password_len
        )
     )
  {
    return NULL;
  }

  /* 2003.02.20: */
  if (service_manager_name_len + username_len + password_len > 118) {
    raise_exception(ProgrammingError,
        "The combined length of the host, user, and password cannot exceed 118 bytes."
      );
    return NULL;
  }

  conn = PyObject_New( ServicesConnectionObject, &ServicesConnectionType );
  if (conn == NULL) {
    return PyErr_NoMemory();
  }
  conn->service_handle = NULL;

  spb_length =
    SPB_BOILERPLATE_SIZE
      + 1 /* the code isc_spb_user_name */
        + 1 /* the one-byte length username_len */
          + username_len /* the contents of username */
      + 1 /* the code isc_spb_password */
        + 1 /* the one-byte length password_len */
          + password_len /* the contents of the password */
    ;
  spb = kimem_main_malloc(spb_length);
  if (spb == NULL) {
    return PyErr_NoMemory();
  }

  spb_walk = spb;

  /* SPB_BOILERPLATE_SIZE refers to the next two entries: */
  *spb_walk++ = isc_spb_version;
  *spb_walk++ = isc_spb_current_version;

  *spb_walk++ = isc_spb_user_name;
  /* 2003.02.20: */
  *spb_walk++ = (char) username_len;
  strncpy(spb_walk, username, username_len);
  spb_walk += username_len;

  *spb_walk++ = isc_spb_password;
  /* 2003.02.20: */
  *spb_walk++ = (char) password_len;
  strncpy(spb_walk, password, password_len);
  spb_walk += password_len;

  assert (spb_length == spb_walk - spb);

  ENTER_DB
  isc_service_attach( conn->status,
      (unsigned short)service_manager_name_len, service_manager_name,
      &conn->service_handle,
      spb_length, spb
    );
  LEAVE_DB

  kimem_main_free(spb);

  if ( DB_API_ERROR(conn->status) ) {
    raise_sql_exception( OperationalError, "_kiservices.pyob_connection_connect: ",
      conn->status );

    Py_DECREF(conn);
    return NULL;
  }

  return (PyObject *) conn;
} /* pyob_connection_connect */


static int connection_close(
  ServicesConnectionObject *conn, boolean allowed_to_raise_exception
) {
  if (conn->service_handle != NULL) {
    ENTER_DB
    isc_service_detach(conn->status, &conn->service_handle);
    LEAVE_DB

    /* Set NULL to prevent segfault on "double jeopardy disconnect" (where
    ** exception is raised by con.close(), then con.__del__ calls this function
    ** again with an invalid service handle).
    ** Note:  isc_service_detach apparently follows the model of other
    ** detach/close functions in the Firebird C API, i.e. we need not manually
    ** free the service handle's memory. */
    conn->service_handle = NULL;

    if ( DB_API_ERROR(conn->status) ) {
      if (allowed_to_raise_exception) {
        raise_sql_exception(OperationalError, "_kiservices could not cleanly"
            " disconnect from the service manager: ",
            conn->status
          );
      }

      return -1;
    }
  }

  assert (conn->service_handle == NULL);
  return 0;
} /* connection_close */


static PyObject *pyob_connection_close( PyObject *self, PyObject *args ) {
  ServicesConnectionObject *conn = NULL;

  if ( !PyArg_ParseTuple( args, "O!", &ServicesConnectionType, &conn ) ) {
    return NULL;
  }

  if (connection_close(conn, TRUE) != 0) {
    return NULL;
  }

  RETURN_PY_NONE;
} /* pyob_connection_close */


static void pyob_connection_del(PyObject *conn) {
  ServicesConnectionObject *_conn = (ServicesConnectionObject *) conn;

  connection_close(_conn, FALSE); /* Ignore any errors in closing. */

  /* 2003.02.20: Moved the memory-freeing of a service handle to the
  ** connection_close function.  It should have freed the handle and set it
  ** NULL regardless of whether it successfully disconnected. */
  assert (_conn->service_handle == NULL);

  PyObject_Del(conn);
} /* pyob_connection_del */
/****** SERCIVES MANAGER CONNECTION CREATION/DELETION FUNCTIONS:END *******/

/*********************** ACTION FUNCTIONS:BEGIN *****************************/

static PyObject *pyob_action_thin( PyObject *self, PyObject *args ) {
  ServicesConnectionObject *conn = NULL;
  char *request_buf = NULL;
  int request_buf_size = -1;

  if ( !PyArg_ParseTuple( args, "O!s#", &ServicesConnectionType, &conn,
        &request_buf, &request_buf_size)
      )
  {
    return NULL;
  }

  if (request_buf_size > USHRT_MAX) {
    raise_exception(ProgrammingError,
        "The size of the request buffer must not exceed 65535."
      );
    return NULL;
  }

  ENTER_DB
  isc_service_start( conn->status,
      &conn->service_handle,
      NULL,
      (unsigned short) request_buf_size,
      request_buf
    );
  LEAVE_DB

  if ( DB_API_ERROR(conn->status) ) {
    raise_sql_exception(OperationalError, "_kiservices could not perform"
        " the action: ",
        conn->status
      );

    return NULL;
  }

  RETURN_PY_NONE;
} /* pyob_action_thin */

/*********************** ACTION FUNCTIONS:END *****************************/


/********* SERVICES MANAGER QUERY FUNCTIONS:BEGIN **********/
static PyObject *pyob_query_base( PyObject *self, PyObject *args ) {
  ServicesConnectionObject *conn = NULL;
  char req_items[] = " ";
  int req_item;
  #define Q_P_STR_REQ_ITEM_COUNT (unsigned short) 1
  int query_return_type;
  long timeout = INFINITE_TIMEOUT;

  char spb[6];
  char *spb_walk = spb;

  char *raw_result = NULL;
  unsigned int raw_result_size;
  char *raw_result_walk;

  PyObject *return_object;
  unsigned short return_object_length;
  unsigned long return_value_as_ulong;

  if ( !PyArg_ParseTuple( args, "O!ii|l", &ServicesConnectionType, &conn,
           &req_item, &query_return_type, &timeout
         )
      )
   {
    return NULL;
   }

  if (req_item < 0 || req_item > UCHAR_MAX) {
    char *base_msg = "The service query request_buf code must fall between 0 and %d.";
    char *err_msg = kimem_main_malloc(strlen(base_msg) + 32);
    if (err_msg == NULL) {
      return PyErr_NoMemory();
    }
    /* Possibility of buffer overflow considered and dealt with.  UCHAR_MAX
    ** won't exceed 34 digits in length anytime soon. */
    sprintf(err_msg, base_msg, UCHAR_MAX);

    PyErr_SetString(PyExc_ValueError, err_msg);
    kimem_main_free(err_msg);
    return NULL;
  }
  req_items[0] = (char)req_item;

  if (timeout != INFINITE_TIMEOUT) {
    *spb_walk++ = isc_info_svc_timeout;
    ADD_SPB_NUMERIC_DSR(spb_walk, timeout);
  }

  raw_result_size = SERVICE_RESULT_BUFFER_INITIAL_SIZE;
  /* Loop, enlarging the raw_result buffer, until the query's results can fit
  ** in raw_result. */
  for (;;) {
    raw_result = kimem_main_realloc(raw_result, raw_result_size);
    if (raw_result == NULL) {
      return PyErr_NoMemory();
    }

    ENTER_DB
    isc_service_query( conn->status,
        &conn->service_handle,
        NULL,
        (unsigned short) (spb_walk - spb), spb,
        Q_P_STR_REQ_ITEM_COUNT, req_items,
        (unsigned short) raw_result_size, raw_result
      );
    LEAVE_DB

    if ( DB_API_ERROR(conn->status) ) {
      raise_sql_exception(OperationalError, "_kiservices could not query: ",
        conn->status);
      kimem_main_free(raw_result);
      return NULL;
    }

    if (raw_result[0] == isc_info_truncated) {
      /* We need to allocate a bigger buffer because the service results
      ** couldn't fit in the one we initially supplied. */
      raw_result_size *= 2;
      continue;
    }

    break; /* raw_result was big enough; move on. */
  } /* ends for loop */

  raw_result_walk = raw_result;
  assert (*raw_result_walk == req_items[0]);
  raw_result_walk++;

  switch (query_return_type) {
    case QUERY_TYPE_PLAIN_STRING:
      return_object_length = (unsigned short) isc_vax_integer(raw_result_walk,
        sizeof(unsigned short));
      raw_result_walk += sizeof (unsigned short);

      return_object = PyString_FromStringAndSize(NULL, return_object_length);

      strncpy(PyString_AsString(return_object), raw_result_walk, return_object_length);
      raw_result_walk += return_object_length;
      assert (*raw_result_walk == isc_info_end);

      break;

    case QUERY_TYPE_PLAIN_INTEGER:
      return_value_as_ulong = (unsigned long)isc_vax_integer (raw_result_walk,
        sizeof(unsigned long));

      return_object = PyLong_FromLongLong( (LONG_LONG)return_value_as_ulong );

      raw_result_walk += sizeof (unsigned long);
      assert (*raw_result_walk == isc_info_end);

      break;

    case QUERY_TYPE_RAW:
      raw_result_walk = raw_result + (raw_result_size - 1);
      while (*raw_result_walk == '\0') {
        raw_result_walk--;
      }

      /* The return string might contain NULL bytes (Python strings have no
      ** problem with that). */
      return_object = PyString_FromStringAndSize(raw_result,
        raw_result_walk - raw_result);

      break;

    default:
      PyErr_SetString(PyExc_TypeError, "_kiservices.query_base is not equipped"
          " to handle this query type."
        );

      return_object = NULL;
      goto PYOB_QUERY_BASE_FINISH;
  }

  /* Under normal circumstances, execution will fall through to the cleanup
  ** section after return_object has been constructed. */

PYOB_QUERY_BASE_FINISH:
  kimem_main_free(raw_result);

  return return_object;
} /* pyob_query_base */
/********* SERVICES MANAGER QUERY FUNCTIONS:END **********/

/************** SERVICES UTILITIY FUNCTIONS:BEGIN ***************/
static PyObject *pyob_isc_vax_integer( PyObject *self, PyObject *args ) {
  /* isc_vax_integer reverses the byte order of an integer.
  ** This Python wrapper function is used in services.py when parsing the
  ** raw return buffer from a Services Manager query. */
  char *raw_bytes;
  int raw_length;

  if ( !PyArg_ParseTuple( args, "s#", &raw_bytes, &raw_length) ) {
    return NULL;
  }

  if (raw_length != 4 && raw_length != 2 && raw_length != 1) {
    raise_exception(InternalError, "pyob_isc_vax_integer: len(buf) must be in (1,2,4)");
    return NULL;
  }

  return PyInt_FromLong( (long)
      isc_vax_integer(raw_bytes, (unsigned short)raw_length)
    );
} /* pyob_isc_vax_integer */
/************** SERVICES UTILITIY FUNCTIONS:END *****************/

/********* PYTHON TYPE OBJECT SETUP:BEGIN **********/
static PyTypeObject ServicesConnectionType = {
    PyObject_HEAD_INIT( NULL )

    0,
    "_kiservices.ServicesConnection",
    sizeof( ServicesConnectionObject ),
    0,
    pyob_connection_del,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0
  }; /* ServicesConnectionType */
/********* PYTHON TYPE OBJECT SETUP:END **********/


/****************** MODULE ADMINISTRATION FUNCTIONS:BEGIN ********************/
static PyObject *pyob_initialize_from( PyObject *self, PyObject *args ) {
  /* Makes kinterbasdb's global thread lock and references to its Python DB API
  ** exception classes readily available to this C code. */
  PyObject *source = NULL;

  if ( !PyArg_ParseTuple(args, "O", &source) ) {
    return NULL;
  }

  /* DSR:2002.10.14:begin block */
  #if SHOULD_MANAGE_GIL
  { /* First, retrieve the Python object of type thread.Locktype.  Then,
    ** retrieve the C-level lock object from the Python lock. */
    PyObject *svc_module_python_thread_lock =
      PyObject_GetAttrString(source, "_module_python_thread_lock_object");
    if (svc_module_python_thread_lock == NULL) {
      PyErr_SetString(PyExc_RuntimeError, "Could not retrieve reference to"
          " global thread lock in main library."
        );
      return NULL;
    }
    module_thread_lock =
      ((_bruteforce_lockobject *) svc_module_python_thread_lock)->lock_lock;
  }
  #endif /* SHOULD_MANAGE_GIL */
  /* DSR:2002.10.14:end block */

  /* 2003.07.02: Don't set the Python module's exception class references here,
  ** because the name 'Warning' shadows a built-in and causes a deprecation
  ** warning with Python 2.3b2.  Now, the services.py module acquires
  ** references to the classes from k_exceptions. */
  /* PyObject_GetAttrString returns a new reference; no need for us to INCREF. */
  Warning = PyObject_GetAttrString(source, "Warning");
  Error = PyObject_GetAttrString(source, "Error");
  InterfaceError = PyObject_GetAttrString(source, "InterfaceError");
  DatabaseError = PyObject_GetAttrString(source, "DatabaseError");
  DataError = PyObject_GetAttrString(source, "DataError");
  OperationalError = PyObject_GetAttrString(source, "OperationalError");
  IntegrityError = PyObject_GetAttrString(source, "IntegrityError");
  InternalError = PyObject_GetAttrString(source, "InternalError");
  ProgrammingError = PyObject_GetAttrString(source, "ProgrammingError");
  NotSupportedError = PyObject_GetAttrString(source, "NotSupportedError");

  RETURN_PY_NONE;
} /* initialize_from */


int _init_kiservices_ibase_header_constants(PyObject *d) {
  /* Makes a bunch of database engine constants available for reference in
  ** Python. */

  /* SIC is just a shortcut for entering header constants into dict d. */
  #define SIC(name, value) if ( -1 == PyDict_SetItemString(d, name, PyInt_FromLong(value)) ) { return -1; }

  SIC("SIZEOF_SHORT",                   sizeof(short));
  SIC("SIZEOF_SHORT_UNSIGNED",          sizeof(unsigned short));
  SIC("SIZEOF_LONG",                    sizeof(long));
  SIC("SIZEOF_LONG_UNSIGNED",           sizeof(unsigned long));

  /* Locally defined header constants that must be accessible from Python: */
  SIC("QUERY_TYPE_PLAIN_STRING",          QUERY_TYPE_PLAIN_STRING);
  SIC("QUERY_TYPE_PLAIN_INTEGER",         QUERY_TYPE_PLAIN_INTEGER);
  SIC("QUERY_TYPE_RAW",                   QUERY_TYPE_RAW);

  /* Drawn from ibase.h: */
  /* YYY: Perhaps should cull these constants? */
  SIC("isc_spb_version1"                 , isc_spb_version1                   );
  SIC("isc_spb_current_version"          , isc_spb_current_version            );
  SIC("isc_spb_version"                  , isc_spb_version                    );
  SIC("isc_spb_current_version"          , isc_spb_current_version            );
  SIC("isc_spb_user_name"                , isc_spb_user_name                  );
  SIC("isc_spb_sys_user_name"            , isc_spb_sys_user_name              );
  SIC("isc_spb_sys_user_name_enc"        , isc_spb_sys_user_name_enc          );
  SIC("isc_spb_password"                 , isc_spb_password                   );
  SIC("isc_spb_password_enc"             , isc_spb_password_enc               );
  SIC("isc_spb_command_line"             , isc_spb_command_line               );
  SIC("isc_spb_dbname"                   , isc_spb_dbname                     );
  SIC("isc_spb_verbose"                  , isc_spb_verbose                    );
  SIC("isc_spb_options"                  , isc_spb_options                    );
  SIC("isc_spb_connect_timeout"          , isc_spb_connect_timeout            );
  SIC("isc_spb_dummy_packet_interval"    , isc_spb_dummy_packet_interval      );
  SIC("isc_spb_sql_role_name"            , isc_spb_sql_role_name              );

  SIC("isc_dpb_sys_user_name"            , isc_dpb_sys_user_name              );
  SIC("isc_dpb_sys_user_name_enc"        , isc_dpb_sys_user_name_enc          );
  SIC("isc_dpb_password"                 , isc_dpb_password                   );
  SIC("isc_dpb_password_enc"             , isc_dpb_password_enc               );
  SIC("isc_dpb_connect_timeout"          , isc_dpb_connect_timeout            );
  SIC("isc_dpb_dummy_packet_interval"    , isc_dpb_dummy_packet_interval      );
  SIC("isc_dpb_user_name"                , isc_dpb_user_name                  );
  SIC("isc_dpb_sql_role_name"            , isc_dpb_sql_role_name              );

  SIC("isc_info_end"                     , isc_info_end                       );
  SIC("isc_info_truncated"               , isc_info_truncated                 );
  SIC("isc_info_error"                   , isc_info_error                     );
  SIC("isc_info_data_not_ready"          , isc_info_data_not_ready            );
  SIC("isc_info_flag_end"                , isc_info_flag_end                  );
  SIC("isc_info_db_id"                   , isc_info_db_id                     );
  SIC("isc_info_reads"                   , isc_info_reads                     );
  SIC("isc_info_writes"                  , isc_info_writes                    );
  SIC("isc_info_fetches"                 , isc_info_fetches                   );
  SIC("isc_info_marks"                   , isc_info_marks                     );
  SIC("isc_info_implementation"          , isc_info_implementation            );
#ifdef isc_info_firebird_version
  SIC("isc_info_isc_version"             , isc_info_isc_version               );
#endif /* isc_info_firebird_version */
  SIC("isc_info_base_level"              , isc_info_base_level                );
  SIC("isc_info_page_size"               , isc_info_page_size                 );
  SIC("isc_info_num_buffers"             , isc_info_num_buffers               );
  SIC("isc_info_limbo"                   , isc_info_limbo                     );
  SIC("isc_info_current_memory"          , isc_info_current_memory            );
  SIC("isc_info_max_memory"              , isc_info_max_memory                );
  SIC("isc_info_window_turns"            , isc_info_window_turns              );
  SIC("isc_info_license"                 , isc_info_license                   );
  SIC("isc_info_allocation"              , isc_info_allocation                );
  SIC("isc_info_attachment_id"           , isc_info_attachment_id             );
  SIC("isc_info_read_seq_count"          , isc_info_read_seq_count            );
  SIC("isc_info_read_idx_count"          , isc_info_read_idx_count            );
  SIC("isc_info_insert_count"            , isc_info_insert_count              );
  SIC("isc_info_update_count"            , isc_info_update_count              );
  SIC("isc_info_delete_count"            , isc_info_delete_count              );
  SIC("isc_info_backout_count"           , isc_info_backout_count             );
  SIC("isc_info_purge_count"             , isc_info_purge_count               );
  SIC("isc_info_expunge_count"           , isc_info_expunge_count             );
  SIC("isc_info_sweep_interval"          , isc_info_sweep_interval            );
  SIC("isc_info_ods_version"             , isc_info_ods_version               );
  SIC("isc_info_ods_minor_version"       , isc_info_ods_minor_version         );
  SIC("isc_info_no_reserve"              , isc_info_no_reserve                );
  SIC("isc_info_logfile"                 , isc_info_logfile                   );
  SIC("isc_info_cur_logfile_name"        , isc_info_cur_logfile_name          );
  SIC("isc_info_cur_log_part_offset"     , isc_info_cur_log_part_offset       );
  SIC("isc_info_num_wal_buffers"         , isc_info_num_wal_buffers           );
  SIC("isc_info_wal_buffer_size"         , isc_info_wal_buffer_size           );
  SIC("isc_info_wal_ckpt_length"         , isc_info_wal_ckpt_length           );
  SIC("isc_info_wal_cur_ckpt_interval"   , isc_info_wal_cur_ckpt_interval     );
  SIC("isc_info_wal_prv_ckpt_fname"      , isc_info_wal_prv_ckpt_fname        );
  SIC("isc_info_wal_prv_ckpt_poffset"    , isc_info_wal_prv_ckpt_poffset      );
  SIC("isc_info_wal_recv_ckpt_fname"     , isc_info_wal_recv_ckpt_fname       );
  SIC("isc_info_wal_recv_ckpt_poffset"   , isc_info_wal_recv_ckpt_poffset     );
  SIC("isc_info_wal_grpc_wait_usecs"     , isc_info_wal_grpc_wait_usecs       );
  SIC("isc_info_wal_num_io"              , isc_info_wal_num_io                );
  SIC("isc_info_wal_avg_io_size"         , isc_info_wal_avg_io_size           );
  SIC("isc_info_wal_num_commits"         , isc_info_wal_num_commits           );
  SIC("isc_info_wal_avg_grpc_size"       , isc_info_wal_avg_grpc_size         );
  SIC("isc_info_forced_writes"           , isc_info_forced_writes             );
  SIC("isc_info_user_names"              , isc_info_user_names                );
  SIC("isc_info_page_errors"             , isc_info_page_errors               );
  SIC("isc_info_record_errors"           , isc_info_record_errors             );
  SIC("isc_info_bpage_errors"            , isc_info_bpage_errors              );
  SIC("isc_info_dpage_errors"            , isc_info_dpage_errors              );
  SIC("isc_info_ipage_errors"            , isc_info_ipage_errors              );
  SIC("isc_info_ppage_errors"            , isc_info_ppage_errors              );
  SIC("isc_info_tpage_errors"            , isc_info_tpage_errors              );
  SIC("isc_info_set_page_buffers"        , isc_info_set_page_buffers          );
  SIC("isc_info_db_sql_dialect"          , isc_info_db_sql_dialect            );
  SIC("isc_info_db_read_only"            , isc_info_db_read_only              );
  SIC("isc_info_db_size_in_pages"        , isc_info_db_size_in_pages          );
#ifdef isc_info_firebird_version
  SIC("isc_info_db_class"                , isc_info_db_class                  );
  SIC("isc_info_firebird_version"        , isc_info_firebird_version          );
  SIC("isc_info_oldest_transaction"      , isc_info_oldest_transaction        );
  SIC("isc_info_oldest_active"           , isc_info_oldest_active             );
  SIC("isc_info_oldest_snapshot"         , isc_info_oldest_snapshot           );
  SIC("isc_info_next_transaction"        , isc_info_next_transaction          );
  SIC("isc_info_db_provider"             , isc_info_db_provider               );
  SIC("isc_info_db_last_value"           , isc_info_db_last_value             );
  SIC("isc_info_version"                 , isc_info_version                   );
  SIC("isc_info_isc_version"             , isc_info_isc_version               );
#endif /* isc_info_firebird_version */
  SIC("isc_info_db_impl_rdb_vms"         , isc_info_db_impl_rdb_vms           );
  SIC("isc_info_db_impl_rdb_eln"         , isc_info_db_impl_rdb_eln           );
  SIC("isc_info_db_impl_rdb_eln_dev"     , isc_info_db_impl_rdb_eln_dev       );
  SIC("isc_info_db_impl_rdb_vms_y"       , isc_info_db_impl_rdb_vms_y         );
  SIC("isc_info_db_impl_rdb_eln_y"       , isc_info_db_impl_rdb_eln_y         );
  SIC("isc_info_db_impl_jri"             , isc_info_db_impl_jri               );
  SIC("isc_info_db_impl_jsv"             , isc_info_db_impl_jsv               );
  SIC("isc_info_db_impl_isc_apl_68K"     , isc_info_db_impl_isc_apl_68K       );
  SIC("isc_info_db_impl_isc_vax_ultr"    , isc_info_db_impl_isc_vax_ultr      );
  SIC("isc_info_db_impl_isc_vms"         , isc_info_db_impl_isc_vms           );
  SIC("isc_info_db_impl_isc_sun_68k"     , isc_info_db_impl_isc_sun_68k       );
  SIC("isc_info_db_impl_isc_os2"         , isc_info_db_impl_isc_os2           );
  SIC("isc_info_db_impl_isc_sun4"        , isc_info_db_impl_isc_sun4          );
  SIC("isc_info_db_impl_isc_hp_ux"       , isc_info_db_impl_isc_hp_ux         );
  SIC("isc_info_db_impl_isc_sun_386i"    , isc_info_db_impl_isc_sun_386i      );
  SIC("isc_info_db_impl_isc_vms_orcl"    , isc_info_db_impl_isc_vms_orcl      );
  SIC("isc_info_db_impl_isc_mac_aux"     , isc_info_db_impl_isc_mac_aux       );
  SIC("isc_info_db_impl_isc_rt_aix"      , isc_info_db_impl_isc_rt_aix        );
  SIC("isc_info_db_impl_isc_mips_ult"    , isc_info_db_impl_isc_mips_ult      );
  SIC("isc_info_db_impl_isc_xenix"       , isc_info_db_impl_isc_xenix         );
  SIC("isc_info_db_impl_isc_dg"          , isc_info_db_impl_isc_dg            );
  SIC("isc_info_db_impl_isc_hp_mpexl"    , isc_info_db_impl_isc_hp_mpexl      );
  SIC("isc_info_db_impl_isc_hp_ux68K"    , isc_info_db_impl_isc_hp_ux68K      );
  SIC("isc_info_db_impl_isc_sgi"         , isc_info_db_impl_isc_sgi           );
  SIC("isc_info_db_impl_isc_sco_unix"    , isc_info_db_impl_isc_sco_unix      );
  SIC("isc_info_db_impl_isc_cray"        , isc_info_db_impl_isc_cray          );
  SIC("isc_info_db_impl_isc_imp"         , isc_info_db_impl_isc_imp           );
  SIC("isc_info_db_impl_isc_delta"       , isc_info_db_impl_isc_delta         );
  SIC("isc_info_db_impl_isc_next"        , isc_info_db_impl_isc_next          );
  SIC("isc_info_db_impl_isc_dos"         , isc_info_db_impl_isc_dos           );
#ifdef isc_info_firebird_version
  SIC("isc_info_db_impl_m88K"            , isc_info_db_impl_m88K              );
  SIC("isc_info_db_impl_unixware"        , isc_info_db_impl_unixware          );
  SIC("isc_info_db_impl_isc_winnt_x86"   , isc_info_db_impl_isc_winnt_x86     );
  SIC("isc_info_db_impl_isc_epson"       , isc_info_db_impl_isc_epson         );
  SIC("isc_info_db_impl_alpha_osf"       , isc_info_db_impl_alpha_osf         );
  SIC("isc_info_db_impl_alpha_vms"       , isc_info_db_impl_alpha_vms         );
  SIC("isc_info_db_impl_netware_386"     , isc_info_db_impl_netware_386       );
  SIC("isc_info_db_impl_win_only"        , isc_info_db_impl_win_only          );
  SIC("isc_info_db_impl_ncr_3000"        , isc_info_db_impl_ncr_3000          );
  SIC("isc_info_db_impl_winnt_ppc"       , isc_info_db_impl_winnt_ppc         );
  SIC("isc_info_db_impl_dg_x86"          , isc_info_db_impl_dg_x86            );
  SIC("isc_info_db_impl_sco_ev"          , isc_info_db_impl_sco_ev            );
  SIC("isc_info_db_impl_i386"            , isc_info_db_impl_i386              );
  SIC("isc_info_db_impl_freebsd"         , isc_info_db_impl_freebsd           );
  SIC("isc_info_db_impl_netbsd"          , isc_info_db_impl_netbsd            );
  SIC("isc_info_db_impl_darwin"          , isc_info_db_impl_darwin            );
  SIC("isc_info_db_impl_last_value"      , isc_info_db_impl_last_value        );
#endif /* isc_info_firebird_version */
  SIC("isc_info_db_impl_isc_a"           , isc_info_db_impl_isc_a             );
  SIC("isc_info_db_impl_isc_apl_68K"     , isc_info_db_impl_isc_apl_68K       );
  SIC("isc_info_db_impl_isc_u"           , isc_info_db_impl_isc_u             );
  SIC("isc_info_db_impl_isc_vax_ultr"    , isc_info_db_impl_isc_vax_ultr      );
  SIC("isc_info_db_impl_isc_v"           , isc_info_db_impl_isc_v             );
  SIC("isc_info_db_impl_isc_vms"         , isc_info_db_impl_isc_vms           );
  SIC("isc_info_db_impl_isc_s"           , isc_info_db_impl_isc_s             );
  SIC("isc_info_db_impl_isc_sun_68k"     , isc_info_db_impl_isc_sun_68k       );
  SIC("isc_info_db_class_access"         , isc_info_db_class_access           );
  SIC("isc_info_db_class_y_valve"        , isc_info_db_class_y_valve          );
  SIC("isc_info_db_class_rem_int"        , isc_info_db_class_rem_int          );
  SIC("isc_info_db_class_rem_srvr"       , isc_info_db_class_rem_srvr         );
  SIC("isc_info_db_class_pipe_int"       , isc_info_db_class_pipe_int         );
  SIC("isc_info_db_class_pipe_srvr"      , isc_info_db_class_pipe_srvr        );
  SIC("isc_info_db_class_sam_int"        , isc_info_db_class_sam_int          );
  SIC("isc_info_db_class_sam_srvr"       , isc_info_db_class_sam_srvr         );
  SIC("isc_info_db_class_gateway"        , isc_info_db_class_gateway          );
  SIC("isc_info_db_class_cache"          , isc_info_db_class_cache            );
#ifdef isc_info_firebird_version
  SIC("isc_info_db_class_classic_access" , isc_info_db_class_classic_access   );
  SIC("isc_info_db_class_server_access"  , isc_info_db_class_server_access    );
  SIC("isc_info_db_class_last_value"     , isc_info_db_class_last_value       );
  SIC("isc_info_db_code_rdb_eln"         , isc_info_db_code_rdb_eln           );
  SIC("isc_info_db_code_rdb_vms"         , isc_info_db_code_rdb_vms           );
  SIC("isc_info_db_code_interbase"       , isc_info_db_code_interbase         );
  SIC("isc_info_db_code_firebird"        , isc_info_db_code_firebird          );
  SIC("isc_info_db_code_last_value"      , isc_info_db_code_last_value        );
#endif /* isc_info_firebird_version */
  SIC("isc_info_number_messages"         , isc_info_number_messages           );
  SIC("isc_info_max_message"             , isc_info_max_message               );
  SIC("isc_info_max_send"                , isc_info_max_send                  );
  SIC("isc_info_max_receive"             , isc_info_max_receive               );
  SIC("isc_info_state"                   , isc_info_state                     );
  SIC("isc_info_message_number"          , isc_info_message_number            );
  SIC("isc_info_message_size"            , isc_info_message_size              );
  SIC("isc_info_request_cost"            , isc_info_request_cost              );
  SIC("isc_info_access_path"             , isc_info_access_path               );
  SIC("isc_info_req_select_count"        , isc_info_req_select_count          );
  SIC("isc_info_req_insert_count"        , isc_info_req_insert_count          );
  SIC("isc_info_req_update_count"        , isc_info_req_update_count          );
  SIC("isc_info_req_delete_count"        , isc_info_req_delete_count          );
  SIC("isc_info_rsb_end"                 , isc_info_rsb_end                   );
  SIC("isc_info_rsb_begin"               , isc_info_rsb_begin                 );
  SIC("isc_info_rsb_type"                , isc_info_rsb_type                  );
  SIC("isc_info_rsb_relation"            , isc_info_rsb_relation              );
  SIC("isc_info_rsb_plan"                , isc_info_rsb_plan                  );
  SIC("isc_info_rsb_unknown"             , isc_info_rsb_unknown               );
  SIC("isc_info_rsb_indexed"             , isc_info_rsb_indexed               );
  SIC("isc_info_rsb_navigate"            , isc_info_rsb_navigate              );
  SIC("isc_info_rsb_sequential"          , isc_info_rsb_sequential            );
  SIC("isc_info_rsb_cross"               , isc_info_rsb_cross                 );
  SIC("isc_info_rsb_sort"                , isc_info_rsb_sort                  );
  SIC("isc_info_rsb_first"               , isc_info_rsb_first                 );
  SIC("isc_info_rsb_boolean"             , isc_info_rsb_boolean               );
  SIC("isc_info_rsb_union"               , isc_info_rsb_union                 );
  SIC("isc_info_rsb_aggregate"           , isc_info_rsb_aggregate             );
  SIC("isc_info_rsb_merge"               , isc_info_rsb_merge                 );
  SIC("isc_info_rsb_ext_sequential"      , isc_info_rsb_ext_sequential        );
  SIC("isc_info_rsb_ext_indexed"         , isc_info_rsb_ext_indexed           );
  SIC("isc_info_rsb_ext_dbkey"           , isc_info_rsb_ext_dbkey             );
  SIC("isc_info_rsb_left_cross"          , isc_info_rsb_left_cross            );
  SIC("isc_info_rsb_select"              , isc_info_rsb_select                );
  SIC("isc_info_rsb_sql_join"            , isc_info_rsb_sql_join              );
  SIC("isc_info_rsb_simulate"            , isc_info_rsb_simulate              );
  SIC("isc_info_rsb_sim_cross"           , isc_info_rsb_sim_cross             );
  SIC("isc_info_rsb_once"                , isc_info_rsb_once                  );
  SIC("isc_info_rsb_procedure"           , isc_info_rsb_procedure             );
  SIC("isc_info_rsb_and"                 , isc_info_rsb_and                   );
  SIC("isc_info_rsb_or"                  , isc_info_rsb_or                    );
  SIC("isc_info_rsb_dbkey"               , isc_info_rsb_dbkey                 );
  SIC("isc_info_rsb_index"               , isc_info_rsb_index                 );
  SIC("isc_info_req_active"              , isc_info_req_active                );
  SIC("isc_info_req_inactive"            , isc_info_req_inactive              );
  SIC("isc_info_req_send"                , isc_info_req_send                  );
  SIC("isc_info_req_receive"             , isc_info_req_receive               );
  SIC("isc_info_req_select"              , isc_info_req_select                );
  SIC("isc_info_req_sql_stall"           , isc_info_req_sql_stall             );
  SIC("isc_info_blob_num_segments"       , isc_info_blob_num_segments         );
  SIC("isc_info_blob_max_segment"        , isc_info_blob_max_segment          );
  SIC("isc_info_blob_total_length"       , isc_info_blob_total_length         );
  SIC("isc_info_blob_type"               , isc_info_blob_type                 );
  SIC("isc_info_tra_id"                  , isc_info_tra_id                    );

  SIC("isc_action_svc_backup"            , isc_action_svc_backup              );
  SIC("isc_action_svc_restore"           , isc_action_svc_restore             );
  SIC("isc_action_svc_repair"            , isc_action_svc_repair              );
  SIC("isc_action_svc_add_user"          , isc_action_svc_add_user            );
  SIC("isc_action_svc_delete_user"       , isc_action_svc_delete_user         );
  SIC("isc_action_svc_modify_user"       , isc_action_svc_modify_user         );
  SIC("isc_action_svc_display_user"      , isc_action_svc_display_user        );
  SIC("isc_action_svc_properties"        , isc_action_svc_properties          );
  SIC("isc_action_svc_add_license"       , isc_action_svc_add_license         );
  SIC("isc_action_svc_remove_license"    , isc_action_svc_remove_license      );
  SIC("isc_action_svc_db_stats"          , isc_action_svc_db_stats            );
  SIC("isc_action_svc_get_ib_log"        , isc_action_svc_get_ib_log          );

  SIC("isc_info_svc_svr_db_info"         , isc_info_svc_svr_db_info           );
  SIC("isc_info_svc_get_license"         , isc_info_svc_get_license           );
  SIC("isc_info_svc_get_license_mask"    , isc_info_svc_get_license_mask      );
  SIC("isc_info_svc_get_config"          , isc_info_svc_get_config            );
  SIC("isc_info_svc_version"             , isc_info_svc_version               );
  SIC("isc_info_svc_server_version"      , isc_info_svc_server_version        );
  SIC("isc_info_svc_implementation"      , isc_info_svc_implementation        );
  SIC("isc_info_svc_capabilities"        , isc_info_svc_capabilities          );
  SIC("isc_info_svc_user_dbpath"         , isc_info_svc_user_dbpath           );
  SIC("isc_info_svc_get_env"             , isc_info_svc_get_env               );
  SIC("isc_info_svc_get_env_lock"        , isc_info_svc_get_env_lock          );
  SIC("isc_info_svc_get_env_msg"         , isc_info_svc_get_env_msg           );
  SIC("isc_info_svc_line"                , isc_info_svc_line                  );
  SIC("isc_info_svc_to_eof"              , isc_info_svc_to_eof                );
  SIC("isc_info_svc_timeout"             , isc_info_svc_timeout               );
  SIC("isc_info_svc_get_licensed_users"  , isc_info_svc_get_licensed_users    );
  SIC("isc_info_svc_limbo_trans"         , isc_info_svc_limbo_trans           );
  SIC("isc_info_svc_running"             , isc_info_svc_running               );
  SIC("isc_info_svc_get_users"           , isc_info_svc_get_users             );

  SIC("isc_spb_sec_userid"               , isc_spb_sec_userid                 );
  SIC("isc_spb_sec_groupid"              , isc_spb_sec_groupid                );
  SIC("isc_spb_sec_username"             , isc_spb_sec_username               );
  SIC("isc_spb_sec_password"             , isc_spb_sec_password               );
  SIC("isc_spb_sec_groupname"            , isc_spb_sec_groupname              );
  SIC("isc_spb_sec_firstname"            , isc_spb_sec_firstname              );
  SIC("isc_spb_sec_middlename"           , isc_spb_sec_middlename             );
  SIC("isc_spb_sec_lastname"             , isc_spb_sec_lastname               );
/* Won't bother with license features.
  SIC("isc_info_svc_get_license"         , isc_info_svc_get_license           );
  SIC("isc_spb_lic_key"                  , isc_spb_lic_key                    );
  SIC("isc_spb_lic_id"                   , isc_spb_lic_id                     );
  SIC("isc_spb_lic_desc"                 , isc_spb_lic_desc                   );
*/
  /* <BACKUP> */
  SIC("isc_action_svc_backup"            , isc_action_svc_backup              );

  SIC("isc_spb_bkp_file"                 , isc_spb_bkp_file                   );
  SIC("isc_spb_bkp_factor"               , isc_spb_bkp_factor                 );
  SIC("isc_spb_bkp_length"               , isc_spb_bkp_length                 );
  /* begin bitmask components */
  SIC("isc_spb_bkp_ignore_checksums"     , isc_spb_bkp_ignore_checksums       );
  SIC("isc_spb_bkp_ignore_limbo"         , isc_spb_bkp_ignore_limbo           );
  SIC("isc_spb_bkp_metadata_only"        , isc_spb_bkp_metadata_only          );
  SIC("isc_spb_bkp_no_garbage_collect"   , isc_spb_bkp_no_garbage_collect     );
  SIC("isc_spb_bkp_old_descriptions"     , isc_spb_bkp_old_descriptions       );
  SIC("isc_spb_bkp_non_transportable"    , isc_spb_bkp_non_transportable      );
  SIC("isc_spb_bkp_convert"              , isc_spb_bkp_convert                );
  SIC("isc_spb_bkp_expand"               , isc_spb_bkp_expand                 );
  /* end bitmask components */
  /* </BACKUP> */

  SIC("isc_action_svc_properties"        , isc_action_svc_properties          );
  SIC("isc_spb_prp_page_buffers"         , isc_spb_prp_page_buffers           );
  SIC("isc_spb_prp_sweep_interval"       , isc_spb_prp_sweep_interval         );
  SIC("isc_spb_prp_shutdown_db"          , isc_spb_prp_shutdown_db            );
  SIC("isc_spb_prp_deny_new_attachments" , isc_spb_prp_deny_new_attachments   );
  SIC("isc_spb_prp_deny_new_transactions", isc_spb_prp_deny_new_transactions  );
  SIC("isc_spb_prp_reserve_space"        , isc_spb_prp_reserve_space          );
  SIC("isc_spb_prp_write_mode"           , isc_spb_prp_write_mode             );
  SIC("isc_spb_prp_access_mode"          , isc_spb_prp_access_mode            );
  SIC("isc_spb_prp_set_sql_dialect"      , isc_spb_prp_set_sql_dialect        );
  SIC("isc_spb_prp_activate"             , isc_spb_prp_activate               );
  SIC("isc_spb_prp_db_online"            , isc_spb_prp_db_online              );
  SIC("isc_spb_prp_reserve_space"        , isc_spb_prp_reserve_space          );
  SIC("isc_spb_prp_res_use_full"         , isc_spb_prp_res_use_full           );
  SIC("isc_spb_prp_res"                  , isc_spb_prp_res                    );
  SIC("isc_spb_prp_write_mode"           , isc_spb_prp_write_mode             );
  SIC("isc_spb_prp_wm_async"             , isc_spb_prp_wm_async               );
  SIC("isc_spb_prp_wm_sync"              , isc_spb_prp_wm_sync                );
  SIC("isc_spb_prp_access_mode"          , isc_spb_prp_access_mode            );
  SIC("isc_spb_prp_am_readonly"          , isc_spb_prp_am_readonly            );
  SIC("isc_spb_prp_am_readwrite"         , isc_spb_prp_am_readwrite           );

  SIC("isc_action_svc_repair"            , isc_action_svc_repair              );
  SIC("isc_spb_rpr_commit_trans"         , isc_spb_rpr_commit_trans           );
  SIC("isc_spb_rpr_rollback_trans"       , isc_spb_rpr_rollback_trans         );
  SIC("isc_spb_rpr_recover_two_phase"    , isc_spb_rpr_recover_two_phase      );
  SIC("isc_spb_tra_id"                   , isc_spb_tra_id                     );
  SIC("isc_spb_single_tra_id"            , isc_spb_single_tra_id              );
  SIC("isc_spb_multi_tra_id"             , isc_spb_multi_tra_id               );
  SIC("isc_spb_tra_state"                , isc_spb_tra_state                  );
  SIC("isc_spb_tra_state_limbo"          , isc_spb_tra_state_limbo            );
  SIC("isc_spb_tra_state_commit"         , isc_spb_tra_state_commit           );
  SIC("isc_spb_tra_state_rollback"       , isc_spb_tra_state_rollback         );
  SIC("isc_spb_tra_state_unknown"        , isc_spb_tra_state_unknown          );
  SIC("isc_spb_tra_host_site"            , isc_spb_tra_host_site              );
  SIC("isc_spb_tra_remote_site"          , isc_spb_tra_remote_site            );
  SIC("isc_spb_tra_db_path"              , isc_spb_tra_db_path                );
  SIC("isc_spb_tra_advise"               , isc_spb_tra_advise                 );
  SIC("isc_spb_tra_advise_commit"        , isc_spb_tra_advise_commit          );
  SIC("isc_spb_tra_advise_rollback"      , isc_spb_tra_advise_rollback        );
  SIC("isc_spb_tra_advise_unknown"       , isc_spb_tra_advise_unknown         );
  SIC("isc_spb_rpr_validate_db"          , isc_spb_rpr_validate_db            );
  SIC("isc_spb_rpr_sweep_db"             , isc_spb_rpr_sweep_db               );
  SIC("isc_spb_rpr_mend_db"              , isc_spb_rpr_mend_db                );
  SIC("isc_spb_rpr_list_limbo_trans"     , isc_spb_rpr_list_limbo_trans       );
  SIC("isc_spb_rpr_check_db"             , isc_spb_rpr_check_db               );
  SIC("isc_spb_rpr_ignore_checksum"      , isc_spb_rpr_ignore_checksum        );
  SIC("isc_spb_rpr_kill_shadows"         , isc_spb_rpr_kill_shadows           );
  SIC("isc_spb_rpr_full"                 , isc_spb_rpr_full                   );

  SIC("isc_action_svc_restore"           , isc_action_svc_restore             );
  SIC("isc_spb_res_buffers"              , isc_spb_res_buffers                );
  SIC("isc_spb_res_page_size"            , isc_spb_res_page_size              );
  SIC("isc_spb_res_length"               , isc_spb_res_length                 );
  SIC("isc_spb_res_access_mode"          , isc_spb_res_access_mode            );
  SIC("isc_spb_res_deactivate_idx"       , isc_spb_res_deactivate_idx         );
  SIC("isc_spb_res_no_shadow"            , isc_spb_res_no_shadow              );
  SIC("isc_spb_res_no_validity"          , isc_spb_res_no_validity            );
  SIC("isc_spb_res_one_at_a_time"        , isc_spb_res_one_at_a_time          );
  SIC("isc_spb_res_replace"              , isc_spb_res_replace                );
  SIC("isc_spb_res_create"               , isc_spb_res_create                 );
  SIC("isc_spb_res_use_all_space"        , isc_spb_res_use_all_space          );
  SIC("isc_spb_res_access_mode"          , isc_spb_res_access_mode            );
  SIC("isc_spb_res_am_readonly"          , isc_spb_res_am_readonly            );
  SIC("isc_spb_prp_am_readonly"          , isc_spb_prp_am_readonly            );
  SIC("isc_spb_res_am_readwrite"         , isc_spb_res_am_readwrite           );
  SIC("isc_spb_prp_am_readwrite"         , isc_spb_prp_am_readwrite           );

  SIC("isc_info_svc_svr_db_info"         , isc_info_svc_svr_db_info           );
  SIC("isc_spb_num_att"                  , isc_spb_num_att                    );
  SIC("isc_spb_num_db"                   , isc_spb_num_db                     );
  SIC("isc_spb_sts_data_pages"           , isc_spb_sts_data_pages             );
  SIC("isc_spb_sts_db_log"               , isc_spb_sts_db_log                 );
  SIC("isc_spb_sts_hdr_pages"            , isc_spb_sts_hdr_pages              );
  SIC("isc_spb_sts_idx_pages"            , isc_spb_sts_idx_pages              );
  SIC("isc_spb_sts_sys_relations"        , isc_spb_sts_sys_relations          );
  SIC("isc_info_sql_select"              , isc_info_sql_select                );
  SIC("isc_info_sql_bind"                , isc_info_sql_bind                  );
  SIC("isc_info_sql_num_variables"       , isc_info_sql_num_variables         );
  SIC("isc_info_sql_describe_vars"       , isc_info_sql_describe_vars         );
  SIC("isc_info_sql_describe_end"        , isc_info_sql_describe_end          );
  SIC("isc_info_sql_sqlda_seq"           , isc_info_sql_sqlda_seq             );
  SIC("isc_info_sql_message_seq"         , isc_info_sql_message_seq           );
  SIC("isc_info_sql_type"                , isc_info_sql_type                  );
  SIC("isc_info_sql_sub_type"            , isc_info_sql_sub_type              );
  SIC("isc_info_sql_scale"               , isc_info_sql_scale                 );
  SIC("isc_info_sql_length"              , isc_info_sql_length                );
  SIC("isc_info_sql_null_ind"            , isc_info_sql_null_ind              );
  SIC("isc_info_sql_field"               , isc_info_sql_field                 );
  SIC("isc_info_sql_relation"            , isc_info_sql_relation              );
  SIC("isc_info_sql_owner"               , isc_info_sql_owner                 );
  SIC("isc_info_sql_alias"               , isc_info_sql_alias                 );
  SIC("isc_info_sql_sqlda_start"         , isc_info_sql_sqlda_start           );
  SIC("isc_info_sql_stmt_type"           , isc_info_sql_stmt_type             );
  SIC("isc_info_sql_get_plan"            , isc_info_sql_get_plan              );
  SIC("isc_info_sql_records"             , isc_info_sql_records               );
  SIC("isc_info_sql_batch_fetch"         , isc_info_sql_batch_fetch           );
  SIC("isc_info_sql_stmt_select"         , isc_info_sql_stmt_select           );
  SIC("isc_info_sql_stmt_insert"         , isc_info_sql_stmt_insert           );
  SIC("isc_info_sql_stmt_update"         , isc_info_sql_stmt_update           );
  SIC("isc_info_sql_stmt_delete"         , isc_info_sql_stmt_delete           );
  SIC("isc_info_sql_stmt_ddl"            , isc_info_sql_stmt_ddl              );
  SIC("isc_info_sql_stmt_get_segment"    , isc_info_sql_stmt_get_segment      );
  SIC("isc_info_sql_stmt_put_segment"    , isc_info_sql_stmt_put_segment      );
  SIC("isc_info_sql_stmt_exec_procedure" , isc_info_sql_stmt_exec_procedure   );
  SIC("isc_info_sql_stmt_start_trans"    , isc_info_sql_stmt_start_trans      );
  SIC("isc_info_sql_stmt_commit"         , isc_info_sql_stmt_commit           );
  SIC("isc_info_sql_stmt_rollback"       , isc_info_sql_stmt_rollback         );
  SIC("isc_info_sql_stmt_select_for_upd" , isc_info_sql_stmt_select_for_upd   );
  SIC("isc_info_sql_stmt_set_generator"  , isc_info_sql_stmt_set_generator    );


  /* 2003.07.08:  The following symbols are no longer available in FB 1.5rc4.
  ** As far as DSR knows, they were included here for completeness, rather than
  ** because kinterbasdb itself actually used them. */
  #ifdef ISCCFG_LOCKMEM_KEY
  SIC("ISCCFG_LOCKMEM_KEY"               , ISCCFG_LOCKMEM_KEY                 );
  SIC("ISCCFG_LOCKSIG_KEY"               , ISCCFG_LOCKSIG_KEY                 );
  SIC("ISCCFG_EVNTMEM_KEY"               , ISCCFG_EVNTMEM_KEY                 );
  SIC("ISCCFG_DBCACHE_KEY"               , ISCCFG_DBCACHE_KEY                 );
  SIC("ISCCFG_PRIORITY_KEY"              , ISCCFG_PRIORITY_KEY                );
  SIC("ISCCFG_IPCMAP_KEY"                , ISCCFG_IPCMAP_KEY                  );
  SIC("ISCCFG_MEMMIN_KEY"                , ISCCFG_MEMMIN_KEY                  );
  SIC("ISCCFG_MEMMAX_KEY"                , ISCCFG_MEMMAX_KEY                  );
  SIC("ISCCFG_LOCKORDER_KEY"             , ISCCFG_LOCKORDER_KEY               );
  SIC("ISCCFG_ANYLOCKMEM_KEY"            , ISCCFG_ANYLOCKMEM_KEY              );
  SIC("ISCCFG_ANYLOCKSEM_KEY"            , ISCCFG_ANYLOCKSEM_KEY              );
  SIC("ISCCFG_ANYLOCKSIG_KEY"            , ISCCFG_ANYLOCKSIG_KEY              );
  SIC("ISCCFG_ANYEVNTMEM_KEY"            , ISCCFG_ANYEVNTMEM_KEY              );
  SIC("ISCCFG_LOCKHASH_KEY"              , ISCCFG_LOCKHASH_KEY                );
  SIC("ISCCFG_DEADLOCK_KEY"              , ISCCFG_DEADLOCK_KEY                );
  SIC("ISCCFG_LOCKSPIN_KEY"              , ISCCFG_LOCKSPIN_KEY                );
  SIC("ISCCFG_CONN_TIMEOUT_KEY"          , ISCCFG_CONN_TIMEOUT_KEY            );
  SIC("ISCCFG_DUMMY_INTRVL_KEY"          , ISCCFG_DUMMY_INTRVL_KEY            );

  /* Since ISCCFG_TRACE_POOLS_KEY is marked "For internal use only" in ibase.h,
  ** it is not loaded into the module's namespace. */

  SIC("ISCCFG_REMOTE_BUFFER_KEY"         , ISCCFG_REMOTE_BUFFER_KEY           );
  #endif /* ISCCFG_LOCKMEM_KEY:2003.07.08 */

  #ifdef ISCCFG_NO_NAGLE_KEY
    SIC("ISCCFG_NO_NAGLE_KEY"              , ISCCFG_NO_NAGLE_KEY                );
  #endif
  #ifdef ISCCFG_CPU_AFFINITY_KEY
    SIC("ISCCFG_CPU_AFFINITY_KEY"          , ISCCFG_CPU_AFFINITY_KEY            );
  #endif

  return 0;
} /* _init_kiservices_ibase_header_constants */


static PyMethodDef _kiservices_GlobalMethods[] = {
  { "initialize_from",     pyob_initialize_from,             METH_VARARGS },

  { "connect",             pyob_connection_connect,          METH_VARARGS },
  { "close",               pyob_connection_close,            METH_VARARGS },

  { "action_thin",         pyob_action_thin,                 METH_VARARGS },

  { "query_base",          pyob_query_base,                  METH_VARARGS },
  { "vax",                 pyob_isc_vax_integer,             METH_VARARGS },

  {NULL, NULL}
};


DL_EXPORT(void)
init_kiservices(void) {
  /* This function is called automatically when the module is first imported. */
  PyObject *module_dict;

  this_module = Py_InitModule("_kiservices", _kiservices_GlobalMethods);
  module_dict = PyModule_GetDict(this_module);

  /* Cause a bunch of constants defined at the C level to be loaded into the
  ** module dictionary so that they'll be accessible to the Python programmer
  ** using this module (that audience should be limited to kinterbasdb
  ** developers). */
  if ( 0 != _init_kiservices_ibase_header_constants(module_dict) ) {
    PyErr_NoMemory();
  }
} /* init_kiservices */

/****************** MODULE ADMINISTRATION FUNCTIONS:END ********************/
